/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include "SetColoredVertices.hh"
#include "../serialization/Containers.hh"
#include "../serialization/Eigen.hh"

#ifdef __PEEKABOT_SERVER
#  include <boost/math/fpclassify.hpp>

#  include "../ServerExecutionContext.hh"
#  include "../VertexObject.hh"
#  include "../SceneObject.hh"
#endif

#include <cassert>
#include <stdexcept>
#include <boost/foreach.hpp>


using namespace peekabot;


SetColoredVertices::SetColoredVertices() throw()
{
}


SetColoredVertices::SetColoredVertices(
    ObjectID target,
    const std::vector<Eigen::Vector3f> &vertices,
    const std::vector<boost::uint8_t> &colors,
    bool add_to_existing) throw()
    : m_target(target),
      m_vertices(vertices),
      m_colors(colors),
      m_add_to_existing(add_to_existing)
{
}


SetColoredVertices::SetColoredVertices(const SetColoredVertices &x) throw()
    : m_target(x.m_target),
      m_vertices(x.m_vertices),
      m_colors(x.m_colors),
      m_add_to_existing(x.m_add_to_existing)
{
}


SetColoredVertices::~SetColoredVertices() throw()
{
}


Action *SetColoredVertices::clone() const
{
    return new SetColoredVertices(*this);
}


void SetColoredVertices::execute(ServerExecutionContext *context) const
{
#ifdef __PEEKABOT_SERVER
    SceneObject *obj = context->get_object(m_target);

    if( !obj )
        throw std::runtime_error("Object not found");

    BOOST_FOREACH( const Eigen::Vector3f &v, m_vertices )
    {
        for( std::size_t i = 0; i < 3; ++i )
            if( !boost::math::isfinite(v(i)) )
                throw std::logic_error(
                    "Failed to set vertices: vertices must not "
                    "contain infinity or NaN");
    }

    VertexObject *v_obj = dynamic_cast<VertexObject *>(obj);

    if( !v_obj )
        throw std::runtime_error(
            "Target object is not vertex-based");

    if( m_add_to_existing )
        v_obj->add_colored_vertices(m_vertices, m_colors);
    else
        v_obj->set_colored_vertices(m_vertices, m_colors);
#endif
}


void SetColoredVertices::save(SerializationInterface &ar) const
{
    ar << m_target << m_vertices << m_colors << m_add_to_existing;
}

void SetColoredVertices::load(DeserializationInterface &ar)
{
    ar >> m_target >> m_vertices >> m_colors >> m_add_to_existing;
}
