/*
 * Copyright Anders Boberg 2006-2007.
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_XERCES_PARSER_HH_INCLUDED
#define PEEKABOT_XERCES_PARSER_HH_INCLUDED

#include "XMLParser.hh"
#include <xercesc/sax2/SAX2XMLReader.hpp>
#include <xercesc/sax2/DefaultHandler.hpp>
#include <xercesc/util/XercesVersion.hpp>

namespace peekabot
{

    /**
     * \internal
     *
     * \brief An implementation of the XMLParser interface
     * using the Xerces-C++ library.
     *
     * This class takes care of initialization and deinitialization
     * of Xerces-C++ library components, translation between libary
     * dependent types and standard types and throwing exceptions
     * when parsing errors are encountered.
     *
     */
    class XercesParser : public XMLParser
    {
    public:
        XercesParser() : m_handler(0) {}
        
        XercesParser(XMLHandler* handler)
            : m_handler(handler) {}
    
        /** \brief Parses the specified XML file.
         *
         * This method will initialize the Xerces-C++ library,
         * parse the specified document and then deinitialize
         * the library.
         * 
         * Parsing events (except for errors) will be forwarded
         * to the \c XMLHandler object set as this parser's handler.
         */
        virtual void invoke(const std::string& filename)
            throw(std::exception);

        /** \brief Set the \c XMLHandler object that will handle
         * parsing events encountered by this parser.
         */
        virtual void set_handler(XMLHandler *handler)
        {
            m_handler = handler;
        }

        /** \brief Sets the schema to validate XML documents against.
         *
         * This setting overrides any schemas specified in the XML
         * document to be parsed.
         */
        void set_schema(const std::string & schema)
        {
            m_schema = schema;
        }

    private:
        class XercesHandler : public xercesc::DefaultHandler
        {
        public:
            XercesHandler();

            XercesHandler(XMLHandler *handler);

            void set_handler(XMLHandler *handler);

            virtual void setDocumentLocator(
                const xercesc::Locator * const locator);

            virtual void startDocument();
    
            virtual void endDocument();
            
            virtual void startElement(
                const XMLCh* const uri, 
                const XMLCh* const localname, 
                const XMLCh* const qname, 
                const xercesc::Attributes& attrs);
            
            virtual void endElement(
                const XMLCh* const uri, 
                const XMLCh* const localname,
                const XMLCh* const qname);

            virtual void characters(
                const XMLCh* const chars,
#if _XERCES_VERSION >= 30000
                const XMLSize_t length
#else
                const unsigned int length
#endif
                );

            virtual void error(
                const xercesc::SAXParseException& exception)
                throw(std::runtime_error);
            
            virtual void fatalError(
                const xercesc::SAXParseException& exception)
                throw(std::runtime_error);

            XMLSSize_t get_last_line() const throw();

            XMLSSize_t get_last_col() const throw();

            bool has_locator() const throw();

        protected:
            void update_line_info() throw();
            
        private:
            XMLHandler *m_handler;
            
            const xercesc::Locator *m_locator;

            XMLSSize_t m_last_line, m_last_col;
        };

        XMLHandler* m_handler;

        std::string m_schema;
    };

}
#endif // PEEKABOT_XERCES_PARSER_HH_INCLUDED
