/*
 * Copyright Anders Boberg 2006-2007.
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdexcept>
#include <boost/format.hpp>
#include <sstream>
#include <boost/scoped_ptr.hpp>
#include <boost/scoped_array.hpp>
#include <boost/function.hpp>
#include <boost/filesystem/operations.hpp>

#include <xercesc/sax/Locator.hpp>
#include <xercesc/sax2/Attributes.hpp>
#include <xercesc/sax2/SAX2XMLReader.hpp>
#include <xercesc/sax2/XMLReaderFactory.hpp>
#include <xercesc/util/XMLString.hpp>
#include <xercesc/util/XMLUni.hpp>
#include <xercesc/framework/LocalFileInputSource.hpp>

#include "FsToolbox.hh"
#include "XercesParser.hh"
#include "XMLHandler.hh"
#include "MessageHub.hh"


using namespace peekabot;


namespace
{
    /*
     * Small utility class that executes the supplied function when an instance
     * goes out of scope.
     */
    class ScopedInvoke
    {
    public:
        ScopedInvoke(boost::function<void (void)> fun) : m_fun(fun) {}

        ~ScopedInvoke() { m_fun(); }

    private:
        boost::function<void (void)> m_fun;
    };

    class ParseError : public std::runtime_error
    {
    public:
        ParseError(const std::string &what)
            : std::runtime_error(what) {}
    };
}


void XercesParser::invoke(const std::string& filename)
    throw(std::exception)
{
    if( m_handler == 0 )
        throw std::runtime_error("XercesParser error: No XMLHandler set!");

    // Check if the document exits - if it does not, throw
    if( !boost::filesystem::exists(fs::make_path(filename)) || 
        boost::filesystem::is_directory(fs::make_path(filename)) )
        throw std::runtime_error("File '" + filename + "' does not exist");
        

    // Initialize Xerces-C++
    xercesc::XMLPlatformUtils::Initialize();
    // Deinitialize Xerces-C++ when the method terminates
    ScopedInvoke terminte_xerces(&xercesc::XMLPlatformUtils::Terminate);

    {
        // Create and initialize a SAX2 reader.
        boost::scoped_ptr<xercesc::SAX2XMLReader> reader( 
            xercesc::XMLReaderFactory::createXMLReader());
        
        boost::scoped_array<XMLCh> xml_str(
            xercesc::XMLString::transcode(filename.c_str()));

        xercesc::LocalFileInputSource source(xml_str.get());

        boost::scoped_ptr<XercesHandler> xerces_handler(
            new XercesHandler(m_handler));

        reader->setContentHandler(xerces_handler.get());
        reader->setErrorHandler(xerces_handler.get());
        
        // This specifies if a document should be validated against a schema.
        bool validate = true;
        reader->setFeature(xercesc::XMLUni::fgSAX2CoreValidation, validate);

        // If set, the m_schema string overrides the schema specified in the
        // document.
        if( m_schema.length() != 0 )
        {
            const fs::path schema_path = m_schema;
            // Check if the schema exits - if it does not, throw
            if( !boost::filesystem::exists(schema_path) || 
                boost::filesystem::is_directory(schema_path) )
                throw std::runtime_error(
                    "Schema file '" + schema_path.string() + 
                    "' does not exist");

            reader->setProperty(
                xercesc::XMLUni::fgXercesSchemaExternalNoNameSpaceSchemaLocation,
                xercesc::XMLString::transcode(schema_path.string().c_str()));
        }
        else
        {
            TheMessageHub::instance().publish(
                LOG_MESSAGE, 
                "Suspicious behaviour detected! Parsing document '" + 
                filename + "' without an explicit schema set.");
        }

        try
        {
            reader->parse(source);
        }
        catch(ParseError &e)
        {
            // Tell the XMLHandler that we failed to parse the document.
            m_handler->on_failure();

            std::stringstream what;
            what << "Syntax error in file '" << filename << "'";
            if( xerces_handler->has_locator() )
            {
                what << ", at line " << xerces_handler->get_last_line()
                     << " (column " << xerces_handler->get_last_col() << ")";
            }
            what << ":\n  " << e.what();

            throw std::runtime_error(what.str());
        }
        catch(std::exception &e)
        {
            // Tell the XMLHandler that we failed to parse the document.
            m_handler->on_failure();

            std::stringstream what;
            what << "An error occured while parsing file '" << filename << "'";
            if( xerces_handler->has_locator() )
            {
                what << ", triggered at line " << xerces_handler->get_last_line()
                     << " (column " << xerces_handler->get_last_col() << ")";
            }
            what << ":\n  " << e.what();

            throw std::runtime_error(what.str());
        }
    }
}

//
// -------------- XercesHandler implementation ---------------
//


XercesParser::XercesHandler::XercesHandler()
    : m_locator(0),
      m_last_line(0),
      m_last_col(0)
{
}

XercesParser::XercesHandler::XercesHandler(XMLHandler *handler) 
    : m_handler(handler), 
      m_locator(0),
      m_last_line(0),
      m_last_col(0)
{
}

void XercesParser::XercesHandler::set_handler(XMLHandler *handler)
{
    m_handler = handler;
}

void XercesParser::XercesHandler::setDocumentLocator(
    const xercesc::Locator * const locator)
{
    m_locator = locator;
}

void XercesParser::XercesHandler::update_line_info() throw()
{
    if( m_locator )
    {
        m_last_line = m_locator->getLineNumber();
        m_last_col = m_locator->getColumnNumber();
    }
}

XMLSSize_t XercesParser::XercesHandler::get_last_line() const throw()
{
    return m_last_line;
}


XMLSSize_t XercesParser::XercesHandler::get_last_col() const throw()
{
    return m_last_col;
}


bool XercesParser::XercesHandler::has_locator() const throw()
{
    return m_locator != 0;
}


void XercesParser::XercesHandler::startDocument()
{
    update_line_info();

    m_handler->on_start_document();
}

void XercesParser::XercesHandler::endDocument()
{
    update_line_info();

    m_handler->on_end_document();
}
    
void XercesParser::XercesHandler::startElement(const XMLCh* const uri, 
                                               const XMLCh* const localname, 
                                               const XMLCh* const qname, 
                                               const xercesc::Attributes& attrs)
{
    update_line_info();

    std::string tag_name(xercesc::XMLString::transcode(localname));

    XMLHandler::AttributeMap attributes;

    // For every attribute encountered...
    for(unsigned int i = 0; i < attrs.getLength(); ++i)
    {
        // ...put attribute name/value pair in attribute list
        std::string name = xercesc::XMLString::transcode(attrs.getLocalName(i));
        std::string value = xercesc::XMLString::transcode(attrs.getValue(i));
        attributes.insert(std::make_pair(name, value));
    }

    m_handler->on_start_element(tag_name, attributes);
}

void XercesParser::XercesHandler::endElement(const XMLCh* const uri, 
                                             const XMLCh* const localname,
                                             const XMLCh* const qname)
{
    update_line_info();

    std::string tag_name(xercesc::XMLString::transcode(localname));

    m_handler->on_end_element(tag_name);
}

void XercesParser::XercesHandler::characters(
    const XMLCh* const chars,
#if _XERCES_VERSION >= 30000
    const XMLSize_t length
#else
    const unsigned int length
#endif
    )
{
    update_line_info();

    std::string cdata(xercesc::XMLString::transcode(chars));

    m_handler->on_cdata(cdata);
}


void XercesParser::XercesHandler::error(
    const xercesc::SAXParseException& exception)
    throw(std::runtime_error)
{
    update_line_info();

    throw ParseError(xercesc::XMLString::transcode(exception.getMessage()));
}

void XercesParser::XercesHandler::fatalError(
    const xercesc::SAXParseException& exception)
    throw(std::runtime_error)
{
    update_line_info();

    throw ParseError(xercesc::XMLString::transcode(exception.getMessage()));
}
