/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_TRI_MESH_HH_INCLUDED
#define PEEKABOT_TRI_MESH_HH_INCLUDED


#include "SceneObject.hh"
#include "ScalableObject.hh"
#include "VertexObject.hh"
#include "HandlerInformer.hh"

#include <string>
#include <vector>
#include <boost/cstdint.hpp>
#include <boost/signals2.hpp>
#include <Eigen/Core>


namespace peekabot
{
    class ScopedHandler;
    class ObjectVisitor;

    /**
     * \internal
     */
    class TriMesh : public SceneObject,
                    public ScalableObject,
                    public VertexObject
    {
    public:
        typedef std::vector<boost::uint32_t> Indices;

        /**
         * \brief Creates a new empty tri-mesh object.
         */
        TriMesh();

        /**
         * \brief Construct a tri-mesh primitive from XML data.
         */
        TriMesh(ScopedHandler *handler);

        virtual void accept(ObjectVisitor *visitor) throw();

        virtual ObjectType get_object_type() const;

        const Indices &get_indices() const;

        void set_indices(const Indices &indices);

        void add_indices(const Indices &indices);

        /// \name Signals
        /// @{

        typedef boost::signals2::signal<void ()> IndicesSetSignal;

        typedef boost::signals2::signal<void (
            const Indices &new_indices)> IndicesAddedSignal;

        inline IndicesSetSignal &indices_set_signal() const
        {
            return m_indices_set_signal;
        }

        inline IndicesAddedSignal &indices_added_signal() const
        {
            return m_indices_added_signal;
        }

        /// @}

    protected:
        virtual PropMap &get_prop_adapters();

    private:
        static void create_prop_adapters(PropMap &adapters);

        /// \name XML handler methods
        /// @{

        /**
         * \brief Element start handler for XML parsing.
         */
        static void start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        void triangles_start_handler(
            const std::string &tag_name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler);

        void triangles_end_handler(
            const std::string &name,
            ScopedHandler *handler);

        static void triangle_start_handler(
            const std::string &tag_name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler);

        static void triangle_cdata_handler(
            const std::string &cdata,
            ScopedHandler *handler);

        /// @}

    private:
        Indices m_indices;

        static HandlerInformer ms_handler_informer;

        mutable IndicesSetSignal m_indices_set_signal;
        mutable IndicesAddedSignal m_indices_added_signal;
    };
}


#endif // PEEKABOT_TRI_MESH_HH_INCLUDED
