/*
 * Copyright Staffan Gimåker 2007-2010.
 * Copyright Anders Boberg 2007-2008.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Slider.hh"
#include "ObjectVisitor.hh"
#include "ScopedHandler.hh"
#include "MessageHub.hh"
#include "ObjectTypes.hh"
#include "PropKeys.hh"
#include "props/Vector3PropBase.hh"

#include <sstream>
#include <boost/bind.hpp>
#include <boost/ref.hpp>
#include <Eigen/LU>


using namespace peekabot;


HandlerInformer Slider::ms_handler_informer(
    "slider", &Slider::start_handler);




Slider::Slider()
    : Joint("slider"),
      m_axis(0, 0, 1)
{
    set_value(0);
}


Slider::Slider(ScopedHandler *handler)
    : Joint("slider", handler),
      m_axis(0, 0, 1)
{
    set_value(0);


    handler->add_start_handler(
        "axis",
        boost::bind(&Slider::axis_start_handler, this, _1, _2, _3));

    handler->add_start_handler(
        "min",
        boost::bind(&Slider::min_start_handler, this, _1, _2, _3));

    handler->add_start_handler(
        "max",
        boost::bind(&Slider::max_start_handler, this, _1, _2, _3));

    handler->add_start_handler(
        "initial",
        boost::bind(&Slider::initial_start_handler, this, _1, _2, _3));

    handler->add_start_handler(
        "offset",
        boost::bind(&Slider::offset_start_handler, this, _1, _2, _3));
}


Slider::~Slider() throw()
{
}


Slider::Slider(const Eigen::Vector3f &axis)
    : Joint("slider")
{
    set_axis(axis);
    set_value(0);
}


Eigen::Transform3f Slider::calculate_dof_transform(
    float value) const throw()
{
    // Check if the object has been translated/rotated
    check_pre_transform();

    // 1. Calculate DOF transform:
    Eigen::Transform3f dof_xform;
    dof_xform = Eigen::Translation3f(
        m_axis(0) * value,
        m_axis(1) * value,
        m_axis(2) * value);

    // 2. Apply the DOF transform to the stored mtop without any previous
    //    DOF transformations and return it:
    return dof_xform;
}


void Slider::set_axis(
    const Eigen::Vector3f &axis,
    CoordinateSystem coord_sys) throw()
{
    m_axis = axis;
    m_axis.normalize();

    // Transform to the proper coordinate system
    if( coord_sys == LOCAL_COORDINATES )
        m_axis = get_transformation().linear() * m_axis;
    else if( coord_sys == WORLD_COORDINATES )
        m_axis = Eigen::Transform3f(
            get_parent_mtow().inverse(Eigen::Isometry)).linear() * m_axis;
    else
        assert( coord_sys == PARENT_COORDINATES );

    // Force transformation to be recalculated, using the new axis
    // of displacement.
    m_dof_xform = calculate_dof_transform(get_value());
    set_transformation(m_pre_dof_mtop * m_dof_xform);

    m_axis_set_signal();
}


const Eigen::Vector3f &Slider::get_axis() const throw()
{
    return m_axis;
}



void Slider::accept(ObjectVisitor *visitor) throw()
{
    visitor->visit(this);
}


ObjectType Slider::get_object_type() const
{
    return SLIDER_OBJECT;
}


PropMap &Slider::get_prop_adapters()
{
    static PropMap *s_prop_adapters = 0;
    if( !s_prop_adapters )
    {
        s_prop_adapters = new PropMap;
        create_prop_adapters(*s_prop_adapters);
        merge_prop_adapters(*s_prop_adapters, Joint::get_prop_adapters());
    }

    return *s_prop_adapters;
}


void Slider::create_prop_adapters(PropMap &adapters)
{
    class AxisAdapter : public Vector3PropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            Slider *p = dynamic_cast<Slider *>(obj);
            assert( p );
            p->set_axis(any_cast<Eigen::Vector3f>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const Slider *p = dynamic_cast<const Slider *>(obj);
            assert( p );
            return Any(p->get_axis());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            Slider *p = dynamic_cast<Slider *>(obj);
            assert( p );
            return p->axis_set_signal();
        }
    };

    adapters.insert(PropMap::value_type(SLIDER_AXIS_PROP, new AxisAdapter));
}


// ---- XML handler methods ----


void Slider::start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    // Creating a scene object by passing a ScopedHandler will cause it to
    // enter a new scope with all registered tag start handlers plus any
    // specific handlers for SceneObject properties which are registered
    // by the parent constructor.
    SceneObject* tmp = new Slider(handler);

    // Set the new object as the current object
    ScopedMap &variables = handler->get_variables();
    variables.push_variable("current_object", tmp);
}


void Slider::axis_start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
    throw()
{
    CoordinateSystem coord_sys;

    if( attributes["system"] == "local" )
        coord_sys = LOCAL_COORDINATES;
    else if( attributes["system"] == "parent" )
        coord_sys = PARENT_COORDINATES;
    else
        coord_sys = WORLD_COORDINATES;


    ScopedHandler::TagScope scope;
    scope.cdata_functor = boost::bind(&Slider::axis_cdata_handler,
                                      this, coord_sys, _1, _2);

    // Enter a new scope where only cdata is valid, and is handled by
    // the axis_cdata_handler() method.
    handler->enter_scope(scope);
}


void Slider::axis_cdata_handler(
    CoordinateSystem coord_sys,
    const std::string &cdata,
    ScopedHandler *handler)
    throw(std::domain_error, std::runtime_error, boost::bad_any_cast)
{
    Eigen::Vector3f v;
    std::stringstream s(cdata);

    s >> v(0) >> v(1) >> v(2);

    if( v.norm() < 1e-6 )
        throw std::runtime_error("Translation axis must be non-zero");

    set_axis(v, coord_sys);
}


void Slider::min_start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope scope;

    boost::function<void (float)> g = boost::bind(
        &Joint::set_min_value, this, _1);
    boost::function<void (float)> f = boost::bind(
        &Slider::invoke_on_end_element,
        boost::ref(handler->get_current_scope()), g, _1);

    scope.cdata_functor = boost::bind(
        &Slider::generic_cdata_handler<float>,
        _1, _2, f);

    handler->enter_scope(scope);
}

void Slider::max_start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope scope;

    boost::function<void (float)> g = boost::bind(
        &Joint::set_max_value, this, _1);
    boost::function<void (float)> f = boost::bind(
        &Slider::invoke_on_end_element,
        boost::ref(handler->get_current_scope()), g, _1);

    scope.cdata_functor = boost::bind(
        &Slider::generic_cdata_handler<float>,
        _1, _2, f);

    handler->enter_scope(scope);
}


void Slider::initial_start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
    throw()
{
    ScopedHandler::TagScope scope;

    boost::function<void (float)> g = boost::bind(&Joint::set_value, this, _1);
    boost::function<void (float)> f = boost::bind(
        &Slider::invoke_on_end_element,
        boost::ref(handler->get_current_scope()), g, _1);

    scope.cdata_functor = boost::bind(
        &Slider::generic_cdata_handler<float>, _1, _2, f);

    // Enter a new scope where only cdata is valid
    handler->enter_scope(scope);
}


void Slider::offset_start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
    throw()
{
    ScopedHandler::TagScope scope;

    boost::function<void (float)> g = boost::bind(
        &Joint::set_value_offset, this, _1);
    boost::function<void (float)> f = boost::bind(
        &Slider::invoke_on_end_element,
        boost::ref(handler->get_current_scope()), g, _1);

    scope.cdata_functor = boost::bind(
        &Slider::generic_cdata_handler<float>, _1, _2, f);

    // Enter a new scope where only cdata is valid
    handler->enter_scope(scope);
}


void Slider::invoke_on_end_element(
    ScopedHandler::TagScope &dof_scope,
    boost::function<void (float)> f, float val)
{
    dof_scope.end_functors.push_back(boost::bind(f, val));
}
