/*
 * Copyright Staffan Gimåker 2006-2010.
 * Copyright Anders Boberg 2007.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ServerExecutionContext.hh"
#include "SceneTree.hh"
#include "SceneFileLoader.hh"
#include "ActionSource.hh"
#include "Action.hh"
#include "Config.hh"
#include "Path.hh"
#include "UploadCache.hh"
#include "ObjectTypes.hh"

#include "SphereObject.hh"
#include "CubeObject.hh"
#include "CylinderObject.hh"
#include "CircleObject.hh"
#include "PolygonObject.hh"
#include "LineCloud.hh"
#include "PointCloud.hh"
#include "GridObject.hh"
#include "Label.hh"
#include "CameraObject.hh"
#include "OccupancyGrid2D.hh"
#include "OccupancyGrid3D.hh"
#include "TriMesh.hh"
#include "GroupObject.hh"
#include "Slider.hh"
#include "Hinge.hh"
#include "Polyline.hh"
#include "ModelObject.hh"

#include <sstream>
#include <stdexcept>
#include <boost/foreach.hpp>


using namespace peekabot;


namespace
{
    void check_arg_count(const std::vector<Any> &args, std::size_t expected)
    {
        if( args.size() != expected )
        {
            std::ostringstream ss;
            ss << "Invalid number of arguments. Expected " << expected
               << ", got " << args.size();
            throw std::runtime_error(ss.str());
        }
    }

    template<typename T>
    T get_arg(const std::vector<Any> &args, std::size_t idx)
    {
        try
        {
            return any_cast<T>(args[idx]);
        }
        catch(BadAnyCast &)
        {
            std::ostringstream ss;
            ss << "Argument " << idx << " is of an incorrect type";
            throw std::runtime_error(ss.str());
        }
    }
}


ServerExecutionContext::ServerExecutionContext(
    const Config &config,
    SceneTree &scene,
    ActionSource &action_source)
    : m_config(config),
      m_scene(scene),
      m_action_source(action_source)
{
}


const Config &ServerExecutionContext::get_config() const
{
    return m_config;
}


Path ServerExecutionContext::make_data_path(const std::string &path) const
{
    return m_action_source.make_data_path(path);
}


SceneObject *ServerExecutionContext::get_root() throw()
{
    return m_scene.get_root();
}


SceneObject *ServerExecutionContext::get_object(ObjectID id) throw()
{
    try
    {
        return m_scene.get_object(m_action_source.pseudonym_to_real(id));
    }
    catch(...)
    {
        return 0;
    }
}


SceneObject *ServerExecutionContext::get_object(
    const std::string &path,
    SceneObject *path_parent) throw()
{
    return m_scene.get_object(path, path_parent);
}


void ServerExecutionContext::register_pseudonym(
    ObjectID real_id, ObjectID pseudonym_id) throw(std::exception)
{
    // Find object...
    SceneObject *obj = m_scene.get_object(real_id);
    if( !obj )
        throw std::runtime_error("Failed to register pseudonym, object not found");

    m_action_source.add_pseudonym(pseudonym_id, obj->get_object_id());
}


void ServerExecutionContext::deregister_pseudonym(
    ObjectID pseudonym_id) throw(std::exception)
{
    m_action_source.remove_pseudonym(pseudonym_id);
}


/*void ServerExecutionContext::post_action(
    Action *action, const SourceID &dest) throw()
{
    TheActionRouter::instance().post_action(
        action, SERVER_SOURCE_ID, dest);
        }*/


void ServerExecutionContext::post_response(Action *action) throw()
{
    m_action_source.send_response(boost::shared_ptr<Action>(action));
}


std::vector<SceneObject *> ServerExecutionContext::load_scene(const Path &path)
{
    return SceneFileLoader::load_scene(m_config, path);
}


void ServerExecutionContext::add_remote_file(
    const std::string &remote_filename,
    const void *data, std::size_t n)
{
    m_action_source.get_upload_cache()->add_file(remote_filename, data, n);
}


void ServerExecutionContext::remove_remote_file(
    const std::string &remote_filename)
{
    m_action_source.get_upload_cache()->remove_file(remote_filename);
}


SceneObject *ServerExecutionContext::create_object(
    ObjectType type, const std::vector<Any> &args)
{
    switch( type )
    {
    case SPHERE_OBJECT:
        check_arg_count(args, 0);
        return new SphereObject;

    case CUBE_OBJECT:
        check_arg_count(args, 0);
        return new CubeObject;

    case CYLINDER_OBJECT:
        check_arg_count(args, 0);
        return new CylinderObject;

    case CIRCLE_OBJECT:
        check_arg_count(args, 0);
        return new CircleObject;

    case POLYGON_OBJECT:
        check_arg_count(args, 0);
        return new PolygonObject;

    case LINE_CLOUD_OBJECT:
        check_arg_count(args, 0);
        return new LineCloud;

    case POINT_CLOUD_OBJECT:
        check_arg_count(args, 0);
        return new PointCloud;

    case GRID_OBJECT:
        check_arg_count(args, 0);
        return new GridObject;

    case LABEL_OBJECT:
        check_arg_count(args, 0);
        return new Label;

    case CAMERA_OBJECT:
        check_arg_count(args, 0);
        return new CameraObject;

    case OG2D_OBJECT:
        check_arg_count(args, 1);
        return new OccupancyGrid2D(get_arg<float>(args, 0));

    case OG3D_OBJECT:
        check_arg_count(args, 2);
        return new OccupancyGrid3D(
            get_arg<float>(args, 0), get_arg<float>(args, 1));

    case TRI_MESH_OBJECT:
        check_arg_count(args, 0);
        return new TriMesh;

    case GROUP_OBJECT:
        check_arg_count(args, 0);
        return new GroupObject;

    case SLIDER_OBJECT:
        check_arg_count(args, 0);
        return new Slider;

    case HINGE_OBJECT:
        check_arg_count(args, 0);
        return new Hinge;

    case POLYLINE_OBJECT:
        check_arg_count(args, 0);
        return new Polyline;

    case MODEL_OBJECT:
        check_arg_count(args, 1);
        return new ModelObject(
            make_data_path(get_arg<std::string>(args, 0)));

    default:
        std::ostringstream ss;
        ss << "Unknown object type " << type;
        throw std::runtime_error(ss.str());
    }
}
