/*
 * Copyright Staffan Gimåker 2006-2009.
 * Copyright Anders Boberg 2006.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_SCENE_TREE_HH_INCLUDED
#define PEEKABOT_SCENE_TREE_HH_INCLUDED


#include <map>
#include <string>
#include <boost/utility.hpp>
#include <boost/thread/recursive_mutex.hpp>

#include "Types.hh"


namespace peekabot
{
    class SceneObject;
    class SceneTreePtr;


    /**
     * \internal
     */
    class SceneTree : public boost::noncopyable
    {
    public:
        SceneTree();

        virtual ~SceneTree() throw();

        SceneObject *get_root() throw();

        SceneObject *get_object(ObjectID id) throw();

        /**
         * \param path_parent The object the \a path is assumed to be given
         * relative. If null, the path is treated as an absolute path.
         */
        SceneObject *get_object(
            const std::string &path,
            SceneObject *path_parent = 0) throw();

    private:
        friend class SceneTreePtr;

        typedef boost::recursive_mutex SceneTreeMutex;

        SceneTreeMutex m_scene_mutex;

    private:
        SceneObject *m_root;

        /**
         * \internal
         */
        class TranslationTable : public boost::noncopyable
        {
        public:
            /**
             * \brief Given the root of a <tt>SceneObject</tt>-tree, construct a
             * translation table which will automatically keep it self up to 
             * date.
             *
             * \param root The root object of the tree to keep a translation table
             * for.
             */
            TranslationTable(SceneObject *root) throw();

            SceneObject *translate(ObjectID id) const throw();

        private:
            void on_child_attached(SceneObject *object) throw();

            /**
             * \brief For \c object and all of its children, each object will be 
             * removed from the octree and deregistered from triggering the 
             * octree updater.
             *
             * Each object in the subtree will be deregistered with the stimuli 
             * listed in \c on_attach_object(SceneObject *).
             *
             * \param The subtree to remove from the octree and deregister the 
             * updater with.
             */
            void on_detached(SceneObject *object) throw();

            bool add_rule(SceneObject *object) throw();

            bool remove_rule(SceneObject *object) throw();

        private:
            typedef std::map<ObjectID, SceneObject *> TranslationMap;

            TranslationMap m_id_map;
        };

        TranslationTable m_translation_table;
    };
}


#endif // PEEKABOT_SCENE_TREE_HH_INCLUDED
