/*
 * Copyright Staffan Gimåker 2006-2010.
 * Copyright Anders Boberg 2006-2007.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_SCENE_OBJECT_HH_INCLUDED
#define PEEKABOT_SCENE_OBJECT_HH_INCLUDED


#include <utility>
#include <string>
#include <stdexcept>
#include <set>
#include <boost/signals2.hpp>
#include <Eigen/Core>
#include <Eigen/Geometry>
#include <boost/bimap.hpp>
#include <boost/cstdint.hpp>

#include "Types.hh"
#include "XMLHandler.hh"
#include "Any.hh"
#include "PropMap.hh"


namespace peekabot
{
    class SceneObject;
    class ObjectVisitor;
    class ScopedHandler;
    class Prop;

    /**
     * \internal
     *
     * \brief The base object class.
     *
     * The SceneObject class and its derivates constitute the core of peekabot's
     * data repository.
     * The scene, or world if you wish, in peekabot is comprised of a tree of
     * SceneObject-derivates that are manipulated by the user and clients.
     *
     * All objects share a set of common attributes such as a transformation,
     * color, layer, etc.
     *
     * \section object_naming Naming and paths
     *
     * In order for clients to have the ability to manipulate already existing
     * objects all objects are named. A \e path to an object consists of its name
     * appended to its parent's path, separated by a dot. For example
     * <em>root.robot.scanner</em> is a path to the scanner object attached to
     * robot, which is in turned attached to the scene root.
     *
     * Given the explaination of paths above the two restrictions that apply to
     * names seem natural:
     * \li They cannot contain dots.
     * \li All children of an object must have unique names.
     *
     * To ease the burden on the end user, name conflicts can often be solved
     * automatically by peekabot, see NameConflictPolicy.
     */
    class SceneObject
    {
    public:
        class ChildNotFound : public std::runtime_error
        {
        public:
            ChildNotFound(const std::string &name)
                : std::runtime_error("Child '" + name + "' not found") {}
        };



        SceneObject(const std::string &default_name);

        /** \brief Create a \c SceneObject to be initialized by a \c XMLHandler.
         */
        SceneObject(const std::string &default_name, ScopedHandler *handler);

        virtual ~SceneObject();

        /**
         * \brief Retrieves the local-to-world coordinates transformation
         * matrix for the object.
         *
         * \return The local-to-world space transformation matrix.
         */
        inline const Eigen::Transform3f &get_mtow() const throw()
        {
            return m_mtow;
        }

        /**
         * \brief Get the parent's local-to-world coordinates transformation
         * matrix.
         *
         * \return If the object has no parent, its local-to-world matrix will
         * be returned instead.
         */
        inline const Eigen::Transform3f &get_parent_mtow() const throw()
        {
            return m_parent ? m_parent->get_mtow() : get_mtow();
        }

        void detach();

        void rearrange(SceneObject *new_parent, bool retain_mtow);

        /**
         * \brief Returns true if \c object is a descendant of this object.
         */
        bool is_descendant(SceneObject *object) const throw();

        /**
         * \brief Returns the parent of an object.
         *
         * \return The object's parent, or a zero pointer if it has no parent.
         */
        SceneObject *get_parent() throw();

        /**
         * \copydoc get_parent()
         */
        const SceneObject *get_parent() const throw();

        typedef std::set<SceneObject *> Children;
        typedef Children::iterator ChildIterator;
        typedef Children::const_iterator ConstChildIterator;

        /**
         * \brief Get the object's children.
         *
         * \return A list of references to all the object's children.
         *         The list can be empty.
         */
        Children &get_children() throw();

        const Children &get_children() const throw();

        ChildIterator begin();

        ChildIterator end();

        ConstChildIterator begin() const;

        ConstChildIterator end() const;

        void clear();

        SceneObject *get_child(const std::string &name)
            throw(ChildNotFound);

        const SceneObject *get_child(const std::string &name)
            const throw(ChildNotFound);

        bool has_child(const std::string &name) const throw();

        /**
         * \brief Check whether the given (relative) path is a descendent.
         *
         * \param rel_path The path of the descendent to return, relative the
         * object.
         * \return \c true if \a rel_path is a descendent of the object,
         * \c false otherwise.
         */
        bool has_descendent(const std::string &rel_path) const throw();

        /**
         * \brief Return the descendent described by the given (relative) path.
         *
         * \param rel_path The path of the descendent to return, relative the
         * object.
         * \return The descendent object.
         * \throw std::runtime_error Thrown if \a rel_path points to a
         * non-existant object.
         */
        SceneObject *get_descendent(const std::string &rel_path)
            throw(std::runtime_error);

        /**
         * \copydoc SceneObject::get_descendent(const std::string &)
         */
        const SceneObject *get_descendent(const std::string &rel_path)
            const throw(std::runtime_error);

        /** \brief Returns the number of children of this object.
         */
        std::size_t child_count() const throw();

        /**
         * \brief Get the object's \c ObjectID.
         *
         * \return The object's \c ObjectID, if it has not been assigned an ID
         *         the value returned is undefined.
         */
        ObjectID get_object_id() const throw();

        /**
         * \brief Attach a child object to the object.
         *
         * In order for this operation to succeed one condition must be met:
         * No child of the object has the same name as the object to be added.
         *
         * \return \c true is returned if auto-enumeration was enabled and the
         * name was changed to resolve a name conflict.
         *
         * \exception std::runtime_error Cast if adding the subtree would've
         * caused a name uniqueness violation.
         */
        bool attach(SceneObject *subtree,
                    NameConflictPolicy conflict_policy = FAIL_ON_CONFLICT)
            throw(std::runtime_error);

        /**
         * \brief Get which layer the object is (shown) in.
         *
         * If no layer has been assigned to the object, the layer it is in
         * is undefined, but consistent among all objects. The default layer
         * is guaranteed to be a valid layer.
         *
         * \return The layer in which the object is in.
         */
        boost::uint8_t get_layer() const throw();

        /**
         * \brief Set the layer the object is (shown) in.
         *
         * Not all values for \c layer are valid. Negative numbers
         * are never allowed and other number might be rejected. In case
         * of failure to comply, the operation will return \c true.
         *
         * \param layer The layer in which to move the object into.
         * \return \c false if the operation succeeded, \c true if it failed.
         */
        void set_layer(boost::uint8_t layer);

        /**
         * \brief Get the object's name.
         *
         * \note The reference returned isn't guaranteed to have a long
         *       lifetime - it may change or be deleted after the calling
         *       method has exited.
         * \note An object's name is always unique among its parent's children.
         *
         * \return A reference to the object's meta name.
         * \sa default_name()
         */
        const std::string &get_name() const throw();

        /**
         * \brief Set the object's name.
         *
         * The name of an object must be unique among it's parent's children.
         *
         * \exception std::runtime_error Thrown if the name is not unique among
         * its siblings.
         */
        void set_name(const std::string &name);

        /**
         * \brief Get the object's opacity value.
         *
         * \return The object's opacity in the range [0,1].
         * \sa \c get_accumulated_opacity()
         */
        float get_opacity() const throw();

        /**
         * \brief Set the object's opacity.
         *
         * \param opacity The desired opacity value in the range [0,1].
         */
        void set_opacity(float opacity);

        /**
         * \brief Sets the visibility of an object.
         *
         * Sets whether this object is hidden or visible.
         *
         * \param hide \c true to set object as hidden, \c false to set as
         * shown.
         */
        void set_hidden(bool hidden);

        /**
         * \brief Returns whether this object is hidden or not.
         *
         * \return \c true if the object is hidden, \c false if it is shown.
         */
        bool is_hidden() const throw();

        /**
         * \brief Returns \c false if the object or any of its ancestors are
         * hidden.
         */
        bool is_visible() const throw();

        /**
         * \brief Get the object's accumulated opacity value.
         *
         * \return The object's accumulated opacity.
         */
        float get_accumulated_opacity() const throw();

        virtual void accept(ObjectVisitor *visitor) throw() = 0;

        /**
         * \brief Returns a unique per class id that identifies the type.
         */
        virtual ObjectType get_object_type() const = 0;

        /** \brief Returns a constant reference to the transformation matrix of
         * this \c SceneObject.
         */
        inline const Eigen::Transform3f &get_transformation() const throw()
        {
            return m_mtop;
        }

        /**
         * \brief Set the object's model to parent transformation.
         */
        void set_transformation(
            const Eigen::Transform3f &m,
            CoordinateSystem system = PARENT_COORDINATES);

        /**
         * \brief Apply the given transformation to the model space to parent
         * space transformation.
         *
         * The given transformation is applied through multiplication from the
         * right side.
         */
        void apply_transformation(const Eigen::Transform3f &transform) throw();

        /**
         * \brief Set the object's position, in parent coordinates.
         */
        void set_position(const Eigen::Vector3f &pos) throw();

        /**
         * \brief Sets the object's position, in world coordinates.
         */
        void set_world_position(const Eigen::Vector3f &pos) throw();

        /**
         * \brief Set the rotation of the object from roll, pitch and yaw
         * angles.
         *
         * The angles are specified as Tait-Bryan angles. The rotations
         * applied are extrinsic (used a fixed coordinate system) and applied
         * in the order roll first, then pitch and yaw last (X-Y-Z notation).
         */
        void set_rpy(
            float roll, float pitch, float yaw,
            CoordinateSystem system);

        /** \brief Rotates the object around a given axis.
         *
         * \param axis The axis vector of rotation.
         * \param angle The angle of rotation in radians.
         */
        void rotate(const Eigen::Vector3f &axis, float angle);

        /** \brief Rotates the object around its x axis.
         *
         * \param angle The angle of rotation in radians.
         */
        void rotate_x(float angle) throw();

        /** \brief Rotates the object around its y axis.
         *
         * \param angle The angle of rotation in radians.
         */
        void rotate_y(float angle) throw();

        /** \brief Rotates the object around its z axis.
         *
         * \param angle The angle of rotation in radians.
         */
        void rotate_z(float angle) throw();

        void set_orientation(
            const Eigen::Vector3f &v, CoordinateSystem system);

        /** \brief Sets the color of this object to the specified RGB value.
         *
         * \param color The RGB color value.
         * \param recursive If true the color will also be set on the
         * descendents of the object.
         */
        void set_color(const RGBColor &color, bool recursive = false);

        /** \brief Gets the current color value of this object.
         */
        RGBColor get_color() const throw();

        void set_selected(bool select);

        bool is_selected() const throw();

        bool has_selected_ancestor() const throw();

        /// \name Properties
        /// @{

        typedef PropMap::iterator PropIterator;
        typedef PropMap::const_iterator ConstPropIterator;

        Prop *get_prop(PropKey key);

        const Prop *get_prop(PropKey key) const;

        Any get_prop_value(PropKey key) const;

        void set_prop_value(PropKey key, const Any &val);

        template<typename T> inline
        T get_prop_value(PropKey key) const
        {
            return any_cast<T>(get_prop_value(key));
        }

        template<typename T> inline
        void set_prop_value(PropKey key, const T &val)
        {
            return set_prop_value(key, Any(val));
        }

        std::size_t prop_adapter_count() const;

        PropIterator prop_adapters_begin();

        ConstPropIterator prop_adapters_begin() const;

        PropIterator prop_adapters_end();

        ConstPropIterator prop_adapters_end() const;

        std::size_t prop_count() const;

        PropIterator props_begin();

        ConstPropIterator props_begin() const;

        PropIterator props_end();

        ConstPropIterator props_end() const;

        void erase_prop(PropKey key);

        void erase_prop(PropIterator it);

        bool has_prop(PropKey key) const;

        PropKey get_prop_key(const std::string &name) const;

        PropKey add_prop(const std::string &name, Prop *prop);

        const std::string &get_prop_name(PropKey key) const;

        /// @}
        /// \name Signals
        /// @{

        typedef boost::signals2::signal<void (
            PropKey, Prop *)> PropAddedSignal;

        typedef boost::signals2::signal<void (
            PropKey, Prop *)> PropErasedSignal;

        typedef boost::signals2::signal<void (
            SceneObject *child)> ChildAttachedSignal;

        typedef boost::signals2::signal<void ()> DetachedSignal;

        typedef boost::signals2::signal<void ()> TransformationSetSignal;

        typedef boost::signals2::signal<void ()> NameSetSignal;

        typedef boost::signals2::signal<void ()> LayerSetSignal;

        typedef boost::signals2::signal<void ()> OpacitySetSignal;

        typedef boost::signals2::signal<void ()> ColorSetSignal;

        typedef boost::signals2::signal<void ()> HiddenSetSignal;

        typedef boost::signals2::signal<void ()> AccumOpacityChangedSignal;

        typedef boost::signals2::signal<void ()> MtowChangedSignal;

        typedef boost::signals2::signal<void ()> VisibilityChangedSignal;

        typedef boost::signals2::signal<void ()> SelectedSetSignal;

        typedef boost::signals2::signal<void (
            )> HasSelectedAncestorChangedSignal;

        // ---

        inline PropAddedSignal &prop_added_signal() const
        {
            return m_prop_added_signal;
        }

        inline PropErasedSignal &prop_erased_signal() const
        {
            return m_prop_erased_signal;
        }

        inline ChildAttachedSignal &child_attached_signal() const
        {
            return m_child_attached_signal;
        }

        inline DetachedSignal &detached_signal() const
        {
            return m_detached_signal;
        }

        inline TransformationSetSignal &transformation_set_signal() const
        {
            return m_transformation_set_signal;
        }

        inline NameSetSignal &name_set_signal() const
        {
            return m_name_set_signal;
        }

        inline LayerSetSignal &layer_set_signal() const
        {
            return m_layer_set_signal;
        }

        inline OpacitySetSignal &opacity_set_signal() const
        {
            return m_opacity_set_signal;
        }

        inline ColorSetSignal &color_set_signal() const
        {
            return m_color_set_signal;
        }

        inline HiddenSetSignal &hidden_set_signal() const
        {
            return m_hidden_set_signal;
        }

        inline AccumOpacityChangedSignal &accum_opacity_changed_signal() const
        {
            return m_accum_opacity_changed_signal;
        }

        inline MtowChangedSignal &mtow_changed_signal() const
        {
            return m_mtow_changed_signal;
        }

        inline VisibilityChangedSignal &visibility_changed_signal() const
        {
            return m_visibility_changed_signal;
        }

        inline SelectedSetSignal &selected_set_signal() const
        {
            return m_selected_set_signal;
        }

        inline HasSelectedAncestorChangedSignal &
        has_selected_ancestor_changed_signal() const
        {
            return m_has_selected_ancestor_changed_signal;
        }

        /// @}

    private:
        /**
         * \brief Recursively re-compute the local-to-world transformations
         * for the subtree parented by the object.
         */
        void recalc_mtows();

        /**
         * \brief Recursively re-compute the visible of the subtree
         * parented by the object.
         */
        void recalc_visibility();

        /**
         * \brief Orthonormalize the local-to-parent space coordinate
         * transformation matrix.
         *
         * In a perfect world, this wouldn't be needed, but due to numerical
         * errors we have to explicitly orthonormalize the transformations
         * once every so often or we'd end up with very weird erroneous
         * results over time.
         */
        void orthonormalize_transform() throw();

        SceneObject *_get_descendent(
            const std::vector<std::string>::const_iterator first,
            const std::vector<std::string>::const_iterator last) throw();

        const SceneObject *_get_descendent(
            const std::vector<std::string>::const_iterator first,
            const std::vector<std::string>::const_iterator last) const throw();

        static void create_prop_adapters(PropMap &adapters);

    protected:
        virtual PropMap &get_prop_adapters();

        const PropMap &get_prop_adapters() const;

        static void merge_prop_adapters(PropMap &to, const PropMap &from);

        /// \name XML handler methods
        /// @{

        /** \brief Element end handler for XML parsing.
         */
        void end_handler(
            const std::string &name,
            ScopedHandler *handler) throw(std::runtime_error);

        /** \brief Name start handler for XML parsing.
         */
        void name_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        /** \brief Name cdata handler for XML parsing.
         */
        void name_cdata_handler(
            const std::string &cdata,
            ScopedHandler *handler) throw(std::runtime_error);

        /** \brief Opacity start handler for XML parsing
         */
        void opacity_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        /** \brief Opacity cdata handler for XML parsing
         */
        void opacity_cdata_handler(
            bool absolute,
            const std::string &cdata,
            ScopedHandler *handler) throw();

        /** \brief Hidden start handler for XML parsing
         */
        void hidden_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        /** \brief XML handler for the transform tag.
         */
        void transform_start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        /** \brief XML handler for the transform end tag.
         */
        void transform_end_handler(
            const std::string &name, ScopedHandler *handler) throw();

        /** \brief XML handler for the transformation/rotate tag.
         */
        void rotate_start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        /** \brief rotate cdata handler for XML parsing
         */
        void rotate_cdata_handler(
            boost::uint32_t tag,
            CoordinateSystem system,
            const std::string &cdata,
            ScopedHandler *handler) throw();

        /** \brief XML handler for the end tag of rotate
         */
        void rotate_end_handler(
            const std::string &name, ScopedHandler *handler) throw();

        /** \brief XML handler for the transformation/translate tag.
         */
        void translate_start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        /** \brief translate cdata handler for XML parsing
         */
        void translate_cdata_handler(
            CoordinateSystem system,
            const std::string &cdata,
            ScopedHandler *handler) throw();

        /** \brief XML handler for the transformation/matrix tag.
         */
        void matrix_start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        /** \brief matrix cdata handler for XML parsing
         */
        void matrix_cdata_handler(
            CoordinateSystem system,
            const std::string &cdata,
            ScopedHandler *handler) throw();

        /** \brief XML handler for the <children> tag.
         */
        void children_start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        /// @}

        //
        // ----------------------- Members ---------------------------
        //

    private:
        PropMap m_props;

        typedef boost::bimap<PropKey, std::string> PropNames;
        static PropNames ms_prop_names;

        static PropKey ms_next_prop_key;

        static ObjectID ms_next_object_id;

        /** \brief \c true if this object is hidden, \c false if it is visible.
         */
        bool m_is_hidden;

        /**
         * \brief False if the object or any of its ancestors are hidden.
         */
        bool m_is_visible;

        /** \brief A pointer to the parent of this object in the scene tree.
         */
        SceneObject *m_parent;

        /** \brief The visibility layer that this object resides in.
         */
        boost::uint8_t m_layer;

        /** \brief The opacity value of this object.
         */
        float m_opacity;

        /** \brief The name of the object.
         */
        std::string m_name;

        /** \brief The unique ObjectID number of the object.
         */
        ObjectID m_id;

        int m_transforms_since_normalization;

        /**
         * \brief The object's color in RGB.
         */
        RGBColor m_color;

        /**
         * \brief A list of pointers to this object's children.
         */
        Children m_children;

        /**
         * \brief Model space to parent space transformation.
         */
        Eigen::Transform3f m_mtop;

        /**
         * \brief Model space to world space transformation.
         */
        Eigen::Transform3f m_mtow;

        bool m_is_selected;


        mutable PropAddedSignal m_prop_added_signal;
        mutable PropErasedSignal m_prop_erased_signal;
        mutable ChildAttachedSignal m_child_attached_signal;
        mutable DetachedSignal m_detached_signal;
        mutable TransformationSetSignal m_transformation_set_signal;
        mutable NameSetSignal m_name_set_signal;
        mutable LayerSetSignal m_layer_set_signal;
        mutable OpacitySetSignal m_opacity_set_signal;
        mutable ColorSetSignal m_color_set_signal;
        mutable HiddenSetSignal m_hidden_set_signal;
        mutable AccumOpacityChangedSignal m_accum_opacity_changed_signal;
        mutable MtowChangedSignal m_mtow_changed_signal;
        mutable VisibilityChangedSignal m_visibility_changed_signal;
        mutable SelectedSetSignal m_selected_set_signal;
        mutable HasSelectedAncestorChangedSignal m_has_selected_ancestor_changed_signal;
    };
}

#endif // PEEKABOT_SCENE_OBJECT_HH_INCLUDED
