/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Path.hh"
#include "UploadCache.hh"

#include <boost/filesystem/operations.hpp>


using namespace peekabot;


Path::Path()
{
}


Path::Path(const std::string &path)
    : m_path(path)
{
}


Path::Path(
    const std::string &path,
    const boost::shared_ptr<UploadCache> &upload_cache,
    const std::list<boost::filesystem::path> &search_dirs)
    : m_path(path),
      m_upload_cache(upload_cache),
      m_search_dirs(search_dirs)
{
}


Path::Path(const Path &other)
    : m_path(other.m_path),
      m_upload_cache(other.m_upload_cache),
      m_search_dirs(other.m_search_dirs)
{
}


Path &Path::operator=(const Path &other)
{
    m_path = other.m_path;
    m_upload_cache = other.m_upload_cache;
    m_search_dirs = other.m_search_dirs;
    return *this;
}


boost::filesystem::path Path::expand(
    const std::string &path,
    const std::list<boost::filesystem::path> &extra_dirs) const
{
    if( m_upload_cache )
    {
        boost::filesystem::path p;
        if( !m_upload_cache->get_local_path(path, p) )
            return p;
    }

    typedef std::list<boost::filesystem::path> SearchDirs;

    for( SearchDirs::const_iterator it = m_search_dirs.begin();
         it != m_search_dirs.end(); ++it )
    {
        if( boost::filesystem::exists(*it / path) )
            return *it / path;
    }

    for( SearchDirs::const_iterator it = extra_dirs.begin();
         it != extra_dirs.end(); ++it )
    {
        if( boost::filesystem::exists(*it / path) )
            return *it / path;
    }

    return boost::filesystem::path(path);
}


boost::filesystem::path Path::get_expanded_path() const
{
    return expand(m_path, std::list<boost::filesystem::path>());
}


void Path::set_path(const std::string &path)
{
    m_path = path;
}


const std::string &Path::get_path() const
{
    return m_path;
}


void Path::set_upload_cache(const boost::shared_ptr<UploadCache> &upload_cache)
{
    m_upload_cache = upload_cache;
}


boost::shared_ptr<const UploadCache> Path::get_upload_cache() const
{
    return m_upload_cache;
}


void Path::append_search_dir(const boost::filesystem::path &dir)
{
    m_search_dirs.push_back(dir);
}


void Path::prepend_search_dir(const boost::filesystem::path &dir)
{
    m_search_dirs.push_front(dir);
}


void Path::set_search_dirs(
    const std::list<boost::filesystem::path> &search_dirs)
{
    m_search_dirs = search_dirs;
}
