/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_OCCUPANCY_GRID_2D_HH_INCLUDED
#define PEEKABOT_OCCUPANCY_GRID_2D_HH_INCLUDED

#include <vector>
#include <Eigen/Core>

#include "SceneObject.hh"
#include "ObjectVisitor.hh"
#include "HandlerInformer.hh"


namespace peekabot
{
    class ScopedHandler;

    class OccupancyGrid2D : public SceneObject
    {
    public:
        /** \brief Creates a new \c OccupancyGrid2D from the given model file.
         *
         * \param model_file_name The file name of the model file to import.
         */
        OccupancyGrid2D(float cell_size) throw();

        /** \brief Construct object from XML data.
         */
        OccupancyGrid2D(ScopedHandler *handler);

        virtual void accept(ObjectVisitor *visitor) throw();

        virtual ObjectType get_object_type() const;

        float get_cell_size() const;

        const RGBColor &get_unoccupied_color() const;

        void set_unoccupied_color(const RGBColor &color);

        const RGBColor &get_occupied_color() const;

        void set_occupied_color(const RGBColor &color);

        // belief = -1 <=> clear cell
        void set_cell(const Eigen::Vector2f &x, float belief) throw();

        void set_cells(const std::vector<std::pair<Eigen::Vector2f, float> > &cells) throw();

        /// \name Signals
        /// @{

        typedef boost::signals2::signal<
            void (const std::vector<std::pair<Eigen::Vector2f, float> > &cells)
            > CellsSetSignal;

        typedef boost::signals2::signal<void ()> UnoccupiedColorSetSignal;

        typedef boost::signals2::signal<void ()> OccupiedColorSetSignal;

        inline CellsSetSignal &cells_set_signal() const
        {
            return m_cells_set_signal;
        }

        inline UnoccupiedColorSetSignal &unoccupied_color_set_signal() const
        {
            return m_unoccupied_color_set_signal;
        }

        inline OccupiedColorSetSignal &occupied_color_set_signal() const
        {
            return m_occupied_color_set_signal;
        }

        /// @}

    protected:
        virtual PropMap &get_prop_adapters();

    private:
        static void create_prop_adapters(PropMap &adapters);

        /// \name XML handler methods
        /// @{

        /** \brief Element start handler for XML parsing.
         */
        static void start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        /// @}

    private:
        //static HandlerInformer ms_handler_informer;

        float m_cell_size;

        RGBColor m_unoccupied_color;
        RGBColor m_occupied_color;

        mutable CellsSetSignal m_cells_set_signal;
        mutable UnoccupiedColorSetSignal m_unoccupied_color_set_signal;
        mutable OccupiedColorSetSignal m_occupied_color_set_signal;
    };
}

#endif // PEEKABOT_OCCUPANCY_GRID_2D_HH_INCLUDED
