/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <cstdarg>
#include <cstdio>

#include "MessageHub.hh"
#include "Message.hh"
#include "Subscriber.hh"


using namespace peekabot;


MessageHub::~MessageHub()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    while( !m_subscribers.empty() )
    {
        MessageType type = m_subscribers.begin()->first;
        std::set<Subscriber *> subs = m_subscribers.begin()->second;

        for( std::set<Subscriber *>::const_iterator it = subs.begin();
             it != subs.end(); ++it )
            unsubscribe( type, *it );
    }
}


void MessageHub::publish(MessageType type, const std::string &msg, ...) 
    const throw()
{
    char msgbuf[2048];
    va_list ap;

    va_start(ap, msg);
    vsnprintf(msgbuf, 2048, msg.c_str(), ap);

    publish( Message(type, msgbuf) );
}


void MessageHub::publish(
    MessageType type, SourceID source, const std::string &msg, ...) 
    const throw()
{
    char msgbuf[2048];
    va_list ap;

    va_start(ap, msg);
    vsnprintf(msgbuf, 2048, msg.c_str(), ap);

    publish( Message(type, source, msgbuf) );
}


void MessageHub::subscribe(MessageType type, Subscriber *recipient) throw()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    SubscriberMap::iterator it = m_subscribers.find(type);

    if( it == m_subscribers.end() )
    {
        // The list for the given type exists, create it:
        it = m_subscribers.insert( 
            std::make_pair(type, SubscriberSet()) ).first;
    }

    it->second.insert( recipient );
    recipient->add_subscription(type, this);
}


void MessageHub::unsubscribe(MessageType type, Subscriber *recipient) throw()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    SubscriberMap::iterator it = m_subscribers.find(type);

    if( it == m_subscribers.end() )
        return;

    recipient->remove_subscription(type, this);
    it->second.erase( recipient );

    if( it->second.empty() )
        m_subscribers.erase(type);
}


void MessageHub::publish(const Message &message) const throw()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    SubscriberMap::const_iterator it = m_subscribers.find(message.get_type());

    if( it == m_subscribers.end() )
        return; // There are no recipients to this message

    for( SubscriberSet::const_iterator jt = it->second.begin();
         jt != it->second.end(); ++jt )
    {
        (*jt)->handle_message( message );
    }
}
