/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __PEEKABOT_MESSAGE_HH
#define __PEEKABOT_MESSAGE_HH


#include <string>
#include <boost/date_time/posix_time/posix_time_types.hpp>

#include "Types.hh"
#include "MessageHub.hh"


namespace peekabot
{


    /**
     * \internal
     *
     * \brief A message class used to pass around information that's to be shown
     * to the user or logged. Typically carries error or information messages.
     *
     * The message class wasn't designed to be infinitely flexible (as there's
     * no real need), but has a certain amount of flexibility built in by 
     * letting the creator of the message leave out certain meta data which
     * not all messages have, such as a (known) source.
     */
    class Message
    {
        /**
         * \brief The message text itself.
         */
        const std::string m_message;
        
        /**
         * \brief What kind of message this is (e.g. debug, or system).
         */
        const MessageType m_type;

        /**
         * \brief Set to true if the message has a source, e.g. the value of
         * \c m_source is valid.
         */
        const bool m_has_source;

        /**
         * \brief The source of the message. Only valid iff \c m_has_source is 
         * true, otherwise undefined.
         */
        const SourceID m_source;

        /**
         * \brief The time the message object was created.
         */
        const boost::posix_time::ptime m_timestamp;

    public:
        /**
         * \brief Construct a new message, tagged with a information about its source.
         *
         * \param type The type of the message.
         * \param source The source that generated the message.
         * \param message The message text.
         */
        Message(
            MessageType type, SourceID source, 
            const std::string &message) throw();


        /**
         * \brief Construct a new message, without information about its source.
         *
         * \param type The type of the message.
         * \param message The message text.
         */
        Message(MessageType type, const std::string &message) throw();

        /**
         * \brief Get the message text.
         *
         * \return A reference to the message text, guaranteed to remain valid 
         * for the lifetime of the message.
         */
        const std::string &get_message() const throw();

        /**
         * \brief Get the message type.
         */
        MessageType get_type() const throw();

        /**
         * \brief Returns true if the message has meta data about its source
         * attached.
         */
        bool has_source() const throw();

        /**
         * \brief Get the source (specified at creation of the message object) 
         * of the message, if applicable.
         *
         * \pre <tt>has_source() == true</tt>
         */
        SourceID get_source() const throw();

        /**
         * \brief Get the time when the \c Message-object was created.
         */
        boost::posix_time::ptime get_timestamp() const throw();
    };



}


#endif // __PEEKABOT_MESSAGE_HH
