/*
 * Copyright Staffan Gimåker 2006-2011.
 * Copyright Anders Boberg 2007.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Types.hh"
#include "gui/Gui.hh"
#include "actions/LoadScene.hh"
#include "Version.hh"
#include "Server.hh"
#include "SceneTree.hh"
#include "SceneObject.hh"
#include "SceneFileLoader.hh"

#include "Log.hh"
#include "PrintfFormatter.hh"
#include "Config.hh"
#include "ModelObject.hh"
#include "Subscriber.hh"
#include "FsToolbox.hh"
#include "Init.hh"

#include <sstream>
#include <cassert>
#include <cstdlib>
#include <iostream>
#include <cmath>
#include <boost/format.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/program_options.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <boost/foreach.hpp>
#include <boost/ref.hpp>


using namespace peekabot;
namespace po = boost::program_options;



static const int DEFAULT_PORT = 5050;


void init_logs(const Config &config) throw();

void setup_config(Config &config) throw();

void init_scene(
    ServerData &sd, const Config &config,
    const std::vector<std::string> &scenes);


class Console : public Subscriber
{
    bool m_is_active;

    /**
     * \brief The formatter used to format the log messages.
     */
    const PrintfFormatter m_formatter;
    
public:
    Console() throw() : m_is_active(false), m_formatter("%m\n") {}

    /**
     * \brief Writes the message to file immediately (i.e. after writing it,
     * it flushes the buffers).
     */
    virtual void handle_message(const Message &message) throw()
    {
        std::cout << m_formatter.format(message);
        std::cout.flush();
    }

    void activate() throw()
    {
        if( !m_is_active )
        {
            TheMessageHub::instance().subscribe(INFO_MESSAGE, this);
            TheMessageHub::instance().subscribe(WARNING_MESSAGE, this);
            TheMessageHub::instance().subscribe(ERROR_MESSAGE, this);
            m_is_active = true;
        }
    }

    void deactivate() throw()
    {
        if( m_is_active )
        {
            TheMessageHub::instance().unsubscribe(INFO_MESSAGE, this);
            TheMessageHub::instance().unsubscribe(WARNING_MESSAGE, this);
            TheMessageHub::instance().unsubscribe(ERROR_MESSAGE, this);
            m_is_active = false;
        }
    }
};


int main(int argc, char** argv)
{
    // Initialize gtkmm and gtkglextmm
    Gtk::Main kit(argc, argv);
    Gtk::GL::init(argc, argv);

    Config config;

    std::vector<std::string> scene_files, overrides;

    po::variables_map vm;
    po::options_description desc("Allowed options");
    desc.add_options()
        ("help,h", "Show this help message and exit")
        ("version,v", "Show the peekabot version and exit")
        ("load-scene,s", po::value<std::vector<std::string> >(&scene_files),
         "Load the specified scene file at start-up (can be used multiple times)")
        ("override,D", po::value<std::vector<std::string> >(&overrides),
         "Override a configuration option. Example: -D debug.debug_log=true");

    po::positional_options_description p;
    p.add("input", 1);
    p.add("output", 2);

    try
    {
        po::store(po::command_line_parser(argc, argv).
                  options(desc).positional(p).run(), vm);
        po::notify(vm);
    }
    catch(po::error &e)
    {
        std::cerr << "Error parsing command line arguments: " 
                  << e.what() << std::endl;
        return -1;
    }

    if( vm.count("help") > 0 )
    {
        std::cout << desc << std::endl;
        return 0;
    }

    if( vm.count("version") > 0 )
    {
        std::cout << "peekabot " << PEEKABOT_VERSION_STRING << std::endl
                  << std::endl
                  << "Compatible with clients using version "
                  << PEEKABOT_COMPATIBLE_VERSION_STRING
                  << " and later." << std::endl
                  << "Compatible with recordings created with version "
                  << PEEKABOT_COMPATIBLE_VERSION_STRING
                  << " and later." << std::endl
                  << std::endl
                  << "Copyright (C) 2006-2011 Staffan Gimåker, "
                  << "Anders Boberg" << std::endl << "http://www.peekabot.org/"
                  << std::endl << std::endl
                  << "peekabot comes with ABSOLUTELY NO WARRANTY. peekabot "
                  << "is free software, and\nyou may use, modify and "
                  << "redistribute it under the terms of the GNU\n"
                  << "General Public License version 3 or later." << std::endl;
        return 0;
    }

    // Parse configuration overrides
    BOOST_FOREACH(const std::string &kv, overrides)
    {
        std::vector<std::string> components;
        boost::split( components, kv, boost::is_any_of("=") );
        if( components.size() != 2 )
        {
            std::cerr << "Error parsing configuration option override: " << kv
                      << " (<key>=<val> expected)" << std::endl;
            return -1;
        }

        boost::trim(components[0]);
        boost::trim(components[1]);

        config.get_overrides().set_raw(components[0], components[1]);
    }

    Console console;
    console.activate();

    fs::create_dir_if_inexistant(fs::get_resource_path());
    fs::create_dir_if_inexistant(fs::get_resource_path() / "data");
    fs::create_dir_if_inexistant(fs::get_resource_path() / "snapshots");
    fs::create_dir_if_inexistant(fs::get_temp_path() / "peekabot");

    //
    // Load configuration and setup default options
    //
    setup_config(config);

    init_logs(config);
    console.deactivate();


    //
    // Log various settings and data that might be useful in dismissing 
    // bug reports from users ;)
    //
    TheMessageHub::instance().publish(
        LOG_MESSAGE, "Version: " +
        PEEKABOT_VERSION_STRING);
    TheMessageHub::instance().publish(
        LOG_MESSAGE, "Compatible with clients using version " +
        PEEKABOT_COMPATIBLE_VERSION_STRING + " and later");
    TheMessageHub::instance().publish(
        LOG_MESSAGE, "Compatible with recordings created using version " +
        PEEKABOT_PBAR_COMPATIBLE_VERSION_STRING + " and later");

    TheMessageHub::instance().publish(
        LOG_MESSAGE, "Resource directory: " + 
        fs::get_resource_path().string());

    TheMessageHub::instance().publish(
        LOG_MESSAGE, "Pkgdatadir: " + 
        fs::get_pkgdata_path().string());

    TheMessageHub::instance().publish(
        LOG_MESSAGE, "Share path: " +
        config.get<fs::path>("resources.share_path").string());

    // Log the command line arguments
    {
        std::string args;
        for( int i = 1; i < argc; ++i )
        {
            args += " ";
            args += argv[i];
        }
        TheMessageHub::instance().publish(
            LOG_MESSAGE, "Command line arguments:" + args);
    }

    peekabot::init();

    Server server(config);

    int port = config.get<int>("network.listen_port", DEFAULT_PORT);
    try
    {
        server.listen_ipv4(port);
    }
    catch(std::exception &e)
    {
        std::stringstream ss;
        ss << "Failed to initialize peekabot on port " << port;
        Gtk::MessageDialog dialog(ss.str(), false, Gtk::MESSAGE_ERROR);
        dialog.set_secondary_text(e.what());
        dialog.run();
        return -1;
    }

    // Load scenes
    scene_files.push_back(
        config.get<boost::filesystem::path>(
            "default_scene", boost::filesystem::path("default_scene.xml")
            ).string());
    server.post(
        boost::bind(
            &init_scene, _1, boost::cref(config), boost::cref(scene_files)));

    server.start();

    gui::Gui gui(config, server);
    gui.run();

    server.stop();


    peekabot::cleanup();

    return 0;
}


void init_logs(const Config &config) throw()
{
    static PrintfFormatter formatter("[%t]: %m (%y, %s)\n");

    static Log log(
        (fs::get_resource_path() / "peekabot.log").string(), formatter);

    TheMessageHub::instance().subscribe(LOG_MESSAGE, &log);
    TheMessageHub::instance().subscribe(INFO_MESSAGE, &log);
    TheMessageHub::instance().subscribe(WARNING_MESSAGE, &log);
    TheMessageHub::instance().subscribe(ERROR_MESSAGE, &log);

    if( config.get<bool>("debug.debug_log", false) )
    {
        static Log dbg_log(
            (fs::get_resource_path() / fs::make_path("debug.log")).string(), formatter);
        TheMessageHub::instance().subscribe(DEBUG_MESSAGE, &dbg_log);
    }
}


bool is_share_path(const boost::filesystem::path &path)
{
    return (
        boost::filesystem::exists(path) &&
        boost::filesystem::is_directory(path) &&
        boost::filesystem::exists(path / "schemas") &&
        boost::filesystem::is_directory(path / "schemas") &&
        boost::filesystem::exists(path / "icons") &&
        boost::filesystem::is_directory(path / "icons") &&
        boost::filesystem::exists(path / "fonts") &&
        boost::filesystem::is_directory(path / "fonts") );
}


boost::filesystem::path get_share_path()
{
    {
        Gtk::MessageDialog dialog("peekabot resources not found");
        dialog.set_secondary_text(
            "peekabot couldn't determine where its shared resources (icons, "
            "XML schemas, etc.) are installed. You will have to manually "
            "select in directory where they can be found."
            "\n\n"
            "On Windows, the directory where the shared resources are "
            "installed is usually INSTALL_DIR/share/peekabot. "
            "On other operating systems, it's typically '/usr/share/peekabot' "
            "or similar."
            "\n\n"
            "This setting will be saved in your configuration file.");
        dialog.run();
    }

    int response;
    std::string dir;

    Gtk::FileChooserDialog file_chooser(
        "Choose shared resources directory",
        Gtk::FILE_CHOOSER_ACTION_SELECT_FOLDER);
    file_chooser.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
    file_chooser.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_OK);

    while( true )
    {
        response = file_chooser.run();
        dir = file_chooser.get_filename();

        if( response != Gtk::RESPONSE_OK || is_share_path(dir) )
        {
            break;
        }
        else
        {
            Gtk::MessageDialog dialog(
                "The directory you specified doesn't seem to contain the "
                "shared peekabot resources - try again.");
            dialog.run();
        }
    }

    if( response == Gtk::RESPONSE_OK && is_share_path(dir) )
    {
        return boost::filesystem::path(dir);
    }

    throw std::runtime_error(
        "Couldn't find the location where the peekabot "
        "shared resources were installed");
}



std::list<boost::filesystem::path> get_data_paths(const Config &config)
{
    std::list<boost::filesystem::path> ret;

    ret = config.get_non_generated("resources.data_paths", ret);

    ret.push_front(fs::get_resource_path() / "models"); // deprecated
    ret.push_front(fs::get_resource_path() / "scenes"); // deprecated

    ret.push_front(fs::get_pkgdata_path() / "data");
    ret.push_front(
        config.get<boost::filesystem::path>("resources.share_path") / "data");
    ret.push_front(fs::get_resource_path() / "data");

    ret.push_back(fs::get_pkgdata_path() / "scenes"); // deprecated
    ret.push_back(fs::get_pkgdata_path() / "models"); // deprecated

    return ret;
}




void setup_config(Config &config) throw()
{
    //
    // Load config file
    //
    boost::filesystem::path conf_path =
        fs::get_resource_path() / fs::make_path("peekabot.conf");

    if( boost::filesystem::exists(conf_path) )
    {
        config.load(conf_path.string());
    }
    else
    {
        TheMessageHub::instance().publish(
            INFO_MESSAGE,
            "No configuration file found - "
            "creating ~/.peekabot/peekabot.conf");

        std::ofstream tmp(conf_path.string().c_str());
    }

    //
    // Setup default values
    //
    AnyMap &defaults = config.get_defaults();
    defaults.set("resources.share_path", fs::get_pkgdata_path());

    //
    // Determine share-path
    //

    // If the currently configured share_path isn't valid, we need the user
    // to input the share_path and save it to in configuration file.
    if( !is_share_path(
            config.get<boost::filesystem::path>("resources.share_path")) )
    {
        boost::filesystem::path share_path = get_share_path();

        // Store share_path in configuration file
        config.get_persistent().set("resources.share_path", share_path);

        // ... and save it!
        config.save((fs::get_resource_path() / "peekabot.conf").string());
    }

    //
    // Set up special/generated options
    //
    AnyMap &gen = config.get_generated();

    gen.insert_adapter<std::list<boost::filesystem::path> >(
        "data_paths", boost::bind(&get_data_paths, boost::ref(config)));
}


void init_scene(
    ServerData &sd, const Config &config,
    const std::vector<std::string> &scenes)
{
    // TODO: error handling

    SceneObject * const root = sd.m_scene->get_root();

    Path p;
    p.set_search_dirs(
        config.get<std::list<boost::filesystem::path> >("data_paths"));

    BOOST_FOREACH( const std::string &path, scenes )
    {
        p.set_path(path);

        std::vector<SceneObject *> objs =
            SceneFileLoader::load_scene(config, p);

        for( std::size_t i = 0; i < objs.size(); ++i )
            root->attach(objs[i], AUTO_ENUMERATE_ON_CONFLICT);
    }
}
