/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdexcept>
#include <boost/bind.hpp>
#include <boost/algorithm/string/case_conv.hpp>

#include "SceneObject.hh"
#include "Label.hh"
#include "ObjectVisitor.hh"
#include "ScopedHandler.hh"
#include "ScopedMap.hh"
#include "PropKeys.hh"
#include "props/StringPropBase.hh"
#include "props/EnumPropBase.hh"
#include "ObjectTypes.hh"


using namespace peekabot;


HandlerInformer Label::ms_handler_informer(
    "label", &Label::start_handler);


Label::Label() throw()
    : SceneObject("label"),
      m_text("text"),
      m_alignment(ALIGN_LEFT)
{
    // Make labels 10 times bigger by default (otherwise they are *tiny*)
    set_scale(10.0f, 10.0f, 10.0f);
}


Label::Label(ScopedHandler* handler) throw()
    : SceneObject("label", handler),
      ScalableObject(handler),
      m_text("text"),
      m_alignment(ALIGN_LEFT)
{
    // Make labels 10 times bigger by default (otherwise they are *tiny*)
    set_scale(10.0f, 10.0f, 10.0f);

    // Register the text tag start handler
    handler->add_start_handler(
        "text",
        boost::bind(&Label::text_start_handler,
                    this, _1, _2, _3));

    // Register the align tag start handler
    handler->add_start_handler(
        "align",
        boost::bind(&Label::align_start_handler,
                    this, _1, _2, _3));
}


void Label::accept(ObjectVisitor* visitor) throw()
{
    visitor->visit(this);
}


ObjectType Label::get_object_type() const
{
    return LABEL_OBJECT;
}


void Label::set_text(const std::string &text) throw()
{
    if( m_text != text )
    {
        m_text = text;
        m_text_set_signal();
    }
}

const std::string &Label::get_text() const throw()
{
    return m_text;
}

void Label::set_alignment(TextAlignment alignment) throw()
{
    if( m_alignment != alignment )
    {
        m_alignment = alignment;
        m_alignment_set_signal();
    }
}

TextAlignment Label::get_alignment() const throw()
{
    return m_alignment;
}


PropMap &Label::get_prop_adapters()
{
    static PropMap *s_prop_adapters = 0;
    if( !s_prop_adapters )
    {
        s_prop_adapters = new PropMap;
        create_prop_adapters(*s_prop_adapters);
        merge_prop_adapters(
            *s_prop_adapters, SceneObject::get_prop_adapters());
        merge_prop_adapters(
            *s_prop_adapters, ScalableObject::get_prop_adapters());
    }

    return *s_prop_adapters;
}


void Label::create_prop_adapters(PropMap &adapters)
{
    class TextAdapter : public StringPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            Label *lbl = dynamic_cast<Label *>(obj);
            assert( lbl );
            lbl->set_text(any_cast<std::string>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const Label *lbl = dynamic_cast<const Label *>(obj);
            assert( lbl );
            return Any(lbl->get_text());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            Label *lbl = dynamic_cast<Label *>(obj);
            assert( lbl );
            return lbl->text_set_signal();
        }
    };

    class AlignmentAdapter : public EnumPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            Label *p = dynamic_cast<Label *>(obj);
            assert( p );
            p->set_alignment(any_cast<TextAlignment>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const Label *p = dynamic_cast<const Label *>(obj);
            assert( p );
            return Any(p->get_alignment());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            Label *p = dynamic_cast<Label *>(obj);
            assert( p );
            return p->alignment_set_signal();
        }

        virtual void get_enum_values(std::vector<std::string> &v) const
        {
            v.push_back("Left");
            v.push_back("Center");
            v.push_back("Right");
        }

        virtual void set_from_text(const std::string &text, SceneObject *obj)
        {
            Label *p = dynamic_cast<Label *>(obj);
            assert( p );

            if( text == "Left" )
                p->set_alignment(ALIGN_LEFT);
            else if( text == "Center" )
                p->set_alignment(ALIGN_CENTER);
            else if( text == "Right" )
                p->set_alignment(ALIGN_RIGHT);
            else
                assert( false );
        }

        virtual std::string get_as_text(const SceneObject *obj) const
        {
            const Label *p = dynamic_cast<const Label *>(obj);
            assert( p );

            if( p->get_alignment() == ALIGN_LEFT )
                return "Left";
            else if( p->get_alignment() == ALIGN_CENTER )
                return "Center";
            else if( p->get_alignment() == ALIGN_RIGHT )
                return "Right";
            else
                assert( false );
        }
    };

    adapters.insert(
        PropMap::value_type(LABEL_TEXT_PROP, new TextAdapter));

    adapters.insert(
        PropMap::value_type(LABEL_ALIGNMENT_PROP, new AlignmentAdapter));
}


//
// --------------- XML handler methods --------------------
//

void Label::start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    // Creating a scene object by passing a ScopedHandler will cause it to
    // enter a new scope with all registered tag start handlers plus any
    // specific handlers for SceneObject properties which are registered
    // by the parent constructor.
    SceneObject *tmp = new Label(handler);

    // Set the new object as the current object
    ScopedMap &variables = handler->get_variables();
    variables.push_variable("current_object", tmp);
}


void Label::text_start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope scope;

    scope.cdata_functor = boost::bind(
        &Label::text_cdata_handler, this, _1, _2);

    handler->enter_scope(scope);
}

void Label::text_cdata_handler(
    const std::string &cdata,
    ScopedHandler *handler) throw()
{
    set_text(cdata);
}


void Label::align_start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope scope;

    scope.cdata_functor = boost::bind(
        &Label::align_cdata_handler, this, _1, _2);

    handler->enter_scope(scope);
}

void Label::align_cdata_handler(
    const std::string &cdata,
    ScopedHandler *handler)
{
    std::string alignment = cdata;
    boost::to_lower(alignment);

    if( alignment == "left" )
        set_alignment(ALIGN_LEFT);
    else if( alignment == "right" )
        set_alignment(ALIGN_RIGHT);
    else if( alignment == "center" )
        set_alignment(ALIGN_CENTER);
    else
        throw std::runtime_error(
            "Invalid alignment '" + alignment +
            "' encountered in definition file");
}
