/*
 * Copyright Staffan Gimåker 2006-2010.
 * Copyright Anders Boberg 2007.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_JOINT_HH_INCLUDED
#define PEEKABOT_JOINT_HH_INCLUDED


#include <stdexcept>
#include <Eigen/Geometry>

#include "Types.hh"
#include "SceneObject.hh"
#include "XMLHandler.hh"


namespace peekabot
{
    class ScopedHandler;


    /**
     * \internal
     *
     * \brief An interface for providing a robot with a degree of freedom, in a
     * way that's convenient to work with for the the users of both the server
     * and proxy library.
     */
    class Joint : public SceneObject
    {
    public:
        /**
         * \brief Construct a new DOF object.
         *
         * \post <tt>get_value() == 0</tt>, <tt>get_min_value() == 0</tt>,
         * <tt>get_max_value() == 0</tt>
         */
        Joint(const std::string &default_name) throw();

        Joint(const std::string &default_name,
            ScopedHandler *handler) throw();

        virtual ~Joint() throw();

        /**
         * \brief Get the lower bound of allowed values (inclusive).
         */
        float get_min_value() const throw();

        /**
         * \brief Set the lower bound of allowed values (inclusive).
         *
         * If the DOF's current value is less than the new lower bound,
         * the value will automatically be set to the lowest allowed value.
         *
         * \pre <tt>min <= get_max_value()</tt>
         * \post <tt>get_min_value() <= get_max_value()</tt>
         *
         * \exception std::domain_error Cast if \f$min>get\_max\_value()\f$.
         */
        void set_min_value(float min) throw(std::domain_error);

        /**
         * \brief Get the upper bound of allowed values (inclusive).
         */
        float get_max_value() const throw();

        /**
         * \brief Set the upper bound of allowed values (inclusive).
         *
         * If the DOF's current value is greater than the new upper bound,
         * the value will automatically be set to the largest allowed value.
         *
         * \pre <tt>max >= get_min_value()</tt>
         * \post <tt>get_min_value() <= get_max_value()</tt>
         *
         * \exception std::domain_error Cast if \f$max<get\_min\_value()\f$.
         */
        void set_max_value(float max) throw(std::domain_error);

        /**
         * \brief Get the DOF's current value.
         *
         * \invariant Let \f$x\f$ be the return value, then
         * \f$get\_min\_value() \leq x \leq get\_max\_value()\f$.
         */
        float get_value() const throw();

        void set_value_offset(float offset) throw();

        float get_value_offset() const throw();

        /**
         * \brief Set the value of the DOF, actual value semantics
         * are subclass-specific.
         *
         * \exception std::domain_error Cast if the specified value lies outside
         * the \f$[get\_min\_value(),get\_max\_value()]\f$ range.
         */
        void set_value(float value) throw(std::domain_error);

       /**
         * \brief Subclass-specific method for calculating how a given DOF value
         * should affect the transformation matrix.
         *
         * This method will call check_pre_mtop() to update its pre-dof
         * transformation matrix.
         */
        virtual Eigen::Transform3f calculate_dof_transform(float value)
            const throw() = 0;

        Eigen::Transform3f get_full_transform(float value)
            const throw();

        /// \name Signals
        /// @{

        typedef boost::signals2::signal<void ()> JointValueSetSignal;

        typedef boost::signals2::signal<void ()> JointMinSetSignal;

        typedef boost::signals2::signal<void ()> JointMaxSetSignal;

        typedef boost::signals2::signal<void ()> JointOffsetSetSignal;

        inline JointValueSetSignal &joint_value_set_signal()
        {
            return m_joint_value_set_signal;
        }

        inline JointMinSetSignal &joint_min_set_signal()
        {
            return m_joint_min_set_signal;
        }

        inline JointMaxSetSignal &joint_max_set_signal()
        {
            return m_joint_max_set_signal;
        }

        inline JointOffsetSetSignal &joint_offset_set_signal()
        {
            return m_joint_offset_set_signal;
        }

        /// @}

    protected:
        /**
         * \brief This method is used to check if the object's transformation
         * matrix has changed due to non-dof-related translation/rotation.
         */
        void check_pre_transform() const throw();


        /**
         * \brief The DOF's mtop prior to applying the DOF-transformation
         * governed by the DOF's value.
         */
        mutable Eigen::Transform3f m_pre_dof_mtop;


        /**
         * \brief The DOF transform for the current DOF-value.
         *
         * We need to store the previous transform in order
         * to correctly calculate it the next time the DOF is set, or
         * it's transformation changed.
         */
        Eigen::Transform3f m_dof_xform;

        virtual PropMap &get_prop_adapters();

    private:
        static void create_prop_adapters(PropMap &adapters);

    private:
        /**
         * \brief The lower bound (inclusive) on allowed values.
         *
         * The initial value is zero.
         */
        float m_min;

        /**
         * \brief The upper bound (inclusive) on allowed values.
         *
         * The initial value is zero.
         */
        float m_max;

        /**
         * \brief The DOF's current value.
         *
         * The initial value is zero.
         */
        float m_val;

        float m_offset;

        JointValueSetSignal m_joint_value_set_signal;
        JointMinSetSignal m_joint_min_set_signal;
        JointMaxSetSignal m_joint_max_set_signal;
        JointOffsetSetSignal m_joint_offset_set_signal;
    };
}


#endif // PEEKABOT_JOINT_HH_INCLUDED
