/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_GRID_OBJECT_HH_INCLUDED
#define PEEKABOT_GRID_OBJECT_HH_INCLUDED


#include "SceneObject.hh"
#include "ScalableObject.hh"
#include "LineBased.hh"
#include "XMLHandler.hh"
#include "HandlerInformer.hh"


namespace peekabot
{
    class ScopedHandler;

    /**
     * \internal
     *
     * \brief A grid object.
     *
     * Enables showing radial and cartesian grids.
     */
    class GridObject : public SceneObject,
                       public ScalableObject,
                       public LineBased
    {
    public:
        GridObject() throw();

        GridObject(ScopedHandler *handler) throw();

        virtual void accept(ObjectVisitor* visitor) throw();

        virtual ObjectType get_object_type() const;

        GridType get_grid_type() const throw();

        void set_grid_type(GridType grid_type) throw();

        boost::uint32_t get_segment_count() const throw();

        void set_segment_count(boost::uint32_t segment_count) throw();

        float get_central_angle() const throw();

        void set_central_angle(float central_angle) throw();

        /// \name Signals
        /// @{

        typedef boost::signals2::signal<void ()> SegmentCountSetSignal;

        typedef boost::signals2::signal<void ()> GridTypeSetSignal;

        typedef boost::signals2::signal<void ()> CentralAngleSetSignal;

        inline SegmentCountSetSignal &segment_count_set_signal()
        {
            return m_segment_count_set_signal;
        }

        inline GridTypeSetSignal &grid_type_set_signal()
        {
            return m_grid_type_set_signal;
        }

        inline CentralAngleSetSignal &central_angle_set_signal()
        {
            return m_central_angle_set_signal;
        }

        /// @}

    protected:
        virtual PropMap &get_prop_adapters();

    private:
        static void create_prop_adapters(PropMap &adapters);

        /**
         * \brief Element start handler for XML parsing.
         */
        static void start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        void type_start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        void type_cdata_handler(
            const std::string &cdata,
            ScopedHandler *handler) throw();

        void segments_start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        void segments_cdata_handler(
            const std::string &cdata,
            ScopedHandler *handler) throw();

        void central_angle_start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        void radians_start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        void degrees_start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        void central_angle_cdata_handler(
            const std::string &cdata,
            ScopedHandler *handler,
            bool value_in_degrees) throw();

    private:
        static HandlerInformer ms_handler_informer;

        GridType m_grid_type;

        boost::uint32_t m_segment_count;

        float m_central_angle;

        SegmentCountSetSignal m_segment_count_set_signal;
        GridTypeSetSignal m_grid_type_set_signal;
        CentralAngleSetSignal m_central_angle_set_signal;
    };
}

#endif // PEEKABOT_GRID_OBJECT_HH_INCLUDED
