/*
 * Copyright Staffan Gimåker 2009-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_CONFIG_HH_INCLUDED
#define PEEKABOT_CONFIG_HH_INCLUDED


#include <string>
#include <istream>
#include <ostream>
#include <boost/utility.hpp>
#include <boost/thread/recursive_mutex.hpp>

#include "AnyMap.hh"
#include "LexicalMap.hh"


namespace peekabot
{
    class Config : public boost::noncopyable
    {
    public:
        Config();

        ~Config();

        template<class T>
        T get(const std::string &key) const
        {
            boost::recursive_mutex::scoped_lock lock(m_mutex);

            if( m_generated.count(key) )
                return m_generated.get<T>(key);
            else if( m_overrides.count(key) )
                return m_overrides.get<T>(key);
            else if( m_persistent.count(key) )
                return m_persistent.get<T>(key);
            else if( m_defaults.count(key) )
                return m_defaults.get<T>(key);
            else
                throw std::runtime_error("Key '" + key + "' not found");
        }


        template<class T>
        T get(const std::string &key, const T &default_val) const
        {
            boost::recursive_mutex::scoped_lock lock(m_mutex);

            if( m_generated.count(key) )
                return m_generated.get<T>(key);
            else if( m_overrides.count(key) )
                return m_overrides.get<T>(key);
            else if( m_persistent.count(key) )
                return m_persistent.get<T>(key);
            else if( m_defaults.count(key) )
                return m_defaults.get<T>(key);
            else
                return default_val;
        }


        template<class T>
        T get_non_generated(const std::string &key) const
        {
            boost::recursive_mutex::scoped_lock lock(m_mutex);

            if( m_overrides.count(key) )
                return m_overrides.get<T>(key);
            else if( m_persistent.count(key) )
                return m_persistent.get<T>(key);
            else if( m_defaults.count(key) )
                return m_defaults.get<T>(key);
            else
                throw std::runtime_error("Key '" + key + "' not found");
        }


        template<class T>
        T get_non_generated(const std::string &key, const T &default_val) const
        {
            boost::recursive_mutex::scoped_lock lock(m_mutex);

            if( m_overrides.count(key) )
                return m_overrides.get<T>(key);
            else if( m_persistent.count(key) )
                return m_persistent.get<T>(key);
            else if( m_defaults.count(key) )
                return m_defaults.get<T>(key);
            else
                return default_val;
        }

        /**
         * \deprecated Use get instead.
         */
        template<class T>
        inline T get_option(const std::string &key, const T &default_val) const
        {
            return get<T>(key, default_val);
        }

        /**
         * \deprecated Use get instead.
         */
        template<class T>
        inline T get_option(const std::string &key) const
        {
            return get<T>(key);
        }

        template<class T>
        void set_persistent(const std::string &key, const T &val)
        {
            boost::recursive_mutex::scoped_lock lock(m_mutex);
            m_persistent.set(key, val);
        }

        bool has(const std::string &key) const;


        template<class T>
        bool has_type(const std::string &key) const
        {
            boost::recursive_mutex::scoped_lock lock(m_mutex);

            if( m_generated.count(key) )
                return m_generated.has_type<T>(key);
            else if( m_overrides.count(key) )
                return m_overrides.has_type<T>(key);
            else if( m_persistent.count(key) )
                return m_persistent.has_type<T>(key);
            else if( m_defaults.count(key) )
                return m_defaults.has_type<T>(key);
            else
                return false;
        }


        void load(const std::string &filename);

        void save(const std::string &filename) const;


        AnyMap &get_generated();

        const AnyMap &get_generated() const;

        AnyMap &get_defaults();

        const AnyMap &get_defaults() const;

        LexicalMap &get_overrides();

        const LexicalMap &get_overrides() const;

        LexicalMap &get_persistent();

        const LexicalMap &get_persistent() const;

    private:
        void load_ini(LexicalMap &lm, std::istream &is);

        void load_ini(LexicalMap &lm, const std::string &filename);

        void save_ini(const LexicalMap &lm, std::ostream &os) const;

        void save_ini(const LexicalMap &lm, const std::string &filename) const;

    private:
        mutable boost::recursive_mutex m_mutex;

        AnyMap m_generated;
        LexicalMap m_overrides;
        LexicalMap m_persistent;
        AnyMap m_defaults;
    };
}


#endif // PEEKABOT_CONFIG_HH_INCLUDED
