/*
 * Copyright Staffan Gimåker 2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Config.hh"
#include "MessageHub.hh"

#include <fstream>


using namespace peekabot;


Config::Config()
{
}


Config::~Config()
{
}


bool Config::has(const std::string &key) const
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    return ( m_generated.count(key) ||
             m_overrides.count(key) ||
             m_persistent.count(key) ||
             m_defaults.count(key) );
}


void Config::load(const std::string &filename)
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    try
    {
        load_ini(m_persistent, filename);
    }
    catch(std::exception &e)
    {
        TheMessageHub::instance().publish(
            ERROR_MESSAGE,
            "Failed to load configuration: %s", e.what());
    }
}


void Config::save(const std::string &filename) const
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    try
    {
        save_ini(m_persistent, filename);
    }
    catch(std::exception &e)
    {
        TheMessageHub::instance().publish(
            ERROR_MESSAGE,
            "Failed to save configuration: %s", e.what());
    }
}


void Config::load_ini(LexicalMap &lm, std::istream &is)
{
    std::string curr_section = "default";

    while( !is.eof() )
    {
        std::string str;
        std::getline(is, str);

        // Skip blank lines
        if( str.empty() )
            continue;

        std::vector<std::string> components;
        boost::split( components, str, boost::is_any_of("#;") );
        assert( !components.empty() );

        std::string &line = components[0];
        boost::trim(line);


        if( line.length() == 0  )
            continue;

        if( line[0] == '[' )
        {
            // Section directive
            if( line[line.length()-1] != ']' )
                throw std::runtime_error(
                    "Malformed section directive: " + line);

            std::string section_name(line, 1, line.length()-2);
            boost::trim(section_name);

            if( section_name.empty() )
                throw std::runtime_error(
                    "Zero-length section names are not allowed: " + line);

            curr_section = section_name;
        }
        else
        {
            // Option directive
            std::vector<std::string> components;
            boost::split( components, line, boost::is_any_of("=") );

            if( components.size() != 2 )
                throw std::runtime_error(
                    "Malformed option directive: " + line);

            std::string &lhs = components[0];
            std::string &rhs = components[1];

            boost::trim(lhs);
            boost::trim(rhs);

            /*if( curr_section.empty() )
                throw std::runtime_error(
                    "Option directive encountered without "
                    "a previous section directive");*/

            lm.set_raw(curr_section + "." + lhs, rhs);
        }
    }
}


void Config::load_ini(LexicalMap &lm, const std::string &filename)
{
    std::ifstream ifs(filename.c_str());
    if( !ifs )
        throw std::runtime_error(
            "Failed to open config file '" + filename + "' for reading");
    load_ini(lm, ifs);
}


void Config::save_ini(const LexicalMap &lm, std::ostream &os) const
{
    std::string prev_section = "default";

    for( LexicalMap::const_iterator it = lm.begin(); it != lm.end(); ++it )
    {
        std::string section, option;

        // Split the key into a <section-name, option-name> pair
        std::vector<std::string> components;
        boost::split( components, it->first, boost::is_any_of(".") );

        if( components.size() == 1 )
        {
            section = "default";
            option = components[0];
        }
        else
        {
            section = components[0];
            for( size_t i = 1; i < components.size(); ++i )
                option += components[i];
        }

        if( section != prev_section )
        {
            os << "\n[" << section << "]\n\n";
            prev_section = section;
        }

        os << option << " = " << it->second << "\n";
    }

    os.flush();
}


void Config::save_ini(const LexicalMap &lm, const std::string &filename) const
{
    std::ofstream ofs(filename.c_str());
    if( !ofs )
        throw std::runtime_error(
            "Failed to open config file '" + filename + "' for writing");
    save_ini(lm, ofs);
}


AnyMap &Config::get_generated()
{
    return m_generated;
}


const AnyMap &Config::get_generated() const
{
    return m_generated;
}


AnyMap &Config::get_defaults()
{
    return m_defaults;
}


const AnyMap &Config::get_defaults() const
{
    return m_defaults;
}


LexicalMap &Config::get_overrides()
{
    return m_overrides;
}


const LexicalMap &Config::get_overrides() const
{
    return m_overrides;
}


LexicalMap &Config::get_persistent()
{
    return m_persistent;
}


const LexicalMap &Config::get_persistent() const
{
    return m_persistent;
}
