/*
 * Copyright Staffan Gimåker 2006-2008.
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <algorithm>
#include <cstring>
#include <cmath>
#include <cassert>

#include "ChunkedBuffer.hh"


using namespace peekabot;



//
// ---------- ChunkedBuffer::Chunk implementation -----------
//

ChunkedBuffer::Chunk::Chunk(size_t _size) throw() 
    : m_capacity(_size), 
      m_wptr(0), 
      m_rptr(0)
{
    m_data = new uint8_t[m_capacity];
}


ChunkedBuffer::Chunk::~Chunk() throw()
{
    delete[] m_data;
}


void ChunkedBuffer::Chunk::clear() throw()
{
    m_wptr = m_rptr = 0;
}



//
// ---------- Node implementation -----------
//


ChunkedBuffer::Node::Node(Node *next, Node *prev, Chunk *chunk) throw()
    : m_next(next), 
      m_prev(prev), 
      m_chunk(chunk)
{
}

ChunkedBuffer::Node::~Node() throw()
{
    if( m_chunk )
        delete m_chunk;
    
    if( m_prev )
        m_prev->m_next = m_next;
    
    if( m_next )
        m_next->m_prev = m_prev;
}


//
// ---------- ChunkedBuffer implementation -----------
//


ChunkedBuffer::ChunkedBuffer(size_t chunk_size) throw()
    : m_chunk_size(chunk_size),
      m_size(0),
      m_chunk_count(1),
      m_read_node(new Node(0, 0, new Chunk(m_chunk_size) )),
      m_write_node(m_read_node)
{
}


ChunkedBuffer::ChunkedBuffer(const ChunkedBuffer &buf) throw()
    : m_chunk_size(buf.m_chunk_size),
      m_size(0),
      m_chunk_count(1),
      m_read_node(new Node(0, 0, new Chunk(m_chunk_size) )),
      m_write_node(m_read_node)
{
    size_t n = buf.get_size();
    if( n > 0 )
    {
        uint8_t *data = new uint8_t[n];
        assert( buf.peek(data, n) == n);
        write(data, n);
        delete[] data;
    }
}


ChunkedBuffer::~ChunkedBuffer() throw()
{
    clear();
    assert( m_read_node == m_write_node );
    delete m_read_node;
}


size_t ChunkedBuffer::read(void *buf, size_t max_bytes) throw()
{
    // Read max_bytes, or as many as bytes as there are available
    size_t n = std::min(max_bytes, m_size);
    size_t remaining = n;
    
    while( remaining > 0 )
    {
        // Request all the remaining bytes, cope with only
        // getting some at a time...
        remaining -= m_read_node->m_chunk->read(
            (uint8_t *)buf+(n-remaining), remaining);
        
        // If we exhausted a chunk we should and must delete it --
        // shrink_on_demand() has this for us.
        //
        // Note that the check to see if the chunk is exhausted isn't
        // necessary since shrink_on_demand() only actually does anything
        // IF it is exhausted -- however, checking it prior to calling
        // shrink_on_demand() eliminates a lot of function calls!
        // Yes, I have profiled it and it makes a BIG difference under 
        // heavy load.
        if( m_read_node->m_chunk->get_available_count() == 0 )
            shrink_on_demand();
    }
    
    m_size -= n;
    
    return n;
}


size_t ChunkedBuffer::discard(size_t max_bytes) throw()
{
    // Read max_bytes, or as many as bytes as there are available
    size_t n = std::min(max_bytes, m_size);
    size_t remaining = n;
    
    while( remaining > 0 )
    {
        // Request all the remaining bytes, cope with only
        // getting some at a time...
        remaining -= m_read_node->m_chunk->discard(remaining);
        
        // If we exhausted a chunk we should and must delete it --
        // shrink_on_demand() has this for us.
        //
        // Note that the check to see if the chunk is exhausted isn't
        // necessary since shrink_on_demand() only actually does anything
        // IF it is exhausted -- however, checking it prior to calling
        // shrink_on_demand() eliminates a lot of function calls!
        // Yes, I have profiled it and it makes a BIG difference under 
        // heavy load.
        if( m_read_node->m_chunk->get_available_count() == 0 )
            shrink_on_demand();
    }
    
    m_size -= n;
    
    return n;
}


uint8_t ChunkedBuffer::read_byte() throw()
{
    uint8_t x;
    read(&x, 1);
    return x;
}


size_t ChunkedBuffer::peek(void *buf, size_t max_bytes) const throw()
{
    // Read max_bytes, or as many as bytes as there are available
    size_t n = std::min(max_bytes, m_size);
    size_t remaining = n;

    Node *node = m_read_node;
    
    while( remaining > 0 )
    {
        assert( node );

        // Request all the remaining bytes, cope with only
        // getting some at a time...
        remaining -= node->m_chunk->peek(
            (uint8_t *)buf+(n-remaining), remaining);

        node = node->m_next;
    }
    
    return (n - remaining);
}


bool ChunkedBuffer::peek(uint8_t &x) const throw()
{
    return !peek(&x, 1);
}


void ChunkedBuffer::write(const void *buf, size_t n) throw()
{
    size_t remaining = n;
    while( remaining > 0 )
    {
        // Heed the order!
        //
        // If we grow first and then write we won't end up in the 
        // undesirable situation that we allocate a chunk that never
        // gets used.
        grow_on_demand();

        remaining -= m_write_node->m_chunk->write(
            (uint8_t *)buf+(n-remaining), remaining);
    }
    
    m_size += n;
}


void ChunkedBuffer::write_byte(uint8_t x) throw()
{
    write(&x, 1);
}


bool ChunkedBuffer::overwrite(const void *buf, size_t n, size_t pos) throw()
{
    Node *node;
    

    // 1) First off, we need to find the node at which the sequence of bytes 
    // to be overwritten begins - the result is stored in node.
    //
    // We can reduce the complexity by a factor two, down to 
    // O(1/2*chunk_size * n) worst case, by searching from either the head 
    // (read node) or the tail (write node) depending on which of the two that
    // lies closest to pos.

    size_t p = m_read_node->m_chunk->get_bytes_read();

    // Calculate the number of nodes we have to skip to reach the start node
    // of the sequence to be overwritten.
    // rsteps and lsteps is the number of steps we have to take if we start
    // from the head and tail respectively.
    int rsteps = (pos+p) / get_chunk_size();
    int lsteps = m_chunk_count-1-rsteps;

    // Find the relevant node, start at the end which lies closest to it.
    if( rsteps <= lsteps )
    {
        node = m_read_node;       
        while( --rsteps >= 0 )
            node = node->m_next;
    }
    else
    {
        node = m_write_node;
        while( --lsteps >= 0 )
            node = node->m_prev;
    }


    if( !node )
        return true;


    // The number of bytes successfully overwritten thus far.
    size_t n_written = 0;


    // We found the right node, now we just have the change the 
    // data in its chunk, and possibly of the following nodes too.
    //
    // 2) Try to write all bytes to the first node's chunk with 
    // an offset, then in 3) write the remaining bytes ...
    n_written += node->m_chunk->overwrite(
        (uint8_t *)buf+n_written, 
        n-n_written, 
        (p+pos)%get_chunk_size());

    // 3) ... to the chunks following the first node, with an 
    // offset of 0 bytes into their chunk data.
    while( n_written < n )
    {
        node = node->m_next;
        if( !node )
            return true;

        n_written += node->m_chunk->overwrite(
            (uint8_t *)buf+n_written, n-n_written, 0);
    }
    

    return false;
}


void ChunkedBuffer::clear() throw()
{
    // Delete every node but the first one:
    Node *node = m_read_node->m_next;
    while( node )
    {
        Node *tmp = node->m_next;
        delete node;
        node = tmp;
    }

    m_write_node = m_read_node;
    m_write_node->m_chunk->clear();
    
    m_size = 0;
    m_chunk_count = 1;
}


size_t ChunkedBuffer::get_size() const throw()
{
    return m_size;
}


bool ChunkedBuffer::is_empty() const throw()
{
    return m_size == 0;
}


size_t ChunkedBuffer::get_chunk_size() const throw()
{
    return m_chunk_size;
}


void ChunkedBuffer::grow_on_demand() throw()
{
    if( m_write_node->m_chunk->get_remaining_capacity() == 0 )
    {
        Node *node = new Node(0, m_write_node, new Chunk(m_chunk_size));
        m_write_node->m_next = node;
        m_write_node = node;
        ++m_chunk_count;
    }
}


void ChunkedBuffer::shrink_on_demand() throw()
{
    // Keep removing nodes until we find one that's used (or that 
    // it is the only node):
    while( m_read_node->m_chunk->get_available_count() == 0 &&
           m_write_node != m_read_node )
    {
        Node *tmp = m_read_node->m_next;
        delete m_read_node;
        m_read_node = tmp;
        --m_chunk_count;
    }

    // If there's only one node and it's empty then its chunk 
    // should be cleared.
    if( m_write_node == m_read_node &&
        m_read_node->m_chunk->get_available_count() == 0 )
        m_read_node->m_chunk->clear();
}



// ------------ StreambufAdapter --------------



ChunkedBuffer::StreambufAdapter::StreambufAdapter(ChunkedBuffer &buf)
    : m_buf(buf)
{
}

ChunkedBuffer::StreambufAdapter::int_type 
ChunkedBuffer::StreambufAdapter::overflow(int_type c)
{
    if( c != traits_type::eof() )
        m_buf.write_byte(c);
                
    return c;
}

ChunkedBuffer::StreambufAdapter::int_type 
ChunkedBuffer::StreambufAdapter::underflow()
{
    uint8_t tmp;

    if( m_buf.peek(tmp) )
        return traits_type::eof();
    else
        return (int_type)tmp;
}

ChunkedBuffer::StreambufAdapter::int_type 
ChunkedBuffer::StreambufAdapter::uflow()
{
    if( m_buf.is_empty() )
        return traits_type::eof();
    else
        return m_buf.read_byte();
}

std::streamsize ChunkedBuffer::StreambufAdapter::xsputn(
    const char *s, std::streamsize n)
{
    m_buf.write(s, n);
    return n;
}

std::streamsize ChunkedBuffer::StreambufAdapter::xsgetn(
    char *s, std::streamsize n)
{
    return (std::streamsize)m_buf.read(s, n);
}
