/*
 * Copyright Staffan Gimåker 2009.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef PEEKABOT_ANY_MAP_HH_INCLUDED
#define PEEKABOT_ANY_MAP_HH_INCLUDED


#include <map>
#include <string>
#include <stdexcept>
#include <boost/any.hpp>
#include <boost/function.hpp>
#include <boost/utility/enable_if.hpp>
#include <boost/type_traits.hpp>
#include <boost/mpl/not.hpp>


namespace peekabot
{
    /**
     * \internal
     *
     * \brief A map with heterogeneous value storage (currently through the use
     * of Boost.Any), and support for extrinsic value storage through the use
     * of adapters.
     */
    class AnyMap
    {
        class ValHolder
        {
        public:
            virtual ~ValHolder() {}

            virtual ValHolder *clone() const = 0;

            virtual boost::any get() const = 0;

            virtual void set(const boost::any &val) = 0;
        };

        class IntrinsicVal : public ValHolder
        {
        public:
            IntrinsicVal(const boost::any &val) : m_val(val) {}

            virtual ValHolder *clone() const
            {
                return new IntrinsicVal(m_val);
            }

            virtual boost::any get() const
            {
                return m_val;
            }

            virtual void set(const boost::any &val)
            {
                m_val = val;
            }

        private:
            boost::any m_val;
        };

        template<typename T>
        class Adapter : public ValHolder
        {
        public:
            Adapter(
                boost::function<T ()> getter,
                boost::function<void (const T &)> setter)
                : m_getter(getter), m_setter(setter) {}

            virtual ValHolder *clone() const
            {
                return new Adapter<T>(m_getter, m_setter);
            }

            virtual boost::any get() const
            {
                return wrap<T>(m_getter());
            }

            virtual void set(const boost::any &val)
            {
                if( m_setter )
                    m_setter(unwrap<T>(val));
                else
                    throw std::runtime_error(
                        "The (adapted) value is read only");
            }

        private:
            template<typename U>
            static inline typename boost::enable_if<
                boost::mpl::not_<boost::is_same<U, boost::any> >,
                boost::any>::type wrap(const U &x)
            {
                return boost::any(x);
            }

            template<typename U>
            static inline typename boost::enable_if<
                boost::is_same<U, boost::any>,
                boost::any>::type wrap(const boost::any &x)
            {
                return x;
            }

            template<typename U>
            static inline typename boost::enable_if<
                boost::mpl::not_<boost::is_same<U, boost::any> >,
                U>::type unwrap(const boost::any &x)
            {
                return boost::any_cast<U>(x);
            }

            template<typename U>
            static inline typename boost::enable_if<
                boost::is_same<U, boost::any>,
                boost::any>::type unwrap(const boost::any &x)
            {
                return x;
            }

        private:
            boost::function<T (void)> m_getter;
            boost::function<void (const T &)> m_setter;
        };

    public:
        typedef std::string KeyType;

        class NoSuchKey : public std::runtime_error
        {
        public:
            NoSuchKey(const std::string &msg) : std::runtime_error(msg) {}
        };

        class TypeMismatch : public std::runtime_error
        {
        public:
            TypeMismatch(const std::string &msg) : std::runtime_error(msg) {}
        };

        // ---

        AnyMap() {}

        AnyMap(const AnyMap &other)
        {
            for( KVMap::const_iterator it = other.m_kv_map.begin();
                 it != other.m_kv_map.end(); ++it )
            {
                m_kv_map.insert(std::make_pair(it->first, it->second->clone()));
            }
        }

        ~AnyMap()
        {
            clear();
        }

        AnyMap &operator=(const AnyMap &other)
        {
            clear();

            for( KVMap::const_iterator it = other.m_kv_map.begin();
                 it != other.m_kv_map.end(); ++it )
            {
                m_kv_map.insert(std::make_pair(it->first, it->second->clone()));
            }

            return *this;
        }

        template<typename T>
        T get(const KeyType &key) const
        {
            KVMap::const_iterator it = m_kv_map.find(key);
            if( it == m_kv_map.end() )
                throw NoSuchKey("No such key");
            else
            {
                try
                {
                    return boost::any_cast<T>(it->second->get());
                }
                catch(boost::bad_any_cast &)
                {
                    throw TypeMismatch("Value type mismatch");
                }
            }
        }

        template<typename T> inline
        void set(const KeyType &key, const T &val)
        {
            set(key, boost::any(val));
        }

        void set(const KeyType &key, const boost::any &val)
        {
            KVMap::iterator it = m_kv_map.find(key);
            if( it == m_kv_map.end() )
                insert(key, val);
            else
                it->second->set(boost::any(val));
        }

        void erase(const KeyType &key)
        {
            KVMap::iterator it = m_kv_map.find(key);
            if( it == m_kv_map.end() )
                throw std::runtime_error("No such key");
            else
            {
                delete it->second;
                m_kv_map.erase(it);
            }
        }

        void clear()
        {
            for( KVMap::iterator it = m_kv_map.begin();
                 it != m_kv_map.end(); ++it )
            {
                delete it->second;
            }
            m_kv_map.clear();
        }

        inline std::size_t count(const KeyType &key) const
        {
            return m_kv_map.count(key);
        }

        template<typename T>
        bool has_type(const KeyType &key) const
        {
            KVMap::const_iterator it = m_kv_map.find(key);
            if( it == m_kv_map.end() )
                return false;
            else
            {
                try
                {
                    boost::any_cast<T>(it->second->get());
                    return true;
                }
                catch(...)
                {
                    return false;
                }
            }
        }

        inline void insert(const KeyType &key, const boost::any &val)
        {
            ValHolder *p = new IntrinsicVal(val);
            insert(key, p);
        }

        template<typename T> inline
        void insert(const KeyType &key, const T &val)
        {
            insert(key, boost::any(val));
        }

        template<typename T> inline
        void insert_adapter(
            const KeyType &key,
            boost::function<T ()> getter,
            boost::function<void (const T &)> setter =
                boost::function<void (const T &)>())
        {
            ValHolder *p = new Adapter<T>(getter, setter);
            insert(key, p);
        }

    private:
        void insert(const KeyType &key, ValHolder *p) throw()
        {
            KVMap::iterator it = m_kv_map.find(key);
            if( it != m_kv_map.end() )
            {
                delete it->second;
                it->second = p;
            }
            else
            {
                m_kv_map.insert(std::make_pair(key, p));
            }
        }

    private:
        typedef std::map<KeyType, ValHolder *> KVMap;

        KVMap m_kv_map;
    };
}


#endif // PEEKABOT_ANY_MAP_HH_INCLUDED
