/*
 * Copyright Staffan Gimåker 2009-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "BoSLAM.hh"


BoSLAM::BoSLAM(double sigma1, double sigma2, double sigma3)
    : m_filter(sigma1, sigma2, sigma3),
      m_curr_x(0), m_curr_y(0), m_curr_theta(0),
      m_curr_odom_x(0), m_curr_odom_y(0), m_curr_odom_theta(0)
{
    m_client.connect("localhost");

    m_grp.add(m_client, "bo-slam", peekabot::REPLACE_ON_CONFLICT);
    m_grp_est.add(m_grp, "estimated");
    m_grp_odom.add(m_grp, "odometry");

    m_path_est.add(m_grp_est, "path");
    m_path_est.set_color(0,0,1);
    m_path_odom.add(m_grp_odom, "path");
    m_path_odom.set_color(0,0.7,0);

    // Add robot model
    m_robot.add(m_grp_est, "robot");
    peekabot::ModelProxy chassis;
    m_client.upload_file("../data/pioneer_3dx.pbmf");
    chassis.add(m_robot, "chassis", "../data/pioneer_3dx.pbmf");
    chassis.set_opacity(0.2);
    peekabot::CylinderProxy sensor;
    sensor.add(chassis, "sensor");
    sensor.set_scale(0.03, 0.03, 0.13);
    sensor.set_position(0,0,0.27);
    sensor.set_color(0,0,0);

    m_pose_cov.add(m_robot, "2x std");
    m_pose_cov.set_color(0,0,0);

    m_obs.add(sensor, "observations");
    m_obs.set_color(0,0,0);
    m_obs.set_line_style("dashed");
}


void BoSLAM::update(
    double ds, double dtheta,
    std::vector<BearingOnlySLAM2D::Observation> &obs)
{
    // Update filter
    ublas::matrix<double> pose_cov;
    m_filter.update(ds, dtheta, obs);
    m_filter.get_pose(m_curr_x, m_curr_y, m_curr_theta, pose_cov);

    // Update odometry
    m_curr_odom_x += ds*cos(m_curr_odom_theta);
    m_curr_odom_y += ds*sin(m_curr_odom_theta);
    m_curr_odom_theta += dtheta;

    // Update visualization
    peekabot::Status s = m_client.noop().status();
    m_client.begin_bundle();
    update_visualization(obs, pose_cov);
    m_client.end_bundle();
    s.wait_until_completed();
}


void BoSLAM::update_visualization(
    const std::vector<BearingOnlySLAM2D::Observation> &obs,
    const ublas::matrix<double> &pose_cov)
{
    m_robot.set_pose(m_curr_x, m_curr_y, 0, m_curr_theta);
    update_cov2d(m_pose_cov, pose_cov);

    peekabot::VertexSet vs;
    vs.add_vertex(m_curr_x, m_curr_y, 0);
    m_path_est.add_vertices(vs);
    vs.clear();
    vs.add_vertex(m_curr_odom_x, m_curr_odom_y, 0);
    m_path_odom.add_vertices(vs);

    m_obs.clear_vertices();
    for( std::size_t i = 0; i < obs.size(); ++i )
    {
        peekabot::VertexSet vs;
        vs.add_vertex(0, 0, 0);
        vs.add_vertex(20*cos(obs[i].m_bearing), 20*sin(obs[i].m_bearing), 0);
        m_obs.add_vertices(vs);
    }


    for( std::size_t i = 0; i < m_filter.feature_count(); ++i )
    {
        double x, y;
        ublas::matrix<double> xy_cov;

        m_filter.get_cartesian_feature(i, x, y, xy_cov);

        if( m_feats.count(i) == 0 )
        {
            peekabot::CircleProxy cp;
            cp.add(m_grp_est, "feature");
            cp.set_color(0,0,0);
            m_feats[i] = cp;

            peekabot::CylinderProxy cyl;
            cyl.add(cp, "cyl");
            cyl.set_opacity(0.3);
            cyl.set_position(0,0,0.5);
            cyl.set_scale(0.1, 0.1, 1);
            cyl.set_color(0,0,1);
        }

        m_feats[i].set_position(x, y, 0);
        update_cov2d(m_feats[i], xy_cov);
    }
}


void BoSLAM::update_cov2d(
    peekabot::CircleProxy &p,
    const ublas::matrix<double> &C)
{
    double a = C(0,0);
    double b = C(0,1);
    double c = C(1,0);
    double d = C(1,1);

    double tmp = sqrt((a+d)*(a+d)/4 - a*d + b*c);
    double lambda1 = (a+d)/2 + tmp;
    double lambda2 = (a+d)/2 - tmp;

    double v_x, v_y;
    if( c != 0 )
    {
        v_x = lambda1-d;
        v_y = c;
    }
    else if( b != 0 )
    {
        v_x = b;
        v_y = lambda1-a;
    }
    else
    {
        v_x = 1;
        v_y = 0;
    }

    p.set_scale(2*sqrt(lambda1), 2*sqrt(lambda2), 1);
    p.set_rotation(atan2(v_y, v_x));
}
