/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

namespace peekabot {

/**

\page manual_scenedef Scene files

\li \subpage manual_scenedef_overview
\li \subpage manual_scenedef_reference
\li \subpage manual_scenedef_advanced

*/



/**

\page manual_scenedef_overview Overview and motivation

\section manual_scenedef_overview_intro Introduction

Scene files allow you to define geometry (and more) which can be loaded into 
peekabot, without writing any code! Scene files are specified in 
human-readable XML mark-up and have a potential number of benefits over doing
everything from your client(s):

\li Provides a good way of interoperating with existing datasets stored in 
similar formats.
\li Facilitates easy reuse and sharing.
\li The visual appearance of robots/environments/etc. can changed without
changing the client.
\li Use can use the same scene file with clients written in any language (if 
the peekabot client library is ported to other languages in the future, that is)

Typical usage example for scene files are:

\li Defining the visual appearance and joints of a robotic arm.
\li Defining a reference map of the environment the robot is operating in.

A scene file contains elements for instantiating objects and their properties.
The general layout of a scene file is:

\verbatim
<?xml version="1.0"?> <!-- required -->
<scene> <!-- required -->
  <object_type1>
    <object_prop1>...</object_prop1>
    <object_prop2>...</object_prop2>

    <children>
      <object_type2>...</object_type2>
    </children>
  </object_type1>
</scene>
\endverbatim

Of course, there are plenty of exceptions to this general rule of thumb. 
For example, layer and color is specified by enclosing an object (or indeed
several objects) in a <tt><layer></tt> or <tt><color></tt> tag, e.g.:

\verbatim
<?xml version="1.0"?>
<scene>
  <color r="0" g="0" b="1">
    <sphere/>
    <cube/>
  </color>
</scene>
\endverbatim

Object instantiations, layer and color aside, there is a handful other elements
that can be used - which are discussed in the section 
\ref manual_scenedef_advanced.


\section manual_scenedef_overview_example An example - defining a pan-tilt unit

The example below shows a real-life example of what a definition file might look
like. It combines three models into a complete pan-tilt unit with 2 rotational
DOFs that can be controlled from clients.

\verbatim
<?xml version="1.0"?>
<scene>
  <model>
    <!-- file is specific to the model object, and simply defines 
         which model to load -->
    <file>dp-ptu-d46/bottom.pbmf</file>
    <!-- set the name of the object to "ptu_bottom" -->
    <name>ptu_bottom</name>
  </model>

  <hinge>
    <name>pan</name>
    <!-- set the hinge's rotational axis -->
    <axis>0 0 1</axis>
    <children>
      <!-- add a few children to the hinge... -->
      <model>
        <file>dp-ptu-d46/middle.pbmf</file>
        <name>ptu_middle</name>
      </model>

      <hinge>
        <name>tilt</name>
        <axis>0 1 0</axis>

        <transform>
          <!-- move the hinge 9.11 cm along the Z-axis -->
          <translate>0 0 0.0911</translate>
        </transform>

        <children>
	      <model>
	        <file>dp-ptu-d46/top.pbmf</file>
	        <name>ptu_top</name>
          </model>
        </children>
      </hinge>
    </children>
  </hinge>
</scene>
\endverbatim

*/



/**

\page manual_scenedef_reference Syntax reference

\section manual_scenedef_layer_and_color Layer and color

The layer and color elements may be contained directly within the 
<tt><scene></tt> element, within a scene object element, or inside each other 
(color inside layer, or layer inside color):

\par
\verbatim<layer number="x"> ... </layer>\endverbatim
\par
All objects declared inside this element will be added to 
layer \f$x \in \{1,2, \dots, 16\}\f$, unless overriden by another layer 
element. The default layer, used when no layer tag is present, is layer 1.

\par
\verbatim<color r="R" g="G" b="B"> ... </color>\endverbatim
\par
All objects declared inside this element will have the color specified by the
given RGB triplet. \f$R,G,B \in [0,1]\f$.


\section manual_scenedef_external External

See \ref manual_scenedef_advanced_external for more information about 
\c external.

\par Syntax
\verbatim
...
<external>
  <!-- any well-formed XML -->
</external>
\endverbatim


\section manual_scenedef_include Include

\par Syntax
\verbatim
...
<include file="source_file">
  <inject at="injection_path"> <!-- optional -->
    <!-- any number of object definitions, includes or external tags are allowed here -->
  </inject>
  ...
  <inject at="injection_path">
    ...
  </inject>
</include>
\endverbatim

The inject tags are optional. Aside from the inject tag, external tags are also
allowed inside \c include elements.

For more information about include injection, see 
\ref manual_scenedef_advanced_inject.


\section manual_scenedef_common Properties common to all objects

The definition of an object of any type can contain the following tags, 
which must come in the order they are specified here. 
<em>None of these elements are required.</em>

\par
\verbatim<name> ... </name>\endverbatim
\par
Specifies the name of the object. Names may contain all characters except 
periods. Remember that object names must be unique among siblings.
\par
If no name is explicitly specified the object will be named according to 
its type and enumerated as needed to preserve name uniqueness among its 
siblings. For example, with this scheme, a <tt><sphere></tt>-element might
yield an object named \c sphere5.

\par
\verbatim<meta_name> ... </meta_name>\endverbatim
\par
Specifies the meta name of the object - a more descriptional name used only
for informational purposes.

\par
\verbatim<opacity absolute="x">a</opacity>\endverbatim
\par
Specifies the opacity of an object. \c x must be \c true or \c false, where 
\c false is the default. \c a is a number in the interval \f$[0, 1]\f$
specifying the object's absolute opacity, or its opacity relative its parent
node.

\par
\verbatim<hidden/>\endverbatim
\par
If this tag is present, the object and all its children will be hidden by 
default.

\par
\verbatim<transform> ... </transform>\endverbatim
\par
Specifies any transforms that are to be applied to the object -
more on this later.

\par
\verbatim<children> ... </children>\endverbatim
\par
May contain any number of scene object (or derived types), color or layer 
elements. The objects defined in this element will be the children of the 
current object.

\subsection manual_scenedef_transform Position and orientation

Inside the transform element, any of the following elements may appear in any
order (they will be applied in the order they appear):

\par
\verbatim<translate system="s"> x y z </translate>\endverbatim
\par
Moves the object by the vector (x, y, z) along to the axes of coordinate 
system \a s, which can be \c parent, \c world (the root of the scene file) 
or \c local, where \c world is the default. 

\par
\verbatim<rotate> ... </rotate>\endverbatim
\par
Rotates the object as specified by the subelements:

\par
\verbatim<axis system="s"> x y z </axis>\endverbatim
\par
The vector (x, y, z) specifies the axis of rotation, and may be specified in 
different coordinate systems as with the <tt><translate></tt> element.
The default value is (1, 0, 0) and world coordinates.

\par
\verbatim<pivot system="s"> x y z </pivot>\endverbatim
\par
Specifies the pivot point around which to rotate the object. The system 's' is specified as with the <translate> tag.
The default is (0, 0, 0) and local coordinates.
\par
\verbatim<radians> r </radians> or <degrees> d </degrees>\endverbatim
\par
Specifies magnitude of rotation in either radians or degrees.

\par Example
\verbatim<transform>
  <translate>0 0 1.2</translate>
  <rotate>
    <axis>0 1 0</axis>
    <degrees>45</degrees>
  </rotate>
</transform>\endverbatim

The matrix element, specified with the syntax given below, loads a 
transformation matrix, describing the object's transformation in the given
coordinate system \a s.

\par
\verbatim
<matrix system="s">
  x1 x2 x3 x4
  y1 y2 y3 y4
  z1 z2 z3 z4
  0   0  0  1
</matrix>
\endverbatim
\par
\a s may be either \c world or \c parent, where the latter is the default. 
The first, second and third column vectors represent the x-, y- and z-axis 
respectively. The fourth column vector is the translation relative the 
coordinate system's origin.

\par
The upper left 3x3 submatrix must specify an orthonormal basis of \f$R^3\f$.

\section manual_scenedef_scale Scale

Most objects (geometric primitives, mostly) allow adjusting the scale along 
the local axes using the <tt><scale></tt>-element. 

\par
\verbatim<scale>x y z</scale>\endverbatim
\par
Where x, y and z are the scale factors for the different axes. 


\section manual_scenedef_line_based Line parameters

For primitives comprised of lines (line clouds, grids, etc.) the
width and the pattern of the lines can be changed.

The line style parameters are optional and the default is solid lines of 
width one.

\par Syntax
\verbatim<line_width>w</line_width>
<line_style factor="x">solid|dashed|dotted|dash-dot|dash-dot-dot</line_style> <!-- x = 1,2,... -->
\endverbatim
\par
Where w is a floating point number and x is an integer describing how often
the line pattern repeats.


\section manual_scenedef_camera Cameras

Camera object declarations may contain the following properties in 
addition to those common to all scene objets:

\par
\verbatim<orthographic/>\endverbatim
\par
If present, the view mode of the camera is set to orthographic 
(perspective mode is the default).

\par
\verbatim<fov>
  <degrees>d</degrees>
  OR
  <radians>d</radians>
</fov>
\endverbatim
\par
Sets the vertical field of view of the camera to d degrees or radians.
d must be in the range \f$(0,180)\f$ degrees.

\section manual_scenedef_group Groups

Prodives a way to group similar objects without adding new graphical 
elements to the scene.

\par Example
\verbatim<group>
  ... 
  <children>...</children>
</group>\endverbatim


\section manual_scenedef_primitives1 Spheres, cylinders, cubes and circles

The \c sphere, \c cylinder, \c cube and \c circle elements are all 
\ref manual_scenedef_scale "scalable" objects. Except for the optional scale 
parameters, neither of them add any extra markup.

\par Example
\verbatim<sphere>
  <children>
    <cylinder>
      <scale>0.5 0.5 2</scale> <!-- diameter=1, height=2 -->
    </cylinder>
  </children>
</sphere>\endverbatim



\section manual_scenedef_primitives2 Polygons, polylines, point clouds and line clouds

The \c point-cloud, \c line-cloud, \c polyline and \c polygon objects share a
few important traits: they're are all \ref manual_scenedef_scale "scalable"
and their geometry is defined by vertices.

The \c <vertices> element is used to specify which vertices the object is
comprised of:

\verbatim<vertices>
  <vertex>x1 y1 z1</vertex>
  <vertex>x2 y2 z2</vertex>
  ...
</vertices>\endverbatim

N vertices yields N points for a point cloud, N/2 lines a line cloud,
an open polygonal chain with N-1 lines for a polyline and a closed polygon
with N sides for a polygon.

\par Vertex order
For polygons, vertices must be specified in counterclockwise order.


\section manual_scenedef_tri_mesh Tri-meshes

Tri-meshes are specified through a set of vertices, just like polygons and
line clouds, but also require a sequence of index triplets which define which
vertices that should form triangles. The syntax for specifying the vertices is
the same as for \ref manual_scenedef_primitives2 "polygons".

The \c <triangles> element is used to specify which vertices to connect
together into triangles:

\verbatim<triangles>
  <vertex>index-1 index-2 index-3</vertex>
  <vertex>index-1 index-2 index-3</vertex>
  ...
</triangles>\endverbatim

The indices are zero-indexed.

\par Polygon winding
The triangles are specified in counterclockwise order.



\section manual_scenedef_model Models

Models are \ref manual_scenedef_scale "scalable" and require one additional 
element to be present:

\par
\verbatim<file> ... </file>\endverbatim
\par
The \c <file> tag specifies which model file to use. See 
\ref manual_tutorial_dot_peekabot "the tutorial" for information on which 
locations peekabot searches for models in. See the section on 
\ref manual_model_conversion "model conversion" for information on which model 
formats that are supported by peekabot.

\section manual_scenedef_grid Grids

\par Syntax
\verbatim<grid>
  ...
  <type> regular|radial|angular </type> <!-- optional, default: regular -->
  <segments> X </segments> <!-- required, X must be an integer larger than 0 -->
  <central_angle> <!-- optional, default: 360 degrees -->
    <degrees>Y</degrees>
    OR
    <radians>Y</radians>
  </central_angle>
</grid>\endverbatim

The grid type is specified with the \c <type> element and defaults to 
\c regular if the type element is left out.

Common to all grid types is that the number of segements in the grid must be
specified somehow, more specifically, using the \c <segments> element.
A regular grid with 10 segments will have 10*10 = 100 grid elements and a 
radial grid with 10 segments will consist of 10 concentric circular segments.

The optional \c <central_angle> element determines how far radial and 
angular grids extend - it has no effect on regular grids. If left out, it 
defaults to 360 degrees.

\par Examples
\verbatim<!-- A 10x10 m cartesian grid with a resolution of 10 cm -->
<grid>
  <scale>0.1 0.1 0.1</scale>
  <segments>100</segments>
</grid>\endverbatim
\par
\verbatim<!-- An 8 m radius radial grid with a resolution of 1 m -->
<grid>
  <type>radial</type>
  <segments>8</segments>
</grid>\endverbatim
\par
\verbatim<!-- An 8 m radius angular, 180 degree grid with 10 degree resolution -->
<grid>
  <scale>8 8 8</scale>
  <type>angular</type>
  <segments>18</segments>
  <central_angle>
    <degrees>180</degrees>
  </central_angle>
</grid>\endverbatim



\section manual_scenedef_label Labels

\par Syntax
\verbatim<label>
  ...
  <text>label text</text> <!-- optional -->
  <align>left | right | center</align> <!-- optional -->
</label>\endverbatim

By default, labels are left-aligned and empty.


\section manual_scenedef_joints Joints

All joint types have a few common properties:
\li A minimum joint value, specified with the \c <min> element.
\li A maximum joint value, specified with the \c <max> element.
\li A default value, that's set when the joint has been parsed. The default value 
is specified with the \c <initial> element.
\li An offset value that specifies the zero configuration relative the
configuration in the definition file, specified with the \c <offset> element.

The joint value offset should be used if the configuration of the joint
in the definition file is not the zero configuration. E.g. if the configuration 
of a 3-DOF arm in the definition file corresponds to "0, -90, 0" the offset
should be set to 0, -90 and 0 for the three joints respectively.


\subsection manual_scenedef_hinge Hinges

Hinges are a rotational joints that result in rotation in the plane 
perpendicular to the given rotational axis. All rotations are done about the 
specified pivot point.

\par Syntax
\verbatim<hinge>
  ...
  <min>
    <radians>a</radians> or <degrees>a</degrees>
  </min>
  <max>
    <radians>b</radians> or <degrees>b</degrees>
  </max>
  <initial>
    <radians>c</radians> or <degrees>c</degrees>
  </initial>
  <offset>
    <radians>c</radians> or <degrees>c</degrees>
  </offset>

  <axis system="s">
    x y z
  </axis>

  <pivot system="s">
    x y z
  </pivot>
</hinge>\endverbatim
\par
Where \a s is the coordinate system the point/vector is given in. \a s can 
be any of \c local, \c parent and \c world.

The order of the elements is not relevant and all elements are optional.
The defaults are listed below:
\li \e min: -inf
\li \e max: inf
\li \e initial: 0
\li \e offset: 0
\li \e axis: (0, 0, 1) in parent coordinates
\li \e pivot: (0, 0, 0) in parent coordinates


\subsection manual_scenedef_slider Sliders

Sliders (also commonly called prismatic joints) provide translational DOFs,
that move all child objects along the axis specified by the \c <axis> element.

\par Syntax
\verbatim<slider>
  ...
  <min>a</min>
  <max>b</max>
  <initial>c</initial>
  <offset>c</offset>

  <axis system="s">
    x y z
  </axis>
</slider>\endverbatim
\par
Where \a s is the coordinate system the point/vector is given in. \a s can 
be any of \c local, \c parent and \c world.

The order of the elements is not relevant and all elements are optional.
The defaults are listed below:
\li \e min: -inf
\li \e max: inf
\li \e initial: 0
\li \e offset: 0
\li \e axis: (0, 0, 1) in parent coordinates


*/



/**

\page manual_scenedef_advanced Advanced topics

\section manual_scenedef_advanced_inject Injecting data in included files

A typical case is that you have:
\li One scene file defining a pan-tilt unit
\li Another scene file defining a stereo camera

And you want to combine the two, i.e. put the camera on the pan-tilt unit. One 
way to achieve the desired result is to just copy-paste the contents of 
the stereo camera defintion into the pan-tilt definition file. This works but 
is less than ideal if you have several different types of appliances you use on 
the same pan-tilt unit model. Not to mention that the files become messy rather 
quickly.

To facilitate a more modular approach, peekabot allows injection of objects 
into included subtrees.
Assume we want the stereo camera (defined in "sth.xml") added under the 
"pan.tilt" object from "ptu.xml". This is simple with <tt><inject></tt> 
element:

\verbatim<?xml version="1.0"?>
<scene>
  <include file="ptu.xml">
    <inject at="pan.tilt">
      <include file="sth.xml"/>
    </inject>
  </include>
</scene>\endverbatim

Of course, if you need to inject objects at several places can use several
<tt><inject></tt> elements.


\section manual_scenedef_advanced_external External blocks

To support interoperability with other tools and applications, scene
files can contain <tt><external></tt> elements that are ignored by peekabot. 
Although markup contained in an external-block is ignored by peekabot, its 
contents still has to be well-formed.

External blocks are generally allowed everywhere in scene files - in the top 
scope of scene files, in color elements, in layer elements and object 
definition elements (e.g. \c <sphere>). They are not, however, allowed e.g. 
inside <tt><scale></tt>-elements.

A typical usage scenario where <tt><external></tt> blocks is storing non-peekabot
data in the same file, e.g. information used by a simulator simulating the
specified world.

\par Example
\verbatim<scene>
  <external>
    <g>9.82</g>
    <drag_coeff>2</drag_coeff>
    <disable_collisions/>
  </external>
  ...
</scene>\endverbatim

*/

}
