<?php
/**
 * FIT Example Music
 * 
 * $Id: MusicLibrary.php 12 2007-08-12 18:34:25Z gerd $
 * 
 * @author gERD Schaufelberger <gerd@php-tools.net>
 * @package FIT
 * @subpackage Fixture
 * @license LGPL http://www.gnu.org/copyleft/lesser.html
 */

require_once dirname( __FILE__ ) . '/Music.php';

/**
 * FIT Music library
 * 
 * Example music library loader
 * 
 * @version 0.1.0
 * @package FIT
 * @subpackage example
 */
class MusicLibrary 
{
   /**
    * loaded library
    * @var array
    */
    private static $library  =   array();
    
   /**
    * selected entries
    * @var array
    */
    private static $selected =   array();
    
   /**
    * the entry to be displayed
    * @var object
    */
    private static $looking  =   null;
		
   /**
    * load music database 
    * 
    * Database is a tab-separated list (CSV-file)
    * 
    * @param string $path to file
    */      
	public static function load( $path ) 
    {
		$fp =   fopen( dirname( __FILE__ ) .'/' . $path, 'r' );
		
        // skip column headings
		fgets( $fp, 4096 ); 
		
		while( !feof( $fp ) ) {
            $entry              =   fgetcsv( $fp, 4095, "\t" );
			self::$library[]    =   self::map( $entry );
		}
		fclose( $fp );
        return true;      
	}
	    
   /**
    * map indexed to associative array
    * 
    * The source list is an indexed array with 10 fields:
    * title, artist, artist, genre, size, secondss, trackNumber, trackCount, 
    * year and date. Those fields will be mapped a corresponding associative
    * array. 
    * 
    * @param array $list indexed array 
    * @return array associative array
    */      
    private static function map( $list ) 
    {
        $map    =   array(
            'title',
            'artist',
            'album',
            'genre',
            'size',
            'seconds',
            'trackNumber',
            'trackCount',
            'year',
            'date'
        );
        
        $a  =   array();
        for( $i = 0; $i < count( $map ); ++$i ) {
        
            // use dummy entry - failsafe
            if( !isset( $list[$i] ) ) {
                $list[$i]  =   '';
            }
            
            $a[$map[$i]]    =   $list[$i];
        }

        return $a;
    }
    
   /**
    * select a song
    * 
    * @param int $pos
    * @param bool true on success
    */
    static function select( $pos ) 
    {
        if( $pos < 1 || $pos > count( self::$library ) ) {
            return false;
        }
        
        --$pos;
        
        // we are currently looking at this one
        self::$looking = $pos;
        
        // add this one to selected items
        if( !in_array( $pos, self::$selected ) ) {
            self::$selected[]   =   $pos;
        }
        return true;
    }
    
   /**
    * receive current song's title
    * 
    * @param string song's title
    */
    public function title() 
    {
        return self::$library[self::$looking]['title'];
    }
    
   /**
    * receive current song's artist
    * 
    * @param string song's artist
    */
    public function artist() 
    {
        return self::$library[self::$looking]['artist'];
    }
    
   /**
    * receive current song's album
    * 
    * @param string song's album
    */
    public function album() 
    {
        return self::$library[self::$looking]['album'];
    }
    
   /**
    * receive current song's year
    * 
    * @param string song's year
    */
    public function year() 
    {
        return self::$library[self::$looking]['year'];
    }
    
   /**
    * receive current song's genre
    * 
    * @param string song's genre
    */
    public function genre() 
    {
        return self::$library[self::$looking]['genre'];
    }
    
   /**
    * receive current song's size
    * 
    * @param int song's size
    */
    public function size() 
    {
        return self::$library[self::$looking]['size'];
    }
    
   /**
    * receive track number as shown on display
    * 
    * e.g. "5 of 10"
    * @return string 
    */
    public function track() 
    {
        $result =   self::$library[self::$looking]['trackNumber'] 
               .    ' of ' 
               .    self::$library[self::$looking]['trackCount'];
               
        return $result;
    }

   /**
    * plaing time
    * 
    * @return float
    */
    public function time() 
    {
        $sec    =   self::$library[self::$looking]['seconds'];
        return round( $sec / 0.6 ) / 100.0;
    }    
    
   /**
    * count entries in library
    * 
    * @return int
    */
    static function count() 
    {
        return count( self::$library );
    }
    
   /**
    * list of content
    * 
    * @return array $display music items
    */
    static function getSelection() 
    {
        $look       =   self::$looking;
        $display    =   array();
        $i          =   0;
        foreach( self::$selected as $sel ) {
            self::$looking          =   $sel;
            
            $display[$i]            =   self::$library[$sel];
            $display[$i]['time']    =   self::time();
            $display[$i]['track']   =   self::track();
            
            ++$i;
        }
        
        self::$looking  =   $look;
        self::$selected =   array();
        return $display;
    }
}
?>