<?php
/**
* OPEN API LICENSE
*
* Copyright (c) 2007 SharedBook.  All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification,are permitted provided that the following conditions are met:
*
* 1. Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the disclaimer below.
* 2. Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the disclaimer below in the
*    documentation and/or other materials provided with the distribution.
* 3. The name of SharedBook Ltd., SharedBook Inc., or the names of its
*    contributor may not be used to endorse or promote products derived from
*    this software without specific prior written permission
*
* Disclaimer
* THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR IMPLIED
* WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
* AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SHAREDBOOK LTD
* AND SHAREDBOOK INC. BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
* EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
* PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
* OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
* WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.
*
*
* Services_SharedBook is the main class that represents the book.
* Full documentation is found on the SharedBook site:
* http://www.sharedbook.com/biz/dev/index.html
*
* PHP version 5
*
* @category Services
* @package  Services_SharedBook
* @author   Alex Lavrov <sharedbook.php@gmail.com>
* @license  http://www.php.net/license/3_0.txt  LGPL License
* @link     not yet
*
*/

require_once 'Services/SharedBook/Exception.php';
require_once 'HTTP/Client.php';

/**
* Services_SharedBook is the main class that represents the book.
* Full documentation is found on the SharedBook site:
* http://www.sharedbook.com/biz/dev/index.html
*
* PHP version 5
*
* @category Services
* @package  Services_SharedBook
* @author   Alex Lavrov <sharedbook.php@gmail.com>
* @license  http://www.php.net/license/3_0.txt  LGPL License 3.0
* @link     not yet
*
*/
class Services_SharedBook
{
    /**
     * Class constants
     */
    const WRAPPER_ERROR                = 2;
    const OPEN_API_URL                 = 'http://api.sharedbook.com/v0.6/';
    const METHOD_AUTH_LOGIN            = 'auth/login';
    const METHOD_SESSION_TOKEN         = 'auth/getSessionToken';
    const METHOD_BMSCREATE_INIT        = 'bmscreate/init';
    const METHOD_BMSCREATE_PUBLISH     = 'bmscreate/publish';
    const METHOD_BMS_ADD_COMMENT       = 'bms/addComment';
    const METHOD_BMS_ADD_PHOTO         = 'bms/addPhoto';
    const METHOD_PHOTO_UPLOAD          = 'photo/upload';
    const METHOD_BMS_FRONT_PHOTO       = 'bms/setFrontCoverPhoto';
    const METHOD_BMS_BACK_PHOTO        = 'bms/setBackCoverPhoto';
    const METHOD_BMS_PUBLISH           = 'bms/publish';
    const METHOD_BOOKCREATE_INIT       = 'bookcreate/init';
    const METHOD_BOOKCREATE_DEDICATION = 'bookcreate/setDedication';
    const METHOD_BOOKCREATE_SUBTITLE   = 'bookcreate/setSubtitle';
    const METHOD_BOOKCREATE_COVERTHEME = 'bookcreate/setCoverTheme';
    const METHOD_BOOKCREATE_PUBLISH    = 'bookcreate/publish';
    const METHOD_BOOK_PREVIEW          = 'book/preview';

    /**#@+
    * @access private
    */
    /**
    * Api key of the registered client
    * @var string   contains API key of the client
    */
    private $apiKey;

    /**
    * Secret word
    * @var string   contains secret word of the client
    */
    private $secretWord;

    /**
    * Authentication token
    * @var string   contains authentication token
    */
    private $authToken;

    /**
    * Session token
    * @var string   contains session token
    */
    private $sessionToken;

    /**
    * Book making space ID
    * @var string   contains id of the book making space
    */
    private $bmsId;

    /**
    * Book ID
    * @var string   contains id of the book
    */
    private $bookId;

    /**
    * Url for the current API version
    * @var string   contains url of the open api methods
    */
    private $openApiUrl;
    
    /**
    * Last API response
    * @var string   contains last response of the API
    */
    private $lastAPIResponse;
    /**#@-*/

    /**
     * Constructor of the class
     *
     * @param string $apiKey     Api key of the client.
     * @param string $secretWord Secret word.
     * @param string $openApiUrl Open API url, can be changed by user.
     */
    public function __construct($apiKey, $secretWord,
                                $openApiUrl = self::OPEN_API_URL)
    {
        $this->apiKey          = $apiKey;
        $this->secretWord      = $secretWord;
        $this->authToken       = null;
        $this->sessionToken    = null;
        $this->bmsId           = null;
        $this->bookId          = null;
        $this->openApiUrl      = $openApiUrl;
        $this->lastAPIResponse = array();
    }
    
    /**
     * Sets the members manualy in the case session already exists 
     * on the server
     *
     * @param array  $parametersArray Array with all the parameters
     *
     * @access public
     *
     * @return none
     */
    public function setMembers($parametersArray)
    {
        if (!empty($parametersArray['apiKey']))       $this->apiKey       = $parametersArray['apiKey'];
        if (!empty($parametersArray['secretWord']))   $this->secretWord   = $parametersArray['secretWord'];
        if (!empty($parametersArray['authToken']))    $this->authToken    = $parametersArray['authToken'];
        if (!empty($parametersArray['sessionToken'])) $this->sessionToken = $parametersArray['sessionToken'];
        if (!empty($parametersArray['bmsId']))        $this->bmsId        = $parametersArray['bmsId'];
        if (!empty($parametersArray['bookId']))       $this->bookId       = $parametersArray['bookId'];
        if (!empty($parametersArray['openApiUrl']))   $this->openApiUrl   = $parametersArray['openApiUrl'];   
    }
    
    /**
     * Gets the members manualy in the case session already exists 
     * on the server
     *
     * @access public
     *
     * @return array $parametersArray Array with all the parameters
     */
    public function getMembers()
    {
        $parametersArray = array();
        
        if (!empty($this->apiKey))       $parametersArray['apiKey']       = $this->apiKey;
        if (!empty($this->secretWord))   $parametersArray['secretWord']   = $this->secretWord;
        if (!empty($this->authToken))    $parametersArray['authToken']    = $this->authToken;
        if (!empty($this->sessionToken)) $parametersArray['sessionToken'] = $this->sessionToken;
        if (!empty($this->bmsId))        $parametersArray['bmsId']        = $this->bmsId;
        if (!empty($this->bookId))       $parametersArray['bookId']       = $this->bookId;
        if (!empty($this->openApiUrl))   $parametersArray['openApiUrl']   = $this->openApiUrl;
        
        return $parametersArray;
    }

    /**
     * Function that parses XML replies from open api.
     *
     * @param string $xml XML reply to be parsed.
     *
     * @return array                         Array of the elements and their value.
     * @throws Services_SharedBook_Exception When status id "fail"
     */
    private function _handleXml($xml)
    {
        $apiAnswer = @simplexml_load_string($xml);

        if (!empty($apiAnswer))
        {
            if ($apiAnswer->attributes()->status == 'fail') {
                throw new Services_SharedBook_Exception($apiAnswer->error->attributes()->msg,
                                                        (string)$apiAnswer->error->attributes()->code);
            } else {
                return $apiAnswer;
            }
        } else {
           return @simplexml_load_string('<dummy>empty</dummy>'); 
        }
    }

    /**
     * Function that performs open api call from SharedBook servers.
     *
     * @param string $method Service and method to be called as 
     *                       defined by the constants
     * @param array  $params Array of parameters for the call.
     * @param string $files  List of the files to be uploaded as 
     *                       required by HTTP_Request.
     *
     * @access private
     *
     * @return array Reply from the open api
     */
    private function _callAPI($method, $params, $files = array())
    {
        $returnVal = array();
        
        // Choose the http method for the current api call
        switch ($method)
        {
        case self::METHOD_SESSION_TOKEN:
        case self::METHOD_BOOK_PREVIEW:
            $httpMethod = 'get';
            break;
        default:
            $httpMethod = 'post';
            break;
        }
        
        // Create new HTTP_Client class
        $callApi = new HTTP_Client();
        
        // No need to follow redirects
        $callApi->setMaxRedirects(0);
        echo $this->openApiUrl.$method."\n";
        
        // Call SharedBook api
        $returnVal['code'] = $callApi->{$httpMethod}($this->openApiUrl.$method, 
                                                     $params, false, $files);
        $returnVal['current'] = $callApi->currentResponse();
        
        // Set last API response
        $this->lastAPIResponse = $returnVal; 
        
        
        // Return code and response body
        return $returnVal;
    }

    /**
     * Function returns last API response
     *
     * @access  public
     *
     * @return string API response
     */
    public function getLastResponse ()
    {
        return $this->lastAPIResponse;
    }
    
    /**
     * Function that calculates signature of the parameters
     *
     * @param array $params Array of parameters
     *
     * @access  private
     *
     * @return string MD5 signature of the parameters
     */
    private function  _calcSignature ($params = array())
    {
        // Sort arguments, concate them and return md5 signature with the secret word
        ksort($params);
        
        $tmpStr = '';
        foreach ($params as $key => $currParam) $tmpStr .= $key.$currParam;

        return md5(utf8_encode($this->secretWord.$tmpStr));
    }


    /****************************************************************/
    /* SERVICE: auth METHOD: login                                  */
    /****************************************************************/

    /**
     * Function calls for auth/login.
     *
     * @access public
     *
     * @return boolen Whether the login succeeded.
     * @throws Services_SharedBook_Exception When invalid auth token, session token
     *                                       returned
     */
    public function authLogin()
    {
        // Set parameters for the API call
        $params = array(
                          'apiKey'   => $this->apiKey
                       );
        
        
        $response = $this->_callApi(self::METHOD_AUTH_LOGIN, $params);
        
        // Check the response code.
        switch ($response['code'])
        {
        // XML response from login is probably an error
        case 200:
            // Parses the XML response and will throw and exception
            $this->_handleXml($response['current']['body']);
            break;
        // Redirect is success
        case 302:
            // Cut authentication token from the response
            $tmpStr = explode('authToken=', 
                              $response['current']['headers']['location']);
                                   
            $this->authToken = trim($tmpStr[1]);
            
            if (!empty($this->authToken)) {
                // Get session token from the open api
                $params['authToken'] = $this->authToken;
        
                $response = $this->_callApi(self::METHOD_SESSION_TOKEN, $params);
        
                $parsedXML = $this->_handleXml($response['current']['body']);
        
                // Check the session token
                if (!empty($parsedXML->sessionToken)) {
                    $this->sessionToken = (string)$parsedXML->sessionToken;
                } else {
                    // Error, invalid session token
                    throw new Services_SharedBook_Exception('Invalid sessionToken', 
                                                            self::WRAPPER_ERROR);
                }
        
            } else {
                // Error, invalid authentication token
                throw new Services_SharedBook_Exception('Invalid authToken', 
                                                        self::WRAPPER_ERROR);
            }

            return true;
            break;
        default:
            // Probably will never happen
            throw new Services_SharedBook_Exception('Invalid response code', 
                                                    self::WRAPPER_ERROR);
        }
        
        return false;
    }


    /****************************************************************/
    /* SERVICE: bmscreate  METHOD: init                             */
    /****************************************************************/

    /**
       * Function calls for bmscreate/init.
       *
       * @param string $bookTitle       Book title
       * @param string $chapterTitle    Chapter title
       * @param string $chapterTtext    Chapter text
       * @param string $theme           Theme for the book (optional)
       * @param string $postCommentLink Comment link (optional)
       * @param string $postPhotoLink   Photo link (optional)
       *
       * @access public
       *
       * @return string ID of book making space
       * @throws Services_SharedBook_Exception When no bms ID returned
       */
    public function bmsCreateInit($bookTitle, $chapterTitle, 
                                  $chapterTtext, $theme = 'sb.theme.vanilla', 
                                  $postCommentLink = null, $postPhotoLink = null)
    {
        // Set parameters for the API call
        $params = array(
                          'apiKey'       => $this->apiKey,
                          'authToken'    => $this->authToken,
                          'sessionToken' => $this->sessionToken,
                          'bookTitle'    => $bookTitle,
                          'chapterText'  => $chapterTtext,
                          'chapterTitle' => $chapterTitle,
                          'theme'        => $theme
                       );
        
       
        // Check if the optional parameters supplied and populate the params array
        if (!empty($postCommentLink)) $params['postCommentLink'] = $postCommentLink;
        if (!empty($postPhotoLink))   $params['postPhotoLink']   = $postPhotoLink;
        
        // Calculate signature
        $params['signature'] = $this->_calcSignature($params);
        
        // Call api
        $response = $this->_callApi(self::METHOD_BMSCREATE_INIT, $params);
        
        // Parse XML reply
        $parsedXML = $this->_handleXml($response['current']['body']);
        
        // Check BMS ID returned
        if (empty($parsedXML->bms->attributes()->id)) {
            throw new Services_SharedBook_Exception('Invalid bms ID', 
                                                    self::WRAPPER_ERROR);
        }
        
        $this->bmsId = (string)$parsedXML->bms->attributes()->id;

        return $this->bmsId;
    }
    
    /****************************************************************/
    /* SERVICE: bmscreate  METHOD: init                             */
    /****************************************************************/

    /**
       * Function calls for bmscreate/init with multiple arcticles support
       *
       * @param string $bookTitle       Book title
       * @param array  $articles        Articles array. Each cell should contain
       *                                (chapterTitle, chapterText, 
       *                                 [postCommentLink], [postPhotoLink])                                 
       * @param string $theme           Theme for the book (optional)
       *
       * @access public
       *
       * @return string ID of book making space
       * @throws Services_SharedBook_Exception When no bms ID returned
       */
    public function bmsCreateInitMultipleArticles($bookTitle, $articles, 
                                                  $theme = 'sb.theme.vanilla', $groups)
    {
        // Set parameters for the API call
        $params = array(
                          'apiKey'       => $this->apiKey,
                          'authToken'    => $this->authToken,
                          'sessionToken' => $this->sessionToken,
                          'bookTitle'    => $bookTitle,
                          'theme'        => $theme
                       );
        
        $counter = 1;
        foreach ($groups as $group)
        {
            $params["groupTitle".$counter] = $group["groupTitle"];
            $params["groupPhotoUrl".$counter] = $group["groupPhotoUrl"];
            ++ $counter;
        }
                       
        $counter = 1;
                 
        foreach ($articles as $article)
        {
            $params['chapterTitle'.$counter] = $article['chapterTitle'];
            $params['chapterText'.$counter]  = $article['chapterText'];
            
            // Check if the optional parameters supplied and populate the params array
            if (!empty($article['postCommentLink'])) 
                $params['postCommentLink'.$counter] = $article['postCommentLink'];
                
            if (!empty($article['postPhotoLink']))   
                $params['postPhotoLink'.$counter]   = $article['postPhotoLink'];
            
            if (!empty($article["chapterGroup"])) 
                $params["chapterGroup".$counter] = $article["chapterGroup"]; 
                
            ++$counter;
        }
        
        // Calculate signature
        $params['signature'] = $this->_calcSignature($params);
        
        // Call api
        $response = $this->_callApi(self::METHOD_BMSCREATE_INIT, $params);
        
        // Parse XML reply
        $parsedXML = $this->_handleXml($response['current']['body']);
        
        // Check BMS ID returned
        if (empty($parsedXML->bms->attributes()->id)) {
            throw new Services_SharedBook_Exception('Invalid bms ID', 
                                                    self::WRAPPER_ERROR);
        }
        
        $this->bmsId = (string)$parsedXML->bms->attributes()->id;

        return $this->bmsId;
    }

    /****************************************************************/
    /* SERVICE: bmscreate  METHOD: publish                          */
    /****************************************************************/

    /**
       * Function calls for bmscreate/publish.
       *
       * @access public
       *
       * @return boolean Wheter bms publishing succeeded
       */
    public function bmsCreatePublish()
    {
        // Set parameters for the API call
        $params = array(
                          'apiKey'       => $this->apiKey,
                          'authToken'    => $this->authToken,
                          'sessionToken' => $this->sessionToken,
                          'bmsId'        => $this->bmsId
                       );
        
        // Calculate signature
        $params['signature'] = $this->_calcSignature($params);
        
        // Call api
        $response = $this->_callApi(self::METHOD_BMSCREATE_PUBLISH, $params);
        
        // Parse XML reply
        $parsedXML = $this->_handleXml($response['current']['body']);
        
        return true;
    }

    /****************************************************************/
    /* SERVICE: bms   METHOD: addComment                            */
    /****************************************************************/

    /**
       * Function calls for bms/addComment.
       *
       * @param string $ownerName     Comment owner (must be valid owner name)
       * @param string $commentTitle  Comment title
       * @param string $commentText   Comment text
       * @param string $chapterNumber If multiple article used, to which article 
       *                              goes this comment (optional)
       * @param string $textInChapter Text in chapter (optional)
       * @param string $commentId     Comment id (optional)
       * @param string $time          Comment time (optional)
       *
       * @access public
       *
       * @return string    ID of the comment added
       * @throws Services_SharedBook_Exception When no comment ID returned
       */
    public function bmsAddComment($ownerName, $commentTitle, $commentText,
                                  $chapterNumber = null, $textInChapter = null, 
                                  $commentId = null, $time = null)
    {
        // Set parameters for the API call
        $params = array(
                          'apiKey'       => $this->apiKey,
                          'authToken'    => $this->authToken,
                          'sessionToken' => $this->sessionToken,
                          'bmsId'        => $this->bmsId,
                          'ownerName'    => $ownerName,
                          'commentTitle' => $commentTitle,
                          'commentText'  => $commentText
                       );
        
        
        // Check if the optional parameters supplied and populate the params array
        if (!empty($textInChapter))  $params['textInChapter'] = $textInChapter;
        if (!empty($time))           $params['time']          = $time;
        if (!empty($commentId))      $params['commentId']     = $commentId;
        if (!empty($chapterNumber))  $params['chapterNumber'] = $chapterNumber;
        
        // Calculate signature
        $params["signature"] = $this->_calcSignature($params);
        
        // Call api
        $response = $this->_callApi(self::METHOD_BMS_ADD_COMMENT, $params);
        
        // Parse XML reply
        $parsedXML = $this->_handleXml($response['current']['body']);
        
        // Check comment ID returned
        if (empty($parsedXML->comment->attributes()->id)) {
            throw new Services_SharedBook_Exception('Invalid comment ID', 
                                                    self::WRAPPER_ERROR);
        }
        
        return $parsedXML->comment->attributes()->id;
    }

    /**
       * Function handles the photos uploads
       *
       * @param string $method    Api method to be called, since there are several 
                                  methods that add photos
       * @param string $filePath  Path for the photo to be uploaded
       * @param string $ownerName Photo owner (must be valid owner name)
       * @param string $fileName  Name for the photo (optional)
       * @param string $photoId   Photo id (optional)
       * @param string $caption   Photo caption (optional)
       * @param string $time      Photo time (optional)
       *
       * @access private
       *
       * @return string ID of the photo added
       * @throws Services_SharedBook_Exception When photo ID is not returned
       */
    private function _addPhoto($method, $filePath, $ownerName = null, $fileName = null,
                                $photoId = null, $caption = null, $time = null)
    {
        // Set parameters for the API call
        $params = array(
                          'apiKey'       => $this->apiKey,
                          'authToken'    => $this->authToken,
                          'sessionToken' => $this->sessionToken,
                          'bmsId'        => $this->bmsId
                       );

        // Check if the optional parameters supplied and populate the params array
        if (!empty($ownerName)) $params['ownerName']  = $ownerName;
        if (!empty($photoId))   $params['photoId']  = $photoId;
        if (!empty($time))      $params['time']     = $time;
        if (!empty($caption))   $params['caption']  = $caption;
        if (!empty($filename))  $params['filename'] = $fileName;

        // Create files array as requested by HTTP_Request
        $files = array();
        
        if (preg_match('/^http|^ftp|^https/', $filePath))
        {
            $params['url'] = $filePath;
        } else {
            $files = array(array("photo", $filePath));
        }
        
        // Calculate signature
        $params['signature'] = $this->_calcSignature($params);
        
        
        // Call api
        $response = $this->_callApi($method, $params, $files);

        // Parse XML reply
        $parsedXML = $this->_handleXml($response['current']['body']);

        // Check photo ID or URL returned
        
        $id  = @$parsedXML->photo->attributes()->id;
        $url = @$parsedXML->photo->attributes()->url;
        
        if (@empty($id) && @empty($url)) {
            throw new Services_SharedBook_Exception('Invalid photo ID/URL', 
                                                    self::WRAPPER_ERROR);
        }

        if (@!empty($url)) return $url;
        if (@!empty($id))  return $id;
        
        
    }

    /****************************************************************/
    /* SERVICE: bms   METHOD: addPhoto                              */
    /****************************************************************/

    /**
       * Function calls for bms/addPhoto
       *
       * @param string $filePath  Path for the photo to be uploaded
       * @param string $ownerName Photo owner (must be valid owner name)
       * @param string $fileName  Name for the photo (optional)
       * @param string $photoId   Photo id (optional)
       * @param string $caption   Photo caption (optional)
       * @param string $time      Photo time (optional)
       *
       * @access public
       *
       * @return string           ID of the photo added
       */
    public function bmsAddPhoto($filePath, $ownerName, 
                                $fileName = null, $photoId = null, 
                                $caption = null, $time = null)
    {
         // Call function that uploads photos
         return $this->_addPhoto(self::METHOD_BMS_ADD_PHOTO, $filePath, $ownerName, 
                                 $fileName,  $photoId, $caption, $time);
    }
    
    /****************************************************************/
    /* SERVICE: photo   METHOD: upload                              */
    /****************************************************************/

    /**
       * Function calls for photo/upload
       *
       * @param string $photoData  byte array fo the photo to be uploaded
       *
       * @access public
       *
       * @return string           URL of the photo added
       */
    public function photoUpload($filePath)
    {
         
         // Call function that uploads photos
         return $this->_addPhoto(self::METHOD_PHOTO_UPLOAD, $filePath);
    }

    /****************************************************************/
    /* SERVICE: bms   METHOD: setFrontCoverPhoto                    */
    /****************************************************************/

    /**
       * Function calls for bms/setFrontCoverPhoto
       *
       * @param string $filePath  Path for the photo to be uploaded
       * @param string $ownerName Photo owner (must be valid owner name)
       * @param string $fileName  Name for the photo (optional)
       * @param string $photoId   Photo id (optional)
       * @param string $caption   Photo caption (optional)
       * @param string $time      Photo time (optional)
       *
       * @access public
       *
       * @return string           ID of the photo added
       */
    public function bmsSetFrontCoverPhoto($filePath, $ownerName, 
                                          $fileName = null, $photoId = null, 
                                          $caption = null, $time = null)
    {
        // Call function that uploads photos
        return $this->_addPhoto(self::METHOD_BMS_FRONT_PHOTO, $filePath, $ownerName, 
                                $fileName, $photoId, $caption, $time);
    }

    /****************************************************************/
    /* SERVICE: bms   METHOD: setBackCoverPhoto                     */
    /****************************************************************/

    /**
       * Function calls for bms/setBackCoverPhoto
       *
       * @param string $filePath  Path for the photo to be uploaded
       * @param string $ownerName Photo owner (must be valid owner name)
       * @param string $fileName  Name for the photo (optional)
       * @param string $photoId   Photo id (optional)
       * @param string $caption   Photo caption (optional)
       * @param string $time      Photo time (optional)
       *
       * @access public
       *
       * @return string           ID of the photo added
       */
    public function bmsSetBackCoverPhoto($filePath, $ownerName, 
                                         $fileName = null, $photoId = null, 
                                         $caption = null, $time = null)
    {
        // Call function that uploads photos
        return $this->_addPhoto(self::METHOD_BMS_BACK_PHOTO, $filePath, 
                                $ownerName, $fileName, $photoId, $caption, $time);
    }


    /****************************************************************/
    /* SERVICE: bms   METHOD: publish                               */
    /****************************************************************/

    /**
      * Function calls for bms/publish
      *
      * @access public
      *
      * @return boolean Whether book making space publishing succeeded
      */
    public function bmsPublish()
    {
         // Set parameters for the API call
         $params = array(
                           'apiKey'       => $this->apiKey,
                           'authToken'    => $this->authToken,
                           'sessionToken' => $this->sessionToken,
                           'bmsId'        => $this->bmsId
                        );

         // Calculate signature
         $params['signature'] = $this->_calcSignature($params);

         // Call api
         $response = $this->_callApi(self::METHOD_BMS_PUBLISH, $params);

         // Parse XML reply
         $parsedXML = $this->_handleXml($response['current']['body']);

         return true;
    }

    /****************************************************************/
    /* SERVICE: bookcreate    METHOD: init                          */
    /****************************************************************/

    /**
     * Function calls for bookcreate/init
     *
     * @access public
     *
     * @return boolean Whether book creating init succeeded
     */
    public function bookCreateInit()
    {
        // Set parameters for the API call
        $params = array(
                          'apiKey'       => $this->apiKey,
                          'authToken'    => $this->authToken,
                          'sessionToken' => $this->sessionToken,
                          'bmsId'        => $this->bmsId
                       );

        // Calculate signature
        $params['signature'] = $this->_calcSignature($params);

        // Call api
        $response = $this->_callApi(self::METHOD_BOOKCREATE_INIT, $params);

        // Parse XML reply
        $parsedXML = $this->_handleXml($response['current']['body']);

        return true;
    }

    /****************************************************************/
    /* SERVICE: bookcreate    METHOD: setDedication                 */
    /****************************************************************/

    /**
     * Function calls for bookcreate/setDedication
     *
     * @param string $dedicationText Text for dedication page
     *
     * @access public
     *
     * @return boolean                Whether setting the dedication succeeded
     */
    public function bookCreateSetDedication($dedicationText)
    {
        // Set parameters for the API call
        $params = array(
                          'apiKey'         => $this->apiKey,
                          'authToken'      => $this->authToken,
                          'sessionToken'   => $this->sessionToken,
                          'bmsId'          => $this->bmsId,
                          'dedicationText' => $dedicationText
                       );

        // Calculate signature
        $params['signature'] = $this->_calcSignature($params);

        // Call api
        $response = $this->_callApi(self::METHOD_BOOKCREATE_DEDICATION, $params);

        // Parse XML reply
        $parsedXML = $this->_handleXml($response['current']['body']);

        return true;
    }
    
    /****************************************************************/
    /* SERVICE: bookcreate    METHOD: setSubtitle                   */
    /****************************************************************/

    /**
     * Function calls for bookcreate/setSubtitle
     *
     * @param string $subtitleText Text for subtitle in the cover
     *
     * @access public
     *
     * @return boolean                Whether setting the subtitle succeeded
     */
    public function bookCreateSetSubtitle($subtitleText)
    {
        // Set parameters for the API call
        $params = array(
                          'apiKey'         => $this->apiKey,
                          'authToken'      => $this->authToken,
                          'sessionToken'   => $this->sessionToken,
                          'bmsId'          => $this->bmsId,
                          'subtitle'       => $subtitleText
                       );

        // Calculate signature
        $params['signature'] = $this->_calcSignature($params);

        // Call api
        $response = $this->_callApi(self::METHOD_BOOKCREATE_SUBTITLE, $params);

        // Parse XML reply
        $parsedXML = $this->_handleXml($response['current']['body']);

        return true;
    }
    
    /****************************************************************/
    /* SERVICE: bookcreate    METHOD: setCoverTheme                 */
    /****************************************************************/

    /**
     * Function calls for bookcreate/setSubtitle
     *
     * @param string $subtitleText Text for subtitle in the cover
     *
     * @access public
     *
     * @return boolean                Whether setting the subtitle succeeded
     */
    public function bookCreateSetCoverTheme($coverTheme)
    {
        // Set parameters for the API call
        $params = array(
                          'apiKey'         => $this->apiKey,
                          'authToken'      => $this->authToken,
                          'sessionToken'   => $this->sessionToken,
                          'bmsId'          => $this->bmsId,
                          'coverTheme'     => $coverTheme
                       );

        // Calculate signature
        $params['signature'] = $this->_calcSignature($params);

        // Call api
        $response = $this->_callApi(self::METHOD_BOOKCREATE_COVERTHEME, $params);

        // Parse XML reply
        $parsedXML = $this->_handleXml($response['current']['body']);

        return true;
    }

    /****************************************************************/
    /* SERVICE: bookcreate    METHOD: publish                       */
    /****************************************************************/

    /**
     * Function calls for bookcreate/publish
     *
     * @access  public
     *
     * @return  string    Book ID of the published book
     * @throws            Services_SharedBook_Exception When no book ID returned
     */
    public function bookCreatePublish()
    {
        // Set parameters for the API call
        $params = array(
                          'apiKey'       => $this->apiKey,
                          'authToken'    => $this->authToken,
                          'sessionToken' => $this->sessionToken,
                          'bmsId'        => $this->bmsId
                       );

        // Calculate signature
        $params['signature'] = $this->_calcSignature($params);

        // Call api
        $response = $this->_callApi(self::METHOD_BOOKCREATE_PUBLISH, $params);

        // Parse XML reply
        $parsedXML = $this->_handleXml($response['current']['body']);

        // Check book ID returned
        if (@empty($parsedXML->book->attributes()->id)) {
            throw new Services_SharedBook_Exception('Invalid book ID', 
                                                    self::WRAPPER_ERROR);
        }

        $this->bookId = $parsedXML->book->attributes()->id;

        return $this->bookId;
    }
    
    

    /****************************************************************/
    /* SERVICE: book    METHOD: preview                             */
    /****************************************************************/

    /**
     * Function calls for book/preview
     *
     * @access  public
     *
     * @return  string    URL to the book itself
     * @throws            Services_SharedBook_Exception When no URL returned
     */
    public function bookPreview()
    {
        // Set parameters for the API call
        $params = array(
                        'apiKey'       => $this->apiKey,
                        'authToken'    => $this->authToken,
                        'sessionToken' => $this->sessionToken,
                        'bmsId'        => $this->bmsId,
                        'bookId'       => $this->bookId
                       );

        // Calculate signature
        $params['signature'] = $this->_calcSignature($params);

        // Call api
        $response = $this->_callApi(self::METHOD_BOOK_PREVIEW, $params);

        // Parse XML reply
        $parsedXML = $this->_handleXml($response['current']['body']);


        // Check book URL returned
        if (@empty($parsedXML->url)) {
            throw new Services_SharedBook_Exception('Invalid book URL',
                                                    self::WRAPPER_ERROR);
        }

        return $parsedXML->url;
    }

}

?>