<?php

/* vim: set expandtab tabstop=4 shiftwidth=4: */

/**
 * ReST-based package to access Compete web services.
 *
 * To use this package you will have to sign-up for an Compete developer
 * user account and register application to get your personal API key:
 * <ol>
 * <li>Go to http://developer.compete.com/</li>
 * <li>Register for a developer user account</li>
 * <li>Register your application to use API</li>
 * </ol>
 *
 * PHP versions 5
 *
 * @category Web Services
 * @package  Services_Compete
 * @author   AKIMOTO Hiroki <REPLACE-HERE-SIRNAME@gmail.com>
 * @author   Jonathan Street <jonathan@torrentialwebdev.com>
 * @license  PHP
 * @version  CVS: $Id$
 * @link     http://developer.compete.com/
 */

/**
 * Requires base PEAR class (for error handling)
 */
require_once 'PEAR.php';

require_once 'HTTP/Request.php';

/**
 * Class for accessing and retrieving information from Compete SnapShot API.
 *
 * Example:
 * <code>
 * require_once 'Services/Compete.php';
 *
 * $compete = new Services_Compete(COMPETE_PERSONAL_KEY);
 * $result = $compete->get('mixi.jp');
 *
 * print_r($result);
 * </code>
 *
 * @package Services_Compete
 * @author  AKIMOTO Hiroki <REPLACE-HERE-SIRNAME@gmail.com>
 * @version Release: @package_version@
 * @link    http://developer.compete.com/
 */
class Services_Compete
{
    const BASE_URL = 'http://api.compete.com/fast-cgi/MI';
    private $api_key = null;
    
    private $results = NULL;

    /**
     * Compete service wrapper
     *
     * @param string $api_key API key provided by Compete
     */
    public function __construct($api_key)
    {
        $this->api_key = $api_key;
    }

    /**
     * Compete service wrapper
     *
     * @param string $domain domain to get statistics
     * @param string $icon_size options 'small' or 'large' - default
     * @return bool true on success
     */
    public function query($domain, $icon_size = 'large')
    {
        $params = array(
            'd' => $domain,
            'ver' => 3,
            'apikey' => $this->api_key,
            'size' => $icon_size,
        );
        $url = Services_Compete::BASE_URL . '?' . http_build_query($params);

        $httpOptions = array(
            'method' => 'GET',
            'http' => '1.1',
        );
        $gottenPage = $this->sendHTTPRequest($url, $httpOptions);
        if ($gottenPage === false) {
            throw new Services_Compete_Exception(
                'failed to fetch Compete response'
            );
        }

        $xml = simplexml_load_string($gottenPage);
        if ($xml === false) {
            throw new Services_Compete_Exception(
                'failed to parse Compete response as XML'
            );
        }
        if (isset($xml->dmn->e)) {
            throw new Services_Compete_Exception(
                'Compete API error [' . $xml->dmn->e . ']'
            );
        }
        $this->results = $xml;
        return TRUE;
    }

    /**
     * Get trust data from fetched data
     *
     * @return object simplified trust data
     */
    public function getTrust ()
    {
        if(!is_null($this->results)) {
            $results->val = trim((string)$this->results->dmn->trust->val);  
            $results->link = trim((string)$this->results->dmn->trust->link);
            $results->icon = trim((string)$this->results->dmn->trust->icon);
            return($results);
        } else {
            throw new Services_Compete_Exception(
                'Attempted to access data before it has been fetched from the web service'
            );
        }
    }

    /**
     * Get traffic data from fetched data
     *
     * @return object simplified traffic data
     */ 
    public function getTraffic ()
    {
        if(!is_null($this->results)) {
            $results->year = trim((string)$this->results->dmn->metrics->val->yr);
            $results->month = trim((string)$this->results->dmn->metrics->val->mth);
            $results->ranking = trim((string)$this->results->dmn->metrics->val->uv->ranking);
            
            $results->count = trim((string)$this->results->dmn->metrics->val->uv->count);
            $results->count_int = (int)implode('', explode(',', $results->count));
            
            $results->link = trim((string)$this->results->dmn->metrics->link);
            $results->icon = trim((string)$this->results->dmn->metrics->icon);
            return($results);
        } else {
            throw new Services_Compete_Exception(
                'Attempted to access data before it has been fetched from the web service'
            );
        }
    }

    /**
     * Get deals data from fetched data
     *
     * @return object simplified deals data
     */ 
    public function getDeals ()
    {
        if(!is_null($this->results)) {
            $results->val = (int)$this->results->dmn->deals->val;
            $results->link = trim((string)$this->results->dmn->deals->link);
            $results->icon = trim((string)$this->results->dmn->deals->icon);
            return $results;
        } else {
            throw new Services_Compete_Exception(
                'Attempted to access data before it has been fetched from the web service'
            );
        }
    }


    /**
     * Get all data from fetched data
     *
     * @return object all simplified data
     */ 
    public function get ()
    {
        if(!is_null($this->results)) {
            $results->domain = (string)$this->results->dmn->nm;
            $results->trust = $this->getTrust();
            $results->traffic = $this->getTraffic();
            $results->deals = $this->getDeals();
            return $results;
        } else {
            throw new Services_Compete_Exception(
                'Attempted to access data before it has been fetched from the web service'
            );
        }     
    }
    
    /**
     * Get all data from fetched data
     *
     * @return object unprocessed simplexml object
     */ 
    public function getRaw ()
    {
        if(!is_null($this->results)) {
            return $this->results;
        } else {
            throw new Services_Compete_Exception(
                'Attempted to access data before it has been fetched from the web service'
            );
        }
        
    }
    
    
    
    /**
     * Send HTTP request
     *
     * @param string $uri URI.
     * @return string responded page
     */
    private function sendHTTPRequest($uri, $params)
    {
        $request = new HTTP_Request($uri, $params);

        $request->sendRequest();
        if (PEAR::isError($request)) {
            throw new Services_Compete_Exception(
                $request->getMessage()
            );
        }

        if ($request->getResponseCode() != 200) {
            throw new Services_Compete_Exception(
                'undesirable response [' . $request->getResponseCode() . ']'
            );
        }

        return $request->getResponseBody();
    }
}

require_once 'PEAR/Exception.php';

class Services_Compete_Exception extends PEAR_Exception
{
}

?>