(*
  Copyright Florian Hatat, Tom Hirschowitz, Pierre Hyvernat,
  Pierre-Etienne Meunier, Christophe Raffalli, Guillaume Theyssier 2012.

  This file is part of Patoline.

  Patoline is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  Patoline is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Patoline.  If not, see <http://www.gnu.org/licenses/>.
*)
open Typography
open Typography.Document
open Typography.Complete
open Typography.Fonts.FTypes
open Typography.Util
open Typography.ConfigUtil
open Typography.Fonts
open Typography.Box
open Typography.Break
open CamomileLibrary
let _=Random.self_init ()

module Euler = Euler
module Numerals = Numerals

let replace_utf8 x y z=if String.length x>0 then (
  let buf=Buffer.create (String.length x) in
  let repl=UTF8.init 1 (fun _->UChar.chr y) in
  let rec add_it i=
    if not (UTF8.out_of_range z i) then (
      try
        let rec comp j=
          if UTF8.out_of_range x j then j else
            if UTF8.out_of_range z (i+j) then raise Not_found else
              if UTF8.look z (i+j) <> UTF8.look x j then raise Not_found else
                comp (UTF8.next x j)
        in
        let j=comp 0 in
        Buffer.add_string buf repl;
        add_it (i+j)
      with
          Not_found->(
            Buffer.add_string buf (String.sub z i (UTF8.next z i-i));
            add_it (UTF8.next z i)
          )
    )
  in
  add_it 0;
  Buffer.contents buf
) else z

let word_subst=
  (fun x->List.fold_left (fun y f->f y) x
    [
      replace_utf8 ("``") 8220;
      replace_utf8 ("''") 8221
    ]
  )

let hyphenate_dict dict=
  try
    let i=open_in_bin (findHyph dict) in
    let inp=input_value i in
    close_in i;
    (fun str->
      let hyphenated=Hyphenate.hyphenate inp str in
      let pos=Array.make (List.length hyphenated-1) ("","") in
      let rec hyph l i cur=match l with
          []->()
        | h::s->(
          pos.(i)<-(cur^"-", List.fold_left (^) "" l);
          hyph s (i+1) (cur^h)
        )
      in
      match hyphenated with
          []->[||]
        | h::s->(hyph s 0 h; pos));
  with
      File_not_found (f,p)->
	(Printf.fprintf stderr "Warning : no hyphenation dictionary (%s not found). Path :\n" f;
         List.iter (Printf.fprintf stderr "%s\n") p;
         fun x->[||])



let alegreya=
  [ Regular,
    (Lazy.lazy_from_fun
       (fun ()->
         (Fonts.loadFont (findFont "Alegreya/Alegreya-Regular.otf")),
         (fun x->x),
         (fun x->List.fold_left (fun a f->f a) x
           [make_ligature [168;175] {glyph_utf8="fi";glyph_index=245};
            make_ligature [168;181] {glyph_utf8="fl";glyph_index=246};
            make_ligature [168;177] {glyph_utf8="fj";glyph_index=383};
            make_ligature [175;177] {glyph_utf8="ij";glyph_index=176};
           ]),
         (fun x->x)),
     Lazy.lazy_from_fun
       (fun ()->
         (Fonts.loadFont (findFont "Alegreya/Alegreya-Italic.otf")),
         (fun x->x),
         (fun x->List.fold_left (fun a f->f a) x
           [make_ligature [162;170] {glyph_utf8="fi";glyph_index=477};
            make_ligature [162;175] {glyph_utf8="fl";glyph_index=478};
            make_ligature [162;171] {glyph_utf8="fj";glyph_index=482};
            make_ligature [170;171] {glyph_utf8="ij";glyph_index=476};
           ]),
         (fun x->x)));
    Bold,
    (Lazy.lazy_from_fun
       (fun ()->
         (Fonts.loadFont (findFont "Alegreya/Alegreya-Bold.otf")),
         (fun x->x),
         (fun x->List.fold_left (fun a f->f a) x
           [make_ligature [168;175] {glyph_utf8="fi";glyph_index=245};
            make_ligature [168;181] {glyph_utf8="fl";glyph_index=246};
            make_ligature [168;177] {glyph_utf8="fj";glyph_index=383};
            make_ligature [175;177] {glyph_utf8="ij";glyph_index=176};
           ]),
         (fun x->x)),
     Lazy.lazy_from_fun
       (fun ()->
         (Fonts.loadFont (findFont "Alegreya/Alegreya-BoldItalic.otf")),
         (fun x->x),
         (fun x->List.fold_left (fun a f->f a) x
           [make_ligature [162;170] {glyph_utf8="fi";glyph_index=477};
            make_ligature [162;175] {glyph_utf8="fl";glyph_index=478};
            make_ligature [162;171] {glyph_utf8="fj";glyph_index=482};
            make_ligature [170;171] {glyph_utf8="ij";glyph_index=476};
           ]),
         (fun x->x)));
    Caps,
    (simpleFamilyMember (fun ()->Fonts.loadFont (findFont "Alegreya/AlegreyaSC-Regular.otf")),
     simpleFamilyMember (fun ()->Fonts.loadFont (findFont "Alegreya/AlegreyaSC-Italic.otf")))]

let philosopher=[
  Regular,
  (simpleFamilyMember (fun ()->Fonts.loadFont (findFont "Philosopher/Philosopher-Regular.otf")),
   simpleFamilyMember (fun ()->Fonts.loadFont (findFont "Philosopher/Philosopher-Italic.otf")));
  Bold,
  (simpleFamilyMember (fun ()->Fonts.loadFont (findFont "Philosopher/Philosopher-Bold.otf")),
   simpleFamilyMember (fun ()->Fonts.loadFont (findFont "Philosopher/Philosopher-BoldItalic.otf")))
]

let texgyrecursor=
  [ Regular,
    (Lazy.lazy_from_fun
       (fun ()->
         (Fonts.loadFont (findFont "TexGyreCursor/texgyrecursor-regular.otf")),
          (fun x->x),
          (fun x->x),
          (fun x->x)),
     Lazy.lazy_from_fun
       (fun ()->
          (Fonts.loadFont (findFont "TexGyreCursor/texgyrecursor-italic.otf")),
          (fun x->x),
          (fun x->x),
          (fun x->x)));
    Bold,
    (Lazy.lazy_from_fun
       (fun ()->
          (Fonts.loadFont (findFont "TexGyreCursor/texgyrecursor-bold.otf")),
          (fun x->x),
          (fun x->x),
          (fun x->x)),
     Lazy.lazy_from_fun
       (fun ()->
          (Fonts.loadFont (findFont "TexGyreCursor/texgyrecursor-bolditalic.otf")),
          (fun x->x),
          (fun x->x),
          (fun x->x)));

  ]

let bitstreamverasansmono=
  [ Regular,
    (Lazy.lazy_from_fun
       (fun ()->
         (Fonts.loadFont (findFont "BitstreamVeraSansMono/BitstreamVeraSansMono-Roman.otf")),
          (fun x->x),
          (fun x->x),
          (fun x->x)),
     Lazy.lazy_from_fun
       (fun ()->
          (Fonts.loadFont (findFont "BitstreamVeraSansMono/BitstreamVeraSansMono-Oblique.otf")),
          (fun x->x),
          (fun x->x),
          (fun x->x)));
    Bold,
    (Lazy.lazy_from_fun
       (fun ()->
          (Fonts.loadFont (findFont "BitstreamVeraSansMono/BitstreamVeraSansMono-Bold.otf")),
          (fun x->x),
          (fun x->x),
          (fun x->x)),
     Lazy.lazy_from_fun
       (fun ()->
          (Fonts.loadFont (findFont "BitstreamVeraSansMono/BitstreamVeraSansMono-BoldOb.otf")),
          (fun x->x),
          (fun x->x),
          (fun x->x)));

  ]

let all_fonts = [alegreya; texgyrecursor] (* trick to force same type *)

let font_size_ratio font1 font2 =
  let x_h f =
    let f,_,_,_ = Lazy.force (fst (List.assoc Regular f)) in
    let x=Fonts.loadGlyph f
      ({empty_glyph with glyph_index=Fonts.glyph_of_char f 'x'}) in
    Fonts.glyph_y1 x -.  Fonts.glyph_y0 x
  in
  x_h font1 /. x_h font2

let parameters env paragraphs figures last_parameters last_figures last_users (last_line:line) (line:line)=
  let page_footnotes=ref 0 in
  let frame_measure=env.normalMeasure in
  let measure=IntMap.fold (fun i aa m->match aa with
      Break.Placed a->
        (if page line=page a &&
           line.height>=
           a.height+.figures.(i).drawing_y0
         && line.height<=
           a.height+. figures.(i).drawing_y1
         then
            frame_measure -. figures.(i).drawing_nominal_width
         else m)
    | _->m
  ) last_figures frame_measure
  in
  let p={ measure=measure;
    left_margin=env.normalLeftMargin;
    local_optimization=0;
    min_page_before=0;
    min_page_after=0;
    min_height_before=0.;
    min_height_after=0.;
    not_last_line=false;
    not_first_line=false;
    min_lines_before=1;
    min_lines_after=0;
    absolute=false
  }
  in
  fold_left_line paragraphs (fun p0 x->match x with
      Parameters fp->(
        let p1=fp p0 in
        p1
      )
    | _->p0
  ) p line


module type Output=
  sig
    type output
    val outputParams : output

    val output :
      output ->
      Typography.Document.tree ->
      Typography.Document.environment -> string -> unit
  end

module Format=functor (D:Document.DocumentStructure)->(
  struct

    let sourcePosition(file,line,column,char) =
      [tT (Printf.sprintf "%s: %d,%d (%d)" file line column char)]

    let node ?(node_env=(fun env->env)) l=
      Document.Node
        {Document.empty with
          Document.node_env=node_env;
          Document.children=List.fold_left
            (fun m (l,_)->Util.IntMap.add (Util.IntMap.cardinal m) l m) Util.IntMap.empty l},
      []

    let parameters=parameters
    let center = do_center parameters
    let ragged_right = do_ragged_right parameters
    let ragged_left = do_ragged_left parameters

    let postprocess_tree tree=
      let has_institute=ref false in
      let has_author=ref false in
      let with_institute=match tree with
          Node n when not (List.mem_assoc "title already typset" n.node_tags)->(try
                     let cont=[tT (List.assoc "Institute" n.node_tags)] in
                     let par=Paragraph {
                       par_contents=cont;
                       par_env=(fun env->{env with par_indent=[]});
                       par_post_env=(fun env1 env2 -> { env1 with names=names env2; counters=env2.counters;
                         user_positions=user_positions env2 });
                       par_parameters=
                         (fun a b c d e f g line->
                           { (center a b c d e f g line) with
                             min_lines_after=if line.lineEnd>=Array.length b.(line.paragraph) then
                                 1 else 0;
                             min_lines_before=if g.lineEnd>=Array.length b.(g.paragraph) then
                                 1 else 0
                           });
                       par_badness=(badness);
                       par_completeLine=Complete.normal;
                       par_states=IntSet.empty;
                       par_paragraph=(-1) }
                     in
                     has_institute:=true;
                     fst (up (newChildBefore (tree,[]) par))
            with
                Not_found->tree)
        | _->tree
      in

      let with_author=match with_institute with
          Node n when not (List.mem_assoc "title already typset" n.node_tags)->(try
                     let cont=[tT (List.assoc "Author" n.node_tags)] in
                     let par=Paragraph {
                       par_contents=cont;
                       par_env=(fun env->{env with par_indent=[]});
                       par_post_env=(fun env1 env2 -> { env1 with names=names env2; counters=env2.counters;
                         user_positions=user_positions env2 });
                       par_parameters=
                         (fun a b c d e f g line->
                           { (center a b c d e f g line) with
                             min_lines_after=
                               if line.lineEnd>=Array.length b.(line.paragraph)then
                                 if !has_institute then
                                   2
                                 else
                                   4
                               else 1;
                             min_height_before=if line.lineEnd>=Array.length b.(line.paragraph) then
                                 2.*.a.normalLead else 0.
                           });
                       par_badness=(badness);
                       par_completeLine=Complete.normal;
                       par_states=IntSet.empty;
                       par_paragraph=(-1) }
                     in
                     has_author:=true;
                     fst (up (newChildBefore (with_institute,[]) par))
            with
                Not_found->with_institute)
        | _->with_institute
      in

      let with_title=match tree with
          Node n when not (List.mem_assoc "title already typeset" n.node_tags)
              && n.displayname<>[]->
            let par=Paragraph {
              par_contents=n.displayname;
              par_env=(fun env->resize_env (env.size*.2.) {env with par_indent=[]; hyphenate=(fun _->[||])});
              par_post_env=(fun env1 env2 -> { env1 with names=names env2; counters=env2.counters;
                user_positions=user_positions env2 });
              par_parameters=
                (fun a b c d e f g line->
                  { (center a b c d e f g line) with
                    min_lines_after=
                      if n.displayname<>[] && line.lineEnd>=Array.length b.(line.paragraph) then
                        if !has_author || !has_institute then
                          3
                        else
                          6
                      else 1;
                    min_height_before=0. });
              par_badness=(badness);
              par_completeLine=Complete.normal;
              par_states=IntSet.empty;
              par_paragraph=(-1)}
            in
            fst (up (newChildBefore (with_author,[]) par))
        | _->with_author
      in

      let rec sectionize path=function
      Node n when List.mem_assoc "structural" n.node_tags ->
        let section_name=
          if List.mem_assoc "numbered" n.node_tags  then
            [C (fun env->
              let a,b=try StrMap.find "_structure" env.counters with Not_found -> -1,[0] in
              bB (fun _->[Marker (Structure path)])
              ::tT (String.concat "." (List.map (fun x->string_of_int (x+1)) (List.rev (drop 1 b))))
              ::tT " "
              ::n.displayname
            )]
          else
            [C(fun env->
              bB (fun env->[Marker (Structure path)])::
                n.displayname)]
        in
        let par=Paragraph {
          par_contents=section_name;
          par_env=(fun env->
            let a,b=try StrMap.find "_structure" env.counters with Not_found -> -1,[0] in
            { (envAlternative (Fonts.Opentype.oldStyleFigures::env.fontFeatures) Caps env) with
              size=(if List.length b <= 2 then sqrt phi else
                  sqrt (sqrt phi))*.env.size;
            });
          par_post_env=(fun env1 env2 -> { env1 with names=names env2; counters=env2.counters;
            user_positions=user_positions env2 });
          par_parameters=
            (fun a b c d e f g line->
              { (parameters a b c d e f g line) with
                min_height_before=if line.lineStart=0 then a.normalLead else 0.;
                min_height_after=if line.lineEnd>=Array.length b.(line.paragraph) then a.normalLead else 0.;
                not_last_line=true });
          par_badness=(badness);
          par_completeLine=Complete.normal;
          par_states=IntSet.empty;
          par_paragraph=(-1) }
        in
        fst (up (newChildBefore (
          Node { n with children=IntMap.mapi (fun k a->sectionize (k::path) a)
              n.children }, []) par
        ))
        | a->a
      in
      let with_chapters=match with_title with
          Node n->Node { n with children=IntMap.map (sectionize []) n.children }
        | _->with_title
      in
      with_chapters

    let paragraph ?(parameters=parameters) ?(par_env=(fun x->x)) cont=
      (Paragraph {par_contents=cont; par_env=par_env;
                  par_post_env=(fun env1 env2 -> { env1 with names=env2.names;
                                                     counters=env2.counters;
                                                     user_positions=env2.user_positions });
                  par_badness=(badness);
                  par_parameters=parameters; par_completeLine=Complete.normal;
                  par_states=IntSet.empty;
                  par_paragraph=(-1)}, [])


    let indent ()=[bB (fun env->env.par_indent);Env (fun env->{env with par_indent=[]})]


    let defaultEnv:environment=
      let f,str,subst,pos=selectFont alegreya Regular false in
      let fsize=3.7 in
      let feat= [ Opentype.standardLigatures ] in
      let loaded_feat=Fonts.select_features f [ Opentype.standardLigatures ] in
        {
          fontFamily=alegreya;
          fontMonoFamily=bitstreamverasansmono (*texgyrecursor*);
	  fontMonoRatio=font_size_ratio alegreya bitstreamverasansmono (*texgyrecursor*);
          fontItalic=false;
          fontAlternative=Regular;
          fontFeatures=feat;
          fontColor=OutputCommon.black;
          font=f;
          mathsEnvironment=Euler.default;
	  mathStyle=Document.Mathematical.Text;
          word_substitutions=
            (fun x->List.fold_left (fun y f->f y) x
               [
                 replace_utf8 ("``") 8220;
                 replace_utf8 ("''") 8221
               ]
            );
          substitutions=(fun glyphs->Fonts.apply_features f loaded_feat (subst glyphs));
          positioning=(fun x->pos (positioning f x));
          footnote_y=10.;
          size=fsize;
          lead=13./.10.*.fsize;
          normalMeasure=(fst a4)*.2./.3.;
          normalLead=13./.10.*.fsize;
          normalLeftMargin=0.;
          normalPageFormat=a4;
          par_indent = [Drawing { drawing_min_width= 4.0 *. phi;
                                  drawing_max_width= 4.0 *. phi;
				  drawing_width_fixed = true;
				  drawing_adjust_before = false;
                                  drawing_y0=0.;drawing_y1=0.;
                                  drawing_nominal_width= 4.0 *. phi;
                                  drawing_contents=(fun _->[]);
                                  drawing_break_badness=0.;
                                  drawing_states=IntSet.empty;
                                  drawing_badness=fun _-> 0. }];
          hyphenate=hyphenate_dict "hyph-en-us.hdict";
          counters=StrMap.empty;
          names=StrMap.empty;
          user_positions=MarkerMap.empty;
          new_page=Document.default_new_page a4;
          new_line=(fun env node params nextNode nextParams layout height->
            if node==nextNode && node.layout==layout then (
              let min_height=min height (node.height-.params.min_height_after) in
              let h0=min_height/.env.lead in
              let h1=if (ceil h0-.h0)<=1e-10 then ceil h0 else floor h0 in
              let next_height=env.lead*.h1 in
              let hh=if next_height>=height then next_height-.env.lead else next_height in
              (* Printf.fprintf stderr "cas 1 %f\n" hh;flush stderr; *)
              hh
            ) else
              let d=if node.layout=layout then (
                let min_height=min (nextNode.height-.env.lead) (node.height -. max params.min_height_after nextParams.min_height_before) in
                let h0=min_height/.env.lead in
                let h1=if (ceil h0-.h0)<=1e-10 then ceil h0 else floor h0 in
                (* Printf.fprintf stderr "cas 2.1 %f %f %f \n" min_height h0 h1;flush stderr; *)
                env.lead*.h1
              ) else (
                let l=(fst layout).frame_y1 in
                let min_height=(height-. env.lead) in
                let h0=(floor (min_height/.env.lead)) in
                let h1=if (ceil h0-.h0)<=1e-10 then ceil h0 else floor h0 in
                (* Printf.fprintf stderr "cas 2.2 %f %f %f %f\n" l min_height h0 h1;flush stderr; *)
                env.lead*.h1
              )
              in
              d
          );
	  show_boxes=false;
	  show_frames=false;
	  adjust_optical_alpha=3.1416 /. 4.;
	  adjust_optical_beta=0.2; (* kerning between math and text while spacing between word is not kerned requires a small beta *)
	  adjust_epsilon=5e-2;
	  adjust_min_space=1./.9.;
	  math_break_badness = 250.0; (* testé juste sur tests/test_break_badness *)
        }


    let title str ?label ?(extra_tags=[]) displayname =
      let displayname=[C (fun _->env_accessed:=true;displayname)] in
      try
	let name = string_of_contents displayname in
	let t0',path=
          match top !str with
            Node n,path ->
	      if List.mem_assoc "maintitle" n.node_tags then
		raise Exit;
	      Node { n with
                name=name;
                node_tags=("maintitle","")::("structural","")::("intoc","")::extra_tags@n.node_tags;
                displayname = displayname},path
          | t,path->
	    Node { empty with
              name=name;
              node_tags=["structural","";"intoc",""];
              displayname=displayname;
	      children=IntMap.singleton 1 t;
              node_env=(fun x->x);
              node_post_env=(fun x y->{ x with names=y.names; counters=y.counters;
                user_positions=y.user_positions });
              node_states=IntSet.empty
            },path
	in
        str:=follow (t0',[]) (List.map fst (List.rev path)); true
      with
	Exit ->
	  newStruct D.structure displayname; false

    module TableOfContents=struct
      let do_begin_env ()=
        let max_depth=2 in
        TableOfContents.these center D.structure D.structure max_depth
      let do_end_env ()=()
    end


    let split_space is_letter is_special s =
      let gl env =
	let font,_,_,_=selectFont env.fontFamily Regular false in
	glyph_of_string env.substitutions env.positioning font env.size env.fontColor " "
      in
      let space = bB(fun env -> gl env) in
      let len = String.length s in
      let rec fn acc w i0 i =
	if i >= len then
	  List.rev (tT (String.sub s i0 (len - i0))::acc)
	else if s.[i] = ' ' then
	  let acc = if i <> i0 then
	      space::tT (String.sub s i0 (i - i0))::acc
	    else
	      space::acc
	  in
	  fn acc None (i+1) (i+1)
	else if is_special s.[i] then
	  let acc = if i <> i0 then
	    tT(String.sub s i 1)::tT (String.sub s i0 (i - i0))::acc
	    else
	      tT(String.sub s i 1)::acc
	  in
	  fn acc None (i+1) (i+1)
	else if w =None then
	  fn acc (Some (is_letter s.[i])) i0 (i+1)
	else if w = Some (is_letter s.[i]) then
	  fn acc  w i0 (i + 1)
	else
	  fn (tT (String.sub s i0 (i - i0))::acc) (Some (is_letter s.[i])) i (i+1)
      in fn [] None 0 0

    let is_letter_ml = fun c ->
      let c = Char.code c in
      (Char.code 'a' <= c && c <= Char.code 'z') ||
      (Char.code 'A' <= c && c <= Char.code 'Z') ||
      (Char.code '0' <= c && c <= Char.code '9') ||
      Char.code '_' = c || c = Char.code '\''


    let verb_counter filename =
      let get_line env =
	  try 
	    match StrMap.find filename env.counters
	    with a,[line] -> line
	    | _ -> raise Not_found
	  with Not_found -> 1
      in
      C (fun env ->
	let line = string_of_int (get_line env) in
	let miss = 4 - String.length line in
	let rec fn acc n = if n = 0 then acc else fn (tT " "::acc) (n - 1) in
	fn [tT line;tT " "] miss)::
      Env (fun env ->
	let line = get_line env in
	{env with counters = StrMap.add filename (-1,[line+1]) env.counters})::
	[]

    let lang_ML keywords specials s =
      let l = split_space is_letter_ml (fun c -> List.mem c specials) s in
      List.rev (List.fold_left (fun a x ->
	match x with
            T (s,_) as x when List.mem s keywords -> bold [x]@a
	| x -> x::a) [] l)

    let lang_default str = split_space (fun _ -> true) (fun _ -> false) str

    let lang_SML s=
      let specials = ['(';')';';'] in
      let keywords = ["fun";"as";"fn";"*";"(";")";",";";";"val";"and";"=>";"->";"type";"|";"=";
			  "case";"of";"datatype";"let";"rec";"end"] in
      lang_ML keywords specials s

    let lang_OCaml s=
      let specials = ['(';')';';'] in
      let keywords = ["fun";"as";"function";"(";")";"*";";";",";"val";"and";"=>";"->";"type";"|";"=";
		      "match";"with";"rec";"let";"begin";"end";"while";"for";"do";"done";
		      "struct"; "sig"; "module"; "functor"] in
      lang_ML keywords specials s


    let minipage env str=
      let env',fig_params,params,new_page_list,new_line_list,compl,bads,pars,par_trees,figures,figure_trees=flatten env D.fixable (fst str) in
      let (_,pages,fig',user')=TS.typeset
        ~completeLine:compl
        ~figure_parameters:fig_params
        ~figures:figures
        ~parameters:params
        ~new_page:new_page_list
        ~new_line:new_line_list
        ~badness:bads
        pars
      in
      (OutputDrawing.output pars figures
         env'
         pages)

    let minipage' env str=
      let env',fig_params,params,new_page_list,new_line_list,compl,bads,pars,par_trees,figures,figure_trees=flatten env D.fixable (fst str) in
      let (_,pages,fig',user')=TS.typeset
        ~completeLine:compl
        ~figure_parameters:fig_params
        ~figures:figures
        ~parameters:params
        ~new_page:new_page_list
        ~new_line:new_line_list
        ~badness:bads
        pars
      in
      (OutputDrawing.output pars figures
         env'
         pages,env')

    let env_stack=ref []

    module Env_minipage=struct
      let do_begin_env ()=
        D.structure:=newChildAfter !D.structure (Node empty);
        env_stack:=(List.map fst (snd !D.structure)) :: !env_stack


      let do_end_env ()=
	D.structure:=follow (top !D.structure) (List.rev (List.hd !env_stack));
        env_stack:=List.tl !env_stack;

        let t,num=match !D.structure with
            t,(h,_)::_->t,h
          | t,[]->t,0
        in
        (match up !D.structure with
            Node n,x->
              D.structure:=(Node { n with children=IntMap.remove num n.children },x);
          | x->D.structure:=x);
        let cont=
          [bB (fun env->List.map (fun x->Drawing x) (Array.to_list (minipage env (t,[]))))]
        in
        match lastChild !D.structure with
            Paragraph x,y->
              D.structure:=up(Paragraph {x with par_contents=x.par_contents@cont},y);
          | _->(
            newPar D.structure Complete.normal parameters cont;
            (* D.structure:=lastChild !D.structure *)
          )

    end


    let figure ?(parameters=center) ?(name="") ?(caption=[]) ?(scale=1.) drawing=
      let drawing' env=
        let dr_=drawing env in
        let dr=
          if scale<>1. then
            match resize scale (Drawing dr_) with Drawing f->f | _->assert false
          else dr_
        in
        let lvl,num=try StrMap.find "figure" env.counters with Not_found -> -1,[] in
        let _,str_counter=try StrMap.find "_structure" env.counters with Not_found -> -1,[] in
        let sect_num=drop (List.length str_counter - max 0 lvl+1) str_counter in
	let caption =
	  minipage {env with normalLeftMargin=0.} (
	    paragraph ((
              [ tT "Figure"; tT " ";
                tT (String.concat "." (List.map (fun x->string_of_int (x+1)) (List.rev (num@sect_num)))) ]
              @(if caption=[] then [] else tT" "::tT"–"::tT" "::caption)
            )))
	in
        let caption= caption.(0) in
        let fig=if caption.drawing_nominal_width<=dr.drawing_nominal_width then
          drawing_blit dr
            ((dr.drawing_nominal_width-.caption.drawing_nominal_width)/.2.)
            (dr.drawing_y0-.2.*.caption.drawing_y1) caption
        else
          drawing_blit caption
            ((caption.drawing_nominal_width-.dr.drawing_nominal_width)/.2.)
            (2.*.caption.drawing_y1-.dr.drawing_y0) dr
        in
        { fig with drawing_y0=fig.drawing_y0-.env.lead }
      in
      figure ~name:name D.structure center drawing'



    type tableParams={ widths:environment->float array; h_spacing:float; v_spacing:float }

    let table params tab=
      [ bB (fun env->
             let widths0=params.widths env in
             let widths=Array.make (Array.length widths0) 0. in
             let heights=Array.make (Array.length tab) 0. in
             let tab_formatted=Array.mapi
               (fun i x->
                  Array.mapi (fun j y->
                    let minip=(minipage
                                 { env with normalMeasure=widths0.(j) } y).(0) in
                    widths.(j)<-max widths.(j) (minip.drawing_max_width);
                    heights.(i)<-max heights.(i) (minip.drawing_y1-.minip.drawing_y0);
                    minip
                  ) x
               )
               tab
             in
             for i=0 to Array.length heights-1 do
               heights.(i)<-(ceil (heights.(i)/.env.normalLead))*.env.normalLead
             done;
             let contents=ref [] in
             let x=ref 0. in
             let y=ref 0. in
             let max_x=ref 0. in
             let max_y=ref (-.infinity) in
             let min_y=ref infinity in
             let ymin=ref 0. in
             let ymax=ref 0. in
               for i=0 to Array.length tab_formatted-1 do
                 x:=0.;
                 ymin:=0.;
                 ymax:= -.infinity;
                 let conts=ref [] in
                 for j=0 to Array.length tab_formatted.(i)-1 do
                   let cont=tab_formatted.(i).(j) in
                   conts:=(List.map (OutputCommon.translate !x 0.)
                             (cont.drawing_contents (widths.(j)))) @ (!conts);
                   ymin := min !ymin cont.drawing_y0;
                   ymax := max !ymax cont.drawing_y1;
                   x:= !x +. widths0.(j) +. params.h_spacing
                 done;
                 max_x:=max !x !max_x;
                 contents:=(List.map (OutputCommon.translate 0. !y) !conts)@(!contents);
                 max_y:=max !max_y (!y+. !ymax);
                 min_y:=min !min_y (!y+. !ymin);
                 y:=(!y)-. heights.(i)-.params.v_spacing;
               done;

               [Drawing {
                  drawing_min_width= !x;
                  drawing_max_width= !x;
                  drawing_nominal_width= !x;
		  drawing_width_fixed = true;
		  drawing_adjust_before = false;
                  drawing_y0= !min_y;
                  drawing_y1= !max_y;
                  drawing_break_badness=0.;
                  drawing_states=IntSet.empty;
                  drawing_badness=(fun _->0.);
                  drawing_contents=(fun _-> List.map (OutputCommon.translate 0. 0.) !contents)
                }]
          )]

    module Env_env (M:sig val arg1:Document.environment->Document.environment end)=struct
      let do_begin_env ()=
        env_stack:=(List.map fst (snd !D.structure)) :: !env_stack ;
        D.structure:=newChildAfter !D.structure (Node { empty with node_env=M.arg1 })


      let do_end_env ()=
	D.structure :=follow (top !D.structure) (List.rev (List.hd !env_stack)) ;
        env_stack:=List.tl !env_stack

    end

    module Env_noindent=struct
      let do_begin_env ()=
        env_stack:=(List.map fst (snd !D.structure)) :: !env_stack ;
        D.structure:=newChildAfter !D.structure
          (Node { empty with node_tags=("noindent","")::empty.node_tags })


      let do_end_env ()=
	D.structure :=follow (top !D.structure) (List.rev (List.hd !env_stack)) ;
        env_stack:=List.tl !env_stack
    end

    let noindent ()=
      []

    let hfill t = [bB (fun env-> let x = env.normalMeasure in
				  [match glue 0. env.size (x /. t) with
				    Glue x -> Drawing x
				  | _ -> assert false
				  ])]

    let hand () = hfill 4. @ hspace 0. @ hfill 4.

    module Env_mathpar = struct

      let do_begin_env () = 
	D.structure:=newChildAfter !D.structure (Node Document.empty) ;
	env_stack := (List.map fst (snd !D.structure)) :: !env_stack 

      let do_end_env () = 
	D.structure := follow (top !D.structure) (List.rev (List.hd !env_stack)) ;
	env_stack:=List.tl !env_stack ;
	let rec truc t = match t with
	  | Paragraph p -> Paragraph { p with par_contents =
	      (hfill 2.) @ p.par_contents @ (hfill 2.) }
	  | Node n -> Node ({ n with children = IntMap.map truc n.children })
	  | _ -> t
	in
	D.structure := up (truc (fst !D.structure), (snd !D.structure))

    end

    let displayedFormula a b c d e f g line=
      { (center a b c d e f g line) with
        min_height_before=3.*.a.lead/.4.;
        min_height_after=3.*.a.lead/.4.;
        not_first_line=true }

    module Env_center = struct

      let do_begin_env ()=
        D.structure:=newChildAfter (!D.structure) (Node { empty with node_env=(fun env->{env with par_indent=[]})});
        env_stack:=(List.map fst (snd !D.structure)) :: !env_stack

      let do_end_env ()=
        let center p = { p with par_parameters=Document.do_center p.par_parameters } in
        let res0, path0=(follow (top !D.structure) (List.rev (List.hd !env_stack))) in
        let res = map_paragraphs center res0 in
          D.structure:=up (res, path0);
          env_stack:=List.tl !env_stack

    end
    module Env_raggedRight = struct

      let do_begin_env ()=
        D.structure:=newChildAfter (!D.structure) (Node empty);
        env_stack:=(List.map fst (snd !D.structure)) :: !env_stack

      let do_end_env ()=
        let rag p = { p with par_parameters=ragged_right } in
        let res0, path0=(follow (top !D.structure) (List.rev (List.hd !env_stack))) in
        let res = map_paragraphs rag res0 in
          D.structure:=up (res, path0);
          env_stack:=List.tl !env_stack

    end
    module Env_raggedLeft = struct

      let do_begin_env ()=
        D.structure:=newChildAfter (!D.structure) (Node empty);
        env_stack:=(List.map fst (snd !D.structure)) :: !env_stack

      let do_end_env ()=
        let rag p = { p with par_parameters=ragged_left } in
        let res0, path0=(follow (top !D.structure) (List.rev (List.hd !env_stack))) in
        let res = map_paragraphs rag res0 in
          D.structure:=up (res, path0);
          env_stack:=List.tl !env_stack

    end

    let tiret_w env=phi*.env.size

    module type Enumeration=sig
      val from_counter:int list->content list
    end
    module Enumerate = functor (M:Enumeration)->struct
      let do_begin_env ()=
        D.structure:=newChildAfter (!D.structure)
          (Node { empty with
            node_env=
              (fun env->
                let lvl,cou=try StrMap.find "enumerate" env.counters with Not_found-> -1,[] in
                { env with
                  normalMeasure=env.normalMeasure-.tiret_w env;
                  normalLeftMargin=env.normalLeftMargin+.tiret_w env;
                  counters=StrMap.add "enumerate" (lvl,(-1)::cou) env.counters }
              );
            node_post_env=
              (fun env0 env1->
                let cou=try
                          let lvl,enum=StrMap.find "enumerate" env1.counters in
                          StrMap.add "enumerate" (lvl,drop 1 enum) env1.counters
                  with Not_found-> env1.counters
                in
                { env0 with names=env1.names;user_positions=env1.user_positions;counters=cou });
            node_tags=("structure","")::empty.node_tags;
          });
        env_stack:=(List.map fst (snd !D.structure)) :: !env_stack

      module Item=struct
        let do_begin_env ()=
          D.structure:=follow (top !D.structure) (List.rev (List.hd !env_stack));
          D.structure:=newChildAfter (follow (top !D.structure) (List.rev (List.hd !env_stack)))
            (Node { empty with
              node_tags=("item","")::("structural","")::empty.node_tags;
              node_env=(incr_counter "enumerate")
            })
        let do_end_env()=()
      end

      let do_end_env ()=
        let params parameters env a1 a2 a3 a4 a5 a6 line=
          let p=parameters env a1 a2 a3 a4 a5 a6 line in
            if not p.absolute && line.lineStart=0 then (
              let rec findMark w j=
                if j>=line.lineEnd then 0. else
                  if a1.(line.paragraph).(j) = Marker AlignmentMark then w else
                    let (_,ww,_)=box_interval a1.(line.paragraph).(j) in
                      findMark (w+.ww) (j+1)
              in
              let w=findMark 0. 0 in
                { p with
                    left_margin=p.left_margin-.w;
                    measure=p.measure+.w }
            ) else
              p
        in
        let comp complete mes a1 a2 a3 a4 line a6=
          if line.lineStart>0 then complete mes a1 a2 a3 a4 line a6 else (
            let rec findMark w j=
              if j>=Array.length a1.(line.paragraph) then 0. else
                if a1.(line.paragraph).(j) = Marker AlignmentMark then w else
                  let (_,ww,_)=box_interval a1.(line.paragraph).(j) in
                    findMark (w+.ww) (j+1)
            in
              complete { mes with normalMeasure=mes.normalMeasure+.findMark 0. 0 } a1 a2 a3 a4 line a6
          )
        in
        let is_first_par=ref false in
        let rec enumerate do_it t=match t with
            Node n when List.mem_assoc "item" n.node_tags && not do_it ->(
              is_first_par:=true;
              Node { n with children=IntMap.map (enumerate true) n.children }
            )
          | Node n when List.mem_assoc "item" n.node_tags -> Node n
          | Node n->Node { n with children=IntMap.map (enumerate do_it) n.children }
          | Paragraph p when do_it->
            let par_contents=
              if !is_first_par then (
                let item=bB (fun env->
                  let _,enum=try StrMap.find "enumerate" env.counters with Not_found->(-1),[0] in
                  let bb=boxify_scoped env (M.from_counter enum) in
                  let fix g= { g with drawing_min_width=g.drawing_nominal_width;
                    drawing_max_width=g.drawing_nominal_width }
                  in
                  let boxes=List.map (function Glue g->Glue (fix g) | Drawing g->Drawing (fix g) | x->x) bb in
                  boxes@[Marker AlignmentMark])
                in
                is_first_par:=false;
                item::p.par_contents
              ) else p.par_contents
            in
            Paragraph { p with
              par_parameters=params p.par_parameters;
              par_completeLine=comp p.par_completeLine;
              par_contents=par_contents
            }
          | _->t
        in
        D.structure:=follow (top !D.structure) (List.rev (List.hd !env_stack));
        let a,b= !D.structure in
        D.structure:=(enumerate false a,b);
        D.structure:=(up !D.structure);
        env_stack:=List.tl !env_stack
    end

    module Env_itemize =
      Enumerate(struct
                  let from_counter _ =
                    [
                      bB (fun env->[Drawing (
                                     let y=env.size/.4. in
                                     let x0=tiret_w env/.phi in
                                     let x1=tiret_w env-.x0 in
                                       { drawing_min_width=tiret_w env;
                                         drawing_nominal_width=tiret_w env;
					 drawing_width_fixed = true;
					 drawing_adjust_before = false;
                                         drawing_max_width=tiret_w env;
                                         drawing_y0=y; drawing_y1=y;
                                         drawing_break_badness=0.;
                                         drawing_states=IntSet.empty;
                                         drawing_badness=(fun _->0.);
                                         drawing_contents=(fun _->
                                                             [OutputCommon.Path
                                                                ({OutputCommon.default with
                                                                    OutputCommon.lineWidth=0.1},
                                                                 [[|[|x0;x1|],[|y;y;|]|]])
                                                             ]) }
                                   )])
                    ]
                end)

    type number_kind = 
      Arabic | AlphaLower | AlphaUpper | RomanLower | RomanUpper

    module type Enumerate_Pattern = sig
      val arg1 : number_kind * (string -> content list)
    end
 
    module Env_genumerate = functor (Pat:Enumerate_Pattern) ->
      Enumerate(struct
	let c, f = Pat.arg1
	let g = match c with
	    Arabic -> string_of_int
	  | AlphaLower -> Numerals.alphabetic ~capital:false
	  | AlphaUpper -> Numerals.alphabetic ~capital:true
	  | RomanLower -> Numerals.roman ~capital:false
	  | RomanUpper -> Numerals.roman ~capital:true
	let from_counter x =
	  let x = List.hd x + 1 in
	  f (g x)
      end)

    module Env_enumerate =
      Enumerate(struct
                  let from_counter x =
                    [ tT(string_of_int (List.hd x + 1));tT".";
                      bB (fun env->let w=env.size/.phi in [glue w w w])]
                end)

    module Env_abstract = struct

      let do_begin_env ()=
        D.structure:=newChildAfter !D.structure (Node empty);
        env_stack:=(List.map fst (snd !D.structure)) :: !env_stack

      let do_end_env ()=

        let stru,path=follow (top !D.structure) (List.rev (List.hd !env_stack)) in
        let p=find_last is_paragraph stru in
        let a,b=follow (stru,path) p in
        let a'=match a with
            Paragraph p->
              Paragraph { p with
                par_parameters=(fun a b c d e f g line->
                  let pp=(p.par_parameters a b c d e f g line) in
                  { pp with
                    min_lines_after=
                      if line.lineEnd>=Array.length b.(line.paragraph) then 2 else pp.min_lines_after;
                  });
              }
          | _->assert false
        in
        D.structure:=up_n (List.length p) (a',b);

        D.structure:=
          up (change_env !D.structure
                (fun x->
                  { x with
                    normalLeftMargin=x.normalLeftMargin+.(fst x.normalPageFormat)/.18.;
                    normalMeasure=x.normalMeasure-.2.*.(fst x.normalPageFormat)/.18.}));
        env_stack:=List.tl !env_stack

    end

    module type Theorem=sig
      val refType:string
      val counter:string
      val counterLevel:int
      val display:string->content list
    end
    module Env_gproof(X : sig val arg1 : content list end)=struct
      let do_begin_env ()=
        D.structure:=newChildAfter !D.structure (Node empty);
        env_stack:=(List.map fst (snd !D.structure)) :: !env_stack

      let do_end_env ()=
        let par a b c d e f g line={ (ragged_right a b c d e f g line) with not_first_line=true;
          min_lines_before=0;min_lines_after=2; }
        in
        let bad env a b c d e f g h i j k l m=if d.isFigure then infinity else
          Document.badness env a b c d e f g h i j k l m
        in

        let par_proof a b c d e f g line=
          { (parameters a b c d e f g line) with min_height_before=if line.lineStart=0 then a.lead else 0. }
        in
        let cont=X.arg1 in

        let rec add_proof t=match t with
            Node x->(
              try
                if List.mem_assoc "structure" x.node_tags then raise Not_found;
                let a,b=IntMap.min_binding x.children in
                Node { x with children=IntMap.add a (add_proof b) x.children }
              with
                  Not_found->
                    let a=try fst (IntMap.min_binding x.children) with _->1 in
                    let par,_=(paragraph ~par_env:(fun env->{env with par_indent=[]})
                                 ~parameters:par_proof cont)
                    in
                    Node { x with children=IntMap.add (a-1) par x.children}
            )
          | Paragraph p->
            Paragraph { p with
              par_env=(fun env->{(p.par_env env) with par_indent=[]});
              par_parameters=(fun a b c d e f g h->
                let p=p.par_parameters a b c d e f g h in
                if h.lineStart=0 then {p with min_height_before=a.lead} else p);
              par_contents=cont@p.par_contents
            }
          | _->raise Not_found
        in
        let retag t=match t with
            Node x,y->Node { x with node_tags=("structure","")::x.node_tags },y
          | _->assert false
        in
        D.structure:=(follow (top !D.structure) (List.rev (List.hd !env_stack)));
        D.structure:=up (retag (add_proof (fst !D.structure), snd !D.structure));
        newPar D.structure ~badness:bad Complete.normal par
          [bB (fun env->
                let w=env.size/.phi in
                  [Drawing (
                     drawing [OutputCommon.Path ({ OutputCommon.default with
                                                     OutputCommon.close=true;
                                                     OutputCommon.lineWidth=0.1 },
                                                 [OutputCommon.rectangle (0.,0.) (w,w)]
                                                )])
                  ]
             )];
        env_stack:=List.tl !env_stack
    end
    module Env_proof = Env_gproof (struct 
      let arg1 = italic [tT "Proof.";bB (fun env->let w=env.size in [glue w w w])]
    end)
    module Env_proofOf(X : sig val arg1 : content list end) = Env_gproof (struct 
      let arg1 = italic (X.arg1 @ [tT ".";bB (fun env->let w=env.size in [glue w w w])])
    end)

    module Proof = Env_proof (* probably useless, just for compatibility *)

    module Make_theorem=functor (Th:Theorem)->struct

      let reference name=generalRef Th.refType name

      let do_begin_env ()=
        D.structure:=newChildAfter !D.structure (Node empty);
        env_stack:=(List.map fst (snd !D.structure)) :: !env_stack

      let do_end_env ()=
        let rec last_par=function
            Paragraph p->
              Paragraph { p with
                            par_parameters=(fun a b c d e f g line->
                              let pp=(p.par_parameters a b c d e f g line) in
                              { pp with
                                min_lines_after=
                                  if line.lineEnd>=Array.length b.(line.paragraph) then 2 else pp.min_lines_after;
                              });
              }
          | Node n->(try
                       let k0,a0=IntMap.max_binding n.children in
                       Node { n with children=IntMap.add k0 (last_par a0) n.children }
                     with Not_found -> Node n)
          | x -> x
        in
        let stru,path=follow (top !D.structure) (List.rev (List.hd !env_stack)) in



        let cont=
          Env (incr_counter ~level:Th.counterLevel Th.counter)::
            C (fun env->
              let lvl,num=try (StrMap.find Th.counter env.counters) with
                  Not_found -> -1,[0]
              in
              let _,str_counter=try
                                  StrMap.find "_structure" env.counters
                with Not_found -> -1,[0]
              in
              let sect_num=drop (max 1 (List.length str_counter - lvl+1))
                str_counter
              in
              Th.display (String.concat "." (List.map (fun x->string_of_int (x+1)) ((List.rev sect_num)@num)))
            )::
            [tT " "]
        in
        let rec add_name t=match t with
            Node x->(
              try
                let a,b=IntMap.min_binding x.children in
                match b with
                    Node y when List.mem_assoc "structure" y.node_tags->raise Not_found
                  | _->Node { x with children=IntMap.add a (add_name b) x.children }
              with
                  Not_found->
                    let a=try fst (IntMap.min_binding x.children) with _->1 in
                    let par,_=paragraph ~par_env:(fun env->{env with par_indent=[]}) cont in
                    Node { x with children=IntMap.add (a-1) par x.children}
            )
          | Paragraph p->
            Paragraph { p with
              par_env=(fun env->{(p.par_env env) with par_indent=[]});
              par_parameters=(fun a b c d e f g h->
                let p=p.par_parameters a b c d e f g h in
                if h.lineStart=0 then {p with min_lines_before=2} else p);
              par_contents=cont@p.par_contents
            }
          | _->raise Not_found
        in
        let retag t=match t with
            Node x,y->Node { x with node_tags=("structure","")::x.node_tags },y
          | _->assert false
        in
	D.structure := up (retag (last_par (add_name stru),path));
        env_stack:=List.tl !env_stack
    end


    module Env_title=struct
      let title ()=
        match fst (top !D.structure) with
            Node n->n.displayname
          | _->[]

      let do_begin_env ()=
        env_stack:=(List.map fst (snd !D.structure)) :: !env_stack


      let do_end_env ()=
        D.structure:=
          (match fst (top !D.structure) with
              Node n->
                Node { n with node_tags=("title already typeset","")::n.node_tags },[]
            | x->x,[]);
	D.structure :=follow (top !D.structure) (List.rev (List.hd !env_stack));
        env_stack:=List.tl !env_stack
    end



    module Output(M:OutputPaper.Driver)=struct
      (** Output routines. An output routine is just a functor taking a driver module *)
      open OutputPaper
      open OutputCommon
      module type Driver=OutputPaper.Driver
      type output=unit
      (* { *)
      (*   mutable format:Box.box array array->(Document.tree*Document.cxt) array-> *)
      (*                  Box.drawingBox array->(Document.tree*Document.cxt) array-> *)
      (*                  (Layout.parameters*Layout.line) list -> OutputPaper.page; *)
      (*   mutable pageNumbers:OutputPaper.page->environment->int->unit *)
      (* } *)
      let outputParams=()
      let max_iterations=ref 3

      let basic_output _ tree defaultEnv file=
        let rec resolve i env0=
          Printf.printf "Compilation %d\n" i; flush stdout;
          let fixable=ref false in
          let env1,fig_params,params,new_page_list,new_line_list,compl,badness,paragraphs,paragraph_trees,figures,figure_trees=flatten env0 fixable tree in
          Printf.fprintf stderr "Début de l'optimisation : %f s\n" (Sys.time ());
          let (logs,opt_pages,figs',user')=TS.typeset
            ~completeLine:compl
            ~figure_parameters:fig_params
            ~figures:figures
            ~parameters:params
            ~new_page:new_page_list
            ~new_line:new_line_list
            ~badness:badness
            paragraphs
          in
          Printf.fprintf stderr "Fin de l'optimisation : %f s\n" (Sys.time ());
          let env, reboot=update_names env1 figs' user' in
          let env=reset_counters env in
          if i < !max_iterations-1 && reboot && !fixable then (
            resolve (i+1) env
          ) else (

            List.iter (fun x->Printf.fprintf stderr "%s\n" (Typography.TypoLanguage.message x)) logs;

            let positions=Array.make (Array.length paragraphs) (0,0.,0.) in
            let pages=Array.make (Array.length opt_pages) { pageFormat=0.,0.;pageContents=[] } in
            let par=ref (-1) in
            let crosslinks=ref [] in (* (page, link, destination) *)
            let crosslink_opened=ref false in
            let destinations=ref StrMap.empty in
            let urilinks=ref None in

            let continued_link=ref None in

            let draw_page i (layout,p)=
              let rec page_level lo=
                match snd lo with
                    _::_::_->page_level (frame_up lo)
                  | _->lo
              in
              let lo=fst (page_level layout) in
              pages.(i)<-{ pageFormat=(lo.frame_x1-.lo.frame_x0,lo.frame_y1-.lo.frame_y0);
                           pageContents=[] };

              let page=pages.(i) in

              let endlink cont=
                continued_link:=None;
                if !crosslink_opened then (
                  let rec link_contents u l=match l with
                      []->[]
                    | (Link h)::s->(
                      if cont then continued_link:=Some (Link h);
                      let x0,y0,x1,y1=bounding_box u in
                      Link { h with
                        link_x0=x0;link_y0=y0;
                        link_x1=x1;link_y1=y1;
                        link_contents=List.rev u
                      }
                    )::s
                    | h::s->link_contents (h::u) s
                  in
                  page.pageContents<-link_contents [] page.pageContents;
                  crosslink_opened:=false;
                )
              in

              (match !continued_link with
                  None->()
                | Some l->(
                  page.pageContents<-l::page.pageContents;
                  crosslink_opened:=true;
                  continued_link:=None
                )
              );

              let footnotes=ref [] in
              let footnote_y=ref (-.infinity) in
              let pp=Array.of_list p in
              let w,h=page.pageFormat in
              (* Affichage des frames (demouchage) *)
              let h=Hashtbl.create 100 in

              for j=0 to Array.length pp-1 do
                let param=pp.(j).line_params
                and line=pp.(j).line in

                (* Affichage des frames (demouchage) *)
                let rec draw_frames (t,cxt)=
                  if cxt<>[] then (
                    let r=(t.frame_x0,t.frame_y0,t.frame_x1,t.frame_y1) in
                    if not (Hashtbl.mem h r) then (
                      Hashtbl.add h r ();
                      page.pageContents<-Path (default,[rectangle (t.frame_x0,t.frame_y0) (t.frame_x1,t.frame_y1)])::page.pageContents;
                    );
                    draw_frames (Box.frame_up (t,cxt))
                  )
                in
                if env.show_frames then draw_frames line.layout;
                (* * *)


                if pp.(j).line.isFigure then (
                  let fig=figures.(pp.(j).line.lastFigure) in
                  let y=
                    if j>0 && j<Array.length pp-1 then
                      let milieu=
                        (pp.(j-1).line.height+.fst (line_height paragraphs figures pp.(j-1).line)
                         +.(pp.(j+1).line.height+.snd (line_height paragraphs figures pp.(j+1).line)))/.2.
                      in
                      milieu-.(fig.drawing_y1+.fig.drawing_y0)/.2.
                    else
                      pp.(j).line.height
                  in
	          if env.show_boxes then
                    page.pageContents<- Path ({OutputCommon.default with close=true;lineWidth=0.1 },
                                              [rectangle (param.left_margin,y+.fig.drawing_y0)
                                                  (param.left_margin+.fig.drawing_nominal_width,
                                                   y+.fig.drawing_y1)]) :: page.pageContents;
                  page.pageContents<- (List.map (translate ((fst pp.(j).line.layout).frame_x0+.param.left_margin) y)
                                         (fig.drawing_contents fig.drawing_nominal_width))
                  @ page.pageContents;

                ) else if line.paragraph<Array.length paragraphs then (

                  if line.paragraph<> !par then (
                    par:=line.paragraph;
                    positions.(!par)<-
                      (i,0.,
                       line.height +. phi*.snd (line_height paragraphs figures line))
                  );

                  let comp=compression paragraphs param line in
                  let rec draw_box x y box=
                    let lowy=y+.lower_y box in
                    let uppy=y+.upper_y box in
                    (match !urilinks with
                        None->()
                      | Some h->(
                        h.link_y0<-min h.link_y0 lowy;
                        h.link_y1<-max h.link_y1 uppy
                      ));
                    if !crosslink_opened then
                      (match !crosslinks with
                          []->()
                        | (_,h,_)::_->(
                          h.link_y0<-min h.link_y0 lowy;
                          h.link_y1<-max h.link_y1 uppy
                        ));
                    match box with
                        Kerning kbox ->(
                          let w=draw_box (x+.kbox.kern_x0) (y+.kbox.kern_y0) kbox.kern_contents in
                          w+.kbox.advance_width
                        )
                      | Hyphen h->(
                        (Array.fold_left (fun x' box->
                          let w=draw_box (x+.x') y box in
                          x'+.w) 0. h.hyphen_normal)
                      )
                      | GlyphBox a->(
                        page.pageContents<-translate x y (Glyph a):: page.pageContents;
                        a.glyph_size*.Fonts.glyphWidth a.glyph/.1000.
                      )
                      | Glue g
                      | Drawing g ->(
                        let w=g.drawing_min_width+.comp*.(g.drawing_max_width-.g.drawing_min_width) in
                        page.pageContents<- (List.map (translate x y) (g.drawing_contents w)) @ page.pageContents;
		        if env.show_boxes then
                          page.pageContents<- Path ({OutputCommon.default with close=true;lineWidth=0.1 }, [rectangle (x,y+.g.drawing_y0) (x+.w,y+.g.drawing_y1)]) :: page.pageContents;
                        w
                      )
                      | Marker (BeginURILink l)->(
                        let link={ link_x0=x;link_y0=y;link_x1=x;link_y1=y;uri=l;
                                   link_order=0;
                                   dest_page=(-1);dest_x=0.;dest_y=0.;is_internal=false;
                                   link_contents=[] }
                        in
                        crosslinks:=(i, link, l) :: !crosslinks;
                        crosslink_opened:=true;
                        page.pageContents<-Link link::page.pageContents;
                        0.
                      )
                      | Marker (BeginLink l)->(
                        let link={ link_x0=x;link_y0=y;link_x1=x;link_y1=y;uri=l;
                                   link_order=0;
                                   dest_page=Box.page line;dest_x=0.;dest_y=0.;is_internal=true;
                                   link_contents=[]
                                 }
                        in
                        crosslinks:=(i, link, l) :: !crosslinks;
                        crosslink_opened:=true;
                        page.pageContents<-Link link::page.pageContents;
                        0.
                      )
                      | Marker EndLink->(
                        endlink false;
                        0.
                      )
                      | Marker (Label l)->(
                        let y0,y1=line_height paragraphs figures line in
                        destinations:=StrMap.add l
                          (i,(fst line.layout).frame_x0+.param.left_margin,
                           y+.y0,y+.y1) !destinations;
                        0.
                      )
                      (* | Marker (Footnote (_,g))->( *)
                      (*   footnotes:= g::(!footnotes); *)
                      (*   footnote_y:=max !footnote_y (h-.topMargin-.param.page_height); *)
                      (*   0. *)
                      (* ) *)
                      | b->box_width comp b
                  in

                  (* Si un lien est commencé sur la ligne précédente,
                     le reprendre *)
                  if !crosslink_opened then
                    crosslinks:=(match !crosslinks with
                        []->[]
                      | (a,h,c)::s->
                        (a, { h with
                          link_x0=(fst line.layout).frame_x0+.param.left_margin;
                          link_x1=(fst line.layout).frame_x0+.param.left_margin;
                          link_y0=line.height;link_y1=line.height }, c)::(a,h,c)::s);

                  (* Écrire la page *)
                  let _=
                    fold_left_line paragraphs (fun x b->x+.draw_box x line.height b)
                      ((fst line.layout).frame_x0+.param.left_margin) line
                  in

                  (* Fermer les liens, et préparer la continuation sur
                     la prochaine ligne. *)
                  endlink true;
                  (match !continued_link with
                      None->()
                    | Some l->(
                      page.pageContents<-l::page.pageContents;
                      crosslink_opened:=true;
                      continued_link:=None
                    )
                  );
                )
              done;

              endlink true;

              (match !urilinks with
                  None->()
                | Some h->page.pageContents<-Link h::page.pageContents; urilinks:=None);
              ignore (
                List.fold_left (
                  fun y footnote->
                    page.pageContents<- (List.map (translate (env.normalLeftMargin) (y-.footnote.drawing_y1-.env.footnote_y))
                                           (footnote.drawing_contents footnote.drawing_nominal_width)) @ page.pageContents;
                    y-.(footnote.drawing_y1-.footnote.drawing_y0)
                ) !footnote_y !footnotes
              );
              if !footnotes<>[] then (
                page.pageContents<- (Path ({OutputCommon.default with lineWidth=0.01 }, [ [| [| env.normalLeftMargin;
                                                                                                env.normalLeftMargin+.env.normalMeasure*.(2.-.phi) |],
                                                                                           [| !footnote_y-.env.footnote_y;
                                                                                              !footnote_y-.env.footnote_y |] |] ]))::page.pageContents
              );


              let num=boxify_scoped defaultEnv [tT (string_of_int (i+1))] in
              let _,w,_=boxes_interval (Array.of_list num) in
              page.pageContents<-
                List.map (translate ((fst page.pageFormat-.w)/.2.) 30.)
                (draw_boxes env num)
                @page.pageContents;

              page.pageContents<-List.rev page.pageContents
            in
            for i=0 to Array.length pages-1 do draw_page i opt_pages.(i) done;

            let pages=Array.map (fun p->
              { p with
                pageContents=List.map (fun a->match a with
                    Link l when l.is_internal->(
                      try
                        let (p',x,y0,y1)=StrMap.find l.uri !destinations in
                        let dx0,dy0,dx1,dy1=bounding_box l.link_contents in
                        Link { l with dest_page=p'; dest_x=x; dest_y=y0+.(y1-.y0)*.phi;
                          link_x0=dx0;link_x1=dx1;
                          link_y0=dy0;link_y1=dy1
                             }
                      with
                          Not_found->a
                    )
                  | a->a
                ) p.pageContents
              }
            ) pages
            in

            (* List.iter (fun (p,link,dest)->try *)
            (*                                 let (p',x,y0,y1)=StrMap.find dest !destinations in *)
            (*                                 pages.(p).pageContents<-Link { link with dest_page=p'; dest_x=x; dest_y=y0+.(y1-.y0)*.phi } *)
            (*                                 ::pages.(p).pageContents *)
            (*   with *)
            (*       Not_found->() *)
            (* ) !crosslinks; *)


            M.output ~structure:(make_struct positions tree) pages file
          )
        in
        resolve 0 defaultEnv

      let output out_params structure defaultEnv file=
        basic_output out_params (postprocess_tree structure) defaultEnv file
    end
  end)

module MathFonts = struct
  let asana_font=Lazy.lazy_from_fun (fun ()->Typography.Fonts.loadFont (findFont "Asana-Math/Asana-Math.otf"))
  let asana name code = Maths.symbol ~name (Lazy.force asana_font) [code]

  let adjusted_asana_delimiters' name ls =
    match ls with
      [] -> assert false
    | x::ls ->
      let x0 = vkern_percent_under' x 0.166 in
      (fun envs st -> snd (x0 envs st)) :: List.map (fun x envs st -> 
	let center, _ = x0 envs st in vkern_center x center envs st) ls

  let adjusted_asana_delimiters name ls =
    adjusted_asana_delimiters' name (List.map (asana name) ls)

  (* This function is for asana delimiters that are not compatible with other
     Asana delimiters !!! *)
  let fix_asana_delimiters name ls =
    let rec map2 f l1 l2 = (* allows for longer second list *)
      match l1, l2 with
	[], _ -> []
      | (x1::l1), (x2::l2) -> f x1 x2::map2 f l1 l2
    in
    adjusted_asana_delimiters' name (map2
      (fun g g' -> vkern_as (asana name g)
	(asana "[" g'))
      ls [61;3340;3341;3342])

  let euler_font=Lazy.lazy_from_fun (fun ()->Typography.Fonts.loadFont (findFont "Euler/euler.otf"))
  let euler name code = Maths.symbol ~name (Lazy.force euler_font) [code]

  let adjusted_euler_delimiters name ls =
    let rec map2 f l1 l2 = (* allows for longer second list *)
      match l1, l2 with
	[], _ -> []
      | (x1::l1), (x2::l2) -> f x1 x2::map2 f l1 l2
    in
    adjusted_asana_delimiters' name (map2
      (fun g g' -> vkern_as (euler name g)
	(asana "[" g'))
      ls [61;3340;3341;3342])

  let ams_font=Lazy.lazy_from_fun (fun ()->Typography.Fonts.loadFont (findFont "AMS/ams.otf"))
  let ams name code = Maths.symbol ~name (Lazy.force ams_font) [code]
end




module MathsFormat=struct
    (* Symboles et polices de maths *)

    module MathFonts = MathFonts
    let mathsText t0=
      [Maths.Ordinary (Maths.noad (fun env st->boxify_scoped
        { env with size=env.size*.(Maths.env_style env.mathsEnvironment st).Mathematical.mathsSize }
        t0 ))]
    let mathcal a=[Maths.Scope(fun _ _-> Maths.Env (Euler.changeFont [Euler.Font `Cal]):: a)]
    let cal a=mathcal a
    let fraktur a=[Maths.Scope(fun _ _-> Maths.Env (Euler.changeFont [Euler.Font `Fraktur]) :: a)]
    let mathbf a=[Maths.Scope(fun _ _-> Maths.Env (fun env -> Euler.changeFont [Euler.Graisse `Gras] (envAlternative [] Bold env)) :: a)]
    let mathsc a=
      [Maths.Scope(fun _ _->
                     Maths.Env (fun env->envAlternative [] Caps env)::
                       Maths.Env (fun env->Maths.change_fonts env env.font)::
                       a
                  )]

    let bbFont=Lazy.lazy_from_fun (fun ()->Fonts.loadFont (findFont "AMS/ams.otf"))
    let mathbb a=[Maths.Scope (fun _ _->Maths.Env (fun env->Maths.change_fonts 
                                                     (change_font (Lazy.force bbFont) env) (Lazy.force bbFont))::a)]

    let mathrm a=[Maths.Scope(
                    fun _ _->Maths.Env (fun env->Maths.change_fonts env env.font)::a
                  )]

    let displayStyle a=[Maths.Scope(
      fun _ _->Maths.Env (fun env-> { env with mathStyle
        = Mathematical.Display })::a
    )]

    let mathsize alpha a=[Maths.Scope(
      fun _ _->Maths.Env (fun env-> { env with size=alpha })::a
    )]

    let mathSpace x =
      [Maths.Scope (fun env style ->
	let mathsEnv=Maths.env_style env.mathsEnvironment style in
	let x = x *. mathsEnv.Mathematical.mathsSize *. env.size in
	[Maths.Glue { drawing_min_width= x;
		    drawing_max_width= x;
		    drawing_y0=infinity; drawing_y1= -.infinity;
		    drawing_nominal_width= x;
		    drawing_width_fixed = true;
		    drawing_adjust_before = false;
		    drawing_contents=(fun _->[]);
                    drawing_states=IntSet.empty;
                    drawing_break_badness=0.;
		    drawing_badness=knuth_h_badness x }])]

    let oline a=
      [Maths.Ordinary
         (Maths.noad
            (fun envs st->
               let dr=draw_boxes envs (Maths.draw [envs] a) in
               let env=Maths.env_style envs.mathsEnvironment st in
               let (x0,y0,x1,y1)=OutputCommon.bounding_box_full dr in
               let drawn=(drawing ~offset:y0 dr) in
               let rul=(env.Mathematical.default_rule_thickness)*.env.Mathematical.mathsSize in
                 [Box.Drawing {
                    drawn with
                      drawing_y1=drawn.drawing_y1*.sqrt phi+.rul;
                      drawing_contents=
                      (fun w->
                         OutputCommon.Path ({OutputCommon.default with
			   OutputCommon.fillColor=Some envs.fontColor;
			   OutputCommon.strokingColor=Some envs.fontColor;
			   OutputCommon.lineWidth=rul},
                                            [[|[|x0;x1|],
                                               [|y1*.sqrt phi+.2.*.rul;y1*.sqrt phi+.2.*.rul|]|]])
                         ::drawn.drawing_contents w)
                  }]
            ))]

    let binomial a b=
      [Maths.Fraction { Maths.numerator=b; Maths.denominator=a;
                        Maths.line=(fun _ _->{OutputCommon.default with OutputCommon.fillColor=None;OutputCommon.strokingColor=None}) }]

    (* Une chirurgie esthétique de glyphs. Ce n'est sans doute pas très
       bien fait, et il faut kerner en haut. Un truc generique pour
       allonger toutes les flêches est à réfléchir *)

    let oRightArrow a=
      [Maths.Ordinary
         (Maths.noad
            (fun envs st->
               let boxes=(Maths.draw [envs] a) in
               let boxes_w=
                 (List.fold_left (fun w x->
                                    let _,w_x,_=box_interval x in
                                      w+.w_x) 0. boxes)
               in
               let dr=draw_boxes envs boxes in
               let (x0_,y0_,x1_,y1_)=OutputCommon.bounding_box_full dr in

               let env=Maths.env_style envs.mathsEnvironment st in
               let font=Lazy.force (env.Mathematical.mathsFont) in
               let utf8_arr={glyph_index=(Fonts.glyph_of_uchar font (UChar.chr 0x2192));
                             glyph_utf8="\033\146"} in
               let gl_arr=Fonts.loadGlyph font utf8_arr in
               let arr=Fonts.outlines gl_arr in
               let w1=List.fold_left (List.fold_left (fun y (v,_)->max y (max v.(0) v.(Array.length v-1)))) 0. arr in
               let y0,y1=List.fold_left (List.fold_left (fun (yy0,yy1) (_,v)->
                                                           let a,b=Bezier.bernstein_extr v in
                                                             min yy0 a, max yy1 b)) (0.,0.) arr in
               let size=envs.size*.env.Mathematical.mathsSize/.(1000.*.phi) in
               let space=env.Mathematical.default_rule_thickness in
               let arr'=
                 List.map (fun x->
                             Array.of_list (List.map (fun (u,v)->
                                                        Array.map (fun y->if y>=w1/.4. then (y*.size)+.(max 0. (x1_-.w1*.size)) else y*.size) u,
                                                        Array.map (fun y->y*.size-.y0+.y1_+.space) v
                                                     ) x)) arr
               in
                 [Box.Drawing {
                    drawing_nominal_width=max (w1*.size) boxes_w;
                    drawing_min_width=max (w1*.size) boxes_w;
                    drawing_max_width=max (w1*.size) boxes_w;
		    drawing_width_fixed = true;
		    drawing_adjust_before = false;
                    drawing_y0=y0_;
                    drawing_y1=y1_+.space-.(y0+.y1)*.size;
                    drawing_badness=(fun _->0.);
                    drawing_break_badness=0.;
                    drawing_states=IntSet.empty;
                    drawing_contents=
                      (fun w->
                         OutputCommon.Path ({OutputCommon.default with
                                               OutputCommon.strokingColor=None;
                                               OutputCommon.fillColor=Some envs.fontColor
                                            },arr')
                         ::(List.map (OutputCommon.translate (max 0. ((w1*.size-.x1_)/.2.)) 0.) dr))
                  }]
            ))]

    let oLeftArrow a=
      [Maths.Ordinary
         (Maths.noad
            (fun envs st->
               let boxes=(Maths.draw [envs] a) in
               let boxes_w=
                 (List.fold_left (fun w x->
                                    let _,w_x,_=box_interval x in
                                      w+.w_x) 0. boxes)
               in
               let dr=draw_boxes envs boxes in
               let (x0_,y0_,x1_,y1_)=OutputCommon.bounding_box_full dr in

               let env=Maths.env_style envs.mathsEnvironment st in
               let font=Lazy.force (env.Mathematical.mathsFont) in
               let utf8_arr={glyph_index=(Fonts.glyph_of_uchar font (UChar.chr 0x2190));
                             glyph_utf8="\033\144"} in
               let gl_arr=Fonts.loadGlyph font utf8_arr in
               let arr=Fonts.outlines gl_arr in
               let w1=List.fold_left (List.fold_left (fun y (v,_)->max y (max v.(0) v.(Array.length v-1)))) 0. arr in
               let y0,y1=List.fold_left (List.fold_left (fun (yy0,yy1) (_,v)->
                                                           let a,b=Bezier.bernstein_extr v in
                                                             min yy0 a, max yy1 b)) (0.,0.) arr in
               let size=envs.size*.env.Mathematical.mathsSize/.(1000.*.phi) in
               let space=env.Mathematical.default_rule_thickness in
               let arr'=
                 List.map (fun x->
                             Array.of_list (List.map (fun (u,v)->
                                                        Array.map (fun y->if y>=w1*.0.75 then (y*.size)+.(max 0. (x1_-.w1*.size)) else y*.size) u,
                                                        Array.map (fun y->y*.size-.y0+.y1_+.space) v
                                                     ) x)) arr
               in
                 [Box.Drawing {
                    drawing_nominal_width=max (w1*.size) boxes_w;
                    drawing_min_width=max (w1*.size) boxes_w;
                    drawing_max_width=max (w1*.size) boxes_w;
		    drawing_width_fixed = true;
		    drawing_adjust_before = false;
                    drawing_y0=y0_;
                    drawing_y1=y1_+.space-.(y0+.y1)*.size;
                    drawing_badness=(fun _->0.);
                    drawing_break_badness=0.;
                    drawing_states=IntSet.empty;
                    drawing_contents=
                      (fun w->
                         OutputCommon.Path ({OutputCommon.default with
                                               OutputCommon.strokingColor=None;
                                               OutputCommon.fillColor=Some envs.fontColor
                                            },arr')
                         ::(List.map (OutputCommon.translate (max 0. ((w1*.size-.x1_)/.2.)) 0.) dr))
                  }]
            ))]

    let vec = oRightArrow

    let cev = oLeftArrow
        (*******************************************************)

end
