#ifndef SUBTITLELANGUAGE_H
#define SUBTITLELANGUAGE_H

#include <string>
#include <boost/json.hpp>

namespace Nickvision::TubeConverter::Shared::Models
{
    /**
     * @brief A model of a subtitle language.
     */
    class SubtitleLanguage
    {
    public:
        /**
         * @brief Constructs a SubtitleLanguage.
         * @param language The language of the subtitle
         * @param isAutoGenerated Whether or not the subtitle is auto-generated
         */
        SubtitleLanguage(const std::string& language, bool isAutoGenerated = false);
        /**
         * @brief Constructs a SubtitleLanguage.
         * @param json The JSON object to construct the SubtitleLanguage from
         */
        SubtitleLanguage(boost::json::object json);
        /**
         * @brief Gets the language of the subtitle.
         * @return The language of the subtitle
         */
        const std::string& getLanguage() const;
        /**
         * @brief Gets whether or not the subtitle is auto-generated.
         * @return True if auto-generated, else false
         */
        bool isAutoGenerated() const;
        /**
         * @brief Gets a string representation of the subtitle language.
         * @return A string representation of the subtitle language
         */
        std::string str() const;
        /**
         * @brief Converts the SubtitleLanguage to a JSON object.
         * @return The JSON object
         */
        boost::json::object toJson() const;
        /**
         * @brief Compares two SubtitleLanguages via ==.
         * @param other The other SubtitleLanguage to compare
         * @return True if this == other, false otherwise
         */
        bool operator==(const SubtitleLanguage& other) const;
        /**
         * @brief Compares two SubtitleLanguages via !=.
         * @param other The other SubtitleLanguage to compare
         * @return True if this != other, false otherwise
         */
        bool operator!=(const SubtitleLanguage& other) const;
        /**
         * @brief Compares two SubtitleLanguages via <.
         * @param other The other SubtitleLanguage to compare
         * @return True if this < other, false otherwise
         */
        bool operator<(const SubtitleLanguage& other) const;
        /**
         * @brief Compares two SubtitleLanguages via >.
         * @param other The other SubtitleLanguage to compare
         * @return True if this > other, false otherwise
         */
        bool operator>(const SubtitleLanguage& other) const;

    private:
        std::string m_language;
        bool m_isAutoGenerated;
    };
}

#endif //SUBTITLELANGUAGE_H