#define _XOPEN_SOURCE

#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <stdexcept>
#include <utility>

#include "libpalm/File.h"
#include "libflatfile/Database.h"
#include "libflatfile/Factory.h"
#include "clp.h"
#include "strop.h"
#include "infofile.h"
#include "csvfile.h"

extern std::ostream* err;

StrOps::string_list_t DataFile::CSVFile::line2array(std::istream& f,
													std::string line,
													const DataFile::CSVConfig& state,
													int linenum)
{
    std::ostringstream error;
	StrOps::string_list_t array;
    try {
        if (state.extended_csv_mode) {
            array = StrOps::str_to_array(line, state.field_sep, false, false);
        } else {
            array = StrOps::csv_to_array(line, state.field_sep[0], state.quoted_string);
		}
    } catch (const StrOps::csv_parse_error& e) {
        error << "csv file" << ':' << linenum << ": " << e.what() << std::endl;
        throw CLP::parse_error(error.str());
    } catch (const StrOps::csv_unterminated_quote& e) {


		std::string line2(StrOps::readline(f));
		++linenum;
        if (!f)
            throw StrOps::csv_unterminated_quote("unterminated quotes");
		line2 = StrOps::strip_back(line2, "\r\n");
		line2 = line + "\n" + line2;

		array = line2array(f, line2, state, linenum);
	}
	return array;
}

void DataFile::CSVFile::read(PalmLib::FlatFile::Database& db,
            const DataFile::CSVConfig& state)
{
    std::ostringstream error;
    if (m_FileName == std::string("stdIO")) {
        read(std::cin, db, state);
    } else {
        std::ifstream f(m_FileName.c_str());
        if (!f) {
            error << m_FileName << " not found\n";
            throw CLP::parse_error(error.str());
        }
        read(f, db, state);
        f.close();
    }
}

void DataFile::CSVFile::read(std::istream& f, PalmLib::FlatFile::Database& db,
            const DataFile::CSVConfig& state)
{
    std::ostringstream error;
    int linenum = 0;
    while (1) {
        // Read the next line from the file.
        std::string line = StrOps::readline(f);
        ++linenum;
        if (!f)
            break;

        // Strip the trailing newline character.
        line = StrOps::strip_back(line, "\r\n");
        // Convert the text into an argv-style array.
        StrOps::string_list_t array;
        try {
			array = line2array(f, line, state, linenum);
		} catch (const StrOps::csv_unterminated_quote& e) {
            error << "csv file" << ':' << linenum << ": " << e.what() << std::endl;
            throw CLP::parse_error(error.str());
		}

        /* Make sure that the correct number of fields were received. */
        if (array.size() != db.getNumOfFields()) {
            error << "csv file" << ':' << linenum
                 << ": number of fields doesn't match\n";
            error << "csv file: nb fields found " << array.size() 
                 << " nb field requested " << db.getNumOfFields() << std::endl;
            throw CLP::parse_error(error.str());
        }

        PalmLib::FlatFile::Record record;
        for (unsigned i = 0; i < db.getNumOfFields(); ++i) {
            try {
                record.appendField(string2field(db.field_type(i), array[i], state));
            } catch (const StrOps::csv_parse_error& e) {
                error << "csv file" << ':' << linenum << ": " << e.what() << std::endl;
                throw CLP::parse_error(error.str());
            }
        }
        
        // Add the record to the database.
        db.appendRecord(record);
    }
}

void DataFile::CSVFile::write(const PalmLib::FlatFile::Database& db,
                const DataFile::CSVConfig& state)
{
    std::ostringstream error;
    if (m_FileName == std::string("stdIO")) {
        write(std::cout, db, state);
    } else {
        std::ofstream csv(m_FileName.c_str());
        if (!csv) {
            error << "unable to create\n";
            throw CLP::parse_error(error.str());
        }
        write(csv, db, state);
        csv.close();
    }
}

void DataFile::CSVFile::write(std::ostream& csv, const PalmLib::FlatFile::Database& db,
                const DataFile::CSVConfig& state)
{
    std::ostringstream error;
    int numRecords = db.getNumRecords();

    for (int i = 0; i < numRecords; ++i) {
        bool first = true;
        PalmLib::FlatFile::Record record = db.getRecord(i);

        for (unsigned int i = 0; i < db.getNumOfFields(); i++) {
            // If this is not the first field, output the field seperator.
            if (!first) csv << state.field_sep;
            first = false;

            // Do not output anything if the field is NULL.
#ifdef __LIBSTDCPP_PARTIAL_SUPPORT__
            if ((record.fields())[i].no_value)
                continue;
            // Output the field contents.
            csv << field2string((record.fields())[i], state);
#else
            if (record.fields().at(i).no_value)
                continue;
            // Output the field contents.
            csv << field2string(record.fields().at(i), state);
#endif

        }

        // Finally, output the newline to finish the line.
        csv << std::endl;
    }
}

PalmLib::FlatFile::Field DataFile::CSVFile::string2field(PalmLib::FlatFile::Field::FieldType type,
                const std::string fldstr,
                const DataFile::CSVConfig& state)
{
    std::ostringstream error;
    PalmLib::FlatFile::Field field;
    std::string format;

    switch (type) {
    case PalmLib::FlatFile::Field::STRING:
        field.type = PalmLib::FlatFile::Field::STRING;
        field.v_string = fldstr;
        break;

    case PalmLib::FlatFile::Field::BOOLEAN:
        field.type = PalmLib::FlatFile::Field::BOOLEAN;
        field.v_boolean = StrOps::string2boolean(fldstr);
        break;

    case PalmLib::FlatFile::Field::INTEGER:
        field.type = PalmLib::FlatFile::Field::INTEGER;
        StrOps::convert_string(fldstr, field.v_integer);
        break;

    case PalmLib::FlatFile::Field::FLOAT:
        field.type = PalmLib::FlatFile::Field::FLOAT;
        StrOps::convert_string(fldstr, field.v_float);
        break;

    case PalmLib::FlatFile::Field::DATE:
        field.type = PalmLib::FlatFile::Field::DATE;
        format = state.format_date;
        goto datetime_parse;

    case PalmLib::FlatFile::Field::TIME:
        field.type = PalmLib::FlatFile::Field::TIME;
        format = state.format_time;
        goto datetime_parse;

    case PalmLib::FlatFile::Field::DATETIME:
        field.type = PalmLib::FlatFile::Field::DATETIME;
        format = state.format_datetime;
        goto datetime_parse;

datetime_parse:
        struct tm time;
        if (!fldstr.empty()) {
#ifdef strptime
            if (!strptime(fldstr.c_str(), format.c_str(), &time))
#else
            if (!StrOps::strptime(fldstr.c_str(), format.c_str(), &time))
#endif
            {
                error << "invalid date in field ";
                throw CLP::parse_error(error.str());
            }
            field.v_date.month = time.tm_mon + 1;
            field.v_date.day = time.tm_mday;
            field.v_date.year = time.tm_year + 1900;
            field.v_time.hour = time.tm_hour;
            field.v_time.minute = time.tm_min;
        } else {
            field.v_date.month = 0;
            field.v_date.day = 0;
            field.v_date.year = 0;
            field.v_time.hour = 24;
            field.v_time.minute = 0;
        }
        break;

    case PalmLib::FlatFile::Field::NOTE:
        field.type = PalmLib::FlatFile::Field::NOTE;
        field.v_string = fldstr.substr(0,NOTETITLE_LENGTH - 1);
        field.v_note = fldstr;
        break;

    case PalmLib::FlatFile::Field::LIST:
        field.type = PalmLib::FlatFile::Field::LIST;
        field.v_string = fldstr;
        break;

    case PalmLib::FlatFile::Field::LINK:
        field.type = PalmLib::FlatFile::Field::LINK;
        field.v_integer = 0;
        field.v_string = fldstr;
        break;

    case PalmLib::FlatFile::Field::LINKED:
        field.type = PalmLib::FlatFile::Field::LINKED;
        field.v_string = fldstr;
        break;

    case PalmLib::FlatFile::Field::CALCULATED:
       field.type = PalmLib::FlatFile::Field::CALCULATED;
       field.v_string = fldstr;
       break;

    default:
        error << "unsupported field type ";
        throw CLP::parse_error(error.str());
    }

    return field;
}

std::string DataFile::CSVFile::field2string(PalmLib::FlatFile::Field field,
                const DataFile::CSVConfig& state)
{
    std::ostringstream fldstr; 

    switch (field.type) {
    case PalmLib::FlatFile::Field::STRING:
    case PalmLib::FlatFile::Field::LIST:
    case PalmLib::FlatFile::Field::LINK:
    case PalmLib::FlatFile::Field::LINKED:
    case PalmLib::FlatFile::Field::CALCULATED:
        if (state.extended_csv_mode || state.quoted_string) {
            fldstr << StrOps::quote_string(field.v_string, state.extended_csv_mode);
        } else {
            fldstr << field.v_string;
        }
        break;
                
    case PalmLib::FlatFile::Field::BOOLEAN:
        if (field.v_boolean)
            fldstr << "true";
        else
            fldstr << "false";
        break;

    case PalmLib::FlatFile::Field::INTEGER:
        fldstr << field.v_integer;
        break;

    case PalmLib::FlatFile::Field::FLOAT:
        fldstr << field.v_float;
        break;

    case PalmLib::FlatFile::Field::DATE:
        // Fill in the date and normalize the remaining fields.
        if ( field.v_date.month != 0) {
            struct tm tm;

            tm.tm_mon = field.v_date.month - 1;
            tm.tm_mday = field.v_date.day;
            tm.tm_year = field.v_date.year - 1900;
            tm.tm_hour = 0;
            tm.tm_min = 0;
            tm.tm_sec = 0;
            tm.tm_wday = 0;
            tm.tm_yday = 0;
            tm.tm_isdst = -1;
            (void) ::mktime(&tm);

             char buf[1024];

             // Clear out the output buffer.
             memset(buf, 0, sizeof(buf));
             // Convert and output the date using the format.
             ::strftime(buf, sizeof(buf), state.format_date.c_str(), &tm);
             fldstr << buf;
        }
        break;            

    case PalmLib::FlatFile::Field::TIME:
        {
            char buf[1024];
            struct tm tm;
            const struct tm * tm_ptr;
            time_t now;

            // Clear out the output buffer.
            memset(buf, 0, sizeof(buf));

            // Get a normalized structure and change the time.
            ::time(&now);
            tm_ptr = ::localtime(&now);
            memcpy(&tm, tm_ptr, sizeof(tm));
            tm.tm_hour = field.v_time.hour;
            tm.tm_min = field.v_time.minute;
            tm.tm_sec = 0;
            // Convert and output the time using the format.
            ::strftime(buf, sizeof(buf), state.format_time.c_str(), &tm);
            fldstr << buf;
        }
        break;

    case PalmLib::FlatFile::Field::DATETIME:
        {
            char buf[1024];
            struct tm tm;

            // Clear out the output buffer.
            memset(buf, 0, sizeof(buf));

            // Fill in the datetime and normalize the remaining fields.
            tm.tm_mon = field.v_date.month - 1;
            tm.tm_mday = field.v_date.day;
            tm.tm_year = field.v_date.year - 1900;
            tm.tm_hour = field.v_time.hour;
            tm.tm_min = field.v_time.minute;
            tm.tm_sec = 0;
            tm.tm_wday = 0;
            tm.tm_yday = 0;
            tm.tm_isdst = -1;
            (void) mktime(&tm);

            // Convert and output the time using the format.
            ::strftime(buf, sizeof(buf), state.format_datetime.c_str(), &tm);
            fldstr << buf;
        }
        break;

    case PalmLib::FlatFile::Field::NOTE:
        if (state.extended_csv_mode || state.quoted_string)
            fldstr << StrOps::quote_string(field.v_note, state.extended_csv_mode);
        else
            fldstr << field.v_note;
        break;
                
    default:
        // Leave the field blank if we don't support the field type.
        break;
    }
    return fldstr.str();
}
