/*
 * palm-db-tools: Read/write DB-format databases
 * Copyright (C) 1999 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#include <stdexcept>
#include <cstring>

#include "libsupport/strop.h"

#include "OldDB.h"

using namespace PalmLib::FlatFile;

bool PalmLib::FlatFile::OldDB::classify(PalmLib::Database& pdb)
{
    return (pdb.creator() == PalmLib::mktag('D','B','O','S'))
        && (pdb.type()    == PalmLib::mktag('D','B','9','9'));
}

bool PalmLib::FlatFile::OldDB::match_name(const std::string& name)
{
    return (name == "OldDB") || (name == "olddb");
}

PalmLib::FlatFile::OldDB::OldDB(PalmLib::Database& pdb)
    : Database("olddb", pdb), m_flags(0), m_top_visible_record(0)
{
    pi_char_t *p;
    size_t size;
    unsigned numFields, i;
    ListView lv;

    // The database title comes from the PalmOS database's name.
    title(pdb.name());

    // Retrieve the application information block.
    PalmLib::Block app_info = pdb.getAppInfoBlock();
    if (! (app_info.size() > 0))
        throw PalmLib::error("no app info block");
    p = app_info.data();
    size = app_info.size();

    // Parse the initial portion of the header.
    if (size < 6)
        throw PalmLib::error("database header is corrupt");
    m_flags = get_short(p);
    p += 2;
    m_top_visible_record = get_short(p);
    p += 2;
    numFields = get_short(p);
    p += 2;

    // Make sure that we have enough header left for the field info.
    if (size < 6 + (numFields * (2 + 32 + 2 + 2)))
        throw PalmLib::error("database header is corrupt");

    for (i = 0; i < numFields; ++i) {
        Field::FieldType type;
        char name[32];

        // Extract the type from the descriptor.
        switch (get_short(p)) {
        case 0:
            type = Field::STRING;
            break;
        case 1:
            type = Field::BOOLEAN;
            break;
        case 2:
            type = Field::INTEGER;
            break;
        default:
            throw PalmLib::error("database header is corrupt: unknown field type");
        }

        // Extract the name from the descriptor.
        strncpy(name, (char *) (p + 2), sizeof(name));
        name[sizeof(name)-1] = '\0';

        // Extract the width from the descriptor.
        lv.push_back(ListViewColumn(i, get_short(p + 2 + 32 + 2)));

        // Add this field to the schema.
        appendField(name, type);

        // Advance to the next field descriptor.
        p += (2 + 32 + 2 + 2);
    }

    // Add the constructed list view as the only list view.
    appendListView(lv);

    // Extract the records.
    for (i = 0; i < pdb.getNumRecords(); ++i) {
        PalmLib::Record data = pdb.getRecord(i);
        Record rec;

        p = data.raw_data();
        for (unsigned j = 0; j < getNumOfFields(); ++j) {
            Field f;

            f.type = field_type(j);
            switch (field_type(j)) {
            case Field::STRING:
                f.v_string = std::string((char *) p);
                p += strlen((char *) p) + 1;
                break;
            case Field::BOOLEAN:
                f.v_boolean = (*p) ? true : false;
                p += 1;
                break;
            case Field::INTEGER:
                f.v_integer = get_long(p);
                p += 4;
                break;
            default:
                throw PalmLib::error("invalid field type");
            }
            
            rec.appendField(f);
        }

        appendRecord(rec);
    }
}  

void PalmLib::FlatFile::OldDB::outputPDB(PalmLib::Database& pdb) const
{
    unsigned i;
    size_t size;

    // Let the superclass have a chance.
    SUPERCLASS(PalmLib::FlatFile, Database, outputPDB, (pdb));

    // Set the database's type and creator.
    pdb.type(PalmLib::mktag('D','B','9','9'));
    pdb.creator(PalmLib::mktag('D','B','O','S'));

    // Allocate a temporary buffer for the block.
    PalmLib::Block appinfo(3 * (2) + getNumOfFields() * (2 + 32 + 2 + 2));
    PalmLib::Block::pointer p = appinfo.data();

    // Write the fixed header information into the block.
    PalmLib::set_short(p, m_flags); p += 2;
    PalmLib::set_short(p, m_top_visible_record); p += 2;
    PalmLib::set_short(p, getNumOfFields()); p += 2;

    // Extract the single list view so the widths can be packed.
    PalmLib::FlatFile::ListView lv = getListView(0);
    PalmLib::FlatFile::ListView::const_iterator iter = lv.begin();
 
    for (i = 0; i < getNumOfFields(); ++i, ++iter) {
        // Pack the field type.
        switch (field_type(i)) {
        case PalmLib::FlatFile::Field::STRING:
        default:
            PalmLib::set_short(p, 0);
            break;

        case PalmLib::FlatFile::Field::BOOLEAN:
            PalmLib::set_short(p, 1);
            break;

        case PalmLib::FlatFile::Field::INTEGER:
            PalmLib::set_short(p, 2);
            break;
        }

        // Advance to the field name position.
        p += 2;

        // Pack the field name.
        memset(p, 0, 32);
        strncpy((char *) p, field_name(i).c_str(), 31);
        p += 32;

        // Pack the length (unused in all 0.2.x versions).
        PalmLib::set_short(p, 32);
        p += 2;

        // Pack the column width.
        PalmLib::set_short(p, (*iter).width);
        p += 2;
    }

    // Put the app info block into place.
    pdb.setAppInfoBlock(appinfo);

    // Output each record to the PalmOS database.
    for (i = 0; i < getNumRecords(); ++i) {
        PalmLib::FlatFile::Record rec = getRecord(i);

        // Determine the final packed size of this record.
        size = 0;
        for (unsigned int j = 0; j < getNumOfFields(); j++) {
#ifdef __LIBSTDCPP_PARTIAL_SUPPORT__
            switch (rec.fields()[j].type) {
#else
            switch (rec.fields().at(j).type) {
#endif
            case PalmLib::FlatFile::Field::STRING:
#ifdef __LIBSTDCPP_PARTIAL_SUPPORT__
                size += rec.fields()[j].v_string.length() + 1;
#else
                size += rec.fields().at(j).v_string.length() + 1;
#endif
                break;

            case PalmLib::FlatFile::Field::BOOLEAN:
                size += 1;
                break;

            case PalmLib::FlatFile::Field::INTEGER:
                size += 4;
                break;

            default:
                throw std::invalid_argument("unsupported field type");
            }
        }

        PalmLib::Record pdb_record(0, 0, size);
        PalmLib::Record::pointer p = pdb_record.data();

#ifdef __OLD_CPP_ISO__
        for (j = 0; j < getNumOfFields(); j++) {
#else
        for (unsigned int j = 0; j < getNumOfFields(); j++) {
#endif
#ifdef __LIBSTDCPP_PARTIAL_SUPPORT__
			const Field field = rec.fields()[j];
#else
			const Field field = rec.fields().at(j);
#endif
            switch (field.type) {
            case PalmLib::FlatFile::Field::STRING:
                memcpy((void *) p, field.v_string.c_str(),
                       field.v_string.length());
                p += field.v_string.length();
                *p++ = '\0';
                break;

            case PalmLib::FlatFile::Field::BOOLEAN:
                if (field.v_boolean)
                    *p++ = 1;
                else
                    *p++ = 0;
                break;

            case PalmLib::FlatFile::Field::INTEGER:
                PalmLib::set_long(p, field.v_integer);
                p += 4;
                break;

            default:
                throw std::invalid_argument("unsupported field type");
            }
        }

        // Pack the record into the PDB object.
        pdb.appendRecord(pdb_record);
    }
}

unsigned PalmLib::FlatFile::OldDB::getMaxNumOfFields() const
{
    return 0;
}

bool
PalmLib::FlatFile::OldDB::supportsFieldType(const Field::FieldType& type) const
{
    switch (type) {
    case PalmLib::FlatFile::Field::STRING:
    case PalmLib::FlatFile::Field::BOOLEAN:
    case PalmLib::FlatFile::Field::INTEGER:
        return true;
    default:
        return false;
    }
}

unsigned PalmLib::FlatFile::OldDB::getMaxNumOfListViews() const
{
    return 1;
}

void PalmLib::FlatFile::OldDB::doneWithSchema()
{
    // Let the superclass have a chance.
    SUPERCLASS(PalmLib::FlatFile, Database, doneWithSchema, ());

    if (getNumOfListViews() < 1)
        throw PalmLib::error("a list view must be specified");

    PalmLib::FlatFile::ListView lv = getListView(0);

    if (lv.size() != getNumOfFields())
        throw PalmLib::error("the list view must have the same number of columns as fields");

    PalmLib::FlatFile::ListView::const_iterator p = lv.begin();
    unsigned field = 0;

    for (; p != lv.end(); ++p, ++field) {
        const PalmLib::FlatFile::ListViewColumn& col = (*p);
        if (field != col.field) {
            throw PalmLib::error("the list view columns must be in the same order as the fields");
        }
    }
}

void PalmLib::FlatFile::OldDB::setOption(const std::string& name,
                                         const std::string& value)
{
    if (name == "find") {
        if (StrOps::string2boolean(value))
            m_flags &= ~(1);
        else
            m_flags |= 1;
    } else {
        SUPERCLASS(PalmLib::FlatFile, Database, setOption, (name, value));
    }
}

PalmLib::FlatFile::Database::options_list_t
PalmLib::FlatFile::OldDB::getOptions(void) const
{
    PalmLib::FlatFile::Database::options_list_t result;
    typedef PalmLib::FlatFile::Database::options_list_t::value_type value;

    result = SUPERCLASS(PalmLib::FlatFile, Database, getOptions, ());

    if (m_flags & 1)
        result.push_back(value("find", "true"));
    else
        result.push_back(value("find", "false"));

    return result;
}
