/*
 * palm-db-tools: CSV->PDB conversion tool
 * Copyright (C) 1998-2000 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <iostream>
#include <fstream>
#include <string>
#include <stdexcept>
#include <utility>

#ifndef WIN32
# if TIME_WITH_SYS_TIME
#  include <sys/time.h>
#  include <time.h>
# else
#  if HAVE_SYS_TIME_H
#   include <sys/time.h>
#  else
#   include <time.h>
#  endif
# endif
#else
# include <time.h>
#endif
 
#include "pdbtools.h"

using namespace PalmLib;

static std::string program;
static std::ostream* err = &std::cerr;

void usage(void)
{
    std::cout << "Usage: " << program <<  " [-v | -h | -i FILE | [others options]] [-m RULE] [CSV] [PDB]\n";
	std::cout << "  Database description options:\n";
    std::cout << "  -i FILE, --info=FILE    Read database metadata commands from FILE\n";
    std::cout << "  -o NAME=VALUE\n";
    std::cout << "     --option=NAME=VALUE  Set the database option NAME to VALUE\n";
    std::cout << "                          This option can be passed for each option to add\n";
    std::cout << "  -f SPEC, --field=SPEC   Add the field described by SPEC to the database.\n";
    std::cout << "                          SPEC format is \"NAME,TYPE,[DEFAULT VALUE]\"\n";
    std::cout << "                          This option can be passed for each field to add\n";
    std::cout << "  -b, --backup            Set pdb backup flag\n";
    std::cout << "  -e, --extended          Use extended mode when reading the CSV file\n";
    std::cout << "  -T TITLE, --title=TITLE Set the database's title\n";
    std::cout << "  -s SEP, --separator=SEP Change the CSV field separator from a comma to SEP\n";
    std::cout << "  -n FILE, --errors=FILE  Send all error messages to FILE\n";
    std::cout << "  -t TYPE, --type=TYPE    Set the target format. Choices are:\n";
    std::cout << "                            DB, db (DB 0.3.x format) (Default)\n";
    std::cout << "                            OldDB, olddb (DB 0.2.x format)\n";
    std::cout << "                            MobileDB, mdb (MobileDB format)\n";
    std::cout << "                            List, listdb, list (Freeware List format)\n";
    std::cout << "                            JFile3, jfile3, JF3, jf3 (JFile v3.x)\n";
    std::cout << "Note: Info option can not be present with the others of this options\n";
	std::cout << "  Action options:\n";
    std::cout << "  -m RULE, --merge=RULE    Merge the csv records to the pdb file records.\n";
    std::cout << "                           RULE could be:\n";
    std::cout << "                            Append (default): append all CSV records in the PDB\n";
    std::cout << "                            Diff: append the CSV records once in the PDB\n";
    std::cout << "                            Rebuild: create a new PDB with CSV file and\n";
    std::cout << "                                     Meta informations from old PDB\n";
	std::cout << "  Others options:\n";
    std::cout << "  -h, --help              Display this help screen\n";
    std::cout << "  -v, --version           Display the program version\n";
    std::cout << "Note: The data are read from the standard input,\n";
	std::cout << "      if the CSV file name is not present\n";
    std::cout << "      and if info option doesn't give a file with the \"csvfile\" element.\n";
    std::cout << "Note: If the pdb file name is not present and if info option is not empty,\n";
	std::cout << "      the pdb file name will be the info file name with .pdb as extention\n";
}

typedef std::vector< std::pair<std::string, std::string> > schema_args_t;

static void opt_callback(const std::string& key, const std::string& value,
                         void * data)
{
    schema_args_t* schema_args = reinterpret_cast<schema_args_t*> (data);
    (*schema_args).push_back(std::make_pair(key, value));
}

int main(int argc, char *argv[])
{
    std::string l_PDB_fname;

    schema_args_t schema_args;
    static const CLP::option_t options[] = {
        { "type", 't', "type", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
        { "help", 'h', "help", CLP::ARGUMENT_NONE, CLP::ACTION_MAP },
        { "version", 'v', "version", CLP::ARGUMENT_NONE, CLP::ACTION_MAP },
        { "errors", 'n', "errors", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
        { "sep", 's', "separator", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
        { "merge", 'm', "merge", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
        { "info", 'i', "info", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
        { "extended", 'e', "extended", CLP::ARGUMENT_NONE, CLP::ACTION_MAP },
        { "backup", 'b', "backup", CLP::ARGUMENT_NONE, CLP::ACTION_MAP },
        { "title", 'T', "title", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
        { "field", 'f', "field", CLP::ARGUMENT_STRING,
          CLP::ACTION_CALLBACK, &opt_callback, &schema_args },
        { "option", 'o', "option", CLP::ARGUMENT_STRING,
          CLP::ACTION_CALLBACK, &opt_callback, &schema_args },
        { 0 },
    };
    CLP::args_t args;
    CLP::option_map_t opts;
    DataFile::CSVConfig state;

    program = argv[0];
    try {
        CLP::parse(argc - 1, argv + 1, &options[0], opts, args);
    } catch (const CLP::missing_value_error& e) {
        *err << program << ": option `" << e.option_name()
             << "' takes an argument\n";
        usage();
        return 1;
    } catch (const CLP::invalid_option_error& e) {
        *err << program << ": unknown option `" << e.option_name()
             << "' was passed\n";
        usage();
        return 1;
    } catch (const CLP::value_present_error& e) {
        *err << program << ": option `" << e.option_name()
             << "' does not take an argument\n";
        usage();
        return 1;
    }

    // Redirect errors to a file. (Added to support a GUI wraparound on Win32.)
    if (opts.find("errors") != opts.end()) {
        std::ofstream* f = new std::ofstream(opts["errors"].c_str());
        if (!f) {
            *err << program << ": warning: unable to open error file `"
                 << opts["errors"] << "'\n";
        } else {
            err = f;
        }
    }

    // Display the usage screen if requested or no arguments present.
    if (argc == 1 || opts.find("help") != opts.end()) {
        usage();
        return 0;
    }

    // Display the program version if requested.
    if (opts.find("version") != opts.end()) {
        std::cout << "csv2pdb (" << PACKAGE_NAME << " " << PACKAGE_VERSION << ")\n";
        return 0;
    }

    //Initialize the library
    PDBTools::setConfigLib(err,".");

    // Create the flat-file database object that will receive the data.
    PalmLib::FlatFile::Factory factory;
    PalmLib::FlatFile::Database* flatfile;
	if (opts.find("info") != opts.end()) {
        if (opts.find("type") != opts.end() ||
            opts.find("extended") != opts.end() ||
            opts.find("title") != opts.end() ||
            opts.find("backup") != opts.end() ||
            opts.find("option") != opts.end() ||
            opts.find("field") != opts.end() ||
            opts.find("sep") != opts.end()) {
            *err << "-i option not supported with at least one of the others options.\n";
            return 1;
        }
        try {
            DataFile::InfoFile l_IfoFile(opts["info"]);
            l_IfoFile.read(state);
            PDBTools::setConfigLib(err,l_IfoFile.readPDBPath(), state);
            flatfile = factory.makeDatabase(l_IfoFile);
        }
        catch (const std::exception& e) {
            *err << opts["info"] << ": " << e.what() << std::endl;
            return 1;
        }
    } else {
	    if (opts.find("type") != opts.end()) {
	        flatfile = factory.makeDatabase(opts["type"]);
	        // If the factory couldn't create the object, tell the user.
	        if (!flatfile) {
	            *err << program << ": an unknown database type `" << opts["type"]
	                 << "' was specified\n";
	            return 1;
	        }
	    } else {
	        flatfile = factory.makeDatabase("db");
	        if (!flatfile) {
	            *err << program << ": an unknown database type 'db' was specified\n";
	            return 1;
	        }
		}    

	    // Determine the CSV separator to use.
	    if (opts.find("sep") != opts.end()) {
	        if (opts["sep"].length() != 1) {
	            *err << program << ": separator option must be a single character\n";
	            return 1;
	        }
	        state.field_sep = opts["sep"];
	    }

	    // Determine the CSV string format to use.
	    if (opts.find("extended") != opts.end()) {
	        state.extended_csv_mode = true;
	    }

	    // Determine the database title to use.
	    if (opts.find("title") != opts.end()) {
	        if (opts["title"].length() == 0) {
	            *err << program << ": title option must contain a string\n";
	            return 1;
	        }
	        flatfile->title(opts["title"]);
	    }

	    // Determine the database backup rigth to use.
	    if (opts.find("backup") != opts.end()) {
	        flatfile->setOption("backup", "true");
	    }

	    // Process all of the arguments that determine the schema and
	    // database options.
	    state.extended_csv_mode = false;
	    for (schema_args_t::const_iterator iter = schema_args.begin();
			         iter != schema_args.end(); ++iter) {
	        const std::string& key = (*iter).first;
	        const std::string& value = (*iter).second;

	        if (key == "option") {
	            std::string opt_name;
	            std::string opt_value;

	            // Determine if the option has an associated value.
	            std::string::size_type equals_pos = value.find("=", 0);
	            if (equals_pos != std::string::npos) {
	                opt_name = value.substr(0, equals_pos);
	                opt_value = value.substr(equals_pos + 1);
	            } else {
	                opt_name = value;
	                opt_value = "";
	            }

	            // Tell the database code about the option.
	            flatfile->setOption(opt_name, opt_value);
	        } else if (key == "field") {
	            StrOps::string_list_t array;

	            // Retrieve the name, field type, and optional width.
	            try {
	                array = StrOps::str_to_array(value, ",", false, false);
	            } catch (const std::exception& e) {
	                *err << program << ": bad -f option: " << e.what() << "\n";
	                return 1;
	            }

	            // Make sure the user gave the correct number of subarguments.
	            if (array.size() < 2 || array.size() > 3) {
    	            *err << program
	                     << ": -f option takes comma-delimited field descriptor\n";
	                return 1;
	            }

	            // Add this described field to the database.
	            try {
	                PalmLib::FlatFile::Field::FieldType type = StrOps::string2type(array[1]);

    	            if (array.size() == 2)
	                    flatfile->appendField(array[0], type);
	                else
    	                flatfile->appendField(array[0], type, array[2]);
	            } catch (const std::exception& e) {
	                *err << program << ": bad -f option: " << e.what() << "\n";
	                return 1;
	            } catch (const std::string& s) {
	                *err << program << ": bad -f option: " << s << "\n";
	                return 1;
	            }
	        }
    	}
	}

    // If an implict list view was given and there are also explicit
    // list views, then complain to the user.
    if (flatfile->getNumOfListViews() >= 1
        && state.implicit_listview.size() > 0) {
        *err << program << ": warning: ignoring implicit list view due to view directives\n";
    }

    // If no explicit list views were specified, then we need to check
    // for an implicit list view from widths given in field arguments.
    if (flatfile->getNumOfListViews() == 0) {
        if (state.implicit_listview.size() == 0) {
        } else {
            try {
                flatfile->appendListView(state.implicit_listview);
            } catch (const std::exception& e) {
                *err << program << ": " << e.what() << std::endl;
                return 1;
            }
        }
    }

    // Ensure that all 3 arguements were specified.
    switch (args.size()){
    case 2:
        // Assign better names to the arguments.
        state.csv_fname = std::string(args[0]);
        l_PDB_fname = std::string(args[1]);
    break;
    case 1:
        state.csv_fname = std::string("stdIO");
        l_PDB_fname = std::string(args[0]);
    break;
    case 0:
        state.csv_fname = std::string("stdIO");
        if (opts.find("info") != opts.end()) {
            std::string l_tempo = std::string(opts["info"]);
            l_PDB_fname = l_tempo.substr(0,l_tempo.rfind('.')) + std::string(".pdb");
            break;
        }
    default:    
        *err << program << ": 2 arguments were expected\n";
        usage();
        return 1;
    }

    PalmLib::File pdb(l_PDB_fname);
    if (opts.find("merge") == opts.end()) {
	    //we don't merge the records

        // Allow the database type code to perform final checks on the
        // schema. After this call, we promise the database type to not
        // change the schema or any other metadata.
        try {
            flatfile->doneWithSchema();
        } catch (const std::exception& e) {
            *err << program << ": " << e.what() << std::endl;
            return 1;
        }

        try {
            DataFile::CSVFile csvfile(state.csv_fname);
            csvfile.read(*flatfile, state);
        } catch (const std::exception& e) {
            *err << program << ": " << e.what() << std::endl;
            return 1;
        }

        // Create the PDB file object and set the creation time.
        pi_int32_t now = StrOps::get_current_time();
        pdb.creation_time(now);
        pdb.modification_time(now);
        pdb.backup_time(now);

    } else {
		// TR: turn the merge option value to lower case
		StrOps::lower(opts["merge"]);

        //we merge the records from the pdb database and the csv database
        if (l_PDB_fname.empty()) {
            *err << program << ": " << "pdb file not defined\n";
            return 1;
        }

        try {
            pdb.load();
        } catch (const PalmLib::error& e) {
            *err << l_PDB_fname << ": " << e.what() << std::endl;
            return 1;
        }
        pi_int32_t now = StrOps::get_current_time();
        pdb.modification_time(now);

        // Instantiate an object based on parameters.
        PalmLib::FlatFile::Factory factory;
        PalmLib::FlatFile::Database* db = 0;

		//we don't keep the previous informations about the database
        //the pdb file informations is used. We keep only the "state"
        //informations.
        delete flatfile;
        try {
            db = factory.makeDatabase(pdb);
            flatfile = factory.makeDatabase(pdb);
            if (!db)
                throw PalmLib::error("unable to determine database type");
        } catch (const PalmLib::error& e) {
            *err << l_PDB_fname << ": " << e.what() << std::endl;
            return 1;
		}

		// We replace the flatfile's records by the CSV records.
        flatfile->clearRecords();
        try {
            DataFile::CSVFile csvfile(state.csv_fname);
            csvfile.read(*flatfile, state);
    
        } catch (const PalmLib::error& e) {
            *err << state.csv_fname << ": " << e.what() << std::endl;
            return 1;
        }
		if (opts["merge"] == std::string("rebuild")) {
			pdb.clearRecords();
			unsigned int i;
            for (i = 0; i < flatfile->getNumRecords(); i++) {
				flatfile->getRecord(i).created(true);
            }
		} else if (opts["merge"] == std::string("delete")) {
			unsigned int i;
			for (i = 0; i < db->getNumRecords(); i++) {
                PalmLib::FlatFile::Record record = db->getRecord(i);
				unsigned int j;
				for (j = 0; j < flatfile->getNumRecords(); j++) {
					if (record.fields() == flatfile->getRecord(j).fields()) {
						//we set the delete flag directly in the PDB file
						pdb.getRecord(j).deleted(true);
					}
				}
			}
			//We must not append the records from the CSV file
			flatfile->clearRecords();
/*
		} else if (opts["merge"] == std::string("modify")) {
			if (flatfile->getNumRecords() != db->getNumRecords()) {
		        *err << state.csv_fname << " and " << l_PDB_fname << "must have the same number of records\n";
		        return 1;
			}

			unsigned int i;
			for (i = 0; i < db->getNumRecords(); i++) {
				if (record.fields() != flatfile->getRecord(j).fields()) {
					PalmLib::Record rec;
					flatfile->make_record(rec,
					//we set the delete flag directly in the PDB file
					pdb.setRecord(j).deleted(true);
				}
			}
			//We must not append the records from the CSV file
			flatfile->clearRecords();
*/
		} else {
			unsigned int i;
			if (opts["merge"] == std::string("diff")) {
				// We delete all records in flatfile (from the CSV file) which is
				// contained by the PDB file.
				for (i = 0; i < db->getNumRecords(); i++) {
	                PalmLib::FlatFile::Record record = db->getRecord(i);
					unsigned int j;
					for (j = 0; j < flatfile->getNumRecords(); j++) {
						if (record.fields() == flatfile->getRecord(j).fields()) {
							flatfile->deleteRecord(j);
						}
					}
				}
			}
			// We don't append all records from the flatfile to the PDB file,
			// because the recors are already inside the pdb.
            for (i = 0; i < flatfile->getNumRecords(); i++) {
				flatfile->getRecord(i).created(true);
                //db->appendRecord( flatfile->getRecord(i));
            }
		}
*err << "csv2pdb 6.\n";
    }

	// Save the AppInfo Block
    PalmLib::Block appinfo = pdb.getAppInfoBlock();

    // Tell the flat-file database to output to the PDB.
    //the records are appenned to the pdb file
    try {
        flatfile->outputPDB(pdb);
    } catch (...) {
        *err << program << ": error while copying into PDB object\n";
        return 1;
    }

	// Restore the AppInfo Block if merge mode
    if (opts.find("merge") != opts.end()) {
		pdb.setAppInfoBlock(appinfo);
	}

    // Save the database as a PDB file.
    try {
        pdb.save();
    } catch (const PalmLib::error& e) {
        *err << l_PDB_fname << ": " << e.what() << std::endl;
        return 1;
    } catch (const std::exception& e) {
        *err << l_PDB_fname << ": " << e.what() << std::endl;
    }
    return 0;
}
