#!/usr/bin/ruby -w
# -*- ruby -*-

require './env'
require './log'


class P4DeltaOptions
  include Log
  
  attr_accessor :confirm
  attr_accessor :execute
  attr_accessor :force_diff
  attr_accessor :package
  attr_accessor :quiet
  attr_accessor :verbose
  attr_accessor :version

  def initialize(package = "undef", version = "1.2.3.4")
    @package     = package
    @version     = version
    @verbose     = false        # whether to spew debugging output
    @quiet       = false        # whether to suppress warnings
    @confirm     = false        # whether to confirm remove commands
    @execute     = false        # whether to execute
    @force_diff  = false        # whether to do a forced-diff
  end

  def run
    read_rcfile
    read_environment_variable
    read_options
  end

  def read_environment_variable
    # process the environment variable
    if ENV["P4DELTAOPTS"]
      log "reading environment variable"
      options = ENV["P4DELTAOPTS"].split(/\s+/)
      while options.length > 0
        opt = options.shift
        Log.log "processing opt " + opt
        arg = options.shift
        process_option(opt, options)
      end
    end
  end

  def read_options
    log "reading options"

    while ARGV.length > 0
      arg = ARGV.shift
      log "processing arg #{arg}"
      
      if process_option(arg, ARGV)
        ARGV.unshift(arg)
        break 
      end
    end

    log "done reading options"
  end

  # Returns whether the value matches a true value, such as "yes", "true", or
  # "on".

  def to_boolean(value)
    [ "yes", "true", "on" ].include?(value.to_s.downcase)
  end

  def process_option(opt, args = nil)
    opt.gsub!(/^\-+/, "")

    case opt

    when "e", "execute"
      @execute = true
    
    when "h", "help"
      show_help
      exit

    when "f", "force"
      @force_diff = true

    when "q", "quiet"
      @quiet = true

    when "V", "verbose"
      @verbose = true

    when "v", "version"
      print @package, ", version ", @version, "\n"
      print "Written by Jeff Pace (jpace@incava.org).\n"
      print "Released under the Lesser GNU Public License.\n"
      exit 1

    else
      return true
    end
    
    return false
  end
  
  def read_rc_file(rc)
    log "reading rc file #{rc}"
    
    IO.readlines(rc).each do |line|
      line.sub!(/\s*#.*/, "")
      line.chomp!
      name, value = line.split(/\s*[=:]\s*/)
      next unless name && value

      case name
      when "execute"
        @execute = to_boolean(value)
      when "quiet"
        @quiet = to_boolean(value)
      when "verbose"
        @verbose = to_boolean(value)
      when /^force/
        @force_diff = to_boolean(value)
      end
    end
  end

  def read_rcfile
    # process the rc file
    if hd = home_directory
      rc = hd + "/.p4deltarc"
      Log.log "reading RC file: #{rc}"
      if File.exists?(rc)
        read_rc_file(rc)
      else
        log "no such file: #{rc}"
      end
    end
  end

  def show_help

    puts "USAGE"
    puts "    p4delta [options] directory..."
    puts ""
    puts "OPTIONS"
    puts "    -c, --confirm"
    puts "        Confirm deletions from Perforce. Valid only with the execute option."
    puts ""
    puts "    -e, --execute"
    puts "        Run the add and remove commands for the appropriate files."
    puts ""
    puts "    -f, --force"
    puts "        Compare all local files against those in Perforce."
    puts ""
    puts "    -h, --help"
    puts "        Display this help message."
    puts ""
    puts "    -q, --quiet"
    puts "        Run with minimum output."
    puts ""
    puts "    -v, --version"
    puts "        Display the version and exit."
    puts ""
    puts "    -V, --verbose"
    puts "        Run with maximum output."
    puts ""
    puts "See the man page for more information."
    puts ""

  end

end


if __FILE__ == $0
  dopt = CVSDeltaOptions.new
  dopt.run
end
