#ifndef OSMIUM_UTIL_CAST_HPP
#define OSMIUM_UTIL_CAST_HPP

/*

This file is part of Osmium (https://osmcode.org/libosmium).

Copyright 2013-2022 Jochen Topf <jochen@topf.org> and others (see README).

Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

*/

#include <osmium/util/compatibility.hpp>

#ifndef assert
# include <cassert>
#endif

#include <cstdint>
#include <limits>
#include <type_traits>

namespace osmium {

    // These functions are wrappers around static_cast<>() that call assert()
    // to check that there is no integer overflow happening before doing the
    // cast. There are several versions of this templated function here
    // depending on the types of the input and output. In any case, both input
    // and output have to be integral types. If the cast can't overflow, no
    // check is done.

    template <typename A, typename B>
    struct are_real_integers :
        std::integral_constant<bool,
            std::is_integral<A>::value &&
            std::is_integral<B>::value &&
            !std::is_same<A, bool>::value &&
            !std::is_same<B, bool>::value> {
    };

    template <typename T, typename F, typename std::enable_if<are_real_integers<T, F>::value && std::is_same<T, F>::value, int>::type = 0>
    OSMIUM_DEPRECATED inline T static_cast_with_assert(const F value) {
        return value;
    }

    template <typename T, typename F, typename std::enable_if<are_real_integers<T, F>::value && !std::is_same<T, F>::value && (sizeof(T) > sizeof(F)), int>::type = 0>
    OSMIUM_DEPRECATED inline T static_cast_with_assert(const F value) {
        return static_cast<T>(value);
    }

    template <typename T, typename F, typename std::enable_if<are_real_integers<T, F>::value && !std::is_same<T, F>::value && std::is_signed<T>::value == std::is_signed<F>::value && (sizeof(T) == sizeof(F)), int>::type = 0>
    OSMIUM_DEPRECATED inline T static_cast_with_assert(const F value) {
        return static_cast<T>(value);
    }

    template <typename T, typename F, typename std::enable_if<are_real_integers<T, F>::value && !std::is_same<T, F>::value && (sizeof(T) < sizeof(F)) && std::is_signed<T>::value && std::is_signed<F>::value, int>::type = 0>
    OSMIUM_DEPRECATED inline T static_cast_with_assert(const F value) {
        assert(value >= std::numeric_limits<T>::min() && value <= std::numeric_limits<T>::max());
        return static_cast<T>(value);
    }

    template <typename T, typename F, typename std::enable_if<are_real_integers<T, F>::value && !std::is_same<T, F>::value && (sizeof(T) <= sizeof(F)) && std::is_unsigned<T>::value && std::is_signed<F>::value, int>::type = 0>
    OSMIUM_DEPRECATED inline T static_cast_with_assert(const F value) {
        assert(value >= 0 && static_cast<typename std::make_unsigned<F>::type>(value) <= std::numeric_limits<T>::max());
        return static_cast<T>(value);
    }

    template <typename T, typename F, typename std::enable_if<are_real_integers<T, F>::value && !std::is_same<T, F>::value && (sizeof(T) < sizeof(F)) && std::is_unsigned<T>::value && std::is_unsigned<F>::value, int>::type = 0>
    OSMIUM_DEPRECATED inline T static_cast_with_assert(const F value) {
        assert(value <= std::numeric_limits<T>::max());
        return static_cast<T>(value);
    }

    template <typename T, typename F, typename std::enable_if<are_real_integers<T, F>::value && !std::is_same<T, F>::value && (sizeof(T) <= sizeof(F)) && std::is_signed<T>::value && std::is_unsigned<F>::value, int>::type = 0>
    OSMIUM_DEPRECATED inline T static_cast_with_assert(const F value) {
        assert(static_cast<int64_t>(value) <= static_cast<int64_t>(std::numeric_limits<T>::max()));
        return static_cast<T>(value);
    }

} // namespace osmium

#endif // OSMIUM_UTIL_CAST_HPP
