/* MatMult test */

#include <oski/common.h>
#include <oski/matcreate.h>
#include <oski/matmult.h>
#include "util/array_util.h"

/* ------------------------------------------------- */

#define A0_M 4    /* # of rows */
#define A0_N 5    /* # of cols */
#define A0_NNZ 6  /* # of stored non-zeros */
static oski_index_t A0_ptr[A0_M+1] = {1, 2, 3, 5, 7}; /* end row pointers for A0 */
static oski_index_t A0_ind[A0_NNZ] = {1, 2, 1, 3, 1, 3}; /* end column indices pointers for A0 */
static oski_value_t A0_val[A0_NNZ] = {4, 2, 10, 8, -10, 3}; /* end stored non-zero values for A0 */

#define x0_LEN 5   /* vector length */
#define x0_VW  3   /* vector width */
static oski_value_t x0_val[x0_LEN*x0_VW] = {-2, 4, 8, 5, -5, -10, 5, -4, 2, 5, 10, -3, -6, 10, 4}; /* data */

#define y0_LEN 4   /* vector length */
#define y0_VW  3   /* vector width */
static oski_value_t y0_val[y0_LEN*y0_VW] = {7, 4, -10, 4, 10, -5, -1, 7, 0, -6, -5, -3}; /* data */

/*
 *  BEGIN TEST 0
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_0( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {-50, -8, 92, -14, -70, 10, 26, -34, 96, 71, -22, -56 };
	oski_value_t alpha = 1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_0, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_0", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_0() */
/* END TEST 0 */

/*
 *  BEGIN TEST 1
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_1( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {8, -16, -32, -10, 10, 20, -20, -8, -96, -35, 52, 74 };
	oski_value_t alpha = -1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_1, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_1", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_1() */
/* END TEST 1 */

/*
 *  BEGIN TEST 2
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_2( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {49, 28, -70, 28, 70, -35, -7, 49, 0, -42, -35, -21 };
	oski_value_t alpha = 0;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_2, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_2", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_2() */
/* END TEST 2 */

/*
 *  BEGIN TEST 3
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_3( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {-112, 80, 304, 38, -150, -100, 148, 0, 672, 293, -324, -494 };
	oski_value_t alpha = 7;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_3, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_3", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_3() */
/* END TEST 3 */

/*
 *  BEGIN TEST 4
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_4( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {64, 164, 46, 43, -15, -80, 9, 13, 5, 35, 70, -21, -42, 70, 28 };
	oski_value_t alpha = 1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_4, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_4", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_4() */
/* END TEST 4 */

/*
 *  BEGIN TEST 5
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_5( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {-78, -136, 10, -8, -20, 10, 26, -41, 9, 0, 0, 0, 0, 0, 0 };
	oski_value_t alpha = -1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_5, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_5", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_5() */
/* END TEST 5 */

/*
 *  BEGIN TEST 6
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_6( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_6, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_6", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_6() */
/* END TEST 6 */

/*
 *  BEGIN TEST 7
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_7( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {-252, -372, 102, 21, -105, -60, 123, -159, 45, 45, 90, -27, -54, 90, 36 };
	oski_value_t alpha = -3;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_7, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_7", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_7() */
/* END TEST 7 */

/*
 *  BEGIN TEST 8
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_8( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {-8, 10, 20, 35, 16, -10, 8, -52, 32, -20, 96, -74 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_8, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_8", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_8() */
/* END TEST 8 */

/*
 *  BEGIN TEST 9
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_9( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {-20, -26, 20, -51, -56, 30, -4, 24, -32, 44, -76, 86 };
	oski_value_t alpha = -1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_9, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_9", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_9() */
/* END TEST 9 */

/*
 *  BEGIN TEST 10
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_10( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {-49, -28, 70, -28, -70, 35, 7, -49, 0, 42, 35, 21 };
	oski_value_t alpha = 0;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_10, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_10", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_10() */
/* END TEST 10 */

/*
 *  BEGIN TEST 11
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_11( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {-120, 48, 240, 248, 48, -40, 72, -472, 256, -112, 808, -568 };
	oski_value_t alpha = 8;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_11, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_11", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_11() */
/* END TEST 11 */

/*
 *  BEGIN TEST 12
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_12( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {-112, -40, -20, 8, -10, -12, -68, 13, -49, 0, 0, 0, 0, 0, 0 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_12, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_12", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_12() */
/* END TEST 12 */

/*
 *  BEGIN TEST 13
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_13( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {116, 32, 4, -18, 20, 32, 58, -5, 45, -10, -20, 6, 12, -20, -8 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_13, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_13", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_13() */
/* END TEST 13 */

/*
 *  BEGIN TEST 14
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_14( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {8, -16, -32, -20, 20, 40, -20, 16, -8, -20, -40, 12, 24, -40, -16 };
	oski_value_t alpha = 0;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_14, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_14", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_14() */
/* END TEST 14 */

/*
 *  BEGIN TEST 15
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_15( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {-786, -276, -132, 61, -75, -94, -471, 87, -341, 5, 10, -3, -6, 10, 4 };
	oski_value_t alpha = 7;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_15, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_15", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_15() */
/* END TEST 15 */

/*
 *  BEGIN TEST 16
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_16( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {-15, -44, 50, 4, 0, -1, 45, -139, 52, 50, 93, -115 };
	oski_value_t alpha = 1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_16, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_16", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_16() */
/* END TEST 16 */

/*
 *  BEGIN TEST 17
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_17( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {64, 72, -120, 24, 70, -34, -52, 188, -52, -92, -128, 94 };
	oski_value_t alpha = -1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_17, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_17", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_17() */
/* END TEST 17 */

/*
 *  BEGIN TEST 18
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_18( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {-56, -32, 80, -32, -80, 40, 8, -56, 0, 48, 40, 24 };
	oski_value_t alpha = 0;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_18, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_18", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_18() */
/* END TEST 18 */

/*
 *  BEGIN TEST 19
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_19( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {-5, -180, 150, 60, 100, -55, 215, -625, 260, 190, 415, -605 };
	oski_value_t alpha = 5;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_19, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_19", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_19() */
/* END TEST 19 */

/*
 *  BEGIN TEST 20
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_20( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {66, 32, 22, 30, -30, 76, 50, 17, 12, 30, 50, -28, -45, 60, 24 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_20, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_20", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_20() */
/* END TEST 20 */

/*
 *  BEGIN TEST 21
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_21( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {-64, -36, -30, -35, 35, -66, -55, -13, -14, -35, -60, 31, 51, -70, -28 };
	oski_value_t alpha = -1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_21, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_21", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_21() */
/* END TEST 21 */

/*
 *  BEGIN TEST 22
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_22( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {18, -36, -72, -45, 45, 90, -45, 36, -18, -45, -90, 27, 54, -90, -36 };
	oski_value_t alpha = 0;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_22, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_22", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_22() */
/* END TEST 22 */

/*
 *  BEGIN TEST 23
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_23( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {220, 52, -22, 35, -35, 338, 95, 95, 14, 35, 40, -51, -69, 70, 28 };
	oski_value_t alpha = 3;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_23, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_23", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_23() */
/* END TEST 23 */

/*
 *  BEGIN TEST 24
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_24( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {6, 16, 24, 52, -20, 0, -134, 102, 40, -18, 42, -124 };
	oski_value_t alpha = 1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_24, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_24", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_24() */
/* END TEST 24 */

/*
 *  BEGIN TEST 25
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_25( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {36, 8, -84, -28, 80, -30, 128, -60, -40, -18, -72, 106 };
	oski_value_t alpha = -1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_25, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_25", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_25() */
/* END TEST 25 */

/*
 *  BEGIN TEST 26
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_26( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {28, 16, -40, 16, 40, -20, -4, 28, 0, -24, -20, -12 };
	oski_value_t alpha = 0;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_26, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_26", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_26() */
/* END TEST 26 */

/*
 *  BEGIN TEST 27
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_27( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {-23, 100, 326, 424, -290, 55, -1195, 841, 360, -96, 433, -1083 };
	oski_value_t alpha = 9;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_27, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_27", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_27() */
/* END TEST 27 */

/*
 *  BEGIN TEST 28
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_28( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {-102, -12, -108, -25, 25, 10, -35, 33, -10, -25, -70, 3, -19, -50, -20 };
	oski_value_t alpha = 1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_28, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_28", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_28() */
/* END TEST 28 */

/*
 *  BEGIN TEST 29
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_29( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {118, -20, 44, -15, 15, 70, -5, -1, -6, -15, -10, 21, 67, -30, -12 };
	oski_value_t alpha = -1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_29, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_29", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_29() */
/* END TEST 29 */

/*
 *  BEGIN TEST 30
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_30( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {6, -12, -24, -15, 15, 30, -15, 12, -6, -15, -30, 9, 18, -30, -12 };
	oski_value_t alpha = 0;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_30, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_30", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_30() */
/* END TEST 30 */

/*
 *  BEGIN TEST 31
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_31( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {-4, 8, 16, 10, -10, -20, 10, -8, 4, 10, 20, -6, -12, 20, 8 };
	oski_value_t alpha = 0;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_31, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_31", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_31() */
/* END TEST 31 */

#define A32_M 4    /* # of rows */
#define A32_N 5    /* # of cols */
#define A32_NNZ 6  /* # of stored non-zeros */
static oski_index_t A32_ptr[A32_M+1] = {0, 0, 3, 4, 6}; /* end row pointers for A32 */
static oski_index_t A32_ind[A32_NNZ] = {0, 2, 4, 2, 0, 2}; /* end column indices pointers for A32 */
static oski_value_t A32_val[A32_NNZ] = {6, 1, 10, 3, 9, -10}; /* end stored non-zero values for A32 */

#define x32_LEN 5   /* vector length */
#define x32_VW  3   /* vector width */
static oski_value_t x32_val[x32_LEN*x32_VW] = {-6, 5, -7, -8, -4, -10, -1, -9, 5, 10, -5, -6, -3, 9, 4}; /* data */

#define y32_LEN 4   /* vector length */
#define y32_VW  3   /* vector width */
static oski_value_t y32_val[y32_LEN*y32_VW] = {-10, 3, -4, -4, -1, -10, 1, 7, 1, 10, 0, -10}; /* data */

/*
 *  BEGIN TEST 32
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_32( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	const oski_value_t y_true[] = {-90, 27, -36, -103, 102, -87, 6, 36, 24, 46, 135, -203 };
	oski_value_t alpha = 1;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_32, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_32", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_32() */
/* END TEST 32 */

/*
 *  BEGIN TEST 33
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_33( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	const oski_value_t y_true[] = {20, -6, 8, 75, -109, 17, 1, 13, -17, 24, -135, 133 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_33, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_33", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_33() */
/* END TEST 33 */

/*
 *  BEGIN TEST 34
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_34( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	const oski_value_t y_true[] = {60, -18, 24, 24, 6, 60, -6, -42, -6, -60, 0, 60 };
	oski_value_t alpha = 0;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_34, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_34", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_34() */
/* END TEST 34 */

/*
 *  BEGIN TEST 35
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_35( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	const oski_value_t y_true[] = {40, -12, 16, -51, 115, 43, -7, -55, 11, -84, 135, -73 };
	oski_value_t alpha = 1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_35, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_35", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_35() */
/* END TEST 35 */

/*
 *  BEGIN TEST 36
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_36( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	const oski_value_t y_true[] = {66, -6, -150, 0, 0, 0, -101, 20, 93, 0, 0, 0, -40, -10, -100 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_36, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_36", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_36() */
/* END TEST 36 */

/*
 *  BEGIN TEST 37
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_37( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	const oski_value_t y_true[] = {-60, 1, 157, 8, 4, 10, 102, -11, -98, -10, 5, 6, 43, 1, 96 };
	oski_value_t alpha = -1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_37, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_37", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_37() */
/* END TEST 37 */

/*
 *  BEGIN TEST 38
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_38( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	const oski_value_t y_true[] = {-48, 40, -56, -64, -32, -80, -8, -72, 40, 80, -40, -48, -24, 72, 32 };
	oski_value_t alpha = 0;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_38, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_38", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_38() */
/* END TEST 38 */

/*
 *  BEGIN TEST 39
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_39( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	const oski_value_t y_true[] = {-432, 17, 1085, 40, 20, 50, 712, -95, -676, -50, 25, 30, 295, 25, 680 };
	oski_value_t alpha = -7;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_39, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_39", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_39() */
/* END TEST 39 */

/*
 *  BEGIN TEST 40
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_40( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	const oski_value_t y_true[] = {30, -76, 9, -32, 3, 141, -30, 114, -3, -27, 15, -83 };
	oski_value_t alpha = 1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_40, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_40", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_40() */
/* END TEST 40 */

/*
 *  BEGIN TEST 41
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_41( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	const oski_value_t y_true[] = {-20, 73, -5, 36, -2, -131, 29, -121, 2, 17, -15, 93 };
	oski_value_t alpha = -1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_41, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_41", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_41() */
/* END TEST 41 */

/*
 *  BEGIN TEST 42
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_42( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	const oski_value_t y_true[] = {70, -21, 28, 28, 7, 70, -7, -49, -7, -70, 0, 70 };
	oski_value_t alpha = 0;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_42, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_42", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_42() */
/* END TEST 42 */

/*
 *  BEGIN TEST 43
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_43( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	const oski_value_t y_true[] = {-20, -195, -17, -140, -2, 313, -79, 419, 2, 29, 45, -359 };
	oski_value_t alpha = 3;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_43, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_43", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_43() */
/* END TEST 43 */

/*
 *  BEGIN TEST 44
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_44( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	const oski_value_t y_true[] = {-48, 28, -65, -40, -20, -50, 26, -122, 135, 50, -25, -30, 15, -55, 120 };
	oski_value_t alpha = 1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_44, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_44", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_44() */
/* END TEST 44 */

/*
 *  BEGIN TEST 45
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_45( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	const oski_value_t y_true[] = {48, -28, 65, 40, 20, 50, -26, 122, -135, -50, 25, 30, -15, 55, -120 };
	oski_value_t alpha = -1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_45, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_45", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_45() */
/* END TEST 45 */

/*
 *  BEGIN TEST 46
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_46( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	const oski_value_t y_true[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_46, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_46", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_46() */
/* END TEST 46 */

/*
 *  BEGIN TEST 47
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_ROWMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_47( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_ROWMAJ, x32_VW );
	const oski_value_t y_true[] = {66, -31, 95, 40, 20, 50, -57, 199, -245, -50, 25, 30, -45, 155, -220 };
	oski_value_t alpha = -2;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_47, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_47", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_47() */
/* END TEST 47 */

/*
 *  BEGIN TEST 48
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_48( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	const oski_value_t y_true[] = {80, -24, 32, -51, 39, 87, -29, -83, -17, -64, 0, 65 };
	oski_value_t alpha = 1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_48, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_48", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_48() */
/* END TEST 48 */

/*
 *  BEGIN TEST 49
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_49( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	const oski_value_t y_true[] = {-70, 21, -28, 55, -38, -77, 28, 76, 16, 54, 0, -55 };
	oski_value_t alpha = -1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_49, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_49", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_49() */
/* END TEST 49 */

/*
 *  BEGIN TEST 50
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_50( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	const oski_value_t y_true[] = {10, -3, 4, 4, 1, 10, -1, -7, -1, -10, 0, 10 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_50, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_50", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_50() */
/* END TEST 50 */

/*
 *  BEGIN TEST 51
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_51( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	const oski_value_t y_true[] = {-80, 24, -32, 466, -194, -122, 134, 218, 62, -16, 0, 10 };
	oski_value_t alpha = -6;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_51, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_51", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_51() */
/* END TEST 51 */

/*
 *  BEGIN TEST 52
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_52( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	const oski_value_t y_true[] = {78, -10, -87, 16, -32, 14, 2, 38, -10, -30, -140, 12, 99, -18, -108 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_52, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_52", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_52() */
/* END TEST 52 */

/*
 *  BEGIN TEST 53
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_53( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	const oski_value_t y_true[] = {-24, -35, 150, 56, 68, 76, 7, 43, -35, -60, 185, 42, -72, -63, 72 };
	oski_value_t alpha = -1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_53, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_53", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_53() */
/* END TEST 53 */

/*
 *  BEGIN TEST 54
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_54( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	const oski_value_t y_true[] = {-42, 35, -49, -56, -28, -70, -7, -63, 35, 70, -35, -42, -21, 63, 28 };
	oski_value_t alpha = 0;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_54, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_54", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_54() */
/* END TEST 54 */

/*
 *  BEGIN TEST 55
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_55( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_ROWMAJ, y32_VW );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	const oski_value_t y_true[] = {-426, -30, 749, 48, 304, 102, 6, -86, -30, 10, 1080, 36, -633, -54, 676 };
	oski_value_t alpha = -7;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_55, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_55", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_55() */
/* END TEST 55 */

/*
 *  BEGIN TEST 56
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_56( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	const oski_value_t y_true[] = {80, -107, 11, 48, 8, 111, -35, -56, -8, -73, -9, 65 };
	oski_value_t alpha = 1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_56, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_56", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_56() */
/* END TEST 56 */

/*
 *  BEGIN TEST 57
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_57( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	const oski_value_t y_true[] = {20, 77, 29, -8, 2, -11, 25, -14, -2, -27, 9, 35 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_57, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_57", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_57() */
/* END TEST 57 */

/*
 *  BEGIN TEST 58
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_58( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	const oski_value_t y_true[] = {40, -12, 16, 16, 4, 40, -4, -28, -4, -40, 0, 40 };
	oski_value_t alpha = 0;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_58, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_58", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_58() */
/* END TEST 58 */

/*
 *  BEGIN TEST 59
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_59( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	oski_value_t* y_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	const oski_value_t y_true[] = {-50, 596, 127, -132, -5, -267, 194, 35, 5, 1, 63, 55 };
	oski_value_t alpha = -7;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_59, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_59", y_true, y_val, y32_LEN*y32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_59() */
/* END TEST 59 */

/*
 *  BEGIN TEST 60
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_60( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	const oski_value_t y_true[] = {-12, -5, 38, 8, 34, 13, 1, -68, -5, -110, -25, 6, 113, -9, 96 };
	oski_value_t alpha = 1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_60, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_60", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_60() */
/* END TEST 60 */

/*
 *  BEGIN TEST 61
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_61( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	const oski_value_t y_true[] = {36, -15, -10, 24, -18, 27, 3, 104, -15, 70, 45, 18, -101, -27, -112 };
	oski_value_t alpha = -1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_61, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_61", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_61() */
/* END TEST 61 */

/*
 *  BEGIN TEST 62
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_62( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	const oski_value_t y_true[] = {6, -5, 7, 8, 4, 10, 1, 9, -5, -10, 5, 6, 3, -9, -4 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_62, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_62", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_62() */
/* END TEST 62 */

/*
 *  BEGIN TEST 63
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x32', LAYOUT_COLMAJ>
 *  y : <'y32', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_63( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A32_ptr, A32_ind, A32_val,
		A32_M, A32_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y32_val,
		y32_LEN * y32_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y32_LEN, y32_VW, LAYOUT_COLMAJ, y32_LEN );
	oski_value_t* y_val = array_Duplicate( x32_val,
		x32_LEN * x32_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x32_LEN, x32_VW, LAYOUT_COLMAJ, x32_LEN );
	const oski_value_t y_true[] = {-156, 40, 130, -64, 148, -62, -8, -534, 40, -520, -220, -48, 636, 72, 632 };
	oski_value_t alpha = 6;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_63, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_63", y_true, y_val, x32_LEN*x32_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_63() */
/* END TEST 63 */

#define A64_M 4    /* # of rows */
#define A64_N 5    /* # of cols */
#define A64_NNZ 6  /* # of stored non-zeros */
static oski_index_t A64_ptr[A64_M+1] = {1, 2, 4, 6, 7}; /* end row pointers for A64 */
static oski_index_t A64_ind[A64_NNZ] = {5, 3, 5, 4, 5, 3}; /* end column indices pointers for A64 */
static oski_value_t A64_val[A64_NNZ] = {1, 2, 10, 9, 6, -10}; /* end stored non-zero values for A64 */

#define x64_LEN 5   /* vector length */
#define x64_VW  3   /* vector width */
static oski_value_t x64_val[x64_LEN*x64_VW] = {1, -8, -1, 3, 10, 0, 6, 10, 6, 8, 6, 7, -10, 10, 2}; /* data */

#define y64_LEN 4   /* vector length */
#define y64_VW  3   /* vector width */
static oski_value_t y64_val[y64_LEN*y64_VW] = {-1, -10, 5, 0, 4, 4, -7, 10, 9, 9, 1, -8}; /* data */

/*
 *  BEGIN TEST 64
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_64( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	const oski_value_t y_true[] = {-8, 12, -4, -85, 126, 28, 25, 114, 72, -61, -95, -45 };
	oski_value_t alpha = 1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_64, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_64", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_64() */
/* END TEST 64 */

/*
 *  BEGIN TEST 65
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_65( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	const oski_value_t y_true[] = {0, -92, 44, 85, -94, 4, -81, -34, 0, 133, 103, -19 };
	oski_value_t alpha = -1;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_65, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_65", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_65() */
/* END TEST 65 */

/*
 *  BEGIN TEST 66
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_66( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	const oski_value_t y_true[] = {5, 50, -25, 0, -20, -20, 35, -50, -45, -45, -5, 40 };
	oski_value_t alpha = 0;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_66, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_66", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_66() */
/* END TEST 66 */

/*
 *  BEGIN TEST 67
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_67( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	const oski_value_t y_true[] = {12, 28, -16, 85, -142, -44, 3, -154, -108, 25, 91, 77 };
	oski_value_t alpha = -1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_67, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_67", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_67() */
/* END TEST 67 */

/*
 *  BEGIN TEST 68
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_68( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	const oski_value_t y_true[] = {-1, -10, 5, 0, 4, 4, -97, 8, 97, -54, 91, 73, -43, 90, 99 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_68, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_68", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_68() */
/* END TEST 68 */

/*
 *  BEGIN TEST 69
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_69( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	const oski_value_t y_true[] = {8, -46, -12, 21, 66, -4, 139, 62, -55, 110, -49, -24, -27, -20, -85 };
	oski_value_t alpha = -1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_69, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_69", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_69() */
/* END TEST 69 */

/*
 *  BEGIN TEST 70
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_70( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	const oski_value_t y_true[] = {-1, 8, 1, -3, -10, 0, -6, -10, -6, -8, -6, -7, 10, -10, -2 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_70, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_70", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_70() */
/* END TEST 70 */

/*
 *  BEGIN TEST 71
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_71( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	const oski_value_t y_true[] = {7, -38, -11, 18, 56, -4, 133, 52, -61, 102, -55, -31, -17, -30, -87 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_71, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_71", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_71() */
/* END TEST 71 */

/*
 *  BEGIN TEST 72
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_72( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	const oski_value_t y_true[] = {-7, -65, 8, -52, -6, 122, 138, -114, -17, 14, 79, -37 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_72, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_72", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_72() */
/* END TEST 72 */

/*
 *  BEGIN TEST 73
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_73( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	const oski_value_t y_true[] = {14, 135, -43, 52, -22, -150, -89, 44, -46, -77, -86, 93 };
	oski_value_t alpha = -1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_73, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_73", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_73() */
/* END TEST 73 */

/*
 *  BEGIN TEST 74
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_74( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	const oski_value_t y_true[] = {-9, -90, 45, 0, 36, 36, -63, 90, 81, 81, 9, -72 };
	oski_value_t alpha = 0;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_74, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_74", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_74() */
/* END TEST 74 */

/*
 *  BEGIN TEST 75
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_75( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	const oski_value_t y_true[] = {52, 490, -98, 312, -4, -772, -758, 584, 12, -174, -484, 302 };
	oski_value_t alpha = -6;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_75, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_75", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_75() */
/* END TEST 75 */

/*
 *  BEGIN TEST 76
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_76( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	const oski_value_t y_true[] = {1, -12, 7, -4, 24, 9, -3, -79, 111, 61, -41, 15, -91, 22, 109 };
	oski_value_t alpha = 1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_76, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_76", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_76() */
/* END TEST 76 */

/*
 *  BEGIN TEST 77
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_77( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	const oski_value_t y_true[] = {-8, 68, 0, -17, -94, -9, -39, 9, -153, -117, -1, -64, 161, -92, -123 };
	oski_value_t alpha = -1;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_77, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_77", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_77() */
/* END TEST 77 */

/*
 *  BEGIN TEST 78
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_78( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	const oski_value_t y_true[] = {-9, 72, 9, -27, -90, 0, -54, -90, -54, -72, -54, -63, 90, -90, -18 };
	oski_value_t alpha = 0;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_78, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_78", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_78() */
/* END TEST 78 */

/*
 *  BEGIN TEST 79
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_ROWMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_79( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_ROWMAJ, x64_VW );
	const oski_value_t y_true[] = {-1, -12, 41, -38, 60, 45, -51, -455, 519, 257, -241, 33, -395, 50, 533 };
	oski_value_t alpha = 5;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_79, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_79", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_79() */
/* END TEST 79 */

/*
 *  BEGIN TEST 80
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_80( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	const oski_value_t y_true[] = {14, 38, -7, 90, 94, -5, 107, 82, 65, -14, -97, 134 };
	oski_value_t alpha = 1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_80, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_80", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_80() */
/* END TEST 80 */

/*
 *  BEGIN TEST 81
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_81( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	const oski_value_t y_true[] = {-10, 2, -13, -90, -110, -11, -79, -122, -101, -22, 93, -102 };
	oski_value_t alpha = -1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_81, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_81", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_81() */
/* END TEST 81 */

/*
 *  BEGIN TEST 82
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_82( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	const oski_value_t y_true[] = {-3, -30, 15, 0, 12, 12, -21, 30, 27, 27, 3, -24 };
	oski_value_t alpha = 0;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_82, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_82", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_82() */
/* END TEST 82 */

/*
 *  BEGIN TEST 83
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_83( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	const oski_value_t y_true[] = {38, 74, -1, 270, 298, 1, 293, 286, 231, -6, -287, 370 };
	oski_value_t alpha = 3;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_83, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_83", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_83() */
/* END TEST 83 */

/*
 *  BEGIN TEST 84
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_84( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	const oski_value_t y_true[] = {5, -48, -103, -36, 17, -10, 40, 68, 127, 138, 41, 46, 37, 133, 111 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_84, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_84", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_84() */
/* END TEST 84 */

/*
 *  BEGIN TEST 85
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_85( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	const oski_value_t y_true[] = {5, -32, 93, 66, 83, 10, 20, 32, -67, -58, 19, 24, -137, -33, -91 };
	oski_value_t alpha = -1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_85, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_85", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_85() */
/* END TEST 85 */

/*
 *  BEGIN TEST 86
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_86( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	const oski_value_t y_true[] = {6, -48, -6, 18, 60, 0, 36, 60, 36, 48, 36, 42, -60, 60, 12 };
	oski_value_t alpha = 0;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_86, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_86", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_86() */
/* END TEST 86 */

/*
 *  BEGIN TEST 87
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_87( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_ROWMAJ, y64_VW );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	const oski_value_t y_true[] = {-1, 64, 687, 354, 221, 70, -76, -136, -685, -694, -83, -84, -599, -591, -709 };
	oski_value_t alpha = -7;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_87, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_87", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_87() */
/* END TEST 87 */

/*
 *  BEGIN TEST 88
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_88( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	const oski_value_t y_true[] = {6, 40, 111, 13, 28, 126, 77, -44, 53, 52, 97, 70 };
	oski_value_t alpha = 1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_88, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_88", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_88() */
/* END TEST 88 */

/*
 *  BEGIN TEST 89
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_89( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	const oski_value_t y_true[] = {-13, -110, -76, -13, 0, -98, -126, 114, 10, 11, -90, -126 };
	oski_value_t alpha = -1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_89, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_89", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_89() */
/* END TEST 89 */

/*
 *  BEGIN TEST 90
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_90( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	const oski_value_t y_true[] = {5, 50, -25, 0, -20, -20, 35, -50, -45, -45, -5, 40 };
	oski_value_t alpha = 0;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_90, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_90", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_90() */
/* END TEST 90 */

/*
 *  BEGIN TEST 91
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_91( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	oski_value_t* y_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	const oski_value_t y_true[] = {-47, -370, -470, -65, -72, -562, -504, 390, -112, -107, -468, -486 };
	oski_value_t alpha = -5;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_91, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_91", y_true, y_val, y64_LEN*y64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_91() */
/* END TEST 91 */

/*
 *  BEGIN TEST 92
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_92( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	const oski_value_t y_true[] = {-3, 6, -13, 39, -91, 4, -8, -119, -65, -14, -3, -5, 119, -19, 101 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_92, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_92", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_92() */
/* END TEST 92 */

/*
 *  BEGIN TEST 93
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_93( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	const oski_value_t y_true[] = {9, -54, 7, -21, 151, -4, 44, 179, 101, 62, 39, 47, -179, 79, -89 };
	oski_value_t alpha = -1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_93, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_93", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_93() */
/* END TEST 93 */

/*
 *  BEGIN TEST 94
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_94( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	const oski_value_t y_true[] = {8, -64, -8, 24, 80, 0, 48, 80, 48, 64, 48, 56, -80, 80, 16 };
	oski_value_t alpha = 0;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_94, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_94", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_94() */
/* END TEST 94 */

/*
 *  BEGIN TEST 95
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x64', LAYOUT_COLMAJ>
 *  y : <'y64', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_95( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A64_ptr, A64_ind, A64_val,
		A64_M, A64_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y64_val,
		y64_LEN * y64_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y64_LEN, y64_VW, LAYOUT_COLMAJ, y64_LEN );
	oski_value_t* y_val = array_Duplicate( x64_val,
		x64_LEN * x64_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x64_LEN, x64_VW, LAYOUT_COLMAJ, x64_LEN );
	const oski_value_t y_true[] = {-2, 52, 34, -102, 102, -8, -32, 158, 82, -36, -42, -46, -158, -42, -218 };
	oski_value_t alpha = -2;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_95, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_95", y_true, y_val, x64_LEN*x64_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_95() */
/* END TEST 95 */

#define A96_M 4    /* # of rows */
#define A96_N 5    /* # of cols */
#define A96_NNZ 6  /* # of stored non-zeros */
static oski_index_t A96_ptr[A96_M+1] = {0, 1, 3, 4, 6}; /* end row pointers for A96 */
static oski_index_t A96_ind[A96_NNZ] = {3, 0, 4, 0, 1, 4}; /* end column indices pointers for A96 */
static oski_value_t A96_val[A96_NNZ] = {-9, 10, -7, -10, -7, 2}; /* end stored non-zero values for A96 */

#define x96_LEN 5   /* vector length */
#define x96_VW  3   /* vector width */
static oski_value_t x96_val[x96_LEN*x96_VW] = {7, 10, 7, 9, 7, 6, -10, 5, -5, -3, 7, -8, 9, 3, 3}; /* data */

#define y96_LEN 4   /* vector length */
#define y96_VW  3   /* vector width */
static oski_value_t y96_val[y96_LEN*y96_VW] = {2, -5, 6, -10, -9, 0, 10, 9, 2, 7, -5, -10}; /* data */

/*
 *  BEGIN TEST 96
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_96( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {36, -58, 85, 6, 77, 55, -70, -86, -73, -41, -41, -54 };
	oski_value_t alpha = 1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_96, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_96", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_96() */
/* END TEST 96 */

/*
 *  BEGIN TEST 97
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_97( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {-40, 68, -97, 14, -59, -55, 50, 68, 69, 27, 51, 74 };
	oski_value_t alpha = -1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_97, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_97", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_97() */
/* END TEST 97 */

/*
 *  BEGIN TEST 98
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_98( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {4, -10, 12, -20, -18, 0, 20, 18, 4, 14, -10, -20 };
	oski_value_t alpha = 0;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_98, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_98", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_98() */
/* END TEST 98 */

/*
 *  BEGIN TEST 99
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_99( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {-2, 5, -6, 10, 9, 0, -10, -9, -2, -7, 5, 10 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_99, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_99", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_99() */
/* END TEST 99 */

/*
 *  BEGIN TEST 100
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_100( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {-226, -225, -42, -95, -2, 46, 50, -11, 22, 1, 12, -32, 48, 41, -32 };
	oski_value_t alpha = 1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_100, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_100", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_100() */
/* END TEST 100 */

/*
 *  BEGIN TEST 101
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_101( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {184, 165, 0, 41, -40, -82, 10, -19, 8, 17, -54, 80, -102, -59, 14 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_101, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_101", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_101() */
/* END TEST 101 */

/*
 *  BEGIN TEST 102
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_102( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {-14, -20, -14, -18, -14, -12, 20, -10, 10, 6, -14, 16, -18, -6, -6 };
	oski_value_t alpha = 0;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_102, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_102", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_102() */
/* END TEST 102 */

/*
 *  BEGIN TEST 103
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_103( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {198, 185, 14, 59, -26, -70, -10, -9, -2, 11, -40, 64, -84, -53, 20 };
	oski_value_t alpha = -1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_103, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_103", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_103() */
/* END TEST 103 */

/*
 *  BEGIN TEST 104
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_104( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {30, 26, -92, -28, -35, 86, -115, -54, 75, 41, -65, -24 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_104, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_104", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_104() */
/* END TEST 104 */

/*
 *  BEGIN TEST 105
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_105( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {-42, 4, 56, 88, 89, -86, 55, 0, -87, -83, 95, 84 };
	oski_value_t alpha = -1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_105, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_105", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_105() */
/* END TEST 105 */

/*
 *  BEGIN TEST 106
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_106( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {-16, 40, -48, 80, 72, 0, -80, -72, -16, -56, 40, 80 };
	oski_value_t alpha = 0;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_106, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_106", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_106() */
/* END TEST 106 */

/*
 *  BEGIN TEST 107
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_107( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {-76, -12, 136, 136, 142, -172, 150, 36, -166, -138, 170, 128 };
	oski_value_t alpha = -2;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_107, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_107", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_107() */
/* END TEST 107 */

/*
 *  BEGIN TEST 108
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_108( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {-87, -79, 143, 92, -42, 95, -24, 25, -20, -37, 111, -52, 42, 27, -60 };
	oski_value_t alpha = 1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_108, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_108", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_108() */
/* END TEST 108 */

/*
 *  BEGIN TEST 109
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_109( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {129, 139, -101, -38, 84, -59, -36, 5, -10, 19, -69, 4, 12, -9, 78 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_109, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_109", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_109() */
/* END TEST 109 */

/*
 *  BEGIN TEST 110
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_110( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_110, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_110", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_110() */
/* END TEST 110 */

/*
 *  BEGIN TEST 111
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_111( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {-404, -396, 516, 296, -224, 332, -16, 60, -40, -124, 388, -144, 96, 84, -264 };
	oski_value_t alpha = 4;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_111, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_111", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_111() */
/* END TEST 111 */

/*
 *  BEGIN TEST 112
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_112( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {-58, 11, 28, -49, -1, 41, 17, 17, -45, 9, 19, -15 };
	oski_value_t alpha = 1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_112, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_112", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_112() */
/* END TEST 112 */

/*
 *  BEGIN TEST 113
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_113( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {72, -46, 14, -21, -62, -41, 53, 46, 59, 40, -54, -55 };
	oski_value_t alpha = -1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_113, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_113", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_113() */
/* END TEST 113 */

/*
 *  BEGIN TEST 114
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_114( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {16, -40, 48, -80, -72, 0, 80, 72, 16, 56, -40, -80 };
	oski_value_t alpha = 0;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_114, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_114", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_114() */
/* END TEST 114 */

/*
 *  BEGIN TEST 115
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_115( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {220, -148, 54, -83, -204, -123, 179, 156, 181, 134, -172, -185 };
	oski_value_t alpha = -3;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_115, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_115", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_115() */
/* END TEST 115 */

/*
 *  BEGIN TEST 116
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_116( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {-198, -59, 10, -11, 84, -185, 26, 9, 40, 53, -14, 70, 2, -64, -20 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_116, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_116", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_116() */
/* END TEST 116 */

/*
 *  BEGIN TEST 117
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_117( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {219, 89, 11, 38, -63, 203, -56, 6, -55, -62, 35, -94, 25, 73, 29 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_117, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_117", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_117() */
/* END TEST 117 */

/*
 *  BEGIN TEST 118
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_118( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {-14, -20, -14, -18, -14, -12, 20, -10, 10, 6, -14, 16, -18, -6, -6 };
	oski_value_t alpha = 0;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_118, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_118", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_118() */
/* END TEST 118 */

/*
 *  BEGIN TEST 119
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_119( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {-333, -28, 83, 59, 231, -316, -38, 63, 35, 79, 35, 68, 85, -101, -13 };
	oski_value_t alpha = 2;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_119, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_119", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_119() */
/* END TEST 119 */

/*
 *  BEGIN TEST 120
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_120( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {-86, 61, -99, 13, 105, 71, -115, 5, -32, -1, -31, 125 };
	oski_value_t alpha = 1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_120, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_120", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_120() */
/* END TEST 120 */

/*
 *  BEGIN TEST 121
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_121( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {80, -46, 81, 17, -78, -71, 85, -32, 26, -20, 46, -95 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_121, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_121", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_121() */
/* END TEST 121 */

/*
 *  BEGIN TEST 122
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_122( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {-6, 15, -18, 30, 27, 0, -30, -27, -6, -21, 15, 30 };
	oski_value_t alpha = 0;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_122, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_122", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_122() */
/* END TEST 122 */

/*
 *  BEGIN TEST 123
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_123( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {-74, 31, -63, -47, 51, 71, -55, 59, -20, 41, -61, 65 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_123, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_123", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_123() */
/* END TEST 123 */

/*
 *  BEGIN TEST 124
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_124( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {-66, 125, 48, 26, 57, -73, -123, 40, 60, 0, 164, 29, 49, -10, -51 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_124, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_124", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_124() */
/* END TEST 124 */

/*
 *  BEGIN TEST 125
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_125( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {45, -155, -69, -53, -78, 55, 153, -55, -45, 9, -185, -5, -76, 1, 42 };
	oski_value_t alpha = -1;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_125, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_125", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_125() */
/* END TEST 125 */

/*
 *  BEGIN TEST 126
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_126( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {-28, -40, -28, -36, -28, -24, 40, -20, 20, 12, -28, 32, -36, -12, -12 };
	oski_value_t alpha = 0;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_126, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_126", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_126() */
/* END TEST 126 */

/*
 *  BEGIN TEST 127
 *  Value type: real
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_127( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {-317, 205, 25, -75, 52, -321, -199, 35, 265, 51, 373, 223, -6, -81, -204 };
	oski_value_t alpha = 3;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_127, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_127", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_127() */
/* END TEST 127 */

#define A128_M 4    /* # of rows */
#define A128_N 5    /* # of cols */
#define A128_NNZ 5  /* # of stored non-zeros */
static oski_index_t A128_ptr[A128_M+1] = {1, 2, 3, 4, 6}; /* end row pointers for A128 */
static oski_index_t A128_ind[A128_NNZ] = {4, 4, 3, 4, 5}; /* end column indices pointers for A128 */
static oski_value_t A128_val[A128_NNZ] = {8, 10, -7, -10, -8}; /* end stored non-zero values for A128 */

#define x128_LEN 5   /* vector length */
#define x128_VW  3   /* vector width */
static oski_value_t x128_val[x128_LEN*x128_VW] = {10, 7, 7, 0, -10, 3, 1, -9, 2, -1, -5, -2, 4, 5, 6}; /* data */

#define y128_LEN 4   /* vector length */
#define y128_VW  3   /* vector width */
static oski_value_t y128_val[y128_LEN*y128_VW] = {3, -3, 10, -1, 7, -3, -10, -8, 3, 9, -5, -9}; /* data */

/*
 *  BEGIN TEST 128
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_128( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	const oski_value_t y_true[] = {-32, -16, -96, -2, -106, 4, 73, 127, -38, -94, 50, 44 };
	oski_value_t alpha = 1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_128, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_128", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_128() */
/* END TEST 128 */

/*
 *  BEGIN TEST 129
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_129( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	const oski_value_t y_true[] = {-10, 58, -44, 16, 8, 38, 67, -15, -4, -32, 20, 82 };
	oski_value_t alpha = -1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_129, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_129", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_129() */
/* END TEST 129 */

/*
 *  BEGIN TEST 130
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_130( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	const oski_value_t y_true[] = {12, -12, 40, -4, 28, -12, -40, -32, 12, 36, -20, -36 };
	oski_value_t alpha = 0;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_130, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_130", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_130() */
/* END TEST 130 */

/*
 *  BEGIN TEST 131
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_131( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	const oski_value_t y_true[] = {8, 184, -16, 48, 144, 104, 108, -188, 32, 16, 0, 184 };
	oski_value_t alpha = -4;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_131, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_131", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_131() */
/* END TEST 131 */

/*
 *  BEGIN TEST 132
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_132( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	const oski_value_t y_true[] = {0, 0, 0, 0, 0, 0, 70, 56, -21, -76, 96, 140, -72, 40, 72 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_132, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_132", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_132() */
/* END TEST 132 */

/*
 *  BEGIN TEST 133
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_133( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	const oski_value_t y_true[] = {-10, -7, -7, 0, 10, -3, -71, -47, 19, 77, -91, -138, 68, -45, -78 };
	oski_value_t alpha = -1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_133, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_133", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_133() */
/* END TEST 133 */

/*
 *  BEGIN TEST 134
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_134( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	const oski_value_t y_true[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_134, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_134", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_134() */
/* END TEST 134 */

/*
 *  BEGIN TEST 135
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_135( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	const oski_value_t y_true[] = {-60, -42, -42, 0, 60, -18, -76, -2, 9, 82, -66, -128, 48, -70, -108 };
	oski_value_t alpha = -1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_135, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_135", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_135() */
/* END TEST 135 */

/*
 *  BEGIN TEST 136
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_136( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	const oski_value_t y_true[] = {-20, 2, -47, -18, -68, -38, 103, 42, -28, -56, 6, 8 };
	oski_value_t alpha = 1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_136, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_136", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_136() */
/* END TEST 136 */

/*
 *  BEGIN TEST 137
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_137( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	const oski_value_t y_true[] = {26, -8, 67, 16, 82, 32, -123, -58, 34, 74, -16, -26 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_137, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_137", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_137() */
/* END TEST 137 */

/*
 *  BEGIN TEST 138
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_138( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	const oski_value_t y_true[] = {12, -12, 40, -4, 28, -12, -40, -32, 12, 36, -20, -36 };
	oski_value_t alpha = 0;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_138, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_138", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_138() */
/* END TEST 138 */

/*
 *  BEGIN TEST 139
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_139( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	const oski_value_t y_true[] = {1, -19, 23, -25, -19, -59, 33, -14, -7, 7, -29, -55 };
	oski_value_t alpha = 1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_139, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_139", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_139() */
/* END TEST 139 */

/*
 *  BEGIN TEST 140
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_140( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	const oski_value_t y_true[] = {-30, -21, -21, 0, 30, -9, -73, 97, 29, 7, 121, 210, -4, 49, 54 };
	oski_value_t alpha = 1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_140, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_140", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_140() */
/* END TEST 140 */

/*
 *  BEGIN TEST 141
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_141( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	const oski_value_t y_true[] = {-80, -56, -56, 0, 80, -24, 62, 2, -51, 4, -66, -188, -40, -104, -120 };
	oski_value_t alpha = -1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_141, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_141", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_141() */
/* END TEST 141 */

/*
 *  BEGIN TEST 142
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_142( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	const oski_value_t y_true[] = {90, 63, 63, 0, -90, 27, 9, -81, 18, -9, -45, -18, 36, 45, 54 };
	oski_value_t alpha = 0;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_142, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_142", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_142() */
/* END TEST 142 */

/*
 *  BEGIN TEST 143
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_ROWMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_143( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_ROWMAJ, x128_VW );
	const oski_value_t y_true[] = {-90, -63, -63, 0, 90, -27, -79, 151, 17, 13, 151, 222, -28, 19, 18 };
	oski_value_t alpha = 1;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_143, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_143", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_143() */
/* END TEST 143 */

/*
 *  BEGIN TEST 144
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_144( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	const oski_value_t y_true[] = {18, -2, 100, -6, 62, 32, -109, 15, -10, 134, -42, -152 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_144, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_144", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_144() */
/* END TEST 144 */

/*
 *  BEGIN TEST 145
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_145( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	const oski_value_t y_true[] = {-24, 8, -120, 8, -76, -26, 129, 1, 4, -152, 52, 170 };
	oski_value_t alpha = -1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_145, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_145", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_145() */
/* END TEST 145 */

/*
 *  BEGIN TEST 146
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_146( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	const oski_value_t y_true[] = {-6, 6, -20, 2, -14, 6, 20, 16, -6, -18, 10, 18 };
	oski_value_t alpha = 0;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_146, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_146", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_146() */
/* END TEST 146 */

/*
 *  BEGIN TEST 147
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_147( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	const oski_value_t y_true[] = {3, -3, 10, -1, 7, -3, -10, -8, 3, 9, -5, -9 };
	oski_value_t alpha = 0;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_147, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_147", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_147() */
/* END TEST 147 */

/*
 *  BEGIN TEST 148
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_148( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	const oski_value_t y_true[] = {40, 28, 98, -76, -112, 12, 4, 20, 104, 36, -20, -8, -5, 160, 96 };
	oski_value_t alpha = 1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_148, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_148", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_148() */
/* END TEST 148 */

/*
 *  BEGIN TEST 149
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_149( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	const oski_value_t y_true[] = {30, 21, -49, 76, 42, 9, 3, -83, -90, -43, -15, -6, 33, -125, -54 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_149, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_149", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_149() */
/* END TEST 149 */

/*
 *  BEGIN TEST 150
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_150( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	const oski_value_t y_true[] = {10, 7, 7, 0, -10, 3, 1, -9, 2, -1, -5, -2, 4, 5, 6 };
	oski_value_t alpha = 0;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_150, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_150", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_150() */
/* END TEST 150 */

/*
 *  BEGIN TEST 151
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_151( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_ROWMAJ, y128_VW );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	const oski_value_t y_true[] = {-70, -49, -609, 608, 646, -21, -7, -385, -782, -313, 35, 14, 140, -1155, -618 };
	oski_value_t alpha = -8;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_151, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_151", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_151() */
/* END TEST 151 */

/*
 *  BEGIN TEST 152
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_152( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	const oski_value_t y_true[] = {27, -27, 41, 71, 79, -7, -27, -84, 67, 131, -73, -179 };
	oski_value_t alpha = 1;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_152, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_152", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_152() */
/* END TEST 152 */

/*
 *  BEGIN TEST 153
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_153( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	const oski_value_t y_true[] = {-18, 18, -11, -74, -58, -2, -3, 60, -58, -104, 58, 152 };
	oski_value_t alpha = -1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_153, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_153", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_153() */
/* END TEST 153 */

/*
 *  BEGIN TEST 154
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_154( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	const oski_value_t y_true[] = {-9, 9, -30, 3, -21, 9, 30, 24, -9, -27, 15, 27 };
	oski_value_t alpha = 0;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_154, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_154", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_154() */
/* END TEST 154 */

/*
 *  BEGIN TEST 155
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_155( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	oski_value_t* y_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	const oski_value_t y_true[] = {6, -6, -78, 158, 46, 34, 106, -40, 86, 118, -66, -214 };
	oski_value_t alpha = 2;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_155, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_155", y_true, y_val, y128_LEN*y128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_155() */
/* END TEST 155 */

/*
 *  BEGIN TEST 156
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_156( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	const oski_value_t y_true[] = {-70, -49, -119, 4, 78, -21, -7, 133, 92, 71, 35, 14, 7, 169, 30 };
	oski_value_t alpha = 1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_156, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_156", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_156() */
/* END TEST 156 */

/*
 *  BEGIN TEST 157
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_157( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	const oski_value_t y_true[] = {10, 7, 77, -4, -18, 3, 1, -79, -104, -65, -5, -2, -31, -199, -66 };
	oski_value_t alpha = -1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_157, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_157", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_157() */
/* END TEST 157 */

/*
 *  BEGIN TEST 158
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_158( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	const oski_value_t y_true[] = {80, 56, 56, 0, -80, 24, 8, -72, 16, -8, -40, -16, 32, 40, 48 };
	oski_value_t alpha = 0;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_158, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_158", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_158() */
/* END TEST 158 */

/*
 *  BEGIN TEST 159
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x128', LAYOUT_COLMAJ>
 *  y : <'y128', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_159( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A128_ptr, A128_ind, A128_val,
		A128_M, A128_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y128_val,
		y128_LEN * y128_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y128_LEN, y128_VW, LAYOUT_COLMAJ, y128_LEN );
	oski_value_t* y_val = array_Duplicate( x128_val,
		x128_LEN * x128_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x128_LEN, x128_VW, LAYOUT_COLMAJ, x128_LEN );
	const oski_value_t y_true[] = {60, 42, 252, -12, -84, 18, 6, -264, -306, -198, -30, -12, -81, -582, -180 };
	oski_value_t alpha = -3;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_159, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_159", y_true, y_val, x128_LEN*x128_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_159() */
/* END TEST 159 */

#define A160_M 4    /* # of rows */
#define A160_N 5    /* # of cols */
#define A160_NNZ 4  /* # of stored non-zeros */
static oski_index_t A160_ptr[A160_M+1] = {0, 1, 3, 3, 4}; /* end row pointers for A160 */
static oski_index_t A160_ind[A160_NNZ] = {0, 1, 2, 3}; /* end column indices pointers for A160 */
static oski_value_t A160_val[A160_NNZ] = {-10, -6, 4, 10}; /* end stored non-zero values for A160 */

#define x160_LEN 5   /* vector length */
#define x160_VW  3   /* vector width */
static oski_value_t x160_val[x160_LEN*x160_VW] = {-8, 10, -10, -9, 0, 6, 9, -5, -3, 6, 0, 5, 4, 4, 5}; /* data */

#define y160_LEN 4   /* vector length */
#define y160_VW  3   /* vector width */
static oski_value_t y160_val[y160_LEN*y160_VW] = {-10, -5, -5, 5, 7, 6, -1, -5, -4, 7, -6, 10}; /* data */

/*
 *  BEGIN TEST 160
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_160( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	const oski_value_t y_true[] = {140, -70, 130, 60, -62, -84, 6, 30, 24, 18, 36, -10 };
	oski_value_t alpha = 1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_160, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_160", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_160() */
/* END TEST 160 */

/*
 *  BEGIN TEST 161
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_161( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	const oski_value_t y_true[] = {0, 140, -60, -130, -36, 0, 8, 40, 32, -116, 48, -130 };
	oski_value_t alpha = -1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_161, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_161", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_161() */
/* END TEST 161 */

/*
 *  BEGIN TEST 162
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_162( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	const oski_value_t y_true[] = {-30, -15, -15, 15, 21, 18, -3, -15, -12, 21, -18, 30 };
	oski_value_t alpha = 0;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_162, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_162", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_162() */
/* END TEST 162 */

/*
 *  BEGIN TEST 163
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_163( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	const oski_value_t y_true[] = {-640, 660, -740, -590, 196, 384, -8, -40, -32, -364, -48, -270 };
	oski_value_t alpha = -7;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_163, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_163", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_163() */
/* END TEST 163 */

/*
 *  BEGIN TEST 164
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_164( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	const oski_value_t y_true[] = {100, 50, 50, -30, -42, -36, 20, 28, 24, 70, -60, 100, 0, 0, 0 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_164, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_164", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_164() */
/* END TEST 164 */

/*
 *  BEGIN TEST 165
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_165( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	const oski_value_t y_true[] = {-116, -30, -70, 12, 42, 48, -2, -38, -30, -58, 60, -90, 8, 8, 10 };
	oski_value_t alpha = -1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_165, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_165", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_165() */
/* END TEST 165 */

/*
 *  BEGIN TEST 166
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_166( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	const oski_value_t y_true[] = {-56, 70, -70, -63, 0, 42, 63, -35, -21, 42, 0, 35, 28, 28, 35 };
	oski_value_t alpha = 0;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_166, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_166", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_166() */
/* END TEST 166 */

/*
 *  BEGIN TEST 167
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_167( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	const oski_value_t y_true[] = {-916, -430, -470, 252, 378, 336, -162, -262, -222, -618, 540, -890, 8, 8, 10 };
	oski_value_t alpha = -9;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_167, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_167", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_167() */
/* END TEST 167 */

/*
 *  BEGIN TEST 168
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_168( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	const oski_value_t y_true[] = {40, 70, -20, 80, -72, 4, -4, -20, 84, -20, -24, 90 };
	oski_value_t alpha = 1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_168, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_168", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_168() */
/* END TEST 168 */

/*
 *  BEGIN TEST 169
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_169( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	const oski_value_t y_true[] = {-80, -90, 0, -60, 100, 20, 0, 0, -100, 48, 0, -50 };
	oski_value_t alpha = -1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_169, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_169", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_169() */
/* END TEST 169 */

/*
 *  BEGIN TEST 170
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_170( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	const oski_value_t y_true[] = {80, 40, 40, -40, -56, -48, 8, 40, 32, -56, 48, -80 };
	oski_value_t alpha = 0;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_170, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_170", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_170() */
/* END TEST 170 */

/*
 *  BEGIN TEST 171
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_171( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	const oski_value_t y_true[] = {-80, -90, 0, -60, 100, 20, 0, 0, -100, 48, 0, -50 };
	oski_value_t alpha = -1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_171, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_171", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_171() */
/* END TEST 171 */

/*
 *  BEGIN TEST 172
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_172( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	const oski_value_t y_true[] = {60, -20, -10, -15, -36, -12, 25, -1, 13, 80, -50, 125, 20, 20, 25 };
	oski_value_t alpha = 1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_172, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_172", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_172() */
/* END TEST 172 */

/*
 *  BEGIN TEST 173
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_173( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	const oski_value_t y_true[] = {-36, -10, 40, 42, 36, -6, -52, 16, -4, -98, 50, -140, -32, -32, -40 };
	oski_value_t alpha = -1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_173, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_173", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_173() */
/* END TEST 173 */

/*
 *  BEGIN TEST 174
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_174( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	const oski_value_t y_true[] = {8, -10, 10, 9, 0, -6, -9, 5, 3, -6, 0, -5, -4, -4, -5 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_174, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_174", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_174() */
/* END TEST 174 */

/*
 *  BEGIN TEST 175
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_ROWMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_175( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_ROWMAJ, x160_VW );
	const oski_value_t y_true[] = {908, -640, 370, 279, -324, -384, -189, 221, 255, 444, -450, 895, -4, -4, -5 };
	oski_value_t alpha = 9;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_175, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_175", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_175() */
/* END TEST 175 */

/*
 *  BEGIN TEST 176
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_176( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	const oski_value_t y_true[] = {110, -45, 15, -115, -95, -32, 3, 15, 12, -111, -12, 10 };
	oski_value_t alpha = 1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_176, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_176", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_176() */
/* END TEST 176 */

/*
 *  BEGIN TEST 177
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_177( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	const oski_value_t y_true[] = {-70, 65, 5, 95, 67, 8, 1, 5, 4, 83, 36, -50 };
	oski_value_t alpha = -1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_177, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_177", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_177() */
/* END TEST 177 */

/*
 *  BEGIN TEST 178
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_178( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	const oski_value_t y_true[] = {-40, -20, -20, 20, 28, 24, -4, -20, -16, 28, -24, 40 };
	oski_value_t alpha = 0;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_178, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_178", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_178() */
/* END TEST 178 */

/*
 *  BEGIN TEST 179
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_179( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	const oski_value_t y_true[] = {-340, 330, 30, 470, 328, 34, 6, 30, 24, 408, 186, -260 };
	oski_value_t alpha = -5;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_179, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_179", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_179() */
/* END TEST 179 */

/*
 *  BEGIN TEST 180
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_180( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	const oski_value_t y_true[] = {116, -50, 40, 88, 0, 38, -60, 38, -54, -12, 50, -46, 16, 92, -10 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_180, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_180", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_180() */
/* END TEST 180 */

/*
 *  BEGIN TEST 181
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_181( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	const oski_value_t y_true[] = {-164, 110, -100, -142, 0, -2, 114, -68, 36, 48, -50, 76, 8, -68, 40 };
	oski_value_t alpha = -1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_181, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_181", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_181() */
/* END TEST 181 */

/*
 *  BEGIN TEST 182
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_182( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	const oski_value_t y_true[] = {-64, 80, -80, -72, 0, 48, 72, -40, -24, 48, 0, 40, 32, 32, 40 };
	oski_value_t alpha = 0;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_182, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_182", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_182() */
/* END TEST 182 */

/*
 *  BEGIN TEST 183
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_183( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_ROWMAJ, y160_VW );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	const oski_value_t y_true[] = {-336, 40, 0, -208, 0, -248, 96, -72, 264, -48, -200, 104, -128, -432, -40 };
	oski_value_t alpha = -4;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_183, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_183", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_183() */
/* END TEST 183 */

/*
 *  BEGIN TEST 184
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_184( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	const oski_value_t y_true[] = {150, -65, 35, -125, -109, -116, 7, 5, 28, -63, 42, -30 };
	oski_value_t alpha = 1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_184, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_184", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_184() */
/* END TEST 184 */

/*
 *  BEGIN TEST 185
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_185( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	const oski_value_t y_true[] = {-80, 100, 0, 90, 60, 74, 0, 30, 0, 14, 0, -40 };
	oski_value_t alpha = -1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_185, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_185", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_185() */
/* END TEST 185 */

/*
 *  BEGIN TEST 186
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_186( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	const oski_value_t y_true[] = {60, 30, 30, -30, -42, -36, 6, 30, 24, -42, 36, -60 };
	oski_value_t alpha = 0;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_186, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_186", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_186() */
/* END TEST 186 */

/*
 *  BEGIN TEST 187
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_187( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	oski_value_t* y_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	const oski_value_t y_true[] = {70, 35, 35, -35, -49, -42, 7, 35, 28, -49, 42, -70 };
	oski_value_t alpha = 0;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_187, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_187", y_true, y_val, y160_LEN*y160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_187() */
/* END TEST 187 */

/*
 *  BEGIN TEST 188
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_188( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	const oski_value_t y_true[] = {52, 90, -80, -4, 0, -34, 18, -6, -68, 36, 40, -12, 52, 124, 30 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_188, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_188", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_188() */
/* END TEST 188 */

/*
 *  BEGIN TEST 189
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_189( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	const oski_value_t y_true[] = {-84, -50, 40, -32, 0, 58, 18, -14, 56, -12, -40, 32, -36, -108, -10 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_189, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_189", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_189() */
/* END TEST 189 */

/*
 *  BEGIN TEST 190
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_190( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	const oski_value_t y_true[] = {8, -10, 10, 9, 0, -6, -9, 5, 3, -6, 0, -5, -4, -4, -5 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_190, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_190", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_190() */
/* END TEST 190 */

/*
 *  BEGIN TEST 191
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x160', LAYOUT_COLMAJ>
 *  y : <'y160', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_191( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A160_ptr, A160_ind, A160_val,
		A160_M, A160_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y160_val,
		y160_LEN * y160_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y160_LEN, y160_VW, LAYOUT_COLMAJ, y160_LEN );
	oski_value_t* y_val = array_Duplicate( x160_val,
		x160_LEN * x160_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x160_LEN, x160_VW, LAYOUT_COLMAJ, x160_LEN );
	const oski_value_t y_true[] = {-332, -50, 20, -186, 0, 234, 144, -92, 138, 24, -120, 146, -68, -284, 20 };
	oski_value_t alpha = -3;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_191, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_191", y_true, y_val, x160_LEN*x160_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_191() */
/* END TEST 191 */

#define A192_M 4    /* # of rows */
#define A192_N 5    /* # of cols */
#define A192_NNZ 0  /* # of stored non-zeros */
static oski_index_t A192_ptr[A192_M+1] = {1, 1, 1, 1, 1}; /* end row pointers for A192 */
static oski_index_t* A192_ind = NULL;
static oski_value_t* A192_val = NULL;

#define x192_LEN 5   /* vector length */
#define x192_VW  3   /* vector width */
static oski_value_t x192_val[x192_LEN*x192_VW] = {2, 5, 10, -4, 6, -4, -1, -8, 1, -5, 8, 1, -10, -4, 5}; /* data */

#define y192_LEN 4   /* vector length */
#define y192_VW  3   /* vector width */
static oski_value_t y192_val[y192_LEN*y192_VW] = {-2, -7, 10, -8, 6, 10, -8, 2, -10, -4, -5, 0}; /* data */

/*
 *  BEGIN TEST 192
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_192( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {-10, -37, 70, -52, 42, 56, -49, 4, -59, -29, -22, 1 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_192, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_192", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_192() */
/* END TEST 192 */

/*
 *  BEGIN TEST 193
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_193( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {-14, -47, 50, -44, 30, 64, -47, 20, -61, -19, -38, -1 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_193, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_193", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_193() */
/* END TEST 193 */

/*
 *  BEGIN TEST 194
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_194( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {-8, -28, 40, -32, 24, 40, -32, 8, -40, -16, -20, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_194, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_194", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_194() */
/* END TEST 194 */

/*
 *  BEGIN TEST 195
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_195( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {-16, -50, 20, -28, 12, 62, -37, 34, -53, -5, -49, -3 };
	oski_value_t alpha = -3;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_195, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_195", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_195() */
/* END TEST 195 */

/*
 *  BEGIN TEST 196
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_196( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {2, 3, 30, -16, 18, 2, -10, -14, -8, -14, 11, 2, -20, -8, 10 };
	oski_value_t alpha = 1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_196, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_196", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_196() */
/* END TEST 196 */

/*
 *  BEGIN TEST 197
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_197( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {-14, -33, -90, 40, -54, 22, 16, 62, 2, 44, -59, -8, 80, 32, -40 };
	oski_value_t alpha = -1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_197, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_197", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_197() */
/* END TEST 197 */

/*
 *  BEGIN TEST 198
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_198( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {-16, -40, -80, 32, -48, 32, 8, 64, -8, 40, -64, -8, 80, 32, -40 };
	oski_value_t alpha = 0;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_198, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_198", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_198() */
/* END TEST 198 */

/*
 *  BEGIN TEST 199
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_199( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {22, 67, -10, 28, -6, -80, 43, -52, 65, -1, 70, 5, -50, -20, 25 };
	oski_value_t alpha = -6;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_199, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_199", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_199() */
/* END TEST 199 */

/*
 *  BEGIN TEST 200
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_200( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {18, 52, -81, 59, -43, -74, 56, -8, 90, 28, 41, 1 };
	oski_value_t alpha = 1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_200, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_200", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_200() */
/* END TEST 200 */

/*
 *  BEGIN TEST 201
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_201( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {-2, 4, 1, 5, -5, -6, 8, -8, -10, 4, -1, -1 };
	oski_value_t alpha = -1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_201, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_201", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_201() */
/* END TEST 201 */

/*
 *  BEGIN TEST 202
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_202( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {-16, -56, 80, -64, 48, 80, -64, 16, -80, -32, -40, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_202, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_202", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_202() */
/* END TEST 202 */

/*
 *  BEGIN TEST 203
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_203( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {0, 55, -45, 65, -55, -80, 80, -50, 0, 40, 20, -5 };
	oski_value_t alpha = -5;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_203, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_203", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_203() */
/* END TEST 203 */

/*
 *  BEGIN TEST 204
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_204( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {-16, -29, -80, 21, -32, 24, 17, 48, -12, 27, -54, -7, 70, 28, -35 };
	oski_value_t alpha = 1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_204, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_204", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_204() */
/* END TEST 204 */

/*
 *  BEGIN TEST 205
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_205( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {10, 14, 50, -9, 14, -12, -14, -24, 9, -12, 30, 4, -40, -16, 20 };
	oski_value_t alpha = -1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_205, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_205", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_205() */
/* END TEST 205 */

/*
 *  BEGIN TEST 206
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_206( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {8, 20, 40, -16, 24, -16, -4, -32, 4, -20, 32, 4, -40, -16, 20 };
	oski_value_t alpha = 0;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_206, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_206", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_206() */
/* END TEST 206 */

/*
 *  BEGIN TEST 207
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_207( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {-8, -20, -40, 16, -24, 16, 4, 32, -4, 20, -32, -4, 40, 16, -20 };
	oski_value_t alpha = 0;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_207, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_207", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_207() */
/* END TEST 207 */

/*
 *  BEGIN TEST 208
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_208( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {-6, -32, 48, -27, 23, 41, -22, 0, -50, -20, -19, -4 };
	oski_value_t alpha = 1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_208, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_208", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_208() */
/* END TEST 208 */

/*
 *  BEGIN TEST 209
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_209( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {2, 18, -28, 11, -11, -21, 6, 4, 30, 12, 9, 4 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_209, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_209", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_209() */
/* END TEST 209 */

/*
 *  BEGIN TEST 210
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_210( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {10, 35, -50, 40, -30, -50, 40, -10, 50, 20, 25, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_210, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_210", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_210() */
/* END TEST 210 */

/*
 *  BEGIN TEST 211
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_211( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {-8, 27, -50, -17, -1, -15, -42, 38, 60, 24, 0, 20 };
	oski_value_t alpha = -5;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_211, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_211", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_211() */
/* END TEST 211 */

/*
 *  BEGIN TEST 212
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_212( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {-4, -13, -18, 0, -6, -3, 7, 10, -6, 5, 2, 9, 0, 4, -5 };
	oski_value_t alpha = 1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_212, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_212", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_212() */
/* END TEST 212 */

/*
 *  BEGIN TEST 213
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_213( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {16, 43, 78, -24, 42, -21, -13, -58, 12, -35, 46, -3, -60, -28, 35 };
	oski_value_t alpha = -1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_213, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_213", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_213() */
/* END TEST 213 */

/*
 *  BEGIN TEST 214
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_214( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {18, 45, 90, -36, 54, -36, -9, -72, 9, -45, 72, 9, -90, -36, 45 };
	oski_value_t alpha = 0;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_214, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_214", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_214() */
/* END TEST 214 */

/*
 *  BEGIN TEST 215
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_215( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {-14, -32, -72, 36, -48, 39, 2, 62, -3, 40, -74, -18, 90, 32, -40 };
	oski_value_t alpha = -1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_215, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_215", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_215() */
/* END TEST 215 */

/*
 *  BEGIN TEST 216
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_216( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {2, 5, 10, -4, -4, -1, -8, 1, 8, 1, -10, -4 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_216, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_216", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_216() */
/* END TEST 216 */

/*
 *  BEGIN TEST 217
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_217( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {14, 51, -90, 68, -44, -79, 72, -17, 72, 31, 50, 4 };
	oski_value_t alpha = -1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_217, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_217", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_217() */
/* END TEST 217 */

/*
 *  BEGIN TEST 218
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_218( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {-10, -35, 50, -40, 30, 50, -40, 10, -50, -20, -25, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_218, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_218", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_218() */
/* END TEST 218 */

/*
 *  BEGIN TEST 219
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_219( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {-8, -32, 80, -56, 28, 58, -64, 14, -44, -22, -50, -8 };
	oski_value_t alpha = 2;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_219, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_219", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_219() */
/* END TEST 219 */

/*
 *  BEGIN TEST 220
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_220( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {-8, -22, -20, 4, -18, 18, 13, 16, -1, 15, -34, -7, 25, 12, -15 };
	oski_value_t alpha = 1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_220, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_220", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_220() */
/* END TEST 220 */

/*
 *  BEGIN TEST 221
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_221( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {-8, -18, -60, 28, -30, 14, -5, 48, -7, 25, -30, -1, 55, 20, -25 };
	oski_value_t alpha = -1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_221, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_221", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_221() */
/* END TEST 221 */

/*
 *  BEGIN TEST 222
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_222( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {2, 5, 10, -4, 6, -4, -1, -8, 1, -5, 8, 1, -10, -4, 5 };
	oski_value_t alpha = 0;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_222, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_222", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_222() */
/* END TEST 222 */

/*
 *  BEGIN TEST 223
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_223( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {-2, -3, -30, 16, -12, 2, -8, 24, -4, 10, -6, 2, 25, 8, -10 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_223, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_223", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_223() */
/* END TEST 223 */

#define A224_M 4    /* # of rows */
#define A224_N 5    /* # of cols */
#define A224_NNZ 4  /* # of stored non-zeros */
static oski_index_t A224_ptr[A224_M+1] = {0, 1, 2, 3, 4}; /* end row pointers for A224 */
static oski_index_t A224_ind[A224_NNZ] = {2, 2, 4, 4}; /* end column indices pointers for A224 */
static oski_value_t A224_val[A224_NNZ] = {10, -7, -10, 2}; /* end stored non-zero values for A224 */

#define x224_LEN 5   /* vector length */
#define x224_VW  3   /* vector width */
static oski_value_t x224_val[x224_LEN*x224_VW] = {-10, 2, 9, 1, 5, 6, 5, -10, -6, 4, -3, -9, -7, 10, -6}; /* data */

#define y224_LEN 4   /* vector length */
#define y224_VW  3   /* vector width */
static oski_value_t y224_val[y224_LEN*y224_VW] = {-2, 2, -7, -10, 5, 6, 5, -9, 2, 10, 10, -1}; /* data */

/*
 *  BEGIN TEST 224
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_224( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	const oski_value_t y_true[] = {40, -98, -51, -34, 75, 48, 75, -110, 54, -10, 17, -21 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_224, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_224", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_224() */
/* END TEST 224 */

/*
 *  BEGIN TEST 225
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_225( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	const oski_value_t y_true[] = {-52, 110, 9, -26, -45, -12, -45, 56, -42, 70, 43, 15 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_225, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_225", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_225() */
/* END TEST 225 */

/*
 *  BEGIN TEST 226
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_226( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	const oski_value_t y_true[] = {-16, 16, -56, -80, 40, 48, 40, -72, 16, 80, 80, -8 };
	oski_value_t alpha = 0;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_226, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_226", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_226() */
/* END TEST 226 */

/*
 *  BEGIN TEST 227
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_227( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	const oski_value_t y_true[] = {72, -188, -130, -108, 170, 120, 170, -256, 116, 20, 74, -46 };
	oski_value_t alpha = 2;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_227, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_227", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_227() */
/* END TEST 227 */

/*
 *  BEGIN TEST 228
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_228( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	const oski_value_t y_true[] = {-2, 2, -7, -10, 5, 6, 55, -24, -110, 10, 10, -1, -30, 110, -22 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_228, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_228", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_228() */
/* END TEST 228 */

/*
 *  BEGIN TEST 229
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_229( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	const oski_value_t y_true[] = {2, -2, 7, 10, -5, -6, -55, 24, 110, -10, -10, 1, 30, -110, 22 };
	oski_value_t alpha = -1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_229, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_229", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_229() */
/* END TEST 229 */

/*
 *  BEGIN TEST 230
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_230( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	const oski_value_t y_true[] = {60, -12, -54, -6, -30, -36, -30, 60, 36, -24, 18, 54, 42, -60, 36 };
	oski_value_t alpha = 0;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_230, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_230", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_230() */
/* END TEST 230 */

/*
 *  BEGIN TEST 231
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_231( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	const oski_value_t y_true[] = {-28, 20, -54, -89, 50, 60, 500, -226, -996, 94, 87, -18, -277, 1000, -204 };
	oski_value_t alpha = 9;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_231, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_231", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_231() */
/* END TEST 231 */

/*
 *  BEGIN TEST 232
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_232( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	const oski_value_t y_true[] = {48, -42, 103, 30, -118, 51, -130, 53, -59, 8, 14, -17 };
	oski_value_t alpha = 1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_232, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_232", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_232() */
/* END TEST 232 */

/*
 *  BEGIN TEST 233
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_233( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	const oski_value_t y_true[] = {-42, 36, -82, 0, 103, -69, 115, -26, 53, -38, -44, 20 };
	oski_value_t alpha = -1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_233, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_233", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_233() */
/* END TEST 233 */

/*
 *  BEGIN TEST 234
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_234( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	const oski_value_t y_true[] = {-8, 8, -28, -40, 20, 24, 20, -36, 8, 40, 40, -4 };
	oski_value_t alpha = 0;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_234, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_234", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_234() */
/* END TEST 234 */

/*
 *  BEGIN TEST 235
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_235( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	const oski_value_t y_true[] = {134, -116, 274, 40, -329, 183, -365, 114, -167, 74, 92, -56 };
	oski_value_t alpha = 3;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_235, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_235", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_235() */
/* END TEST 235 */

/*
 *  BEGIN TEST 236
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_236( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	const oski_value_t y_true[] = {-12, 7, 11, 3, 11, 16, -36, 3, -46, -6, -12, -10, 43, -58, -108 };
	oski_value_t alpha = 1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_236, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_236", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_236() */
/* END TEST 236 */

/*
 *  BEGIN TEST 237
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_237( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	const oski_value_t y_true[] = {-18, -1, 16, 0, 4, 2, 51, -33, 28, 18, 3, -17, -64, 88, 90 };
	oski_value_t alpha = -1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_237, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_237", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_237() */
/* END TEST 237 */

/*
 *  BEGIN TEST 238
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_238( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	const oski_value_t y_true[] = {-30, 6, 27, 3, 15, 18, 15, -30, -18, 12, -9, -27, -21, 30, -18 };
	oski_value_t alpha = 0;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_238, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_238", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_238() */
/* END TEST 238 */

/*
 *  BEGIN TEST 239
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_ROWMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_239( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_ROWMAJ, x224_VW );
	const oski_value_t y_true[] = {-10, 2, 9, 1, 5, 6, 5, -10, -6, 4, -3, -9, -7, 10, -6 };
	oski_value_t alpha = 0;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_239, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_239", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_239() */
/* END TEST 239 */

/*
 *  BEGIN TEST 240
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_240( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	const oski_value_t y_true[] = {76, -90, -87, -81, 85, 52, -31, -68, 57, 31, 22, -4 };
	oski_value_t alpha = 1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_240, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_240", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_240() */
/* END TEST 240 */

/*
 *  BEGIN TEST 241
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_241( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	const oski_value_t y_true[] = {-78, 92, 80, 71, -80, -46, 36, 59, -55, -21, -12, 3 };
	oski_value_t alpha = -1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_241, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_241", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_241() */
/* END TEST 241 */

/*
 *  BEGIN TEST 242
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_242( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	const oski_value_t y_true[] = {12, -12, 42, 60, -30, -36, -30, 54, -12, -60, -60, 6 };
	oski_value_t alpha = 0;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_242, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_242", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_242() */
/* END TEST 242 */

/*
 *  BEGIN TEST 243
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_243( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	const oski_value_t y_true[] = {-8, 8, -28, -40, 20, 24, 20, -36, 8, 40, 40, -4 };
	oski_value_t alpha = 0;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_243, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_243", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_243() */
/* END TEST 243 */

/*
 *  BEGIN TEST 244
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_244( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	const oski_value_t y_true[] = {78, -26, -17, 2, -70, -46, -35, 56, 58, 78, 17, 78, -54, -81, 26 };
	oski_value_t alpha = 1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_244, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_244", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_244() */
/* END TEST 244 */

/*
 *  BEGIN TEST 245
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_245( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	const oski_value_t y_true[] = {82, -6, -127, -18, -10, -50, -45, 104, 38, -142, 31, 66, 166, -79, 70 };
	oski_value_t alpha = -1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_245, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_245", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_245() */
/* END TEST 245 */

/*
 *  BEGIN TEST 246
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_246( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	const oski_value_t y_true[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_246, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_246", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_246() */
/* END TEST 246 */

/*
 *  BEGIN TEST 247
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_247( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_ROWMAJ, y224_VW );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	const oski_value_t y_true[] = {86, 14, -237, -38, 50, -54, -55, 152, 18, -362, 45, 54, 386, -77, 114 };
	oski_value_t alpha = -3;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_247, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_247", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_247() */
/* END TEST 247 */

/*
 *  BEGIN TEST 248
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_248( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	const oski_value_t y_true[] = {90, -71, -6, 61, -119, 45, -75, 47, -83, -10, 3, 3 };
	oski_value_t alpha = 1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_248, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_248", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_248() */
/* END TEST 248 */

/*
 *  BEGIN TEST 249
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_249( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	const oski_value_t y_true[] = {-66, 47, 90, 59, 59, -117, 15, 61, 59, -110, -123, 9 };
	oski_value_t alpha = -1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_249, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_249", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_249() */
/* END TEST 249 */

/*
 *  BEGIN TEST 250
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_250( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	const oski_value_t y_true[] = {4, -4, 14, 20, -10, -12, -10, 18, -4, -20, -20, 2 };
	oski_value_t alpha = 0;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_250, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_250", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_250() */
/* END TEST 250 */

/*
 *  BEGIN TEST 251
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_251( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	oski_value_t* y_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	const oski_value_t y_true[] = {76, -57, -55, -9, -84, 87, -40, -16, -69, 60, 73, -4 };
	oski_value_t alpha = 1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_251, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_251", y_true, y_val, y224_LEN*y224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_251() */
/* END TEST 251 */

/*
 *  BEGIN TEST 252
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_252( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	const oski_value_t y_true[] = {-22, 6, -23, -8, 60, 17, 16, -7, -21, -60, -4, -8, -54, 19, -114 };
	oski_value_t alpha = 1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_252, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_252", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_252() */
/* END TEST 252 */

/*
 *  BEGIN TEST 253
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_253( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	const oski_value_t y_true[] = {-78, 14, 113, 18, -10, 43, 34, -93, -39, 100, -26, -82, -16, 81, 54 };
	oski_value_t alpha = -1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_253, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_253", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_253() */
/* END TEST 253 */

/*
 *  BEGIN TEST 254
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_254( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	const oski_value_t y_true[] = {-50, 10, 45, 5, 25, 30, 25, -50, -30, 20, -15, -45, -35, 50, -30 };
	oski_value_t alpha = 0;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_254, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_254", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_254() */
/* END TEST 254 */

/*
 *  BEGIN TEST 255
 *  Value type: real
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x224', LAYOUT_COLMAJ>
 *  y : <'y224', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_255( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A224_ptr, A224_ind, A224_val,
		A224_M, A224_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y224_val,
		y224_LEN * y224_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y224_LEN, y224_VW, LAYOUT_COLMAJ, y224_LEN );
	oski_value_t* y_val = array_Duplicate( x224_val,
		x224_LEN * x224_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x224_LEN, x224_VW, LAYOUT_COLMAJ, x224_LEN );
	const oski_value_t y_true[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_255, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_255", y_true, y_val, x224_LEN*x224_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_255() */
/* END TEST 255 */

#define A256_M 4    /* # of rows */
#define A256_N 5    /* # of cols */
#define A256_NNZ 2  /* # of stored non-zeros */
static oski_index_t A256_ptr[A256_M+1] = {1, 1, 1, 3, 3}; /* end row pointers for A256 */
static oski_index_t A256_ind[A256_NNZ] = {2, 3}; /* end column indices pointers for A256 */
static oski_value_t A256_val[A256_NNZ] = {10, -10}; /* end stored non-zero values for A256 */

#define x256_LEN 5   /* vector length */
#define x256_VW  3   /* vector width */
static oski_value_t x256_val[x256_LEN*x256_VW] = {-7, 9, -5, 4, -10, 10, -4, 8, -4, 10, 1, -8, -1, 5, -1}; /* data */

#define y256_LEN 4   /* vector length */
#define y256_VW  3   /* vector width */
static oski_value_t y256_val[y256_LEN*y256_VW] = {-9, 10, -9, 1, -2, -3, 10, -2, -2, -10, 5, 2}; /* data */

/*
 *  BEGIN TEST 256
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_256( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	const oski_value_t y_true[] = {45, -50, 45, -5, 10, 15, 30, -170, 150, 50, -25, -10 };
	oski_value_t alpha = 1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_256, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_256", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_256() */
/* END TEST 256 */

/*
 *  BEGIN TEST 257
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_257( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	const oski_value_t y_true[] = {63, -70, 63, -7, 14, 21, -150, 194, -126, 70, -35, -14 };
	oski_value_t alpha = -1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_257, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_257", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_257() */
/* END TEST 257 */

/*
 *  BEGIN TEST 258
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_258( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	const oski_value_t y_true[] = {-81, 90, -81, 9, -18, -27, 90, -18, -18, -90, 45, 18 };
	oski_value_t alpha = 0;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_258, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_258", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_258() */
/* END TEST 258 */

/*
 *  BEGIN TEST 259
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_259( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	const oski_value_t y_true[] = {0, 0, 0, 0, 0, 0, 560, -1260, 980, 0, 0, 0 };
	oski_value_t alpha = 7;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_259, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_259", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_259() */
/* END TEST 259 */

/*
 *  BEGIN TEST 260
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_260( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	const oski_value_t y_true[] = {14, -18, 10, 92, 0, -40, -92, 4, 28, -20, -2, 16, 2, -10, 2 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_260, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_260", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_260() */
/* END TEST 260 */

/*
 *  BEGIN TEST 261
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_261( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	const oski_value_t y_true[] = {-35, 45, -25, -80, -30, 70, 80, 20, -40, 50, 5, -40, -5, 25, -5 };
	oski_value_t alpha = -1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_261, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_261", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_261() */
/* END TEST 261 */

/*
 *  BEGIN TEST 262
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_262( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	const oski_value_t y_true[] = {42, -54, 30, -24, 60, -60, 24, -48, 24, -60, -6, 48, 6, -30, 6 };
	oski_value_t alpha = 0;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_262, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_262", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_262() */
/* END TEST 262 */

/*
 *  BEGIN TEST 263
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_263( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	const oski_value_t y_true[] = {56, -72, 40, -632, 200, 40, 632, -184, -88, -80, -8, 64, 8, -40, 8 };
	oski_value_t alpha = -6;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_263, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_263", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_263() */
/* END TEST 263 */

/*
 *  BEGIN TEST 264
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_264( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	const oski_value_t y_true[] = {54, -60, 134, -6, 12, 18, -240, 12, 12, 60, 110, -12 };
	oski_value_t alpha = 1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_264, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_264", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_264() */
/* END TEST 264 */

/*
 *  BEGIN TEST 265
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_265( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	const oski_value_t y_true[] = {9, -10, -71, -1, 2, 3, 170, 2, 2, 10, -145, -2 };
	oski_value_t alpha = -1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_265, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_265", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_265() */
/* END TEST 265 */

/*
 *  BEGIN TEST 266
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_266( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	const oski_value_t y_true[] = {18, -20, 18, -2, 4, 6, -20, 4, 4, 20, -10, -4 };
	oski_value_t alpha = 0;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_266, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_266", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_266() */
/* END TEST 266 */

/*
 *  BEGIN TEST 267
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_267( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	const oski_value_t y_true[] = {9, -10, 569, -1, 2, 3, -1270, 2, 2, 10, 975, -2 };
	oski_value_t alpha = 7;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_267, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_267", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_267() */
/* END TEST 267 */

/*
 *  BEGIN TEST 268
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_268( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	const oski_value_t y_true[] = {14, -18, 10, -98, 120, 30, 98, -116, -42, -20, -2, 16, 2, -10, 2 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_268, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_268", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_268() */
/* END TEST 268 */

/*
 *  BEGIN TEST 269
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_269( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	const oski_value_t y_true[] = {-21, 27, -15, 102, -130, -20, -102, 124, 38, 30, 3, -24, -3, 15, -3 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_269, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_269", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_269() */
/* END TEST 269 */

/*
 *  BEGIN TEST 270
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_270( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	const oski_value_t y_true[] = {21, -27, 15, -12, 30, -30, 12, -24, 12, -30, -3, 24, 3, -15, 3 };
	oski_value_t alpha = 0;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_270, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_270", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_270() */
/* END TEST 270 */

/*
 *  BEGIN TEST 271
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_ROWMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_271( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_ROWMAJ, x256_VW );
	const oski_value_t y_true[] = {-42, 54, -30, 744, -860, -340, -744, 848, 376, 60, 6, -48, -6, 30, -6 };
	oski_value_t alpha = -8;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_271, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_271", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_271() */
/* END TEST 271 */

/*
 *  BEGIN TEST 272
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_272( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	const oski_value_t y_true[] = {-63, 70, -63, 7, -14, -21, 210, -134, -84, -70, 35, 14 };
	oski_value_t alpha = 1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_272, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_272", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_272() */
/* END TEST 272 */

/*
 *  BEGIN TEST 273
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_273( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	const oski_value_t y_true[] = {-72, 80, -72, 8, -16, -24, -60, 104, 54, -80, 40, 16 };
	oski_value_t alpha = -1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_273, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_273", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_273() */
/* END TEST 273 */

/*
 *  BEGIN TEST 274
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_274( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	const oski_value_t y_true[] = {-63, 70, -63, 7, -14, -21, 70, -14, -14, -70, 35, 14 };
	oski_value_t alpha = 0;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_274, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_274", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_274() */
/* END TEST 274 */

/*
 *  BEGIN TEST 275
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_275( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	const oski_value_t y_true[] = {54, -60, 54, -6, 12, 18, 640, -588, -338, 60, -30, -12 };
	oski_value_t alpha = 5;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_275, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_275", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_275() */
/* END TEST 275 */

/*
 *  BEGIN TEST 276
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_276( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	const oski_value_t y_true[] = {-7, 109, -105, 4, -10, 10, -24, 28, -4, 10, 1, -28, 19, 5, -1 };
	oski_value_t alpha = 1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_276, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_276", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_276() */
/* END TEST 276 */

/*
 *  BEGIN TEST 277
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_277( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	const oski_value_t y_true[] = {56, -172, 140, -32, 80, -80, 52, -84, 32, -80, -8, 84, -12, -40, 8 };
	oski_value_t alpha = -1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_277, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_277", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_277() */
/* END TEST 277 */

/*
 *  BEGIN TEST 278
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_278( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	const oski_value_t y_true[] = {-35, 45, -25, 20, -50, 50, -20, 40, -20, 50, 5, -40, -5, 25, -5 };
	oski_value_t alpha = 0;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_278, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_278", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_278() */
/* END TEST 278 */

/*
 *  BEGIN TEST 279
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_279( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_ROWMAJ, y256_VW );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	const oski_value_t y_true[] = {-49, 63, -35, 28, -70, 70, -28, 56, -28, 70, 7, -56, -7, 35, -7 };
	oski_value_t alpha = 0;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_279, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_279", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_279() */
/* END TEST 279 */

/*
 *  BEGIN TEST 280
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_280( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	const oski_value_t y_true[] = {36, -40, 176, -4, 8, 12, -160, 8, 8, 40, -90, -8 };
	oski_value_t alpha = 1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_280, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_280", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_280() */
/* END TEST 280 */

/*
 *  BEGIN TEST 281
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_281( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	const oski_value_t y_true[] = {-63, 70, -203, 7, -14, -21, 190, -14, -14, -70, 105, 14 };
	oski_value_t alpha = -1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_281, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_281", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_281() */
/* END TEST 281 */

/*
 *  BEGIN TEST 282
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_282( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	const oski_value_t y_true[] = {-18, 20, -18, 2, -4, -6, 20, -4, -4, -20, 10, 4 };
	oski_value_t alpha = 0;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_282, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_282", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_282() */
/* END TEST 282 */

/*
 *  BEGIN TEST 283
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_283( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	oski_value_t* y_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	const oski_value_t y_true[] = {-54, 60, -334, 6, -12, -18, 300, -12, -12, -60, 170, 12 };
	oski_value_t alpha = -2;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_283, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_283", y_true, y_val, y256_LEN*y256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_283() */
/* END TEST 283 */

/*
 *  BEGIN TEST 284
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_284( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	const oski_value_t y_true[] = {0, -90, 90, 0, 0, 0, 100, -100, 0, 0, 0, 50, -50, 0, 0 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_284, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_284", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_284() */
/* END TEST 284 */

/*
 *  BEGIN TEST 285
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_285( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	const oski_value_t y_true[] = {-56, 162, -130, 32, -80, 80, -132, 164, -32, 80, 8, -114, 42, 40, -8 };
	oski_value_t alpha = -1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_285, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_285", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_285() */
/* END TEST 285 */

/*
 *  BEGIN TEST 286
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_286( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	const oski_value_t y_true[] = {7, -9, 5, -4, 10, -10, 4, -8, 4, -10, -1, 8, 1, -5, 1 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_286, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_286", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_286() */
/* END TEST 286 */

/*
 *  BEGIN TEST 287
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x256', LAYOUT_COLMAJ>
 *  y : <'y256', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_287( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A256_ptr, A256_ind, A256_val,
		A256_M, A256_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y256_val,
		y256_LEN * y256_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y256_LEN, y256_VW, LAYOUT_COLMAJ, y256_LEN );
	oski_value_t* y_val = array_Duplicate( x256_val,
		x256_LEN * x256_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x256_LEN, x256_VW, LAYOUT_COLMAJ, x256_LEN );
	const oski_value_t y_true[] = {-28, -54, 70, 16, -40, 40, 84, -68, -16, 40, 4, 18, -54, 20, -4 };
	oski_value_t alpha = 1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_287, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_287", y_true, y_val, x256_LEN*x256_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_287() */
/* END TEST 287 */

#define A288_M 4    /* # of rows */
#define A288_N 5    /* # of cols */
#define A288_NNZ 4  /* # of stored non-zeros */
static oski_index_t A288_ptr[A288_M+1] = {0, 1, 1, 3, 4}; /* end row pointers for A288 */
static oski_index_t A288_ind[A288_NNZ] = {0, 0, 2, 0}; /* end column indices pointers for A288 */
static oski_value_t A288_val[A288_NNZ] = {4, -10, 10, 9}; /* end stored non-zero values for A288 */

#define x288_LEN 5   /* vector length */
#define x288_VW  3   /* vector width */
static oski_value_t x288_val[x288_LEN*x288_VW] = {-9, -3, -3, 4, -9, -10, -9, -1, 4, -6, -3, 10, 2, 3, 9}; /* data */

#define y288_LEN 4   /* vector length */
#define y288_VW  3   /* vector width */
static oski_value_t y288_val[y288_LEN*y288_VW] = {-3, 8, 7, -2, -10, 0, 0, -1, 0, 6, 10, -5}; /* data */

/*
 *  BEGIN TEST 288
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_288( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {-18, -60, -54, 12, 60, 0, 0, 26, 70, -117, -87, 3 };
	oski_value_t alpha = 1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_288, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_288", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_288() */
/* END TEST 288 */

/*
 *  BEGIN TEST 289
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_289( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {57, -44, -37, 14, 70, 0, 0, -13, -70, 39, -43, 62 };
	oski_value_t alpha = -1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_289, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_289", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_289() */
/* END TEST 289 */

/*
 *  BEGIN TEST 290
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_290( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {24, -64, -56, 16, 80, 0, 0, 8, 0, -48, -80, 40 };
	oski_value_t alpha = 0;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_290, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_290", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_290() */
/* END TEST 290 */

/*
 *  BEGIN TEST 291
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_291( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {-318, -124, -122, 4, 20, 0, 0, 182, 630, -741, -263, -233 };
	oski_value_t alpha = 9;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_291, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_291", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_291() */
/* END TEST 291 */

/*
 *  BEGIN TEST 292
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_292( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {87, 147, -2, -20, 45, 50, 45, -5, -20, 30, 15, -50, -10, -15, -45 };
	oski_value_t alpha = 1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_292, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_292", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_292() */
/* END TEST 292 */

/*
 *  BEGIN TEST 293
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_293( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {-114, -156, -7, 32, -72, -80, -72, 2, 32, -48, -24, 80, 16, 24, 72 };
	oski_value_t alpha = -1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_293, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_293", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_293() */
/* END TEST 293 */

/*
 *  BEGIN TEST 294
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_294( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {-54, -18, -18, 24, -54, -60, -54, -6, 24, -36, -18, 60, 12, 18, 54 };
	oski_value_t alpha = 0;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_294, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_294", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_294() */
/* END TEST 294 */

/*
 *  BEGIN TEST 295
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_295( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {-147, -639, 106, -28, 63, 70, 63, 57, -28, 42, 21, -70, -14, -21, -63 };
	oski_value_t alpha = -5;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_295, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_295", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_295() */
/* END TEST 295 */

/*
 *  BEGIN TEST 296
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_296( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {-21, -40, -35, -71, 38, 0, 20, -22, -12, -30, 20, -2 };
	oski_value_t alpha = 1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_296, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_296", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_296() */
/* END TEST 296 */

/*
 *  BEGIN TEST 297
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_297( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {45, -24, -21, 87, 42, 0, -20, 30, 12, -18, -100, 42 };
	oski_value_t alpha = -1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_297, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_297", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_297() */
/* END TEST 297 */

/*
 *  BEGIN TEST 298
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_298( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {-3, 8, 7, -2, -10, 0, 0, -1, 0, 6, 10, -5 };
	oski_value_t alpha = 0;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_298, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_298", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_298() */
/* END TEST 298 */

/*
 *  BEGIN TEST 299
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_299( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {30, 16, 14, 77, -8, 0, -20, 25, 12, 12, -50, 17 };
	oski_value_t alpha = -1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_299, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_299", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_299() */
/* END TEST 299 */

/*
 *  BEGIN TEST 300
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_300( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {-91, -46, -142, -4, 9, 10, 79, 1, 96, 6, 3, -10, -2, -3, -9 };
	oski_value_t alpha = 1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_300, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_300", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_300() */
/* END TEST 300 */

/*
 *  BEGIN TEST 301
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_301( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {118, 55, 151, -8, 18, 20, -52, 2, -108, 12, 6, -20, -4, -6, -18 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_301, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_301", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_301() */
/* END TEST 301 */

/*
 *  BEGIN TEST 302
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_302( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {18, 6, 6, -8, 18, 20, 18, 2, -8, 12, 6, -20, -4, -6, -18 };
	oski_value_t alpha = 0;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_302, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_302", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_302() */
/* END TEST 302 */

/*
 *  BEGIN TEST 303
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_303( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {-191, -95, -287, -4, 9, 10, 149, 1, 196, 6, 3, -10, -2, -3, -9 };
	oski_value_t alpha = 2;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_303, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_303", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_303() */
/* END TEST 303 */

/*
 *  BEGIN TEST 304
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_304( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {-18, -88, -54, 12, 60, 0, 60, 96, 50, -117, -150, 3 };
	oski_value_t alpha = 1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_304, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_304", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_304() */
/* END TEST 304 */

/*
 *  BEGIN TEST 305
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_305( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {27, 64, 33, -6, -30, 0, -60, -93, -50, 99, 120, 12 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_305, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_305", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_305() */
/* END TEST 305 */

/*
 *  BEGIN TEST 306
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_306( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {-27, 72, 63, -18, -90, 0, 0, -9, 0, 54, 90, -45 };
	oski_value_t alpha = 0;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_306, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_306", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_306() */
/* END TEST 306 */

/*
 *  BEGIN TEST 307
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_307( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {-138, -176, -62, 4, 20, 0, 240, 362, 200, -336, -380, -98 };
	oski_value_t alpha = 4;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_307, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_307", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_307() */
/* END TEST 307 */

/*
 *  BEGIN TEST 308
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_308( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {-12, -18, -18, 24, -54, 72, -54, -16, 24, -36, -35, 60, 12, 18, 54 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_308, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_308", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_308() */
/* END TEST 308 */

/*
 *  BEGIN TEST 309
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_309( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {-96, -18, -18, 24, -54, -192, -54, 4, 24, -36, -1, 60, 12, 18, 54 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_309, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_309", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_309() */
/* END TEST 309 */

/*
 *  BEGIN TEST 310
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_310( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {-45, -15, -15, 20, -45, -50, -45, -5, 20, -30, -15, 50, 10, 15, 45 };
	oski_value_t alpha = 0;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_310, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_310", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_310() */
/* END TEST 310 */

/*
 *  BEGIN TEST 311
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_311( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {-3, 27, 27, -36, 81, -174, 81, 29, -36, 54, 61, -90, -18, -27, -81 };
	oski_value_t alpha = -2;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_311, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_311", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_311() */
/* END TEST 311 */

/*
 *  BEGIN TEST 312
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_312( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {-36, 0, 60, -81, -40, 0, 90, -90, -12, 0, 50, -27 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_312, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_312", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_312() */
/* END TEST 312 */

/*
 *  BEGIN TEST 313
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_313( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {27, 24, -39, 75, 10, 0, -90, 87, 12, 18, -20, 12 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_313, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_313", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_313() */
/* END TEST 313 */

/*
 *  BEGIN TEST 314
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_314( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {-9, 24, 21, -6, -30, 0, 0, -3, 0, 18, 30, -15 };
	oski_value_t alpha = 0;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_314, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_314", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_314() */
/* END TEST 314 */

/*
 *  BEGIN TEST 315
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_315( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {-123, 40, 215, -253, -170, 0, 270, -275, -36, 30, 200, -106 };
	oski_value_t alpha = 3;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_315, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_315", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_315() */
/* END TEST 315 */

/*
 *  BEGIN TEST 316
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_316( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {-82, 6, 76, -8, 18, -29, 18, 2, -8, 12, -139, -20, 96, -6, -18 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_316, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_316", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_316() */
/* END TEST 316 */

/*
 *  BEGIN TEST 317
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_317( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {127, 9, -61, -12, 27, 79, 27, 3, -12, 18, 154, -30, -106, -9, -27 };
	oski_value_t alpha = -1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_317, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_317", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_317() */
/* END TEST 317 */

/*
 *  BEGIN TEST 318
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_318( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {18, 6, 6, -8, 18, 20, 18, 2, -8, 12, 6, -20, -4, -6, -18 };
	oski_value_t alpha = 0;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_318, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_318", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_318() */
/* END TEST 318 */

/*
 *  BEGIN TEST 319
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_319( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {273, -9, -219, 12, -27, 117, -27, -3, 12, -18, 426, 30, -294, 9, 27 };
	oski_value_t alpha = -3;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_319, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_319", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_319() */
/* END TEST 319 */

#define A320_M 4    /* # of rows */
#define A320_N 5    /* # of cols */
#define A320_NNZ 1  /* # of stored non-zeros */
static oski_index_t A320_ptr[A320_M+1] = {1, 1, 1, 1, 2}; /* end row pointers for A320 */
static oski_index_t A320_ind[A320_NNZ] = {2}; /* end column indices pointers for A320 */
static oski_value_t A320_val[A320_NNZ] = {10}; /* end stored non-zero values for A320 */

#define x320_LEN 5   /* vector length */
#define x320_VW  3   /* vector width */
static oski_value_t x320_val[x320_LEN*x320_VW] = {-10, 9, 3, -3, 0, -3, 2, 7, 10, -5, 1, -1, -9, -8, -5}; /* data */

#define y320_LEN 4   /* vector length */
#define y320_VW  3   /* vector width */
static oski_value_t y320_val[y320_LEN*y320_VW] = {0, 7, -4, 7, -7, -10, -5, 10, 1, -5, -3, 2}; /* data */

/*
 *  BEGIN TEST 320
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_320( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	const oski_value_t y_true[] = {-10, 65, -29, 53, -56, -83, -38, 87, 18, -75, -23, -15 };
	oski_value_t alpha = 1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_320, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_320", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_320() */
/* END TEST 320 */

/*
 *  BEGIN TEST 321
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_321( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	const oski_value_t y_true[] = {10, -23, 5, -11, 14, 23, 8, -27, -12, 45, 5, 27 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_321, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_321", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_321() */
/* END TEST 321 */

/*
 *  BEGIN TEST 322
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_322( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	const oski_value_t y_true[] = {0, -56, 32, -56, 56, 80, 40, -80, -8, 40, 24, -16 };
	oski_value_t alpha = 0;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_322, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_322", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_322() */
/* END TEST 322 */

/*
 *  BEGIN TEST 323
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_323( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	const oski_value_t y_true[] = {10, 5, -11, 17, -14, -17, -12, 13, -8, 25, -7, 35 };
	oski_value_t alpha = -1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_323, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_323", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_323() */
/* END TEST 323 */

/*
 *  BEGIN TEST 324
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_324( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	const oski_value_t y_true[] = {-40, 43, 8, -55, -37, -2, 3, 38, 41, -25, 1, -2, -36, -32, -20 };
	oski_value_t alpha = 1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_324, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_324", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_324() */
/* END TEST 324 */

/*
 *  BEGIN TEST 325
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_325( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	const oski_value_t y_true[] = {-70, 56, 25, 22, 37, -31, 19, 39, 69, -30, 10, -9, -63, -56, -35 };
	oski_value_t alpha = -1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_325, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_325", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_325() */
/* END TEST 325 */

/*
 *  BEGIN TEST 326
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_326( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	const oski_value_t y_true[] = {70, -63, -21, 21, 0, 21, -14, -49, -70, 35, -7, 7, 63, 56, 35 };
	oski_value_t alpha = 0;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_326, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_326", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_326() */
/* END TEST 326 */

/*
 *  BEGIN TEST 327
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_327( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	const oski_value_t y_true[] = {-30, -22, 37, 292, 259, -79, 41, -49, 23, 20, 24, -17, -27, -24, -15 };
	oski_value_t alpha = -7;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_327, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_327", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_327() */
/* END TEST 327 */

/*
 *  BEGIN TEST 328
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_328( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	const oski_value_t y_true[] = {-10, 39, -22, 7, -33, -60, -23, 61, 9, -33, -8, -19 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_328, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_328", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_328() */
/* END TEST 328 */

/*
 *  BEGIN TEST 329
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_329( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	const oski_value_t y_true[] = {10, 10, -6, 42, -16, -10, -12, 9, -2, -2, -13, 33 };
	oski_value_t alpha = -1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_329, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_329", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_329() */
/* END TEST 329 */

/*
 *  BEGIN TEST 330
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_330( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	const oski_value_t y_true[] = {0, 21, -12, 21, -21, -30, -15, 30, 3, -15, -9, 6 };
	oski_value_t alpha = 0;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_330, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_330", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_330() */
/* END TEST 330 */

/*
 *  BEGIN TEST 331
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_331( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	const oski_value_t y_true[] = {-90, -90, 54, -378, 144, 90, 108, -81, 18, 18, 117, -297 };
	oski_value_t alpha = 9;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_331, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_331", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_331() */
/* END TEST 331 */

/*
 *  BEGIN TEST 332
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_332( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	const oski_value_t y_true[] = {-80, 65, 25, 53, 90, -9, 12, 51, 77, -33, 18, -6, -72, -64, -40 };
	oski_value_t alpha = 1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_332, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_332", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_332() */
/* END TEST 332 */

/*
 *  BEGIN TEST 333
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_333( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	const oski_value_t y_true[] = {50, -38, -16, -62, -90, 0, -6, -30, -47, 18, -15, 3, 45, 40, 25 };
	oski_value_t alpha = -1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_333, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_333", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_333() */
/* END TEST 333 */

/*
 *  BEGIN TEST 334
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_334( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	const oski_value_t y_true[] = {-80, 72, 24, -24, 0, -24, 16, 56, 80, -40, 8, -8, -72, -64, -40 };
	oski_value_t alpha = 0;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_334, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_334", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_334() */
/* END TEST 334 */

/*
 *  BEGIN TEST 335
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_ROWMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_335( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_ROWMAJ, x320_VW );
	const oski_value_t y_true[] = {40, -36, -12, 12, 0, 12, -8, -28, -40, 20, -4, 4, 36, 32, 20 };
	oski_value_t alpha = 0;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_335, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_335", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_335() */
/* END TEST 335 */

/*
 *  BEGIN TEST 336
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_336( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	const oski_value_t y_true[] = {-10, 25, -15, 37, -26, -41, -17, 47, -5, 67, 18, -10 };
	oski_value_t alpha = 1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_336, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_336", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_336() */
/* END TEST 336 */

/*
 *  BEGIN TEST 337
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_337( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	const oski_value_t y_true[] = {10, 17, -9, 5, -16, -19, -13, 13, 11, -97, -36, 22 };
	oski_value_t alpha = -1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_337, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_337", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_337() */
/* END TEST 337 */

/*
 *  BEGIN TEST 338
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_338( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	const oski_value_t y_true[] = {0, 42, -24, 42, -42, -60, -30, 60, 6, -30, -18, 12 };
	oski_value_t alpha = 0;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_338, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_338", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_338() */
/* END TEST 338 */

/*
 *  BEGIN TEST 339
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_339( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	const oski_value_t y_true[] = {-60, 3, -6, 75, -9, -36, 3, 72, -51, 507, 171, -102 };
	oski_value_t alpha = 6;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_339, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_339", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_339() */
/* END TEST 339 */

/*
 *  BEGIN TEST 340
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_340( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	const oski_value_t y_true[] = {-40, -7, 7, -17, 0, -5, -29, 38, 37, -20, 0, 6, -35, -30, -20 };
	oski_value_t alpha = 1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_340, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_340", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_340() */
/* END TEST 340 */

/*
 *  BEGIN TEST 341
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_341( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	const oski_value_t y_true[] = {-30, 70, 14, -4, 0, -16, 43, 11, 33, -15, 7, -13, -28, -26, -15 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_341, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_341", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_341() */
/* END TEST 341 */

/*
 *  BEGIN TEST 342
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_342( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	const oski_value_t y_true[] = {30, -27, -9, 9, 0, 9, -6, -21, -30, 15, -3, 3, 27, 24, 15 };
	oski_value_t alpha = 0;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_342, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_342", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_342() */
/* END TEST 342 */

/*
 *  BEGIN TEST 343
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_343( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_ROWMAJ, y320_VW );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	const oski_value_t y_true[] = {-30, -16, 4, -14, 0, -2, -31, 31, 27, -15, -1, 7, -26, -22, -15 };
	oski_value_t alpha = 1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_343, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_343", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_343() */
/* END TEST 343 */

/*
 *  BEGIN TEST 344
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_344( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	const oski_value_t y_true[] = {-10, -12, 15, 66, 18, 32, 22, 0, -2, 14, 0, -24 };
	oski_value_t alpha = 1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_344, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_344", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_344() */
/* END TEST 344 */

/*
 *  BEGIN TEST 345
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_345( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	const oski_value_t y_true[] = {10, 40, -31, -38, -46, -72, -42, 40, 6, -34, -12, 32 };
	oski_value_t alpha = -1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_345, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_345", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_345() */
/* END TEST 345 */

/*
 *  BEGIN TEST 346
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_346( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	const oski_value_t y_true[] = {0, -7, 4, -7, 7, 10, 5, -10, -1, 5, 3, -2 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_346, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_346", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_346() */
/* END TEST 346 */

/*
 *  BEGIN TEST 347
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_347( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	oski_value_t* y_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	const oski_value_t y_true[] = {-80, 44, 40, 668, 4, 56, 76, 200, 4, 12, -60, -152 };
	oski_value_t alpha = 8;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_347, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_347", y_true, y_val, y320_LEN*y320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_347() */
/* END TEST 347 */

/*
 *  BEGIN TEST 348
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_348( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	const oski_value_t y_true[] = {-80, 149, 20, -17, 0, -31, 106, 51, 90, -40, 9, 7, -75, -62, -40 };
	oski_value_t alpha = 1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_348, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_348", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_348() */
/* END TEST 348 */

/*
 *  BEGIN TEST 349
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_349( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	const oski_value_t y_true[] = {70, -140, -17, 14, 0, 28, -104, -44, -80, 35, -8, -8, 66, 54, 35 };
	oski_value_t alpha = -1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_349, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_349", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_349() */
/* END TEST 349 */

/*
 *  BEGIN TEST 350
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_350( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	const oski_value_t y_true[] = {20, -18, -6, 6, 0, 6, -4, -14, -20, 10, -2, 2, 18, 16, 10 };
	oski_value_t alpha = 0;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_350, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_350", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_350() */
/* END TEST 350 */

/*
 *  BEGIN TEST 351
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x320', LAYOUT_COLMAJ>
 *  y : <'y320', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_351( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A320_ptr, A320_ind, A320_val,
		A320_M, A320_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y320_val,
		y320_LEN * y320_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y320_LEN, y320_VW, LAYOUT_COLMAJ, y320_LEN );
	oski_value_t* y_val = array_Duplicate( x320_val,
		x320_LEN * x320_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x320_LEN, x320_VW, LAYOUT_COLMAJ, x320_LEN );
	const oski_value_t y_true[] = {-80, 534, 0, 18, 0, -66, 556, 26, 140, -40, 14, 82, -90, -52, -40 };
	oski_value_t alpha = 6;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_351, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_351", y_true, y_val, x320_LEN*x320_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_351() */
/* END TEST 351 */

#define A352_M 4    /* # of rows */
#define A352_N 5    /* # of cols */
#define A352_NNZ 3  /* # of stored non-zeros */
static oski_index_t A352_ptr[A352_M+1] = {0, 0, 0, 2, 3}; /* end row pointers for A352 */
static oski_index_t A352_ind[A352_NNZ] = {0, 1, 0}; /* end column indices pointers for A352 */
static oski_value_t A352_val[A352_NNZ] = {2, -10, 10}; /* end stored non-zero values for A352 */

#define x352_LEN 5   /* vector length */
#define x352_VW  3   /* vector width */
static oski_value_t x352_val[x352_LEN*x352_VW] = {-10, 6, 5, 1, 3, -9, -2, 10, -1, 5, 3, -9, 7, -3, 0}; /* data */

#define y352_LEN 4   /* vector length */
#define y352_VW  3   /* vector width */
static oski_value_t y352_val[y352_LEN*y352_VW] = {-7, -9, 8, 10, 2, -10, 4, -1, 1, 8, -2, 4}; /* data */

/*
 *  BEGIN TEST 352
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_352( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	const oski_value_t y_true[] = {25, 51, -35, -49, -7, 41, -52, -3, 94, -135, 73, 21 };
	oski_value_t alpha = 1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_352, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_352", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_352() */
/* END TEST 352 */

/*
 *  BEGIN TEST 353
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_353( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	const oski_value_t y_true[] = {10, -6, -5, -1, -3, 9, 32, 8, -99, 95, -63, -41 };
	oski_value_t alpha = -1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_353, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_353", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_353() */
/* END TEST 353 */

/*
 *  BEGIN TEST 354
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_354( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	const oski_value_t y_true[] = {28, 36, -32, -40, -8, 40, -16, 4, -4, -32, 8, -16 };
	oski_value_t alpha = 0;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_354, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_354", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_354() */
/* END TEST 354 */

/*
 *  BEGIN TEST 355
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_355( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	const oski_value_t y_true[] = {-45, 93, 0, -42, 14, -22, -276, -59, 787, -800, 514, 308 };
	oski_value_t alpha = 8;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_355, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_355", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_355() */
/* END TEST 355 */

/*
 *  BEGIN TEST 356
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_356( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	const oski_value_t y_true[] = {1, 17, 90, -22, 36, -92, -12, 79, -7, 48, 22, -68, 56, -24, 0 };
	oski_value_t alpha = 1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_356, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_356", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_356() */
/* END TEST 356 */

/*
 *  BEGIN TEST 357
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_357( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	const oski_value_t y_true[] = {-81, 31, -50, 30, -12, 20, -4, 1, -1, -8, 2, -4, 0, 0, 0 };
	oski_value_t alpha = -1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_357, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_357", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_357() */
/* END TEST 357 */

/*
 *  BEGIN TEST 358
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_358( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	const oski_value_t y_true[] = {70, -42, -35, -7, -21, 63, 14, -70, 7, -35, -21, 63, -49, 21, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_358, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_358", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_358() */
/* END TEST 358 */

/*
 *  BEGIN TEST 359
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_359( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	const oski_value_t y_true[] = {588, -212, 430, -234, 114, -214, 20, 52, 2, 94, 2, -22, 42, -18, 0 };
	oski_value_t alpha = 8;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_359, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_359", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_359() */
/* END TEST 359 */

/*
 *  BEGIN TEST 360
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_360( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	const oski_value_t y_true[] = {25, 46, -72, -145, -4, 53, -28, 68, 0, -49, 109, 21 };
	oski_value_t alpha = 1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_360, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_360", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_360() */
/* END TEST 360 */

/*
 *  BEGIN TEST 361
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_361( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	const oski_value_t y_true[] = {73, 80, -40, 5, -24, 87, -28, -54, -14, -63, -81, -77 };
	oski_value_t alpha = -1;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_361, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_361", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_361() */
/* END TEST 361 */

/*
 *  BEGIN TEST 362
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_362( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	const oski_value_t y_true[] = {-35, -45, 40, 50, 10, -50, 20, -5, 5, 40, -10, 20 };
	oski_value_t alpha = 0;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_362, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_362", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_362() */
/* END TEST 362 */

/*
 *  BEGIN TEST 363
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_363( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	const oski_value_t y_true[] = {24, -80, 320, 840, -32, -104, 96, -512, -32, 136, -808, -296 };
	oski_value_t alpha = -8;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_363, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_363", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_363() */
/* END TEST 363 */

/*
 *  BEGIN TEST 364
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_364( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	const oski_value_t y_true[] = {79, 18, 52, -86, -41, 1, 2, 34, -5, 25, 8, -23, 21, -9, 0 };
	oski_value_t alpha = 1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_364, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_364", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_364() */
/* END TEST 364 */

/*
 *  BEGIN TEST 365
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_365( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	const oski_value_t y_true[] = {-119, 6, -32, 90, 53, -37, -10, 6, 1, -5, 4, -13, 7, -3, 0 };
	oski_value_t alpha = -1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_365, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_365", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_365() */
/* END TEST 365 */

/*
 *  BEGIN TEST 366
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_366( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	const oski_value_t y_true[] = {70, -42, -35, -7, -21, 63, 14, -70, 7, -35, -21, 63, -49, 21, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_366, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_366", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_366() */
/* END TEST 366 */

/*
 *  BEGIN TEST 367
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_ROWMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_367( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_ROWMAJ, x352_VW );
	const oski_value_t y_true[] = {-823, 36, -229, 629, 368, -250, -68, 32, 8, -40, 25, -82, 42, -18, 0 };
	oski_value_t alpha = -7;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_367, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_367", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_367() */
/* END TEST 367 */

/*
 *  BEGIN TEST 368
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_368( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	const oski_value_t y_true[] = {-10, -9, 3, 6, -2, -9, -75, 12, 103, -99, -91, 27 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_368, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_368", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_368() */
/* END TEST 368 */

/*
 *  BEGIN TEST 369
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_369( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	const oski_value_t y_true[] = {-11, -18, 21, 24, 8, -21, 87, -15, -100, 123, 85, -15 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_369, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_369", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_369() */
/* END TEST 369 */

/*
 *  BEGIN TEST 370
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_370( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	const oski_value_t y_true[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_370, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_370", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_370() */
/* END TEST 370 */

/*
 *  BEGIN TEST 371
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_371( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	const oski_value_t y_true[] = {49, 63, -56, -70, -14, 70, -28, 7, -7, -56, 14, -28 };
	oski_value_t alpha = 0;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_371, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_371", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_371() */
/* END TEST 371 */

/*
 *  BEGIN TEST 372
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_372( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	const oski_value_t y_true[] = {11, 12, 39, 15, 21, -94, -2, 69, -9, 35, 71, -83, 50, -17, 0 };
	oski_value_t alpha = 1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_372, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_372", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_372() */
/* END TEST 372 */

/*
 *  BEGIN TEST 373
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_373( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	const oski_value_t y_true[] = {-111, 48, 11, -5, 9, 4, -18, 31, -1, 15, -41, -7, 20, -13, 0 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_373, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_373", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_373() */
/* END TEST 373 */

/*
 *  BEGIN TEST 374
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_374( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	const oski_value_t y_true[] = {40, -24, -20, -4, -12, 36, 8, -40, 4, -20, -12, 36, -28, 12, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_374, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_374", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_374() */
/* END TEST 374 */

/*
 *  BEGIN TEST 375
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_375( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_ROWMAJ, y352_VW );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	const oski_value_t y_true[] = {-142, 48, -18, -18, -6, 80, -20, -18, 6, -10, -106, 58, -16, -2, 0 };
	oski_value_t alpha = -2;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_375, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_375", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_375() */
/* END TEST 375 */

/*
 *  BEGIN TEST 376
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_376( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	const oski_value_t y_true[] = {-52, -48, -27, -39, 3, -62, 36, -97, 9, 39, 91, 51 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_376, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_376", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_376() */
/* END TEST 376 */

/*
 *  BEGIN TEST 377
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_377( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	const oski_value_t y_true[] = {45, 39, 35, 49, -1, 52, -32, 96, -8, -31, -93, -47 };
	oski_value_t alpha = -1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_377, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_377", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_377() */
/* END TEST 377 */

/*
 *  BEGIN TEST 378
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_378( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	const oski_value_t y_true[] = {49, 63, -56, -70, -14, 70, -28, 7, -7, -56, 14, -28 };
	oski_value_t alpha = 0;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_378, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_378", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_378() */
/* END TEST 378 */

/*
 *  BEGIN TEST 379
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_379( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	oski_value_t* y_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	const oski_value_t y_true[] = {-61, -3, -276, -366, -30, -38, 60, -367, 15, -12, 406, 120 };
	oski_value_t alpha = 4;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_379, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_379", y_true, y_val, y352_LEN*y352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_379() */
/* END TEST 379 */

/*
 *  BEGIN TEST 380
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_380( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	const oski_value_t y_true[] = {159, -119, -17, 5, -15, 45, -40, -46, 4, -25, 22, 73, -37, 19, 0 };
	oski_value_t alpha = 1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_380, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_380", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_380() */
/* END TEST 380 */

/*
 *  BEGIN TEST 381
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_381( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	const oski_value_t y_true[] = {-169, 125, 22, -4, 18, -54, 38, 56, -5, 30, -19, -82, 44, -22, 0 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_381, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_381", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_381() */
/* END TEST 381 */

/*
 *  BEGIN TEST 382
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_382( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	const oski_value_t y_true[] = {90, -54, -45, -9, -27, 81, 18, -90, 9, -45, -27, 81, -63, 27, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_382, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_382", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_382() */
/* END TEST 382 */

/*
 *  BEGIN TEST 383
 *  Value type: real
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x352', LAYOUT_COLMAJ>
 *  y : <'y352', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_383( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A352_ptr, A352_ind, A352_val,
		A352_M, A352_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y352_val,
		y352_LEN * y352_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y352_LEN, y352_VW, LAYOUT_COLMAJ, y352_LEN );
	oski_value_t* y_val = array_Duplicate( x352_val,
		x352_LEN * x352_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x352_LEN, x352_VW, LAYOUT_COLMAJ, x352_LEN );
	const oski_value_t y_true[] = {-703, 587, -86, -76, -18, 54, 362, -88, 13, -30, -277, -142, -28, -10, 0 };
	oski_value_t alpha = -7;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_383, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_383", y_true, y_val, x352_LEN*x352_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_383() */
/* END TEST 383 */

#define A384_M 5    /* # of rows */
#define A384_N 5    /* # of cols */
#define A384_NNZ 12  /* # of stored non-zeros */
static oski_index_t A384_ptr[A384_M+1] = {1, 2, 5, 7, 11, 13}; /* end row pointers for A384 */
static oski_index_t A384_ind[A384_NNZ] = {4, 3, 4, 5, 2, 4, 1, 2, 3, 5, 2, 4}; /* end column indices pointers for A384 */
static oski_value_t A384_val[A384_NNZ] = {-7, -6, -10, -1, -6, 10, -7, -10, 10, 4, -1, 4}; /* end stored non-zero values for A384 */

#define x384_LEN 5   /* vector length */
#define x384_VW  3   /* vector width */
static oski_value_t x384_val[x384_LEN*x384_VW] = {-8, -7, -1, 7, -6, -1, -10, 10, -8, -5, 0, 7, -5, -6, -4}; /* data */

#define y384_LEN 5   /* vector length */
#define y384_VW  3   /* vector width */
static oski_value_t y384_val[y384_LEN*y384_VW] = {-2, -6, 10, 6, 3, 2, -3, 10, -6, -8, 7, 5, -10, 6, -8}; /* data */

/*
 *  BEGIN TEST 384
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_384( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {21, -42, 21, 157, -33, -4, -113, 106, 34, -190, 234, -44, -97, 48, -27 };
	oski_value_t alpha = 1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_384, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_384", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_384() */
/* END TEST 384 */

/*
 *  BEGIN TEST 385
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_385( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {-41, -18, 79, -97, 63, 24, 83, -6, -94, 110, -164, 94, -3, 12, -53 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_385, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_385", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_385() */
/* END TEST 385 */

/*
 *  BEGIN TEST 386
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_386( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {6, 18, -30, -18, -9, -6, 9, -30, 18, 24, -21, -15, 30, -18, 24 };
	oski_value_t alpha = 0;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_386, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_386", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_386() */
/* END TEST 386 */

/*
 *  BEGIN TEST 387
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_387( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {-60, 30, 48, -260, 93, 26, 199, -122, -122, 308, -405, 133, 104, -42, -18 };
	oski_value_t alpha = -2;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_387, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_387", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_387() */
/* END TEST 387 */

/*
 *  BEGIN TEST 388
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_388( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {80, -28, -32, 87, -118, -3, -86, 22, 62, -101, 136, -203, -23, 43, 30 };
	oski_value_t alpha = 1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_388, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_388", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_388() */
/* END TEST 388 */

/*
 *  BEGIN TEST 389
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_389( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {16, 112, 44, -171, 190, 15, 206, -142, 34, 161, -136, 119, 83, 29, 18 };
	oski_value_t alpha = -1;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_389, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_389", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_389() */
/* END TEST 389 */

/*
 *  BEGIN TEST 390
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_390( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {-56, -49, -7, 49, -42, -7, -70, 70, -56, -35, 0, 49, -35, -42, -28 };
	oski_value_t alpha = 0;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_390, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_390", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_390() */
/* END TEST 390 */

/*
 *  BEGIN TEST 391
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_391( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {296, -231, -173, 526, -668, -28, -560, 240, 206, -570, 680, -924, -180, 137, 98 };
	oski_value_t alpha = 5;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_391, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_391", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_391() */
/* END TEST 391 */

/*
 *  BEGIN TEST 392
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_392( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {27, 91, -52, -110, -15, 8, -66, 76, 161, -26, -21, 2, 36, -55, -3 };
	oski_value_t alpha = 1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_392, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_392", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_392() */
/* END TEST 392 */

/*
 *  BEGIN TEST 393
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_393( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {-47, -151, 152, 170, 45, 12, 36, 24, -221, -54, 91, 48, -136, 115, -77 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_393, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_393", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_393() */
/* END TEST 393 */

/*
 *  BEGIN TEST 394
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_394( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {-14, -42, 70, 42, 21, 14, -21, 70, -42, -56, 49, 35, -70, 42, -56 };
	oski_value_t alpha = 0;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_394, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_394", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_394() */
/* END TEST 394 */

/*
 *  BEGIN TEST 395
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_395( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {-239, -787, 614, 920, 180, -6, 387, -282, -1277, -18, 322, 111, -502, 535, -179 };
	oski_value_t alpha = -7;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_395, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_395", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_395() */
/* END TEST 395 */

/*
 *  BEGIN TEST 396
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_396( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {-106, -14, -50, -67, -40, 0, 16, 38, -34, 146, 84, -175, -10, -69, -13 };
	oski_value_t alpha = 1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_396, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_396", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_396() */
/* END TEST 396 */

/*
 *  BEGIN TEST 397
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_397( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {58, -28, 44, 109, 4, -6, -76, 22, -14, -176, -84, 217, -20, 33, -11 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_397, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_397", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_397() */
/* END TEST 397 */

/*
 *  BEGIN TEST 398
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_398( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {24, 21, 3, -21, 18, 3, 30, -30, 24, 15, 0, -21, 15, 18, 12 };
	oski_value_t alpha = 0;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_398, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_398", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_398() */
/* END TEST 398 */

/*
 *  BEGIN TEST 399
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_399( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {328, -343, 335, 991, -70, -65, -778, 346, -248, -1493, -672, 1855, -245, 162, -156 };
	oski_value_t alpha = -8;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_399, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_399", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_399() */
/* END TEST 399 */

/*
 *  BEGIN TEST 400
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_400( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {-59, 26, 92, -28, 40, 104, 97, 30, -132, 52, 222, -111, -15, 8, -71 };
	oski_value_t alpha = 1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_400, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_400", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_400() */
/* END TEST 400 */

/*
 *  BEGIN TEST 401
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_401( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {37, -92, 18, 94, -7, -82, -130, 80, 66, -140, -145, 166, -95, 58, -17 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_401, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_401", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_401() */
/* END TEST 401 */

/*
 *  BEGIN TEST 402
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_402( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_402, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_402", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_402() */
/* END TEST 402 */

/*
 *  BEGIN TEST 403
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_403( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {-300, 318, 282, -330, 159, 570, 663, -90, -630, 528, 1143, -801, 180, -114, -210 };
	oski_value_t alpha = 6;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_403, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_403", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_403() */
/* END TEST 403 */

/*
 *  BEGIN TEST 404
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_404( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {64, 115, -115, -123, -32, -48, -126, 42, 144, 30, -35, -13, 43, -176, 22 };
	oski_value_t alpha = 1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_404, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_404", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_404() */
/* END TEST 404 */

/*
 *  BEGIN TEST 405
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_405( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {-32, -87, 119, 95, 56, 52, 166, -82, -112, -10, 35, -15, -23, 200, -6 };
	oski_value_t alpha = -1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_405, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_405", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_405() */
/* END TEST 405 */

/*
 *  BEGIN TEST 406
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_406( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {24, 21, 3, -21, 18, 3, 30, -30, 24, 15, 0, -21, 15, 18, 12 };
	oski_value_t alpha = 0;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_406, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_406", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_406() */
/* END TEST 406 */

/*
 *  BEGIN TEST 407
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_407( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {-288, -488, 456, 520, 104, 188, 464, -128, -608, -140, 140, 80, -192, 680, -104 };
	oski_value_t alpha = -4;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_407, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_407", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_407() */
/* END TEST 407 */

/*
 *  BEGIN TEST 408
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_408( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {-53, -70, 132, 104, 41, 60, 19, 0, 175, -38, 56, 104, -122, -124, -47 };
	oski_value_t alpha = 1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_408, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_408", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_408() */
/* END TEST 408 */

/*
 *  BEGIN TEST 409
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_409( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {41, 34, -72, -68, -23, -48, -37, 60, -211, -10, -14, -74, 62, 160, -1 };
	oski_value_t alpha = -1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_409, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_409", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_409() */
/* END TEST 409 */

/*
 *  BEGIN TEST 410
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_410( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {12, 36, -60, -36, -18, -12, 18, -60, 36, 48, -42, -30, 60, -36, 48 };
	oski_value_t alpha = 0;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_410, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_410", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_410() */
/* END TEST 410 */

/*
 *  BEGIN TEST 411
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_411( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {-237, -266, 520, 436, 163, 272, 137, -140, 959, -78, 182, 450, -470, -704, -123 };
	oski_value_t alpha = 5;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_411, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_411", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_411() */
/* END TEST 411 */

/*
 *  BEGIN TEST 412
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_412( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {-58, -137, 94, 200, 18, 40, -12, -22, 68, -31, -42, 22, 20, -243, 11 };
	oski_value_t alpha = 1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_412, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_412", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_412() */
/* END TEST 412 */

/*
 *  BEGIN TEST 413
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_413( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {-6, 81, -102, -144, -66, -48, -68, 102, -132, -9, 42, 34, -60, 195, -43 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_413, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_413", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_413() */
/* END TEST 413 */

/*
 *  BEGIN TEST 414
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_414( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {24, 21, 3, -21, 18, 3, 30, -30, 24, 15, 0, -21, 15, 18, 12 };
	oski_value_t alpha = 0;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_414, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_414", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_414() */
/* END TEST 414 */

/*
 *  BEGIN TEST 415
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_415( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {-28, -197, 199, 323, 102, 91, 86, -154, 224, -7, -84, -33, 95, -420, 66 };
	oski_value_t alpha = 2;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_415, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_415", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_415() */
/* END TEST 415 */

#define A416_M 5    /* # of rows */
#define A416_N 5    /* # of cols */
#define A416_NNZ 6  /* # of stored non-zeros */
static oski_index_t A416_ptr[A416_M+1] = {0, 2, 3, 4, 5, 6}; /* end row pointers for A416 */
static oski_index_t A416_ind[A416_NNZ] = {1, 3, 0, 4, 0, 2}; /* end column indices pointers for A416 */
static oski_value_t A416_val[A416_NNZ] = {1, 10, 1, -10, 10, -10}; /* end stored non-zero values for A416 */

#define x416_LEN 5   /* vector length */
#define x416_VW  3   /* vector width */
static oski_value_t x416_val[x416_LEN*x416_VW] = {3, -10, 10, 6, -2, -10, -8, 10, 8, 3, 7, -6, 4, 2, -6}; /* data */

#define y416_LEN 5   /* vector length */
#define y416_VW  3   /* vector width */
static oski_value_t y416_val[y416_LEN*y416_VW] = {-5, -10, 0, -1, 10, -10, 1, -2, 8, 7, 3, -3, 3, 4, 4}; /* data */

/*
 *  BEGIN TEST 416
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_416( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	const oski_value_t y_true[] = {56, 108, -70, 7, -50, 50, -44, -12, 28, 2, -112, 112, 68, -116, -96 };
	oski_value_t alpha = 1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_416, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_416", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_416() */
/* END TEST 416 */

/*
 *  BEGIN TEST 417
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_417( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	const oski_value_t y_true[] = {9, 22, 70, 6, -80, 80, 31, 38, -132, -93, 73, -73, -107, 64, 44 };
	oski_value_t alpha = -1;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_417, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_417", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_417() */
/* END TEST 417 */

/*
 *  BEGIN TEST 418
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_418( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	const oski_value_t y_true[] = {-45, -90, 0, -9, 90, -90, 9, -18, 72, 63, 27, -27, 27, 36, 36 };
	oski_value_t alpha = 0;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_418, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_418", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_418() */
/* END TEST 418 */

/*
 *  BEGIN TEST 419
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_419( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	const oski_value_t y_true[] = {-195, -370, 350, -18, 80, -80, 203, 94, -276, -129, 509, -509, -391, 512, 412 };
	oski_value_t alpha = -5;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_419, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_419", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_419() */
/* END TEST 419 */

/*
 *  BEGIN TEST 420
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_420( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	const oski_value_t y_true[] = {72, 30, -30, 1, -12, -10, -38, -30, -32, -47, -93, -6, -6, 22, -86 };
	oski_value_t alpha = 1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_420, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_420", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_420() */
/* END TEST 420 */

/*
 *  BEGIN TEST 421
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_421( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	const oski_value_t y_true[] = {-75, -20, 20, -7, 14, 20, 46, 20, 24, 44, 86, 12, 2, -24, 92 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_421, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_421", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_421() */
/* END TEST 421 */

/*
 *  BEGIN TEST 422
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_422( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	const oski_value_t y_true[] = {-21, 70, -70, -42, 14, 70, 56, -70, -56, -21, -49, 42, -28, -14, 42 };
	oski_value_t alpha = 0;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_422, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_422", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_422() */
/* END TEST 422 */

/*
 *  BEGIN TEST 423
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_423( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	const oski_value_t y_true[] = {-90, 30, -30, -37, 24, 70, 86, -30, -16, 29, 51, 42, -18, -34, 122 };
	oski_value_t alpha = -1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_423, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_423", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_423() */
/* END TEST 423 */

/*
 *  BEGIN TEST 424
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_424( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	const oski_value_t y_true[] = {26, -17, -40, 28, 100, 48, -8, -24, -84, -86, -64, 4, 66, 108, -72 };
	oski_value_t alpha = 1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_424, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_424", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_424() */
/* END TEST 424 */

/*
 *  BEGIN TEST 425
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_425( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	const oski_value_t y_true[] = {-41, -13, 40, -31, -70, -78, 11, 18, 108, 107, 73, -13, -57, -96, 84 };
	oski_value_t alpha = -1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_425, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_425", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_425() */
/* END TEST 425 */

/*
 *  BEGIN TEST 426
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_426( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	const oski_value_t y_true[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_426, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_426", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_426() */
/* END TEST 426 */

/*
 *  BEGIN TEST 427
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_427( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	const oski_value_t y_true[] = {292, 101, -280, 218, 480, 556, -78, -124, -764, -756, -514, 94, 396, 668, -592 };
	oski_value_t alpha = 7;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_427, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_427", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_427() */
/* END TEST 427 */

/*
 *  BEGIN TEST 428
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_428( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	const oski_value_t y_true[] = {-26, 101, 17, -17, -6, 23, -84, -90, -56, -56, -114, 42, -8, 16, -18 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_428, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_428", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_428() */
/* END TEST 428 */

/*
 *  BEGIN TEST 429
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_429( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	const oski_value_t y_true[] = {-1, -11, -107, -37, 24, 67, 156, 0, -16, 29, 51, 12, -28, -34, 72 };
	oski_value_t alpha = -1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_429, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_429", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_429() */
/* END TEST 429 */

/*
 *  BEGIN TEST 430
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_430( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	const oski_value_t y_true[] = {-12, 40, -40, -24, 8, 40, 32, -40, -32, -12, -28, 24, -16, -8, 24 };
	oski_value_t alpha = 0;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_430, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_430", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_430() */
/* END TEST 430 */

/*
 *  BEGIN TEST 431
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_ROWMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_431( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_ROWMAJ, x416_VW );
	const oski_value_t y_true[] = {-99, 416, 292, 12, -74, -52, -656, -350, -184, -279, -551, 138, 28, 134, -222 };
	oski_value_t alpha = 6;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_431, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_431", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_431() */
/* END TEST 431 */

/*
 *  BEGIN TEST 432
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_432( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	const oski_value_t y_true[] = {20, 12, 14, -3, 50, -53, 26, -42, 108, 72, -82, 52, -82, -76, -16 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_432, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_432", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_432() */
/* END TEST 432 */

/*
 *  BEGIN TEST 433
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_433( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	const oski_value_t y_true[] = {-90, -152, -14, -11, 90, -87, -12, 14, 4, 26, 124, -94, 124, 132, 72 };
	oski_value_t alpha = -1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_433, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_433", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_433() */
/* END TEST 433 */

/*
 *  BEGIN TEST 434
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_434( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	const oski_value_t y_true[] = {-30, -60, 0, -6, 60, -60, 6, -12, 48, 42, 18, -18, 18, 24, 24 };
	oski_value_t alpha = 0;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_434, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_434", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_434() */
/* END TEST 434 */

/*
 *  BEGIN TEST 435
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_435( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	const oski_value_t y_true[] = {220, 300, 70, 9, 10, -25, 106, -162, 348, 192, -482, 332, -482, -476, -176 };
	oski_value_t alpha = 5;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_435, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_435", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_435() */
/* END TEST 435 */

/*
 *  BEGIN TEST 436
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_436( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	const oski_value_t y_true[] = {66, 5, -40, -56, -8, 50, -2, -50, -108, 17, -47, 6, -44, -2, -74 };
	oski_value_t alpha = 1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_436, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_436", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_436() */
/* END TEST 436 */

/*
 *  BEGIN TEST 437
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_437( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	const oski_value_t y_true[] = {-60, -25, 60, 68, 4, -70, -14, 70, 124, -11, 61, -18, 52, 6, 62 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_437, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_437", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_437() */
/* END TEST 437 */

/*
 *  BEGIN TEST 438
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_438( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	const oski_value_t y_true[] = {9, -30, 30, 18, -6, -30, -24, 30, 24, 9, 21, -18, 12, 6, -18 };
	oski_value_t alpha = 0;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_438, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_438", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_438() */
/* END TEST 438 */

/*
 *  BEGIN TEST 439
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_439( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_ROWMAJ, y416_VW );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	const oski_value_t y_true[] = {411, -20, -190, -306, -58, 250, -52, -250, -608, 117, -247, 6, -244, -2, -474 };
	oski_value_t alpha = 6;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_439, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_439", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_439() */
/* END TEST 439 */

/*
 *  BEGIN TEST 440
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_440( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	const oski_value_t y_true[] = {80, 63, 20, 36, -160, 132, -16, -18, -148, -142, -4, 25, 42, 46, -64 };
	oski_value_t alpha = 1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_440, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_440", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_440() */
/* END TEST 440 */

/*
 *  BEGIN TEST 441
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_441( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	const oski_value_t y_true[] = {-40, 17, -20, -28, 80, -52, 8, 34, 84, 86, -20, -1, -66, -78, 32 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_441, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_441", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_441() */
/* END TEST 441 */

/*
 *  BEGIN TEST 442
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_442( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	const oski_value_t y_true[] = {-40, -80, 0, -8, 80, -80, 8, -16, 64, 56, 24, -24, 24, 32, 32 };
	oski_value_t alpha = 0;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_442, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_442", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_442() */
/* END TEST 442 */

/*
 *  BEGIN TEST 443
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_443( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	oski_value_t* y_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	const oski_value_t y_true[] = {-40, 17, -20, -28, 80, -52, 8, 34, 84, 86, -20, -1, -66, -78, 32 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_443, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_443", y_true, y_val, y416_LEN*y416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_443() */
/* END TEST 443 */

/*
 *  BEGIN TEST 444
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_444( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	const oski_value_t y_true[] = {-20, -5, -100, -50, 0, 81, -10, -70, -100, 20, 37, 3, -40, 30, -30 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_444, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_444", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_444() */
/* END TEST 444 */

/*
 *  BEGIN TEST 445
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_445( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	const oski_value_t y_true[] = {5, 55, 50, 20, 10, -31, 50, 20, 60, -35, -72, 27, 20, -40, 60 };
	oski_value_t alpha = -1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_445, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_445", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_445() */
/* END TEST 445 */

/*
 *  BEGIN TEST 446
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_446( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	const oski_value_t y_true[] = {21, -70, 70, 42, -14, -70, -56, 70, 56, 21, 49, -42, 28, 14, -42 };
	oski_value_t alpha = 0;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_446, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_446", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_446() */
/* END TEST 446 */

/*
 *  BEGIN TEST 447
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x416', LAYOUT_COLMAJ>
 *  y : <'y416', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_447( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A416_ptr, A416_ind, A416_val,
		A416_M, A416_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y416_val,
		y416_LEN * y416_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y416_LEN, y416_VW, LAYOUT_COLMAJ, y416_LEN );
	oski_value_t* y_val = array_Duplicate( x416_val,
		x416_LEN * x416_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x416_LEN, x416_VW, LAYOUT_COLMAJ, x416_LEN );
	const oski_value_t y_true[] = {-201, 25, -970, -492, 14, 799, -34, -700, -956, 159, 284, 69, -388, 256, -228 };
	oski_value_t alpha = 9;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_447, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_447", y_true, y_val, x416_LEN*x416_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_447() */
/* END TEST 447 */

#define A448_M 5    /* # of rows */
#define A448_N 5    /* # of cols */
#define A448_NNZ 8  /* # of stored non-zeros */
static oski_index_t A448_ptr[A448_M+1] = {1, 2, 3, 4, 8, 9}; /* end row pointers for A448 */
static oski_index_t A448_ind[A448_NNZ] = {4, 4, 4, 1, 2, 3, 5, 4}; /* end column indices pointers for A448 */
static oski_value_t A448_val[A448_NNZ] = {-5, -10, 8, -5, -10, 8, -8, -8}; /* end stored non-zero values for A448 */

#define x448_LEN 5   /* vector length */
#define x448_VW  3   /* vector width */
static oski_value_t x448_val[x448_LEN*x448_VW] = {-3, 7, -10, -2, 5, -5, 9, 10, 10, 5, -4, 6, -7, 2, 0}; /* data */

#define y448_LEN 5   /* vector length */
#define y448_VW  3   /* vector width */
static oski_value_t y448_val[y448_LEN*y448_VW] = {-7, -8, 1, -1, 5, 1, 10, 6, 0, -9, -10, -1, -7, 1, -5}; /* data */

/*
 *  BEGIN TEST 448
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_448( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	const oski_value_t y_true[] = {-49, 3, -37, -55, 60, -62, 79, -4, 58, 141, -55, 183, -68, 37, -63 };
	oski_value_t alpha = 1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_448, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_448", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_448() */
/* END TEST 448 */

/*
 *  BEGIN TEST 449
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_449( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	const oski_value_t y_true[] = {84, 37, 32, 60, -85, 57, -129, -26, -58, -96, 105, -178, 103, -42, 88 };
	oski_value_t alpha = -1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_449, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_449", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_449() */
/* END TEST 449 */

/*
 *  BEGIN TEST 450
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_450( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	const oski_value_t y_true[] = {-42, -48, 6, -6, 30, 6, 60, 36, 0, -54, -60, -6, -42, 6, -30 };
	oski_value_t alpha = 0;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_450, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_450", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_450() */
/* END TEST 450 */

/*
 *  BEGIN TEST 451
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_451( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	const oski_value_t y_true[] = {-175, 213, -283, -361, 300, -458, 313, -172, 406, 1203, -145, 1305, -308, 235, -321 };
	oski_value_t alpha = 7;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_451, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_451", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_451() */
/* END TEST 451 */

/*
 *  BEGIN TEST 452
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_452( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	const oski_value_t y_true[] = {56, 0, 66, 101, 75, 41, -116, -134, -68, 142, 44, -12, 107, 69, 3 };
	oski_value_t alpha = 1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_452, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_452", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_452() */
/* END TEST 452 */

/*
 *  BEGIN TEST 453
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_453( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	const oski_value_t y_true[] = {-50, -14, -46, -97, -85, -31, 98, 114, 48, -152, -36, 0, -93, -73, -3 };
	oski_value_t alpha = -1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_453, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_453", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_453() */
/* END TEST 453 */

/*
 *  BEGIN TEST 454
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_454( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	const oski_value_t y_true[] = {-9, 21, -30, -6, 15, -15, 27, 30, 30, 15, -12, 18, -21, 6, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_454, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_454", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_454() */
/* END TEST 454 */

/*
 *  BEGIN TEST 455
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_455( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	const oski_value_t y_true[] = {-82, -70, -32, -182, -200, -32, 142, 168, 36, -334, -48, -36, -144, -158, -6 };
	oski_value_t alpha = -2;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_455, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_455", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_455() */
/* END TEST 455 */

/*
 *  BEGIN TEST 456
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_456( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	const oski_value_t y_true[] = {-35, -60, 50, 167, -42, 28, 55, -16, -25, 25, -50, -66, 51, 187, -53 };
	oski_value_t alpha = 1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_456, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_456", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_456() */
/* END TEST 456 */

/*
 *  BEGIN TEST 457
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_457( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	const oski_value_t y_true[] = {28, 52, -49, -168, 47, -27, -45, 22, 25, -34, 40, 65, -58, -186, 48 };
	oski_value_t alpha = -1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_457, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_457", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_457() */
/* END TEST 457 */

/*
 *  BEGIN TEST 458
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_458( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	const oski_value_t y_true[] = {-14, -16, 2, -2, 10, 2, 20, 12, 0, -18, -20, -2, -14, 2, -10 };
	oski_value_t alpha = 0;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_458, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_458", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_458() */
/* END TEST 458 */

/*
 *  BEGIN TEST 459
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_459( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	const oski_value_t y_true[] = {-238, -432, 394, 1342, -366, 218, 380, -164, -200, 254, -340, -522, 450, 1490, -394 };
	oski_value_t alpha = 8;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_459, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_459", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_459() */
/* END TEST 459 */

/*
 *  BEGIN TEST 460
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_460( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	const oski_value_t y_true[] = {10, -27, 25, 10, -10, 9, -43, -34, -39, 62, 31, 21, 41, -17, -13 };
	oski_value_t alpha = 1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_460, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_460", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_460() */
/* END TEST 460 */

/*
 *  BEGIN TEST 461
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_461( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	const oski_value_t y_true[] = {26, -57, 95, 14, -50, 51, -65, -86, -81, -122, 17, -93, 43, -7, 13 };
	oski_value_t alpha = -1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_461, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_461", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_461() */
/* END TEST 461 */

/*
 *  BEGIN TEST 462
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_462( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	const oski_value_t y_true[] = {-6, 14, -20, -4, 10, -10, 18, 20, 20, 10, -8, 12, -14, 4, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_462, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_462", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_462() */
/* END TEST 462 */

/*
 *  BEGIN TEST 463
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_ROWMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_463( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_ROWMAJ, x448_VW );
	const oski_value_t y_true[] = {33, -66, 135, 12, -75, 78, -60, -108, -93, -291, -9, -189, 24, 9, 39 };
	oski_value_t alpha = -3;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_463, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_463", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_463() */
/* END TEST 463 */

/*
 *  BEGIN TEST 464
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_464( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	const oski_value_t y_true[] = {49, -7, -20, 33, -121, -20, -86, 54, 9, -123, 45, -88, 63, -81, 14 };
	oski_value_t alpha = 1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_464, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_464", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_464() */
/* END TEST 464 */

/*
 *  BEGIN TEST 465
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_465( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	const oski_value_t y_true[] = {-21, 39, 16, -29, 101, 16, 46, -78, -9, 159, -5, 92, -35, 77, 6 };
	oski_value_t alpha = -1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_465, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_465", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_465() */
/* END TEST 465 */

/*
 *  BEGIN TEST 466
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_466( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	const oski_value_t y_true[] = {7, 8, -1, 1, -5, -1, -10, -6, 0, 9, 10, 1, 7, -1, 5 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_466, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_466", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_466() */
/* END TEST 466 */

/*
 *  BEGIN TEST 467
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_467( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	const oski_value_t y_true[] = {-21, -87, -10, 23, -71, -10, 14, 114, 9, -213, -55, -98, -7, -71, -36 };
	oski_value_t alpha = 1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_467, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_467", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_467() */
/* END TEST 467 */

/*
 *  BEGIN TEST 468
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_468( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	const oski_value_t y_true[] = {44, 75, -42, 176, 55, 52, 87, -94, 0, 71, 14, -1, 6, 20, 3 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_468, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_468", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_468() */
/* END TEST 468 */

/*
 *  BEGIN TEST 469
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_469( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	const oski_value_t y_true[] = {-32, -103, 82, -168, -75, -32, -123, 54, -40, -91, 2, -23, 22, -28, -3 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_469, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_469", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_469() */
/* END TEST 469 */

/*
 *  BEGIN TEST 470
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_470( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	const oski_value_t y_true[] = {6, -14, 20, 4, -10, 10, -18, -20, -20, -10, 8, -12, 14, -4, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_470, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_470", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_470() */
/* END TEST 470 */

/*
 *  BEGIN TEST 471
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_471( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_ROWMAJ, y448_VW );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	const oski_value_t y_true[] = {-269, -616, 424, -1206, -450, -299, -726, 528, -130, -562, -46, -71, 49, -166, -21 };
	oski_value_t alpha = -7;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_471, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_471", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_471() */
/* END TEST 471 */

/*
 *  BEGIN TEST 472
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_472( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	const oski_value_t y_true[] = {14, 35, -27, -176, 16, -56, -101, 84, -15, -66, -4, -13, 16, -95, -11 };
	oski_value_t alpha = 1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_472, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_472", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_472() */
/* END TEST 472 */

/*
 *  BEGIN TEST 473
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_473( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	const oski_value_t y_true[] = {42, 29, 19, 184, -56, 48, 21, -132, 15, 138, 84, 21, 40, 87, 51 };
	oski_value_t alpha = -1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_473, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_473", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_473() */
/* END TEST 473 */

/*
 *  BEGIN TEST 474
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_474( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	const oski_value_t y_true[] = {21, 24, -3, 3, -15, -3, -30, -18, 0, 27, 30, 3, 21, -3, 15 };
	oski_value_t alpha = 0;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_474, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_474", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_474() */
/* END TEST 474 */

/*
 *  BEGIN TEST 475
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_475( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	oski_value_t* y_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	const oski_value_t y_true[] = {14, 73, -77, -532, 68, -164, -263, 276, -45, -234, -52, -43, 20, -281, -53 };
	oski_value_t alpha = 3;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_475, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_475", y_true, y_val, y448_LEN*y448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_475() */
/* END TEST 475 */

/*
 *  BEGIN TEST 476
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_476( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	const oski_value_t y_true[] = {-23, 51, -77, 68, 48, -34, 73, 76, 85, 26, -43, 31, -48, 59, -13 };
	oski_value_t alpha = 1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_476, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_476", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_476() */
/* END TEST 476 */

/*
 *  BEGIN TEST 477
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_477( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	const oski_value_t y_true[] = {-1, 5, -3, -84, -8, -6, -1, 4, -5, 14, 11, 17, -8, -43, 13 };
	oski_value_t alpha = -1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_477, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_477", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_477() */
/* END TEST 477 */

/*
 *  BEGIN TEST 478
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_478( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	const oski_value_t y_true[] = {-27, 63, -90, -18, 45, -45, 81, 90, 90, 45, -36, 54, -63, 18, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_478, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_478", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_478() */
/* END TEST 478 */

/*
 *  BEGIN TEST 479
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x448', LAYOUT_COLMAJ>
 *  y : <'y448', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_479( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A448_ptr, A448_ind, A448_val,
		A448_M, A448_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y448_val,
		y448_LEN * y448_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y448_LEN, y448_VW, LAYOUT_COLMAJ, y448_LEN );
	oski_value_t* y_val = array_Duplicate( x448_val,
		x448_LEN * x448_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x448_LEN, x448_VW, LAYOUT_COLMAJ, x448_LEN );
	const oski_value_t y_true[] = {16, -16, 56, -656, -104, -8, -80, -48, -120, 72, 120, 88, -8, -360, 104 };
	oski_value_t alpha = -8;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_479, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_479", y_true, y_val, x448_LEN*x448_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_479() */
/* END TEST 479 */

#define A480_M 5    /* # of rows */
#define A480_N 5    /* # of cols */
#define A480_NNZ 6  /* # of stored non-zeros */
static oski_index_t A480_ptr[A480_M+1] = {0, 2, 2, 4, 5, 6}; /* end row pointers for A480 */
static oski_index_t A480_ind[A480_NNZ] = {2, 4, 0, 3, 2, 0}; /* end column indices pointers for A480 */
static oski_value_t A480_val[A480_NNZ] = {-8, -10, -8, -5, -5, -10}; /* end stored non-zero values for A480 */

#define x480_LEN 5   /* vector length */
#define x480_VW  3   /* vector width */
static oski_value_t x480_val[x480_LEN*x480_VW] = {9, 10, -3, -8, -3, -4, 8, -10, -5, 5, 6, 3, 4, 3, -7}; /* data */

#define y480_LEN 5   /* vector length */
#define y480_VW  3   /* vector width */
static oski_value_t y480_val[y480_LEN*y480_VW] = {-6, 5, 10, -6, -10, -6, -6, -6, 4, 5, 1, -3, 0, -3, -8}; /* data */

/*
 *  BEGIN TEST 480
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_480( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {-137, 95, 177, -50, -73, -46, -131, -162, 32, 0, 63, 7, -86, -118, -33 };
	oski_value_t alpha = 1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_480, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_480", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_480() */
/* END TEST 480 */

/*
 *  BEGIN TEST 481
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_481( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {101, -65, -117, 14, 13, 10, 95, 126, -8, 30, -57, -25, 86, 100, -15 };
	oski_value_t alpha = -1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_481, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_481", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_481() */
/* END TEST 481 */

/*
 *  BEGIN TEST 482
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_482( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {36, -30, -60, 36, 60, 36, 36, 36, -24, -30, -6, 18, 0, 18, 48 };
	oski_value_t alpha = 0;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_482, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_482", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_482() */
/* END TEST 482 */

/*
 *  BEGIN TEST 483
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_483( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {-333, 220, 401, -72, -89, -60, -315, -408, 44, -65, 176, 60, -258, -315, 5 };
	oski_value_t alpha = 3;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_483, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_483", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_483() */
/* END TEST 483 */

/*
 *  BEGIN TEST 484
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_484( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {105, 153, 37, -62, -31, -34, 73, -121, -96, 70, 73, -2, 88, -32, -157 };
	oski_value_t alpha = 1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_484, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_484", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_484() */
/* END TEST 484 */

/*
 *  BEGIN TEST 485
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_485( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {-15, -53, -67, -18, 1, -6, 7, 21, 46, -20, -13, 32, -48, 62, 87 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_485, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_485", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_485() */
/* END TEST 485 */

/*
 *  BEGIN TEST 486
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_486( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {-45, -50, 15, 40, 15, 20, -40, 50, 25, -25, -30, -15, -20, -15, 35 };
	oski_value_t alpha = 0;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_486, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_486", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_486() */
/* END TEST 486 */

/*
 *  BEGIN TEST 487
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_487( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {-333, -588, -321, 108, 87, 72, -174, 396, 411, -255, -240, 111, -396, 291, 711 };
	oski_value_t alpha = -6;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_487, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_487", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_487() */
/* END TEST 487 */

/*
 *  BEGIN TEST 488
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_488( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {-59, -38, -149, 1, -26, 96, 33, -84, 32, -127, 101, 14, 4, 46, 71 };
	oski_value_t alpha = 1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_488, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_488", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_488() */
/* END TEST 488 */

/*
 *  BEGIN TEST 489
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_489( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {137, -27, 19, 77, 156, -18, 45, 162, -84, 62, -114, 25, -4, -7, 33 };
	oski_value_t alpha = -1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_489, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_489", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_489() */
/* END TEST 489 */

/*
 *  BEGIN TEST 490
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_490( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {12, -10, -20, 12, 20, 12, 12, 12, -8, -10, -2, 6, 0, 6, 16 };
	oski_value_t alpha = 0;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_490, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_490", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_490() */
/* END TEST 490 */

/*
 *  BEGIN TEST 491
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_491( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {-143, 32, -9, -83, -166, 12, -51, -168, 88, -57, 115, -28, 4, 4, -41 };
	oski_value_t alpha = 1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_491, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_491", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_491() */
/* END TEST 491 */

/*
 *  BEGIN TEST 492
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_492( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {-31, -58, 96, 45, 9, 17, 48, 72, 32, -81, 4, -18, 30, 50, 17 };
	oski_value_t alpha = 1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_492, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_492", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_492() */
/* END TEST 492 */

/*
 *  BEGIN TEST 493
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_493( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {-95, -82, -54, 67, 33, 39, -160, 68, 38, 11, -88, -24, -86, -92, 81 };
	oski_value_t alpha = -1;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_493, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_493", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_493() */
/* END TEST 493 */

/*
 *  BEGIN TEST 494
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_494( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {-72, -80, 24, 64, 24, 32, -64, 80, 40, -40, -48, -24, -32, -24, 56 };
	oski_value_t alpha = 0;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_494, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_494", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_494() */
/* END TEST 494 */

/*
 *  BEGIN TEST 495
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_495( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {161, 14, 546, -21, -63, -49, 672, 84, 14, -357, 280, 0, 378, 476, -175 };
	oski_value_t alpha = 7;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_495, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_495", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_495() */
/* END TEST 495 */

/*
 *  BEGIN TEST 496
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_496( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {21, 61, 114, -32, -62, -39, -77, 5, -31, 42, 52, -38, -93, 24, -123 };
	oski_value_t alpha = 1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_496, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_496", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_496() */
/* END TEST 496 */

/*
 *  BEGIN TEST 497
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_497( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {-69, -21, -34, -16, -18, -9, 29, -53, 63, -2, -44, 14, 93, -48, 59 };
	oski_value_t alpha = -1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_497, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_497", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_497() */
/* END TEST 497 */

/*
 *  BEGIN TEST 498
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_498( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {36, -30, -60, 36, 60, 36, 36, 36, -24, -30, -6, 18, 0, 18, 48 };
	oski_value_t alpha = 0;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_498, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_498", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_498() */
/* END TEST 498 */

/*
 *  BEGIN TEST 499
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_499( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {-36, 30, 60, -36, -60, -36, -36, -36, 24, 30, 6, -18, 0, -18, -48 };
	oski_value_t alpha = 0;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_499, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_499", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_499() */
/* END TEST 499 */

/*
 *  BEGIN TEST 500
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_500( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {114, 74, -7, -29, 36, 51, 54, -131, -9, -13, 106, 18, -29, 1, -164 };
	oski_value_t alpha = 1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_500, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_500", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_500() */
/* END TEST 500 */

/*
 *  BEGIN TEST 501
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_501( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {-33, 16, -20, -43, -63, -87, 18, 41, -36, 58, -52, 9, 65, 26, 101 };
	oski_value_t alpha = -1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_501, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_501", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_501() */
/* END TEST 501 */

/*
 *  BEGIN TEST 502
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_502( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {9, 10, -3, -8, -3, -4, 8, -10, -5, 5, 6, 3, 4, 3, -7 };
	oski_value_t alpha = 0;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_502, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_502", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_502() */
/* END TEST 502 */

/*
 *  BEGIN TEST 503
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_503( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {3, 56, -32, -75, -75, -103, 50, 1, -56, 78, -28, 21, 81, 38, 73 };
	oski_value_t alpha = -1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_503, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_503", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_503() */
/* END TEST 503 */

/*
 *  BEGIN TEST 504
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_504( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {81, -5, -65, 25, -63, 44, 26, 65, 33, 30, 41, 12, -59, -8, -43 };
	oski_value_t alpha = 1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_504, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_504", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_504() */
/* END TEST 504 */

/*
 *  BEGIN TEST 505
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_505( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {-27, -40, -25, 29, 153, 10, 28, -11, -69, -75, -50, 15, 59, 35, 115 };
	oski_value_t alpha = -1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_505, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_505", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_505() */
/* END TEST 505 */

/*
 *  BEGIN TEST 506
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_506( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_506, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_506", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_506() */
/* END TEST 506 */

/*
 *  BEGIN TEST 507
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_507( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {-162, 10, 130, -50, 126, -88, -52, -130, -66, -60, -82, -24, 118, 16, 86 };
	oski_value_t alpha = -2;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_507, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_507", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_507() */
/* END TEST 507 */

/*
 *  BEGIN TEST 508
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_508( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {86, 85, 64, -120, 26, -40, 58, -58, -6, 105, 129, 21, 39, 21, -74 };
	oski_value_t alpha = 1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_508, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_508", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_508() */
/* END TEST 508 */

/*
 *  BEGIN TEST 509
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_509( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {-95, -95, -61, 128, -23, 44, -66, 68, 11, -110, -135, -24, -43, -24, 81 };
	oski_value_t alpha = -1;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_509, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_509", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_509() */
/* END TEST 509 */

/*
 *  BEGIN TEST 510
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_510( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {-36, -40, 12, 32, 12, 16, -32, 40, 20, -20, -24, -12, -16, -12, 28 };
	oski_value_t alpha = 0;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_510, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_510", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_510() */
/* END TEST 510 */

/*
 *  BEGIN TEST 511
 *  Value type: real
 *  A : <MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_511( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {49, 65, -109, 0, -71, -20, 62, -92, -69, -30, -39, 24, 21, 24, -31 };
	oski_value_t alpha = -1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_511, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_511", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_511() */
/* END TEST 511 */

#define A512_M 5    /* # of rows */
#define A512_N 5    /* # of cols */
#define A512_NNZ 8  /* # of stored non-zeros */
static oski_index_t A512_ptr[A512_M+1] = {1, 3, 6, 8, 9, 9}; /* end row pointers for A512 */
static oski_index_t A512_ind[A512_NNZ] = {1, 3, 3, 4, 5, 4, 5, 4}; /* end column indices pointers for A512 */
static oski_value_t A512_val[A512_NNZ] = {-3, 1, -8, 9, -8, 10, 2, -10}; /* end stored non-zero values for A512 */

#define x512_LEN 5   /* vector length */
#define x512_VW  3   /* vector width */
static oski_value_t x512_val[x512_LEN*x512_VW] = {10, -8, -4, -4, -3, -7, -4, -3, 3, 2, -10, -4, -9, 1, 0}; /* data */

#define y512_LEN 5   /* vector length */
#define y512_VW  3   /* vector width */
static oski_value_t y512_val[y512_LEN*y512_VW] = {-4, -5, -1, 3, -3, -10, 10, -8, -4, 9, -3, -7, 10, -8, -9}; /* data */

/*
 *  BEGIN TEST 512
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_512( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	const oski_value_t y_true[] = {-66, -19, 7, 146, -98, -140, 124, -146, -20, -24, 19, -49, 104, -46, -10 };
	oski_value_t alpha = 1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_512, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_512", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_512() */
/* END TEST 512 */

/*
 *  BEGIN TEST 513
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_513( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	const oski_value_t y_true[] = {14, -46, -20, -107, 59, 10, 6, 42, -32, 141, -58, -42, 26, -58, -107 };
	oski_value_t alpha = -1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_513, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_513", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_513() */
/* END TEST 513 */

/*
 *  BEGIN TEST 514
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_514( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	const oski_value_t y_true[] = {20, 25, 5, -15, 15, 50, -50, 40, 20, -45, 15, 35, -50, 40, 45 };
	oski_value_t alpha = 0;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_514, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_514", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_514() */
/* END TEST 514 */

/*
 *  BEGIN TEST 515
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_515( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	const oski_value_t y_true[] = {254, -127, -101, -866, 530, 460, -348, 606, -68, 636, -289, -21, -208, -94, -398 };
	oski_value_t alpha = -7;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_515, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_515", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_515() */
/* END TEST 515 */

/*
 *  BEGIN TEST 516
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_516( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	const oski_value_t y_true[] = {12, 15, 3, -75, 104, 48, 86, -24, -12, 35, -67, -56, 5, 7, 72 };
	oski_value_t alpha = 1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_516, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_516", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_516() */
/* END TEST 516 */

/*
 *  BEGIN TEST 517
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_517( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	const oski_value_t y_true[] = {-72, 33, 21, 99, -86, -6, -62, 42, -6, -47, 127, 80, 49, -13, -72 };
	oski_value_t alpha = -1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_517, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_517", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_517() */
/* END TEST 517 */

/*
 *  BEGIN TEST 518
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_518( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	const oski_value_t y_true[] = {-40, 32, 16, 16, 12, 28, 16, 12, -12, -8, 40, 16, 36, -4, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_518, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_518", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_518() */
/* END TEST 518 */

/*
 *  BEGIN TEST 519
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_519( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	const oski_value_t y_true[] = {92, -49, -29, -107, 80, -8, 54, -48, 12, 51, -147, -88, -67, 15, 72 };
	oski_value_t alpha = 1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_519, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_519", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_519() */
/* END TEST 519 */

/*
 *  BEGIN TEST 520
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_520( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	const oski_value_t y_true[] = {-58, 92, 38, -78, 6, -39, -14, -130, 19, 72, -3, -102, 72, -41, 8 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_520, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_520", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_520() */
/* END TEST 520 */

/*
 *  BEGIN TEST 521
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_521( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	const oski_value_t y_true[] = {22, -137, -47, 105, -33, -51, 104, 58, -55, 9, -24, 39, 18, -31, -89 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_521, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_521", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_521() */
/* END TEST 521 */

/*
 *  BEGIN TEST 522
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_522( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	const oski_value_t y_true[] = {-4, -5, -1, 3, -3, -10, 10, -8, -4, 9, -3, -7, 10, -8, -9 };
	oski_value_t alpha = 0;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_522, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_522", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_522() */
/* END TEST 522 */

/*
 *  BEGIN TEST 523
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_523( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	const oski_value_t y_true[] = {-198, 575, 213, -459, 99, 35, -300, -466, 187, 153, 54, -349, 130, -21, 247 };
	oski_value_t alpha = 5;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_523, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_523", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_523() */
/* END TEST 523 */

/*
 *  BEGIN TEST 524
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_524( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	const oski_value_t y_true[] = {71, -26, -5, 35, -62, -122, 36, -130, -27, -73, -10, 93, -16, -90, 76 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_524, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_524", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_524() */
/* END TEST 524 */

/*
 *  BEGIN TEST 525
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_525( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	const oski_value_t y_true[] = {39, -62, -39, -79, 29, 45, -80, 97, 60, 95, -100, -137, -83, 101, -76 };
	oski_value_t alpha = -1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_525, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_525", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_525() */
/* END TEST 525 */

/*
 *  BEGIN TEST 526
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_526( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	const oski_value_t y_true[] = {-50, 40, 20, 20, 15, 35, 20, 15, -15, -10, 50, 20, 45, -5, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_526, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_526", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_526() */
/* END TEST 526 */

/*
 *  BEGIN TEST 527
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_ROWMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_527( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_ROWMAJ, x512_VW );
	const oski_value_t y_true[] = {59, 230, 187, 547, -384, -692, 556, -996, -417, -773, 490, 1069, 378, -868, 684 };
	oski_value_t alpha = 9;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_527, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_527", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_527() */
/* END TEST 527 */

/*
 *  BEGIN TEST 528
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_528( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	const oski_value_t y_true[] = {-26, 28, 23, 14, 41, 101, 8, 75, 40, -90, -90, -122, 36, 42, 32 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_528, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_528", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_528() */
/* END TEST 528 */

/*
 *  BEGIN TEST 529
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_529( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	const oski_value_t y_true[] = {50, 2, -17, -32, -23, -41, -68, -27, -16, 36, 108, 164, -96, 6, 22 };
	oski_value_t alpha = -1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_529, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_529", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_529() */
/* END TEST 529 */

/*
 *  BEGIN TEST 530
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_530( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	const oski_value_t y_true[] = {-16, -20, -4, 12, -12, -40, 40, -32, -16, 36, -12, -28, 40, -32, -36 };
	oski_value_t alpha = 0;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_530, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_530", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_530() */
/* END TEST 530 */

/*
 *  BEGIN TEST 531
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_531( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	const oski_value_t y_true[] = {-326, 137, 184, 195, 300, 679, 302, 491, 268, -603, -879, -1259, 554, 194, 81 };
	oski_value_t alpha = 9;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_531, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_531", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_531() */
/* END TEST 531 */

/*
 *  BEGIN TEST 532
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_532( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	const oski_value_t y_true[] = {12, -71, 86, 41, -1, 14, 105, -24, -80, 6, 9, 45, 0, -61, 72 };
	oski_value_t alpha = 1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_532, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_532", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_532() */
/* END TEST 532 */

/*
 *  BEGIN TEST 533
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_533( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	const oski_value_t y_true[] = {8, 55, -94, -49, -5, -28, -113, 18, 86, -2, -29, -53, -18, 63, -72 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_533, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_533", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_533() */
/* END TEST 533 */

/*
 *  BEGIN TEST 534
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_534( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	const oski_value_t y_true[] = {-10, 8, 4, 4, 3, 7, 4, 3, -3, -2, 10, 4, 9, -1, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_534, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_534", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_534() */
/* END TEST 534 */

/*
 *  BEGIN TEST 535
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_535( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_ROWMAJ, y512_VW );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	const oski_value_t y_true[] = {-214, 601, -550, -235, 46, -7, -683, 207, 521, -68, 67, -263, 117, 414, -504 };
	oski_value_t alpha = -7;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_535, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_535", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_535() */
/* END TEST 535 */

/*
 *  BEGIN TEST 536
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_536( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	const oski_value_t y_true[] = {-62, -15, 21, -51, 35, -52, 105, 3, -124, 89, 0, 32, 102, -192, -49 };
	oski_value_t alpha = 1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_536, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_536", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_536() */
/* END TEST 536 */

/*
 *  BEGIN TEST 537
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_537( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	const oski_value_t y_true[] = {34, -20, -28, 72, -56, -18, -35, -59, 96, -26, -21, -81, -32, 136, -14 };
	oski_value_t alpha = -1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_537, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_537", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_537() */
/* END TEST 537 */

/*
 *  BEGIN TEST 538
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_538( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	const oski_value_t y_true[] = {-32, -40, -8, 24, -24, -80, 80, -64, -32, 72, -24, -56, 80, -64, -72 };
	oski_value_t alpha = 0;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_538, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_538", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_538() */
/* END TEST 538 */

/*
 *  BEGIN TEST 539
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_539( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	oski_value_t* y_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	const oski_value_t y_true[] = {-222, 160, 200, -516, 404, 166, 205, 445, -656, 146, 159, 595, 184, -920, 134 };
	oski_value_t alpha = 7;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_539, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_539", y_true, y_val, y512_LEN*y512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_539() */
/* END TEST 539 */

/*
 *  BEGIN TEST 540
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_540( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	const oski_value_t y_true[] = {31, 43, 52, -93, 32, 8, -52, -118, 56, -92, -1, -88, -63, 119, 76 };
	oski_value_t alpha = 1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_540, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_540", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_540() */
/* END TEST 540 */

/*
 *  BEGIN TEST 541
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_541( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	const oski_value_t y_true[] = {79, -131, -96, 49, -65, -85, 8, 85, -23, 114, -109, 44, -36, -108, -76 };
	oski_value_t alpha = -1;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_541, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_541", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_541() */
/* END TEST 541 */

/*
 *  BEGIN TEST 542
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_542( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	const oski_value_t y_true[] = {10, -8, -4, -4, -3, -7, -4, -3, 3, 2, -10, -4, -9, 1, 0 };
	oski_value_t alpha = 0;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_542, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_542", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_542() */
/* END TEST 542 */

/*
 *  BEGIN TEST 543
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x512', LAYOUT_COLMAJ>
 *  y : <'y512', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_543( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A512_ptr, A512_ind, A512_val,
		A512_M, A512_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y512_val,
		y512_LEN * y512_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y512_LEN, y512_VW, LAYOUT_COLMAJ, y512_LEN );
	oski_value_t* y_val = array_Duplicate( x512_val,
		x512_LEN * x512_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x512_LEN, x512_VW, LAYOUT_COLMAJ, x512_LEN );
	const oski_value_t y_true[] = {-14, 418, 392, -478, 252, 188, -232, -648, 276, -592, 194, -448, -198, 694, 456 };
	oski_value_t alpha = 6;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_543, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_543", y_true, y_val, x512_LEN*x512_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_543() */
/* END TEST 543 */

#define A544_M 5    /* # of rows */
#define A544_N 5    /* # of cols */
#define A544_NNZ 7  /* # of stored non-zeros */
static oski_index_t A544_ptr[A544_M+1] = {0, 4, 5, 6, 6, 7}; /* end row pointers for A544 */
static oski_index_t A544_ind[A544_NNZ] = {1, 2, 3, 4, 3, 3, 4}; /* end column indices pointers for A544 */
static oski_value_t A544_val[A544_NNZ] = {-4, 1, -10, 7, -3, 10, -9}; /* end stored non-zero values for A544 */

#define x544_LEN 5   /* vector length */
#define x544_VW  3   /* vector width */
static oski_value_t x544_val[x544_LEN*x544_VW] = {7, -7, -2, 10, 4, -4, -10, 10, -9, 0, -4, -7, 7, -8, 3}; /* data */

#define y544_LEN 5   /* vector length */
#define y544_VW  3   /* vector width */
static oski_value_t y544_val[y544_LEN*y544_VW] = {-3, 1, -1, 2, 4, -10, 7, -2, 1, 10, 5, 6, 6, -1, -2}; /* data */

/*
 *  BEGIN TEST 544
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_544( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	const oski_value_t y_true[] = {14, -27, 103, -38, 20, 79, -28, -37, -77, -250, 133, -88, -44, 28, -31 };
	oski_value_t alpha = 1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_544, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_544", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_544() */
/* END TEST 544 */

/*
 *  BEGIN TEST 545
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_545( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	const oski_value_t y_true[] = {-11, 26, -102, 36, -24, -69, 21, 39, 76, 240, -138, 82, 38, -27, 33 };
	oski_value_t alpha = -1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_545, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_545", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_545() */
/* END TEST 545 */

/*
 *  BEGIN TEST 546
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_546( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	const oski_value_t y_true[] = {15, -5, 5, -10, -20, 50, -35, 10, -5, -50, -25, -30, -30, 5, 10 };
	oski_value_t alpha = 0;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_546, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_546", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_546() */
/* END TEST 546 */

/*
 *  BEGIN TEST 547
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_547( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	const oski_value_t y_true[] = {3, -68, 296, -88, 112, 107, 7, -137, -218, -620, 464, -186, -54, 71, -119 };
	oski_value_t alpha = 3;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_547, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_547", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_547() */
/* END TEST 547 */

/*
 *  BEGIN TEST 548
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_548( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	const oski_value_t y_true[] = {-87, -47, -25, -58, -35, 2, 137, 11, 95, 94, -26, 78, -103, 48, -1 };
	oski_value_t alpha = 1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_548, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_548", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_548() */
/* END TEST 548 */

/*
 *  BEGIN TEST 549
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_549( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	const oski_value_t y_true[] = {38, 96, 39, -12, 7, 26, -67, -81, -32, -94, 54, -29, 54, 8, -20 };
	oski_value_t alpha = -1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_549, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_549", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_549() */
/* END TEST 549 */

/*
 *  BEGIN TEST 550
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_550( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	const oski_value_t y_true[] = {-28, 28, 8, -40, -16, 16, 40, -40, 36, 0, 16, 28, -28, 32, -12 };
	oski_value_t alpha = 0;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_550, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_550", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_550() */
/* END TEST 550 */

/*
 *  BEGIN TEST 551
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_551( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	const oski_value_t y_true[] = {-121, -281, -115, 26, -25, -74, 211, 233, 105, 282, -158, 94, -169, -16, 57 };
	oski_value_t alpha = 3;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_551, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_551", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_551() */
/* END TEST 551 */

/*
 *  BEGIN TEST 552
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_552( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	const oski_value_t y_true[] = {-16, -23, 2, -190, 6, -72, 75, -57, 163, 73, 123, 59, -42, -63, -51 };
	oski_value_t alpha = 1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_552, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_552", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_552() */
/* END TEST 552 */

/*
 *  BEGIN TEST 553
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_553( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	const oski_value_t y_true[] = {-20, 35, -14, 214, 42, -48, 9, 33, -151, 47, -63, 13, 114, 51, 27 };
	oski_value_t alpha = -1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_553, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_553", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_553() */
/* END TEST 553 */

/*
 *  BEGIN TEST 554
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_554( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	const oski_value_t y_true[] = {27, -9, 9, -18, -36, 90, -63, 18, -9, -90, -45, -54, -54, 9, 18 };
	oski_value_t alpha = 0;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_554, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_554", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_554() */
/* END TEST 554 */

/*
 *  BEGIN TEST 555
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_555( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	const oski_value_t y_true[] = {15, -260, 71, -1816, -158, -118, 304, -407, 1414, 127, 842, 213, -696, -514, -353 };
	oski_value_t alpha = 9;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_555, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_555", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_555() */
/* END TEST 555 */

/*
 *  BEGIN TEST 556
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_556( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	const oski_value_t y_true[] = {-11, 44, -18, -14, 29, -9, 37, -20, 13, 17, 67, 6, -71, -144, 47 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_556, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_556", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_556() */
/* END TEST 556 */

/*
 *  BEGIN TEST 557
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_557( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	const oski_value_t y_true[] = {-45, 12, 34, -66, -61, 41, 43, -60, 59, -17, -35, 50, 15, 208, -71 };
	oski_value_t alpha = -1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_557, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_557", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_557() */
/* END TEST 557 */

/*
 *  BEGIN TEST 558
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_558( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	const oski_value_t y_true[] = {35, -35, -10, 50, 20, -20, -50, 50, -45, 0, -20, -35, 35, -40, 15 };
	oski_value_t alpha = 0;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_558, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_558", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_558() */
/* END TEST 558 */

/*
 *  BEGIN TEST 559
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_ROWMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_559( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_ROWMAJ, x544_VW );
	const oski_value_t y_true[] = {-13, -284, 194, -34, -325, 145, -173, 20, 27, -153, -539, 58, 527, 1424, -471 };
	oski_value_t alpha = -9;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_559, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_559", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_559() */
/* END TEST 559 */

/*
 *  BEGIN TEST 560
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_560( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	const oski_value_t y_true[] = {-70, 148, 128, -42, 75, -40, 163, -110, -76, 11, 210, 179, 61, -36, -71 };
	oski_value_t alpha = 1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_560, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_560", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_560() */
/* END TEST 560 */

/*
 *  BEGIN TEST 561
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_561( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	const oski_value_t y_true[] = {67, -147, -129, 44, -71, 30, -156, 108, 77, -1, -205, -173, -55, 35, 69 };
	oski_value_t alpha = -1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_561, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_561", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_561() */
/* END TEST 561 */

/*
 *  BEGIN TEST 562
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_562( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	const oski_value_t y_true[] = {-3, 1, -1, 2, 4, -10, 7, -2, 1, 10, 5, 6, 6, -1, -2 };
	oski_value_t alpha = 0;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_562, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_562", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_562() */
/* END TEST 562 */

/*
 *  BEGIN TEST 563
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_563( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	const oski_value_t y_true[] = {365, -1119, -1089, 466, -340, -330, -849, 750, 673, 562, -1355, -1042, -98, 223, 438 };
	oski_value_t alpha = -8;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_563, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_563", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_563() */
/* END TEST 563 */

/*
 *  BEGIN TEST 564
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_564( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	const oski_value_t y_true[] = {-73, -4, 101, 74, -83, -67, 1, 31, -24, 16, -25, 0, 45, 66, 5 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_564, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_564", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_564() */
/* END TEST 564 */

/*
 *  BEGIN TEST 565
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_565( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	const oski_value_t y_true[] = {59, 18, -97, -94, 75, 75, 19, -51, 42, -16, 33, 14, -59, -50, -11 };
	oski_value_t alpha = -1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_565, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_565", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_565() */
/* END TEST 565 */

/*
 *  BEGIN TEST 566
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_566( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	const oski_value_t y_true[] = {-14, 14, 4, -20, -8, 8, 20, -20, 18, 0, 8, 14, -14, 16, -6 };
	oski_value_t alpha = 0;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_566, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_566", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_566() */
/* END TEST 566 */

/*
 *  BEGIN TEST 567
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_567( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_ROWMAJ, y544_VW );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	const oski_value_t y_true[] = {125, 29, -196, -178, 154, 146, 28, -92, 75, -32, 62, 21, -111, -108, -19 };
	oski_value_t alpha = -2;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_567, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_567", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_567() */
/* END TEST 567 */

/*
 *  BEGIN TEST 568
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_568( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	const oski_value_t y_true[] = {-70, -50, 99, -53, 45, 60, 99, -110, 178, 52, 176, 88, -36, 123, -71 };
	oski_value_t alpha = 1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_568, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_568", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_568() */
/* END TEST 568 */

/*
 *  BEGIN TEST 569
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_569( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	const oski_value_t y_true[] = {34, 62, -111, 77, 3, -180, -15, 86, -166, 68, -116, -16, 108, -135, 47 };
	oski_value_t alpha = -1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_569, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_569", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_569() */
/* END TEST 569 */

/*
 *  BEGIN TEST 570
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_570( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	const oski_value_t y_true[] = {-6, 2, -2, 4, 8, -20, 14, -4, 2, 20, 10, 12, 12, -2, -4 };
	oski_value_t alpha = 0;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_570, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_570", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_570() */
/* END TEST 570 */

/*
 *  BEGIN TEST 571
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_571( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	oski_value_t* y_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	const oski_value_t y_true[] = {426, 518, -959, 613, -133, -1220, -415, 854, -1534, 212, -1244, -384, 732, -1175, 503 };
	oski_value_t alpha = -9;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_571, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_571", y_true, y_val, y544_LEN*y544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_571() */
/* END TEST 571 */

/*
 *  BEGIN TEST 572
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_572( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	const oski_value_t y_true[] = {-11, 20, 21, -3, -65, 38, 57, -20, 77, -160, -14, -3, -19, 8, 47 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_572, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_572", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_572() */
/* END TEST 572 */

/*
 *  BEGIN TEST 573
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_573( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	const oski_value_t y_true[] = {25, -34, -25, 23, 73, -46, -77, 40, -95, 160, 6, -11, 33, -24, -41 };
	oski_value_t alpha = -1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_573, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_573", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_573() */
/* END TEST 573 */

/*
 *  BEGIN TEST 574
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_574( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	const oski_value_t y_true[] = {7, -7, -2, 10, 4, -4, -10, 10, -9, 0, -4, -7, 7, -8, 3 };
	oski_value_t alpha = 0;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_574, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_574", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_574() */
/* END TEST 574 */

/*
 *  BEGIN TEST 575
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x544', LAYOUT_COLMAJ>
 *  y : <'y544', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_575( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A544_ptr, A544_ind, A544_val,
		A544_M, A544_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y544_val,
		y544_LEN * y544_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y544_LEN, y544_VW, LAYOUT_COLMAJ, y544_LEN );
	oski_value_t* y_val = array_Duplicate( x544_val,
		x544_LEN * x544_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x544_LEN, x544_VW, LAYOUT_COLMAJ, x544_LEN );
	const oski_value_t y_true[] = {-25, 34, 25, -23, -73, 46, 77, -40, 95, -160, -6, 11, -33, 24, 41 };
	oski_value_t alpha = 1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_575, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_575", y_true, y_val, x544_LEN*x544_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_575() */
/* END TEST 575 */

#define A576_M 5    /* # of rows */
#define A576_N 5    /* # of cols */
#define A576_NNZ 5  /* # of stored non-zeros */
static oski_index_t A576_ptr[A576_M+1] = {1, 4, 5, 6, 6, 6}; /* end row pointers for A576 */
static oski_index_t A576_ind[A576_NNZ] = {2, 3, 5, 4, 5}; /* end column indices pointers for A576 */
static oski_value_t A576_val[A576_NNZ] = {10, -10, -2, -7, -4}; /* end stored non-zero values for A576 */

#define x576_LEN 5   /* vector length */
#define x576_VW  3   /* vector width */
static oski_value_t x576_val[x576_LEN*x576_VW] = {0, -1, 5, -10, -7, -8, -3, -3, -6, -8, 8, -6, 8, 6, 10}; /* data */

#define y576_LEN 5   /* vector length */
#define y576_VW  3   /* vector width */
static oski_value_t y576_val[y576_LEN*y576_VW] = {-5, -5, 1, 4, 0, 0, -3, 7, -2, -10, 8, 9, 1, 10, -2}; /* data */

/*
 *  BEGIN TEST 576
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_576( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	const oski_value_t y_true[] = {-96, -63, -33, 54, -73, 84, -41, -3, -100, 42, 73, 68, 22, 40, 20 };
	oski_value_t alpha = 1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_576, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_576", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_576() */
/* END TEST 576 */

/*
 *  BEGIN TEST 577
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_577( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	const oski_value_t y_true[] = {56, 23, 41, -22, 73, -84, 17, 59, 84, -122, -9, 4, -14, 40, -36 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_577, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_577", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_577() */
/* END TEST 577 */

/*
 *  BEGIN TEST 578
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_578( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	const oski_value_t y_true[] = {10, 10, -2, -8, 0, 0, 6, -14, 4, 20, -16, -18, -2, -20, 4 };
	oski_value_t alpha = 0;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_578, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_578", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_578() */
/* END TEST 578 */

/*
 *  BEGIN TEST 579
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_579( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	const oski_value_t y_true[] = {683, 419, 281, -364, 584, -672, 277, 143, 766, -506, -448, -391, -159, -150, -194 };
	oski_value_t alpha = -8;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_579, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_579", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_579() */
/* END TEST 579 */

/*
 *  BEGIN TEST 580
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_580( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	const oski_value_t y_true[] = {63, -88, -10, 94, -57, 3, 64, 38, 38, 18, -48, 51, -33, -50, -66 };
	oski_value_t alpha = 1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_580, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_580", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_580() */
/* END TEST 580 */

/*
 *  BEGIN TEST 581
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_581( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	const oski_value_t y_true[] = {-63, 94, -20, -34, 99, 45, -46, -20, -2, 30, 0, -15, -15, 14, 6 };
	oski_value_t alpha = -1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_581, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_581", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_581() */
/* END TEST 581 */

/*
 *  BEGIN TEST 582
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_582( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	const oski_value_t y_true[] = {0, -8, 40, -80, -56, -64, -24, -24, -48, -64, 64, -48, 64, 48, 80 };
	oski_value_t alpha = 0;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_582, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_582", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_582() */
/* END TEST 582 */

/*
 *  BEGIN TEST 583
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_583( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	const oski_value_t y_true[] = {189, -280, 50, 122, -283, -119, 144, 66, 18, -74, -16, 57, 29, -54, -38 };
	oski_value_t alpha = 3;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_583, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_583", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_583() */
/* END TEST 583 */

/*
 *  BEGIN TEST 584
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_584( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	const oski_value_t y_true[] = {-91, 41, -34, 66, 20, -53, -76, -10, 55, 10, -27, 93, -95, 60, 22 };
	oski_value_t alpha = 1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_584, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_584", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_584() */
/* END TEST 584 */

/*
 *  BEGIN TEST 585
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_585( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	const oski_value_t y_true[] = {56, -76, 41, -38, -20, 53, 55, 59, -69, -80, 83, -30, 102, 10, -36 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_585, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_585", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_585() */
/* END TEST 585 */

/*
 *  BEGIN TEST 586
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_586( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	const oski_value_t y_true[] = {20, 20, -4, -16, 0, 0, 12, -28, 8, 40, -32, -36, -4, -40, 8 };
	oski_value_t alpha = 0;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_586, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_586", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_586() */
/* END TEST 586 */

/*
 *  BEGIN TEST 587
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_587( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	const oski_value_t y_true[] = {213, -183, 114, -150, -60, 159, 192, 114, -189, -150, 177, -171, 297, -60, -90 };
	oski_value_t alpha = -3;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_587, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_587", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_587() */
/* END TEST 587 */

/*
 *  BEGIN TEST 588
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_588( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	const oski_value_t y_true[] = {-65, -89, 137, -173, -52, -53, 24, 20, -125, -33, 91, -107, 78, 16, 68 };
	oski_value_t alpha = 1;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_588, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_588", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_588() */
/* END TEST 588 */

/*
 *  BEGIN TEST 589
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_589( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	const oski_value_t y_true[] = {65, 78, -82, 63, -25, -35, -57, -53, 59, -55, -3, 41, 10, 50, 42 };
	oski_value_t alpha = -1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_589, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_589", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_589() */
/* END TEST 589 */

/*
 *  BEGIN TEST 590
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_590( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	const oski_value_t y_true[] = {0, 3, -15, 30, 21, 24, 9, 9, 18, 24, -24, 18, -24, -18, -30 };
	oski_value_t alpha = 0;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_590, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_590", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_590() */
/* END TEST 590 */

/*
 *  BEGIN TEST 591
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_591( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	const oski_value_t y_true[] = {0, 8, -40, 80, 56, 64, 24, 24, 48, 64, -64, 48, -64, -48, -80 };
	oski_value_t alpha = 0;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_591, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_591", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_591() */
/* END TEST 591 */

/*
 *  BEGIN TEST 592
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_592( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	const oski_value_t y_true[] = {-76, -22, -146, 93, -41, 32, 15, 151, -124, -63, 63, 102, -21, 80, -50 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_592, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_592", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_592() */
/* END TEST 592 */

/*
 *  BEGIN TEST 593
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_593( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	const oski_value_t y_true[] = {31, -23, 155, -57, 41, -32, -42, -88, 106, -27, 9, -21, 30, 10, 32 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_593, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_593", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_593() */
/* END TEST 593 */

/*
 *  BEGIN TEST 594
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_594( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	const oski_value_t y_true[] = {-10, -10, 2, 8, 0, 0, -6, 14, -4, -20, 16, 18, 2, 20, -4 };
	oski_value_t alpha = 0;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_594, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_594", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_594() */
/* END TEST 594 */

/*
 *  BEGIN TEST 595
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_595( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	const oski_value_t y_true[] = {255, -15, 755, -365, 205, -160, -150, -580, 570, 65, -115, -285, 130, -150, 200 };
	oski_value_t alpha = -5;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_595, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_595", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_595() */
/* END TEST 595 */

/*
 *  BEGIN TEST 596
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_596( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	const oski_value_t y_true[] = {63, 30, 13, 22, 65, -47, -88, 35, 44, 40, -23, -17, -52, -27, -56 };
	oski_value_t alpha = 1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_596, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_596", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_596() */
/* END TEST 596 */

/*
 *  BEGIN TEST 597
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_597( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	const oski_value_t y_true[] = {-63, -20, -63, 78, 5, 127, 118, -5, 16, 40, -57, 77, -28, -33, -44 };
	oski_value_t alpha = -1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_597, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_597", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_597() */
/* END TEST 597 */

/*
 *  BEGIN TEST 598
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_598( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	const oski_value_t y_true[] = {0, 5, -25, 50, 35, 40, 15, 15, 30, 40, -40, 30, -40, -30, -50 };
	oski_value_t alpha = 0;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_598, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_598", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_598() */
/* END TEST 598 */

/*
 *  BEGIN TEST 599
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_599( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	const oski_value_t y_true[] = {315, 113, 250, -260, 66, -531, -551, 64, -2, -96, 181, -307, 36, 87, 90 };
	oski_value_t alpha = 5;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_599, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_599", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_599() */
/* END TEST 599 */

/*
 *  BEGIN TEST 600
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_600( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	const oski_value_t y_true[] = {-71, 44, 38, 17, -27, 8, -56, 144, 5, -30, -112, 77, -107, 98, -48 };
	oski_value_t alpha = 1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_600, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_600", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_600() */
/* END TEST 600 */

/*
 *  BEGIN TEST 601
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_601( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	const oski_value_t y_true[] = {81, -34, -40, -25, 27, -8, 62, -158, -1, 50, 96, -95, 105, -118, 52 };
	oski_value_t alpha = -1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_601, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_601", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_601() */
/* END TEST 601 */

/*
 *  BEGIN TEST 602
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_602( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	const oski_value_t y_true[] = {30, 30, -6, -24, 0, 0, 18, -42, 12, 60, -48, -54, -6, -60, 12 };
	oski_value_t alpha = 0;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_602, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_602", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_602() */
/* END TEST 602 */

/*
 *  BEGIN TEST 603
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_603( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	const oski_value_t y_true[] = {205, -370, -160, 35, 135, -40, 190, -510, -85, -150, 800, -115, 565, -190, 180 };
	oski_value_t alpha = -5;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_603, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_603", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_603() */
/* END TEST 603 */

/*
 *  BEGIN TEST 604
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_604( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	const oski_value_t y_true[] = {-65, -83, 51, 39, 6, -80, 11, 47, 19, -38, 92, 19, -71, -53, -22 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_604, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_604", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_604() */
/* END TEST 604 */

/*
 *  BEGIN TEST 605
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_605( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	const oski_value_t y_true[] = {65, 85, -61, -19, 8, 96, -5, -41, -7, 54, -108, -7, 55, 41, 2 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_605, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_605", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_605() */
/* END TEST 605 */

/*
 *  BEGIN TEST 606
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_606( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	const oski_value_t y_true[] = {0, -7, 35, -70, -49, -56, -21, -21, -42, -56, 56, -42, 56, 42, 70 };
	oski_value_t alpha = 0;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_606, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_606", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_606() */
/* END TEST 606 */

/*
 *  BEGIN TEST 607
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_COLMAJ>
 *  y : <'y576', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_607( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y576_LEN, y576_VW, LAYOUT_COLMAJ, y576_LEN );
	oski_value_t* y_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x576_LEN, x576_VW, LAYOUT_COLMAJ, x576_LEN );
	const oski_value_t y_true[] = {195, 247, -143, -137, -32, 224, -39, -147, -69, 98, -260, -69, 229, 171, 86 };
	oski_value_t alpha = -3;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_607, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_607", y_true, y_val, x576_LEN*x576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_607() */
/* END TEST 607 */

#define A608_M 5    /* # of rows */
#define A608_N 5    /* # of cols */
#define A608_NNZ 5  /* # of stored non-zeros */
static oski_index_t A608_ptr[A608_M+1] = {0, 0, 2, 4, 5, 5}; /* end row pointers for A608 */
static oski_index_t A608_ind[A608_NNZ] = {2, 3, 3, 4, 4}; /* end column indices pointers for A608 */
static oski_value_t A608_val[A608_NNZ] = {9, -1, -4, 4, -10}; /* end stored non-zero values for A608 */

#define x608_LEN 5   /* vector length */
#define x608_VW  3   /* vector width */
static oski_value_t x608_val[x608_LEN*x608_VW] = {1, 4, 4, 5, 3, 10, -4, 6, -10, -8, -9, 9, -6, 6, -5}; /* data */

#define y608_LEN 5   /* vector length */
#define y608_VW  3   /* vector width */
static oski_value_t y608_val[y608_LEN*y608_VW] = {-8, 5, -8, -3, 10, 10, 9, -9, 5, 3, -3, -3, -10, -2, 9}; /* data */

/*
 *  BEGIN TEST 608
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_608( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	const oski_value_t y_true[] = {-47, 34, -44, -41, 126, -29, 103, 39, 54, 81, -114, 71, -2, 108, -81 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_608, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_608", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_608() */
/* END TEST 608 */

/*
 *  BEGIN TEST 609
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_609( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	const oski_value_t y_true[] = {-65, 36, -68, -1, 14, 169, 23, -165, 16, -39, 72, -113, -138, -136, 207 };
	oski_value_t alpha = -1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_609, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_609", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_609() */
/* END TEST 609 */

/*
 *  BEGIN TEST 610
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_610( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	const oski_value_t y_true[] = {-72, 45, -72, -27, 90, 90, 81, -81, 45, 27, -27, -27, -90, -18, 81 };
	oski_value_t alpha = 0;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_610, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_610", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_610() */
/* END TEST 610 */

/*
 *  BEGIN TEST 611
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_611( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	const oski_value_t y_true[] = {56, 2, 80, -166, 468, -772, 338, 798, 162, 486, -750, 730, 524, 972, -1134 };
	oski_value_t alpha = 8;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_611, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_611", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_611() */
/* END TEST 611 */

/*
 *  BEGIN TEST 612
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_612( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	const oski_value_t y_true[] = {-12, -11, -24, 55, -80, 18, -54, 61, 183, 102, 79, -159, 20, -32, 79 };
	oski_value_t alpha = 1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_612, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_612", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_612() */
/* END TEST 612 */

/*
 *  BEGIN TEST 613
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_613( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	const oski_value_t y_true[] = {6, -13, 0, -85, 62, -78, 78, -97, -123, -54, -25, 105, 16, -4, -49 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_613, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_613", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_613() */
/* END TEST 613 */

/*
 *  BEGIN TEST 614
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_614( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	const oski_value_t y_true[] = {5, 20, 20, 25, 15, 50, -20, 30, -50, -40, -45, 45, -30, 30, -25 };
	oski_value_t alpha = 0;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_614, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_614", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_614() */
/* END TEST 614 */

/*
 *  BEGIN TEST 615
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_615( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	const oski_value_t y_true[] = {-35, 45, -20, 400, -325, 340, -370, 455, 665, 310, 170, -570, -50, -10, 270 };
	oski_value_t alpha = 5;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_615, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_615", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_615() */
/* END TEST 615 */

/*
 *  BEGIN TEST 616
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_616( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	const oski_value_t y_true[] = {9, -28, 57, 66, 48, -6, 57, 102, -101, 117, 7, -86, 34, 91, -144 };
	oski_value_t alpha = 1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_616, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_616", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_616() */
/* END TEST 616 */

/*
 *  BEGIN TEST 617
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_617( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	const oski_value_t y_true[] = {7, 18, -41, -60, -68, -14, -75, -84, 91, -123, -1, 92, -14, -87, 126 };
	oski_value_t alpha = -1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_617, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_617", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_617() */
/* END TEST 617 */

/*
 *  BEGIN TEST 618
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_618( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	const oski_value_t y_true[] = {-16, 10, -16, -6, 20, 20, 18, -18, 10, 6, -6, -6, -20, -4, 18 };
	oski_value_t alpha = 0;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_618, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_618", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_618() */
/* END TEST 618 */

/*
 *  BEGIN TEST 619
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_619( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	const oski_value_t y_true[] = {-29, -49, 115, 177, 214, 52, 234, 243, -268, 372, 0, -279, 32, 259, -369 };
	oski_value_t alpha = 3;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_619, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_619", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_619() */
/* END TEST 619 */

/*
 *  BEGIN TEST 620
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_620( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	const oski_value_t y_true[] = {-2, 34, 21, -34, -59, -31, 65, 100, -53, -124, -52, 5, -28, -47, -41 };
	oski_value_t alpha = 1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_620, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_620", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_620() */
/* END TEST 620 */

/*
 *  BEGIN TEST 621
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_621( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	const oski_value_t y_true[] = {4, -26, -13, 44, 65, 51, -73, -88, 33, 108, 34, 13, 16, 59, 31 };
	oski_value_t alpha = -1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_621, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_621", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_621() */
/* END TEST 621 */

/*
 *  BEGIN TEST 622
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_622( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	const oski_value_t y_true[] = {5, 20, 20, 25, 15, 50, -20, 30, -50, -40, -45, 45, -30, 30, -25 };
	oski_value_t alpha = 0;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_622, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_622", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_622() */
/* END TEST 622 */

/*
 *  BEGIN TEST 623
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_ROWMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_623( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_ROWMAJ, x608_VW );
	const oski_value_t y_true[] = {41, -88, -10, 349, 441, 476, -506, -426, 28, 512, 51, 231, -6, 456, 101 };
	oski_value_t alpha = -6;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_623, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_623", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_623() */
/* END TEST 623 */

/*
 *  BEGIN TEST 624
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_624( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	const oski_value_t y_true[] = {-39, 35, -49, 20, 110, -1, 77, -67, 56, -30, 35, 56, -81, 106, -44 };
	oski_value_t alpha = 1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_624, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_624", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_624() */
/* END TEST 624 */

/*
 *  BEGIN TEST 625
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_625( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	const oski_value_t y_true[] = {-73, 35, -63, -62, 30, 141, 49, -59, 14, 72, -77, -98, -59, -134, 170 };
	oski_value_t alpha = -1;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_625, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_625", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_625() */
/* END TEST 625 */

/*
 *  BEGIN TEST 626
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_626( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	const oski_value_t y_true[] = {64, -40, 64, 24, -80, -80, -72, 72, -40, -24, 24, 24, 80, 16, -72 };
	oski_value_t alpha = 0;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_626, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_626", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_626() */
/* END TEST 626 */

/*
 *  BEGIN TEST 627
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_627( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	const oski_value_t y_true[] = {-49, 20, -39, -53, 0, 111, 22, -32, -1, 63, -68, -89, -29, -128, 143 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_627, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_627", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_627() */
/* END TEST 627 */

/*
 *  BEGIN TEST 628
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_628( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	const oski_value_t y_true[] = {-6, 83, -62, 80, 2, 25, -76, 97, 23, -24, -26, 76, 131, -111, 49 };
	oski_value_t alpha = 1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_628, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_628", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_628() */
/* END TEST 628 */

/*
 *  BEGIN TEST 629
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_629( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	const oski_value_t y_true[] = {16, -43, 102, -30, 28, 75, 36, -37, -123, -56, -64, 14, -191, 171, -99 };
	oski_value_t alpha = -1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_629, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_629", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_629() */
/* END TEST 629 */

/*
 *  BEGIN TEST 630
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_630( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	const oski_value_t y_true[] = {-1, -4, -4, -5, -3, -10, 4, -6, 10, 8, 9, -9, 6, -6, 5 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_630, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_630", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_630() */
/* END TEST 630 */

/*
 *  BEGIN TEST 631
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_631( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_ROWMAJ, y608_VW );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	const oski_value_t y_true[] = {-59, 620, -540, 585, -17, 90, -564, 710, 294, -104, -109, 509, 1114, -954, 447 };
	oski_value_t alpha = 8;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_631, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_631", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_631() */
/* END TEST 631 */

/*
 *  BEGIN TEST 632
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_632( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	const oski_value_t y_true[] = {17, 25, 48, -39, -51, -10, 42, -4, 40, 110, -3, -45, 51, 75, -107 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_632, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_632", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_632() */
/* END TEST 632 */

/*
 *  BEGIN TEST 633
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_633( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	const oski_value_t y_true[] = {39, -60, 8, 60, -19, -60, -105, 67, -75, -131, 24, 66, 19, -61, 44 };
	oski_value_t alpha = -1;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_633, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_633", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_633() */
/* END TEST 633 */

/*
 *  BEGIN TEST 634
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_634( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	const oski_value_t y_true[] = {-48, 30, -48, -18, 60, 60, 54, -54, 30, 18, -18, -18, -60, -12, 54 };
	oski_value_t alpha = 0;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_634, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_634", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_634() */
/* END TEST 634 */

/*
 *  BEGIN TEST 635
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_635( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	oski_value_t* y_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	const oski_value_t y_true[] = {32, -305, -216, 375, 198, -130, -525, 221, -425, -943, 87, 423, -198, -558, 667 };
	oski_value_t alpha = -8;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_635, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_635", y_true, y_val, y608_LEN*y608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_635() */
/* END TEST 635 */

/*
 *  BEGIN TEST 636
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_636( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	const oski_value_t y_true[] = {-15, -92, 61, -111, -13, -60, -49, 22, 72, -27, 60, -154, 49, -91, 24 };
	oski_value_t alpha = 1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_636, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_636", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_636() */
/* END TEST 636 */

/*
 *  BEGIN TEST 637
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_637( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	const oski_value_t y_true[] = {15, 92, -61, 111, 13, 60, 49, -22, -72, 27, -60, 154, -49, 91, -24 };
	oski_value_t alpha = -1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_637, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_637", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_637() */
/* END TEST 637 */

/*
 *  BEGIN TEST 638
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_638( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	const oski_value_t y_true[] = {-8, -32, -32, -40, -24, -80, 32, -48, 80, 64, 72, -72, 48, -48, 40 };
	oski_value_t alpha = 0;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_638, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_638", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_638() */
/* END TEST 638 */

/*
 *  BEGIN TEST 639
 *  Value type: real
 *  A : <MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x608', LAYOUT_COLMAJ>
 *  y : <'y608', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_639( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A608_ptr, A608_ind, A608_val,
		A608_M, A608_N, SHARE_INPUTMAT,
		5, MAT_SYMM_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y608_val,
		y608_LEN * y608_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y608_LEN, y608_VW, LAYOUT_COLMAJ, y608_LEN );
	oski_value_t* y_val = array_Duplicate( x608_val,
		x608_LEN * x608_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x608_LEN, x608_VW, LAYOUT_COLMAJ, x608_LEN );
	const oski_value_t y_true[] = {-58, -488, 736, -578, 82, 140, -640, 548, -44, -712, -78, -674, 20, -356, -118 };
	oski_value_t alpha = 8;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_639, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_639", y_true, y_val, x608_LEN*x608_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_639() */
/* END TEST 639 */

#define A640_M 5    /* # of rows */
#define A640_N 5    /* # of cols */
#define A640_NNZ 9  /* # of stored non-zeros */
static oski_index_t A640_ptr[A640_M+1] = {1, 1, 1, 4, 5, 10}; /* end row pointers for A640 */
static oski_index_t A640_ind[A640_NNZ] = {1, 2, 3, 2, 1, 2, 3, 4, 5}; /* end column indices pointers for A640 */
static oski_value_t A640_val[A640_NNZ] = {-5, 1, 6, -3, -3, -1, -10, -3, 10}; /* end stored non-zero values for A640 */

#define x640_LEN 5   /* vector length */
#define x640_VW  3   /* vector width */
static oski_value_t x640_val[x640_LEN*x640_VW] = {5, -4, -6, -1, 5, -5, -8, 6, 3, 5, -10, 10, -7, -6, 6}; /* data */

#define y640_LEN 5   /* vector length */
#define y640_VW  3   /* vector width */
static oski_value_t y640_val[y640_LEN*y640_VW] = {6, 10, 5, 5, -6, -6, -10, 7, 6, -6, -1, -5, 4, 7, -9}; /* data */

/*
 *  BEGIN TEST 640
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_640( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	const oski_value_t y_true[] = {73, 8, -23, -6, 30, -45, -24, 135, -5, 12, 1, -13, -11, -69, 5 };
	oski_value_t alpha = 1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_640, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_640", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_640() */
/* END TEST 640 */

/*
 *  BEGIN TEST 641
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_641( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	const oski_value_t y_true[] = {-49, 32, 43, 26, -54, 21, -16, -107, 29, -36, -5, -7, 27, 97, -41 };
	oski_value_t alpha = -1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_641, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_641", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_641() */
/* END TEST 641 */

/*
 *  BEGIN TEST 642
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_642( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	const oski_value_t y_true[] = {-6, -10, -5, -5, 6, 6, 10, -7, -6, 6, 1, 5, -4, -7, 9 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_642, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_642", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_642() */
/* END TEST 642 */

/*
 *  BEGIN TEST 643
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_643( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	const oski_value_t y_true[] = {-403, 124, 251, 132, -318, 207, -12, -819, 143, -192, -25, 1, 149, 609, -197 };
	oski_value_t alpha = -7;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_643, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_643", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_643() */
/* END TEST 643 */

/*
 *  BEGIN TEST 644
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_644( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	const oski_value_t y_true[] = {-7, -20, 51, 13, -42, 75, -53, -138, 68, -72, 87, -45, 198, 33, -198 };
	oski_value_t alpha = 1;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_644, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_644", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_644() */
/* END TEST 644 */

/*
 *  BEGIN TEST 645
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_645( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	const oski_value_t y_true[] = {-48, 64, 15, -2, -13, -20, 141, 72, -101, 17, 23, -65, -121, 33, 132 };
	oski_value_t alpha = -1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_645, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_645", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_645() */
/* END TEST 645 */

/*
 *  BEGIN TEST 646
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_646( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	const oski_value_t y_true[] = {5, -4, -6, -1, 5, -5, -8, 6, 3, 5, -10, 10, -7, -6, 6 };
	oski_value_t alpha = 0;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_646, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_646", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_646() */
/* END TEST 646 */

/*
 *  BEGIN TEST 647
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_647( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	const oski_value_t y_true[] = {-251, 380, 3, -31, -6, -225, 851, 606, -656, 204, -9, -285, -966, 129, 1026 };
	oski_value_t alpha = -7;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_647, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_647", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_647() */
/* END TEST 647 */

/*
 *  BEGIN TEST 648
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_648( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	const oski_value_t y_true[] = {55, -26, -9, 19, -13, -6, 52, 114, -3, -77, -32, -28, -21, -10, 32 };
	oski_value_t alpha = 1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_648, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_648", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_648() */
/* END TEST 648 */

/*
 *  BEGIN TEST 649
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_649( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	const oski_value_t y_true[] = {-109, -64, -36, -64, 67, 60, 38, -177, -51, 131, 41, 73, -15, -53, 49 };
	oski_value_t alpha = -1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_649, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_649", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_649() */
/* END TEST 649 */

/*
 *  BEGIN TEST 650
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_650( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	const oski_value_t y_true[] = {-30, -50, -25, -25, 30, 30, 50, -35, -30, 30, 5, 25, -20, -35, 45 };
	oski_value_t alpha = 0;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_650, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_650", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_650() */
/* END TEST 650 */

/*
 *  BEGIN TEST 651
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_651( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	const oski_value_t y_true[] = {-171, 68, 22, -62, 45, 24, -146, -349, 3, 237, 97, 89, 59, 23, -87 };
	oski_value_t alpha = -3;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_651, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_651", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_651() */
/* END TEST 651 */

/*
 *  BEGIN TEST 652
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_652( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	const oski_value_t y_true[] = {-17, -9, 19, -2, -15, 2, 86, 110, 108, -22, 68, 22, -139, -108, -155 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_652, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_652", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_652() */
/* END TEST 652 */

/*
 *  BEGIN TEST 653
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_653( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	const oski_value_t y_true[] = {12, 13, -13, 3, 10, 3, -78, -116, -111, 17, -58, -32, 146, 114, 149 };
	oski_value_t alpha = -1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_653, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_653", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_653() */
/* END TEST 653 */

/*
 *  BEGIN TEST 654
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_654( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	const oski_value_t y_true[] = {25, -20, -30, -5, 25, -25, -40, 30, 15, 25, -50, 50, -35, -30, 30 };
	oski_value_t alpha = 0;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_654, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_654", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_654() */
/* END TEST 654 */

/*
 *  BEGIN TEST 655
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_ROWMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_655( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_ROWMAJ, x640_VW );
	const oski_value_t y_true[] = {81, 116, -86, 27, 65, 39, -600, -946, -897, 121, -434, -286, 1189, 930, 1174 };
	oski_value_t alpha = -8;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_655, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_655", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_655() */
/* END TEST 655 */

/*
 *  BEGIN TEST 656
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_656( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	const oski_value_t y_true[] = {-33, -125, -23, -48, 40, 53, -35, -53, -90, 45, 17, -8, 70, -52, 240 };
	oski_value_t alpha = 1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_656, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_656", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_656() */
/* END TEST 656 */

/*
 *  BEGIN TEST 657
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_657( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	const oski_value_t y_true[] = {-51, -15, -47, -22, 44, 31, 175, -45, 6, 39, -3, 78, -126, -46, -114 };
	oski_value_t alpha = -1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_657, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_657", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_657() */
/* END TEST 657 */

/*
 *  BEGIN TEST 658
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_658( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	const oski_value_t y_true[] = {-6, -10, -5, -5, 6, 6, 10, -7, -6, 6, 1, 5, -4, -7, 9 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_658, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_658", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_658() */
/* END TEST 658 */

/*
 *  BEGIN TEST 659
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_659( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	const oski_value_t y_true[] = {72, -300, 87, -63, -30, 48, -660, -3, -270, 0, 57, -273, 600, 3, 1035 };
	oski_value_t alpha = 6;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_659, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_659", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_659() */
/* END TEST 659 */

/*
 *  BEGIN TEST 660
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_660( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	const oski_value_t y_true[] = {28, 12, -113, -25, 125, -46, 19, -96, -9, -31, 17, 10, 109, 57, -156 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_660, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_660", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_660() */
/* END TEST 660 */

/*
 *  BEGIN TEST 661
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_661( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	const oski_value_t y_true[] = {-8, -28, 89, 21, -105, 26, -51, 120, 21, 51, -57, 30, -137, -81, 180 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_661, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_661", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_661() */
/* END TEST 661 */

/*
 *  BEGIN TEST 662
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_662( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	const oski_value_t y_true[] = {25, -20, -30, -5, 25, -25, -40, 30, 15, 25, -50, 50, -35, -30, 30 };
	oski_value_t alpha = 0;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_662, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_662", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_662() */
/* END TEST 662 */

/*
 *  BEGIN TEST 663
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_663( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_ROWMAJ, y640_VW );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	const oski_value_t y_true[] = {-122, -40, 464, 102, -510, 194, -60, 372, 30, 114, -48, -60, -422, -216, 612 };
	oski_value_t alpha = -4;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_663, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_663", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_663() */
/* END TEST 663 */

/*
 *  BEGIN TEST 664
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_664( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	const oski_value_t y_true[] = {39, 32, -95, 17, 78, -69, -48, 31, 33, -20, 13, -15, -26, -20, 132 };
	oski_value_t alpha = 1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_664, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_664", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_664() */
/* END TEST 664 */

/*
 *  BEGIN TEST 665
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_665( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	const oski_value_t y_true[] = {27, 78, 150, 38, -144, 3, -62, 46, 33, -46, -24, -40, 70, 97, -231 };
	oski_value_t alpha = -1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_665, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_665", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_665() */
/* END TEST 665 */

/*
 *  BEGIN TEST 666
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_666( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	const oski_value_t y_true[] = {42, 70, 35, 35, -42, -42, -70, 49, 42, -42, -7, -35, 28, 49, -63 };
	oski_value_t alpha = 0;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_666, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_666", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_666() */
/* END TEST 666 */

/*
 *  BEGIN TEST 667
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_667( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	oski_value_t* y_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	const oski_value_t y_true[] = {-66, -44, 200, -24, -168, 126, 76, -48, -54, 28, -28, 20, 60, 54, -282 };
	oski_value_t alpha = -2;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_667, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_667", y_true, y_val, y640_LEN*y640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_667() */
/* END TEST 667 */

/*
 *  BEGIN TEST 668
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_668( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	const oski_value_t y_true[] = {-2, -8, 64, -13, -148, -22, -13, 128, 51, -115, -3, 2, 107, 36, -137 };
	oski_value_t alpha = 1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_668, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_668", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_668() */
/* END TEST 668 */

/*
 *  BEGIN TEST 669
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_669( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	const oski_value_t y_true[] = {12, 0, -76, 11, 158, 12, -3, -116, -45, 125, -17, 18, -121, -48, 149 };
	oski_value_t alpha = -1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_669, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_669", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_669() */
/* END TEST 669 */

/*
 *  BEGIN TEST 670
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_670( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	const oski_value_t y_true[] = {-15, 12, 18, 3, -15, 15, 24, -18, -9, -15, 30, -30, 21, 18, -18 };
	oski_value_t alpha = 0;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_670, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_670", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_670() */
/* END TEST 670 */

/*
 *  BEGIN TEST 671
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x640', LAYOUT_COLMAJ>
 *  y : <'y640', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_671( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A640_ptr, A640_ind, A640_val,
		A640_M, A640_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y640_val,
		y640_LEN * y640_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y640_LEN, y640_VW, LAYOUT_COLMAJ, y640_LEN );
	oski_value_t* y_val = array_Duplicate( x640_val,
		x640_LEN * x640_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x640_LEN, x640_VW, LAYOUT_COLMAJ, x640_LEN );
	const oski_value_t y_true[] = {12, -48, 232, -56, -572, -108, -84, 536, 216, -440, -52, 48, 400, 120, -524 };
	oski_value_t alpha = 4;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_671, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_671", y_true, y_val, x640_LEN*x640_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_671() */
/* END TEST 671 */

#define A672_M 5    /* # of rows */
#define A672_N 5    /* # of cols */
#define A672_NNZ 8  /* # of stored non-zeros */
static oski_index_t A672_ptr[A672_M+1] = {0, 1, 2, 2, 4, 8}; /* end row pointers for A672 */
static oski_index_t A672_ind[A672_NNZ] = {0, 1, 1, 3, 0, 1, 2, 3}; /* end column indices pointers for A672 */
static oski_value_t A672_val[A672_NNZ] = {10, 3, -4, -6, -10, -4, -1, -3}; /* end stored non-zero values for A672 */

#define x672_LEN 5   /* vector length */
#define x672_VW  3   /* vector width */
static oski_value_t x672_val[x672_LEN*x672_VW] = {-7, -9, -4, 2, -3, -6, -10, 5, -9, -5, 6, -3, -4, -3, 10}; /* data */

#define y672_LEN 5   /* vector length */
#define y672_VW  3   /* vector width */
static oski_value_t y672_val[y672_LEN*y672_VW] = {5, 7, -2, -6, 3, 4, 7, 7, 10, 1, 8, -6, -2, -10, 1}; /* data */

/*
 *  BEGIN TEST 672
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_672( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	const oski_value_t y_true[] = {-5, -25, -150, 12, -6, -26, 39, 38, 40, 39, 25, -18, 77, 29, 87 };
	oski_value_t alpha = 1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_672, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_672", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_672() */
/* END TEST 672 */

/*
 *  BEGIN TEST 673
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_673( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	const oski_value_t y_true[] = {70, 116, 124, -90, 45, 78, 52, 53, 90, -26, 79, -60, -103, -159, -74 };
	oski_value_t alpha = -1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_673, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_673", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_673() */
/* END TEST 673 */

/*
 *  BEGIN TEST 674
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_674( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	const oski_value_t y_true[] = {5, 7, -2, -6, 3, 4, 7, 7, 10, 1, 8, -6, -2, -10, 1 };
	oski_value_t alpha = 0;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_674, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_674", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_674() */
/* END TEST 674 */

/*
 *  BEGIN TEST 675
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_675( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	const oski_value_t y_true[] = {-25, -53, -142, 36, -18, -42, 11, 10, 0, 35, -7, 6, 85, 69, 83 };
	oski_value_t alpha = 1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_675, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_675", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_675() */
/* END TEST 675 */

/*
 *  BEGIN TEST 676
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_676( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	const oski_value_t y_true[] = {7, 89, -66, 4, -10, -22, -88, 55, -82, -21, 24, -10, -72, -140, 102 };
	oski_value_t alpha = 1;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_676, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_676", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_676() */
/* END TEST 676 */

/*
 *  BEGIN TEST 677
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_677( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	const oski_value_t y_true[] = {-98, -206, 14, 22, -29, -56, -42, 10, -35, -44, 54, -29, 20, 101, 28 };
	oski_value_t alpha = -1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_677, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_677", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_677() */
/* END TEST 677 */

/*
 *  BEGIN TEST 678
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_678( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	const oski_value_t y_true[] = {49, 63, 28, -14, 21, 42, 70, -35, 63, 35, -42, 21, 28, 21, -70 };
	oski_value_t alpha = 0;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_678, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_678", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_678() */
/* END TEST 678 */

/*
 *  BEGIN TEST 679
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_679( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	const oski_value_t y_true[] = {7, 9, 4, -2, 3, 6, 10, -5, 9, 5, -6, 3, 4, 3, -10 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_679, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_679", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_679() */
/* END TEST 679 */

/*
 *  BEGIN TEST 680
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_680( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	const oski_value_t y_true[] = {-30, 42, 4, 34, 87, -60, -21, 3, -15, 79, -140, -46, -10, 12, 82 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_680, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_680", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_680() */
/* END TEST 680 */

/*
 *  BEGIN TEST 681
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_681( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	const oski_value_t y_true[] = {35, -35, -6, -40, -84, 64, 28, 4, 25, -78, 148, 40, 8, -22, -81 };
	oski_value_t alpha = -1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_681, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_681", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_681() */
/* END TEST 681 */

/*
 *  BEGIN TEST 682
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_682( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	const oski_value_t y_true[] = {10, 14, -4, -12, 6, 8, 14, 14, 20, 2, 16, -12, -4, -20, 2 };
	oski_value_t alpha = 0;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_682, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_682", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_682() */
/* END TEST 682 */

/*
 *  BEGIN TEST 683
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_683( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	const oski_value_t y_true[] = {200, -224, -32, -228, -510, 376, 154, 10, 130, -470, 872, 252, 52, -112, -488 };
	oski_value_t alpha = -6;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_683, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_683", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_683() */
/* END TEST 683 */

/*
 *  BEGIN TEST 684
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_684( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	const oski_value_t y_true[] = {20, 30, 70, 33, -23, 18, -3, -1, -1, -1, -91, 81, -58, -105, -24 };
	oski_value_t alpha = 1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_684, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_684", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_684() */
/* END TEST 684 */

/*
 *  BEGIN TEST 685
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_685( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	const oski_value_t y_true[] = {1, -3, -58, -39, 32, 0, 33, -14, 28, 16, 73, -72, 70, 114, -6 };
	oski_value_t alpha = -1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_685, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_685", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_685() */
/* END TEST 685 */

/*
 *  BEGIN TEST 686
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_686( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	const oski_value_t y_true[] = {-14, -18, -8, 4, -6, -12, -20, 10, -18, -10, 12, -6, -8, -6, 20 };
	oski_value_t alpha = 0;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_686, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_686", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_686() */
/* END TEST 686 */

/*
 *  BEGIN TEST 687
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_ROWMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_687( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_ROWMAJ, x672_VW );
	const oski_value_t y_true[] = {-209, -303, -588, -250, 163, -186, -46, 43, -55, -27, 770, -669, 436, 819, 262 };
	oski_value_t alpha = -8;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_687, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_687", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_687() */
/* END TEST 687 */

/*
 *  BEGIN TEST 688
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_688( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	const oski_value_t y_true[] = {-15, 25, -50, -53, 41, -17, 38, 40, 40, 38, 149, -30, 94, 72, -30 };
	oski_value_t alpha = 1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_688, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_688", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_688() */
/* END TEST 688 */

/*
 *  BEGIN TEST 689
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_689( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	const oski_value_t y_true[] = {40, 10, 40, 23, -26, 37, -3, -5, 10, -33, -109, 0, -104, -122, 35 };
	oski_value_t alpha = -1;
	oski_value_t beta = 0;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_689, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_689", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_689() */
/* END TEST 689 */

/*
 *  BEGIN TEST 690
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_690( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	const oski_value_t y_true[] = {-10, -14, 4, 12, -6, -8, -14, -14, -20, -2, -16, 12, 4, 20, -2 };
	oski_value_t alpha = 0;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_690, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_690", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_690() */
/* END TEST 690 */

/*
 *  BEGIN TEST 691
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_691( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	const oski_value_t y_true[] = {10, -32, 52, 59, -44, 13, -45, -47, -50, -39, -157, 36, -92, -62, 29 };
	oski_value_t alpha = -1;
	oski_value_t beta = -6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_691, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_691", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_691() */
/* END TEST 691 */

/*
 *  BEGIN TEST 692
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_692( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	const oski_value_t y_true[] = {49, -41, -10, 30, -45, 152, -13, 25, -57, -128, -12, 23, -13, 8, 42 };
	oski_value_t alpha = 1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_692, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_692", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_692() */
/* END TEST 692 */

/*
 *  BEGIN TEST 693
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_693( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	const oski_value_t y_true[] = {-14, 86, 30, -40, 60, -122, 63, -50, 102, 153, -18, -8, 33, 7, -92 };
	oski_value_t alpha = -1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_693, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_693", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_693() */
/* END TEST 693 */

/*
 *  BEGIN TEST 694
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_694( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	const oski_value_t y_true[] = {-21, -27, -12, 6, -9, -18, -30, 15, -27, -15, 18, -9, -12, -9, 30 };
	oski_value_t alpha = 0;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_694, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_694", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_694() */
/* END TEST 694 */

/*
 *  BEGIN TEST 695
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_695( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_ROWMAJ, y672_VW );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	const oski_value_t y_true[] = {-392, 120, 4, -152, 228, -996, -62, -80, 216, 698, 156, -180, 22, -90, -112 };
	oski_value_t alpha = -6;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_695, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_695", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_695() */
/* END TEST 695 */

/*
 *  BEGIN TEST 696
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_696( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	const oski_value_t y_true[] = {0, 33, -13, -15, 128, 22, 82, 61, 189, 130, 24, -85, -26, -80, -27 };
	oski_value_t alpha = 1;
	oski_value_t beta = 8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_696, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_696", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_696() */
/* END TEST 696 */

/*
 *  BEGIN TEST 697
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_697( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	const oski_value_t y_true[] = {60, 51, -11, -57, -92, 26, 2, 23, -69, -118, 72, 13, 2, -40, 39 };
	oski_value_t alpha = -1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_697, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_697", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_697() */
/* END TEST 697 */

/*
 *  BEGIN TEST 698
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_698( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	const oski_value_t y_true[] = {30, 42, -12, -36, 18, 24, 42, 42, 60, 6, 48, -36, -12, -60, 6 };
	oski_value_t alpha = 0;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_698, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_698", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_698() */
/* END TEST 698 */

/*
 *  BEGIN TEST 699
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_699( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	oski_value_t* y_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	const oski_value_t y_true[] = {145, 71, -6, -114, -425, 28, -125, -41, -466, -491, 136, 166, 46, 30, 137 };
	oski_value_t alpha = -4;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_699, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_699", y_true, y_val, y672_LEN*y672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_699() */
/* END TEST 699 */

/*
 *  BEGIN TEST 700
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_700( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	const oski_value_t y_true[] = {-1, 6, -15, 5, -67, 12, -53, 14, -118, -120, 88, 9, -13, 72, 6 };
	oski_value_t alpha = 1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_700, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_700", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_700() */
/* END TEST 700 */

/*
 *  BEGIN TEST 701
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_701( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	const oski_value_t y_true[] = {-62, -87, -21, 13, 40, -66, -37, 31, 37, 75, -34, -36, -23, -99, 84 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_701, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_701", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_701() */
/* END TEST 701 */

/*
 *  BEGIN TEST 702
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_702( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	const oski_value_t y_true[] = {7, 9, 4, -2, 3, 6, 10, -5, 9, 5, -6, 3, 4, 3, -10 };
	oski_value_t alpha = 0;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_702, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_702", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_702() */
/* END TEST 702 */

/*
 *  BEGIN TEST 703
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x672', LAYOUT_COLMAJ>
 *  y : <'y672', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_703( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A672_ptr, A672_ind, A672_val,
		A672_M, A672_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y672_val,
		y672_LEN * y672_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y672_LEN, y672_VW, LAYOUT_COLMAJ, y672_LEN );
	oski_value_t* y_val = array_Duplicate( x672_val,
		x672_LEN * x672_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x672_LEN, x672_VW, LAYOUT_COLMAJ, x672_LEN );
	const oski_value_t y_true[] = {-13, -24, 7, -1, 61, -24, 33, -4, 100, 110, -76, -15, 5, -78, 14 };
	oski_value_t alpha = -1;
	oski_value_t beta = -1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_703, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_703", y_true, y_val, x672_LEN*x672_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_703() */
/* END TEST 703 */

#define A704_M 5    /* # of rows */
#define A704_N 5    /* # of cols */
#define A704_NNZ 5  /* # of stored non-zeros */
static oski_index_t A704_ptr[A704_M+1] = {1, 1, 1, 3, 5, 6}; /* end row pointers for A704 */
static oski_index_t A704_ind[A704_NNZ] = {1, 2, 1, 3, 4}; /* end column indices pointers for A704 */
static oski_value_t A704_val[A704_NNZ] = {-7, -10, -4, -7, -8}; /* end stored non-zero values for A704 */

#define x704_LEN 5   /* vector length */
#define x704_VW  3   /* vector width */
static oski_value_t x704_val[x704_LEN*x704_VW] = {-6, 1, -3, 6, 6, 9, 1, 8, -6, 0, -5, -10, 2, 10, -3}; /* data */

#define y704_LEN 5   /* vector length */
#define y704_VW  3   /* vector width */
static oski_value_t y704_val[y704_LEN*y704_VW] = {-1, 4, -7, 2, -3, 7, -2, -9, 7, -8, 9, 8, 7, -10, 10}; /* data */

/*
 *  BEGIN TEST 704
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_704( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	const oski_value_t y_true[] = {-9, -51, 107, -12, -62, 41, -9, 12, -33, 33, -181, 36, -26, 90, 37 };
	oski_value_t alpha = 1;
	oski_value_t beta = -4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_704, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_704", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_704() */
/* END TEST 704 */

/*
 *  BEGIN TEST 705
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_705( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	const oski_value_t y_true[] = {9, 51, -107, 12, 62, -41, 9, -12, 33, -33, 181, -36, 26, -90, -37 };
	oski_value_t alpha = -1;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_705, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_705", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_705() */
/* END TEST 705 */

/*
 *  BEGIN TEST 706
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_706( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	const oski_value_t y_true[] = {7, -28, 49, -14, 21, -49, 14, 63, -49, 56, -63, -56, -49, 70, -70 };
	oski_value_t alpha = 0;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_706, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_706", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_706() */
/* END TEST 706 */

/*
 *  BEGIN TEST 707
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_707( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	const oski_value_t y_true[] = {-83, -190, 439, -14, -459, 449, -112, -189, 5, -34, -825, 448, 47, 250, 512 };
	oski_value_t alpha = 6;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_707, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_707", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_707() */
/* END TEST 707 */

/*
 *  BEGIN TEST 708
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_708( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	const oski_value_t y_true[] = {99, 22, -61, -32, 33, -144, 32, -142, -16, -46, 181, -3, 53, -172, -27 };
	oski_value_t alpha = 1;
	oski_value_t beta = -9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_708, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_708", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_708() */
/* END TEST 708 */

/*
 *  BEGIN TEST 709
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_709( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	const oski_value_t y_true[] = {-75, -26, 73, 8, -57, 108, -36, 110, 40, 46, -161, 43, -61, 132, 39 };
	oski_value_t alpha = -1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_709, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_709", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_709() */
/* END TEST 709 */

/*
 *  BEGIN TEST 710
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_710( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	const oski_value_t y_true[] = {-12, 2, -6, 12, 12, 18, 2, 16, -12, 0, -10, -20, 4, 20, -6 };
	oski_value_t alpha = 0;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_710, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_710", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_710() */
/* END TEST 710 */

/*
 *  BEGIN TEST 711
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_711( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	const oski_value_t y_true[] = {-411, -278, 789, -192, -777, 576, -368, 638, 624, 414, -1229, 827, -637, 748, 483 };
	oski_value_t alpha = -9;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_711, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_711", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_711() */
/* END TEST 711 */

/*
 *  BEGIN TEST 712
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_712( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	const oski_value_t y_true[] = {-15, 4, -31, 5, -4, -21, -78, -42, -131, 34, 97, 85, 9, 48, 97 };
	oski_value_t alpha = 1;
	oski_value_t beta = 2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_712, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_712", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_712() */
/* END TEST 712 */

/*
 *  BEGIN TEST 713
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_713( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	const oski_value_t y_true[] = {21, -28, 73, -17, 22, -21, 90, 96, 89, 14, -151, -133, -51, 12, -157 };
	oski_value_t alpha = -1;
	oski_value_t beta = -8;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_713, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_713", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_713() */
/* END TEST 713 */

/*
 *  BEGIN TEST 714
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_714( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	const oski_value_t y_true[] = {3, -12, 21, -6, 9, -21, 6, 27, -21, 24, -27, -24, -21, 30, -30 };
	oski_value_t alpha = 0;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_714, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_714", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_714() */
/* END TEST 714 */

/*
 *  BEGIN TEST 715
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_715( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	const oski_value_t y_true[] = {-10, -16, 4, -5, 11, -56, -68, 3, -166, 74, 52, 45, -26, 98, 47 };
	oski_value_t alpha = 1;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_715, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_715", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_715() */
/* END TEST 715 */

/*
 *  BEGIN TEST 716
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_716( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	const oski_value_t y_true[] = {22, 45, -9, 92, 106, -35, -51, -63, -84, 79, 91, -205, -13, -34, 81 };
	oski_value_t alpha = 1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_716, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_716", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_716() */
/* END TEST 716 */

/*
 *  BEGIN TEST 717
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_717( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	const oski_value_t y_true[] = {-76, -36, -18, -38, -52, 116, 60, 135, 30, -79, -136, 115, 31, 124, -108 };
	oski_value_t alpha = -1;
	oski_value_t beta = 6;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_717, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_717", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_717() */
/* END TEST 717 */

/*
 *  BEGIN TEST 718
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_718( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	const oski_value_t y_true[] = {-24, 4, -12, 24, 24, 36, 4, 32, -24, 0, -20, -40, 8, 40, -12 };
	oski_value_t alpha = 0;
	oski_value_t beta = 4;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_718, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_718", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_718() */
/* END TEST 718 */

/*
 *  BEGIN TEST 719
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_ROWMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_719( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_ROWMAJ, x704_VW );
	const oski_value_t y_true[] = {142, 171, -9, 314, 370, -221, -213, -324, -282, 316, 409, -730, -70, -226, 351 };
	oski_value_t alpha = 4;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_719, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_719", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_719() */
/* END TEST 719 */

/*
 *  BEGIN TEST 720
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_720( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	const oski_value_t y_true[] = {-2, -51, -10, 17, -58, -79, 1, 40, 18, 59, -161, -16, -91, 118, -153 };
	oski_value_t alpha = 1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_720, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_720", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_720() */
/* END TEST 720 */

/*
 *  BEGIN TEST 721
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_721( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	const oski_value_t y_true[] = {4, 43, 24, -21, 64, 65, 3, -22, -32, -43, 143, 0, 77, -98, 133 };
	oski_value_t alpha = -1;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_721, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_721", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_721() */
/* END TEST 721 */

/*
 *  BEGIN TEST 722
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_722( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	const oski_value_t y_true[] = {3, -12, 21, -6, 9, -21, 6, 27, -21, 24, -27, -24, -21, 30, -30 };
	oski_value_t alpha = 0;
	oski_value_t beta = -3;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_722, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_722", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_722() */
/* END TEST 722 */

/*
 *  BEGIN TEST 723
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_723( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	const oski_value_t y_true[] = {88, 179, 580, -293, 732, 221, 131, 270, -652, 29, 819, -416, 329, -362, 677 };
	oski_value_t alpha = -9;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_723, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_723", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_723() */
/* END TEST 723 */

/*
 *  BEGIN TEST 724
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_724( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	const oski_value_t y_true[] = {39, 23, 38, -40, 77, 40, 88, -62, 130, -82, -93, -73, -68, -83, -57 };
	oski_value_t alpha = 1;
	oski_value_t beta = 1;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_724, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_724", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_724() */
/* END TEST 724 */

/*
 *  BEGIN TEST 725
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_725( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	const oski_value_t y_true[] = {-87, -15, -62, 88, -29, 32, -80, 126, -178, 82, 53, -7, 84, 163, 33 };
	oski_value_t alpha = -1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_725, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_725", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_725() */
/* END TEST 725 */

/*
 *  BEGIN TEST 726
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_726( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	const oski_value_t y_true[] = {30, -5, 15, -30, -30, -45, -5, -40, 30, 0, 25, 50, -10, -50, 15 };
	oski_value_t alpha = 0;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_726, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_726", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_726() */
/* END TEST 726 */

/*
 *  BEGIN TEST 727
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_727( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_ROWMAJ, y704_VW );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	const oski_value_t y_true[] = {120, 39, 97, -122, 112, 17, 169, -180, 302, -164, -151, -76, -150, -236, -93 };
	oski_value_t alpha = 2;
	oski_value_t beta = -5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_727, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_727", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_727() */
/* END TEST 727 */

/*
 *  BEGIN TEST 728
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_728( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	const oski_value_t y_true[] = {-16, 59, -62, 17, -63, 26, -93, -86, -49, -8, 4, 26, 116, -30, -13 };
	oski_value_t alpha = 1;
	oski_value_t beta = 7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_728, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_728", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_728() */
/* END TEST 728 */

/*
 *  BEGIN TEST 729
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_729( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	const oski_value_t y_true[] = {16, -59, 62, -17, 63, -26, 93, 86, 49, 8, -4, -26, -116, 30, 13 };
	oski_value_t alpha = -1;
	oski_value_t beta = -7;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_729, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_729", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_729() */
/* END TEST 729 */

/*
 *  BEGIN TEST 730
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_730( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	const oski_value_t y_true[] = {2, -8, 14, -4, 6, -14, 4, 18, -14, 16, -18, -16, -14, 20, -20 };
	oski_value_t alpha = 0;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_730, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_730", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_730() */
/* END TEST 730 */

/*
 *  BEGIN TEST 731
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_731( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	oski_value_t* y_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	const oski_value_t y_true[] = {-23, 82, -61, 16, -99, -11, -168, -91, -161, 56, -73, -20, 169, 30, -116 };
	oski_value_t alpha = 2;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_731, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_731", y_true, y_val, y704_LEN*y704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_731() */
/* END TEST 731 */

/*
 *  BEGIN TEST 732
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_732( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	const oski_value_t y_true[] = {52, 72, -48, 67, -31, 24, 86, -103, 118, -64, 10, -42, -70, -195, 96 };
	oski_value_t alpha = 1;
	oski_value_t beta = -2;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_732, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_732", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_732() */
/* END TEST 732 */

/*
 *  BEGIN TEST 733
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_733( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	const oski_value_t y_true[] = {-58, -71, 45, -61, 37, -15, -85, 111, -124, 64, -15, 32, 72, 205, -99 };
	oski_value_t alpha = -1;
	oski_value_t beta = 3;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_733, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_733", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_733() */
/* END TEST 733 */

/*
 *  BEGIN TEST 734
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_734( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	const oski_value_t y_true[] = {-30, 5, -15, 30, 30, 45, 5, 40, -30, 0, -25, -50, 10, 50, -15 };
	oski_value_t alpha = 0;
	oski_value_t beta = 5;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_734, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_734", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_734() */
/* END TEST 734 */

/*
 *  BEGIN TEST 735
 *  Value type: real
 *  A : <MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x704', LAYOUT_COLMAJ>
 *  y : <'y704', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_735( void )
{
	oski_matrix_t A = oski_CreateMatCSR( A704_ptr, A704_ind, A704_val,
		A704_M, A704_N, SHARE_INPUTMAT,
		5, MAT_SYMM_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y704_val,
		y704_LEN * y704_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y704_LEN, y704_VW, LAYOUT_COLMAJ, y704_LEN );
	oski_value_t* y_val = array_Duplicate( x704_val,
		x704_LEN * x704_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x704_LEN, x704_VW, LAYOUT_COLMAJ, x704_LEN );
	const oski_value_t y_true[] = {-374, -583, 405, -578, 206, -255, -695, 768, -902, 512, -45, 406, 546, 1490, -747 };
	oski_value_t alpha = -8;
	oski_value_t beta = 9;
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_735, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_735", y_true, y_val, x704_LEN*x704_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Des