/*F***************************************************************************
 * This file is part of openSMILE.
 * 
 * Copyright (c) audEERING GmbH. All rights reserved.
 * See the file COPYING for details on license terms.
 ***************************************************************************E*/

/*

This is the main commandline application

*/

#include <core/smileCommon.hpp>

#include <core/configManager.hpp>
#include <core/commandlineParser.hpp>
#include <core/componentManager.hpp>

#define MODULE "SMILExtract"


/************** Ctrl+C signal handler **/
#include  <signal.h>

cComponentManager *cmanGlob = NULL;

void INThandler(int);
int ctrlc = 0;

void INThandler(int sig)
{
  signal(sig, SIG_IGN);
  if (cmanGlob != NULL) cmanGlob->requestAbort();
  signal(SIGINT, INThandler);
  ctrlc = 1;
}
/*******************************************/


int main(int argc, const char *argv[])
{
  try {

    smileCommon_fixLocaleEnUs();
    smileCommon_enableVTInWindowsConsole();

    // set up the smile logger
    cSmileLogger *logger = new cSmileLogger();
    logger->useForCurrentThread();
    logger->setLogLevel(1);
    logger->setConsoleOutput(true);

    // commandline parser:
    cCommandlineParser cmdline(argc,argv);
    cmdline.addStr("configfile", 'C', "Path to openSMILE config file", "smile.conf");
    cmdline.addInt("loglevel", 'l', "Verbosity level (0-9)", 2);
#ifdef DEBUG
    cmdline.addBoolean("debug", 'd', "Show debug messages", 0);
#endif
    cmdline.addInt("nticks", 't', "Number of ticks to process (-1 = infinite) (only works for single-thread processing, i.e. nThreads=1)", -1);
    cmdline.addBoolean("components", 'L', "Show component list", 0);
    cmdline.addStr("configHelp", 'H', "Show documentation of registered config types (if an argument is given, show only documentation for config types beginning with the name given in the argument)", NULL, 0);
    cmdline.addStr("configDflt", 0, "Show default config section templates for each config type (if an argument is given, show only documentation for config types beginning with the name given in the argument, OR for a list of components in conjunctions with the 'cfgFileTemplate' option enabled)", NULL, 0);
    cmdline.addBoolean("cfgFileTemplate", 0, "Print a complete template config file for a configuration containing the components specified in a comma separated string as argument to the 'configDflt' option", 0);
    cmdline.addBoolean("cfgFileDescriptions", 0, "Include description in config file templates.", 0);
    cmdline.addBoolean("ccmdHelp", 'c', "Show custom commandline option help (those specified in config file)", 0);
    cmdline.addBoolean("exportHelp", 0, "Print detailed documentation of registered config types in JSON format.", 0);
    cmdline.addStr("logfile", 0, "Set path of log file", "smile.log");
    cmdline.addBoolean("nologfile", 0, "Don't create a log file (e.g. on a read-only filesystem)", 0);
    cmdline.addBoolean("noconsoleoutput", 0, "Don't output any messages to the console (log file is not affected by this option)", 0);
    cmdline.addBoolean("appendLogfile", 0, "Append log messages to an existing logfile instead of overwriting the logfile at every start", 0);

    int help = 0;
    if (cmdline.parse() == -1) {
      logger->setLogLevel(0);
      help = 1;
    }
    if (argc <= 1) {
      printf("\nNo commandline options were given.\nPlease run 'SMILExtract -h' to see usage information!\n\n");
      return 10;
    }

    if (help == 1)
      return EXIT_SUCCESS;

    if (cmdline.getBoolean("nologfile")) {
      logger->setLogFile((const char *)NULL,0);
    } else {
      logger->setLogFile(cmdline.getStr("logfile"),cmdline.getBoolean("appendLogfile"));
    }
    logger->setConsoleOutput(!cmdline.getBoolean("noconsoleoutput"));
    logger->setLogLevel(cmdline.getInt("loglevel"));
    SMILE_MSG(2,"openSMILE starting!");

#ifdef DEBUG 
    if (!cmdline.getBoolean("debug"))
      logger->setLogLevel(LOG_DEBUG, 0);
#endif

    SMILE_MSG(2,"config file is: %s", cmdline.getStr("configfile"));

    cConfigManager *configManager = new cConfigManager(&cmdline);
    cComponentManager *cMan = new cComponentManager(configManager,componentlist);

    if (cmdline.isSet("configHelp")) {
#ifndef EXTERNAL_BUILD
      const char *selStr = cmdline.getStr("configHelp");
      configManager->printTypeHelp(1/*!!! -> 1*/,selStr,0);
#endif
      help = 1;
    }
    if (cmdline.isSet("configDflt")) {
#ifndef EXTERNAL_BUILD
      int fullMode = 0; 
      int wDescr = 0;
      if (cmdline.getBoolean("cfgFileTemplate")) fullMode=1;
      if (cmdline.getBoolean("cfgFileDescriptions")) wDescr=1;
      const char *selStr = cmdline.getStr("configDflt");
      configManager->printTypeDfltConfig(selStr,1,fullMode,wDescr);
#endif
      help = 1;
    }
    if (cmdline.getBoolean("exportHelp")) {
#ifndef EXTERNAL_BUILD
      cMan->exportComponentList();
#endif
      help = 1;
    }
    if (cmdline.getBoolean("components")) {
#ifndef EXTERNAL_BUILD
      cMan->printComponentList();
#endif
      help = 1;
    }

    if (help==1) {
      delete configManager;
      delete cMan;
      delete logger;
      return EXIT_ERROR; 
    }

    // TODO: read config here and print ccmdHelp...
    // add the file config reader:
    try { 
      cFileConfigReader * reader = new cFileConfigReader(
          cmdline.getStr("configfile"), -1, &cmdline);
      configManager->addReader(reader);
      configManager->readConfig();
    } catch (const cConfigException&) {
      delete configManager;
      delete cMan;
      delete logger;
      return EXIT_ERROR;
    }

    /* re-parse the command-line to include options created in the config file */
    cmdline.parse(true, false); // warn if unknown options are detected on the commandline
    if (cmdline.getBoolean("ccmdHelp")) {
      cmdline.showUsage();
      delete configManager;
      delete cMan;
      delete logger;
      return EXIT_ERROR;
    }

    try { 
      /* create all instances specified in the config file */
      cMan->createInstances(0); // 0 = do not read config (we already did that above..)
    } catch (const cConfigException&) {
      delete configManager;
      delete cMan;
      delete logger;
      return EXIT_ERROR;
    }

    /*
    MAIN TICK LOOP :
    */
    cmanGlob = cMan;
    signal(SIGINT, INThandler); // install Ctrl+C signal handler

    bool run = true;

    /* run single or multi-threaded, depending on componentManager config in config file */
    long long nTicks = 0;
    if (run) {
      nTicks = cMan->runMultiThreaded(cmdline.getInt("nticks"));
    }

    /* it is important that configManager is deleted BEFORE componentManager! 
      (since component Manager unregisters plugin DLLs, which might have allocated configTypes, etc.) */
    delete configManager;
    delete cMan;
    delete logger;

  } catch (const cSMILException&) { 
    return EXIT_ERROR; 
  } 

  if (ctrlc) return EXIT_CTRLC;
  return EXIT_SUCCESS;
}
