/*
 * CHANNELS - Handles common operations on channels; also contains the code
 *            to process incoming data from them. For outgoing interface
 *            messages, we just use the generic meta_avtomsg() in metaops.c.
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2001/10/29 - EvB - Created
 * 2002/01/07 - EvB - Moved most of chan_*msgtoav's code to metaops.c,
 *		      to make it independent of channels and rings.
 */

char channels_id[] = "CHANNELS - Copyright (C) 2001 Emile van Bergen.";


/*
 * INCLUDES & DEFINES
 */


#include <stdlib.h>	/* For malloc() / free() */
#include <string.h>	/* For memset() */

#include <channels.h>	/* Also includes srvtypes.h */
#include <jobs.h>	/* For job_toiface(), job_tochan(), job_fromchan() */
#include <language.h>	/* For VM. Also includes metaops for META_AV. */

#define DEBUGLEVEL 1	/* Include D1 statements */
#include <debug.h>


/*
 * FUNCTIONS
 */


/* Add an ASCII pair from a ring to an A/V item list - used by
   chan_handle_read below */

static void chan_ascmsgtoav(META_AV **head, META_AV **tail, 
			    CHAN *ch, ssize_t len)
{
	static char line[C_MAX_MSGSIZE + 16];

	/* Take the indicated amount from the ring - we need to copy for
	   getitembyspec and getvalbyspec anyway. If those worked on rings,
	   we could do everything straight on the ring, but now it doesn't 
	   seem worth it to use prttoa, atoip and atoord directly on the 
	   ring, because we still need to copy the other elements. */

	/* Note that 'len' is already bound by proc_new and proc_msgavailable */
	ring_get(ch->proc->r, line, len);
	D1(msg(F_PROC, L_DEBUG, "- got part of ASCII message: '%s'\n", 
	       dbg_cvtstr(line, len)));

	if (meta_ascmsgtoav(ch->iface->c->m, line, len, head, tail, 
			    ch->iface->flags, ch->iface->recvacl) == -1) {
		msg(F_PROC, L_NOTICE, "chan_ascmsgtoav: ERROR: Could not add AV"
				      " pair received from '%s' (pid %d)!\n", 
		    ch->proc->argv[0], ch->proc->pid);
	}
}


/* Add all binary pairs from a ring to a list of A/V items - used by
   chan_handle_read() below */

static void chan_binmsgtoav(META_AV **head, META_AV **tail, 
			    CHAN *ch, ssize_t len)
{
	U_INT32_T binmsg[(C_MAX_MSGSIZE + 16) >> 2];

	/* Note that 'len' is already bound by proc_new and proc_msgavailable */
	ring_get(ch->proc->r, (char *)binmsg, len);
	D1(msg(F_PROC, L_DEBUG, "- got binary message:\n")); 
	D1(if (msg_thresh[F_PROC] >= L_DEBUG) hexdump((char *)binmsg, len));

	if (meta_binmsgtoav(ch->iface->c->m, binmsg, len, head, tail, 
			    ch->iface->flags, ch->iface->recvacl) == -1) {
		msg(F_PROC, L_NOTICE, "chan_binmsgtoav: ERROR: Could not add AV"
				      " pair(s) received from '%s' (pid %d)!\n",
		    ch->proc->argv[0], ch->proc->pid);
	}
}


/* To be called when select says a channel's subprocess' pipe has data */

void chan_handle_read(CHAN *ch, time_t t)
{
	IFACE *i;
	JOB *j;
	ssize_t len;

	/* Transfer as many bytes to the ring as possible */
	proc_handle_read(ch->proc, t);

	/* Loop while we have jobs on the recv. ring and a (partial) message */
	while(!job_ring_empty(ch) && proc_msgavailable(ch->proc, t, &len)) {

		if (ch->iface->flags & C_DV_FLAGS_ASCII) {
			
			/* ASCII: if partial message: decode and resume loop */
			if (len) {
				chan_ascmsgtoav(&ch->reph, &ch->rept, ch, len);
				ring_discard(ch->proc->r, 1);
				continue;
			}

			/* if full message: skip LF */
			ring_discard(ch->proc->r, 1);
			msg(F_PROC, L_DEBUG, "- end of ASCII message - "
					     "continuing job.\n");
		}
		else {
			/* Decode message */
			chan_binmsgtoav(&ch->reph, &ch->rept, ch, len);
			ch->proc->expectedlen = -1;		/* ack */
			msg(F_PROC, L_DEBUG, "- end of binary message - "
					     "continuing job.\n");
		}

		/* We have a full answer. Take the job from the channel's ring 
		   and run it until it ends or makes another interface call. */
		j = job_fromchan(ch, t);
		i = job_run(ch->iface->c, j); 
		if (i) job_toiface(i, j, t);
	}

	/* If there are no more jobs on recv. queue, clear receiving state */
	if (job_ring_empty(ch)) {
		ch->proc->state &= ~PRS_RECEIVING;
		if (ch->proc->state == PRS_IDLE) ch->proc->timer = 0;
		msg(F_PROC, L_DEBUG, "- no more jobs waiting for pid %d - cleared receiving state\n", ch->proc->pid);
	}
	else {
		ch->proc->timer = ch->jobring[ch->job_r]->expiry;
		msg(F_PROC, L_DEBUG, "- done receiving for now - next oldest job is %d which expires in %d seconds\n", ch->job_r, ch->proc->timer - t);
	}

	/* No message yet or anymore */
	return;
}


/* To be called when select says a channel's subprocess' pipe has more room */

void chan_handle_write(CHAN *ch, time_t t)
{
	JOB *j;

	for(;;) {
		/* Write as many bytes as we can from the tx ring, set 
		   receiving state if we actually transfered anything, clear 
		   sending state if there's nothing more in the tx ring */
		proc_handle_write(ch->proc, t);

		/* See if the tx'er is idle and the iface has more */
		if ((ch->proc->state == PRS_IDLE ||
		     ch->proc->state == PRS_RECEIVING) && 
		    (j=ch->iface->sendqt)) {

			/* It has. Take the job from the shared send queue */
			ch->iface->sendqt = j->next;
			if (ch->iface->sendqt) ch->iface->sendqt->prev = 0;
			else ch->iface->sendqh = 0;
			ch->iface->sendqlen--;

			msg(F_PROC, L_DEBUG, "- tx idle, took another job from "
					     "shared send queue (%d left)\n",
			    ch->iface->sendqlen);

			/* Give it to this channel and loop */
			job_tochan(ch, j, t);
			continue;
		}
		return;
	}
}


void chan_flush(CHAN *ch)
{
	int n;

	for(n = ch->job_r; n != ch->job_w; n = (n + 1) % C_CHAN_JOBRINGSIZE) {
		if (!ch->jobring[n]) continue;
		job_del(ch->jobring[n]); 
		ch->jobring[n] = 0;
		msg(F_PROC, L_DEBUG, "chan_flush: removing job %d from receive queue\n", n);
	}
	ch->job_r = ch->job_w;

	if (ch->reph) {
		meta_freeavlist(ch->reph);
		ch->reph = 0;
	}
	ch->rept = 0;
}
