/*
 * Copyright (c) 2010-2012 Frank Morgner and Dominik Oepen
 *
 * This file is part of OpenPACE.
 *
 * OpenPACE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * OpenPACE is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * OpenPACE.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Additional permission under GNU GPL version 3 section 7
 *
 * If you modify this Program, or any covered work, by linking or combining it
 * with OpenSSL (or a modified version of that library), containing
 * parts covered by the terms of OpenSSL's license, the licensors of
 * this Program grant you additional permission to convey the resulting work.
 * Corresponding Source for a non-source form of such a combination shall include
 * the source code for the parts of OpenSSL used as well as that of the
 * covered work.
 *
 * If you modify this Program, or any covered work, by linking or combining it
 * with OpenSC (or a modified version of that library), containing
 * parts covered by the terms of OpenSC's license, the licensors of
 * this Program grant you additional permission to convey the resulting work. 
 * Corresponding Source for a non-source form of such a combination shall include
 * the source code for the parts of OpenSC used as well as that of the
 * covered work.
 */

/**
 * @file eactest.c
 * @brief Tests for the Extended Access Control protocols
 *
 * @author Frank Morgner <frankmorgner@gmail.com>
 * @author Dominik Oepen <oepen@informatik.hu-berlin.de>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "eac_util.h"
#include "misc.h"
#include "ssl_compat.h"
#include <eac/ca.h>
#include <eac/cv_cert.h>
#include <eac/eac.h>
#include <eac/objects.h>
#include <eac/pace.h>
#include <eac/ri.h>
#include <eac/ta.h>
#include <openssl/asn1.h>
#include <openssl/bio.h>
#include <openssl/buffer.h>
#include <openssl/conf.h>
#include <openssl/crypto.h>
#include <openssl/dh.h>
#include <openssl/ec.h>
#include <openssl/err.h>
#include <openssl/evp.h>
#include <openssl/rand.h>
#include <openssl/x509.h>
#include <stdio.h>
#include <string.h>

#define CHECK(reportverbose, test, message) do { \
    if (test) { \
        if (!verbose) \
            printf("."); \
        else if (debug || (verbose && reportverbose)) { \
            printf("    %-65s [  OK ]\n", message); \
        } \
    } else { \
        if (verbose) { \
            printf("    %-65s [ ERR ]\n", message); \
        } else { \
            printf("\n"); \
        } \
        printf("    Error in %s:%d:%s\n", __FILE__, __LINE__, __FUNCTION__); \
        if (debug) { \
            ERR_load_crypto_strings(); ERR_print_errors_fp(stdout); ERR_free_strings(); \
        } \
        goto err; \
    } \
} while (0)
#define OK do { if (!verbose) printf(" ok\n"); } while (0)
#define TESTEND do { if (!verbose && !failed) printf(" ok\n"); } while (0)
#define END do { TESTEND; return failed; } while (0)

static char verbose=0;
static char debug=0;

/******************************* Test data ************************************/

struct encryption_decryption {
    unsigned int ssc;
    BUF_MEM data;
    BUF_MEM cipher;
};

struct authenticate {
    unsigned int ssc;
    BUF_MEM data;
    BUF_MEM mac;
};

struct eac_worked_example {
    const BUF_MEM ef_cardaccess;
    const BUF_MEM ef_cardsecurity;
    const int pace_info_oid;
    const int pace_version;
    const int pace_curve;
    const char *password;
    const enum s_type password_type;
    const BUF_MEM pace_nonce;
    const BUF_MEM pace_enc_nonce;
    const BUF_MEM pace_static_pcd_priv_key;
    const BUF_MEM pace_static_pcd_pub_key;
    const BUF_MEM pace_static_picc_priv_key;
    const BUF_MEM pace_static_picc_pub_key;
    const BUF_MEM pace_shared_secret_h;
    const BUF_MEM pace_eph_generator;
    const BUF_MEM pace_eph_pcd_priv_key;
    const BUF_MEM pace_eph_pcd_pub_key;
    const BUF_MEM pace_eph_picc_priv_key;
    const BUF_MEM pace_eph_picc_pub_key;
    const BUF_MEM pace_shared_secret_k;
    const BUF_MEM pace_k_mac;
    const BUF_MEM pace_k_enc;
    const BUF_MEM pace_authentication_token_picc;
    const BUF_MEM pace_authentication_token_pcd;
    const struct encryption_decryption *pace_encrypt_decrypt;
    const size_t pace_encrypt_decrypt_len;
    const struct authenticate *pace_authenticate;
    const size_t pace_authenticate_len;
    const int ca_curve;
    const int ca_version;
    const int ca_info_oid;
    const BUF_MEM ca_picc_priv_key;
    const BUF_MEM ca_picc_pub_key;
    const BUF_MEM ca_pcd_priv_key;
    const BUF_MEM ca_pcd_pub_key;
    const BUF_MEM ca_nonce;
    const BUF_MEM ca_picc_token;
    const BUF_MEM ca_shared_secret_k;
    const BUF_MEM ca_k_mac;
    const BUF_MEM ca_k_enc;
    const int ta_curve;
    const BUF_MEM ta_pcd_key;
    const BUF_MEM ta_nonce;
    const BUF_MEM ta_auxdata;
    const BUF_MEM ta_pcd_signature;
    const BUF_MEM ta_cert;
    const BUF_MEM ta_cvca;
    const BUF_MEM ta_dv_cert;
};


static char tc_ecdh_ef_cardaccess[] = {
    0x31, 0x81, 0xC6, 0x30, 0x0D, 0x06, 0x08, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02,  /* 1..0............ */
    0x01, 0x02, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, 0x02, 0x02,  /* ..0............. */
    0x02, 0x01, 0x02, 0x02, 0x01, 0x01, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02,  /* ......0......... */
    0x02, 0x04, 0x02, 0x02, 0x02, 0x01, 0x02, 0x02, 0x01, 0x0D, 0x30, 0x1C, 0x06, 0x09, 0x04, 0x00,  /* ..........0..... */
    0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, 0x02, 0x30, 0x0C, 0x06, 0x07, 0x04, 0x00, 0x7F, 0x00, 0x07,  /* .......0........ */
    0x01, 0x02, 0x02, 0x01, 0x0D, 0x02, 0x01, 0x01, 0x30, 0x2F, 0x06, 0x08, 0x04, 0x00, 0x7F, 0x00,  /* ........0/...... */
    0x07, 0x02, 0x02, 0x06, 0x16, 0x23, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3A, 0x2F, 0x2F, 0x77, 0x77,  /* .....#https://ww */
    0x77, 0x2E, 0x68, 0x6A, 0x70, 0x2D, 0x63, 0x6F, 0x6E, 0x73, 0x75, 0x6C, 0x74, 0x69, 0x6E, 0x67,  /* w.hjp-consulting */
    0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x68, 0x6F, 0x6D, 0x65, 0x30, 0x3E, 0x06, 0x08, 0x04, 0x00, 0x7F,  /* .com/home0>..... */
    0x00, 0x07, 0x02, 0x02, 0x08, 0x31, 0x32, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07,  /* .....120........ */
    0x02, 0x02, 0x03, 0x02, 0x02, 0x02, 0x01, 0x02, 0x02, 0x01, 0x02, 0x30, 0x1C, 0x06, 0x09, 0x04,  /* ...........0.... */
    0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, 0x02, 0x30, 0x0C, 0x06, 0x07, 0x04, 0x00, 0x7F, 0x00,  /* ........0....... */
    0x07, 0x01, 0x02, 0x02, 0x01, 0x0D, 0x02, 0x01, 0x02,                                            /* ......... */
};
static char tc_ecdh_ef_cardsecurity[] = {
    0x30, 0x82, 0x07, 0xE7, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x02, 0xA0, /*0.....*.H.......*/
    0x82, 0x07, 0xD8, 0x30, 0x82, 0x07, 0xD4, 0x02, 0x01, 0x03, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09, /*...0......1.0...*/
    0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x82, 0x01, 0x4D, 0x06, /*`.H.e......0..M.*/
    0x08, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x03, 0x02, 0x01, 0xA0, 0x82, 0x01, 0x3F, 0x04, 0x82, 0x01, /*............?...*/
    0x3B, 0x31, 0x82, 0x01, 0x37, 0x30, 0x0D, 0x06, 0x08, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, /*;1..70..........*/
    0x02, 0x02, 0x01, 0x02, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, /*....0...........*/
    0x02, 0x02, 0x02, 0x01, 0x02, 0x02, 0x01, 0x01, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, /*........0.......*/
    0x07, 0x02, 0x02, 0x04, 0x02, 0x02, 0x02, 0x01, 0x02, 0x02, 0x01, 0x0D, 0x30, 0x1C, 0x06, 0x09, /*............0...*/
    0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, 0x02, 0x30, 0x0C, 0x06, 0x07, 0x04, 0x00, 0x7F, /*.........0......*/
    0x00, 0x07, 0x01, 0x02, 0x02, 0x01, 0x0D, 0x02, 0x01, 0x01, 0x30, 0x2F, 0x06, 0x08, 0x04, 0x00, /*..........0/....*/
    0x7F, 0x00, 0x07, 0x02, 0x02, 0x06, 0x16, 0x23, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3A, 0x2F, 0x2F, /*.......#https://*/
    0x77, 0x77, 0x77, 0x2E, 0x68, 0x6A, 0x70, 0x2D, 0x63, 0x6F, 0x6E, 0x73, 0x75, 0x6C, 0x74, 0x69, /*www.hjp-consulti*/
    0x6E, 0x67, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x68, 0x6F, 0x6D, 0x65, 0x30, 0x17, 0x06, 0x0A, 0x04, /*ng.com/home0....*/
    0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x05, 0x02, 0x03, 0x30, 0x09, 0x02, 0x01, 0x01, 0x02,
    0x01, 0x01, 0x01, 0x01, 0x00, 0x30, 0x17, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, /*.....0..........*/
    0x05, 0x02, 0x03, 0x30, 0x09, 0x02, 0x01, 0x01, 0x02, 0x01, 0x02, 0x01, 0x01, 0xFF, 0x30, 0x19, /*...0..........0.*/
    0x06, 0x09, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x05, 0x02, 0x30, 0x0C, 0x06, 0x07, 0x04, /*...........0....*/
    0x00, 0x7F, 0x00, 0x07, 0x01, 0x02, 0x02, 0x01, 0x0D, 0x30, 0x62, 0x06, 0x09, 0x04, 0x00, 0x7F, /*.........0b.....*/
    0x00, 0x07, 0x02, 0x02, 0x01, 0x02, 0x30, 0x52, 0x30, 0x0C, 0x06, 0x07, 0x04, 0x00, 0x7F, 0x00, /*......0R0.......*/
    0x07, 0x01, 0x02, 0x02, 0x01, 0x0D, 0x03, 0x42, 0x00, 0x04, 0xA4, 0x4E, 0xBE, 0x54, 0x51, 0xDF, /*.......B...N.TQ.*/
    0x7A, 0xAD, 0xB0, 0x1E, 0x45, 0x9B, 0x8C, 0x92, 0x8A, 0x87, 0x74, 0x6A, 0x57, 0x92, 0x7C, 0x8C, /*z...E.....tjW.|.*/
    0x28, 0xA6, 0x77, 0x5C, 0x97, 0xA7, 0xE1, 0xFE, 0x8D, 0x9A, 0x46, 0xFF, 0x4A, 0x1C, 0xC7, 0xE4, /*(.w\......F.J...*/
    0xD1, 0x38, 0x9A, 0xEA, 0x19, 0x75, 0x8E, 0x4F, 0x75, 0xC2, 0x8C, 0x59, 0x8F, 0xD7, 0x34, 0xAE, /*.8...u.Ou..Y..4.*/
    0xBE, 0xB1, 0x35, 0x33, 0x7C, 0xF9, 0x5B, 0xE1, 0x2E, 0x94, 0x02, 0x01, 0x01, 0xA0, 0x82, 0x04, /*..53|.[.........*/
    0x62, 0x30, 0x82, 0x04, 0x5E, 0x30, 0x82, 0x02, 0x92, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x03, /*b0..^0..........*/
    0x01, 0x63, 0x26, 0x30, 0x41, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0A, /*.c&0A..*.H......*/
    0x30, 0x34, 0xA0, 0x0F, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, /*04..0...`.H.e...*/
    0x01, 0x05, 0x00, 0xA1, 0x1C, 0x30, 0x1A, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,
    0x01, 0x01, 0x08, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, /*...0...`.H.e....*/
    0x05, 0x00, 0xA2, 0x03, 0x02, 0x01, 0x20, 0x30, 0x53, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, /*...... 0S1.0...U*/
    0x04, 0x06, 0x13, 0x02, 0x44, 0x45, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, /*....DE1.0...U...*/
    0x0E, 0x48, 0x4A, 0x50, 0x20, 0x43, 0x6F, 0x6E, 0x73, 0x75, 0x6C, 0x74, 0x69, 0x6E, 0x67, 0x31, /*.HJP Consulting1*/
    0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x0E, 0x43, 0x6F, 0x75, 0x6E, 0x74, 0x72, /*.0...U....Countr*/
    0x79, 0x20, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, /*y Signer1.0...U.*/
    0x03, 0x0C, 0x09, 0x48, 0x4A, 0x50, 0x20, 0x50, 0x42, 0x20, 0x43, 0x53, 0x30, 0x1E, 0x17, 0x0D, /*...HJP PB CS0...*/
    0x30, 0x39, 0x30, 0x39, 0x31, 0x38, 0x30, 0x37, 0x35, 0x39, 0x35, 0x33, 0x5A, 0x17, 0x0D, 0x31, /*090918075953Z..1*/
    0x30, 0x30, 0x39, 0x31, 0x33, 0x30, 0x37, 0x35, 0x39, 0x35, 0x33, 0x5A, 0x30, 0x54, 0x31, 0x0B, /*00913075953Z0T1.*/
    0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x44, 0x45, 0x31, 0x17, 0x30, 0x15, 0x06, /*0...U....DE1.0..*/
    0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0E, 0x48, 0x4A, 0x50, 0x20, 0x43, 0x6F, 0x6E, 0x73, 0x75, 0x6C, /*.U....HJP Consul*/
    0x74, 0x69, 0x6E, 0x67, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x0F, 0x44, /*ting1.0...U....D*/
    0x6F, 0x63, 0x75, 0x6D, 0x65, 0x6E, 0x74, 0x20, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x31, 0x12, /*ocument Signer1.*/
    0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x09, 0x48, 0x4A, 0x50, 0x20, 0x50, 0x42,
    0x20, 0x44, 0x53, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, /* DS0.."0...*.H..*/
    0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30, 0x82, 0x01, 0x0A, 0x02, /*...........0....*/
    0x82, 0x01, 0x01, 0x00, 0xB6, 0xC5, 0xA8, 0xEE, 0x57, 0x30, 0x76, 0x79, 0xE3, 0x64, 0xE3, 0xF7, /*........W0vy.d..*/
    0xE7, 0x76, 0xEA, 0x64, 0x07, 0x4E, 0x9A, 0x72, 0xF6, 0xB3, 0x76, 0xC2, 0xD2, 0x31, 0x89, 0x63, /*.v.d.N.r..v..1.c*/
    0x1C, 0x10, 0xBA, 0x65, 0xEA, 0x34, 0x6F, 0xEF, 0x70, 0x3B, 0x52, 0xEF, 0x75, 0x93, 0xD4, 0x96, /*...e.4o.p;R.u...*/
    0xE1, 0x50, 0x0F, 0x71, 0x64, 0xD0, 0x38, 0xE9, 0xA8, 0x80, 0xD0, 0x6E, 0x90, 0xFC, 0xF9, 0x6F, /*.P.qd.8....n...o*/
    0xAD, 0x5B, 0x60, 0x68, 0xB3, 0x42, 0xCC, 0xA8, 0x24, 0x77, 0x0B, 0xAD, 0xF1, 0x42, 0x9E, 0xBB, /*.[`h.B..$w...B..*/
    0xDB, 0x97, 0x88, 0x0A, 0xAE, 0xA4, 0x31, 0x14, 0x62, 0xCA, 0xCE, 0x40, 0xAA, 0xF2, 0x24, 0x78, /*......1.b..@..$x*/
    0x35, 0xAB, 0xC2, 0x59, 0x1E, 0x18, 0x80, 0xAD, 0xD9, 0xFD, 0x35, 0xF2, 0xC0, 0xE4, 0x3C, 0xC6, /*5..Y......5...<.*/
    0xFE, 0xB9, 0x1B, 0x0F, 0x13, 0x7C, 0xC4, 0x2A, 0xD8, 0x34, 0x73, 0x24, 0x93, 0xFD, 0x63, 0xF7, /*.....|.*.4s$..c.*/
    0x8F, 0xC7, 0xDA, 0x75, 0xCD, 0xB4, 0xA1, 0xBD, 0x4C, 0x7D, 0xE1, 0xE0, 0x4A, 0xC1, 0xB4, 0xBD, /*...u....L}..J...*/
    0x4C, 0x62, 0xC4, 0x6F, 0x8D, 0x83, 0xEE, 0x6B, 0xF1, 0xAC, 0x24, 0x05, 0xD5, 0xA1, 0x73, 0x77, /*Lb.o...k..$...sw*/
    0x6A, 0x58, 0x96, 0x0A, 0x79, 0xB1, 0xB5, 0xB9, 0x0B, 0x79, 0x7A, 0x4A, 0x7A, 0x19, 0x84, 0x57, /*jX..y....yzJz..W*/
    0xC7, 0xA0, 0x2A, 0x72, 0xA2, 0xFF, 0x9A, 0x32, 0x7E, 0x55, 0x88, 0x19, 0x67, 0x42, 0xC5,
    0x7C, 0x8B, 0x9D, 0x5E, 0xE6, 0x4B, 0x8A, 0x46, 0x00, 0x3B, 0xC5, 0x6D, 0x24, 0x93, 0xC0, 0xA6, /*|..^.K.F.;.m$...*/
    0x58, 0x82, 0x37, 0x94, 0xAB, 0x23, 0x14, 0xBC, 0xF9, 0x79, 0xC5, 0xEE, 0xDF, 0x32, 0x7C, 0x6C, /*X.7..#...y...2|l*/
    0x11, 0x2E, 0x9E, 0xDD, 0x86, 0xC6, 0xE4, 0x19, 0xF9, 0xAD, 0x35, 0xA9, 0x46, 0x56, 0xFD, 0xE7, /*..........5.FV..*/
    0xED, 0x89, 0x6A, 0xF5, 0xC3, 0x46, 0x43, 0x5A, 0xB3, 0xD7, 0xBE, 0xC0, 0xF8, 0xB9, 0x02, 0x56, /*..j..FCZ.......V*/
    0xA3, 0x10, 0x50, 0xB3, 0x97, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x52, 0x30, 0x50, 0x30, 0x1F, /*..P........R0P0.*/
    0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x0D, 0xFD, 0x5C, 0x02, 0x88, /*..U.#..0.....\..*/
    0xBF, 0xEC, 0xE0, 0xB0, 0x7A, 0x5D, 0x40, 0xFF, 0x80, 0xAC, 0x8A, 0x91, 0x74, 0x3A, 0x9B, 0x30, /*....z]@.....t:.0*/
    0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0x91, 0x93, 0xF4, 0xF0, 0xAA, 0x4A, /*...U...........J*/
    0xCA, 0xC0, 0xD3, 0xA1, 0xB6, 0xAC, 0x83, 0xB2, 0x4F, 0x6F, 0xDC, 0x8F, 0xF2, 0x1B, 0x30, 0x0E, /*........Oo....0.*/
    0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x41, /*..U...........0A*/
    0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0A, 0x30, 0x34, 0xA0, 0x0F, 0x30, /*..*.H......04..0*/
    0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0xA1, 0x1C, /*...`.H.e........*/
    0x30, 0x1A, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x08, 0x30, 0x0D, 0x06, /*0...*.H......0..*/
    0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0xA2, 0x03, 0x02,
    0x01, 0x20, 0x03, 0x82, 0x01, 0x81, 0x00, 0xA3, 0xAF, 0xEC, 0x3B, 0xC5, 0xD3, 0x66, 0xE6, 0x61, /*. ........;..f.a*/
    0x19, 0x4A, 0xCA, 0x8D, 0xFC, 0x39, 0x06, 0x76, 0x06, 0x1D, 0x6E, 0x52, 0x30, 0x18, 0xCA, 0x13, /*.J...9.v..nR0...*/
    0x93, 0x0D, 0x79, 0x40, 0xE6, 0xCE, 0x77, 0x86, 0x1D, 0x18, 0xF6, 0x5F, 0x3C, 0xEF, 0x8C, 0xBF, /*..y@..w...._<...*/
    0x44, 0xE8, 0x7D, 0x59, 0xAA, 0xFA, 0x6F, 0x29, 0xEC, 0x57, 0xFB, 0x19, 0xDB, 0x12, 0x30, 0xF0, /*D.}Y..o).W....0.*/
    0xF2, 0xFC, 0x1B, 0xF6, 0x0D, 0x1D, 0x03, 0x96, 0x33, 0x3C, 0x89, 0xA9, 0x2B, 0xF2, 0x31, 0x3C, /*........3<..+.1<*/
    0x43, 0x60, 0xBA, 0xB2, 0x18, 0xDE, 0x57, 0x71, 0x3F, 0x39, 0x0C, 0xA5, 0xBB, 0xB6, 0x99, 0xCD, /*C`....Wq?9......*/
    0x1A, 0x1E, 0x27, 0x3C, 0x61, 0x8B, 0x25, 0xA7, 0xEE, 0xDA, 0xB5, 0xF0, 0xBA, 0xB0, 0x30, 0x65, /*..'<a.%.......0e*/
    0xAA, 0x74, 0x9D, 0x51, 0x32, 0x60, 0xBE, 0x86, 0x7E, 0xB0, 0x11, 0x29, 0x1D, 0xCF, 0x4A, 0xDC, /*.t.Q2`..~..)..J.*/
    0x83, 0x33, 0xF7, 0x78, 0x4F, 0xDD, 0xE8, 0x17, 0x2F, 0x46, 0xC4, 0xF7, 0x90, 0x42, 0x15, 0xFD, /*.3.xO.../F...B..*/
    0xC9, 0x8F, 0x5C, 0xDE, 0x49, 0x16, 0xF0, 0x3E, 0x24, 0x9C, 0xD3, 0x94, 0x07, 0x62, 0xD2, 0xF8, /*..\.I..>$....b..*/
    0xE9, 0x2F, 0x23, 0x17, 0x16, 0xA6, 0xBF, 0x74, 0x2F, 0xED, 0xC2, 0x62, 0x7E, 0x62, 0xF0, 0x46, /*./#....t/..b~b.F*/
    0x95, 0x6D, 0xB9, 0x7B, 0xAA, 0xD2, 0x5C, 0x04, 0x62, 0x47, 0x54, 0xD4, 0xAF, 0x3E, 0x1A, 0x7E, /*.m.{..\.bGT..>.~*/
    0xC4, 0x72, 0x07, 0xCC, 0x08, 0xBD, 0x15, 0x4E, 0x83, 0x9A, 0x43, 0x55, 0xD0, 0x1F, 0x16, 0xDA, /*.r.....N..CU....*/
    0x2C, 0xC1, 0x61, 0x77, 0xA9, 0x14, 0xD4, 0x42, 0x87, 0xE6, 0x52, 0x25, 0x64, 0xD0, 0x00,
    0x53, 0x9E, 0xC9, 0x6A, 0x2B, 0x0E, 0x1E, 0x6E, 0xBB, 0x89, 0x63, 0x81, 0x86, 0x8B, 0x5A, 0xFE, /*S..j+..n..c...Z.*/
    0x0A, 0x0F, 0xD3, 0xC3, 0x62, 0xF4, 0x19, 0xAF, 0xFD, 0xFF, 0x01, 0x6A, 0x71, 0x17, 0x0A, 0xC8, /*....b......jq...*/
    0xB3, 0x78, 0xA6, 0xE3, 0x99, 0x5D, 0x82, 0xEE, 0x45, 0x95, 0x0E, 0xEB, 0xB4, 0xC9, 0xBB, 0xF6, /*.x...]..E.......*/
    0x31, 0x13, 0x24, 0x82, 0xA5, 0x03, 0xC3, 0x10, 0x26, 0xB4, 0xC2, 0xCD, 0x94, 0x26, 0xE6, 0x66, /*1.$.....&....&.f*/
    0x3D, 0xE4, 0xC4, 0x3E, 0xFE, 0x54, 0x01, 0xF4, 0xD3, 0xBA, 0x76, 0xE5, 0x4F, 0x66, 0x3B, 0x28, /*=..>.T....v.Of;(*/
    0x32, 0x3E, 0xA3, 0x33, 0x1E, 0x96, 0xA7, 0x08, 0x12, 0xF9, 0x43, 0x15, 0xD6, 0x08, 0xA9, 0xE8, /*2>.3......C.....*/
    0xCE, 0x1B, 0xF0, 0x2B, 0x6E, 0xCF, 0x07, 0x01, 0x5D, 0x40, 0xF4, 0x73, 0xDF, 0xE1, 0x6F, 0x5C, /*...+n...]@.s..o\*/
    0x12, 0x14, 0x60, 0x81, 0xC4, 0x4C, 0x14, 0x8D, 0xAB, 0x09, 0x83, 0x50, 0x46, 0x57, 0xA5, 0x3C, /*..`..L.....PFW.<*/
    0xCA, 0x16, 0xBD, 0x54, 0x5D, 0x5A, 0xD5, 0x9A, 0x21, 0xAA, 0x91, 0x9E, 0x7F, 0x9B, 0xB7, 0xB3, /*...T]Z..!.......*/
    0x50, 0x01, 0xAB, 0xEF, 0x61, 0xE7, 0xD5, 0x6E, 0x21, 0xC7, 0xF1, 0x13, 0x73, 0x42, 0x55, 0x71, /*P...a..n!...sBUq*/
    0xA7, 0x91, 0x45, 0xD4, 0x46, 0x2E, 0xB2, 0x6B, 0x31, 0x82, 0x02, 0x05, 0x30, 0x82, 0x02, 0x01, /*..E.F..k1...0...*/
    0x02, 0x01, 0x01, 0x30, 0x5A, 0x30, 0x53, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, /*...0Z0S1.0...U..*/
    0x13, 0x02, 0x44, 0x45, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0E, 0x48, /*..DE1.0...U....H*/
    0x4A, 0x50, 0x20, 0x43, 0x6F, 0x6E, 0x73, 0x75, 0x6C, 0x74, 0x69, 0x6E, 0x67, 0x31, 0x17,
    0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x0E, 0x43, 0x6F, 0x75, 0x6E, 0x74, 0x72, 0x79, /*0...U....Country*/
    0x20, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x03, /* Signer1.0...U..*/
    0x0C, 0x09, 0x48, 0x4A, 0x50, 0x20, 0x50, 0x42, 0x20, 0x43, 0x53, 0x02, 0x03, 0x01, 0x63, 0x26, /*..HJP PB CS...c&*/
    0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0xA0, /*0...`.H.e.......*/
    0x4A, 0x30, 0x17, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x03, 0x31, 0x0A, /*J0...*.H......1.*/
    0x06, 0x08, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x03, 0x02, 0x01, 0x30, 0x2F, 0x06, 0x09, 0x2A, 0x86, /*..........0/..*.*/
    0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x04, 0x31, 0x22, 0x04, 0x20, 0x49, 0xAE, 0xB9, 0x37, 0x52, /*H......1". I..7R*/
    0x8C, 0x26, 0x9E, 0xA7, 0x23, 0xBB, 0xC8, 0xAA, 0xDC, 0x38, 0x5C, 0x9D, 0x6B, 0x1A, 0xE3, 0x75, /*.&..#....8\.k..u*/
    0x16, 0xA5, 0xB8, 0x92, 0x1F, 0xF8, 0xC4, 0x59, 0x18, 0x72, 0x93, 0x30, 0x41, 0x06, 0x09, 0x2A, /*.......Y.r.0A..**/
    0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0A, 0x30, 0x34, 0xA0, 0x0F, 0x30, 0x0D, 0x06, 0x09, /*.H......04..0...*/
    0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0xA1, 0x1C, 0x30, 0x1A, 0x06, /*`.H.e........0..*/
    0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x08, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, /*.*.H......0...`.*/
    0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0xA2, 0x03, 0x02, 0x01, 0x20, 0x04, 0x82, /*H.e.......... ..*/
    0x01, 0x00, 0x97, 0xC2, 0x2D, 0x87, 0xC3, 0x13, 0xD6, 0x48, 0xDF, 0xB9, 0xDE, 0x9D, 0x9C,
    0xCA, 0x3A, 0x41, 0xCB, 0xF8, 0x52, 0x22, 0xE3, 0x6D, 0x3B, 0x9C, 0x7E, 0xB1, 0xCC, 0x0B, 0x1A, /*.:A..R".m;.~....*/
    0x8C, 0xAE, 0x4C, 0x93, 0xE0, 0xF1, 0xCA, 0x02, 0x8A, 0x90, 0xDD, 0x2B, 0x4F, 0x5C, 0xB4, 0x2C, /*..L........+O\.,*/
    0x9E, 0x5B, 0xB5, 0x73, 0xC0, 0x39, 0x77, 0x3E, 0x64, 0x08, 0x41, 0xB3, 0x28, 0x30, 0xDF, 0x83, /*.[.s.9w>d.A.(0..*/
    0x93, 0x22, 0x46, 0xFC, 0x8B, 0xAF, 0x92, 0x8D, 0x67, 0x54, 0x6E, 0x8E, 0x0C, 0x06, 0x65, 0xA9, /*."F.....gTn...e.*/
    0x32, 0x87, 0x48, 0x85, 0x51, 0x8E, 0xA3, 0xD0, 0x20, 0x46, 0xA6, 0x18, 0xCF, 0x1A, 0xF5, 0xA0, /*2.H.Q... F......*/
    0xF5, 0xE4, 0xC4, 0x05, 0x62, 0x4D, 0x2D, 0x66, 0xD1, 0x6B, 0xDA, 0x18, 0xA8, 0x38, 0x22, 0x84, /*....bM-f.k...8".*/
    0x78, 0x8E, 0x81, 0xFE, 0x1C, 0xB5, 0xE2, 0x17, 0x01, 0xCD, 0xD2, 0x09, 0x22, 0x12, 0x0E, 0x68, /*x..........."..h*/
    0x20, 0x30, 0xE8, 0x0D, 0x12, 0xDA, 0x40, 0x6B, 0x01, 0x36, 0xE9, 0xED, 0x8B, 0x23, 0x8F, 0x65, /* 0....@k.6...#.e*/
    0x3C, 0x7D, 0xDC, 0xA9, 0x27, 0x86, 0x60, 0x41, 0x4E, 0xFA, 0x93, 0x73, 0x82, 0x50, 0xCD, 0x08, /*<}..'.`AN..s.P..*/
    0x41, 0x72, 0x7E, 0x0F, 0x68, 0xC4, 0x90, 0x02, 0x64, 0x1D, 0x7E, 0x40, 0x26, 0x28, 0x5B, 0x9B, /*Ar~.h...d.~@&([.*/
    0x53, 0xF2, 0x70, 0xBB, 0xA5, 0x05, 0x8E, 0x46, 0x60, 0x0B, 0x84, 0x35, 0x54, 0x60, 0x5B, 0xF8, /*S.p....F`..5T`[.*/
    0xEC, 0x2C, 0x74, 0x0A, 0xED, 0xC8, 0xB2, 0x4E, 0x2A, 0x64, 0xAC, 0x78, 0xF5, 0x89, 0x97, 0xA1, /*.,t....N*d.x....*/
    0x88, 0x33, 0xA4, 0x05, 0xCB, 0x64, 0xEA, 0x6D, 0xD7, 0xD7, 0x11, 0x5F, 0xD7, 0xC3, 0x51, 0x76, /*.3...d.m..._..Qv*/
    0x72, 0x65, 0x4E, 0x03, 0x02, 0x97, 0x30, 0xFA, 0xB7, 0x25, 0x65, 0xA0, 0x92, 0x65, 0x71,
    0x69, 0x68, 0x01, 0xF3, 0xFE, 0x5A, 0x63, 0xCF, 0x70, 0x92, 0x0A, 0x11, 0x2F, 0xCF, 0x69, 0x29, /*ih...Zc.p.../.i)*/
    0x1B, 0xAE, 0x37, 0xA3,
};
static char tc_ecdh_nonce_enc[] = {
    0xCE, 0x83, 0x4C, 0xDE, 0x69, 0xFF, 0xBB, 0x1D, 0x1E, 0xB2, 0x15, 0x85, 0xCD, 0x70, 0x9F, 0x18,
};
static char tc_ecdh_nonce[] = {
    0x7D, 0x98, 0xC0, 0x0F, 0xC6, 0xC9, 0xE9, 0x54, 0x3B, 0xBF, 0x94, 0xA8, 0x70, 0x73, 0xA1, 0x23,
};
static char tc_ecdh_map_pcd_priv_key[] = {
    0x75, 0x22, 0x87, 0xF5, 0xB0, 0x2D, 0xE3, 0xC4, 0xBC, 0x3E, 0x17, 0x94, 0x51, 0x18, 0xC5, 0x1B,  /* u"...-...>..Q... */
    0x23, 0xC9, 0x72, 0x78, 0xE4, 0xCD, 0x74, 0x80, 0x48, 0xAC, 0x56, 0xBA, 0x5B, 0xDC, 0x3D, 0x46,  /* #.rx..t.H.V.[.=F */
};
static char tc_ecdh_map_pcd_pub_key[] = {
    0x04, 0x3D, 0xD2, 0x9B, 0xBE, 0x59, 0x07, 0xFD, 0x21, 0xA1, 0x52, 0xAD, 0xA4, 0x89, 0x5F, 0xAA,  /* .=...Y..!.R..._. */
    0xE7, 0xAC, 0xC5, 0x5F, 0x5E, 0x50, 0xEF, 0xBF, 0xDE, 0x5A, 0xB0, 0xC6, 0xEB, 0x54, 0xF1, 0x98,  /* ..._^P...Z...T.. */
    0xD6, 0x15, 0x91, 0x36, 0x35, 0xF0, 0xFD, 0xF5, 0xBE, 0xB3, 0x83, 0xE0, 0x03, 0x55, 0xF8, 0x2D,  /* ...65........U.- */
    0x3C, 0x41, 0xED, 0x0D, 0xF2, 0xE2, 0x83, 0x63, 0x43, 0x3D, 0xFB, 0x73, 0x85, 0x6A, 0x15, 0xDC,  /* <A.....cC=.s.j.. */
    0x9F,                                                                                            /* . */
};
static char tc_ecdh_map_picc_priv_key[] = {
    0x19, 0xC4, 0x28, 0x71, 0x56, 0x63, 0xDE, 0x74, 0x5D, 0x18, 0x24, 0xB8, 0x55, 0xD2, 0xB9, 0x67,  /* ..(qVc.t].$.U..g */
    0x89, 0x0C, 0x99, 0xD6, 0x8E, 0xD5, 0xFE, 0xEE, 0x9D, 0xCD, 0xF8, 0xD7, 0xBB, 0xA2, 0x89, 0xD2,  /* ................ */
};
static char tc_ecdh_map_picc_pub_key[] = {
    0x04, 0x9C, 0xFC, 0xF7, 0x58, 0x2A, 0xC9, 0x86, 0xD0, 0xDD, 0x52, 0xFA, 0x53, 0x12, 0x34, 0x14,  /* ....X*....R.S.4. */
    0xC3, 0xE1, 0xB9, 0x6B, 0x4D, 0x00, 0xAB, 0xA8, 0xE5, 0x74, 0x67, 0x9B, 0x70, 0xEF, 0xB5, 0xBC,  /* ...kM....tg.p... */
    0x3B, 0x45, 0xD2, 0xF1, 0x37, 0x29, 0xCC, 0x2A, 0xE1, 0x78, 0xE7, 0xE2, 0x41, 0xB4, 0x43, 0x21,  /* ;E..7).*.x..A.C! */
    0x35, 0x33, 0xB7, 0x7D, 0xBB, 0x44, 0x64, 0x9A, 0x81, 0x5D, 0xDC, 0x4A, 0x23, 0x84, 0xBA, 0x42,  /* 53.}.Dd..].J#..B */
    0x2A,                                                                                            /* * */
    };
static char tc_ecdh_map_shared_secret_h[] = {
    0x04, 0x71, 0x85, 0x0C, 0xFD, 0x80, 0xFB, 0x47, 0x59, 0x47, 0xE5, 0xB1, 0xAF, 0x10, 0xFE, 0x8E,  /* .q.....GYG...... */
    0x66, 0x63, 0x96, 0x7C, 0x2D, 0x26, 0x49, 0x35, 0xB3, 0x19, 0x51, 0xF7, 0x63, 0xA4, 0xB0, 0x3A,  /* fc.|-&I5..Q.c..: */
    0x57, 0x49, 0x16, 0x73, 0x88, 0xF8, 0x8F, 0x52, 0xA1, 0x09, 0x16, 0x7E, 0x3E, 0x65, 0x92, 0xCA,  /* WI.s...R...~>e.. */
    0x08, 0x20, 0x46, 0x8D, 0x11, 0x57, 0xA8, 0xE7, 0x81, 0xD2, 0xF7, 0x04, 0x91, 0x79, 0xB1, 0xD1,  /* . F..W.......y.. */
    0x14,                                                                                            /* . */
};
static char tc_ecdh_map_generator[] = {
    0x04, 0x39, 0x29, 0xD2, 0x8B, 0xA1, 0xE5, 0x33, 0x9D, 0x6C, 0x5D, 0xAD, 0xE5, 0xE3, 0x3B, 0xD3,  /* .9)....3.l]...;. */
    0xC2, 0xF0, 0xBD, 0x14, 0xDD, 0x77, 0xC7, 0x52, 0x15, 0x32, 0x26, 0x16, 0x59, 0xC9, 0x18, 0xFA,  /* .....w.R.2&.Y... */
    0x60, 0x14, 0xDD, 0x48, 0xFA, 0x84, 0xE6, 0x2B, 0xDE, 0x43, 0x8E, 0xDB, 0x4C, 0x97, 0x71, 0xD0,  /* `..H...+.C..L.q. */
    0x42, 0xCD, 0xB2, 0x4B, 0x77, 0x88, 0xBD, 0xBA, 0xB2, 0x03, 0x1C, 0x45, 0x75, 0x1E, 0x77, 0x7F,  /* B..Kw......Eu.w. */
    0x66,                                                                                            /* f */
};
static char tc_ecdh_pcd_priv_key[] = {
    0x00, 0x9D, 0x9A, 0x32, 0xDF, 0x93, 0xA5, 0x7C, 0xCE, 0x33, 0xCA, 0x3C, 0xDD, 0x34, 0x57, 0xE3,  /* ...2...|.3.<.4W. */
    0x3A, 0x97, 0x6F, 0x29, 0x35, 0x46, 0xC7, 0x35, 0x50, 0xF3, 0x97, 0x25, 0x9C, 0x93, 0xBE, 0x01,  /* :.o)5F.5P..%.... */
    0x20,
};
static char tc_ecdh_pcd_pub_key[] = {
    0x04, 0x51, 0x8B, 0xC4, 0xE5, 0x32, 0xAD, 0x2A, 0x9B, 0xD6, 0x52, 0x78, 0x04, 0xD5, 0xD6, 0x65,  /* .Q...2.*..Rx...e */
    0xAB, 0xD5, 0x10, 0x41, 0x03, 0x7A, 0x0C, 0xC8, 0xAA, 0x92, 0x28, 0x04, 0xEB, 0x50, 0x1C, 0x22,  /* ...A.z....(..P." */
    0x2B, 0x34, 0x27, 0x38, 0x85, 0x99, 0xAF, 0xAA, 0xE9, 0xFB, 0xAC, 0xE2, 0xDF, 0x93, 0xE1, 0x3C,  /* +4'8...........< */
    0x3C, 0x49, 0x79, 0xCD, 0x12, 0xF0, 0xAE, 0x3E, 0x3C, 0x01, 0x26, 0x02, 0x83, 0x91, 0x55, 0x45,  /* <Iy....><.&...UE */
    0x82,                                                                                            /* . */
};
static char tc_ecdh_picc_priv_key[] = {
    0x15, 0x87, 0x2C, 0x56, 0x90, 0x8C, 0x14, 0x40, 0x02, 0x17, 0x79, 0x94, 0xCF, 0xAA, 0xED, 0xD5,  /* ..,V...@..y..... */
    0x46, 0x7C, 0xE1, 0x50, 0x85, 0x3C, 0x44, 0x53, 0x50, 0x51, 0xFF, 0x24, 0x18, 0x30, 0x39, 0xD8,  /* F|.P.<DSPQ.$.09. */
};
static char tc_ecdh_picc_pub_key[] = {
    0x04, 0x28, 0x2C, 0xF3, 0x80, 0x73, 0x03, 0x6A, 0xFA, 0xC2, 0x16, 0xAF, 0x13, 0x5B, 0xD9, 0x94,  /* .(,..s.j.....[.. */
    0xDA, 0x0C, 0x35, 0x7F, 0x10, 0xBD, 0x4C, 0x34, 0xAF, 0xEA, 0x10, 0x42, 0xB2, 0xEB, 0x0F, 0xD6,  /* ..5...L4...B.... */
    0x80, 0x4D, 0xF3, 0x65, 0x8B, 0x83, 0x5A, 0xC2, 0xE7, 0x13, 0x3F, 0x13, 0x69, 0x11, 0x84, 0x54,  /* .M.e..Z...?.i..T */
    0x2B, 0xB5, 0x0B, 0x10, 0x99, 0x63, 0xA4, 0x66, 0x2A, 0xBD, 0xC0, 0x8B, 0x97, 0x63, 0xAF, 0x4B,  /* +....c.f*....c.K */
    0x5B,                                                                                            /* [ */
};
static char tc_ecdh_shared_secret_k[] = {
    0x6E, 0x7D, 0x07, 0x7C, 0xCD, 0x36, 0x7C, 0x2E, 0xAA, 0x68, 0x3F, 0x1E, 0x8E, 0xC5, 0x34, 0x30,  /* n}.|.6|..h?...40 */
    0x2E, 0x2D, 0x00, 0xB6, 0xAD, 0xAF, 0x8A, 0x87, 0xA6, 0xED, 0xA7, 0x87, 0x40, 0xF1, 0x76, 0x06,  /* .-..........@.v. */
};
/* Note that  Worked Example for Extended Access Control (EAC) Version 1.0
 * contains a wrong key. The Log however contains a dump of the correct value.
 * Fixed in Worked Example for Extended Access Control (EAC) Version 1.01.
 */
static char tc_ecdh_k_mac[] = {
    0x73, 0xFF, 0x26, 0x87, 0x84, 0xF7, 0x2A, 0xF8, 0x33, 0xFD, 0xC9, 0x46, 0x40, 0x49, 0xAF, 0xC9,  /* s.&...*.3..F@I.. */
};
/* Note that  Worked Example for Extended Access Control (EAC) Version 1.0
 * contains a wrong key. The Log however contains a dump of the correct value.
 * Fixed in Worked Example for Extended Access Control (EAC) Version 1.01.
 */
static char tc_ecdh_k_enc[] = {
    0x68, 0x40, 0x6B, 0x41, 0x62, 0x10, 0x05, 0x63, 0xD9, 0xC9, 0x01, 0xA6, 0x15, 0x4D, 0x29, 0x01,  /* h@kAb..c.....M). */
};
static char tc_ecdh_authentication_token_pcd[] = {
    0xA2, 0x7A, 0xE7, 0xB3, 0x65, 0x73, 0xC1, 0xD9,                          /* .z..es.. */
};
static char tc_ecdh_authentication_token_picc[] = {
    0xA2, 0x65, 0x8C, 0x2F, 0x38, 0x60, 0x0B, 0x0F,                          /* .e./8`.. */
};
static char tc_ecdh_d1[] = {
    0x83, 0x0D, 0x44, 0x45, 0x43, 0x56, 0x43, 0x41, 0x41, 0x54, 0x30, 0x30,
    0x30, 0x30, 0x31,
};
static char tc_ecdh_e1[] = {
    0xbe, 0x90, 0x23, 0x7e, 0xeb, 0x4b, 0xa0, 0xff, 0x25, 0x3e, 0xa2, 0x46,
    0xae, 0x31, 0xc8, 0xb8,
};
static const struct encryption_decryption tc_ecdh_enc_dec[] = {
    {
        1,
        { sizeof tc_ecdh_d1, tc_ecdh_d1, sizeof tc_ecdh_d1, },
        { sizeof tc_ecdh_e1, tc_ecdh_e1, sizeof tc_ecdh_e1, },
    },
};
static char tc_ecdh_ad1[] = {
    0x99, 0x02, 0x90, 0x00,
};
static char tc_ecdh_a1[] = {
    0xA8, 0x95, 0x70, 0xA6, 0x86, 0x64, 0xA7, 0xD6,
};
static const struct authenticate tc_ecdh_pace_authenticate[] = {
    {
        2,
        { sizeof tc_ecdh_ad1, tc_ecdh_ad1, sizeof tc_ecdh_ad1, },
        { sizeof tc_ecdh_a1, tc_ecdh_a1, sizeof tc_ecdh_a1, },
    },
};
static char tc_ecdh_ca_picc_priv_key[] = {
    0x79, 0x84, 0x67, 0x4C, 0xF3, 0xB3, 0xA5, 0x24, 0xBF, 0x92, 0x9C, 0xE8, 0xA6, 0x7F, 0xCF, 0x22,  /* y.gL...$......." */
    0x17, 0x3D, 0xA0, 0xBA, 0xD5, 0x95, 0xEE, 0xD6, 0xDE, 0xB7, 0x2D, 0x22, 0xC5, 0x42, 0xFA, 0x9D,  /* .=........-".B.. */
};
static char tc_ecdh_ca_picc_pub_key[] = {
    0x04, 0xA4, 0x4E, 0xBE, 0x54, 0x51, 0xDF, 0x7A, 0xAD, 0xB0, 0x1E, 0x45, 0x9B, 0x8C, 0x92, 0x8A,  /* ..N.TQ.z...E.... */
    0x87, 0x74, 0x6A, 0x57, 0x92, 0x7C, 0x8C, 0x28, 0xA6, 0x77, 0x5C, 0x97, 0xA7, 0xE1, 0xFE, 0x8D,  /* .tjW.|.(.w\..... */
    0x9A, 0x46, 0xFF, 0x4A, 0x1C, 0xC7, 0xE4, 0xD1, 0x38, 0x9A, 0xEA, 0x19, 0x75, 0x8E, 0x4F, 0x75,  /* .F.J....8...u.Ou */
    0xC2, 0x8C, 0x59, 0x8F, 0xD7, 0x34, 0xAE, 0xBE, 0xB1, 0x35, 0x33, 0x7C, 0xF9, 0x5B, 0xE1, 0x2E,  /* ..Y..4...53|.[.. */
    0x94,                                               /* . */
};
static char tc_ecdh_ca_pcd_priv_key[] = {
    0x00, 0xA6, 0xA4, 0xD2, 0x55, 0xC5, 0xBF, 0x7A, 0x77, 0xEC, 0x3D, 0x05, 0x53, 0xDB, 0x74, 0xF6,  /* ....U..zw.=.S.t. */
    0x93, 0xCF, 0x04, 0x4E, 0x18, 0xC9, 0x83, 0x64, 0xD4, 0x97, 0x7A, 0x29, 0x61, 0x08, 0xAF, 0x19,  /* ...N...d..z)a... */
    0xBD,                                                                                            /* . */
};
static char tc_ecdh_ca_pcd_pub_key[] = {
    0x04, 0x5A, 0x7A, 0x37, 0x7F, 0xC9, 0xCA, 0xFC, 0x03, 0xAC, 0x7F, 0xF4, 0x54, 0x41, 0xA8, 0xB2,  /* .Zz7........TA.. */
    0x90, 0x9D, 0x88, 0xEA, 0xB8, 0xE6, 0xB0, 0x17, 0x38, 0x47, 0xAB, 0x49, 0xB9, 0x49, 0xDF, 0x37,  /* ........8G.I.I.7 */
    0x99, 0xA3, 0x4E, 0xE5, 0x7E, 0xC5, 0x52, 0x68, 0xCF, 0x8B, 0x1C, 0x3E, 0xC4, 0x89, 0xF8, 0xBF,  /* /..N.~.Rh...>.... */
    0x4C, 0xF4, 0xC6, 0x8D, 0x3F, 0xD9, 0x67, 0x0E, 0x89, 0xC0, 0xD5, 0xD3, 0xFF, 0xF1, 0xAA, 0xF8,  /* L...?.g......... */
    0x9F,                                                                                            /* . */
};
static char tc_ecdh_ca_nonce[] = {
    0x42, 0x87, 0xB3, 0x07, 0x2A, 0x3E, 0xDC, 0x60,                          /* B...*>.` */
};
static char tc_ecdh_ca_picc_token[] = {
    0xFF, 0x01, 0x17, 0xD6, 0x8D, 0xEE, 0x8E, 0x72,                          /* .......r */
};
static char tc_ecdh_ca_shared_secret_k[] = {
    0x79, 0x1D, 0xA0, 0x42, 0x73, 0xCC, 0xFE, 0x86, 0x2E, 0x52, 0xDF, 0x60, 0x34, 0x7E, 0x25, 0x57,  /* y..Bs....R.`4~%W */
    0x19, 0x2E, 0x1F, 0x8D, 0x75, 0x17, 0x82, 0x2C, 0xE3, 0xD3, 0x06, 0x05, 0x6C, 0x1C, 0xDE, 0xB4,  /* ....u..,....l... */
};
static char tc_ecdh_ca_k_mac[] = {
    0x78, 0xB5, 0x70, 0x9E, 0x7A, 0xBE, 0xDB, 0x18, 0x5B, 0x42, 0x4D, 0x0E, 0xE3, 0xA8, 0x24, 0x99,  /* x.p.z...[BM...$. */
};
static char tc_ecdh_ca_k_enc[] = {
    0x94, 0xAB, 0xCD, 0x27, 0x1A, 0xB7, 0xD9, 0xA5, 0x59, 0x0B, 0xA5, 0x2C, 0xB5, 0x18, 0xB8, 0x31,  /* ...'....Y..,...1 */
};
static char tc_ecdh_ta_pcd_key[] = {
    0x30, 0x82, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x82, 0x01, 0xaf, 0x06, 0x07, 0x2a, 0x86, 0x48,  /* 0......0.....*.H*/
    0xce, 0x3d, 0x02, 0x01, 0x30, 0x82, 0x01, 0xa2, 0x02, 0x01, 0x01, 0x30, 0x4c, 0x06, 0x07, 0x2a,  /* .=..0......0L..**/
    0x86, 0x48, 0xce, 0x3d, 0x01, 0x01, 0x02, 0x41, 0x00, 0xaa, 0xdd, 0x9d, 0xb8, 0xdb, 0xe9, 0xc4,  /* .H.=...A........*/
    0x8b, 0x3f, 0xd4, 0xe6, 0xae, 0x33, 0xc9, 0xfc, 0x07, 0xcb, 0x30, 0x8d, 0xb3, 0xb3, 0xc9, 0xd2,  /* .?...3....0.....*/
    0x0e, 0xd6, 0x63, 0x9c, 0xca, 0x70, 0x33, 0x08, 0x71, 0x7d, 0x4d, 0x9b, 0x00, 0x9b, 0xc6, 0x68,  /* ..c..p3.q}M....h*/
    0x42, 0xae, 0xcd, 0xa1, 0x2a, 0xe6, 0xa3, 0x80, 0xe6, 0x28, 0x81, 0xff, 0x2f, 0x2d, 0x82, 0xc6,  /* B...*....(../-..*/
    0x85, 0x28, 0xaa, 0x60, 0x56, 0x58, 0x3a, 0x48, 0xf3, 0x30, 0x81, 0x84, 0x04, 0x40, 0x78, 0x30,  /* .(.`VX:H.0...@x0*/
    0xa3, 0x31, 0x8b, 0x60, 0x3b, 0x89, 0xe2, 0x32, 0x71, 0x45, 0xac, 0x23, 0x4c, 0xc5, 0x94, 0xcb,  /* .1.`;..2qE.#L...*/
    0xdd, 0x8d, 0x3d, 0xf9, 0x16, 0x10, 0xa8, 0x34, 0x41, 0xca, 0xea, 0x98, 0x63, 0xbc, 0x2d, 0xed,  /* ..=....4A...c.-.*/
    0x5d, 0x5a, 0xa8, 0x25, 0x3a, 0xa1, 0x0a, 0x2e, 0xf1, 0xc9, 0x8b, 0x9a, 0xc8, 0xb5, 0x7f, 0x11,  /* ]Z.%:...........*/
    0x17, 0xa7, 0x2b, 0xf2, 0xc7, 0xb9, 0xe7, 0xc1, 0xac, 0x4d, 0x77, 0xfc, 0x94, 0xca, 0x04, 0x40,  /* ..+......Mw....@*/
    0x3d, 0xf9, 0x16, 0x10, 0xa8, 0x34, 0x41, 0xca, 0xea, 0x98, 0x63, 0xbc, 0x2d, 0xed, 0x5d, 0x5a,  /* =....4A...c.-.]Z*/
    0xa8, 0x25, 0x3a, 0xa1, 0x0a, 0x2e, 0xf1, 0xc9, 0x8b, 0x9a, 0xc8, 0xb5, 0x7f, 0x11, 0x17, 0xa7,  /* .%:.............*/
    0x2b, 0xf2, 0xc7, 0xb9, 0xe7, 0xc1, 0xac, 0x4d, 0x77, 0xfc, 0x94, 0xca, 0xdc, 0x08, 0x3e, 0x67,  /* +......Mw.....>g*/
    0x98, 0x40, 0x50, 0xb7, 0x5e, 0xba, 0xe5, 0xdd, 0x28, 0x09, 0xbd, 0x63, 0x80, 0x16, 0xf7, 0x23,  /* .@P.^...(..c...#*/
    0x04, 0x81, 0x81, 0x04, 0x81, 0xae, 0xe4, 0xbd, 0xd8, 0x2e, 0xd9, 0x64, 0x5a, 0x21, 0x32, 0x2e,  /* ...........dZ!2.*/
    0x9c, 0x4c, 0x6a, 0x93, 0x85, 0xed, 0x9f, 0x70, 0xb5, 0xd9, 0x16, 0xc1, 0xb4, 0x3b, 0x62, 0xee,  /* .Lj....p.....;b.*/
    0xf4, 0xd0, 0x09, 0x8e, 0xff, 0x3b, 0x1f, 0x78, 0xe2, 0xd0, 0xd4, 0x8d, 0x50, 0xd1, 0x68, 0x7b,  /* .....;.x....P.h{*/
    0x93, 0xb9, 0x7d, 0x5f, 0x7c, 0x6d, 0x50, 0x47, 0x40, 0x6a, 0x5e, 0x68, 0x8b, 0x35, 0x22, 0x09,  /* ..}_|mPG@j^h.5".*/
    0xbc, 0xb9, 0xf8, 0x22, 0x7d, 0xde, 0x38, 0x5d, 0x56, 0x63, 0x32, 0xec, 0xc0, 0xea, 0xbf, 0xa9,  /* ..."}.8]Vc2.....*/
    0xcf, 0x78, 0x22, 0xfd, 0xf2, 0x09, 0xf7, 0x00, 0x24, 0xa5, 0x7b, 0x1a, 0xa0, 0x00, 0xc5, 0x5b,  /* .x".....$.{....[*/
    0x88, 0x1f, 0x81, 0x11, 0xb2, 0xdc, 0xde, 0x49, 0x4a, 0x5f, 0x48, 0x5e, 0x5b, 0xca, 0x4b, 0xd8,  /* .......IJ_H^[.K.*/
    0x8a, 0x27, 0x63, 0xae, 0xd1, 0xca, 0x2b, 0x2f, 0xa8, 0xf0, 0x54, 0x06, 0x78, 0xcd, 0x1e, 0x0f,  /* .'c...+/..T.x...*/
    0x3a, 0xd8, 0x08, 0x92, 0x02, 0x41, 0x00, 0xaa, 0xdd, 0x9d, 0xb8, 0xdb, 0xe9, 0xc4, 0x8b, 0x3f,  /* :....A.........?*/
    0xd4, 0xe6, 0xae, 0x33, 0xc9, 0xfc, 0x07, 0xcb, 0x30, 0x8d, 0xb3, 0xb3, 0xc9, 0xd2, 0x0e, 0xd6,  /* ...3....0.......*/
    0x63, 0x9c, 0xca, 0x70, 0x33, 0x08, 0x70, 0x55, 0x3e, 0x5c, 0x41, 0x4c, 0xa9, 0x26, 0x19, 0x41,  /* c..p3.pU>\AL.&.A*/
    0x86, 0x61, 0x19, 0x7f, 0xac, 0x10, 0x47, 0x1d, 0xb1, 0xd3, 0x81, 0x08, 0x5d, 0xda, 0xdd, 0xb5,  /* .a....G.....]...*/
    0x87, 0x96, 0x82, 0x9c, 0xa9, 0x00, 0x69, 0x02, 0x01, 0x01, 0x04, 0x47, 0x30, 0x45, 0x02, 0x01,  /* ......i....G0E..*/
    0x01, 0x04, 0x40, 0x8a, 0xa2, 0x32, 0x7f, 0x9d, 0x38, 0x70, 0xab, 0x1e, 0xbb, 0x62, 0x85, 0x0f,  /* ..@..2..8p...b..*/
    0x85, 0x97, 0x5e, 0xf7, 0xa0, 0x42, 0xdb, 0x92, 0x0b, 0x60, 0x4b, 0x68, 0x8d, 0xc1, 0xda, 0x27,  /* ..^..B...`Kh...'*/
    0xa6, 0x1c, 0x3f, 0x8f, 0xa5, 0xdf, 0xdf, 0x1f, 0x91, 0xd2, 0x26, 0x62, 0x1f, 0xf1, 0xf9, 0x02,  /* ..?.......&b....*/
    0xf1, 0x4d, 0x8c, 0x90, 0x38, 0x26, 0x42, 0x5a, 0xc4, 0x64, 0x10, 0xd8, 0x7f, 0x3d, 0xda, 0xd7,  /* .M..8&BZ.d...=..*/
    0x2b, 0x0d, 0xbe,                                                                                /* +..*/
};
static char tc_ecdh_ta_nonce[] = {
    0x54, 0x7E, 0x4E, 0xAB, 0x03, 0xB2, 0x35, 0xD2,                          /*  T~N...5. */
};
static char tc_ecdh_ta_pcd_signature[] = {
    0x81, 0xA2, 0xD9, 0xE5, 0x89, 0x1B, 0x86, 0xA0, 0xF0, 0x1A, 0xDD, 0x03, 0x41, 0xC8, 0x9D, 0x6E,  /* ............A..n */
    0x9E, 0xF3, 0xE2, 0x0C, 0x22, 0x38, 0x84, 0xF2, 0x7B, 0x2F, 0x20, 0x40, 0xE9, 0x72, 0x1C, 0x46,  /* ...."8..{/ @.r.F */
    0xC9, 0xC3, 0xCF, 0x7C, 0x0E, 0xE0, 0xA4, 0x22, 0x26, 0x3B, 0xA6, 0xF1, 0xEA, 0xFC, 0x69, 0x3B,  /* ...|..."&;....i; */
    0xD8, 0x73, 0x8F, 0x34, 0x76, 0x0B, 0x59, 0x98, 0x58, 0x4F, 0x58, 0xDD, 0x25, 0xB8, 0xD6, 0x0D,  /* .s.4v.Y.XOX.%... */
    0x99, 0xD7, 0xC4, 0x59, 0x57, 0xC2, 0xE2, 0xF1, 0x0C, 0x0E, 0x5F, 0x91, 0xA8, 0xDC, 0x88, 0xBD,  /* ...YW....._..... */
    0x7C, 0xFA, 0x33, 0xCB, 0xF7, 0x84, 0xA9, 0xDA, 0x83, 0xC8, 0x00, 0x32, 0x41, 0x65, 0x17, 0x9D,  /* |.3........2Ae.. */
    0xBE, 0x71, 0xC6, 0x9B, 0x2C, 0x0F, 0xD8, 0x11, 0x27, 0x0E, 0x43, 0x6C, 0x43, 0x27, 0x81, 0x84,  /* .q..,...'.ClC'.. */
    0xDD, 0xB5, 0x42, 0x14, 0x23, 0x42, 0xDB, 0xF1, 0x56, 0xE8, 0x2D, 0x2A, 0x1B, 0xC1, 0x90, 0x85,  /* ..B.#B..V.-*.... */
};
static char tc_ecdh_ta_cert[] = {
    0x7f, 0x21, 0x82, 0x01, 0x66, 0x7f, 0x4e, 0x81, 0xde, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x0d, 0x44,  /* .!..f.N.._)..B.D */
    0x45, 0x54, 0x45, 0x53, 0x54, 0x44, 0x56, 0x44, 0x45, 0x30, 0x31, 0x39, 0x7f, 0x49, 0x81, 0x90,  /* ETESTDVDE019.I.. */
    0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x05, 0x86, 0x81, 0x81, 0x04,  /* ................ */
    0x33, 0x85, 0xb4, 0x84, 0xe8, 0xe9, 0x94, 0xc9, 0x3f, 0x55, 0xad, 0xf4, 0xa9, 0xc9, 0x2d, 0xa3,  /* 3.......?U....-. */
    0xc0, 0x63, 0xac, 0x7d, 0x74, 0x4e, 0x85, 0xd3, 0x8e, 0xe0, 0x70, 0xbf, 0x6f, 0xf8, 0xd7, 0xab,  /* .c.}tN....p.o... */
    0xa2, 0xde, 0x68, 0x87, 0x24, 0xb2, 0x7d, 0x6b, 0xae, 0x2b, 0x1c, 0x8a, 0xe0, 0x74, 0xc0, 0x9b,  /* ..h.$.}k.+...t.. */
    0xb8, 0x80, 0x8c, 0xc8, 0x30, 0x36, 0xb0, 0x69, 0x28, 0x98, 0x71, 0x1d, 0x3f, 0x6f, 0xc3, 0x21,  /* ....06.i(.q.?o.! */
    0x38, 0x46, 0xe1, 0x4b, 0x31, 0x54, 0x84, 0x50, 0x84, 0xdf, 0xbc, 0x0e, 0xb7, 0x0a, 0xe2, 0xbd,  /* 8F.K1T.P........ */
    0xea, 0x8e, 0xeb, 0xd6, 0x79, 0x6a, 0xcf, 0xa4, 0xc5, 0x6f, 0x57, 0x03, 0xce, 0x5a, 0x6e, 0xfe,  /* ....yj...oW..Zn. */
    0x0a, 0x26, 0x6e, 0x43, 0xc3, 0xf7, 0x1a, 0x1b, 0x7e, 0xec, 0x60, 0xe5, 0xc6, 0xf0, 0x85, 0x0d,  /* .&nC....~.`..... */
    0xc4, 0xd3, 0x45, 0x5f, 0x53, 0x28, 0x2c, 0x9f, 0xbc, 0xa8, 0x5d, 0xc9, 0x47, 0xd1, 0x3b, 0xee,  /* ..E_S(,...].G.;. */
    0x5f, 0x20, 0x0d, 0x44, 0x45, 0x54, 0x45, 0x53, 0x54, 0x41, 0x54, 0x44, 0x45, 0x30, 0x31, 0x39,  /* _ .DETESTATDE00x19, */
    0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, 0x02, 0x53, 0x05,  /* .L............S. */
    0x00, 0x00, 0x00, 0x01, 0x10, 0x5f, 0x25, 0x06, 0x01, 0x00, 0x00, 0x09, 0x03, 0x00, 0x5f, 0x24,  /* ....._%......._$ */
    0x06, 0x01, 0x00, 0x01, 0x00, 0x03, 0x00, 0x5f, 0x37, 0x81, 0x80, 0x34, 0xbb, 0x8c, 0x28, 0x0a,  /* ......._7..4..(. */
    0x7a, 0x59, 0x3c, 0xf7, 0xc7, 0xe9, 0xf3, 0x75, 0xff, 0xff, 0xbf, 0xd9, 0x2c, 0x4e, 0x3f, 0x61,  /* zY<....u....,N?a */
    0x88, 0xa4, 0x48, 0x24, 0x21, 0x5e, 0x11, 0xef, 0x47, 0xd0, 0x34, 0x28, 0xcf, 0xc9, 0x19, 0x36,  /* ..H$!^..G.4(...6 */
    0xa2, 0x60, 0x9c, 0x74, 0x2e, 0xf9, 0x2c, 0x59, 0x68, 0xbc, 0xe6, 0x5b, 0xa4, 0x2a, 0xed, 0x7a,  /* .`.t..,Yh..[.*.z */
    0xad, 0x70, 0xb7, 0xb2, 0xa3, 0x12, 0x01, 0xdc, 0x15, 0x2c, 0xec, 0x93, 0x0d, 0x7d, 0x79, 0x54,  /* .p.......,...}yT */
    0xbf, 0xbe, 0x00, 0x21, 0xad, 0x1f, 0xd2, 0xac, 0xaa, 0x6c, 0x34, 0x9a, 0x6c, 0x2f, 0xb8, 0x65,  /* ...!.....l4.l/.e */
    0x14, 0xb9, 0xf0, 0x3d, 0xfa, 0xb9, 0x98, 0x71, 0xfb, 0xd9, 0x90, 0xdd, 0x84, 0x16, 0xd3, 0xed,  /* ...=...q........ */
    0xa2, 0x88, 0x3e, 0xbd, 0x0b, 0x94, 0x40, 0x1a, 0x7d, 0xbf, 0xe1, 0x6e, 0x5a, 0x1d, 0xb6, 0x43,  /* ..>...@.}..nZ..C */
    0xee, 0x7c, 0x2f, 0xc0, 0xd5, 0xd7, 0x7b, 0xae, 0x36, 0x66, 0xaf,                                /* .|/...{.6f. */
};

static char tc_ecdh_cvca_cert[] = {
    0x7f, 0x21, 0x82, 0x02, 0xf3, 0x7f, 0x4e, 0x82, 0x02, 0x6a, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x0d,  /* .!....N..j_)..B. */
    0x44, 0x45, 0x43, 0x56, 0x43, 0x41, 0x41, 0x54, 0x30, 0x30, 0x30, 0x30, 0x31, 0x7f, 0x49, 0x82,  /* DECVCAAT00001.I. */
    0x02, 0x1f, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x05, 0x81, 0x40,  /* ...............@ */
    0xaa, 0xdd, 0x9d, 0xb8, 0xdb, 0xe9, 0xc4, 0x8b, 0x3f, 0xd4, 0xe6, 0xae, 0x33, 0xc9, 0xfc, 0x07,  /* ........?...3... */
    0xcb, 0x30, 0x8d, 0xb3, 0xb3, 0xc9, 0xd2, 0x0e, 0xd6, 0x63, 0x9c, 0xca, 0x70, 0x33, 0x08, 0x71,  /* .0.......c..p3.q */
    0x7d, 0x4d, 0x9b, 0x00, 0x9b, 0xc6, 0x68, 0x42, 0xae, 0xcd, 0xa1, 0x2a, 0xe6, 0xa3, 0x80, 0xe6,  /* }M....hB...*.... */
    0x28, 0x81, 0xff, 0x2f, 0x2d, 0x82, 0xc6, 0x85, 0x28, 0xaa, 0x60, 0x56, 0x58, 0x3a, 0x48, 0xf3,  /* (../-...(.`VX:H. */
    0x82, 0x40, 0x78, 0x30, 0xa3, 0x31, 0x8b, 0x60, 0x3b, 0x89, 0xe2, 0x32, 0x71, 0x45, 0xac, 0x23,  /* .@x0.1.`;..2qE.# */
    0x4c, 0xc5, 0x94, 0xcb, 0xdd, 0x8d, 0x3d, 0xf9, 0x16, 0x10, 0xa8, 0x34, 0x41, 0xca, 0xea, 0x98,  /* L.....=....4A... */
    0x63, 0xbc, 0x2d, 0xed, 0x5d, 0x5a, 0xa8, 0x25, 0x3a, 0xa1, 0x0a, 0x2e, 0xf1, 0xc9, 0x8b, 0x9a,  /* c.-.]Z.%:....... */
    0xc8, 0xb5, 0x7f, 0x11, 0x17, 0xa7, 0x2b, 0xf2, 0xc7, 0xb9, 0xe7, 0xc1, 0xac, 0x4d, 0x77, 0xfc,  /* ......+......Mw. */
    0x94, 0xca, 0x83, 0x40, 0x3d, 0xf9, 0x16, 0x10, 0xa8, 0x34, 0x41, 0xca, 0xea, 0x98, 0x63, 0xbc,  /* ...@=....4A...c. */
    0x2d, 0xed, 0x5d, 0x5a, 0xa8, 0x25, 0x3a, 0xa1, 0x0a, 0x2e, 0xf1, 0xc9, 0x8b, 0x9a, 0xc8, 0xb5,  /* -.]Z.%:......... */
    0x7f, 0x11, 0x17, 0xa7, 0x2b, 0xf2, 0xc7, 0xb9, 0xe7, 0xc1, 0xac, 0x4d, 0x77, 0xfc, 0x94, 0xca,  /* ....+......Mw... */
    0xdc, 0x08, 0x3e, 0x67, 0x98, 0x40, 0x50, 0xb7, 0x5e, 0xba, 0xe5, 0xdd, 0x28, 0x09, 0xbd, 0x63,  /* ..>g.@P.^...(..c */
    0x80, 0x16, 0xf7, 0x23, 0x84, 0x81, 0x81, 0x04, 0x81, 0xae, 0xe4, 0xbd, 0xd8, 0x2e, 0xd9, 0x64,  /* ...#...........d */
    0x5a, 0x21, 0x32, 0x2e, 0x9c, 0x4c, 0x6a, 0x93, 0x85, 0xed, 0x9f, 0x70, 0xb5, 0xd9, 0x16, 0xc1,  /* Z!2..Lj....p.... */
    0xb4, 0x3b, 0x62, 0xee, 0xf4, 0xd0, 0x09, 0x8e, 0xff, 0x3b, 0x1f, 0x78, 0xe2, 0xd0, 0xd4, 0x8d,  /* .;b......;.x.... */
    0x50, 0xd1, 0x68, 0x7b, 0x93, 0xb9, 0x7d, 0x5f, 0x7c, 0x6d, 0x50, 0x47, 0x40, 0x6a, 0x5e, 0x68,  /* P.h{..}_|mPG@j^h */
    0x8b, 0x35, 0x22, 0x09, 0xbc, 0xb9, 0xf8, 0x22, 0x7d, 0xde, 0x38, 0x5d, 0x56, 0x63, 0x32, 0xec,  /* .5"...."}.8]Vc2. */
    0xc0, 0xea, 0xbf, 0xa9, 0xcf, 0x78, 0x22, 0xfd, 0xf2, 0x09, 0xf7, 0x00, 0x24, 0xa5, 0x7b, 0x1a,  /* .....x".....$.{. */
    0xa0, 0x00, 0xc5, 0x5b, 0x88, 0x1f, 0x81, 0x11, 0xb2, 0xdc, 0xde, 0x49, 0x4a, 0x5f, 0x48, 0x5e,  /* ...[.......IJ_H^ */
    0x5b, 0xca, 0x4b, 0xd8, 0x8a, 0x27, 0x63, 0xae, 0xd1, 0xca, 0x2b, 0x2f, 0xa8, 0xf0, 0x54, 0x06,  /* [.K..'c...+/..T. */
    0x78, 0xcd, 0x1e, 0x0f, 0x3a, 0xd8, 0x08, 0x92, 0x85, 0x40, 0xaa, 0xdd, 0x9d, 0xb8, 0xdb, 0xe9,  /* x...:....@...... */
    0xc4, 0x8b, 0x3f, 0xd4, 0xe6, 0xae, 0x33, 0xc9, 0xfc, 0x07, 0xcb, 0x30, 0x8d, 0xb3, 0xb3, 0xc9,  /* ..?...3....0.... */
    0xd2, 0x0e, 0xd6, 0x63, 0x9c, 0xca, 0x70, 0x33, 0x08, 0x70, 0x55, 0x3e, 0x5c, 0x41, 0x4c, 0xa9,  /* ...c..p3.pU>\AL. */
    0x26, 0x19, 0x41, 0x86, 0x61, 0x19, 0x7f, 0xac, 0x10, 0x47, 0x1d, 0xb1, 0xd3, 0x81, 0x08, 0x5d,  /* &.A.a....G.....] */
    0xda, 0xdd, 0xb5, 0x87, 0x96, 0x82, 0x9c, 0xa9, 0x00, 0x69, 0x86, 0x81, 0x81, 0x04, 0x64, 0xf0,  /* .........i....d. */
    0x9c, 0x61, 0x7c, 0x0d, 0x5a, 0x4e, 0x2e, 0x88, 0xb2, 0x59, 0x8a, 0xf0, 0x68, 0x60, 0x44, 0x0c,  /* .a|.ZN...Y..h`D. */
    0x07, 0xc5, 0xed, 0x35, 0x3a, 0x18, 0xa1, 0x4e, 0x93, 0x8a, 0x6c, 0xbc, 0xe3, 0x05, 0x94, 0xd9,  /* ...5:..N..l..... */
    0x40, 0x79, 0x59, 0x4c, 0xcf, 0xae, 0xfe, 0x28, 0xd9, 0xaa, 0xc9, 0xac, 0x1b, 0xd3, 0x7c, 0x89,  /* @yYL...(......|. */
    0xb6, 0xcc, 0xbd, 0x10, 0xb1, 0x4f, 0xc3, 0xaa, 0x19, 0xde, 0xb1, 0xfd, 0x03, 0xea, 0x15, 0x1a,  /* .....O.......... */
    0x42, 0xb8, 0x92, 0x54, 0x7a, 0x33, 0x96, 0x18, 0xc4, 0xc9, 0xf2, 0x6f, 0xa9, 0x83, 0x85, 0x5d,  /* B..Tz3.....o...] */
    0x89, 0x3f, 0x81, 0x41, 0x3c, 0xd3, 0x20, 0xea, 0x42, 0x30, 0xd3, 0x41, 0x5e, 0xbc, 0xcd, 0xb5,  /* .?.A<. .B0.A^... */
    0x90, 0x8d, 0x91, 0xdc, 0x23, 0xc6, 0x56, 0x6f, 0x47, 0xb8, 0xa0, 0xe0, 0xa1, 0x9c, 0x17, 0x5b,  /* ....#.VoG......[ */
    0xdc, 0x77, 0x5d, 0x88, 0x24, 0x67, 0x6a, 0xac, 0xfa, 0xed, 0xea, 0x0c, 0x16, 0x0e, 0x87, 0x01,  /* .w].$gj......... */
    0x01, 0x5f, 0x20, 0x0d, 0x44, 0x45, 0x43, 0x56, 0x43, 0x41, 0x41, 0x54, 0x30, 0x30, 0x30, 0x30,  /* ._ .DECVCAAT0000 */
    0x31, 0x7f, 0x4c, 0x0e, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, 0x01, 0x53,  /* 1.L............S */
    0x01, 0xc3, 0x5f, 0x25, 0x06, 0x01, 0x00, 0x00, 0x09, 0x03, 0x00, 0x5f, 0x24, 0x06, 0x01, 0x01,  /* .._%......._$... */
    0x00, 0x09, 0x02, 0x05, 0x5f, 0x37, 0x81, 0x80, 0xa5, 0xe7, 0x22, 0x44, 0xd0, 0xac, 0x9c, 0xd3,  /* ...._7...."D.... */
    0xe8, 0xfe, 0x9b, 0x87, 0x8e, 0xa3, 0x2a, 0x24, 0xf4, 0xaa, 0x06, 0x0f, 0xd1, 0x7c, 0x2a, 0x82,  /* ......*$.....|*. */
    0x91, 0xd9, 0x79, 0x7e, 0x8a, 0x60, 0xf2, 0x8a, 0x29, 0x78, 0xe3, 0xf4, 0xea, 0xe8, 0x40, 0xe9,  /* ..y~.`..)x....@. */
    0x5b, 0x6a, 0x64, 0xfd, 0xa4, 0x4d, 0x8d, 0x39, 0xa6, 0x99, 0x52, 0x1c, 0xd5, 0x38, 0x22, 0x03,  /* [jd..M.9..R..8". */
    0x96, 0x49, 0x45, 0x15, 0x0a, 0x6c, 0xc4, 0xa5, 0x0c, 0x49, 0xee, 0x23, 0x4a, 0xb3, 0x6d, 0xf6,  /* .IE..l...I.#J.m. */
    0x75, 0xe4, 0x4c, 0x11, 0x49, 0x90, 0x80, 0x67, 0x6d, 0xd5, 0xb3, 0xac, 0x18, 0x59, 0xd9, 0x8b,  /* u.L.I..gm....Y.. */
    0x30, 0x2c, 0xa9, 0x60, 0x5c, 0xd9, 0x4a, 0x51, 0xd7, 0xda, 0x57, 0x38, 0xbe, 0xaa, 0x49, 0x84,  /* 0,.`\.JQ..W8..I. */
    0x8b, 0xca, 0x77, 0x77, 0xb7, 0xdc, 0x9a, 0xcc, 0xda, 0xa1, 0xd0, 0xb6, 0x44, 0x32, 0xb2, 0xc1,  /* ..ww........D2.. */
    0x38, 0x89, 0xa4, 0x85, 0x4c, 0x9f, 0xf3, 0x46,                                                  /* 8...L..F */
};
static char tc_ecdh_dv_cert[] = {
    0x7f, 0x21, 0x82, 0x01, 0x66, 0x7f, 0x4e, 0x81, 0xde, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x0d, 0x44,  /*.!..f.N.._)..B.D */
    0x45, 0x43, 0x56, 0x43, 0x41, 0x41, 0x54, 0x30, 0x30, 0x30, 0x30, 0x31, 0x7f, 0x49, 0x81, 0x90,  /*ECVCAAT00001.I.. */
    0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x05, 0x86, 0x81, 0x81, 0x04,  /*................ */
    0x0a, 0x74, 0x97, 0x2e, 0x84, 0xb7, 0xd2, 0xc4, 0x28, 0xfb, 0xe4, 0x6d, 0x40, 0xc9, 0x2d, 0x6c,  /*.t......(..m@.-l */
    0xb5, 0x6a, 0xe6, 0xa4, 0xb0, 0xaf, 0x5b, 0x8b, 0xc0, 0x92, 0x7e, 0x5e, 0xf6, 0xf7, 0x32, 0x20,  /*.j....[...~^..2  */
    0x77, 0x6d, 0x31, 0x69, 0x3e, 0x36, 0xd6, 0x85, 0xdf, 0x6c, 0xf3, 0x76, 0x3e, 0x41, 0x72, 0x8d,  /*wm1i>6...l.v>Ar. */
    0x96, 0x7d, 0xc1, 0x96, 0x3f, 0x7c, 0xa7, 0x0f, 0x0e, 0xa7, 0xad, 0xb5, 0xea, 0x85, 0x6d, 0x8a,  /*.}..?|........m. */
    0x13, 0x3b, 0x86, 0x7b, 0x8e, 0xf4, 0x13, 0x2e, 0x7f, 0xfa, 0x3e, 0x8f, 0x32, 0xcd, 0x03, 0x21,  /*.;.{......>.2..! */
    0xfc, 0x22, 0x19, 0x9b, 0x93, 0x23, 0x37, 0x6e, 0x59, 0xd8, 0x4a, 0x06, 0x24, 0x91, 0x94, 0x81,  /*."...#7nY.J.$... */
    0x17, 0x17, 0x8b, 0xf4, 0xdb, 0x6c, 0x0e, 0xe7, 0xa2, 0x35, 0xbf, 0xc3, 0xd4, 0x45, 0x91, 0x83,  /*.....l...5...E.. */
    0xd4, 0x08, 0x23, 0x2a, 0xc7, 0x81, 0xc7, 0x8c, 0xf5, 0xc5, 0x4e, 0x2f, 0xf3, 0x5b, 0xa5, 0x16,  /*..#*......N/.[.. */
    0x5f, 0x20, 0x0d, 0x44, 0x45, 0x54, 0x45, 0x53, 0x54, 0x44, 0x56, 0x44, 0x45, 0x30, 0x31, 0x39,  /*_ .DETESTDVDE00x19, */
    0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, 0x02, 0x53, 0x05,  /*.L............S. */
    0x80, 0x1f, 0xff, 0xff, 0x10, 0x5f, 0x25, 0x06, 0x01, 0x00, 0x00, 0x09, 0x03, 0x00, 0x5f, 0x24,  /*....._%......._$ */
    0x06, 0x01, 0x00, 0x01, 0x00, 0x03, 0x00, 0x5f, 0x37, 0x81, 0x80, 0x98, 0xc6, 0x37, 0xbe, 0xfa,  /*......._7....7.. */
    0x63, 0x05, 0x89, 0x21, 0x90, 0x28, 0x96, 0xa6, 0x05, 0x20, 0x6d, 0x5b, 0xe3, 0xbf, 0xf2, 0xf7,  /*c..!.(... m[.... */
    0xe2, 0x58, 0xeb, 0x0b, 0xb0, 0x6e, 0xa7, 0xe8, 0x4d, 0xfb, 0xc9, 0x8e, 0x35, 0x27, 0x61, 0x10,  /*.X...n..M...5'a. */
    0xc6, 0x84, 0xbb, 0x23, 0xd2, 0x77, 0x35, 0x0e, 0x38, 0x62, 0xe6, 0xa3, 0x05, 0x20, 0x80, 0x8f,  /*...#.w5.8b... .. */
    0x41, 0x7b, 0x89, 0x11, 0x98, 0x58, 0x80, 0xa5, 0x56, 0xe1, 0x75, 0x8c, 0x7f, 0xe6, 0x54, 0x5b,  /*A{...X..V.u...T[ */
    0x3d, 0xa0, 0x33, 0x83, 0xbd, 0x7b, 0x5b, 0x51, 0xde, 0xc0, 0x07, 0x13, 0x9c, 0xbc, 0x44, 0xff,  /*=.3..{[Q......D. */
    0x32, 0x0f, 0x84, 0xf2, 0xb6, 0xad, 0x44, 0xc7, 0xb2, 0x8f, 0xca, 0xeb, 0x2b, 0x7b, 0x98, 0xe3,  /*2.....D.....+{.. */
    0xa2, 0xba, 0xe8, 0x32, 0x2b, 0x11, 0x5a, 0x8f, 0x10, 0xd4, 0x83, 0x49, 0x28, 0x1b, 0x4f, 0x46,  /*...2+.Z....I(.OF */
    0x45, 0xc9, 0xaa, 0xe3, 0x07, 0xcf, 0xae, 0x9a, 0xd6, 0xb1, 0x94,                                /*E.......... */
};

static char tc_dh_ef_cardaccess[] = {
    0x31, 0x81, 0xC6, 0x30, 0x0D, 0x06, 0x08, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02,  /* 1..0............ */
    0x01, 0x02, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, 0x01, 0x02,  /* ..0............. */
    0x02, 0x01, 0x02, 0x02, 0x01, 0x01, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02,  /* ......0......... */
    0x02, 0x04, 0x01, 0x02, 0x02, 0x01, 0x02, 0x02, 0x01, 0x00, 0x30, 0x1C, 0x06, 0x09, 0x04, 0x00,  /* ..........0..... */
    0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, 0x01, 0x30, 0x0C, 0x06, 0x07, 0x04, 0x00, 0x7F, 0x00, 0x07,  /* .......0........ */
    0x01, 0x02, 0x02, 0x01, 0x00, 0x02, 0x01, 0x01, 0x30, 0x2F, 0x06, 0x08, 0x04, 0x00, 0x7F, 0x00,  /* ........0/...... */
    0x07, 0x02, 0x02, 0x06, 0x16, 0x23, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3A, 0x2F, 0x2F, 0x77, 0x77,  /* .....#https://ww */
    0x77, 0x2E, 0x68, 0x6A, 0x70, 0x2D, 0x63, 0x6F, 0x6E, 0x73, 0x75, 0x6C, 0x74, 0x69, 0x6E, 0x67,  /* w.hjp-consulting */
    0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x68, 0x6F, 0x6D, 0x65, 0x30, 0x3E, 0x06, 0x08, 0x04, 0x00, 0x7F,  /* .com/home0>..... */
    0x00, 0x07, 0x02, 0x02, 0x08, 0x31, 0x32, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07,  /* .....120........ */
    0x02, 0x02, 0x03, 0x01, 0x02, 0x02, 0x01, 0x02, 0x02, 0x01, 0x02, 0x30, 0x1C, 0x06, 0x09, 0x04,  /* ...........0.... */
    0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, 0x01, 0x30, 0x0C, 0x06, 0x07, 0x04, 0x00, 0x7F, 0x00,  /* ........0....... */
    0x07, 0x01, 0x02, 0x02, 0x01, 0x00, 0x02, 0x01, 0x02,                                            /* ......... */
};
static char tc_dh_ef_cardsecurity[] = {
    0x30, 0x82, 0x08, 0x2C, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x02, 0xA0, /*0..,..*.H.......*/
    0x82, 0x08, 0x1D, 0x30, 0x82, 0x08, 0x19, 0x02, 0x01, 0x03, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09, /*...0......1.0...*/
    0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x82, 0x01, 0x92, 0x06, /*`.H.e......0....*/
    0x08, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x03, 0x02, 0x01, 0xA0, 0x82, 0x01, 0x84, 0x04, 0x82, 0x01, /*................*/
    0x80, 0x31, 0x82, 0x01, 0x7C, 0x30, 0x0D, 0x06, 0x08, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, /*.1..|0..........*/
    0x02, 0x02, 0x01, 0x02, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, /*....0...........*/
    0x01, 0x02, 0x02, 0x01, 0x02, 0x02, 0x01, 0x01, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, /*........0.......*/
    0x07, 0x02, 0x02, 0x04, 0x01, 0x02, 0x02, 0x01, 0x02, 0x02, 0x01, 0x00, 0x30, 0x1C, 0x06, 0x09, /*............0...*/
    0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, 0x01, 0x30, 0x0C, 0x06, 0x07, 0x04, 0x00, 0x7F, /*.........0......*/
    0x00, 0x07, 0x01, 0x02, 0x02, 0x01, 0x00, 0x02, 0x01, 0x01, 0x30, 0x2F, 0x06, 0x08, 0x04, 0x00, /*..........0/....*/
    0x7F, 0x00, 0x07, 0x02, 0x02, 0x06, 0x16, 0x23, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3A, 0x2F, 0x2F, /*.......#https://*/
    0x77, 0x77, 0x77, 0x2E, 0x68, 0x6A, 0x70, 0x2D, 0x63, 0x6F, 0x6E, 0x73, 0x75, 0x6C, 0x74, 0x69, /*www.hjp-consulti*/
    0x6E, 0x67, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x68, 0x6F, 0x6D, 0x65, 0x30, 0x17, 0x06, 0x0A, 0x04, /*ng.com/home0....*/
    0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x05, 0x01, 0x03, 0x30, 0x09, 0x02, 0x01, 0x01, 0x02,
    0x01, 0x01, 0x01, 0x01, 0x00, 0x30, 0x17, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, /*.....0..........*/
    0x05, 0x01, 0x03, 0x30, 0x09, 0x02, 0x01, 0x01, 0x02, 0x01, 0x02, 0x01, 0x01, 0xFF, 0x30, 0x19, /*...0..........0.*/
    0x06, 0x09, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x05, 0x01, 0x30, 0x0C, 0x06, 0x07, 0x04, /*...........0....*/
    0x00, 0x7F, 0x00, 0x07, 0x01, 0x02, 0x02, 0x01, 0x00, 0x30, 0x81, 0xA6, 0x06, 0x09, 0x04, 0x00, /*.........0......*/
    0x7F, 0x00, 0x07, 0x02, 0x02, 0x01, 0x01, 0x30, 0x81, 0x95, 0x30, 0x0C, 0x06, 0x07, 0x04, 0x00, /*.......0..0.....*/
    0x7F, 0x00, 0x07, 0x01, 0x02, 0x02, 0x01, 0x00, 0x03, 0x81, 0x84, 0x00, 0x02, 0x81, 0x80, 0x1B, /*................*/
    0x33, 0x45, 0xF8, 0xDC, 0x04, 0x34, 0x1B, 0xF8, 0xB2, 0xC9, 0x7F, 0x65, 0x2F, 0xA6, 0x80, 0xE5, /* 0x3E...4.....e/...*/
    0xD4, 0xFA, 0x4C, 0x14, 0x6A, 0xE4, 0xB8, 0x39, 0x43, 0x1A, 0x64, 0x4A, 0x79, 0xBC, 0x36, 0x8C, /*..L.j..9C.dJy.6.*/
    0x48, 0x22, 0xC8, 0x9C, 0xD0, 0x18, 0xA5, 0x7F, 0x95, 0x36, 0x44, 0xBE, 0xDA, 0x67, 0x9C, 0x5B, /*H".......6D..g.[*/
    0x53, 0x29, 0x02, 0x32, 0x0E, 0x83, 0xE1, 0x3B, 0x80, 0xDE, 0xEF, 0x8C, 0x18, 0xAF, 0x3E, 0x7D, /*S).2...;......>}*/
    0x49, 0x3A, 0xE3, 0xF8, 0x81, 0x96, 0x10, 0x1B, 0x9F, 0x78, 0xEA, 0xFE, 0x4B, 0x30, 0x25, 0xEF, /*I:.......x..K0%.*/
    0x8B, 0xFF, 0x91, 0x6B, 0x2F, 0xC0, 0x2D, 0x76, 0x2D, 0x08, 0x38, 0xDE, 0xA2, 0x9C, 0x09, 0xB4, /*...k/.-v-.8.....*/
    0x85, 0x59, 0x1C, 0x2F, 0x47, 0xF8, 0x7C, 0x71, 0xF5, 0x30, 0xBB, 0x35, 0x8F, 0x56, 0xAF, 0x64, /*.Y./G.|q.0.5.V.d*/
    0x59, 0xD8, 0x6D, 0xBF, 0x85, 0xEA, 0xF9, 0xED, 0xBD, 0x96, 0x2C, 0xD3, 0x64, 0xF7, 0xB8,
    0x02, 0x01, 0x01, 0xA0, 0x82, 0x04, 0x62, 0x30, 0x82, 0x04, 0x5E, 0x30, 0x82, 0x02, 0x92, 0xA0, /*......b0..^0....*/
    0x03, 0x02, 0x01, 0x02, 0x02, 0x03, 0x01, 0x63, 0x26, 0x30, 0x41, 0x06, 0x09, 0x2A, 0x86, 0x48, /*.......c&0A..*.H*/
    0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0A, 0x30, 0x34, 0xA0, 0x0F, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, /*......04..0...`.*/
    0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0xA1, 0x1C, 0x30, 0x1A, 0x06, 0x09, 0x2A, /*H.e........0...**/
    0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x08, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, /*.H......0...`.H.*/
    0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0xA2, 0x03, 0x02, 0x01, 0x20, 0x30, 0x53, 0x31, 0x0B, /*e.......... 0S1.*/
    0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x44, 0x45, 0x31, 0x17, 0x30, 0x15, 0x06, /*0...U....DE1.0..*/
    0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0E, 0x48, 0x4A, 0x50, 0x20, 0x43, 0x6F, 0x6E, 0x73, 0x75, 0x6C, /*.U....HJP Consul*/
    0x74, 0x69, 0x6E, 0x67, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x0E, 0x43, /*ting1.0...U....C*/
    0x6F, 0x75, 0x6E, 0x74, 0x72, 0x79, 0x20, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x31, 0x12, 0x30, /*ountry Signer1.0*/
    0x10, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x09, 0x48, 0x4A, 0x50, 0x20, 0x50, 0x42, 0x20, 0x43, /*...U....HJP PB C*/
    0x53, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x39, 0x30, 0x39, 0x31, 0x38, 0x30, 0x37, 0x35, 0x39, 0x35, /*S0...09091807595*/
    0x33, 0x5A, 0x17, 0x0D, 0x31, 0x30, 0x30, 0x39, 0x31, 0x33, 0x30, 0x37, 0x35, 0x39, 0x35, 0x33, /*3Z..100913075953*/
    0x5A, 0x30, 0x54, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x44,
    0x45, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0E, 0x48, 0x4A, 0x50, 0x20, /* 0xE1.0...U....HJP */
    0x43, 0x6F, 0x6E, 0x73, 0x75, 0x6C, 0x74, 0x69, 0x6E, 0x67, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, /*Consulting1.0...*/
    0x55, 0x04, 0x0B, 0x0C, 0x0F, 0x44, 0x6F, 0x63, 0x75, 0x6D, 0x65, 0x6E, 0x74, 0x20, 0x53, 0x69, /*U....Document Si*/
    0x67, 0x6E, 0x65, 0x72, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x09, 0x48, /*gner1.0...U....H*/
    0x4A, 0x50, 0x20, 0x50, 0x42, 0x20, 0x44, 0x53, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, /*JP PB DS0.."0...*/
    0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, /**.H.............*/
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xB6, 0xC5, 0xA8, 0xEE, 0x57, 0x30, 0x76, /*0............W0v*/
    0x79, 0xE3, 0x64, 0xE3, 0xF7, 0xE7, 0x76, 0xEA, 0x64, 0x07, 0x4E, 0x9A, 0x72, 0xF6, 0xB3, 0x76, /*y.d...v.d.N.r..v*/
    0xC2, 0xD2, 0x31, 0x89, 0x63, 0x1C, 0x10, 0xBA, 0x65, 0xEA, 0x34, 0x6F, 0xEF, 0x70, 0x3B, 0x52, /*..1.c...e.4o.p;R*/
    0xEF, 0x75, 0x93, 0xD4, 0x96, 0xE1, 0x50, 0x0F, 0x71, 0x64, 0xD0, 0x38, 0xE9, 0xA8, 0x80, 0xD0, /*.u....P.qd.8....*/
    0x6E, 0x90, 0xFC, 0xF9, 0x6F, 0xAD, 0x5B, 0x60, 0x68, 0xB3, 0x42, 0xCC, 0xA8, 0x24, 0x77, 0x0B, /*n...o.[`h.B..$w.*/
    0xAD, 0xF1, 0x42, 0x9E, 0xBB, 0xDB, 0x97, 0x88, 0x0A, 0xAE, 0xA4, 0x31, 0x14, 0x62, 0xCA, 0xCE, /*..B........1.b..*/
    0x40, 0xAA, 0xF2, 0x24, 0x78, 0x35, 0xAB, 0xC2, 0x59, 0x1E, 0x18, 0x80, 0xAD, 0xD9, 0xFD, 0x35, /*@..$x5..Y......5*/
    0xF2, 0xC0, 0xE4, 0x3C, 0xC6, 0xFE, 0xB9, 0x1B, 0x0F, 0x13, 0x7C, 0xC4, 0x2A, 0xD8, 0x34,
    0x73, 0x24, 0x93, 0xFD, 0x63, 0xF7, 0x8F, 0xC7, 0xDA, 0x75, 0xCD, 0xB4, 0xA1, 0xBD, 0x4C, 0x7D, /*s$..c....u....L}*/
    0xE1, 0xE0, 0x4A, 0xC1, 0xB4, 0xBD, 0x4C, 0x62, 0xC4, 0x6F, 0x8D, 0x83, 0xEE, 0x6B, 0xF1, 0xAC, /*..J...Lb.o...k..*/
    0x24, 0x05, 0xD5, 0xA1, 0x73, 0x77, 0x6A, 0x58, 0x96, 0x0A, 0x79, 0xB1, 0xB5, 0xB9, 0x0B, 0x79, /*$...swjX..y....y*/
    0x7A, 0x4A, 0x7A, 0x19, 0x84, 0x57, 0xC7, 0xA0, 0x2A, 0x72, 0xA2, 0xFF, 0x9A, 0x32, 0x7E, 0x55, /*zJz..W..*r...2~U*/
    0x88, 0x19, 0x67, 0x42, 0xC5, 0x7C, 0x8B, 0x9D, 0x5E, 0xE6, 0x4B, 0x8A, 0x46, 0x00, 0x3B, 0xC5, /*..gB.|..^.K.F.;.*/
    0x6D, 0x24, 0x93, 0xC0, 0xA6, 0x58, 0x82, 0x37, 0x94, 0xAB, 0x23, 0x14, 0xBC, 0xF9, 0x79, 0xC5, /*m$...X.7..#...y.*/
    0xEE, 0xDF, 0x32, 0x7C, 0x6C, 0x11, 0x2E, 0x9E, 0xDD, 0x86, 0xC6, 0xE4, 0x19, 0xF9, 0xAD, 0x35, /*..2|l..........5*/
    0xA9, 0x46, 0x56, 0xFD, 0xE7, 0xED, 0x89, 0x6A, 0xF5, 0xC3, 0x46, 0x43, 0x5A, 0xB3, 0xD7, 0xBE, /*.FV....j..FCZ...*/
    0xC0, 0xF8, 0xB9, 0x02, 0x56, 0xA3, 0x10, 0x50, 0xB3, 0x97, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, /*....V..P........*/
    0x52, 0x30, 0x50, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, /*R0P0...U.#..0...*/
    0x0D, 0xFD, 0x5C, 0x02, 0x88, 0xBF, 0xEC, 0xE0, 0xB0, 0x7A, 0x5D, 0x40, 0xFF, 0x80, 0xAC, 0x8A, /*..\......z]@....*/
    0x91, 0x74, 0x3A, 0x9B, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0x91, /*.t:.0...U.......*/
    0x93, 0xF4, 0xF0, 0xAA, 0x4A, 0xCA, 0xC0, 0xD3, 0xA1, 0xB6, 0xAC, 0x83, 0xB2, 0x4F, 0x6F, 0xDC, /*....J........Oo.*/
    0x8F, 0xF2, 0x1B, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04,
    0x03, 0x02, 0x07, 0x80, 0x30, 0x41, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, /*....0A..*.H.....*/
    0x0A, 0x30, 0x34, 0xA0, 0x0F, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, /*.04..0...`.H.e..*/
    0x02, 0x01, 0x05, 0x00, 0xA1, 0x1C, 0x30, 0x1A, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, /*......0...*.H...*/
    0x01, 0x01, 0x08, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, /*...0...`.H.e....*/
    0x05, 0x00, 0xA2, 0x03, 0x02, 0x01, 0x20, 0x03, 0x82, 0x01, 0x81, 0x00, 0xA3, 0xAF, 0xEC, 0x3B, /*...... ........;*/
    0xC5, 0xD3, 0x66, 0xE6, 0x61, 0x19, 0x4A, 0xCA, 0x8D, 0xFC, 0x39, 0x06, 0x76, 0x06, 0x1D, 0x6E, /*..f.a.J...9.v..n*/
    0x52, 0x30, 0x18, 0xCA, 0x13, 0x93, 0x0D, 0x79, 0x40, 0xE6, 0xCE, 0x77, 0x86, 0x1D, 0x18, 0xF6, /*R0.....y@..w....*/
    0x5F, 0x3C, 0xEF, 0x8C, 0xBF, 0x44, 0xE8, 0x7D, 0x59, 0xAA, 0xFA, 0x6F, 0x29, 0xEC, 0x57, 0xFB, /*_<...D.}Y..o).W.*/
    0x19, 0xDB, 0x12, 0x30, 0xF0, 0xF2, 0xFC, 0x1B, 0xF6, 0x0D, 0x1D, 0x03, 0x96, 0x33, 0x3C, 0x89, /*...0.........3<.*/
    0xA9, 0x2B, 0xF2, 0x31, 0x3C, 0x43, 0x60, 0xBA, 0xB2, 0x18, 0xDE, 0x57, 0x71, 0x3F, 0x39, 0x0C, /*.+.1<C`....Wq?9.*/
    0xA5, 0xBB, 0xB6, 0x99, 0xCD, 0x1A, 0x1E, 0x27, 0x3C, 0x61, 0x8B, 0x25, 0xA7, 0xEE, 0xDA, 0xB5, /*.......'<a.%....*/
    0xF0, 0xBA, 0xB0, 0x30, 0x65, 0xAA, 0x74, 0x9D, 0x51, 0x32, 0x60, 0xBE, 0x86, 0x7E, 0xB0, 0x11, /*...0e.t.Q2`..~..*/
    0x29, 0x1D, 0xCF, 0x4A, 0xDC, 0x83, 0x33, 0xF7, 0x78, 0x4F, 0xDD, 0xE8, 0x17, 0x2F, 0x46, 0xC4, /*)..J..3.xO.../F.*/
    0xF7, 0x90, 0x42, 0x15, 0xFD, 0xC9, 0x8F, 0x5C, 0xDE, 0x49, 0x16, 0xF0, 0x3E, 0x24, 0x9C,
    0xD3, 0x94, 0x07, 0x62, 0xD2, 0xF8, 0xE9, 0x2F, 0x23, 0x17, 0x16, 0xA6, 0xBF, 0x74, 0x2F, 0xED, /*...b.../#....t/.*/
    0xC2, 0x62, 0x7E, 0x62, 0xF0, 0x46, 0x95, 0x6D, 0xB9, 0x7B, 0xAA, 0xD2, 0x5C, 0x04, 0x62, 0x47, /*.b~b.F.m.{..\.bG*/
    0x54, 0xD4, 0xAF, 0x3E, 0x1A, 0x7E, 0xC4, 0x72, 0x07, 0xCC, 0x08, 0xBD, 0x15, 0x4E, 0x83, 0x9A, /*T..>.~.r.....N..*/
    0x43, 0x55, 0xD0, 0x1F, 0x16, 0xDA, 0x2C, 0xC1, 0x61, 0x77, 0xA9, 0x14, 0xD4, 0x42, 0x87, 0xE6, /*CU....,.aw...B..*/
    0x52, 0x25, 0x64, 0xD0, 0x00, 0x53, 0x9E, 0xC9, 0x6A, 0x2B, 0x0E, 0x1E, 0x6E, 0xBB, 0x89, 0x63, /*R%d..S..j+..n..c*/
    0x81, 0x86, 0x8B, 0x5A, 0xFE, 0x0A, 0x0F, 0xD3, 0xC3, 0x62, 0xF4, 0x19, 0xAF, 0xFD, 0xFF, 0x01, /*...Z.....b......*/
    0x6A, 0x71, 0x17, 0x0A, 0xC8, 0xB3, 0x78, 0xA6, 0xE3, 0x99, 0x5D, 0x82, 0xEE, 0x45, 0x95, 0x0E, /*jq....x...]..E..*/
    0xEB, 0xB4, 0xC9, 0xBB, 0xF6, 0x31, 0x13, 0x24, 0x82, 0xA5, 0x03, 0xC3, 0x10, 0x26, 0xB4, 0xC2, /*.....1.$.....&..*/
    0xCD, 0x94, 0x26, 0xE6, 0x66, 0x3D, 0xE4, 0xC4, 0x3E, 0xFE, 0x54, 0x01, 0xF4, 0xD3, 0xBA, 0x76, /*..&.f=..>.T....v*/
    0xE5, 0x4F, 0x66, 0x3B, 0x28, 0x32, 0x3E, 0xA3, 0x33, 0x1E, 0x96, 0xA7, 0x08, 0x12, 0xF9, 0x43, /*.Of;(2>.3......C*/
    0x15, 0xD6, 0x08, 0xA9, 0xE8, 0xCE, 0x1B, 0xF0, 0x2B, 0x6E, 0xCF, 0x07, 0x01, 0x5D, 0x40, 0xF4, /*........+n...]@.*/
    0x73, 0xDF, 0xE1, 0x6F, 0x5C, 0x12, 0x14, 0x60, 0x81, 0xC4, 0x4C, 0x14, 0x8D, 0xAB, 0x09, 0x83, /*s..o\..`..L.....*/
    0x50, 0x46, 0x57, 0xA5, 0x3C, 0xCA, 0x16, 0xBD, 0x54, 0x5D, 0x5A, 0xD5, 0x9A, 0x21, 0xAA, 0x91, /*PFW.<...T]Z..!..*/
    0x9E, 0x7F, 0x9B, 0xB7, 0xB3, 0x50, 0x01, 0xAB, 0xEF, 0x61, 0xE7, 0xD5, 0x6E, 0x21, 0xC7,
    0xF1, 0x13, 0x73, 0x42, 0x55, 0x71, 0xA7, 0x91, 0x45, 0xD4, 0x46, 0x2E, 0xB2, 0x6B, 0x31, 0x82, /*..sBUq..E.F..k1.*/
    0x02, 0x05, 0x30, 0x82, 0x02, 0x01, 0x02, 0x01, 0x01, 0x30, 0x5A, 0x30, 0x53, 0x31, 0x0B, 0x30, /*..0......0Z0S1.0*/
    0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x44, 0x45, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, /*...U....DE1.0...*/
    0x55, 0x04, 0x0A, 0x0C, 0x0E, 0x48, 0x4A, 0x50, 0x20, 0x43, 0x6F, 0x6E, 0x73, 0x75, 0x6C, 0x74, /*U....HJP Consult*/
    0x69, 0x6E, 0x67, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x0E, 0x43, 0x6F, /*ing1.0...U....Co*/
    0x75, 0x6E, 0x74, 0x72, 0x79, 0x20, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x31, 0x12, 0x30, 0x10, /*untry Signer1.0.*/
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x09, 0x48, 0x4A, 0x50, 0x20, 0x50, 0x42, 0x20, 0x43, 0x53, /*..U....HJP PB CS*/
    0x02, 0x03, 0x01, 0x63, 0x26, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, /*...c&0...`.H.e..*/
    0x02, 0x01, 0x05, 0x00, 0xA0, 0x4A, 0x30, 0x17, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, /*.....J0...*.H...*/
    0x01, 0x09, 0x03, 0x31, 0x0A, 0x06, 0x08, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x03, 0x02, 0x01, 0x30, /*...1...........0*/
    0x2F, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x04, 0x31, 0x22, 0x04, 0x20, /*/..*.H......1". */
    0x5F, 0x65, 0x34, 0x5D, 0x29, 0xA8, 0x5A, 0x01, 0xBB, 0x38, 0xAE, 0xA4, 0xEB, 0xD6, 0xC9, 0x8E, /*_e4]).Z..8......*/
    0x7E, 0xCE, 0x6F, 0x9F, 0x0F, 0xF4, 0x6A, 0xF2, 0xFD, 0xD0, 0x7B, 0xDC, 0x18, 0xBA, 0x6F, 0x38, /*~.o...j...{...o8*/
    0x30, 0x41, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0A, 0x30, 0x34,
    0xA0, 0x0F, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, /*..0...`.H.e.....*/
    0x00, 0xA1, 0x1C, 0x30, 0x1A, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x08, /*...0...*.H......*/
    0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0xA2, /*0...`.H.e.......*/
    0x03, 0x02, 0x01, 0x20, 0x04, 0x82, 0x01, 0x00, 0x3B, 0xCE, 0x96, 0x95, 0x4D, 0x09, 0x8B, 0xD4, /*... ....;...M...*/
    0xC6, 0xCC, 0xD6, 0x9D, 0xD2, 0xEB, 0x73, 0xDD, 0x58, 0x85, 0xA4, 0x12, 0xF5, 0x9F, 0x48, 0x93, /*......s.X.....H.*/
    0xB4, 0xC1, 0x6F, 0x4F, 0x15, 0x2C, 0x5A, 0x7F, 0x62, 0x08, 0x24, 0xDA, 0x91, 0x85, 0xFC, 0x23, /*..oO.,Z.b.$....#*/
    0x6F, 0x3B, 0x72, 0x01, 0x8F, 0x77, 0x59, 0xAD, 0xBD, 0xD1, 0x66, 0xC1, 0x16, 0xAB, 0x5D, 0x4D, /*o;r..wY...f...]M*/
    0xE5, 0x4C, 0x89, 0x43, 0x68, 0x0A, 0xD9, 0x77, 0x4D, 0x4B, 0x31, 0x02, 0x8D, 0x6E, 0x3F, 0xA4, /*.L.Ch..wMK1..n?.*/
    0x11, 0x73, 0x5B, 0xED, 0x1E, 0x73, 0xE9, 0x9F, 0x9C, 0x5A, 0x25, 0x5B, 0xB0, 0x3E, 0xF6, 0x94, /*.s[..s...Z%[.>..*/
    0x59, 0xB4, 0xB3, 0xBB, 0xA9, 0x58, 0xD3, 0x0D, 0x0F, 0xF5, 0xC9, 0xFE, 0x0A, 0x0F, 0xCB, 0x09, /*Y....X..........*/
    0x6F, 0x8D, 0x41, 0xCA, 0xB1, 0xD8, 0xFB, 0x0E, 0xA6, 0x15, 0x3F, 0x01, 0xE2, 0xA4, 0x2B, 0x7F, /*o.A.......?...+.*/
    0xE6, 0xB2, 0x30, 0x4D, 0x96, 0x27, 0xE7, 0x30, 0xC6, 0x3F, 0x9B, 0xCB, 0xAB, 0x0A, 0x44, 0x5D, /*..0M.'.0.?....D]*/
    0xE9, 0xDC, 0xDA, 0xE1, 0xF8, 0xE9, 0xC2, 0x76, 0x35, 0x41, 0x0F, 0x49, 0xC8, 0xEB, 0x45, 0xFA, /*.......v5A.I..E.*/
    0xB2, 0x5F, 0x55, 0xC0, 0x5A, 0x78, 0x58, 0x82, 0x03, 0x44, 0x25, 0x46, 0x96, 0x16, 0xFC,
    0xB7, 0x86, 0xEC, 0xBD, 0x7C, 0x92, 0x93, 0x0F, 0x2E, 0x2D, 0xAE, 0xA9, 0x8E, 0xF8, 0x55, 0x73, /*....|....-....Us*/
    0x23, 0x98, 0x78, 0x51, 0x3E, 0x4F, 0x9A, 0xCD, 0x89, 0x75, 0x31, 0xF5, 0x5E, 0x3E, 0x15, 0x8D, /*#.xQ>O...u1.^>..*/
    0x61, 0x81, 0x5B, 0x48, 0x75, 0xD1, 0xD7, 0x6E, 0x0E, 0xE6, 0x38, 0x0D, 0x57, 0x6C, 0xF5, 0xCF, /*a.[Hu..n..8.Wl..*/
    0x8F, 0x2F, 0x0A, 0x8E, 0xEC, 0xB6, 0xCB, 0x51, 0x34, 0xAA, 0x89, 0x44, 0x1A, 0x4B, 0x87, 0x58, /*./.....Q4..D.K.X*/
    0x91, 0xAE, 0x22, 0x81, 0x1B, 0x90, 0x49, 0x45, 0xF8, 0x4C, 0x53, 0xFC, 0x0E, 0x9B, 0xA6, 0x42, /*.."...IE.LS....B*/
    0x93, 0x33, 0x1F, 0x31, 0x5D, 0xA9, 0x68, 0xD8, 0xB8,
};
static char tc_dh_nonce_enc[] = {
    0x9A, 0xBB, 0x88, 0x64, 0xCA, 0x0F, 0xF1, 0x55, 0x1E, 0x62, 0x0D, 0x1E, 0xF4, 0xE1, 0x35, 0x10,  /* ...d...U.b....5. */
};
static char tc_dh_nonce[] = {
    0xFA, 0x5B, 0x7E, 0x3E, 0x49, 0x75, 0x3A, 0x0D, 0xB9, 0x17, 0x8B, 0x7B, 0x9B, 0xD8, 0x98, 0xC8,  /* .[~>Iu:....{.... */
};
static char tc_dh_map_pcd_priv_key[] = {
    0x24, 0xC3, 0xC0, 0xE0, 0xA3, 0x28, 0x0E, 0xCB, 0x94, 0x33, 0x45, 0xD9, 0xDC, 0x2A, 0x7B, 0x72,  /* $....(...3E..*{r */
    0x53, 0x9F, 0xDA, 0x6F, 0xFD, 0xF9, 0x9A, 0xB7, 0xB6, 0xCD, 0xDD, 0xD1, 0xBE, 0x42, 0x5A, 0xF3,  /* S..o.........BZ. */
    0xD0, 0x2C, 0x4E, 0xD0, 0xCD, 0xD7, 0x3E, 0xBB, 0x4B, 0x2E, 0xDF, 0x8C, 0x07, 0xFB, 0x3A, 0x35,  /* .,N...>.K.....:5 */
    0x90, 0x3F, 0x72, 0xB8, 0x4F, 0x37, 0x71, 0xF4, 0xEB, 0xFB, 0x49, 0x52, 0x0D, 0x61, 0xA8, 0xF7,  /* .?r.O7q...IR.a.. */
    0xC7, 0xFB, 0x8C, 0x9E, 0x2A, 0xBC, 0x24, 0xBF, 0x4F, 0xF9, 0xD8, 0xDD, 0xF3, 0x81, 0xA1, 0x93,  /* ....*.$.O....... */
    0x80, 0xC8, 0x5B, 0x62, 0x3A, 0xB0, 0x2A, 0xCB, 0xF6, 0xD2, 0x20, 0xF5, 0x12, 0xBF, 0x40, 0x65,  /* ..[b:.*... ...@e */
    0x83, 0x22, 0xAD, 0x20, 0x9A, 0xC0, 0xBF, 0x9E, 0x6F, 0x8D, 0xB6, 0x02, 0xD5, 0x19, 0x7D, 0x25,  /* .". ....o.....}% */
    0x2B, 0xF6, 0xD1, 0x48, 0x51, 0x0C, 0xA1, 0xB7, 0x40, 0xAF, 0x0F, 0x99, 0xF3, 0x3C, 0xA5, 0xF1,  /* +..HQ...@....<.. */
};
static char tc_dh_map_pcd_pub_key[] = {
    0x23, 0xFB, 0x37, 0x49, 0xEA, 0x03, 0x0D, 0x2A, 0x25, 0xB2, 0x78, 0xD2, 0xA5, 0x62, 0x04, 0x7A,  /* #.7I...*%.x..b.z */
    0xDE, 0x3F, 0x01, 0xB7, 0x4F, 0x17, 0xA1, 0x54, 0x02, 0xCB, 0x73, 0x52, 0xCA, 0x7D, 0x2B, 0x3E,  /* .?..O..T..sR.}+> */
    0xB7, 0x1C, 0x34, 0x3D, 0xB1, 0x3D, 0x1D, 0xEB, 0xCE, 0x9A, 0x36, 0x66, 0xDB, 0xCF, 0xC9, 0x20,  /* ..4=.=....6f...  */
    0xB4, 0x91, 0x74, 0xA6, 0x02, 0xCB, 0x47, 0x96, 0x5C, 0xAA, 0x73, 0xDC, 0x70, 0x24, 0x89, 0xA4,  /* ..t...G.\.s.p$.. */
    0x4D, 0x41, 0xDB, 0x91, 0x4D, 0xE9, 0x61, 0x3D, 0xC5, 0xE9, 0x8C, 0x94, 0x16, 0x05, 0x51, 0xC0,  /* MA..M.a=......Q. */
    0xDF, 0x86, 0x27, 0x4B, 0x93, 0x59, 0xBC, 0x04, 0x90, 0xD0, 0x1B, 0x03, 0xAD, 0x54, 0x02, 0x2D,  /* ..'K.Y.......T.- */
    0xCB, 0x4F, 0x57, 0xFA, 0xD6, 0x32, 0x24, 0x97, 0xD7, 0xA1, 0xE2, 0x8D, 0x46, 0x71, 0x0F, 0x46,  /* .OW..2$.....Fq.F */
    0x1A, 0xFE, 0x71, 0x0F, 0xBB, 0xBC, 0x5F, 0x8B, 0xA1, 0x66, 0xF4, 0x31, 0x19, 0x75, 0xEC, 0x6C,  /* ..q..._..f.1.u.l */
    };
static char tc_dh_map_picc_priv_key[] = {
    0x4E, 0xC0, 0x25, 0xE4, 0x0C, 0x6D, 0x10, 0xB2, 0xAA, 0xF6, 0xFC, 0xAC, 0x98, 0xC4, 0x24, 0x4F,  /* N.%..m........$O */
    0x57, 0x48, 0x1A, 0x49, 0x61, 0xF3, 0xAD, 0xC3, 0x72, 0xA9, 0x5E, 0x40, 0xE0, 0xCC, 0x35, 0x55,  /* WH.Ia...r.^@..5U */
    0xF7, 0x3C, 0xCF, 0xC6, 0x5E, 0x9D, 0xB9, 0x56, 0xDD, 0x61, 0xB1, 0x43, 0xE0, 0xC7, 0xDC, 0x51,  /* .<..^..V.a.C...Q */
    0x9E, 0x7D, 0xD8, 0xED, 0xD8, 0xE3, 0xE4, 0x6A, 0x09, 0x4C, 0xF2, 0x26, 0x4F, 0xD1, 0x93, 0xD0,  /* .}.....j.L.&O... */
    0xBC, 0x4B, 0xC0, 0x5C, 0xDE, 0x6C, 0xA4, 0x43, 0x19, 0xC2, 0x43, 0x9F, 0xD0, 0x4A, 0x46, 0x44,  /* .K.\.l.C..C..JFD */
    0x3C, 0x8D, 0x04, 0x94, 0x48, 0x7F, 0x6F, 0x2F, 0xE9, 0xAC, 0x8B, 0xE9, 0xB9, 0xEE, 0x16, 0xA3,  /* <...H.o/........ */
    0xD2, 0x42, 0x66, 0x8C, 0xBA, 0x4F, 0xFD, 0x42, 0xEE, 0xAC, 0x36, 0x50, 0x9E, 0x16, 0xB4, 0xD1,  /* .Bf..O.B..6P.... */
    0xE6, 0xE8, 0xEE, 0x00, 0x25, 0xFF, 0x82, 0x44, 0xB1, 0x90, 0xF5, 0x7D, 0x44, 0x1E, 0xC3, 0x28,  /* ....%..D...}D..( */
    };
static char tc_dh_map_picc_pub_key[] = {
    0x78, 0x87, 0x9F, 0x57, 0x22, 0x5A, 0xA8, 0x08, 0x0D, 0x52, 0xED, 0x0F, 0xC8, 0x90, 0xA4, 0xB2,  /* x..W"Z...R...... */
    0x53, 0x36, 0xF6, 0x99, 0xAA, 0x89, 0xA2, 0xD3, 0xA1, 0x89, 0x65, 0x4A, 0xF7, 0x07, 0x29, 0xE6,  /* S6........eJ..). */
    0x23, 0xEA, 0x57, 0x38, 0xB2, 0x63, 0x81, 0xE4, 0xDA, 0x19, 0xE0, 0x04, 0x70, 0x6F, 0xAC, 0xE7,  /* #.W8.c......po.. */
    0xB2, 0x35, 0xC2, 0xDB, 0xF2, 0xF3, 0x87, 0x48, 0x31, 0x2F, 0x3C, 0x98, 0xC2, 0xDD, 0x48, 0x82,  /* .5.....H1/<...H. */
    0xA4, 0x19, 0x47, 0xB3, 0x24, 0xAA, 0x12, 0x59, 0xAC, 0x22, 0x57, 0x9D, 0xB9, 0x3F, 0x70, 0x85,  /* ..G.$..Y."W..?p. */
    0x65, 0x5A, 0xF3, 0x08, 0x89, 0xDB, 0xB8, 0x45, 0xD9, 0xE6, 0x78, 0x3F, 0xE4, 0x2C, 0x9F, 0x24,  /* eZ.....E..x?.,.$ */
    0x49, 0x40, 0x03, 0x06, 0x25, 0x4C, 0x8A, 0xE8, 0xEE, 0x9D, 0xD8, 0x12, 0xA8, 0x04, 0xC0, 0xB6,  /* I@..%L.......... */
    0x6E, 0x8C, 0xAF, 0xC1, 0x4F, 0x84, 0xD8, 0x25, 0x89, 0x50, 0xA9, 0x1B, 0x44, 0x12, 0x6E, 0xE6,  /* n...O..%.P..D.n. */
};
static char tc_dh_map_shared_secret_h[] = {
    0x5B, 0xAB, 0xEB, 0xEF, 0x5B, 0x74, 0xE5, 0xBA, 0x94, 0xB5, 0xC0, 0x63, 0xFD, 0xA1, 0x5F, 0x1F,  /* [...[t.....c.._. */
    0x1C, 0xDE, 0x94, 0x87, 0x3E, 0xE0, 0xA5, 0xD3, 0xA2, 0xFC, 0xAB, 0x49, 0xF2, 0x58, 0xD0, 0x7F,  /* ....>......I.X.. */
    0x54, 0x4F, 0x13, 0xCB, 0x66, 0x65, 0x8C, 0x3A, 0xFE, 0xE9, 0xE7, 0x27, 0x38, 0x9B, 0xE3, 0xF6,  /* TO..fe.:...'8... */
    0xCB, 0xBB, 0xD3, 0x21, 0x28, 0xA8, 0xC2, 0x1D, 0xD6, 0xEE, 0xA3, 0xCF, 0x70, 0x91, 0xCD, 0xDF,  /* ...!(.......p... */
    0xB0, 0x8B, 0x8D, 0x00, 0x7D, 0x40, 0x31, 0x8D, 0xCC, 0xA4, 0xFF, 0xBF, 0x51, 0x20, 0x87, 0x90,  /* ....}@1.....Q .. */
    0xFB, 0x4B, 0xD1, 0x11, 0xE5, 0xA9, 0x68, 0xED, 0x6B, 0x6F, 0x08, 0xB2, 0x6C, 0xA8, 0x7C, 0x41,  /* .K....h.ko..l.|A */
    0x0B, 0x3C, 0xE0, 0xC3, 0x10, 0xCE, 0x10, 0x4E, 0xAB, 0xD1, 0x66, 0x29, 0xAA, 0x48, 0x62, 0x0C,  /* .<.....N..f).Hb. */
    0x12, 0x79, 0x27, 0x0C, 0xB0, 0x75, 0x0C, 0x0D, 0x37, 0xC5, 0x7F, 0xFF, 0xE3, 0x02, 0xAE, 0x7F,  /* .y'..u..7....... */
    };
static char tc_dh_map_generator[] = {
    0x7C, 0x9C, 0xBF, 0xE9, 0x8F, 0x9F, 0xBD, 0xDA, 0x8D, 0x14, 0x35, 0x06, 0xFA, 0x7D, 0x93, 0x06,  /* |.........5..}.. */
    0xF4, 0xCB, 0x17, 0xE3, 0xC7, 0x17, 0x07, 0xAF, 0xF5, 0xE1, 0xC1, 0xA1, 0x23, 0x70, 0x24, 0x96,  /* ............#p$. */
    0x84, 0xD6, 0x4E, 0xE3, 0x7A, 0xF4, 0x4B, 0x8D, 0xBD, 0x9D, 0x45, 0xBF, 0x60, 0x23, 0x91, 0x9C,  /* ..N.z.K...E.`#.. */
    0xBA, 0xA0, 0x27, 0xAB, 0x97, 0xAC, 0xC7, 0x71, 0x66, 0x6C, 0x8E, 0x98, 0xFF, 0x48, 0x33, 0x01,  /* ..'....qfl...H3. */
    0xBF, 0xA4, 0x87, 0x2D, 0xED, 0xE9, 0x03, 0x4E, 0xDF, 0xAC, 0xB7, 0x08, 0x14, 0x16, 0x6B, 0x7F,  /* ...-...N......k. */
    0x36, 0x06, 0x76, 0x82, 0x9B, 0x82, 0x6B, 0xEA, 0x57, 0x29, 0x1B, 0x5A, 0xD6, 0x9F, 0xBC, 0x84,  /* 6.v...k.W).Z.... */
    0xEF, 0x1E, 0x77, 0x90, 0x32, 0xA3, 0x05, 0x80, 0x3F, 0x74, 0x34, 0x17, 0x93, 0xE8, 0x69, 0x74,  /* ..w.2...?t4...it */
    0x2D, 0x40, 0x13, 0x25, 0xB3, 0x7E, 0xE8, 0x56, 0x5F, 0xFC, 0xDE, 0xE6, 0x18, 0x34, 0x2D, 0xC5,  /* -@.%.~.V_....4-. */
    };
static char tc_dh_pcd_priv_key[] = {
    0x4B, 0xD0, 0xE5, 0x47, 0x40, 0xF9, 0xA0, 0x28, 0xE6, 0xA5, 0x15, 0xBF, 0xDA, 0xF9, 0x67, 0x84,  /* K..G@..(......g. */
    0x8C, 0x4F, 0x5F, 0x5F, 0xFF, 0x65, 0xAA, 0x09, 0x15, 0x94, 0x7F, 0xFD, 0x1A, 0x0D, 0xF2, 0xFA,  /* .O__.e.......... */
    0x69, 0x81, 0x27, 0x1B, 0xC9, 0x05, 0xF3, 0x55, 0x14, 0x57, 0xB7, 0xE0, 0x3A, 0xC3, 0xB8, 0x06,  /* i.'....U.W..:... */
    0x6D, 0xE4, 0xAA, 0x40, 0x6C, 0x11, 0x71, 0xFB, 0x43, 0xDD, 0x93, 0x9C, 0x4B, 0xA1, 0x61, 0x75,  /* m..@l.q.C...K.au */
    0x10, 0x3B, 0xA3, 0xDE, 0xE1, 0x64, 0x19, 0xAA, 0x24, 0x81, 0x18, 0xF9, 0x0C, 0xC3, 0x6A, 0x3D,  /* .;...d..$.....j= */
    0x6F, 0x4C, 0x37, 0x36, 0x52, 0xE0, 0xC3, 0xCC, 0xE7, 0xF0, 0xF1, 0xD0, 0xC5, 0x42, 0x5B, 0x36,  /* oL76R........B[6 */
    0x00, 0xF0, 0xF0, 0xD6, 0xA6, 0x7F, 0x00, 0x4C, 0x8B, 0xBA, 0x33, 0xF2, 0xB4, 0x73, 0x3C, 0x72,  /* .......L..3..s<r */
    0x52, 0x44, 0x5C, 0x1D, 0xFC, 0x4F, 0x11, 0x07, 0x20, 0x3F, 0x71, 0xD2, 0xEF, 0xB2, 0x81, 0x61,  /* RD\..O.. ?q....a */
};
static char tc_dh_pcd_pub_key[] = {
    0x00, 0x90, 0x7D, 0x89, 0xE2, 0xD4, 0x25, 0xA1, 0x78, 0xAA, 0x81, 0xAF, 0x4A, 0x77, 0x74, 0xEC,  /* ..}...%.x...Jwt. */
    0x8E, 0x38, 0x8C, 0x11, 0x5C, 0xAE, 0x67, 0x03, 0x1E, 0x85, 0xEE, 0xCE, 0x52, 0x0B, 0xD9, 0x11,  /* .8..\.g.....R... */
    0x55, 0x1B, 0x9A, 0xE4, 0xD0, 0x43, 0x69, 0xF2, 0x9A, 0x02, 0x62, 0x6C, 0x86, 0xFB, 0xC6, 0x74,  /* U....Ci...bl...t */
    0x7C, 0xC7, 0xBC, 0x35, 0x26, 0x45, 0xB6, 0x16, 0x1A, 0x2A, 0x42, 0xD4, 0x4E, 0xDA, 0x80, 0xA0,  /* |..5&E...*B.N... */
    0x8F, 0xA8, 0xD6, 0x1B, 0x76, 0xD3, 0xA1, 0x54, 0xAD, 0x8A, 0x5A, 0x51, 0x78, 0x6B, 0x0B, 0xC0,  /* ....v..T..ZQxk.. */
    0x71, 0x47, 0x05, 0x78, 0x71, 0xA9, 0x22, 0x21, 0x2C, 0x5F, 0x67, 0xF4, 0x31, 0x73, 0x17, 0x22,  /* qG.xq."!,_g.1s." */
    0x36, 0xB7, 0x74, 0x7D, 0x16, 0x71, 0xE6, 0xD6, 0x92, 0xA3, 0xC7, 0xD4, 0x0A, 0x0C, 0x3C, 0x5C,  /* 6.t}.q........<\ */
    0xE3, 0x97, 0x54, 0x5D, 0x01, 0x5C, 0x17, 0x5E, 0xB5, 0x13, 0x05, 0x51, 0xED, 0xBC, 0x2E, 0xE5,  /* ..T].\.^...Q.... */
    0xD4,                                                                                            /* . */
};
/* Note that  Worked Example for Extended Access Control (EAC) Version 1.0
 * contains a wrong key. The Log however contains a dump of the correct value.
 * Fixed in Worked Example for Extended Access Control (EAC) Version 1.01.
 */
static char tc_dh_picc_priv_key[] = {
    0x02, 0x0F, 0x01, 0x8C, 0x72, 0x84, 0xB0, 0x47, 0xFA, 0x77, 0x21, 0xA3, 0x37, 0xEF, 0xB7, 0xAC,  /* ....r..G.w!.7... */
    0xB1, 0x44, 0x0B, 0xB3, 0x0C, 0x52, 0x52, 0xBD, 0x41, 0xC9, 0x7C, 0x30, 0xC9, 0x94, 0xBB, 0x78,  /* .D...RR.A.|0...x */
    0xE9, 0xF0, 0xC5, 0xB3, 0x27, 0x44, 0xD8, 0x40, 0x17, 0xD2, 0x1F, 0xFA, 0x68, 0x78, 0x39, 0x6A,  /* ....'D.@....hx9j */
    0x64, 0x69, 0xCA, 0x28, 0x3E, 0xF5, 0xC0, 0x00, 0xDA, 0xF7, 0xD2, 0x61, 0xA3, 0x9A, 0xB8, 0x86,  /* di.(>......a.... */
    0x0E, 0xD4, 0x61, 0x0A, 0xB5, 0x34, 0x33, 0x90, 0x89, 0x7A, 0xAB, 0x5A, 0x77, 0x87, 0xE4, 0xFA,  /* ..a..43..z.Zw... */
    0xEF, 0xA0, 0x64, 0x9C, 0x6A, 0x94, 0xFD, 0xF8, 0x2D, 0x99, 0x1E, 0x8E, 0x3F, 0xC3, 0x32, 0xF5,  /* ..d.j...-...?.2. */
    0x14, 0x27, 0x29, 0xE7, 0x04, 0x0A, 0x3F, 0x7D, 0x5A, 0x4D, 0x3C, 0xD7, 0x5C, 0xBE, 0xE1, 0xF0,  /* .')...?}ZM<.\... */
    0x43, 0xC1, 0xCA, 0xD2, 0xDD, 0x48, 0x4F, 0xEB, 0x4E, 0xD2, 0x2B, 0x59, 0x7D, 0x36, 0x68, 0x8E,  /* C....HO.N.+Y}6h. */
};
/* Note that  Worked Example for Extended Access Control (EAC) Version 1.0
 * contains a wrong key. The Log however contains a dump of the correct value.
 * Fixed in Worked Example for Extended Access Control (EAC) Version 1.01.
 */
static char tc_dh_picc_pub_key[] = {
    0x07, 0x56, 0x93, 0xD9, 0xAE, 0x94, 0x18, 0x77, 0x57, 0x3E, 0x63, 0x4B, 0x6E, 0x64, 0x4F, 0x8E,  /* .V.....wW>cKndO. */
    0x60, 0xAF, 0x17, 0xA0, 0x07, 0x6B, 0x8B, 0x12, 0x3D, 0x92, 0x01, 0x07, 0x4D, 0x36, 0x15, 0x2B,  /* `....k..=...M6.+ */
    0xD8, 0xB3, 0xA2, 0x13, 0xF5, 0x38, 0x20, 0xC4, 0x2A, 0xDC, 0x79, 0xAB, 0x5D, 0x0A, 0xEE, 0xC3,  /* .....8 .*.y.]... */
    0xAE, 0xFB, 0x91, 0x39, 0x4D, 0xA4, 0x76, 0xBD, 0x97, 0xB9, 0xB1, 0x4D, 0x0A, 0x65, 0xC1, 0xFC,  /* ...9M.v....M.e.. */
    0x71, 0xA0, 0xE0, 0x19, 0xCB, 0x08, 0xAF, 0x55, 0xE1, 0xF7, 0x29, 0x00, 0x5F, 0xBA, 0x7E, 0x3F,  /* q......U..)._.~? */
    0xA5, 0xDC, 0x41, 0x89, 0x92, 0x38, 0xA2, 0x50, 0x76, 0x7A, 0x6D, 0x46, 0xDB, 0x97, 0x40, 0x64,  /* ..A..8.PvzmF..@d */
    0x38, 0x6C, 0xD4, 0x56, 0x74, 0x35, 0x85, 0xF8, 0xE5, 0xD9, 0x0C, 0xC8, 0xB4, 0x00, 0x4B, 0x1F,  /* 8l.Vt5........K. */
    0x6D, 0x86, 0x6C, 0x79, 0xCE, 0x05, 0x84, 0xE4, 0x96, 0x87, 0xFF, 0x61, 0xBC, 0x29, 0xAE, 0xA1,  /* m.ly.......a.).. */
};
/* Note that  Worked Example for Extended Access Control (EAC) Version 1.0
 * contains a wrong key. The Log however contains a dump of the correct value.
 * Fixed in Worked Example for Extended Access Control (EAC) Version 1.01.
 */
static char tc_dh_shared_secret_k[] = {
    0x6B, 0xAB, 0xC7, 0xB3, 0xA7, 0x2B, 0xCD, 0x7E, 0xA3, 0x85, 0xE4, 0xC6, 0x2D, 0xB2, 0x62, 0x5B,  /* k....+.~....-.b[ */
    0xD8, 0x61, 0x3B, 0x24, 0x14, 0x9E, 0x14, 0x6A, 0x62, 0x93, 0x11, 0xC4, 0xCA, 0x66, 0x98, 0xE3,  /* .a;$...jb....f.. */
    0x8B, 0x83, 0x4B, 0x6A, 0x9E, 0x9C, 0xD7, 0x18, 0x4B, 0xA8, 0x83, 0x4A, 0xFF, 0x50, 0x43, 0xD4,  /* ..Kj....K..J.PC. */
    0x36, 0x95, 0x0C, 0x4C, 0x1E, 0x78, 0x32, 0x36, 0x7C, 0x10, 0xCB, 0x8C, 0x31, 0x4D, 0x40, 0xE5,  /* 6..L.x26|...1M@. */
    0x99, 0x0B, 0x0D, 0xF7, 0x01, 0x3E, 0x64, 0xB4, 0x54, 0x9E, 0x22, 0x70, 0x92, 0x3D, 0x06, 0xF0,  /* .....>d.T."p.=.. */
    0x8C, 0xFF, 0x6B, 0xD3, 0xE9, 0x77, 0xDD, 0xE6, 0xAB, 0xE4, 0xC3, 0x1D, 0x55, 0xC0, 0xFA, 0x2E,  /* ..k..w......U... */
    0x46, 0x5E, 0x55, 0x3E, 0x77, 0xBD, 0xF7, 0x5E, 0x31, 0x93, 0xD3, 0x83, 0x4F, 0xC2, 0x6E, 0x8E,  /* F^U>w..^1...O.n. */
    0xB1, 0xEE, 0x2F, 0xA1, 0xE4, 0xFC, 0x97, 0xC1, 0x8C, 0x3F, 0x6C, 0xFF, 0xFE, 0x26, 0x07, 0xFD,  /* ../......?l..&.. */
};
static char tc_dh_k_mac[] = {
    0x80, 0x5A, 0x1D, 0x27, 0xD4, 0x5A, 0x51, 0x16, 0xF7, 0x3C, 0x54, 0x46, 0x94, 0x62, 0xB7, 0xD8,  /* .Z.'.ZQ..<TF.b.. */
};
static char tc_dh_k_enc[] = {
    0x2F, 0x7F, 0x46, 0xAD, 0xCC, 0x9E, 0x7E, 0x52, 0x1B, 0x45, 0xD1, 0x92, 0xFA, 0xFA, 0x91, 0x26,  /* /.F...~R.E.....& */
};
static char tc_dh_authentication_token_picc[] = {
    0x91, 0x7F, 0x37, 0xB5, 0xC0, 0xE6, 0xD8, 0xD1,                          /* ..7..... */
};
static char tc_dh_authentication_token_pcd[] = {
    0xB4, 0x6D, 0xD9, 0xBD, 0x4D, 0x98, 0x38, 0x1F,                          /* .m..M.8. */
};
static char tc_dh_d1[] = {
    0x83, 0x0F, 0x44, 0x45, 0x54, 0x45, 0x53, 0x54, 0x43, 0x56, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x33,
};
static char tc_dh_e1[] = {
    0xB3, 0x7B, 0xB5, 0x7D, 0xA1, 0xDB, 0x37, 0xD1, 0xC4, 0x96, 0x04, 0x91,
    0x7B, 0xD6, 0x99, 0xE6, 0x1D, 0x6A, 0x30, 0x74, 0xE6, 0x9E, 0x40, 0x67,
    0xA1, 0xB3, 0x99, 0x03, 0x88, 0x23, 0x36, 0x33,
};
static const struct encryption_decryption tc_dh_enc_dec[] = {
    {
        1,
        { sizeof tc_dh_d1, tc_dh_d1, sizeof tc_dh_d1, },
        { sizeof tc_dh_e1, tc_dh_e1, sizeof tc_dh_e1, },
    },
};
static char tc_dh_ad1[] = {
    0x99, 0x02, 0x90, 0x00,
};
static char tc_dh_a1[] = {
    0xEB, 0xFF, 0x08, 0xD3, 0xB2, 0x0A, 0x04, 0x14,
};
static const struct authenticate tc_dh_pace_authenticate[] = {
    {
        2,
        { sizeof tc_dh_ad1, tc_dh_ad1, sizeof tc_dh_ad1, },
        { sizeof tc_dh_a1, tc_dh_a1, sizeof tc_dh_a1, },
    },
};
static char tc_dh_ca_picc_priv_key[] = {
    0x00, 0xD0, 0xE5, 0xA9, 0x5E, 0xCA, 0x54, 0x1D, 0xEF, 0x4F, 0xE9, 0x2B, 0x9F, 0xF9, 0x92, 0x0D,  /* ....^.T..O.+.... */
    0x49, 0xA4, 0x54, 0xC2, 0x97,                                                                    /* I.T.. */
};
static char tc_dh_ca_picc_pub_key[] = {
    0x1B, 0x33, 0x45, 0xF8, 0xDC, 0x04, 0x34, 0x1B, 0xF8, 0xB2, 0xC9, 0x7F, 0x65, 0x2F, 0xA6, 0x80,  /* .3E...4.....e/.. */
    0xE5, 0xD4, 0xFA, 0x4C, 0x14, 0x6A, 0xE4, 0xB8, 0x39, 0x43, 0x1A, 0x64, 0x4A, 0x79, 0xBC, 0x36,  /* ...L.j..9C.dJy.6 */
    0x8C, 0x48, 0x22, 0xC8, 0x9C, 0xD0, 0x18, 0xA5, 0x7F, 0x95, 0x36, 0x44, 0xBE, 0xDA, 0x67, 0x9C,  /* .H".......6D..g. */
    0x5B, 0x53, 0x29, 0x02, 0x32, 0x0E, 0x83, 0xE1, 0x3B, 0x80, 0xDE, 0xEF, 0x8C, 0x18, 0xAF, 0x3E,  /* [S).2...;......> */
    0x7D, 0x49, 0x3A, 0xE3, 0xF8, 0x81, 0x96, 0x10, 0x1B, 0x9F, 0x78, 0xEA, 0xFE, 0x4B, 0x30, 0x25,  /* }I:.......x..K0% */
    0xEF, 0x8B, 0xFF, 0x91, 0x6B, 0x2F, 0xC0, 0x2D, 0x76, 0x2D, 0x08, 0x38, 0xDE, 0xA2, 0x9C, 0x09,  /* ....k/.-v-.8.... */
    0xB4, 0x85, 0x59, 0x1C, 0x2F, 0x47, 0xF8, 0x7C, 0x71, 0xF5, 0x30, 0xBB, 0x35, 0x8F, 0x56, 0xAF,  /* ..Y./G.|q.0.5.V. */
    0x64, 0x59, 0xD8, 0x6D, 0xBF, 0x85, 0xEA, 0xF9, 0xED, 0xBD, 0x96, 0x2C, 0xD3, 0x64, 0xF7, 0xB8,  /* dY.m.......,.d.. */
};
static char tc_dh_ca_pcd_priv_key[] = {
    0x00, 0xA2, 0xCF, 0xFD, 0x06, 0xC3, 0x4A, 0xFD, 0x62, 0x2E, 0xEE, 0x0F, 0xC3, 0x1F, 0x09, 0x3F,  /* ......J.b......? */
    0xDF, 0xDA, 0x60, 0x9C, 0x67, 0x12, 0x1C, 0xAC, 0xF0, 0xA8, 0xF5, 0x22, 0x91, 0xDE, 0x68, 0x53,  /* ..`.g......"..hS */
    0xBB, 0x5C, 0x93, 0xCF, 0x76, 0x70, 0x57, 0x75, 0xEC, 0xF4, 0x08, 0xA7, 0x43, 0x02, 0x61, 0x3B,  /* .\..vpWu....C.a; */
    0xEE, 0xCB, 0x38, 0x14, 0x47, 0xD3, 0x64, 0x94, 0xC9, 0xE1, 0x89, 0x51, 0xEC, 0x17, 0x25, 0x2D,  /* ..8.G.d....Q..%- */
    0xD2, 0xA8, 0x07, 0xAA, 0xE0, 0x9F, 0xA4, 0xDC, 0x30, 0x18, 0x33, 0x39, 0x01, 0x3D, 0x9C, 0x91,  /* ........0.39.=.. */
    0x91, 0x30, 0x3C, 0xAC, 0xEE, 0x3C, 0x91, 0xE9, 0x26, 0xA3, 0x6D, 0x01, 0x4A, 0x5C, 0xFA, 0x94,  /* .0<..<..&.m.J\.. */
    0x95, 0x0C, 0xAD, 0xB3, 0x7B, 0x53, 0x4F, 0x32, 0xA9, 0xBF, 0x76, 0xB3, 0x79, 0x80, 0x97, 0x93,  /* ....{SO2..v.y... */
    0x04, 0xC5, 0x66, 0x38, 0x71, 0xBD, 0x74, 0x6E, 0xB9, 0xE9, 0x5A, 0x47, 0xCA, 0x47, 0x1B, 0x4E,  /* ..f8q.tn..ZG.G.N */
    0xDE,                                                                                            /* . */
};
static char tc_dh_ca_pcd_pub_key[] = {
    0xA2, 0x83, 0x09, 0x47, 0xA6, 0xFC, 0xAA, 0xCD, 0xE2, 0xFC, 0xB8, 0x8B, 0x29, 0xAB, 0x38, 0xE0,  /* ...G........).8. */
    0x7C, 0x34, 0x53, 0xAB, 0xC4, 0xBC, 0xB4, 0x66, 0x08, 0x7E, 0x11, 0xC7, 0x9F, 0x32, 0xA1, 0x9E,  /* |4S....f.~...2.. */
    0x6E, 0xF2, 0x2B, 0xE1, 0x08, 0xF8, 0xDD, 0x18, 0xFE, 0x82, 0x49, 0xC9, 0x60, 0x95, 0x15, 0x11,  /* n.+.......I.`... */
    0x20, 0x0D, 0xC9, 0x85, 0xAA, 0x3E, 0xC0, 0xCC, 0xAD, 0x59, 0xA5, 0xF9, 0xBB, 0xCC, 0x33, 0xEE,  /* ....>...Y....3. */
    0x5F, 0x15, 0x77, 0xE2, 0x03, 0x30, 0xB4, 0xDD, 0x10, 0xEB, 0x06, 0xB7, 0x40, 0x27, 0x7C, 0x97,  /* _.w..0......@'|. */
    0xA1, 0x89, 0x18, 0x0E, 0xDE, 0x52, 0xBE, 0xE9, 0xD4, 0x29, 0xF1, 0x0F, 0xB7, 0x7F, 0x18, 0x0F,  /* .....R...)...... */
    0x05, 0xD6, 0xA9, 0x9C, 0x49, 0x9C, 0xB5, 0xE1, 0xEC, 0xEE, 0xB8, 0xE9, 0x22, 0x84, 0xF6, 0x6E,  /* ....I......."..n */
    0xA9, 0x84, 0x79, 0x67, 0x4C, 0xE7, 0x3F, 0x53, 0xC5, 0x67, 0xA0, 0x3B, 0x0D, 0x29, 0x78, 0x33,  /* ..ygL.?S.g.;.)x3 */
};
static char tc_dh_ca_nonce[] = {
    0x1A, 0x1A, 0xC1, 0xAB, 0x04, 0xD9, 0x69, 0x65,                          /* ......ie */
};
static char tc_dh_ca_picc_token[] = {
    0x75, 0x68, 0x91, 0x58, 0x0F, 0x70, 0x16, 0x4D,                          /* uh.X.p.M */
};
static char tc_dh_ca_shared_secret_k[] = {
    0x2A, 0x4F, 0xC2, 0xD1, 0x14, 0x90, 0xDF, 0x47, 0x73, 0xDD, 0xFA, 0x6F, 0xF7, 0x05, 0x55, 0x7C,  /* *O.....Gs..o..U| */
    0x51, 0xF4, 0xAD, 0x45, 0x33, 0xE8, 0xD8, 0xA6, 0x6A, 0x30, 0x01, 0xBF, 0xDD, 0x27, 0x1D, 0xB8,  /* Q..E3...j0...'.. */
    0x7B, 0xE3, 0xC1, 0xCA, 0xC2, 0x2A, 0x05, 0xE8, 0xAF, 0x1A, 0x06, 0x6D, 0xD0, 0x29, 0x8E, 0x75,  /* {....*.....m.).u */
    0xDB, 0x92, 0x8A, 0xAF, 0xDF, 0x00, 0xEB, 0x4B, 0xFB, 0x1B, 0xD1, 0x2F, 0x37, 0x23, 0x13, 0xC1,  /* .......K.../7#.. */
    0xCA, 0x64, 0x90, 0x56, 0x51, 0x73, 0x05, 0x63, 0x85, 0x15, 0xD5, 0xA4, 0xFB, 0xE0, 0xAC, 0x59,  /* .d.VQs.c.......Y */
    0xBD, 0xC8, 0x0E, 0x8C, 0x5A, 0x5F, 0x46, 0x25, 0x4D, 0x23, 0x19, 0x16, 0xEA, 0x77, 0xF8, 0x0A,  /* ....Z_F%M#...w.. */
    0xC5, 0x8E, 0x6B, 0x63, 0xA2, 0x61, 0x98, 0xEE, 0x43, 0x87, 0xF1, 0x09, 0x81, 0xE8, 0xE4, 0x6F,  /* ..kc.a..C......o */
    0xFB, 0xA2, 0x37, 0x90, 0xC8, 0x1E, 0x67, 0x93, 0x63, 0xC5, 0x89, 0x58, 0x7D, 0x30, 0xBB, 0xB3,  /* ..7...g.c..X}0.. */
};
static char tc_dh_ca_k_mac[] = {
    0x06, 0x67, 0x21, 0xF3, 0xEB, 0xEA, 0x78, 0x7B, 0x4F, 0x1C, 0x6D, 0xCA, 0x43, 0x1E, 0xC2, 0x29,  /* .g!...x{O.m.C..) */
};
static char tc_dh_ca_k_enc[] = {
    0xE8, 0x13, 0xD0, 0x9D, 0xF6, 0x9F, 0xFC, 0xC6, 0x37, 0x63, 0xC5, 0x37, 0xF8, 0x28, 0x98, 0x5C,  /* ........7c.7.(.\ */
};
static char tc_dh_ta_pcd_key[] = {
    0x30, 0x82, 0x02, 0x76, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,  /* 0..v...0...*.H.. */
    0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82, 0x02, 0x60, 0x30, 0x82, 0x02, 0x5c, 0x02, 0x01,  /* .........`0..\.. */
    0x00, 0x02, 0x81, 0x81, 0x00, 0x9f, 0x7e, 0xf6, 0x8e, 0x15, 0x3d, 0xb4, 0xfd, 0x10, 0x84, 0xdd,  /* ......~...=..... */
    0xed, 0xbe, 0xae, 0x84, 0x2c, 0x55, 0x6d, 0x41, 0x9f, 0xcb, 0x5e, 0xf6, 0x21, 0xaa, 0x37, 0x51,  /* ....,UmA..^.!.7Q */
    0xf0, 0xfc, 0x0c, 0xfd, 0x71, 0x4f, 0xc0, 0xe7, 0x68, 0x86, 0x6b, 0x3f, 0x44, 0xe2, 0x72, 0x5a,  /* ....qO..h.k?D.rZ */
    0xf0, 0x35, 0x1a, 0x97, 0xed, 0xb1, 0xba, 0x88, 0xdf, 0xdd, 0x9b, 0x4d, 0x81, 0xd4, 0x08, 0xfe,  /* .5.........M.... */
    0x07, 0x63, 0x34, 0x6a, 0x77, 0x2c, 0xf6, 0x46, 0x16, 0x46, 0x5c, 0x8f, 0xd9, 0x71, 0xb7, 0x75,  /* .c4jw,.F.F\..q.u */
    0xd2, 0xe1, 0x34, 0x26, 0xc5, 0xbc, 0x11, 0x89, 0x47, 0x95, 0xc5, 0xad, 0x2c, 0x3e, 0x42, 0x68,  /* ..4&....G...,>Bh */
    0x37, 0xf3, 0xa1, 0x01, 0x9f, 0xe9, 0x51, 0x24, 0xea, 0x5d, 0x43, 0x3e, 0x90, 0x6d, 0x79, 0x93,  /* 7.....Q$.]C>.my. */
    0x49, 0x63, 0x21, 0xef, 0xcb, 0xdb, 0xc3, 0x2d, 0x93, 0xc0, 0x68, 0x0b, 0x45, 0xf3, 0xb8, 0xf6,  /* Ic!....-..h.E... */
    0x4a, 0x5d, 0xaf, 0xcf, 0xb9, 0x02, 0x03, 0x01, 0x00, 0x01, 0x02, 0x81, 0x81, 0x00, 0x80, 0xcf,  /* J].............. */
    0xe8, 0x4e, 0x4d, 0xe7, 0x5d, 0x41, 0xb8, 0xd2, 0xa5, 0xed, 0xf9, 0xbf, 0x36, 0x2e, 0x1b, 0xfc,  /* .NM.]A......6... */
    0x36, 0x30, 0xd8, 0xed, 0xe3, 0xb8, 0xc2, 0xa1, 0x15, 0x03, 0x9c, 0x7c, 0xdd, 0x3f, 0x63, 0x5a,  /* 60.........|.?cZ */
    0xf7, 0x49, 0x9a, 0xae, 0x09, 0x16, 0x18, 0x10, 0xa5, 0x53, 0x34, 0x80, 0x3a, 0xcd, 0x94, 0x69,  /* .I.......S4.:..i */
    0x83, 0x5b, 0xd1, 0xbd, 0x38, 0xb6, 0xae, 0xc9, 0x68, 0x68, 0x0c, 0xfd, 0xeb, 0x30, 0xf8, 0x39,  /* .[..8...hh...0.9 */
    0x3a, 0xc3, 0x9f, 0x3a, 0x98, 0xf5, 0x47, 0x80, 0xff, 0x15, 0x11, 0x93, 0xf6, 0x39, 0x80, 0xd3,  /* :..:..G......9.. */
    0xef, 0x60, 0x6a, 0xb5, 0x98, 0x11, 0x0e, 0x7a, 0x44, 0xe9, 0xf0, 0x71, 0x71, 0xc7, 0xc5, 0xf1,  /* .`j....zD..qq... */
    0x95, 0xfc, 0xc1, 0x84, 0xf5, 0x34, 0x5d, 0x0c, 0x2d, 0x09, 0xc9, 0x2f, 0x05, 0xbc, 0xc1, 0x1c,  /* .....4].-../.... */
    0xa7, 0x09, 0x6e, 0x76, 0x52, 0x24, 0xc1, 0xfd, 0x3a, 0x4d, 0xd5, 0xc4, 0xc8, 0x25, 0x02, 0x41,  /* ..nvR$..:M...%.A */
    0x00, 0xd1, 0xa6, 0x33, 0x59, 0xe8, 0x74, 0x23, 0xbb, 0xe5, 0x6e, 0x9b, 0x6b, 0xc9, 0x9a, 0x7d,  /* ...3Y.t#..n.k..} */
    0x42, 0x73, 0xd3, 0xb1, 0xee, 0xd2, 0xba, 0x8e, 0x1a, 0xaa, 0xb2, 0xe1, 0x0f, 0x31, 0x93, 0xc3,  /* Bs...........1.. */
    0xfc, 0x48, 0xa0, 0x6f, 0xfe, 0xfc, 0x4e, 0x48, 0x21, 0x90, 0xa2, 0xda, 0xc2, 0x10, 0xe4, 0x69,  /* .H.o..NH!......i */
    0x19, 0xc5, 0x9b, 0x7c, 0xc1, 0xbe, 0xa4, 0xad, 0x05, 0xbd, 0xcd, 0xc5, 0x28, 0xe4, 0x95, 0xb4,  /* ...|........(... */
    0x03, 0x02, 0x41, 0x00, 0xc2, 0xc2, 0x2b, 0x95, 0x1e, 0x07, 0x6a, 0x2f, 0xa9, 0xf6, 0x06, 0xf6,  /* ..A...+...j/.... */
    0x6d, 0xb1, 0x5a, 0x65, 0xae, 0x78, 0xdb, 0x73, 0x8c, 0x04, 0x41, 0x1e, 0xd8, 0x78, 0xd0, 0x25,  /* m.Ze.x.s..A..x.% */
    0xfa, 0xe3, 0xf0, 0x7b, 0xcc, 0x8f, 0xdb, 0x3d, 0x1c, 0x73, 0xa2, 0xe4, 0x7f, 0xb6, 0x00, 0x16,  /* ...{...=.s...... */
    0x58, 0x64, 0xbc, 0xc6, 0x88, 0x28, 0x7e, 0x2c, 0xd8, 0xc6, 0x0a, 0x37, 0xda, 0x2a, 0xcd, 0x67,  /* Xd...(~,...7.*.g */
    0x37, 0xab, 0x26, 0x93, 0x02, 0x40, 0x2f, 0x08, 0xb9, 0x76, 0x8f, 0x47, 0x17, 0x09, 0x3d, 0x84,  /* 7.&..@/..v.G..=. */
    0xde, 0x0b, 0x85, 0x94, 0x4a, 0x4b, 0xd1, 0x4e, 0x8a, 0x19, 0xc9, 0x5f, 0x1f, 0x28, 0x6e, 0x69,  /* ....JK.N..._.(ni */
    0x69, 0x42, 0x8e, 0x5d, 0xd7, 0x57, 0x4f, 0x37, 0xe7, 0x48, 0x6e, 0x74, 0x65, 0x8c, 0x5a, 0x35,  /* iB.].WO7.Hnte.Z5 */
    0x9f, 0x1c, 0x70, 0xcb, 0xbd, 0xa4, 0x91, 0x87, 0xbb, 0xaa, 0x7f, 0xc9, 0xd4, 0x23, 0x78, 0x9c,  /* ..p..........#x. */
    0xa4, 0x81, 0xf3, 0x1e, 0x7e, 0xdf, 0x02, 0x40, 0x46, 0x86, 0x71, 0xa9, 0xc9, 0xf9, 0xfd, 0x65,  /* ....~..@F.q....e */
    0xcd, 0x32, 0x1e, 0x4b, 0x56, 0x65, 0x86, 0xca, 0x93, 0x67, 0x88, 0x70, 0x4b, 0x1f, 0xaf, 0x45,  /* .2.KVe...g.pK..E */
    0x30, 0x36, 0x6f, 0x89, 0x91, 0x54, 0xc1, 0x5c, 0xd3, 0x32, 0xb4, 0xbd, 0x68, 0x74, 0x27, 0x64,  /* 06o..T.\.2..ht'd */
    0x7f, 0x2a, 0xb4, 0xd5, 0x62, 0xa4, 0xf8, 0x07, 0x1c, 0x7f, 0x2d, 0x51, 0xa8, 0x1d, 0xee, 0xcd,  /* .*..b.....-Q.... */
    0x52, 0xdf, 0x75, 0x0a, 0x0f, 0x8a, 0x96, 0xa9, 0x02, 0x40, 0x06, 0x3b, 0x7a, 0x95, 0x47, 0x78,  /* R.u......@.;z.Gx */
    0x5a, 0x8c, 0x0b, 0x31, 0x89, 0x30, 0xd3, 0x6b, 0x81, 0xa4, 0xca, 0x86, 0x2c, 0xbb, 0x14, 0x7a,  /* Z..1.0.k....,..z */
    0xfb, 0x36, 0x78, 0x02, 0xb7, 0xca, 0x6f, 0xf2, 0x89, 0xf3, 0xc9, 0x2b, 0x9e, 0xce, 0x05, 0x4c,  /* .6x...o....+...L */
    0xa5, 0xa7, 0x78, 0x54, 0x2a, 0x06, 0xc9, 0xc2, 0x5b, 0xd4, 0xa8, 0xe3, 0x77, 0xdc, 0xb2, 0xab,  /* ..xT*...[...w... */
    0xe5, 0xf4, 0x23, 0x7c, 0x4e, 0xe8, 0x03, 0x25, 0xa6, 0xf3,                                      /* ..#|N..%.. */
};
static char tc_dh_ta_nonce[] = {
    0xFC, 0x21, 0x0C, 0x17, 0x4A, 0x80, 0x1D, 0x46,                          /* .!..J..F */
};
static char tc_dh_ta_pcd_signature[] = {
    0x37, 0xC7, 0x36, 0x0C, 0x55, 0x57, 0x9D, 0xE2, 0x8A, 0x41, 0xDF, 0x8A, 0x1D, 0x17, 0x03, 0x2C,  /* 7.6.UW...A....., */
    0x4B, 0x9F, 0x90, 0xDF, 0x7F, 0x64, 0x6D, 0xF3, 0x3F, 0x77, 0xEF, 0x97, 0x6A, 0xE4, 0xC4, 0x5C,  /* K....dm.?w..j..\ */
    0xDB, 0xA3, 0x77, 0x63, 0x99, 0x64, 0x8C, 0x75, 0x50, 0x43, 0x79, 0x0F, 0x0F, 0x19, 0xBF, 0xC1,  /* ..wc.d.uPCy..... */
    0x46, 0xD0, 0x53, 0xB7, 0x70, 0x36, 0xCA, 0x9B, 0x24, 0xBF, 0xB4, 0xB7, 0x93, 0x4F, 0xAC, 0x04,  /* F.S.p6..$....O.. */
    0x26, 0x71, 0xE2, 0x3B, 0xE6, 0xE3, 0xCA, 0x2B, 0x24, 0x6C, 0xE2, 0x06, 0x6C, 0x1F, 0x05, 0x50,  /* &q.;...+$l..l..P */
    0x38, 0x34, 0x18, 0x19, 0x2F, 0x06, 0xD6, 0xAD, 0x6C, 0x0A, 0xCB, 0x1E, 0x70, 0xBD, 0xEA, 0xD6,  /* 84../...l...p... */
    0x0E, 0xA1, 0x0B, 0x31, 0xE0, 0x74, 0xF4, 0x9E, 0x5E, 0x8A, 0xEB, 0x54, 0xE6, 0xF4, 0xF8, 0x3C,  /* ...1.t..^..T...< */
    0xE9, 0x98, 0x95, 0x79, 0x53, 0xF9, 0x7F, 0x87, 0x00, 0x58, 0x01, 0x3D, 0x0E, 0x74, 0x5D, 0x0A,  /* ...yS....X.=.t]. */
};
static char tc_dh_ta_cert[] = {
    0x7f, 0x21, 0x82, 0x01, 0x6a, 0x7f, 0x4e, 0x81, 0xe2, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x0d, 0x44,  /* .!..j.N.._)..B.D */
    0x45, 0x54, 0x45, 0x53, 0x54, 0x44, 0x56, 0x44, 0x45, 0x30, 0x31, 0x39, 0x7f, 0x49, 0x81, 0x94,  /* ETESTDVDE019.I.. */
    0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x01, 0x01, 0x81, 0x81, 0x80, 0x9f,  /* ................ */
    0x7e, 0xf6, 0x8e, 0x15, 0x3d, 0xb4, 0xfd, 0x10, 0x84, 0xdd, 0xed, 0xbe, 0xae, 0x84, 0x2c, 0x55,  /* ~...=.........,U */
    0x6d, 0x41, 0x9f, 0xcb, 0x5e, 0xf6, 0x21, 0xaa, 0x37, 0x51, 0xf0, 0xfc, 0x0c, 0xfd, 0x71, 0x4f,  /* mA..^.!.7Q....qO */
    0xc0, 0xe7, 0x68, 0x86, 0x6b, 0x3f, 0x44, 0xe2, 0x72, 0x5a, 0xf0, 0x35, 0x1a, 0x97, 0xed, 0xb1,  /* ..h.k?D.rZ.5.... */
    0xba, 0x88, 0xdf, 0xdd, 0x9b, 0x4d, 0x81, 0xd4, 0x08, 0xfe, 0x07, 0x63, 0x34, 0x6a, 0x77, 0x2c,  /* .....M.....c4jw, */
    0xf6, 0x46, 0x16, 0x46, 0x5c, 0x8f, 0xd9, 0x71, 0xb7, 0x75, 0xd2, 0xe1, 0x34, 0x26, 0xc5, 0xbc,  /* .F.F\..q.u..4&.. */
    0x11, 0x89, 0x47, 0x95, 0xc5, 0xad, 0x2c, 0x3e, 0x42, 0x68, 0x37, 0xf3, 0xa1, 0x01, 0x9f, 0xe9,  /* ..G...,>Bh7..... */
    0x51, 0x24, 0xea, 0x5d, 0x43, 0x3e, 0x90, 0x6d, 0x79, 0x93, 0x49, 0x63, 0x21, 0xef, 0xcb, 0xdb,  /* Q$.]C>.my.Ic!... */
    0xc3, 0x2d, 0x93, 0xc0, 0x68, 0x0b, 0x45, 0xf3, 0xb8, 0xf6, 0x4a, 0x5d, 0xaf, 0xcf, 0xb9, 0x82,  /* .-..h.E...J].... */
    0x03, 0x01, 0x00, 0x01, 0x5f, 0x20, 0x0d, 0x44, 0x45, 0x54, 0x45, 0x53, 0x54, 0x41, 0x54, 0x44,  /* ...._ .DETESTATD */
    0x45, 0x30, 0x31, 0x39, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01,  /* E019.L.......... */
    0x02, 0x02, 0x53, 0x05, 0x00, 0x00, 0x00, 0x01, 0x10, 0x5f, 0x25, 0x06, 0x01, 0x00, 0x00, 0x03,  /* ..S......_%..... */
    0x02, 0x04, 0x5f, 0x24, 0x06, 0x01, 0x00, 0x00, 0x04, 0x02, 0x04, 0x5f, 0x37, 0x81, 0x80, 0x8c,  /* .._$......._7... */
    0xb1, 0x61, 0x26, 0xa1, 0xfd, 0xbb, 0x82, 0x48, 0xc8, 0x8b, 0xdb, 0x1f, 0xb1, 0x19, 0x9c, 0x3f,  /* .a&....H.......? */
    0x25, 0x38, 0x56, 0xfe, 0x10, 0x83, 0x5f, 0x7b, 0xff, 0x62, 0xa3, 0x0b, 0xd2, 0x81, 0xb8, 0xa1,  /* %8V..._{.b...... */
    0xf0, 0xfe, 0x03, 0x81, 0xa5, 0xb0, 0xa4, 0x26, 0x51, 0xf7, 0x7d, 0xf7, 0x21, 0x52, 0x21, 0xf0,  /* .......&Q.}.!R!. */
    0xed, 0xe4, 0x88, 0xe6, 0x89, 0xea, 0x45, 0xce, 0xe2, 0x0b, 0x19, 0xc7, 0xb1, 0xd1, 0xed, 0xb6,  /* ......E......... */
    0xac, 0x21, 0xf3, 0x40, 0x88, 0x81, 0x9f, 0x6f, 0xd5, 0xdc, 0x33, 0x31, 0x09, 0xe1, 0x5a, 0x15,  /* .!.@...o..31..Z. */
    0xdf, 0xf6, 0x85, 0xa2, 0xb6, 0x9d, 0x17, 0xd5, 0xe2, 0x3d, 0xaf, 0xe3, 0x63, 0xa8, 0xe7, 0x63,  /* .........=..c..c */
    0x31, 0xcc, 0x25, 0xb9, 0x13, 0xfb, 0x6e, 0xd8, 0x30, 0xeb, 0x45, 0x7a, 0xd0, 0xa6, 0x73, 0x96,  /* 1.%...n.0.Ez..s. */
    0xa1, 0x90, 0xca, 0xe3, 0x9c, 0xc6, 0xc2, 0xe4, 0x67, 0x1e, 0x60, 0x52, 0xd3, 0xc2, 0x2d,        /* ........g.`R..- */
};
static char tc_dh_cvca[] = {
    0x7f, 0x21, 0x82, 0x01, 0x6a, 0x7f, 0x4e, 0x81, 0xe2, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x0f, 0x44,  /* .!..j.N.._)..B.D */
    0x45, 0x54, 0x45, 0x53, 0x54, 0x43, 0x56, 0x43, 0x41, 0x30, 0x30, 0x30, 0x30, 0x33, 0x7f, 0x49,  /* ETESTCVCA00003.I */
    0x81, 0x94, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x01, 0x01, 0x81, 0x81,  /* ................ */
    0x80, 0xb5, 0xad, 0xa4, 0x40, 0xf2, 0x64, 0x80, 0x3d, 0xbe, 0xec, 0x57, 0xab, 0x1f, 0xfb, 0xd1,  /* ....@.d.=..W.... */
    0x2b, 0x42, 0x68, 0x74, 0x23, 0xec, 0x29, 0x9d, 0xd5, 0x2a, 0x6b, 0xb0, 0x34, 0x51, 0x93, 0xd1,  /* +Bht#.)..*k.4Q.. */
    0xe2, 0xa5, 0xbb, 0xc1, 0x19, 0x55, 0x90, 0x8a, 0x08, 0x76, 0x0f, 0x40, 0x0f, 0xa0, 0x3d, 0x56,  /* .....U...v.@..=V */
    0x09, 0x7b, 0x5f, 0x21, 0x65, 0x64, 0x8d, 0x85, 0x07, 0x1a, 0x80, 0xa1, 0xb6, 0x6f, 0xfa, 0x93,  /* .{_!ed.......o.. */
    0x21, 0x52, 0x6d, 0xda, 0x41, 0xe1, 0x46, 0xd5, 0x32, 0x99, 0x1c, 0xa0, 0xc1, 0xdc, 0xfc, 0x8d,  /* !Rm.A.F.2....... */
    0x0b, 0xbe, 0x29, 0x8c, 0x1f, 0x0e, 0xdb, 0x57, 0x40, 0x0f, 0xae, 0xd6, 0x7e, 0x13, 0x5d, 0x54,  /* ..)....W@...~.]T */
    0x24, 0x71, 0xc8, 0x2e, 0x35, 0x1b, 0x06, 0xd4, 0x3c, 0x94, 0x66, 0x04, 0xfc, 0xbd, 0x50, 0x45,  /* $q..5...<.f...PE */
    0x05, 0x6c, 0x4b, 0x8a, 0x7b, 0x98, 0xec, 0x38, 0x72, 0xe4, 0x1b, 0xb3, 0xd7, 0xa5, 0x91, 0xcc,  /* .lK.{..8r....... */
    0xad, 0x82, 0x03, 0x01, 0x00, 0x01, 0x5f, 0x20, 0x0f, 0x44, 0x45, 0x54, 0x45, 0x53, 0x54, 0x43,  /* ......_ .DETESTC */
    0x56, 0x43, 0x41, 0x30, 0x30, 0x30, 0x30, 0x33, 0x7f, 0x4c, 0x0e, 0x06, 0x09, 0x04, 0x00, 0x7f,  /* VCA00003.L...... */
    0x00, 0x07, 0x03, 0x01, 0x02, 0x01, 0x53, 0x01, 0xc3, 0x5f, 0x25, 0x06, 0x01, 0x00, 0x00, 0x03,  /* ......S.._%..... */
    0x02, 0x04, 0x5f, 0x24, 0x06, 0x01, 0x01, 0x00, 0x03, 0x01, 0x09, 0x5f, 0x37, 0x81, 0x80, 0xa2,  /* .._$......._7... */
    0x7e, 0xd8, 0x30, 0xe9, 0x90, 0x82, 0xe5, 0x81, 0x30, 0x68, 0xb9, 0xb2, 0x16, 0x62, 0x44, 0x4e,  /* ~.0.....0h...bDN */
    0xd1, 0xc8, 0x71, 0x3a, 0xba, 0x71, 0x98, 0x5b, 0x52, 0x88, 0xf3, 0x76, 0x42, 0x36, 0x08, 0xb8,  /* ..q:.q.[R..vB6.. */
    0xee, 0xc8, 0x25, 0x17, 0x38, 0x62, 0xf8, 0xdf, 0x9f, 0x25, 0x9b, 0xcd, 0x61, 0x1c, 0x96, 0xa2,  /* ..%.8b...%..a... */
    0xdb, 0xa8, 0x6a, 0xc6, 0xc4, 0x8a, 0x33, 0x10, 0xcf, 0xd6, 0xfa, 0xde, 0x9b, 0x4b, 0xd0, 0x2b,  /* ..j...3......K.+ */
    0xb5, 0x20, 0x36, 0x7a, 0xd9, 0x0b, 0x37, 0x9f, 0x6d, 0x35, 0xad, 0x01, 0xd4, 0x06, 0xf8, 0xcb,  /* . 6z..7.m5...... */
    0x1d, 0x94, 0xd0, 0x25, 0x73, 0x0a, 0x98, 0xd7, 0xdf, 0xbe, 0x3b, 0x0f, 0x74, 0x18, 0x35, 0x31,  /* ...%s.....;.t.51 */
    0xde, 0x6e, 0x69, 0x9d, 0x23, 0x65, 0x2c, 0xe3, 0x90, 0x67, 0x63, 0x5a, 0x8d, 0x17, 0x6c, 0x66,  /* .ni.#e,..gcZ..lf */
    0x06, 0x4b, 0x79, 0xb7, 0x27, 0x1a, 0x0e, 0xe6, 0x00, 0xba, 0xc9, 0x4a, 0xd5, 0x1e, 0x4f,        /* .Ky.'......J..O */
};
static char tc_dh_dv_cert[] = {
    0x7f, 0x21, 0x82, 0x01, 0x6c, 0x7f, 0x4e, 0x81, 0xe4, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x0f, 0x44,  /* .!..l.N.._)..B.D */
    0x45, 0x54, 0x45, 0x53, 0x54, 0x43, 0x56, 0x43, 0x41, 0x30, 0x30, 0x30, 0x30, 0x33, 0x7f, 0x49,  /* ETESTCVCA00003.I */
    0x81, 0x94, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x01, 0x01, 0x81, 0x81,  /* ................ */
    0x80, 0xa0, 0x8c, 0x4d, 0x11, 0xd6, 0x99, 0xf4, 0x25, 0xb0, 0xe7, 0x43, 0xbb, 0xa4, 0xf2, 0x19,  /* ...M....%..C.... */
    0x6e, 0x05, 0xbc, 0x9e, 0xf2, 0x4f, 0x53, 0xa6, 0x74, 0x42, 0x90, 0xe6, 0x55, 0x6e, 0x83, 0xe9,  /* n....OS.tB..Un.. */
    0x05, 0x77, 0xa9, 0x30, 0xec, 0x31, 0x4a, 0x4f, 0x9f, 0x03, 0x33, 0xa0, 0xa0, 0x19, 0x93, 0x11,  /* .w.0.1JO..3..... */
    0x0e, 0xc6, 0x34, 0x86, 0xdf, 0x60, 0x7f, 0xd7, 0xb3, 0x04, 0x74, 0x79, 0xb0, 0xec, 0x09, 0x04,  /* ..4..`....ty.... */
    0xac, 0xf8, 0xb6, 0x26, 0x5c, 0xd0, 0xab, 0xc3, 0x53, 0x8f, 0x4d, 0x72, 0x39, 0x5d, 0xd5, 0xf1,  /* ...&\...S.Mr9].. */
    0xe7, 0xa1, 0x08, 0x18, 0xa7, 0xfa, 0xa0, 0x1d, 0x25, 0xff, 0x25, 0xbc, 0x6b, 0xf1, 0x9c, 0xe8,  /* ........%.%.k... */
    0x6a, 0x20, 0x82, 0x33, 0xc5, 0x43, 0x7f, 0xf9, 0x90, 0xfe, 0x94, 0xd1, 0xc2, 0x5d, 0x59, 0xbe,  /* j .3.C.......]Y. */
    0xdb, 0x6a, 0xe7, 0x9e, 0x4a, 0x76, 0xde, 0x22, 0x79, 0xfc, 0xd6, 0xa5, 0xa3, 0xd6, 0x6f, 0xf5,  /* .j..Jv."y.....o. */
    0xf9, 0x82, 0x03, 0x01, 0x00, 0x01, 0x5f, 0x20, 0x0d, 0x44, 0x45, 0x54, 0x45, 0x53, 0x54, 0x44,  /* ......_ .DETESTD */
    0x56, 0x44, 0x45, 0x30, 0x31, 0x39, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07,  /* VDE019.L........ */
    0x03, 0x01, 0x02, 0x02, 0x53, 0x05, 0x80, 0x1f, 0xff, 0xff, 0x10, 0x5f, 0x25, 0x06, 0x01, 0x00,  /* ....S......_%... */
    0x00, 0x03, 0x02, 0x04, 0x5f, 0x24, 0x06, 0x01, 0x00, 0x00, 0x04, 0x02, 0x04, 0x5f, 0x37, 0x81,  /* ...._$......._7. */
    0x80, 0x6b, 0x95, 0x0f, 0x1f, 0xa8, 0xfe, 0xf8, 0x61, 0xee, 0xa7, 0x57, 0x65, 0xc2, 0x80, 0x5d,  /* .k......a..We..] */
    0x79, 0xbb, 0x5d, 0x0d, 0x60, 0x87, 0x8e, 0x93, 0x0a, 0x8c, 0x17, 0xd3, 0xf9, 0x2d, 0xcc, 0x2b,  /* y.].`........-.+ */
    0xe9, 0x54, 0x7d, 0x31, 0xe4, 0x12, 0x6b, 0x75, 0x10, 0xc3, 0x59, 0x27, 0xe8, 0x24, 0xbd, 0x0c,  /* .T}1..ku..Y'.$.. */
    0x64, 0xdc, 0x33, 0x96, 0xf5, 0x39, 0x2a, 0xac, 0xf6, 0xf6, 0x49, 0x9f, 0x1d, 0x88, 0xca, 0xfa,  /* d.3..9*...I..... */
    0xd9, 0x4c, 0xa6, 0x16, 0x24, 0xb6, 0x63, 0x7c, 0x75, 0x1b, 0xd0, 0x35, 0xfc, 0x08, 0x4b, 0xb8,  /* .L..$.c|u..5..K. */
    0x9f, 0x50, 0xa9, 0x00, 0xec, 0xc1, 0x80, 0x71, 0x25, 0x8b, 0x31, 0x6b, 0xdf, 0x3a, 0xf9, 0xd6,  /* .P.....q%.1k.:.. */
    0x10, 0x92, 0xa1, 0x50, 0x05, 0x64, 0x29, 0xe0, 0x2d, 0x1a, 0x70, 0xdf, 0xc1, 0x1e, 0x77, 0xd0,  /* ...P.d).-.p...w. */
    0xfb, 0xba, 0x00, 0xcb, 0x70, 0x0a, 0x63, 0x20, 0x98, 0x05, 0x96, 0x8d, 0xbd, 0x17, 0xd3, 0x6e,  /* ....p.c .......n */
    0x75,                                                                                            /* u */
};

static char ef_cardaccess_old_testcard[] = {
    0x31, 0x82, 0x02, 0x64, 0x30, 0x0d, 0x06, 0x08, 0x04, 0x00, 0x7f,
    0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x01, 0x02, 0x30, 0x0f, 0x06,
    0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x03, 0x02, 0x02,
    0x02, 0x01, 0x02, 0x30, 0x0f, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00,
    0x07, 0x02, 0x02, 0x04, 0x02, 0x02, 0x02, 0x01, 0x01, 0x30, 0x2f,
    0x06, 0x08, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x06, 0x16,
    0x23, 0x41, 0x77, 0x54, 0x20, 0x65, 0x50, 0x41, 0x20, 0x2d, 0x20,
    0x42, 0x44, 0x72, 0x20, 0x47, 0x6d, 0x62, 0x48, 0x20, 0x2d, 0x20,
    0x54, 0x65, 0x73, 0x74, 0x6b, 0x61, 0x72, 0x74, 0x65, 0x20, 0x76,
    0x31, 0x2e, 0x30, 0x30, 0x81, 0xfe, 0x06, 0x09, 0x04, 0x00, 0x7f,
    0x00, 0x07, 0x02, 0x02, 0x03, 0x02, 0x30, 0x81, 0xf0, 0x06, 0x0b,
    0x04, 0x00, 0x7f, 0x00, 0x07, 0x01, 0x01, 0x05, 0x02, 0x02, 0x02,
    0x30, 0x81, 0xe0, 0x02, 0x01, 0x01, 0x30, 0x2c, 0x06, 0x07, 0x2a,
    0x86, 0x48, 0xce, 0x3d, 0x01, 0x01, 0x02, 0x21, 0x00, 0xa9, 0xfb,
    0x57, 0xdb, 0xa1, 0xee, 0xa9, 0xbc, 0x3e, 0x66, 0x0a, 0x90, 0x9d,
    0x83, 0x8d, 0x72, 0x6e, 0x3b, 0xf6, 0x23, 0xd5, 0x26, 0x20, 0x28,
    0x20, 0x13, 0x48, 0x1d, 0x1f, 0x6e, 0x53, 0x77, 0x30, 0x44, 0x04,
    0x20, 0xa9, 0xfb, 0x57, 0xdb, 0xa1, 0xee, 0xa9, 0xbc, 0x3e, 0x66,
    0x0a, 0x90, 0x9d, 0x83, 0x8d, 0x72, 0x6e, 0x3b, 0xf6, 0x23, 0xd5,
    0x26, 0x20, 0x28, 0x20, 0x13, 0x48, 0x1d, 0x1f, 0x6e, 0x53, 0x74,
    0x04, 0x20, 0x66, 0x2c, 0x61, 0xc4, 0x30, 0xd8, 0x4e, 0xa4, 0xfe,
    0x66, 0xa7, 0x73, 0x3d, 0x0b, 0x76, 0xb7, 0xbf, 0x93, 0xeb, 0xc4,
    0xaf, 0x2f, 0x49, 0x25, 0x6a, 0xe5, 0x81, 0x01, 0xfe, 0xe9, 0x2b,
    0x04, 0x04, 0x41, 0x04, 0xa3, 0xe8, 0xeb, 0x3c, 0xc1, 0xcf, 0xe7,
    0xb7, 0x73, 0x22, 0x13, 0xb2, 0x3a, 0x65, 0x61, 0x49, 0xaf, 0xa1,
    0x42, 0xc4, 0x7a, 0xaf, 0xbc, 0x2b, 0x79, 0xa1, 0x91, 0x56, 0x2e,
    0x13, 0x05, 0xf4, 0x2d, 0x99, 0x6c, 0x82, 0x34, 0x39, 0xc5, 0x6d,
    0x7f, 0x7b, 0x22, 0xe1, 0x46, 0x44, 0x41, 0x7e, 0x69, 0xbc, 0xb6,
    0xde, 0x39, 0xd0, 0x27, 0x00, 0x1d, 0xab, 0xe8, 0xf3, 0x5b, 0x25,
    0xc9, 0xbe, 0x02, 0x21, 0x00, 0xa9, 0xfb, 0x57, 0xdb, 0xa1, 0xee,
    0xa9, 0xbc, 0x3e, 0x66, 0x0a, 0x90, 0x9d, 0x83, 0x8d, 0x71, 0x8c,
    0x39, 0x7a, 0xa3, 0xb5, 0x61, 0xa6, 0xf7, 0x90, 0x1e, 0x0e, 0x82,
    0x97, 0x48, 0x56, 0xa7, 0x02, 0x01, 0x01, 0x30, 0x81, 0xfe, 0x06,
    0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x04, 0x02, 0x30,
    0x81, 0xf0, 0x06, 0x0b, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x01, 0x01,
    0x05, 0x02, 0x02, 0x02, 0x30, 0x81, 0xe0, 0x02, 0x01, 0x01, 0x30,
    0x2c, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x01, 0x01, 0x02,
    0x21, 0x00, 0xa9, 0xfb, 0x57, 0xdb, 0xa1, 0xee, 0xa9, 0xbc, 0x3e,
    0x66, 0x0a, 0x90, 0x9d, 0x83, 0x8d, 0x72, 0x6e, 0x3b, 0xf6, 0x23,
    0xd5, 0x26, 0x20, 0x28, 0x20, 0x13, 0x48, 0x1d, 0x1f, 0x6e, 0x53,
    0x77, 0x30, 0x44, 0x04, 0x20, 0xa9, 0xfb, 0x57, 0xdb, 0xa1, 0xee,
    0xa9, 0xbc, 0x3e, 0x66, 0x0a, 0x90, 0x9d, 0x83, 0x8d, 0x72, 0x6e,
    0x3b, 0xf6, 0x23, 0xd5, 0x26, 0x20, 0x28, 0x20, 0x13, 0x48, 0x1d,
    0x1f, 0x6e, 0x53, 0x74, 0x04, 0x20, 0x66, 0x2c, 0x61, 0xc4, 0x30,
    0xd8, 0x4e, 0xa4, 0xfe, 0x66, 0xa7, 0x73, 0x3d, 0x0b, 0x76, 0xb7,
    0xbf, 0x93, 0xeb, 0xc4, 0xaf, 0x2f, 0x49, 0x25, 0x6a, 0xe5, 0x81,
    0x01, 0xfe, 0xe9, 0x2b, 0x04, 0x04, 0x41, 0x04, 0xa3, 0xe8, 0xeb,
    0x3c, 0xc1, 0xcf, 0xe7, 0xb7, 0x73, 0x22, 0x13, 0xb2, 0x3a, 0x65,
    0x61, 0x49, 0xaf, 0xa1, 0x42, 0xc4, 0x7a, 0xaf, 0xbc, 0x2b, 0x79,
    0xa1, 0x91, 0x56, 0x2e, 0x13, 0x05, 0xf4, 0x2d, 0x99, 0x6c, 0x82,
    0x34, 0x39, 0xc5, 0x6d, 0x7f, 0x7b, 0x22, 0xe1, 0x46, 0x44, 0x41,
    0x7e, 0x69, 0xbc, 0xb6, 0xde, 0x39, 0xd0, 0x27, 0x00, 0x1d, 0xab,
    0xe8, 0xf3, 0x5b, 0x25, 0xc9, 0xbe, 0x02, 0x21, 0x00, 0xa9, 0xfb,
    0x57, 0xdb, 0xa1, 0xee, 0xa9, 0xbc, 0x3e, 0x66, 0x0a, 0x90, 0x9d,
    0x83, 0x8d, 0x71, 0x8c, 0x39, 0x7a, 0xa3, 0xb5, 0x61, 0xa6, 0xf7,
    0x90, 0x1e, 0x0e, 0x82, 0x97, 0x48, 0x56, 0xa7, 0x02, 0x01, 0x01
};
static char ef_cardaccess_new_testcard[] = {
    0x31, 0x81, 0x82, 0x30, 0x0D, 0x06, 0x08, 0x04, 0x00, 0x7F, 0x00,
    0x07, 0x02, 0x02, 0x02, 0x02, 0x01, 0x02, 0x30, 0x12, 0x06, 0x0A,
    0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, 0x02, 0x02, 0x02,
    0x01, 0x02, 0x02, 0x01, 0x41, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00,
    0x7F, 0x00, 0x07, 0x02, 0x02, 0x04, 0x02, 0x02, 0x02, 0x01, 0x02,
    0x02, 0x01, 0x0D, 0x30, 0x1C, 0x06, 0x09, 0x04, 0x00, 0x7F, 0x00,
    0x07, 0x02, 0x02, 0x03, 0x02, 0x30, 0x0C, 0x06, 0x07, 0x04, 0x00,
    0x7F, 0x00, 0x07, 0x01, 0x02, 0x02, 0x01, 0x0D, 0x02, 0x01, 0x41,
    0x30, 0x2B, 0x06, 0x08, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02,
    0x06, 0x16, 0x1F, 0x65, 0x50, 0x41, 0x20, 0x2D, 0x20, 0x42, 0x44,
    0x72, 0x20, 0x47, 0x6D, 0x62, 0x48, 0x20, 0x2D, 0x20, 0x54, 0x65,
    0x73, 0x74, 0x6B, 0x61, 0x72, 0x74, 0x65, 0x20, 0x76, 0x32, 0x2E,
    0x30, 0x04, 0x49, 0x17, 0x15, 0x41, 0x19, 0x28, 0x80, 0x0A, 0x01,
    0xB4, 0x21, 0xFA, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x10, 0x10, 0x29, 0x10, 0x10
    /*Note: The new test cards contain garbage at the end of their EF.CardAccess
     * (the last 38 Bytes), which we ignore */
};
static char ef_cardaccess_dan[] = {
    0x31, 0x81, 0xB3, 0x30, 0x0D, 0x06, 0x08, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, /*1..0............*/
    0x01, 0x02, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, 0x02, 0x02, /*..0.............*/
    0x02, 0x01, 0x02, 0x02, 0x01, 0x41, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, /*.....A0.........*/
    0x02, 0x03, 0x02, 0x02, 0x02, 0x01, 0x02, 0x02, 0x01, 0x45, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, /*.........E0.....*/
    0x7F, 0x00, 0x07, 0x02, 0x02, 0x04, 0x02, 0x02, 0x02, 0x01, 0x02, 0x02, 0x01, 0x0D, 0x30, 0x1C, /*..............0.*/
    0x06, 0x09, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, 0x02, 0x30, 0x0C, 0x06, 0x07, 0x04, /*...........0....*/
    0x00, 0x7F, 0x00, 0x07, 0x01, 0x02, 0x02, 0x01, 0x0D, 0x02, 0x01, 0x41, 0x30, 0x1C, 0x06, 0x09, /*...........A0...*/
    0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, 0x02, 0x30, 0x0C, 0x06, 0x07, 0x04, 0x00, 0x7F, /*.........0......*/
    0x00, 0x07, 0x01, 0x02, 0x02, 0x01, 0x0D, 0x02, 0x01, 0x45, 0x30, 0x2A, 0x06, 0x08, 0x04, 0x00, /*.........E0*....*/
    0x7F, 0x00, 0x07, 0x02, 0x02, 0x06, 0x16, 0x1E, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F, 0x2F, 0x62, /*........http://b*/
    0x73, 0x69, 0x2E, 0x62, 0x75, 0x6E, 0x64, 0x2E, 0x64, 0x65, 0x2F, 0x63, 0x69, 0x66, 0x2F, 0x6E, /*si.bund.de/cif/n*/
    0x70, 0x61, 0x2E, 0x78, 0x6D, 0x6C,                                                             /*pa.xml*/
};

static char ef_cardaccess_slowenia[] = {
   0x31, 0x5B, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02,
   0x02, 0x03, 0x02, 0x01, 0x02, 0x01, 0x01, 0x02, 0x01, 0x03, 0x30, 0x1C,
   0x06, 0x09, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, 0x02, 0x30,
   0x0C, 0x06, 0x07, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x01, 0x02, 0x02, 0x01,
   0x0D, 0x02, 0x01, 0x03, 0x30, 0x16, 0x06, 0x08, 0x04, 0x00, 0x7F, 0x00,
   0x07, 0x02, 0x02, 0x02, 0x02, 0x01, 0x01, 0x30, 0x07, 0x04, 0x02, 0x01,
   0x1A, 0x04, 0x01, 0x1A, 0x30, 0x0F, 0x06, 0x07, 0x28, 0x81, 0xC1, 0x17,
   0x03, 0x00, 0x09, 0x02, 0x01, 0x01, 0x02, 0x01, 0x06, 0x62, 0x82
};

static const BUF_MEM ef_cardaccess_files[] = {
    { sizeof ef_cardaccess_new_testcard, ef_cardaccess_new_testcard, sizeof ef_cardaccess_new_testcard, },
    { sizeof ef_cardaccess_old_testcard, ef_cardaccess_old_testcard, sizeof ef_cardaccess_old_testcard, },
    { sizeof ef_cardaccess_dan, ef_cardaccess_dan, sizeof ef_cardaccess_dan, },
    { sizeof ef_cardaccess_slowenia, ef_cardaccess_slowenia, sizeof ef_cardaccess_slowenia, },
};


static char ef_cardsecurity_dan[] = {
    0x30, 0x82, 0x05, 0xA0, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x02, 0xA0, /*0.....*.H.......*/
    0x82, 0x05, 0x91, 0x30, 0x82, 0x05, 0x8D, 0x02, 0x01, 0x03, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09, /*...0......1.0...*/
    0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x82, 0x01, 0x48, 0x06, /*`.H.e......0..H.*/
    0x08, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x03, 0x02, 0x01, 0xA0, 0x82, 0x01, 0x3A, 0x04, 0x82, 0x01, /*............:...*/
    0x36, 0x31, 0x82, 0x01, 0x32, 0x30, 0x0D, 0x06, 0x08, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, /*61..20..........*/
    0x02, 0x02, 0x01, 0x02, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x03, /*....0...........*/
    0x02, 0x02, 0x02, 0x01, 0x02, 0x02, 0x01, 0x41, 0x30, 0x12, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, /*.......A0.......*/
    0x07, 0x02, 0x02, 0x04, 0x02, 0x02, 0x02, 0x01, 0x02, 0x02, 0x01, 0x0D, 0x30, 0x17, 0x06, 0x0A, /*............0...*/
    0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x05, 0x02, 0x03, 0x30, 0x09, 0x02, 0x01, 0x01, 0x02, /*..........0.....*/
    0x01, 0x43, 0x01, 0x01, 0xFF, 0x30, 0x17, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, /*.C...0..........*/
    0x05, 0x02, 0x03, 0x30, 0x09, 0x02, 0x01, 0x01, 0x02, 0x01, 0x44, 0x01, 0x01, 0x00, 0x30, 0x19, /*...0......D...0.*/
    0x06, 0x09, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x05, 0x02, 0x30, 0x0C, 0x06, 0x07, 0x04, /*...........0....*/
    0x00, 0x7F, 0x00, 0x07, 0x01, 0x02, 0x02, 0x01, 0x0D, 0x30, 0x1C, 0x06, 0x09, 0x04, 0x00, 0x7F, /*.........0......*/
    0x00, 0x07, 0x02, 0x02, 0x03, 0x02, 0x30, 0x0C, 0x06, 0x07, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x01, /*......0.........*/
    0x02, 0x02, 0x01, 0x0D, 0x02, 0x01, 0x41, 0x30, 0x2A, 0x06, 0x08, 0x04, 0x00, 0x7F, 0x00, 0x07, /*......A0*.......*/
    0x02, 0x02, 0x06, 0x16, 0x1E, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F, 0x2F, 0x62, 0x73, 0x69, 0x2E, /*.....http://bsi.*/
    0x62, 0x75, 0x6E, 0x64, 0x2E, 0x64, 0x65, 0x2F, 0x63, 0x69, 0x66, 0x2F, 0x6E, 0x70, 0x61, 0x2E, /*bund.de/cif/npa.*/
    0x78, 0x6D, 0x6C, 0x30, 0x62, 0x06, 0x09, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x01, 0x02, /*xml0b...........*/
    0x30, 0x52, 0x30, 0x0C, 0x06, 0x07, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x01, 0x02, 0x02, 0x01, 0x0D, /*0R0.............*/
    0x03, 0x42, 0x00, 0x04, 0x92, 0x5D, 0xB4, 0xE1, 0x7A, 0xDE, 0x58, 0x20, 0x9F, 0x96, 0xFA, 0xA0, /*.B...]..z.X ....*/
    0x7F, 0x1F, 0x8A, 0x22, 0x3F, 0x82, 0x3F, 0x96, 0xCC, 0x5D, 0x78, 0xCB, 0xEF, 0x5D, 0x17, 0x42, /*..."?.?..]x..].B*/
    0x20, 0x88, 0xFD, 0xD5, 0x8E, 0x56, 0xBC, 0x42, 0x50, 0xDE, 0x33, 0x46, 0xB3, 0xC8, 0x32, 0xCA, /* ....V.BP.3F..2.*/
    0xE4, 0x86, 0x35, 0xFB, 0x6C, 0x43, 0x78, 0x9D, 0xE8, 0xB3, 0x10, 0x2F, 0x43, 0x93, 0xB4, 0x18, /*..5.lCx..../C...*/
    0xE2, 0x4A, 0x13, 0xD9, 0x02, 0x01, 0x41, 0xA0, 0x82, 0x03, 0x1C, 0x30, 0x82, 0x03, 0x18, 0x30, /*.J....A....0...0*/
    0x82, 0x02, 0xBC, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x02, 0x01, 0x5C, 0x30, 0x0C, 0x06, 0x08, /*...........\0...*/
    0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x05, 0x00, 0x30, 0x4F, 0x31, 0x0B, 0x30, 0x09, /**.H.=.....0O1.0.*/
    0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x44, 0x45, 0x31, 0x0D, 0x30, 0x0B, 0x06, 0x03, 0x55, /*..U....DE1.0...U*/
    0x04, 0x0A, 0x0C, 0x04, 0x62, 0x75, 0x6E, 0x64, 0x31, 0x0C, 0x30, 0x0A, 0x06, 0x03, 0x55, 0x04, /*....bund1.0...U.*/
    0x0B, 0x0C, 0x03, 0x62, 0x73, 0x69, 0x31, 0x0C, 0x30, 0x0A, 0x06, 0x03, 0x55, 0x04, 0x05, 0x13, /*...bsi1.0...U...*/
    0x03, 0x30, 0x31, 0x33, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0C, 0x63, /*.0131.0...U....c*/
    0x73, 0x63, 0x61, 0x2D, 0x67, 0x65, 0x72, 0x6D, 0x61, 0x6E, 0x79, 0x30, 0x1E, 0x17, 0x0D, 0x31, /*sca-germany0...1*/
    0x30, 0x31, 0x30, 0x30, 0x35, 0x31, 0x31, 0x30, 0x37, 0x35, 0x32, 0x5A, 0x17, 0x0D, 0x32, 0x31, /*01005110752Z..21*/
    0x30, 0x34, 0x30, 0x35, 0x30, 0x38, 0x33, 0x39, 0x34, 0x37, 0x5A, 0x30, 0x47, 0x31, 0x0B, 0x30, /*0405083947Z0G1.0*/
    0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x44, 0x45, 0x31, 0x1D, 0x30, 0x1B, 0x06, 0x03, /*...U....DE1.0...*/
    0x55, 0x04, 0x0A, 0x0C, 0x14, 0x42, 0x75, 0x6E, 0x64, 0x65, 0x73, 0x64, 0x72, 0x75, 0x63, 0x6B, /*U....Bundesdruck*/
    0x65, 0x72, 0x65, 0x69, 0x20, 0x47, 0x6D, 0x62, 0x48, 0x31, 0x0C, 0x30, 0x0A, 0x06, 0x03, 0x55, /*erei GmbH1.0...U*/
    0x04, 0x05, 0x13, 0x03, 0x30, 0x37, 0x34, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x03, /*....0741.0...U..*/
    0x0C, 0x02, 0x44, 0x53, 0x30, 0x82, 0x01, 0x13, 0x30, 0x81, 0xD4, 0x06, 0x07, 0x2A, 0x86, 0x48, /*..DS0...0....*.H*/
    0xCE, 0x3D, 0x02, 0x01, 0x30, 0x81, 0xC8, 0x02, 0x01, 0x01, 0x30, 0x28, 0x06, 0x07, 0x2A, 0x86, /*.=..0.....0(..*.*/
    0x48, 0xCE, 0x3D, 0x01, 0x01, 0x02, 0x1D, 0x00, 0xD7, 0xC1, 0x34, 0xAA, 0x26, 0x43, 0x66, 0x86, /*H.=.......4.&Cf.*/
    0x2A, 0x18, 0x30, 0x25, 0x75, 0xD1, 0xD7, 0x87, 0xB0, 0x9F, 0x07, 0x57, 0x97, 0xDA, 0x89, 0xF5, /**.0%u......W....*/
    0x7E, 0xC8, 0xC0, 0xFF, 0x30, 0x3C, 0x04, 0x1C, 0x68, 0xA5, 0xE6, 0x2C, 0xA9, 0xCE, 0x6C, 0x1C, /*~...0<..h..,..l.*/
    0x29, 0x98, 0x03, 0xA6, 0xC1, 0x53, 0x0B, 0x51, 0x4E, 0x18, 0x2A, 0xD8, 0xB0, 0x04, 0x2A, 0x59, /*)....S.QN.*...*Y*/
    0xCA, 0xD2, 0x9F, 0x43, 0x04, 0x1C, 0x25, 0x80, 0xF6, 0x3C, 0xCF, 0xE4, 0x41, 0x38, 0x87, 0x07, /*...C..%..<..A8..*/
    0x13, 0xB1, 0xA9, 0x23, 0x69, 0xE3, 0x3E, 0x21, 0x35, 0xD2, 0x66, 0xDB, 0xB3, 0x72, 0x38, 0x6C, /*...#i.>!5.f..r8l*/
    0x40, 0x0B, 0x04, 0x39, 0x04, 0x0D, 0x90, 0x29, 0xAD, 0x2C, 0x7E, 0x5C, 0xF4, 0x34, 0x08, 0x23, /*@..9...).,~\.4.#*/
    0xB2, 0xA8, 0x7D, 0xC6, 0x8C, 0x9E, 0x4C, 0xE3, 0x17, 0x4C, 0x1E, 0x6E, 0xFD, 0xEE, 0x12, 0xC0, /*..}...L..L.n....*/
    0x7D, 0x58, 0xAA, 0x56, 0xF7, 0x72, 0xC0, 0x72, 0x6F, 0x24, 0xC6, 0xB8, 0x9E, 0x4E, 0xCD, 0xAC, /*}X.V.r.ro$...N..*/
    0x24, 0x35, 0x4B, 0x9E, 0x99, 0xCA, 0xA3, 0xF6, 0xD3, 0x76, 0x14, 0x02, 0xCD, 0x02, 0x1D, 0x00, /*$5K......v......*/
    0xD7, 0xC1, 0x34, 0xAA, 0x26, 0x43, 0x66, 0x86, 0x2A, 0x18, 0x30, 0x25, 0x75, 0xD0, 0xFB, 0x98, /*..4.&Cf.*.0%u...*/
    0xD1, 0x16, 0xBC, 0x4B, 0x6D, 0xDE, 0xBC, 0xA3, 0xA5, 0xA7, 0x93, 0x9F, 0x02, 0x01, 0x01, 0x03, /*...Km...........*/
    0x3A, 0x00, 0x04, 0x3D, 0x6A, 0x7C, 0x2A, 0x6F, 0x20, 0x5F, 0x83, 0x9B, 0x04, 0x14, 0xEC, 0x58, /*:..=j|*o _.....X*/
    0xC6, 0xC7, 0x1B, 0x75, 0xF5, 0x15, 0xDE, 0xC3, 0xAE, 0x73, 0x3B, 0x5F, 0x47, 0x88, 0xDD, 0xC8, /*...u.....s;_G...*/
    0x15, 0xF0, 0x5B, 0xC1, 0xF6, 0x53, 0x8F, 0xD9, 0x69, 0x54, 0xE1, 0xF8, 0x40, 0xA2, 0xE2, 0x18, /*..[..S..iT..@...*/
    0x99, 0x62, 0xCC, 0xAA, 0x14, 0x90, 0x08, 0x24, 0xC7, 0xDD, 0xB9, 0xA3, 0x81, 0xD1, 0x30, 0x81, /*.b.....$......0.*/
    0xCE, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x07, /*.0...U..........*/
    0x80, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x60, 0x44, /*.0...U.#..0...`D*/
    0xF2, 0x45, 0xF2, 0xE0, 0x71, 0xD4, 0xD5, 0x64, 0xF4, 0xE5, 0x77, 0xD6, 0x36, 0x69, 0xDB, 0xEB, /*.E..q..d..w.6i..*/
    0x18, 0x59, 0x30, 0x41, 0x06, 0x03, 0x55, 0x1D, 0x20, 0x04, 0x3A, 0x30, 0x38, 0x30, 0x36, 0x06, /*.Y0A..U. .:0806.*/
    0x09, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x03, 0x01, 0x01, 0x01, 0x30, 0x29, 0x30, 0x27, 0x06, 0x08, /*..........0)0'..*/
    0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x01, 0x16, 0x1B, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F, /*+.........http:/*/
    0x2F, 0x77, 0x77, 0x77, 0x2E, 0x62, 0x73, 0x69, 0x2E, 0x62, 0x75, 0x6E, 0x64, 0x2E, 0x64, 0x65, /*/www.bsi.bund.de*/
    0x2F, 0x63, 0x73, 0x63, 0x61, 0x30, 0x2B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, /*/csca0+..*.H....*/
    0x09, 0x15, 0x04, 0x1E, 0x04, 0x1C, 0x31, 0x2E, 0x32, 0x2E, 0x32, 0x37, 0x36, 0x2E, 0x30, 0x2E, /*......1.2.276.0.*/
    0x38, 0x30, 0x2E, 0x31, 0x2E, 0x31, 0x32, 0x2E, 0x30, 0x2E, 0x32, 0x30, 0x2E, 0x35, 0x2E, 0x31, /*80.1.12.0.20.5.1*/
    0x2E, 0x30, 0x30, 0x2B, 0x06, 0x03, 0x55, 0x1D, 0x10, 0x04, 0x24, 0x30, 0x22, 0x80, 0x0F, 0x32, /*.00+..U...$0"..2*/
    0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x35, 0x31, 0x31, 0x30, 0x37, 0x35, 0x32, 0x5A, 0x81, 0x0F, /*0101005110752Z..*/
    0x32, 0x30, 0x31, 0x31, 0x30, 0x32, 0x30, 0x35, 0x30, 0x39, 0x33, 0x39, 0x34, 0x37, 0x5A, 0x30, /*20110205093947Z0*/
    0x0C, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x05, 0x00, 0x03, 0x48, 0x00, /*...*.H.=......H.*/
    0x30, 0x45, 0x02, 0x20, 0x13, 0xE9, 0xE1, 0x7A, 0x9E, 0xFE, 0x8B, 0xD7, 0xD7, 0x27, 0x62, 0x92, /*0E. ...z.....'b.*/
    0x30, 0x5B, 0xCC, 0xC3, 0x2B, 0x70, 0xC2, 0xB7, 0x60, 0x40, 0xF4, 0x88, 0x30, 0x66, 0x62, 0x26, /*0[..+p..`@..0fb&*/
    0xCD, 0x6A, 0x4B, 0xF4, 0x02, 0x21, 0x00, 0x87, 0xF4, 0x71, 0xE2, 0x44, 0x35, 0xB4, 0xC3, 0x4A, /*.jK..!...q.D5..J*/
    0xF3, 0x57, 0x30, 0x94, 0xFB, 0x1F, 0x1C, 0x2A, 0x48, 0xB1, 0x3E, 0xE5, 0xED, 0x67, 0xF1, 0x72, /*.W0....*H.>..g.r*/
    0x6D, 0xCF, 0x56, 0xE3, 0x84, 0xE3, 0x6F, 0x31, 0x82, 0x01, 0x09, 0x30, 0x82, 0x01, 0x05, 0x02, /*m.V...o1...0....*/
    0x01, 0x01, 0x30, 0x55, 0x30, 0x4F, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, /*..0U0O1.0...U...*/
    0x02, 0x44, 0x45, 0x31, 0x0D, 0x30, 0x0B, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x04, 0x62, 0x75, /*.DE1.0...U....bu*/
    0x6E, 0x64, 0x31, 0x0C, 0x30, 0x0A, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x03, 0x62, 0x73, 0x69, /*nd1.0...U....bsi*/
    0x31, 0x0C, 0x30, 0x0A, 0x06, 0x03, 0x55, 0x04, 0x05, 0x13, 0x03, 0x30, 0x31, 0x33, 0x31, 0x15, /*1.0...U....0131.*/
    0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0C, 0x63, 0x73, 0x63, 0x61, 0x2D, 0x67, 0x65, /*0...U....csca-ge*/
    0x72, 0x6D, 0x61, 0x6E, 0x79, 0x02, 0x02, 0x01, 0x5C, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, /*rmany...\0...`.H*/
    0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0xA0, 0x4A, 0x30, 0x17, 0x06, 0x09, 0x2A, 0x86, /*.e.......J0...*.*/
    0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x03, 0x31, 0x0A, 0x06, 0x08, 0x04, 0x00, 0x7F, 0x00, 0x07, /*H......1........*/
    0x03, 0x02, 0x01, 0x30, 0x2F, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x04, /*...0/..*.H......*/
    0x31, 0x22, 0x04, 0x20, 0xEF, 0x0F, 0xDA, 0x94, 0x2E, 0x5A, 0x0F, 0x6F, 0xC9, 0xC5, 0x46, 0xEE, /*1". .....Z.o..F.*/
    0x01, 0xF9, 0x10, 0x31, 0x43, 0x64, 0x30, 0xF7, 0x5E, 0x9D, 0x36, 0x54, 0xD3, 0x69, 0x30, 0x9E, /*...1Cd0.^.6T.i0.*/
    0x8B, 0xE7, 0x17, 0x48, 0x30, 0x0C, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, /*...H0...*.H.=...*/
    0x05, 0x00, 0x04, 0x40, 0x30, 0x3E, 0x02, 0x1D, 0x00, 0x89, 0x75, 0x92, 0x5B, 0xE1, 0x31, 0xB7, /*...@0>....u.[.1.*/
    0x7C, 0x95, 0x8C, 0x3E, 0xCB, 0x2A, 0x5C, 0x67, 0xFC, 0x5C, 0xE3, 0x1C, 0xBD, 0x01, 0x41, 0xE3, /*|..>.*\g.\....A.*/
    0x4B, 0xC7, 0xF0, 0xA4, 0x47, 0x02, 0x1D, 0x00, 0xCC, 0x65, 0xE6, 0x2D, 0xDC, 0xF2, 0x93, 0x96, /*K...G....e.-....*/
    0x4B, 0x22, 0xD7, 0xB5, 0x10, 0xD7, 0x81, 0x88, 0x07, 0xC8, 0x95, 0x96, 0xBD, 0x34, 0xD8, 0xF9, /*K"...........4..*/
    0xBB, 0x4C, 0x05, 0x27,                                                                         /*.L.'*/
};
static char csca_germany_013_self_signed_cer[] = {
   0x30, 0x82, 0x03, 0x7f, 0x30, 0x82, 0x03, 0x24, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x02, 0x01,  /* 0...0..$........ */
   0x1d, 0x30, 0x0c, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x05, 0x00, 0x30,  /* .0...*.H.=.....0 */
   0x4f, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x44, 0x45, 0x31, 0x0d,  /* O1.0...U....DE1. */
   0x30, 0x0b, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x04, 0x62, 0x75, 0x6e, 0x64, 0x31, 0x0c, 0x30,  /* 0...U....bund1.0 */
   0x0a, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x03, 0x62, 0x73, 0x69, 0x31, 0x0c, 0x30, 0x0a, 0x06,  /* ...U....bsi1.0.. */
   0x03, 0x55, 0x04, 0x05, 0x13, 0x03, 0x30, 0x31, 0x33, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55,  /* .U....0131.0...U */
   0x04, 0x03, 0x0c, 0x0c, 0x63, 0x73, 0x63, 0x61, 0x2d, 0x67, 0x65, 0x72, 0x6d, 0x61, 0x6e, 0x79,  /* ....csca-germany */
   0x30, 0x1e, 0x17, 0x0d, 0x30, 0x38, 0x30, 0x32, 0x32, 0x36, 0x31, 0x33, 0x34, 0x33, 0x30, 0x34,  /* 0...080226134304 */
   0x5a, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x38, 0x32, 0x36, 0x31, 0x32, 0x33, 0x35, 0x35, 0x36, 0x5a,  /* Z..210826123556Z */
   0x30, 0x4f, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x44, 0x45, 0x31,  /* 0O1.0...U....DE1 */
   0x0d, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x04, 0x62, 0x75, 0x6e, 0x64, 0x31, 0x0c,  /* .0...U....bund1. */
   0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x03, 0x62, 0x73, 0x69, 0x31, 0x0c, 0x30, 0x0a,  /* 0...U....bsi1.0. */
   0x06, 0x03, 0x55, 0x04, 0x05, 0x13, 0x03, 0x30, 0x31, 0x33, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03,  /* ..U....0131.0... */
   0x55, 0x04, 0x03, 0x0c, 0x0c, 0x63, 0x73, 0x63, 0x61, 0x2d, 0x67, 0x65, 0x72, 0x6d, 0x61, 0x6e,  /* U....csca-german */
   0x79, 0x30, 0x82, 0x01, 0x33, 0x30, 0x81, 0xec, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02,  /* y0..30....*.H.=. */
   0x01, 0x30, 0x81, 0xe0, 0x02, 0x01, 0x01, 0x30, 0x2c, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d,  /* .0.....0,..*.H.= */
   0x01, 0x01, 0x02, 0x21, 0x00, 0xa9, 0xfb, 0x57, 0xdb, 0xa1, 0xee, 0xa9, 0xbc, 0x3e, 0x66, 0x0a,  /* ...!...W.....>f. */
   0x90, 0x9d, 0x83, 0x8d, 0x72, 0x6e, 0x3b, 0xf6, 0x23, 0xd5, 0x26, 0x20, 0x28, 0x20, 0x13, 0x48,  /* ....rn;.#.& ( .H */
   0x1d, 0x1f, 0x6e, 0x53, 0x77, 0x30, 0x44, 0x04, 0x20, 0x7d, 0x5a, 0x09, 0x75, 0xfc, 0x2c, 0x30,  /* ..nSw0D. }Z.u.,0 */
   0x57, 0xee, 0xf6, 0x75, 0x30, 0x41, 0x7a, 0xff, 0xe7, 0xfb, 0x80, 0x55, 0xc1, 0x26, 0xdc, 0x5c,  /* W..u0Az....U.&.\ */
   0x6c, 0xe9, 0x4a, 0x4b, 0x44, 0xf3, 0x30, 0xb5, 0xd9, 0x04, 0x20, 0x26, 0xdc, 0x5c, 0x6c, 0xe9,  /* l.JKD.0... &.\l. */
   0x4a, 0x4b, 0x44, 0xf3, 0x30, 0xb5, 0xd9, 0xbb, 0xd7, 0x7c, 0xbf, 0x95, 0x84, 0x16, 0x29, 0x5c,  /* JKD.0....|....)\ */
   0xf7, 0xe1, 0xce, 0x6b, 0xcc, 0xdc, 0x18, 0xff, 0x8c, 0x07, 0xb6, 0x04, 0x41, 0x04, 0x8b, 0xd2,  /* ...k........A... */
   0xae, 0xb9, 0xcb, 0x7e, 0x57, 0xcb, 0x2c, 0x4b, 0x48, 0x2f, 0xfc, 0x81, 0xb7, 0xaf, 0xb9, 0xde,  /* ...~W.,KH/...... */
   0x27, 0xe1, 0xe3, 0xbd, 0x23, 0xc2, 0x3a, 0x44, 0x53, 0xbd, 0x9a, 0xce, 0x32, 0x62, 0x54, 0x7e,  /* '...#.:DS...2bT~ */
   0xf8, 0x35, 0xc3, 0xda, 0xc4, 0xfd, 0x97, 0xf8, 0x46, 0x1a, 0x14, 0x61, 0x1d, 0xc9, 0xc2, 0x77,  /* .5......F..a...w */
   0x45, 0x13, 0x2d, 0xed, 0x8e, 0x54, 0x5c, 0x1d, 0x54, 0xc7, 0x2f, 0x04, 0x69, 0x97, 0x02, 0x21,  /* E.-..T\.T./.i..! */
   0x00, 0xa9, 0xfb, 0x57, 0xdb, 0xa1, 0xee, 0xa9, 0xbc, 0x3e, 0x66, 0x0a, 0x90, 0x9d, 0x83, 0x8d,  /* ...W.....>f..... */
   0x71, 0x8c, 0x39, 0x7a, 0xa3, 0xb5, 0x61, 0xa6, 0xf7, 0x90, 0x1e, 0x0e, 0x82, 0x97, 0x48, 0x56,  /* q.9z..a.......HV */
   0xa7, 0x02, 0x01, 0x01, 0x03, 0x42, 0x00, 0x04, 0x4a, 0x94, 0x49, 0x81, 0x77, 0x9d, 0xdf, 0x1d,  /* .....B..J.I.w... */
   0xa5, 0xe7, 0xc5, 0x27, 0xe2, 0x7d, 0x24, 0x71, 0xa9, 0x28, 0xeb, 0x4d, 0x7b, 0x67, 0x75, 0xae,  /* ...'.}$q.(.M{gu. */
   0x09, 0x0a, 0x51, 0x45, 0x19, 0x9b, 0xd4, 0x7e, 0xa0, 0x81, 0xe5, 0x5e, 0xd4, 0xa4, 0x3f, 0x60,  /* ..QE...~...^..?` */
   0x7c, 0x6a, 0x50, 0xee, 0x36, 0x41, 0x8a, 0x87, 0xff, 0xcd, 0xa6, 0x10, 0x39, 0xca, 0x95, 0x76,  /* |jP.6A......9..v */
   0x7d, 0xae, 0xca, 0xc3, 0x44, 0x3f, 0xe3, 0x2c, 0xa3, 0x82, 0x01, 0x10, 0x30, 0x82, 0x01, 0x0c,  /* }...D?.,....0... */
   0x30, 0x36, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x2f, 0x30, 0x2d, 0x81, 0x18, 0x63, 0x73, 0x63,  /* 06..U.../0-..csc */
   0x61, 0x2d, 0x67, 0x65, 0x72, 0x6d, 0x61, 0x6e, 0x79, 0x40, 0x62, 0x73, 0x69, 0x2e, 0x62, 0x75,  /* a-germany@bsi.bu */
   0x6e, 0x64, 0x2e, 0x64, 0x65, 0x86, 0x11, 0x66, 0x61, 0x78, 0x3a, 0x2b, 0x34, 0x39, 0x32, 0x32,  /* nd.de..fax:+4922 */
   0x38, 0x39, 0x35, 0x38, 0x32, 0x37, 0x32, 0x32, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01,  /* 895827220...U... */
   0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04,  /* ........0...U... */
   0x16, 0x04, 0x14, 0x60, 0x44, 0xf2, 0x45, 0xf2, 0xe0, 0x71, 0xd4, 0xd5, 0x64, 0xf4, 0xe5, 0x77,  /* ...`D.E..q..d..w */
   0xd6, 0x36, 0x69, 0xdb, 0xeb, 0x18, 0x59, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18,  /* .6i...Y0...U.#.. */
   0x30, 0x16, 0x80, 0x14, 0x60, 0x44, 0xf2, 0x45, 0xf2, 0xe0, 0x71, 0xd4, 0xd5, 0x64, 0xf4, 0xe5,  /* 0...`D.E..q..d.. */
   0x77, 0xd6, 0x36, 0x69, 0xdb, 0xeb, 0x18, 0x59, 0x30, 0x41, 0x06, 0x03, 0x55, 0x1d, 0x20, 0x04,  /* w.6i...Y0A..U. . */
   0x3a, 0x30, 0x38, 0x30, 0x36, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x01, 0x01,  /* :0806........... */
   0x30, 0x29, 0x30, 0x27, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x01, 0x16, 0x1b,  /* 0)0'..+......... */
   0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x62, 0x73, 0x69, 0x2e, 0x62,  /* http://www.bsi.b */
   0x75, 0x6e, 0x64, 0x2e, 0x64, 0x65, 0x2f, 0x63, 0x73, 0x63, 0x61, 0x30, 0x12, 0x06, 0x03, 0x55,  /* und.de/csca0...U */
   0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30,  /* .......0.......0 */
   0x2b, 0x06, 0x03, 0x55, 0x1d, 0x10, 0x04, 0x24, 0x30, 0x22, 0x80, 0x0f, 0x32, 0x30, 0x30, 0x38,  /* +..U...$0"..2008 */
   0x30, 0x32, 0x32, 0x36, 0x31, 0x33, 0x34, 0x33, 0x30, 0x34, 0x5a, 0x81, 0x0f, 0x32, 0x30, 0x31,  /* 0226134304Z..201 */
   0x31, 0x30, 0x32, 0x32, 0x36, 0x31, 0x32, 0x33, 0x35, 0x35, 0x36, 0x5a, 0x30, 0x0c, 0x06, 0x08,  /* 10226123556Z0... */
   0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x05, 0x00, 0x03, 0x47, 0x00, 0x30, 0x44, 0x02,  /* *.H.=......G.0D. */
   0x20, 0x1a, 0x39, 0x71, 0x22, 0x7a, 0x96, 0xaa, 0xe7, 0x92, 0x58, 0x8a, 0xa5, 0xdc, 0x0b, 0x48,  /*  .9q"z....X....H */
   0x98, 0xe9, 0x17, 0xa5, 0x8a, 0x30, 0x8e, 0xd1, 0xeb, 0xd1, 0x9a, 0x16, 0x97, 0x94, 0xea, 0x04,  /* .....0.......... */
   0xe7, 0x02, 0x20, 0x2a, 0x7e, 0x4e, 0x7e, 0x9b, 0x22, 0xf3, 0x98, 0xc7, 0x28, 0x49, 0x05, 0x19,  /* .. *~N~."...(I.. */
   0x49, 0x6b, 0x73, 0xd5, 0xc0, 0x2e, 0x4b, 0x29, 0xab, 0x65, 0x92, 0x29, 0x29, 0x52, 0xf3, 0x98,  /* Iks...K).e.))R.. */
   0x94, 0x47, 0xf7,                                                                                /* .G. */
};

struct tc_ef_cardsecurity {
    BUF_MEM ef_cardaccess;
    BUF_MEM ef_cardsecurity;
    BUF_MEM csca;
};

struct tc_ef_cardsecurity ef_cardsecurity_tests[] = {
   {
      { sizeof ef_cardaccess_dan, ef_cardaccess_dan, sizeof ef_cardaccess_dan, },
      { sizeof ef_cardsecurity_dan, ef_cardsecurity_dan, sizeof ef_cardsecurity_dan, },
      { sizeof csca_germany_013_self_signed_cer, csca_germany_013_self_signed_cer, sizeof csca_germany_013_self_signed_cer, },
   },
};

static const unsigned char cvc1[] = {
    0x7F, 0x21, 0x82, 0x01, 0x41, 0x7F, 0x4E, 0x81, 0xFA, 0x5F, 0x29,
    0x01, 0x00, 0x42, 0x0D, 0x5A, 0x5A, 0x44, 0x56, 0x43, 0x41, 0x41,
    0x54, 0x41, 0x30, 0x30, 0x30, 0x33, 0x7F, 0x49, 0x4F, 0x06, 0x0A,
    0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86,
    0x41, 0x04, 0x19, 0xD1, 0x75, 0x45, 0xD3, 0xFE, 0x0B, 0x34, 0x3E,
    0x7E, 0xE2, 0xAE, 0x4E, 0x2B, 0xC9, 0x2D, 0x51, 0x35, 0x1C, 0xC1,
    0x17, 0xA4, 0x7F, 0xA9, 0x51, 0x9A, 0xDB, 0x1E, 0x40, 0x5E, 0xE6,
    0xB8, 0x12, 0x12, 0x80, 0xBC, 0xC2, 0xFF, 0xF0, 0x35, 0x7A, 0x19,
    0x7D, 0xE7, 0x39, 0xA7, 0xFD, 0x2E, 0xF0, 0x22, 0x10, 0xEF, 0x34,
    0x3C, 0xDB, 0xE7, 0x9E, 0xF9, 0x4B, 0x8E, 0x28, 0x59, 0x1B, 0xB9,
    0x5F, 0x20, 0x0B, 0x5A, 0x5A, 0x44, 0x4B, 0x42, 0x32, 0x30, 0x30,
    0x30, 0x30, 0x52, 0x7F, 0x4C, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7F,
    0x00, 0x07, 0x03, 0x01, 0x02, 0x02, 0x53, 0x05, 0x00, 0x03, 0x01,
    0xDF, 0x04, 0x5F, 0x25, 0x06, 0x01, 0x00, 0x00, 0x02, 0x01, 0x07,
    0x5F, 0x24, 0x06, 0x01, 0x00, 0x00, 0x03, 0x03, 0x01, 0x65, 0x5E,
    0x73, 0x2D, 0x06, 0x09, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x03, 0x01,
    0x03, 0x01, 0x80, 0x20, 0x75, 0xE0, 0xC4, 0xAC, 0x36, 0xC2, 0x5A,
    0x33, 0xAC, 0x0E, 0x9A, 0x75, 0xEB, 0x79, 0x2A, 0x72, 0xF3, 0x31,
    0xA5, 0x1E, 0x28, 0x63, 0x4E, 0xCC, 0x2E, 0xD6, 0x2E, 0x54, 0xF3,
    0xC6, 0x93, 0xDA, 0x73, 0x2D, 0x06, 0x09, 0x04, 0x00, 0x7F, 0x00,
    0x07, 0x03, 0x01, 0x03, 0x02, 0x80, 0x20, 0x18, 0x12, 0x65, 0x74,
    0x49, 0xFC, 0xF1, 0xD3, 0xDA, 0xD8, 0x3D, 0x13, 0x14, 0x29, 0x17,
    0x5C, 0x61, 0x8B, 0x21, 0xBA, 0xF0, 0xAF, 0x44, 0xAC, 0xE3, 0x8C,
    0xB2, 0xC1, 0x2C, 0xEB, 0x2A, 0x56, 0x5F, 0x37, 0x40, 0x54, 0x0F,
    0x85, 0x09, 0x12, 0xAB, 0xD3, 0x51, 0xF8, 0xF5, 0x56, 0x9B, 0x53,
    0x4A, 0x5C, 0x8F, 0x64, 0x54, 0x5B, 0x51, 0xA7, 0x34, 0x70, 0xBE,
    0x5A, 0xD2, 0x89, 0xC1, 0x9A, 0x5E, 0x13, 0x52, 0x53, 0xD3, 0xBB,
    0x15, 0x52, 0x26, 0x21, 0x7B, 0x41, 0xE7, 0xF0, 0x68, 0xB3, 0x52,
    0x3F, 0x3A, 0x63, 0x92, 0x22, 0xAF, 0x2B, 0x62, 0x8C, 0x39, 0x7D,
    0x4F, 0xD4, 0x02, 0x1E, 0xDE, 0x00, 0xDC
};
static const unsigned char cvc1_desc[] = {
    0x30, 0x82, 0x01, 0x90, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x03,
    0x01, 0x03, 0x01, 0x01, 0xA1, 0x16, 0x0C, 0x14, 0x42, 0x75, 0x6E, 0x64,
    0x65, 0x73, 0x64, 0x72, 0x75, 0x63, 0x6B, 0x65, 0x72, 0x65, 0x69, 0x20,
    0x47, 0x6D, 0x62, 0x48, 0xA2, 0x24, 0x13, 0x22, 0x68, 0x74, 0x74, 0x70,
    0x3A, 0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x62, 0x75, 0x6E, 0x64, 0x65,
    0x73, 0x64, 0x72, 0x75, 0x63, 0x6B, 0x65, 0x72, 0x65, 0x69, 0x2E, 0x64,
    0x65, 0x2F, 0x64, 0x76, 0x63, 0x61, 0xA3, 0x18, 0x0C, 0x16, 0x44, 0x65,
    0x75, 0x74, 0x73, 0x63, 0x68, 0x65, 0x20, 0x4B, 0x72, 0x65, 0x64, 0x69,
    0x74, 0x62, 0x61, 0x6E, 0x6B, 0x20, 0x41, 0x47, 0xA4, 0x13, 0x13, 0x11,
    0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x64,
    0x6B, 0x62, 0x2E, 0x64, 0x65, 0xA5, 0x82, 0x01, 0x13, 0x0C, 0x82, 0x01,
    0x0F, 0x54, 0x61, 0x75, 0x62, 0x65, 0x6E, 0x73, 0x74, 0x72, 0x2E, 0x20,
    0x37, 0x2D, 0x39, 0x0D, 0x0A, 0x31, 0x30, 0x31, 0x31, 0x37, 0x20, 0x42,
    0x65, 0x72, 0x6C, 0x69, 0x6E, 0x0D, 0x0A, 0x69, 0x6E, 0x66, 0x6F, 0x40,
    0x64, 0x6B, 0x62, 0x2E, 0x64, 0x65, 0x0D, 0x0A, 0x45, 0x72, 0xC3, 0xB6,
    0x66, 0x66, 0x6E, 0x75, 0x6E, 0x67, 0x20, 0x65, 0x69, 0x6E, 0x65, 0x73,
    0x20, 0x4B, 0x6F, 0x6E, 0x74, 0x6F, 0x73, 0x0D, 0x0A, 0x42, 0x65, 0x72,
    0x6C, 0x69, 0x6E, 0x65, 0x72, 0x20, 0x42, 0x65, 0x61, 0x75, 0x66, 0x74,
    0x72, 0x61, 0x67, 0x74, 0x65, 0x72, 0x20, 0x66, 0xC3, 0xBC, 0x72, 0x20,
    0x44, 0x61, 0x74, 0x65, 0x6E, 0x73, 0x63, 0x68, 0x75, 0x74, 0x7A, 0x20,
    0x75, 0x6E, 0x64, 0x20, 0x49, 0x6E, 0x66, 0x6F, 0x72, 0x6D, 0x61, 0x74,
    0x69, 0x6F, 0x6E, 0x73, 0x66, 0x72, 0x65, 0x69, 0x68, 0x65, 0x69, 0x74,
    0x2C, 0x20, 0x41, 0x6E, 0x20, 0x64, 0x65, 0x72, 0x20, 0x55, 0x72, 0x61,
    0x6E, 0x69, 0x61, 0x20, 0x34, 0x2D, 0x31, 0x30, 0x2C, 0x20, 0x31, 0x30,
    0x37, 0x38, 0x37, 0x20, 0x42, 0x65, 0x72, 0x6C, 0x69, 0x6E, 0x2C, 0x20,
    0x30, 0x33, 0x30, 0x2F, 0x31, 0x33, 0x20, 0x38, 0x38, 0x39, 0x2D, 0x30,
    0x2C, 0x20, 0x6D, 0x61, 0x69, 0x6C, 0x62, 0x6F, 0x78, 0x40, 0x64, 0x61,
    0x74, 0x65, 0x6E, 0x73, 0x63, 0x68, 0x75, 0x74, 0x7A, 0x2D, 0x62, 0x65,
    0x72, 0x6C, 0x69, 0x6E, 0x2E, 0x64, 0x65, 0x2C, 0x20, 0x68, 0x74, 0x74,
    0x70, 0x3A, 0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x64, 0x61, 0x74, 0x65,
    0x6E, 0x73, 0x63, 0x68, 0x75, 0x74, 0x7A, 0x2D, 0x62, 0x65, 0x72, 0x6C,
    0x69, 0x6E, 0x2E, 0x64, 0x65, 0x0D, 0x0A, 0x45, 0x72, 0xC3, 0xB6, 0x66,
    0x66, 0x6E, 0x75, 0x6E, 0x67, 0x20, 0x65, 0x69, 0x6E, 0x65, 0x73, 0x20,
    0x4B, 0x6F, 0x6E, 0x74, 0x6F, 0x73, 0x0D, 0x0A
};
/* http://www.internet-sicherheit.de/forschung/aktuelle-forschungsprojekte/elektronischer-personalausweis/berechtigungszertifikat/ */
static unsigned const char cvc2[] = {
    0x7f, 0x21, 0x82, 0x01, 0x89, 0x7f, 0x4e, 0x82, 0x01, 0x49, 0x5f,
    0x29, 0x01, 0x00, 0x42, 0x0e, 0x44, 0x45, 0x43, 0x56, 0x43, 0x41,
    0x45, 0x50, 0x41, 0x53, 0x53, 0x30, 0x30, 0x31, 0x7f, 0x49, 0x81,
    0xfd, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02,
    0x02, 0x02, 0x81, 0x1c, 0xd7, 0xc1, 0x34, 0xaa, 0x26, 0x43, 0x66,
    0x86, 0x2a, 0x18, 0x30, 0x25, 0x75, 0xd1, 0xd7, 0x87, 0xb0, 0x9f,
    0x07, 0x57, 0x97, 0xda, 0x89, 0xf5, 0x7e, 0xc8, 0xc0, 0xff, 0x82,
    0x1c, 0x68, 0xa5, 0xe6, 0x2c, 0xa9, 0xce, 0x6c, 0x1c, 0x29, 0x98,
    0x03, 0xa6, 0xc1, 0x53, 0x0b, 0x51, 0x4e, 0x18, 0x2a, 0xd8, 0xb0,
    0x04, 0x2a, 0x59, 0xca, 0xd2, 0x9f, 0x43, 0x83, 0x1c, 0x25, 0x80,
    0xf6, 0x3c, 0xcf, 0xe4, 0x41, 0x38, 0x87, 0x07, 0x13, 0xb1, 0xa9,
    0x23, 0x69, 0xe3, 0x3e, 0x21, 0x35, 0xd2, 0x66, 0xdb, 0xb3, 0x72,
    0x38, 0x6c, 0x40, 0x0b, 0x84, 0x39, 0x04, 0x0d, 0x90, 0x29, 0xad,
    0x2c, 0x7e, 0x5c, 0xf4, 0x34, 0x08, 0x23, 0xb2, 0xa8, 0x7d, 0xc6,
    0x8c, 0x9e, 0x4c, 0xe3, 0x17, 0x4c, 0x1e, 0x6e, 0xfd, 0xee, 0x12,
    0xc0, 0x7d, 0x58, 0xaa, 0x56, 0xf7, 0x72, 0xc0, 0x72, 0x6f, 0x24,
    0xc6, 0xb8, 0x9e, 0x4e, 0xcd, 0xac, 0x24, 0x35, 0x4b, 0x9e, 0x99,
    0xca, 0xa3, 0xf6, 0xd3, 0x76, 0x14, 0x02, 0xcd, 0x85, 0x1c, 0xd7,
    0xc1, 0x34, 0xaa, 0x26, 0x43, 0x66, 0x86, 0x2a, 0x18, 0x30, 0x25,
    0x75, 0xd0, 0xfb, 0x98, 0xd1, 0x16, 0xbc, 0x4b, 0x6d, 0xde, 0xbc,
    0xa3, 0xa5, 0xa7, 0x93, 0x9f, 0x86, 0x39, 0x04, 0x39, 0x3e, 0xe8,
    0xe0, 0x6d, 0xb6, 0xc7, 0xf5, 0x28, 0xf8, 0xb4, 0x26, 0x0b, 0x49,
    0xaa, 0x93, 0x30, 0x98, 0x24, 0xd9, 0x2c, 0xdb, 0x18, 0x07, 0xe5,
    0x43, 0x7e, 0xe2, 0xe2, 0x6e, 0x29, 0xb7, 0x3a, 0x71, 0x11, 0x53,
    0x0f, 0xa8, 0x6b, 0x35, 0x00, 0x37, 0xcb, 0x94, 0x15, 0xe1, 0x53,
    0x70, 0x43, 0x94, 0x46, 0x37, 0x97, 0x13, 0x9e, 0x14, 0x87, 0x01,
    0x01, 0x5f, 0x20, 0x0e, 0x44, 0x45, 0x43, 0x56, 0x43, 0x41, 0x45,
    0x50, 0x41, 0x53, 0x53, 0x30, 0x30, 0x31, 0x7f, 0x4c, 0x0e, 0x06,
    0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, 0x01, 0x53,
    0x01, 0xc1, 0x5f, 0x25, 0x06, 0x00, 0x07, 0x00, 0x04, 0x00, 0x01,
    0x5f, 0x24, 0x06, 0x00, 0x09, 0x00, 0x04, 0x00, 0x01, 0x5f, 0x37,
    0x38, 0x4c, 0xcf, 0x25, 0xc5, 0x9f, 0x36, 0x12, 0xee, 0xe1, 0x88,
    0x75, 0xf6, 0xc5, 0xf2, 0xe2, 0xd2, 0x1f, 0x03, 0x95, 0x68, 0x3b,
    0x53, 0x2a, 0x26, 0xe4, 0xc1, 0x89, 0xb7, 0x1e, 0xfe, 0x65, 0x9c,
    0x3f, 0x26, 0xe0, 0xeb, 0x9a, 0xea, 0xe9, 0x98, 0x63, 0x10, 0x7f,
    0x9b, 0x0d, 0xad, 0xa1, 0x64, 0x14, 0xff, 0xa2, 0x04, 0x51, 0x6a,
    0xee, 0x2b
};

static unsigned char wrong_tc_ecdh_cvca[] = {
    0x7f, 0x21, 0x82, 0x01, 0xb4, 0x7f, 0x4e, 0x82, 0x01, 0x6c, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x0d,  /*.!....N..l_)..B. */
    0x44, 0x45, 0x41, 0x54, 0x43, 0x56, 0x43, 0x41, 0x30, 0x30, 0x30, 0x30, 0x31, 0x7f, 0x49, 0x82,  /*DEATCVCA00001.I. */
    0x01, 0x1d, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x81, 0x20,  /*...............  */
    0xa9, 0xfb, 0x57, 0xdb, 0xa1, 0xee, 0xa9, 0xbc, 0x3e, 0x66, 0x0a, 0x90, 0x9d, 0x83, 0x8d, 0x72,  /*..W.....>f.....r */
    0x6e, 0x3b, 0xf6, 0x23, 0xd5, 0x26, 0x20, 0x28, 0x20, 0x13, 0x48, 0x1d, 0x1f, 0x6e, 0x53, 0x77,  /*n;.#.& ( .H..nSw */
    0x82, 0x20, 0x7d, 0x5a, 0x09, 0x75, 0xfc, 0x2c, 0x30, 0x57, 0xee, 0xf6, 0x75, 0x30, 0x41, 0x7a,  /*. }Z.u.,0W..u0Az */
    0xff, 0xe7, 0xfb, 0x80, 0x55, 0xc1, 0x26, 0xdc, 0x5c, 0x6c, 0xe9, 0x4a, 0x4b, 0x44, 0xf3, 0x30,  /*....U.&.\l.JKD.0 */
    0xb5, 0xd9, 0x83, 0x20, 0x26, 0xdc, 0x5c, 0x6c, 0xe9, 0x4a, 0x4b, 0x44, 0xf3, 0x30, 0xb5, 0xd9,  /*... &.\l.JKD.0.. */
    0xbb, 0xd7, 0x7c, 0xbf, 0x95, 0x84, 0x16, 0x29, 0x5c, 0xf7, 0xe1, 0xce, 0x6b, 0xcc, 0xdc, 0x18,  /*..|....)\...k... */
    0xff, 0x8c, 0x07, 0xb6, 0x84, 0x41, 0x04, 0x8b, 0xd2, 0xae, 0xb9, 0xcb, 0x7e, 0x57, 0xcb, 0x2c,  /*.....A......~W., */
    0x4b, 0x48, 0x2f, 0xfc, 0x81, 0xb7, 0xaf, 0xb9, 0xde, 0x27, 0xe1, 0xe3, 0xbd, 0x23, 0xc2, 0x3a,  /*KH/......'...#.: */
    0x44, 0x53, 0xbd, 0x9a, 0xce, 0x32, 0x62, 0x54, 0x7e, 0xf8, 0x35, 0xc3, 0xda, 0xc4, 0xfd, 0x97,  /*DS...2bT~.5..... */
    0xf8, 0x46, 0x1a, 0x14, 0x61, 0x1d, 0xc9, 0xc2, 0x77, 0x45, 0x13, 0x2d, 0xed, 0x8e, 0x54, 0x5c,  /*.F..a...wE.-..T\ */
    0x1d, 0x54, 0xc7, 0x2f, 0x04, 0x69, 0x97, 0x85, 0x20, 0xa9, 0xfb, 0x57, 0xdb, 0xa1, 0xee, 0xa9,  /*.T./.i.. ..W.... */
    0xbc, 0x3e, 0x66, 0x0a, 0x90, 0x9d, 0x83, 0x8d, 0x71, 0x8c, 0x39, 0x7a, 0xa3, 0xb5, 0x61, 0xa6,  /*.>f.....q.9z..a. */
    0xf7, 0x90, 0x1e, 0x0e, 0x82, 0x97, 0x48, 0x56, 0xa7, 0x86, 0x41, 0x04, 0x31, 0xa3, 0x8e, 0xe4,  /*......HV..A.1... */
    0x3b, 0xf1, 0xa9, 0xf0, 0x3b, 0x13, 0x6b, 0x67, 0x30, 0x12, 0x92, 0x38, 0xe4, 0x12, 0xd2, 0x4c,  /*;...;.kg0..8...L */
    0x6e, 0x36, 0xdd, 0xf2, 0xe6, 0x26, 0x20, 0x84, 0xd0, 0x5f, 0xaf, 0x3d, 0x9d, 0xed, 0x1c, 0xe1,  /*n6...& .._.=.... */
    0x2e, 0x26, 0x4c, 0xcc, 0xb5, 0xb0, 0xda, 0xd2, 0x1f, 0x22, 0x2a, 0xd2, 0x81, 0xe3, 0x9d, 0x97,  /*.&L......"*..... */
    0x29, 0x5e, 0xb0, 0x4d, 0x18, 0xfc, 0x72, 0x1f, 0x46, 0x9f, 0x7b, 0xd8, 0x87, 0x01, 0x01, 0x5f,  /*)^.M..r.F.{...._ */
    0x20, 0x0d, 0x44, 0x45, 0x41, 0x54, 0x43, 0x56, 0x43, 0x41, 0x30, 0x30, 0x30, 0x30, 0x31, 0x7f,  /* .DEATCVCA00001. */
    0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, 0x02, 0x53, 0x05, 0xfe,  /*L............S.. */
    0x1f, 0xff, 0xff, 0xff, 0x5f, 0x25, 0x06, 0x01, 0x00, 0x00, 0x03, 0x02, 0x04, 0x5f, 0x24, 0x06,  /*...._%......._$. */
    0x01, 0x01, 0x00, 0x03, 0x02, 0x04, 0x5f, 0x37, 0x40, 0x13, 0x48, 0xec, 0xf2, 0xce, 0x5e, 0x0c,  /*......_7@.H...^. */
    0x23, 0x3f, 0xd2, 0xb7, 0x78, 0x42, 0x10, 0x4f, 0x76, 0xfc, 0xe6, 0x02, 0x19, 0x28, 0x8d, 0x2f,  /*#?..xB.Ov....(./ */
    0x67, 0x36, 0x0c, 0xbc, 0xf8, 0x80, 0xc7, 0x92, 0x5d, 0x4f, 0x2f, 0x80, 0x6f, 0x8b, 0xac, 0xa8,  /*g6......]O/.o... */
    0x60, 0x83, 0xbd, 0xc7, 0x6a, 0xb9, 0xdd, 0xa4, 0x2b, 0xc5, 0xe8, 0x26, 0x20, 0xd6, 0x11, 0xf4,  /*`...j...+..& ... */
    0x64, 0x97, 0x4f, 0x6c, 0x28, 0x21, 0x4e, 0x1e, 0xc2,                                            /*d.Ol(!N.. */
};

/* https://www.kba-online.de/vzronline/ */
static unsigned char kba_dvca_cert[] = {
    0x7f, 0x21, 0x82, 0x00, 0xe7, 0x7F,
    0x4E, 0x81, 0xA0, 0x5F, 0x29, 0x01, 0x00, 0x42, 0x0E, 0x44, 0x45, 0x43, 0x56, 0x43, 0x41, 0x65, /*N.._)..B.DECVCAe*/
    0x49, 0x44, 0x30, 0x30, 0x31, 0x30, 0x32, 0x7F, 0x49, 0x4F, 0x06, 0x0A, 0x04, 0x00, 0x7F, 0x00, /*ID00102.IO......*/
    0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86, 0x41, 0x04, 0xA0, 0x1E, 0x1B, 0x0D, 0x37, 0x50, 0x51, /*.......A.....7PQ*/
    0xAD, 0xCA, 0x23, 0x00, 0x33, 0xA8, 0x9F, 0x20, 0xC2, 0x87, 0x06, 0x52, 0x98, 0xD1, 0x4D, 0x2F, /*..#.3.. ...R..M/*/
    0xF9, 0xD0, 0x2A, 0xFF, 0x43, 0xFA, 0x14, 0xE2, 0x44, 0x24, 0xEA, 0x7B, 0xCD, 0x8B, 0xA8, 0x66, /*..*.C...D$.{...f*/
    0xBF, 0xF2, 0x36, 0x1E, 0x94, 0x2C, 0x76, 0x52, 0xA0, 0xF5, 0x22, 0xB4, 0x90, 0xCA, 0xDC, 0x25, /*..6..,vR.."....%*/
    0x97, 0x2E, 0x01, 0xA2, 0x2F, 0xA5, 0x21, 0xB3, 0x28, 0x5F, 0x20, 0x10, 0x44, 0x45, 0x44, 0x56, /*..../.!.(_ .DEDV*/
    0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x32, 0x30, 0x31, 0x7F, 0x4C, 0x12, 0x06, /*eIDDTR101201.L..*/
    0x09, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x03, 0x01, 0x02, 0x02, 0x53, 0x05, 0x40, 0x05, 0x01, 0xFF, /*..........S.@...*/
    0x87, 0x5F, 0x25, 0x06, 0x01, 0x01, 0x00, 0x06, 0x02, 0x04, 0x5F, 0x24, 0x06, 0x01, 0x01, 0x00, /*._%......._$....*/
    0x09, 0x02, 0x04, 0x5F, 0x37, 0x40, 0x5F, 0x22, 0x30, 0xC2, 0x1F, 0xC3, 0x86, 0x28, 0x90, 0x22, /*..._7@_"0....(."*/
    0xF2, 0x7F, 0x0D, 0x54, 0x0E, 0xAB, 0x07, 0x74, 0xFE, 0x85, 0xB6, 0x57, 0x34, 0xE0, 0x23, 0x90, /*...T...t...W4.#.*/
    0x8C, 0x83, 0x3E, 0xCE, 0x2F, 0x93, 0x96, 0xD4, 0xEE, 0x5C, 0x8E, 0x9F, 0xDC, 0x33, 0x3D, 0x9E, /*..>./....\...3=.*/
    0xD9, 0x71, 0xA9, 0x70, 0xEF, 0xF5, 0x78, 0xD4, 0x00, 0x3D, 0x48, 0x62, 0x62, 0x9F, 0x3B, 0xD8, /*.q.p..x..=Hbb.;.*/
    0xFA, 0xCD, 0xED, 0x10, 0xBC, 0x51,
};
static unsigned char kba_cert[] = {
    0x7f, 0x21, 0x82, 0x01, 0x48,
          0x7F, 0x4E, 0x82, 0x01, 0x00, 0x5F, 0x29, 0x01, 0x00, 0x42, 0x10, 0x44, 0x45, 0x44, 0x56,  /*.N..._)..B.DEDV*/
    0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x32, 0x30, 0x31, 0x7F, 0x49, 0x4F, 0x06, /*eIDDTR101201.IO.*/
    0x0A, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86, 0x41, 0x04, 0x6E, 0x43, /*............A.nC*/
    0x02, 0xE9, 0x70, 0xC8, 0x1F, 0x35, 0x60, 0x28, 0xB1, 0x75, 0x7C, 0xB4, 0xC7, 0x14, 0xE0, 0xD8, /*..p..5`(.u|.....*/
    0x75, 0xEB, 0xA6, 0x94, 0xAB, 0xA9, 0xE6, 0x43, 0xF8, 0x3F, 0x72, 0x8A, 0x45, 0xCF, 0x83, 0xDE, /*u......C.?r.E...*/
    0x2E, 0xB1, 0x05, 0x5E, 0x02, 0xC5, 0x2C, 0xDE, 0x74, 0xA2, 0x4B, 0x1D, 0xF9, 0xBB, 0xA3, 0xE8, /*...^..,.t.K.....*/
    0xB1, 0x7B, 0x40, 0x35, 0x56, 0xE5, 0x26, 0x1F, 0x03, 0x12, 0x8E, 0x55, 0x84, 0x3C, 0x5F, 0x20, /*.{@5V.&....U.<_ */
    0x0E, 0x44, 0x45, 0x30, 0x30, 0x30, 0x30, 0x30, 0x37, 0x31, 0x30, 0x30, 0x33, 0x47, 0x33, 0x7F, /*.DE0000071003G3.*/
    0x4C, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7F, 0x00, 0x07, 0x03, 0x01, 0x02, 0x02, 0x53, 0x05, 0x00, /*L............S..*/
    0x01, 0x01, 0x98, 0x04, 0x5F, 0x25, 0x06, 0x01, 0x01, 0x00, 0x09, 0x02, 0x01, 0x5F, 0x24, 0x06, /*...._%......._$.*/
    0x01, 0x01, 0x00, 0x09, 0x02, 0x03, 0x65, 0x5E, 0x73, 0x2D, 0x06, 0x09, 0x04, 0x00, 0x7F, 0x00, /*......e^s-......*/
    0x07, 0x03, 0x01, 0x03, 0x02, 0x80, 0x20, 0x98, 0x71, 0xAC, 0xEA, 0x94, 0x79, 0xF5, 0xF3, 0x36, /*...... .q...y..6*/
    0xB1, 0x24, 0xB8, 0x71, 0xB7, 0xB1, 0xFB, 0x4D, 0xF4, 0x06, 0x46, 0xB8, 0x69, 0x35, 0x7C, 0xBA, /*.$.q...M..F.i5|.*/
    0x77, 0x4D, 0x3C, 0x10, 0x83, 0x41, 0x97, 0x73, 0x2D, 0x06, 0x09, 0x04, 0x00, 0x7F, 0x00, 0x07, /*wM<..A.s-.......*/
    0x03, 0x01, 0x03, 0x01, 0x80, 0x20, 0x8E, 0x2F, 0xCF, 0x13, 0x70, 0xEE, 0x5E, 0x7D, 0xBC, 0xCA, /*..... ./..p.^}..*/
    0x6F, 0xBF, 0xE0, 0x04, 0x41, 0x43, 0xB4, 0x97, 0xF4, 0x27, 0xA5, 0xD9, 0xAF, 0xAF, 0x5E, 0x40, /*o...AC...'....^@*/
    0x33, 0xB2, 0xB1, 0xC3, 0x22, 0x06, 0x5F, 0x37, 0x40, 0x88, 0x49, 0x47, 0xD1, 0x07, 0x23, 0xD4, /*3..."._7@.IG..#.*/
    0xB1, 0xE8, 0x7B, 0x4F, 0x74, 0x2E, 0x1B, 0xEC, 0x1A, 0x56, 0x2F, 0x8E, 0xCE, 0x62, 0xB1, 0x2C, /*..{Ot....V/..b.,*/
    0xEB, 0xE5, 0x8B, 0xF2, 0x70, 0xB2, 0x67, 0xCD, 0x31, 0x95, 0x0C, 0x8B, 0x03, 0xA5, 0xD8, 0x9F, /*....p.g.1.......*/
    0x1A, 0xEB, 0x06, 0x8C, 0x7B, 0x75, 0x18, 0x66, 0xCD, 0xA8, 0xE8, 0xC1, 0xCF, 0x21, 0x0C, 0xF9, /*....{u.f.....!..*/
    0xA5, 0x24, 0x31, 0x04, 0x7C, 0x5C, 0x72, 0x31, 0xA4,                                           /*.$1.|\r1.*/
};
static unsigned char kba_cert_desc[] = {
    0x30, 0x82, 0x02, 0x58, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x01,
    0xa1, 0x0e, 0x0c, 0x0c, 0x44, 0x2d, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x47, 0x6d, 0x62, 0x48,
    0xa2, 0x18, 0x13, 0x16, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x64,
    0x2d, 0x74, 0x72, 0x75, 0x73, 0x74, 0x2e, 0x6e, 0x65, 0x74, 0xa3, 0x16, 0x0c, 0x14, 0x4b, 0x72,
    0x61, 0x66, 0x74, 0x66, 0x61, 0x68, 0x72, 0x74, 0x2d, 0x42, 0x75, 0x6e, 0x64, 0x65, 0x73, 0x61,
    0x6d, 0x74, 0xa4, 0x25, 0x13, 0x23, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77,
    0x77, 0x2e, 0x6b, 0x62, 0x61, 0x2d, 0x6f, 0x6e, 0x6c, 0x69, 0x6e, 0x65, 0x2e, 0x64, 0x65, 0x2f,
    0x76, 0x7a, 0x72, 0x6f, 0x6e, 0x6c, 0x69, 0x6e, 0x65, 0xa5, 0x82, 0x01, 0x97, 0x0c, 0x82, 0x01,
    0x93, 0x4e, 0x61, 0x6d, 0x65, 0x2c, 0x20, 0x41, 0x6e, 0x73, 0x63, 0x68, 0x72, 0x69, 0x66, 0x74,
    0x20, 0x75, 0x6e, 0x64, 0x20, 0x45, 0x2d, 0x4d, 0x61, 0x69, 0x6c, 0x2d, 0x41, 0x64, 0x72, 0x65,
    0x73, 0x73, 0x65, 0x20, 0x64, 0x65, 0x73, 0x20, 0x44, 0x69, 0x65, 0x6e, 0x73, 0x74, 0x61, 0x6e,
    0x62, 0x69, 0x65, 0x74, 0x65, 0x72, 0x73, 0x3a, 0x0d, 0x0a, 0x4b, 0x72, 0x61, 0x66, 0x74, 0x66,
    0x61, 0x68, 0x72, 0x74, 0x2d, 0x42, 0x75, 0x6e, 0x64, 0x65, 0x73, 0x61, 0x6d, 0x74, 0x0d, 0x0a,
    0x46, 0xc3, 0xb6, 0x72, 0x64, 0x65, 0x73, 0x74, 0x72, 0x61, 0x73, 0x73, 0x65, 0x20, 0x31, 0x36,
    0x0d, 0x0a, 0x32, 0x34, 0x39, 0x34, 0x34, 0x20, 0x46, 0x6c, 0x65, 0x6e, 0x73, 0x62, 0x75, 0x72,
    0x67, 0x0d, 0x0a, 0x56, 0x5a, 0x52, 0x2d, 0x48, 0x69, 0x6c, 0x66, 0x65, 0x40, 0x6b, 0x62, 0x61,
    0x2e, 0x64, 0x65, 0x0d, 0x0a, 0x0d, 0x0a, 0x47, 0x65, 0x73, 0x63, 0x68, 0xc3, 0xa4, 0x66, 0x74,
    0x73, 0x7a, 0x77, 0x65, 0x63, 0x6b, 0x3a, 0x0d, 0x0a, 0x2d, 0x20, 0x41, 0x75, 0x73, 0x6b, 0x75,
    0x6e, 0x66, 0x74, 0x20, 0x61, 0x75, 0x73, 0x20, 0x64, 0x65, 0x6d, 0x20, 0x56, 0x65, 0x72, 0x6b,
    0x65, 0x68, 0x72, 0x73, 0x7a, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x72, 0x65, 0x67, 0x69, 0x73,
    0x74, 0x65, 0x72, 0x20, 0x2d, 0x0d, 0x0a, 0x0d, 0x0a, 0x5a, 0x75, 0x73, 0x74, 0xc3, 0xa4, 0x6e,
    0x64, 0x69, 0x67, 0x65, 0x20, 0x44, 0x61, 0x74, 0x65, 0x6e, 0x73, 0x63, 0x68, 0x75, 0x74, 0x7a,
    0x62, 0x65, 0x68, 0xc3, 0xb6, 0x72, 0x64, 0x65, 0x3a, 0x0d, 0x0a, 0x44, 0x65, 0x72, 0x20, 0x42,
    0x75, 0x6e, 0x64, 0x65, 0x73, 0x62, 0x65, 0x61, 0x75, 0x66, 0x74, 0x72, 0x61, 0x67, 0x74, 0x65,
    0x20, 0x66, 0xc3, 0xbc, 0x72, 0x20, 0x64, 0x65, 0x6e, 0x20, 0x44, 0x61, 0x74, 0x65, 0x6e, 0x73,
    0x63, 0x68, 0x75, 0x74, 0x7a, 0x20, 0x75, 0x6e, 0x64, 0x20, 0x64, 0x69, 0x65, 0x20, 0x49, 0x6e,
    0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x66, 0x72, 0x65, 0x69, 0x68, 0x65,
    0x69, 0x74, 0x0d, 0x0a, 0x48, 0x75, 0x73, 0x61, 0x72, 0x65, 0x6e, 0x73, 0x74, 0x72, 0x61, 0xc3,
    0x9f, 0x65, 0x20, 0x33, 0x30, 0x0d, 0x0a, 0x35, 0x33, 0x31, 0x31, 0x37, 0x20, 0x42, 0x6f, 0x6e,
    0x6e, 0x0d, 0x0a, 0x30, 0x32, 0x32, 0x38, 0x2d, 0x39, 0x39, 0x37, 0x37, 0x39, 0x39, 0x2d, 0x30,
    0x0d, 0x0a, 0x70, 0x6f, 0x73, 0x74, 0x73, 0x74, 0x65, 0x6c, 0x6c, 0x65, 0x40, 0x62, 0x66, 0x64,
    0x69, 0x2e, 0x62, 0x75, 0x6e, 0x64, 0x2e, 0x64, 0x65, 0x0d, 0x0a, 0x68, 0x74, 0x74, 0x70, 0x3a,
    0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x62, 0x66, 0x64, 0x69, 0x2e, 0x62, 0x75, 0x6e, 0x64, 0x2e,
    0x64, 0x65, 0x0d, 0x0a, 0xa7, 0x46, 0x31, 0x44, 0x04, 0x20, 0x80, 0x5a, 0xd7, 0x54, 0x56, 0x8e,
    0x47, 0x2c, 0x47, 0x61, 0xd5, 0x2d, 0x41, 0x0f, 0xb9, 0x91, 0x28, 0xab, 0x4c, 0xe2, 0xd7, 0x50,
    0xfd, 0xa3, 0xa8, 0xda, 0x8f, 0xba, 0x67, 0xbb, 0x14, 0xeb, 0x04, 0x20, 0xf4, 0x31, 0x0a, 0x18,
    0x1e, 0x71, 0x06, 0x4c, 0x36, 0xac, 0x95, 0xa0, 0x66, 0x68, 0xf0, 0xb7, 0x7a, 0x30, 0xd5, 0x99,
    0x58, 0x67, 0x71, 0x3a, 0x73, 0xe5, 0xf1, 0x28, 0x6e, 0xd9, 0x90, 0x73,
};
static unsigned char kba_cert_3[] = {
    0x7f, 0x21, 0x82, 0x01, 0x48, 0x7f, 0x4e, 0x82, 0x01, 0x00, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x10, /* .!..H.N..._)..B. */
    0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30, 0x31, /* DEDVeIDDTR101401 */
    0x7f, 0x49, 0x4f, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86, /* .IO............. */
    0x41, 0x04, 0x82, 0x82, 0x88, 0xe4, 0x86, 0x71, 0xb1, 0xad, 0x46, 0x58, 0x9a, 0x50, 0x73, 0x20, /* A......q..FX.Ps  */
    0xa1, 0x66, 0x98, 0xf4, 0x95, 0x65, 0x4a, 0x20, 0x07, 0xbe, 0x7b, 0x54, 0x9d, 0x79, 0x71, 0x5b, /* .f...eJ ..{T.yq[ */
    0x59, 0xc8, 0x61, 0xb6, 0x45, 0x73, 0xe7, 0x5e, 0xf3, 0x7a, 0x74, 0x58, 0xb8, 0x11, 0x5c, 0x78, /* Y.a.Es.^.ztX..\x */
    0x27, 0xa3, 0xee, 0xda, 0xc7, 0xee, 0x47, 0xe1, 0x86, 0x88, 0x08, 0x7e, 0x4b, 0x66, 0x01, 0x7e, /* '.....G....~Kf.~ */
    0x0c, 0x05, 0x5f, 0x20, 0x0e, 0x44, 0x45, 0x30, 0x30, 0x30, 0x30, 0x30, 0x37, 0x31, 0x30, 0x30, /* .._ .DE000007100 */
    0x35, 0x4a, 0x32, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, /* 5J2.L........... */
    0x02, 0x53, 0x05, 0x00, 0x01, 0x01, 0x98, 0x04, 0x5f, 0x25, 0x06, 0x01, 0x02, 0x00, 0x02, 0x00, /* .S......_%...... */
    0x07, 0x5f, 0x24, 0x06, 0x01, 0x02, 0x00, 0x02, 0x00, 0x08, 0x65, 0x5e, 0x73, 0x2d, 0x06, 0x09, /* ._$.......e^s-.. */
    0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x02, 0x80, 0x20, 0x98, 0x71, 0xac, 0xea, 0x94, /* .......... .q... */
    0x79, 0xf5, 0xf3, 0x36, 0xb1, 0x24, 0xb8, 0x71, 0xb7, 0xb1, 0xfb, 0x4d, 0xf4, 0x06, 0x46, 0xb8, /* y..6.$.q...M..F. */
    0x69, 0x35, 0x7c, 0xba, 0x77, 0x4d, 0x3c, 0x10, 0x83, 0x41, 0x97, 0x73, 0x2d, 0x06, 0x09, 0x04, /* i5|.wM<..A.s-... */
    0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x80, 0x20, 0x36, 0x46, 0xb9, 0x42, 0xb1, 0xb1, /* ......... 6F.B.. */
    0x0e, 0x87, 0x7d, 0x25, 0x67, 0xec, 0x2f, 0xf5, 0x57, 0x7c, 0xd9, 0xa5, 0x95, 0x6d, 0x29, 0x44, /* ..}%g./.W|...m)D */
    0x4b, 0xad, 0x45, 0xdc, 0x7c, 0x21, 0x9f, 0xce, 0x6a, 0x71, 0x5f, 0x37, 0x40, 0x92, 0x3e, 0x51, /* K.E.|!..jq_7@.>Q */
    0x82, 0xcc, 0x64, 0x20, 0x77, 0x9b, 0x21, 0xdc, 0xa2, 0x1d, 0x85, 0x57, 0x45, 0xa7, 0x18, 0x5e, /* ..d w.!....WE..^ */
    0x68, 0x1d, 0xf4, 0x23, 0xa0, 0xb5, 0x5b, 0xd6, 0xc5, 0xee, 0xc5, 0x27, 0x8a, 0x84, 0xa6, 0x2c, /* h..#..[....'..., */
    0x76, 0x9c, 0x51, 0xe2, 0xf1, 0xa5, 0x3f, 0x9d, 0xab, 0x09, 0x0c, 0xa9, 0x08, 0xbb, 0xfe, 0x84, /* v.Q...?......... */
    0x41, 0x9d, 0x08, 0xf3, 0xd2, 0x09, 0xd3, 0xc8, 0xa0, 0x73, 0x28, 0xdd, 0x1e,                   /* A........s(.. */
};
static unsigned char kba_cert_2[] = {
    0x7f, 0x21, 0x82, 0x01, 0x48, 0x7f, 0x4e, 0x82, 0x01, 0x00, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x10, /* .!..H.N..._)..B. */
    0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30, 0x31, /* DEDVeIDDTR101401 */
    0x7f, 0x49, 0x4f, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86, /* .IO............. */
    0x41, 0x04, 0x06, 0x84, 0x9e, 0xb7, 0x1d, 0x58, 0x39, 0xae, 0x64, 0xa0, 0xcf, 0x53, 0xbc, 0xa3, /* A......X9.d..S.. */
    0x00, 0xce, 0x90, 0x89, 0xfc, 0x06, 0x00, 0xbd, 0x9b, 0x1b, 0x67, 0x29, 0xb9, 0x37, 0x36, 0xcb, /* ..........g).76. */
    0xa2, 0x96, 0x2f, 0x70, 0xd6, 0x38, 0xde, 0x95, 0x28, 0x6c, 0x90, 0x0d, 0x9f, 0x4e, 0xd3, 0x08, /* ../p.8..(l...N.. */
    0x33, 0xa8, 0xce, 0xf1, 0xda, 0x4d, 0xa3, 0x65, 0x7a, 0x84, 0x6a, 0x96, 0xfc, 0xd4, 0x85, 0xa0, /* 3....M.ez.j..... */
    0x81, 0xcd, 0x5f, 0x20, 0x0e, 0x44, 0x45, 0x30, 0x30, 0x30, 0x30, 0x30, 0x37, 0x31, 0x30, 0x30, /* .._ .DE000007100 */
    0x35, 0x47, 0x4d, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, /* 5GM.L........... */
    0x02, 0x53, 0x05, 0x00, 0x01, 0x01, 0x98, 0x04, 0x5f, 0x25, 0x06, 0x01, 0x02, 0x00, 0x02, 0x00, /* .S......_%...... */
    0x03, 0x5f, 0x24, 0x06, 0x01, 0x02, 0x00, 0x02, 0x00, 0x04, 0x65, 0x5e, 0x73, 0x2d, 0x06, 0x09, /* ._$.......e^s-.. */
    0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x02, 0x80, 0x20, 0x98, 0x71, 0xac, 0xea, 0x94, /* .......... .q... */
    0x79, 0xf5, 0xf3, 0x36, 0xb1, 0x24, 0xb8, 0x71, 0xb7, 0xb1, 0xfb, 0x4d, 0xf4, 0x06, 0x46, 0xb8, /* y..6.$.q...M..F. */
    0x69, 0x35, 0x7c, 0xba, 0x77, 0x4d, 0x3c, 0x10, 0x83, 0x41, 0x97, 0x73, 0x2d, 0x06, 0x09, 0x04, /* i5|.wM<..A.s-... */
    0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x80, 0x20, 0x36, 0x46, 0xb9, 0x42, 0xb1, 0xb1, /* ......... 6F.B.. */
    0x0e, 0x87, 0x7d, 0x25, 0x67, 0xec, 0x2f, 0xf5, 0x57, 0x7c, 0xd9, 0xa5, 0x95, 0x6d, 0x29, 0x44, /* ..}%g./.W|...m)D */
    0x4b, 0xad, 0x45, 0xdc, 0x7c, 0x21, 0x9f, 0xce, 0x6a, 0x71, 0x5f, 0x37, 0x40, 0x5e, 0xb7, 0x6b, /* K.E.|!..jq_7@^.k */
    0x7a, 0xcc, 0x56, 0x59, 0xfb, 0x97, 0xc2, 0x7a, 0x94, 0xeb, 0xf3, 0xfb, 0x32, 0x74, 0x28, 0x1c, /* z.VY...z....2t(. */
    0x63, 0xd4, 0x8e, 0x95, 0x9f, 0x6f, 0xf1, 0xb4, 0x37, 0x18, 0x04, 0x86, 0x3d, 0x68, 0x98, 0xf2, /* c....o..7...=h.. */
    0xcc, 0x62, 0xf2, 0x5d, 0x1c, 0xc0, 0x71, 0x87, 0x26, 0xf7, 0xa6, 0xc2, 0x59, 0xdd, 0x77, 0x54, /* .b.]..q.&...Y.wT */
    0xe0, 0x04, 0x54, 0xab, 0x6d, 0x2e, 0x13, 0x4d, 0x9f, 0xa3, 0x2b, 0x11, 0x67                    /* ..T.m..M..+.g */
    };
static unsigned char kba_dvca_cert_2[] = {
    0x7f, 0x21, 0x81, 0xe7, 0x7f, 0x4e, 0x81, 0xa0, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x0e, 0x44, 0x45, /* .!...N.._)..B.DE */
    0x43, 0x56, 0x43, 0x41, 0x65, 0x49, 0x44, 0x30, 0x30, 0x31, 0x30, 0x32, 0x7f, 0x49, 0x4f, 0x06, /* CVCAeID00102.IO. */
    0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86, 0x41, 0x04, 0x46, 0x5f, /* ............A.F_ */
    0x4b, 0x87, 0x11, 0xde, 0x56, 0xa2, 0x4a, 0x03, 0x1a, 0xec, 0xc3, 0x74, 0x8c, 0x0a, 0x13, 0xa6, /* K...V.J....t.... */
    0x3f, 0x56, 0x67, 0x5c, 0x8d, 0xfc, 0x76, 0x8e, 0x1f, 0xfe, 0x69, 0xf9, 0x92, 0x19, 0x2d, 0xab, /* ?Vg\..v...i...-. */
    0x1a, 0x41, 0xec, 0x37, 0xc3, 0xae, 0x23, 0xe1, 0x53, 0x90, 0x65, 0x44, 0xab, 0x9b, 0x84, 0xb4, /* .A.7..#.S.eD.... */
    0x50, 0xc0, 0xbf, 0x2e, 0xe4, 0x62, 0x57, 0x6a, 0x33, 0xf4, 0xa5, 0x0a, 0x17, 0x07, 0x5f, 0x20, /* P....bWj3....._  */
    0x10, 0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30, /* .DEDVeIDDTR10140 */
    0x31, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, 0x02, 0x53, /* 1.L............S */
    0x05, 0x40, 0x05, 0x01, 0xff, 0x87, 0x5f, 0x25, 0x06, 0x01, 0x01, 0x01, 0x01, 0x01, 0x07, 0x5f, /* .@...._%......._ */
    0x24, 0x06, 0x01, 0x02, 0x00, 0x02, 0x01, 0x07, 0x5f, 0x37, 0x40, 0x1b, 0x93, 0x4a, 0x7e, 0xc4, /* $......._7@..J~. */
    0xd2, 0x01, 0xe8, 0xae, 0xb0, 0xe3, 0xbe, 0xda, 0x06, 0x60, 0x6b, 0x7b, 0x47, 0x10, 0xaf, 0xd3, /* .........`k{G... */
    0x9a, 0x13, 0x14, 0x7d, 0x63, 0x2c, 0xf6, 0x72, 0xd6, 0x4b, 0x2f, 0x94, 0xd9, 0x93, 0x55, 0xad, /* ...}c,.r.K/...U. */
    0x4a, 0x3b, 0x16, 0x07, 0x04, 0x86, 0x14, 0x9e, 0xd3, 0x53, 0xe0, 0xdc, 0x9d, 0x8c, 0xa7, 0x6b, /* J;.......S.....k */
    0x28, 0xf7, 0x94, 0x33, 0x07, 0x3b, 0x43, 0x40, 0xb3, 0x8e, 0x97                                /* (..3.;C@... */
};
static unsigned char kba_cert_desc_2[] = {
    0x30, 0x82, 0x02, 0x7a, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x01,
    0xa1, 0x0e, 0x0c, 0x0c, 0x44, 0x2d, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x47, 0x6d, 0x62, 0x48,
    0xa2, 0x18, 0x13, 0x16, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x64,
    0x2d, 0x74, 0x72, 0x75, 0x73, 0x74, 0x2e, 0x6e, 0x65, 0x74, 0xa3, 0x16, 0x0c, 0x14, 0x4b, 0x72,
    0x61, 0x66, 0x74, 0x66, 0x61, 0x68, 0x72, 0x74, 0x2d, 0x42, 0x75, 0x6e, 0x64, 0x65, 0x73, 0x61,
    0x6d, 0x74, 0xa4, 0x25, 0x13, 0x23, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77,
    0x77, 0x2e, 0x6b, 0x62, 0x61, 0x2d, 0x6f, 0x6e, 0x6c, 0x69, 0x6e, 0x65, 0x2e, 0x64, 0x65, 0x2f,
    0x76, 0x7a, 0x72, 0x6f, 0x6e, 0x6c, 0x69, 0x6e, 0x65, 0xa5, 0x82, 0x01, 0x97, 0x0c, 0x82, 0x01,
    0x93, 0x4e, 0x61, 0x6d, 0x65, 0x2c, 0x20, 0x41, 0x6e, 0x73, 0x63, 0x68, 0x72, 0x69, 0x66, 0x74,
    0x20, 0x75, 0x6e, 0x64, 0x20, 0x45, 0x2d, 0x4d, 0x61, 0x69, 0x6c, 0x2d, 0x41, 0x64, 0x72, 0x65,
    0x73, 0x73, 0x65, 0x20, 0x64, 0x65, 0x73, 0x20, 0x44, 0x69, 0x65, 0x6e, 0x73, 0x74, 0x61, 0x6e,
    0x62, 0x69, 0x65, 0x74, 0x65, 0x72, 0x73, 0x3a, 0x0d, 0x0a, 0x4b, 0x72, 0x61, 0x66, 0x74, 0x66,
    0x61, 0x68, 0x72, 0x74, 0x2d, 0x42, 0x75, 0x6e, 0x64, 0x65, 0x73, 0x61, 0x6d, 0x74, 0x0d, 0x0a,
    0x46, 0xc3, 0xb6, 0x72, 0x64, 0x65, 0x73, 0x74, 0x72, 0x61, 0x73, 0x73, 0x65, 0x20, 0x31, 0x36,
    0x0d, 0x0a, 0x32, 0x34, 0x39, 0x34, 0x34, 0x20, 0x46, 0x6c, 0x65, 0x6e, 0x73, 0x62, 0x75, 0x72,
    0x67, 0x0d, 0x0a, 0x56, 0x5a, 0x52, 0x2d, 0x48, 0x69, 0x6c, 0x66, 0x65, 0x40, 0x6b, 0x62, 0x61,
    0x2e, 0x64, 0x65, 0x0d, 0x0a, 0x0d, 0x0a, 0x47, 0x65, 0x73, 0x63, 0x68, 0xc3, 0xa4, 0x66, 0x74,
    0x73, 0x7a, 0x77, 0x65, 0x63, 0x6b, 0x3a, 0x0d, 0x0a, 0x2d, 0x20, 0x41, 0x75, 0x73, 0x6b, 0x75,
    0x6e, 0x66, 0x74, 0x20, 0x61, 0x75, 0x73, 0x20, 0x64, 0x65, 0x6d, 0x20, 0x56, 0x65, 0x72, 0x6b,
    0x65, 0x68, 0x72, 0x73, 0x7a, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x72, 0x65, 0x67, 0x69, 0x73,
    0x74, 0x65, 0x72, 0x20, 0x2d, 0x0d, 0x0a, 0x0d, 0x0a, 0x5a, 0x75, 0x73, 0x74, 0xc3, 0xa4, 0x6e,
    0x64, 0x69, 0x67, 0x65, 0x20, 0x44, 0x61, 0x74, 0x65, 0x6e, 0x73, 0x63, 0x68, 0x75, 0x74, 0x7a,
    0x62, 0x65, 0x68, 0xc3, 0xb6, 0x72, 0x64, 0x65, 0x3a, 0x0d, 0x0a, 0x44, 0x65, 0x72, 0x20, 0x42,
    0x75, 0x6e, 0x64, 0x65, 0x73, 0x62, 0x65, 0x61, 0x75, 0x66, 0x74, 0x72, 0x61, 0x67, 0x74, 0x65,
    0x20, 0x66, 0xc3, 0xbc, 0x72, 0x20, 0x64, 0x65, 0x6e, 0x20, 0x44, 0x61, 0x74, 0x65, 0x6e, 0x73,
    0x63, 0x68, 0x75, 0x74, 0x7a, 0x20, 0x75, 0x6e, 0x64, 0x20, 0x64, 0x69, 0x65, 0x20, 0x49, 0x6e,
    0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x66, 0x72, 0x65, 0x69, 0x68, 0x65,
    0x69, 0x74, 0x0d, 0x0a, 0x48, 0x75, 0x73, 0x61, 0x72, 0x65, 0x6e, 0x73, 0x74, 0x72, 0x61, 0xc3,
    0x9f, 0x65, 0x20, 0x33, 0x30, 0x0d, 0x0a, 0x35, 0x33, 0x31, 0x31, 0x37, 0x20, 0x42, 0x6f, 0x6e,
    0x6e, 0x0d, 0x0a, 0x30, 0x32, 0x32, 0x38, 0x2d, 0x39, 0x39, 0x37, 0x37, 0x39, 0x39, 0x2d, 0x30,
    0x0d, 0x0a, 0x70, 0x6f, 0x73, 0x74, 0x73, 0x74, 0x65, 0x6c, 0x6c, 0x65, 0x40, 0x62, 0x66, 0x64,
    0x69, 0x2e, 0x62, 0x75, 0x6e, 0x64, 0x2e, 0x64, 0x65, 0x0d, 0x0a, 0x68, 0x74, 0x74, 0x70, 0x3a,
    0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x62, 0x66, 0x64, 0x69, 0x2e, 0x62, 0x75, 0x6e, 0x64, 0x2e,
    0x64, 0x65, 0x0d, 0x0a, 0xa7, 0x68, 0x31, 0x66, 0x04, 0x20, 0x2e, 0x15, 0x78, 0x88, 0x58, 0xe5,
    0x6a, 0x91, 0xa4, 0x59, 0xbb, 0x70, 0x86, 0x94, 0x3a, 0x5a, 0x3a, 0xb8, 0x79, 0xf8, 0x8f, 0x72,
    0xee, 0xe7, 0x2d, 0x5b, 0x82, 0x02, 0xb0, 0x35, 0x94, 0x3d, 0x04, 0x20, 0x80, 0x5a, 0xd7, 0x54,
    0x56, 0x8e, 0x47, 0x2c, 0x47, 0x61, 0xd5, 0x2d, 0x41, 0x0f, 0xb9, 0x91, 0x28, 0xab, 0x4c, 0xe2,
    0xd7, 0x50, 0xfd, 0xa3, 0xa8, 0xda, 0x8f, 0xba, 0x67, 0xbb, 0x14, 0xeb, 0x04, 0x20, 0xf4, 0x31,
    0x0a, 0x18, 0x1e, 0x71, 0x06, 0x4c, 0x36, 0xac, 0x95, 0xa0, 0x66, 0x68, 0xf0, 0xb7, 0x7a, 0x30,
    0xd5, 0x99, 0x58, 0x67, 0x71, 0x3a, 0x73, 0xe5, 0xf1, 0x28, 0x6e, 0xd9, 0x90, 0x73,
};
static unsigned char kba_cert_4[] = {
    0x7f, 0x21, 0x82, 0x01, 0x48, 0x7f, 0x4e, 0x82, 0x01, 0x00, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x10, /* .!..H.N..._)..B. */
    0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30, 0x31, /* DEDVeIDDTR101401 */
    0x7f, 0x49, 0x4f, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86, /* .IO............. */
    0x41, 0x04, 0x4f, 0x27, 0x73, 0x2a, 0x5a, 0x78, 0x79, 0x3e, 0xe6, 0x6d, 0x2c, 0x33, 0x8b, 0x43, /* A.O's*Zxy>.m,3.C */
    0x7c, 0x1f, 0x38, 0x58, 0x0b, 0x1b, 0xa0, 0x68, 0x92, 0xc3, 0xc7, 0x18, 0x48, 0x1f, 0xc4, 0x7e, /* |.8X...h....H..~ */
    0x16, 0xdf, 0x14, 0x1d, 0xf3, 0x81, 0xcd, 0xd1, 0xee, 0x45, 0xf9, 0xeb, 0x60, 0x54, 0x8d, 0x2b, /* .........E..`T.+ */
    0x97, 0x2b, 0x74, 0x65, 0xc2, 0x9a, 0xbb, 0x03, 0x14, 0x1d, 0x7e, 0xe0, 0x38, 0x26, 0xe5, 0x04, /* .+te......~.8&.. */
    0x41, 0xc5, 0x5f, 0x20, 0x0e, 0x44, 0x45, 0x30, 0x30, 0x30, 0x30, 0x30, 0x37, 0x31, 0x30, 0x30, /* A._ .DE000007100 */
    0x35, 0x4b, 0x41, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, /* 5KA.L........... */
    0x02, 0x53, 0x05, 0x00, 0x01, 0x01, 0x98, 0x04, 0x5f, 0x25, 0x06, 0x01, 0x02, 0x00, 0x02, 0x00, /* .S......_%...... */
    0x09, 0x5f, 0x24, 0x06, 0x01, 0x02, 0x00, 0x02, 0x01, 0x00, 0x65, 0x5e, 0x73, 0x2d, 0x06, 0x09, /* ._$.......e^s-.. */
    0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x02, 0x80, 0x20, 0x98, 0x71, 0xac, 0xea, 0x94, /* .......... .q... */
    0x79, 0xf5, 0xf3, 0x36, 0xb1, 0x24, 0xb8, 0x71, 0xb7, 0xb1, 0xfb, 0x4d, 0xf4, 0x06, 0x46, 0xb8, /* y..6.$.q...M..F. */
    0x69, 0x35, 0x7c, 0xba, 0x77, 0x4d, 0x3c, 0x10, 0x83, 0x41, 0x97, 0x73, 0x2d, 0x06, 0x09, 0x04, /* i5|.wM<..A.s-... */
    0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x80, 0x20, 0x36, 0x46, 0xb9, 0x42, 0xb1, 0xb1, /* ......... 6F.B.. */
    0x0e, 0x87, 0x7d, 0x25, 0x67, 0xec, 0x2f, 0xf5, 0x57, 0x7c, 0xd9, 0xa5, 0x95, 0x6d, 0x29, 0x44, /* ..}%g./.W|...m)D */
    0x4b, 0xad, 0x45, 0xdc, 0x7c, 0x21, 0x9f, 0xce, 0x6a, 0x71, 0x5f, 0x37, 0x40, 0x11, 0xb8, 0x53, /* K.E.|!..jq_7@..S */
    0xd8, 0x71, 0x95, 0xb3, 0x40, 0xfd, 0x78, 0xa3, 0x97, 0x01, 0xfe, 0xf3, 0xab, 0xdb, 0x94, 0x12, /* .q..@.x......... */
    0x1e, 0xb5, 0x9c, 0xc5, 0xfd, 0x50, 0x35, 0xdc, 0xfc, 0xa3, 0xd6, 0x52, 0xa0, 0x85, 0x67, 0xaf, /* .....P5....R..g. */
    0x8c, 0x13, 0xc1, 0x3b, 0x02, 0x23, 0x84, 0xc9, 0xc4, 0x76, 0x6c, 0x6c, 0x41, 0x99, 0xa4, 0xa1, /* ...;.#...vllA... */
    0x13, 0x26, 0x6e, 0xbf, 0x5f, 0xe5, 0xd8, 0x73, 0x8e, 0x65, 0x51, 0xb0, 0x14,                   /* .&n._..s.eQ.. */
    };

/* https://www.meineschufa.de/index.php */
static unsigned char schufa_cert[] = {
   0x7f, 0x21, 0x82, 0x01, 0x48, 0x7f, 0x4e, 0x82, 0x01, 0x00, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x10,
   0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30, 0x31,
   0x7f, 0x49, 0x4f, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86,
   0x41, 0x04, 0x30, 0x56, 0xd3, 0xf8, 0x41, 0x9d, 0xc5, 0x2f, 0x8c, 0xcb, 0xfc, 0x26, 0xe5, 0x83,
   0x70, 0x86, 0x63, 0xaa, 0x4b, 0x4a, 0xc6, 0xd9, 0x90, 0x43, 0xa5, 0x9a, 0x0e, 0x08, 0xa1, 0x69,
   0x44, 0x59, 0x2d, 0xbc, 0x38, 0x7d, 0xe0, 0x1a, 0x10, 0x1e, 0x50, 0x35, 0xdb, 0x63, 0x2e, 0x17,
   0xaf, 0x39, 0x12, 0x66, 0x2d, 0x31, 0xce, 0x31, 0x73, 0x56, 0x9e, 0xc9, 0x98, 0xe9, 0xf6, 0x18,
   0xdc, 0x3f, 0x5f, 0x20, 0x0e, 0x44, 0x45, 0x30, 0x30, 0x30, 0x30, 0x30, 0x32, 0x35, 0x30, 0x30,
   0x34, 0x46, 0x46, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02,
   0x02, 0x53, 0x05, 0x00, 0x01, 0x01, 0xd8, 0x04, 0x5f, 0x25, 0x06, 0x01, 0x01, 0x01, 0x02, 0x00,
   0x09, 0x5f, 0x24, 0x06, 0x01, 0x01, 0x01, 0x02, 0x01, 0x00, 0x65, 0x5e, 0x73, 0x2d, 0x06, 0x09,
   0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x02, 0x80, 0x20, 0x8d, 0xd1, 0x91, 0xb4, 0x1b,
   0x37, 0x63, 0x0e, 0x7e, 0x75, 0x9b, 0xb2, 0xe1, 0x0f, 0xb5, 0x0a, 0x7c, 0x59, 0x84, 0xdb, 0x09,
   0x6a, 0x77, 0x14, 0xbe, 0x17, 0xbe, 0x54, 0x0c, 0x7c, 0x2d, 0x9d, 0x73, 0x2d, 0x06, 0x09, 0x04,
   0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x80, 0x20, 0x17, 0xf5, 0x83, 0xb9, 0xdb, 0x3f,
   0x79, 0x01, 0xa6, 0x05, 0xc3, 0x4e, 0xe1, 0x81, 0x02, 0x28, 0xa9, 0xd2, 0x03, 0x98, 0x57, 0x92,
   0xc8, 0x2f, 0xb6, 0x37, 0x28, 0x24, 0x98, 0x5a, 0x66, 0x1d, 0x5f, 0x37, 0x40, 0x5c, 0xb1, 0x30,
   0x87, 0xba, 0x2e, 0xd6, 0xf9, 0x66, 0x47, 0xae, 0x0f, 0xca, 0x6a, 0x7c, 0x51, 0xfa, 0xdb, 0xd9,
   0x1e, 0x20, 0x78, 0xdd, 0x70, 0x68, 0x56, 0xd5, 0x27, 0xa8, 0x34, 0xf9, 0xdc, 0x88, 0x1c, 0x2a,
   0x7b, 0x4c, 0x1d, 0xd8, 0x52, 0xf5, 0x8c, 0xe4, 0x1f, 0xfb, 0xd2, 0x79, 0x21, 0x71, 0x3d, 0xa6,
   0x3a, 0x59, 0xf6, 0xae, 0xe2, 0xed, 0xbe, 0xe0, 0x9c, 0xc2, 0x3c, 0x9f, 0x26,
};
static unsigned char schufa_dvcert[] = {
   0x7f, 0x21, 0x81, 0xe7, 0x7f, 0x4e, 0x81, 0xa0, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x0e, 0x44, 0x45,
   0x43, 0x56, 0x43, 0x41, 0x65, 0x49, 0x44, 0x30, 0x30, 0x31, 0x30, 0x32, 0x7f, 0x49, 0x4f, 0x06,
   0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86, 0x41, 0x04, 0x46, 0x5f,
   0x4b, 0x87, 0x11, 0xde, 0x56, 0xa2, 0x4a, 0x03, 0x1a, 0xec, 0xc3, 0x74, 0x8c, 0x0a, 0x13, 0xa6,
   0x3f, 0x56, 0x67, 0x5c, 0x8d, 0xfc, 0x76, 0x8e, 0x1f, 0xfe, 0x69, 0xf9, 0x92, 0x19, 0x2d, 0xab,
   0x1a, 0x41, 0xec, 0x37, 0xc3, 0xae, 0x23, 0xe1, 0x53, 0x90, 0x65, 0x44, 0xab, 0x9b, 0x84, 0xb4,
   0x50, 0xc0, 0xbf, 0x2e, 0xe4, 0x62, 0x57, 0x6a, 0x33, 0xf4, 0xa5, 0x0a, 0x17, 0x07, 0x5f, 0x20,
   0x10, 0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30,
   0x31, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, 0x02, 0x53,
   0x05, 0x40, 0x05, 0x01, 0xff, 0x87, 0x5f, 0x25, 0x06, 0x01, 0x01, 0x01, 0x01, 0x01, 0x07, 0x5f,
   0x24, 0x06, 0x01, 0x02, 0x00, 0x02, 0x01, 0x07, 0x5f, 0x37, 0x40, 0x1b, 0x93, 0x4a, 0x7e, 0xc4,
   0xd2, 0x01, 0xe8, 0xae, 0xb0, 0xe3, 0xbe, 0xda, 0x06, 0x60, 0x6b, 0x7b, 0x47, 0x10, 0xaf, 0xd3,
   0x9a, 0x13, 0x14, 0x7d, 0x63, 0x2c, 0xf6, 0x72, 0xd6, 0x4b, 0x2f, 0x94, 0xd9, 0x93, 0x55, 0xad,
   0x4a, 0x3b, 0x16, 0x07, 0x04, 0x86, 0x14, 0x9e, 0xd3, 0x53, 0xe0, 0xdc, 0x9d, 0x8c, 0xa7, 0x6b,
   0x28, 0xf7, 0x94, 0x33, 0x07, 0x3b, 0x43, 0x40, 0xb3, 0x8e, 0x97
};
static unsigned char schufa_desc[] = {
   0x30, 0x82, 0x03, 0x1b, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x01,
   0xa1, 0x0e, 0x0c, 0x0c, 0x44, 0x2d, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x47, 0x6d, 0x62, 0x48,
   0xa2, 0x18, 0x13, 0x16, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x64,
   0x2d, 0x74, 0x72, 0x75, 0x73, 0x74, 0x2e, 0x6e, 0x65, 0x74, 0xa3, 0x13, 0x0c, 0x11, 0x53, 0x43,
   0x48, 0x55, 0x46, 0x41, 0x20, 0x48, 0x6f, 0x6c, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x41, 0x47, 0xa4,
   0x1b, 0x13, 0x19, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x6d, 0x65,
   0x69, 0x6e, 0x65, 0x53, 0x43, 0x48, 0x55, 0x46, 0x41, 0x2e, 0x64, 0x65, 0xa5, 0x82, 0x02, 0x21,
   0x0c, 0x82, 0x02, 0x1d, 0x4e, 0x61, 0x6d, 0x65, 0x2c, 0x20, 0x41, 0x6e, 0x73, 0x63, 0x68, 0x72,
   0x69, 0x66, 0x74, 0x20, 0x75, 0x6e, 0x64, 0x20, 0x45, 0x2d, 0x4d, 0x61, 0x69, 0x6c, 0x2d, 0x41,
   0x64, 0x72, 0x65, 0x73, 0x73, 0x65, 0x20, 0x64, 0x65, 0x73, 0x20, 0x44, 0x69, 0x65, 0x6e, 0x73,
   0x74, 0x65, 0x61, 0x6e, 0x62, 0x69, 0x65, 0x74, 0x65, 0x72, 0x73, 0x3a, 0x0d, 0x0a, 0x53, 0x43,
   0x48, 0x55, 0x46, 0x41, 0x20, 0x48, 0x6f, 0x6c, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x41, 0x47, 0x0d,
   0x0a, 0x4b, 0x6f, 0x72, 0x6d, 0x6f, 0x72, 0x61, 0x6e, 0x77, 0x65, 0x67, 0x20, 0x35, 0x0d, 0x0a,
   0x36, 0x35, 0x32, 0x30, 0x31, 0x20, 0x57, 0x69, 0x65, 0x73, 0x62, 0x61, 0x64, 0x65, 0x6e, 0x0d,
   0x0a, 0x65, 0x70, 0x61, 0x40, 0x73, 0x63, 0x68, 0x75, 0x66, 0x61, 0x2e, 0x64, 0x65, 0x0d, 0x0a,
   0x0d, 0x0a, 0x47, 0x65, 0x73, 0x63, 0x68, 0xc3, 0xa4, 0x66, 0x74, 0x73, 0x7a, 0x77, 0x65, 0x63,
   0x6b, 0x3a, 0x0d, 0x0a, 0x2d, 0x20, 0x52, 0x65, 0x67, 0x69, 0x73, 0x74, 0x72, 0x69, 0x65, 0x72,
   0x75, 0x6e, 0x67, 0x20, 0x75, 0x6e, 0x64, 0x20, 0x4c, 0x6f, 0x67, 0x69, 0x6e, 0x20, 0x61, 0x6d,
   0x20, 0x50, 0x6f, 0x72, 0x74, 0x61, 0x6c, 0x20, 0x22, 0x4d, 0x65, 0x69, 0x6e, 0x65, 0x20, 0x53,
   0x43, 0x48, 0x55, 0x46, 0x41, 0x2d, 0x41, 0x75, 0x73, 0x6b, 0x75, 0x6e, 0x66, 0x74, 0x20, 0x4f,
   0x6e, 0x6c, 0x69, 0x6e, 0x65, 0x22, 0x2d, 0x0d, 0x0a, 0x0d, 0x0a, 0x48, 0x69, 0x6e, 0x77, 0x65,
   0x69, 0x73, 0x20, 0x61, 0x75, 0x66, 0x20, 0x64, 0x69, 0x65, 0x20, 0x66, 0xc3, 0xbc, 0x72, 0x20,
   0x64, 0x65, 0x6e, 0x20, 0x44, 0x69, 0x65, 0x6e, 0x73, 0x74, 0x65, 0x61, 0x6e, 0x62, 0x69, 0x65,
   0x74, 0x65, 0x72, 0x20, 0x7a, 0x75, 0x73, 0x74, 0xc3, 0xa4, 0x6e, 0x64, 0x69, 0x67, 0x65, 0x6e,
   0x20, 0x53, 0x74, 0x65, 0x6c, 0x6c, 0x65, 0x6e, 0x2c, 0x20, 0x64, 0x69, 0x65, 0x20, 0x64, 0x69,
   0x65, 0x20, 0x45, 0x69, 0x6e, 0x68, 0x61, 0x6c, 0x74, 0x75, 0x6e, 0x67, 0x20, 0x64, 0x65, 0x72,
   0x20, 0x56, 0x6f, 0x72, 0x73, 0x63, 0x68, 0x72, 0x69, 0x66, 0x74, 0x65, 0x6e, 0x20, 0x7a, 0x75,
   0x6d, 0x20, 0x44, 0x61, 0x74, 0x65, 0x6e, 0x73, 0x63, 0x68, 0x75, 0x74, 0x7a, 0x20, 0x6b, 0x6f,
   0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x6c, 0x69, 0x65, 0x72, 0x65, 0x6e, 0x3a, 0x0d, 0x0a, 0x44, 0x65,
   0x72, 0x20, 0x48, 0x65, 0x73, 0x73, 0x69, 0x73, 0x63, 0x68, 0x65, 0x20, 0x44, 0x61, 0x74, 0x65,
   0x6e, 0x73, 0x63, 0x68, 0x75, 0x74, 0x7a, 0x62, 0x65, 0x61, 0x75, 0x66, 0x74, 0x72, 0x61, 0x67,
   0x74, 0x65, 0x0d, 0x0a, 0x4c, 0x75, 0x69, 0x73, 0x65, 0x6e, 0x70, 0x6c, 0x61, 0x74, 0x7a, 0x20,
   0x32, 0x0d, 0x0a, 0x36, 0x34, 0x32, 0x38, 0x33, 0x20, 0x44, 0x61, 0x72, 0x6d, 0x73, 0x74, 0x61,
   0x64, 0x74, 0x0d, 0x0a, 0x30, 0x36, 0x31, 0x35, 0x31, 0x2f, 0x31, 0x32, 0x2d, 0x30, 0x0d, 0x0a,
   0x44, 0x61, 0x74, 0x65, 0x6e, 0x73, 0x63, 0x68, 0x75, 0x74, 0x7a, 0x40, 0x72, 0x70, 0x64, 0x61,
   0x2e, 0x68, 0x65, 0x73, 0x73, 0x65, 0x6e, 0x2e, 0x64, 0x65, 0x0d, 0x0a, 0x68, 0x74, 0x74, 0x70,
   0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x72, 0x70, 0x2d, 0x64, 0x61, 0x72, 0x6d, 0x73, 0x74,
   0x61, 0x64, 0x74, 0x2e, 0x68, 0x65, 0x73, 0x73, 0x65, 0x6e, 0x2e, 0x64, 0x65, 0x2f, 0x0d, 0x0a,
   0x41, 0x6e, 0x73, 0x70, 0x72, 0x65, 0x63, 0x68, 0x70, 0x61, 0x72, 0x74, 0x6e, 0x65, 0x72, 0x3a,
   0x20, 0x50, 0x72, 0x6f, 0x66, 0x2e, 0x20, 0x44, 0x72, 0x2e, 0x20, 0x4d, 0x69, 0x63, 0x68, 0x61,
   0x65, 0x6c, 0x20, 0x52, 0x6f, 0x6e, 0x65, 0x6c, 0x6c, 0x65, 0x6e, 0x66, 0x69, 0x74, 0x73, 0x63,
   0x68, 0xa7, 0x81, 0x8b, 0x31, 0x81, 0x88, 0x04, 0x20, 0x2e, 0x15, 0x78, 0x88, 0x58, 0xe5, 0x6a,
   0x91, 0xa4, 0x59, 0xbb, 0x70, 0x86, 0x94, 0x3a, 0x5a, 0x3a, 0xb8, 0x79, 0xf8, 0x8f, 0x72, 0xee,
   0xe7, 0x2d, 0x5b, 0x82, 0x02, 0xb0, 0x35, 0x94, 0x3d, 0x04, 0x20, 0x80, 0x5a, 0xd7, 0x54, 0x56,
   0x8e, 0x47, 0x2c, 0x47, 0x61, 0xd5, 0x2d, 0x41, 0x0f, 0xb9, 0x91, 0x28, 0xab, 0x4c, 0xe2, 0xd7,
   0x50, 0xfd, 0xa3, 0xa8, 0xda, 0x8f, 0xba, 0x67, 0xbb, 0x14, 0xeb, 0x04, 0x20, 0xab, 0xf4, 0x75,
   0xcb, 0xa2, 0xf2, 0xc1, 0x10, 0xaf, 0x75, 0x49, 0x38, 0xe6, 0x2d, 0x58, 0xd1, 0xa9, 0x2f, 0x6e,
   0x05, 0x10, 0x1e, 0x98, 0x48, 0xc2, 0x8c, 0x45, 0x82, 0x3d, 0x43, 0xdd, 0x30, 0x04, 0x20, 0xea,
   0x58, 0xa5, 0xe7, 0xcc, 0xa4, 0x9c, 0x0b, 0xa6, 0x19, 0x10, 0x12, 0x2f, 0x8a, 0x5a, 0x37, 0xe2,
   0x8c, 0x2a, 0x8b, 0xde, 0x34, 0xd3, 0xdb, 0xc2, 0xf7, 0xcc, 0xe4, 0x36, 0xb8, 0x09, 0xdc,
};

/* https://makler.allianz.de/wps/myportal/epa/login?TARGETURL=/wps/myportal/tw/makler.allianz.de/index.html?smartcard=false */
static unsigned char allianz_desc[] = {
   0x30, 0x82, 0x03, 0x86, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x01,
   0xa1, 0x0e, 0x0c, 0x0c, 0x44, 0x2d, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x47, 0x6d, 0x62, 0x48,
   0xa2, 0x18, 0x13, 0x16, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x64,
   0x2d, 0x74, 0x72, 0x75, 0x73, 0x74, 0x2e, 0x6e, 0x65, 0x74, 0xa3, 0x18, 0x0c, 0x16, 0x41, 0x6c,
   0x6c, 0x69, 0x61, 0x6e, 0x7a, 0x20, 0x44, 0x65, 0x75, 0x74, 0x73, 0x63, 0x68, 0x6c, 0x61, 0x6e,
   0x64, 0x20, 0x41, 0x47, 0xa4, 0x0c, 0x13, 0x0a, 0x61, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x7a, 0x2e,
   0x64, 0x65, 0xa5, 0x82, 0x02, 0x96, 0x0c, 0x82, 0x02, 0x92, 0x4e, 0x61, 0x6d, 0x65, 0x2c, 0x20,
   0x41, 0x6e, 0x73, 0x63, 0x68, 0x72, 0x69, 0x66, 0x74, 0x20, 0x75, 0x6e, 0x64, 0x20, 0x45, 0x2d,
   0x4d, 0x61, 0x69, 0x6c, 0x2d, 0x41, 0x64, 0x72, 0x65, 0x73, 0x73, 0x65, 0x20, 0x64, 0x65, 0x73,
   0x20, 0x44, 0x69, 0x65, 0x6e, 0x73, 0x74, 0x65, 0x61, 0x6e, 0x62, 0x69, 0x65, 0x74, 0x65, 0x72,
   0x73, 0x3a, 0x0d, 0x0a, 0x41, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x7a, 0x20, 0x44, 0x65, 0x75, 0x74,
   0x73, 0x63, 0x68, 0x6c, 0x61, 0x6e, 0x64, 0x20, 0x41, 0x47, 0x0d, 0x0a, 0x4b, 0xc3, 0xb6, 0x6e,
   0x69, 0x67, 0x69, 0x6e, 0x73, 0x74, 0x72, 0x61, 0xc3, 0x9f, 0x65, 0x20, 0x32, 0x38, 0x0d, 0x0a,
   0x38, 0x30, 0x38, 0x30, 0x32, 0x20, 0x4d, 0xc3, 0xbc, 0x6e, 0x63, 0x68, 0x65, 0x6e, 0x0d, 0x0a,
   0x69, 0x6e, 0x66, 0x6f, 0x40, 0x61, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x7a, 0x2e, 0x64, 0x65, 0x0d,
   0x0a, 0x0d, 0x0a, 0x47, 0x65, 0x73, 0x63, 0x68, 0xc3, 0xa4, 0x66, 0x74, 0x73, 0x7a, 0x77, 0x65,
   0x63, 0x6b, 0x3a, 0x0d, 0x0a, 0x2d, 0x20, 0x44, 0x69, 0x65, 0x20, 0x44, 0x61, 0x74, 0x65, 0x6e,
   0x20, 0x64, 0x65, 0x73, 0x20, 0x50, 0x65, 0x72, 0x73, 0x6f, 0x6e, 0x61, 0x6c, 0x61, 0x75, 0x73,
   0x77, 0x65, 0x69, 0x73, 0x65, 0x73, 0x20, 0x77, 0x65, 0x72, 0x64, 0x65, 0x6e, 0x20, 0x76, 0x6f,
   0x6e, 0x20, 0x64, 0x65, 0x72, 0x20, 0x41, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x7a, 0x20, 0x44, 0x65,
   0x75, 0x74, 0x73, 0x63, 0x68, 0x6c, 0x61, 0x6e, 0x64, 0x20, 0x41, 0x47, 0x20, 0x7a, 0x75, 0x6d,
   0x20, 0x5a, 0x77, 0x65, 0x63, 0x6b, 0x20, 0x64, 0x65, 0x72, 0x20, 0x49, 0x64, 0x65, 0x6e, 0x74,
   0x69, 0x66, 0x69, 0x7a, 0x69, 0x65, 0x72, 0x75, 0x6e, 0x67, 0x20, 0x66, 0xc3, 0xbc, 0x72, 0x20,
   0x64, 0x69, 0x65, 0x20, 0x4e, 0x75, 0x74, 0x7a, 0x75, 0x6e, 0x67, 0x20, 0x76, 0x6f, 0x6e, 0x20,
   0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65, 0x74, 0x20, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65,
   0x73, 0x20, 0x76, 0x65, 0x72, 0x77, 0x65, 0x6e, 0x64, 0x65, 0x74, 0x2e, 0x20, 0x2d, 0x0d, 0x0a,
   0x0d, 0x0a, 0x48, 0x69, 0x6e, 0x77, 0x65, 0x69, 0x73, 0x20, 0x61, 0x75, 0x66, 0x20, 0x64, 0x69,
   0x65, 0x20, 0x66, 0xc3, 0xbc, 0x72, 0x20, 0x64, 0x65, 0x6e, 0x20, 0x44, 0x69, 0x65, 0x6e, 0x73,
   0x74, 0x65, 0x61, 0x6e, 0x62, 0x69, 0x65, 0x74, 0x65, 0x72, 0x20, 0x7a, 0x75, 0x73, 0x74, 0xc3,
   0xa4, 0x6e, 0x64, 0x69, 0x67, 0x65, 0x6e, 0x20, 0x53, 0x74, 0x65, 0x6c, 0x6c, 0x65, 0x6e, 0x2c,
   0x20, 0x64, 0x69, 0x65, 0x20, 0x64, 0x69, 0x65, 0x20, 0x45, 0x69, 0x6e, 0x68, 0x61, 0x6c, 0x74,
   0x75, 0x6e, 0x67, 0x20, 0x64, 0x65, 0x72, 0x20, 0x56, 0x6f, 0x72, 0x73, 0x63, 0x68, 0x72, 0x69,
   0x66, 0x74, 0x65, 0x6e, 0x20, 0x7a, 0x75, 0x6d, 0x20, 0x44, 0x61, 0x74, 0x65, 0x6e, 0x73, 0x63,
   0x68, 0x75, 0x74, 0x7a, 0x20, 0x6b, 0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x6c, 0x69, 0x65, 0x72,
   0x65, 0x6e, 0x3a, 0x0d, 0x0a, 0x44, 0x65, 0x72, 0x20, 0x42, 0x61, 0x79, 0x65, 0x72, 0x69, 0x73,
   0x63, 0x68, 0x65, 0x20, 0x4c, 0x61, 0x6e, 0x64, 0x65, 0x73, 0x62, 0x65, 0x61, 0x75, 0x66, 0x74,
   0x72, 0x61, 0x67, 0x74, 0x65, 0x20, 0x66, 0xc3, 0xbc, 0x72, 0x20, 0x64, 0x65, 0x6e, 0x20, 0x44,
   0x61, 0x74, 0x65, 0x6e, 0x73, 0x63, 0x68, 0x75, 0x74, 0x7a, 0x0d, 0x0a, 0x57, 0x61, 0x67, 0x6d,
   0xc3, 0xbc, 0x6c, 0x6c, 0x65, 0x72, 0x73, 0x74, 0x72, 0x61, 0xc3, 0x9f, 0x65, 0x20, 0x31, 0x38,
   0x0d, 0x0a, 0x38, 0x30, 0x35, 0x33, 0x38, 0x20, 0x4d, 0xc3, 0xbc, 0x6e, 0x63, 0x68, 0x65, 0x6e,
   0x0d, 0x0a, 0x30, 0x38, 0x39, 0x20, 0x2f, 0x20, 0x32, 0x31, 0x20, 0x32, 0x36, 0x20, 0x37, 0x32,
   0x20, 0x2d, 0x20, 0x30, 0x0d, 0x0a, 0x70, 0x6f, 0x73, 0x74, 0x73, 0x74, 0x65, 0x6c, 0x6c, 0x65,
   0x40, 0x64, 0x61, 0x74, 0x65, 0x6e, 0x73, 0x63, 0x68, 0x75, 0x74, 0x7a, 0x2d, 0x62, 0x61, 0x79,
   0x65, 0x72, 0x6e, 0x2e, 0x64, 0x65, 0x0d, 0x0a, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77,
   0x77, 0x77, 0x2e, 0x64, 0x61, 0x74, 0x65, 0x6e, 0x73, 0x63, 0x68, 0x75, 0x74, 0x7a, 0x2d, 0x62,
   0x61, 0x79, 0x65, 0x72, 0x6e, 0x2e, 0x64, 0x65, 0x2f, 0x0d, 0x0a, 0x41, 0x6e, 0x73, 0x70, 0x72,
   0x65, 0x63, 0x68, 0x70, 0x61, 0x72, 0x74, 0x6e, 0x65, 0x72, 0x3a, 0x20, 0x44, 0x72, 0x2e, 0x20,
   0x54, 0x68, 0x6f, 0x6d, 0x61, 0x73, 0x20, 0x50, 0x65, 0x74, 0x72, 0x69, 0xa7, 0x81, 0x8b, 0x31,
   0x81, 0x88, 0x04, 0x20, 0x2e, 0x15, 0x78, 0x88, 0x58, 0xe5, 0x6a, 0x91, 0xa4, 0x59, 0xbb, 0x70,
   0x86, 0x94, 0x3a, 0x5a, 0x3a, 0xb8, 0x79, 0xf8, 0x8f, 0x72, 0xee, 0xe7, 0x2d, 0x5b, 0x82, 0x02,
   0xb0, 0x35, 0x94, 0x3d, 0x04, 0x20, 0x79, 0x2e, 0x9e, 0xca, 0xb7, 0xec, 0x8b, 0x07, 0x40, 0x77,
   0xab, 0x95, 0x4e, 0x3a, 0xfa, 0x9c, 0xea, 0xfe, 0xe9, 0x61, 0x39, 0x2f, 0x91, 0xc1, 0x1e, 0x6f,
   0x27, 0xec, 0x08, 0xa1, 0x6f, 0x6d, 0x04, 0x20, 0x80, 0x5a, 0xd7, 0x54, 0x56, 0x8e, 0x47, 0x2c,
   0x47, 0x61, 0xd5, 0x2d, 0x41, 0x0f, 0xb9, 0x91, 0x28, 0xab, 0x4c, 0xe2, 0xd7, 0x50, 0xfd, 0xa3,
   0xa8, 0xda, 0x8f, 0xba, 0x67, 0xbb, 0x14, 0xeb, 0x04, 0x20, 0xb9, 0x9a, 0x7b, 0x5d, 0x20, 0x3c,
   0x80, 0x29, 0xa4, 0x6d, 0xe4, 0xd5, 0x96, 0xe6, 0x46, 0x41, 0x5a, 0x4c, 0xcf, 0xc9, 0x46, 0x13,
   0xbc, 0xc7, 0x70, 0x88, 0x64, 0x8c, 0xdc, 0x9f, 0xe1, 0xa7,
};
static unsigned char allianz_dvcert[] = {
   0x7f, 0x21, 0x81, 0xe7, 0x7f, 0x4e, 0x81, 0xa0, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x0e, 0x44, 0x45,
   0x43, 0x56, 0x43, 0x41, 0x65, 0x49, 0x44, 0x30, 0x30, 0x31, 0x30, 0x32, 0x7f, 0x49, 0x4f, 0x06,
   0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86, 0x41, 0x04, 0x46, 0x5f,
   0x4b, 0x87, 0x11, 0xde, 0x56, 0xa2, 0x4a, 0x03, 0x1a, 0xec, 0xc3, 0x74, 0x8c, 0x0a, 0x13, 0xa6,
   0x3f, 0x56, 0x67, 0x5c, 0x8d, 0xfc, 0x76, 0x8e, 0x1f, 0xfe, 0x69, 0xf9, 0x92, 0x19, 0x2d, 0xab,
   0x1a, 0x41, 0xec, 0x37, 0xc3, 0xae, 0x23, 0xe1, 0x53, 0x90, 0x65, 0x44, 0xab, 0x9b, 0x84, 0xb4,
   0x50, 0xc0, 0xbf, 0x2e, 0xe4, 0x62, 0x57, 0x6a, 0x33, 0xf4, 0xa5, 0x0a, 0x17, 0x07, 0x5f, 0x20,
   0x10, 0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30,
   0x31, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, 0x02, 0x53,
   0x05, 0x40, 0x05, 0x01, 0xff, 0x87, 0x5f, 0x25, 0x06, 0x01, 0x01, 0x01, 0x01, 0x01, 0x07, 0x5f,
   0x24, 0x06, 0x01, 0x02, 0x00, 0x02, 0x01, 0x07, 0x5f, 0x37, 0x40, 0x1b, 0x93, 0x4a, 0x7e, 0xc4,
   0xd2, 0x01, 0xe8, 0xae, 0xb0, 0xe3, 0xbe, 0xda, 0x06, 0x60, 0x6b, 0x7b, 0x47, 0x10, 0xaf, 0xd3,
   0x9a, 0x13, 0x14, 0x7d, 0x63, 0x2c, 0xf6, 0x72, 0xd6, 0x4b, 0x2f, 0x94, 0xd9, 0x93, 0x55, 0xad,
   0x4a, 0x3b, 0x16, 0x07, 0x04, 0x86, 0x14, 0x9e, 0xd3, 0x53, 0xe0, 0xdc, 0x9d, 0x8c, 0xa7, 0x6b,
   0x28, 0xf7, 0x94, 0x33, 0x07, 0x3b, 0x43, 0x40, 0xb3, 0x8e, 0x97,
};
static unsigned char allianz_cert[] = {
   0x7f, 0x21, 0x82, 0x01, 0x48, 0x7f, 0x4e, 0x82, 0x01, 0x00, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x10,
   0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30, 0x31,
   0x7f, 0x49, 0x4f, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86,
   0x41, 0x04, 0x2d, 0x82, 0x40, 0x33, 0x30, 0xea, 0xa8, 0xd4, 0x9a, 0x01, 0x27, 0xe9, 0xb3, 0x94,
   0x21, 0x7c, 0x3e, 0xbf, 0x05, 0x95, 0x77, 0xa6, 0xb8, 0x4f, 0x6c, 0x37, 0x92, 0xd7, 0x2d, 0x78,
   0xdf, 0x78, 0x3b, 0x40, 0xc8, 0x0f, 0xcc, 0xe1, 0xe4, 0xf7, 0xec, 0x99, 0xd6, 0xd0, 0xd8, 0xff,
   0xf1, 0xbd, 0x3e, 0xab, 0x0f, 0xfc, 0x63, 0x1b, 0xf8, 0xa1, 0x58, 0xca, 0xba, 0xb3, 0xa7, 0x2c,
   0xc9, 0x44, 0x5f, 0x20, 0x0e, 0x44, 0x45, 0x30, 0x30, 0x30, 0x30, 0x30, 0x34, 0x32, 0x30, 0x30,
   0x34, 0x46, 0x51, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02,
   0x02, 0x53, 0x05, 0x00, 0x00, 0x00, 0x18, 0x04, 0x5f, 0x25, 0x06, 0x01, 0x01, 0x01, 0x02, 0x00,
   0x09, 0x5f, 0x24, 0x06, 0x01, 0x01, 0x01, 0x02, 0x01, 0x00, 0x65, 0x5e, 0x73, 0x2d, 0x06, 0x09,
   0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x02, 0x80, 0x20, 0xbb, 0x3c, 0xf6, 0x55, 0x78,
   0x20, 0xd8, 0x98, 0x7b, 0xfb, 0x51, 0x30, 0xd0, 0x54, 0x2c, 0x4e, 0xd4, 0xf9, 0x92, 0xb6, 0x4c,
   0xe7, 0x6d, 0x1d, 0x7c, 0xf6, 0x71, 0xbf, 0xe6, 0xb0, 0xa8, 0x4a, 0x73, 0x2d, 0x06, 0x09, 0x04,
   0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x80, 0x20, 0x36, 0x00, 0x2a, 0xab, 0x88, 0x34,
   0x58, 0x76, 0x89, 0xfd, 0x43, 0xc2, 0x04, 0x8a, 0xdd, 0x1a, 0x5b, 0xe1, 0xa9, 0x44, 0xe5, 0x06,
   0x8f, 0x1b, 0xb3, 0x3e, 0x40, 0x73, 0x33, 0xdb, 0x72, 0xb8, 0x5f, 0x37, 0x40, 0x18, 0x4e, 0xbc,
   0xf9, 0x87, 0x34, 0xf2, 0xe8, 0xc3, 0x2a, 0xbf, 0xc9, 0x04, 0xe2, 0xb4, 0x04, 0x73, 0xb7, 0x67,
   0x57, 0xcd, 0x38, 0x93, 0x2a, 0x45, 0x3e, 0xdc, 0x31, 0x2c, 0x47, 0xeb, 0x03, 0x5a, 0x00, 0x46,
   0xa6, 0xaf, 0x91, 0x73, 0x91, 0x81, 0x4c, 0x3d, 0x0b, 0x20, 0xfb, 0xda, 0x3f, 0x46, 0x5f, 0xc8,
   0x90, 0xd0, 0x45, 0x95, 0xec, 0xed, 0x0c, 0x19, 0x4c, 0xbf, 0xab, 0x14, 0xcd,
};

/* https://www.mein-cockpit.de/api/eid/finish/ */
static unsigned char mein_cockpit_desc[] = {
   0x30, 0x82, 0x02, 0xb5, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x01,
   0xa1, 0x0e, 0x0c, 0x0c, 0x44, 0x2d, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x47, 0x6d, 0x62, 0x48,
   0xa2, 0x18, 0x13, 0x16, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x64,
   0x2d, 0x74, 0x72, 0x75, 0x73, 0x74, 0x2e, 0x6e, 0x65, 0x74, 0xa3, 0x0d, 0x0c, 0x0b, 0x61, 0x6c,
   0x6c, 0x79, 0x76, 0x65, 0x20, 0x47, 0x6d, 0x62, 0x48, 0xa4, 0x1d, 0x13, 0x1b, 0x68, 0x74, 0x74,
   0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x6d, 0x65, 0x69, 0x6e, 0x2d, 0x63, 0x6f,
   0x63, 0x6b, 0x70, 0x69, 0x74, 0x2e, 0x64, 0x65, 0xa5, 0x82, 0x01, 0xe3, 0x0c, 0x82, 0x01, 0xdf,
   0x4e, 0x61, 0x6d, 0x65, 0x2c, 0x20, 0x41, 0x6e, 0x73, 0x63, 0x68, 0x72, 0x69, 0x66, 0x74, 0x20,
   0x75, 0x6e, 0x64, 0x20, 0x45, 0x2d, 0x4d, 0x61, 0x69, 0x6c, 0x2d, 0x41, 0x64, 0x72, 0x65, 0x73,
   0x73, 0x65, 0x20, 0x64, 0x65, 0x73, 0x20, 0x44, 0x69, 0x65, 0x6e, 0x73, 0x74, 0x65, 0x61, 0x6e,
   0x62, 0x69, 0x65, 0x74, 0x65, 0x72, 0x73, 0x3a, 0x0d, 0x0a, 0x61, 0x6c, 0x6c, 0x79, 0x76, 0x65,
   0x20, 0x47, 0x6d, 0x62, 0x48, 0x0d, 0x0a, 0x41, 0x6c, 0x73, 0x74, 0x65, 0x72, 0x74, 0x6f, 0x72,
   0x20, 0x39, 0x0d, 0x0a, 0x32, 0x30, 0x30, 0x39, 0x35, 0x20, 0x48, 0x61, 0x6d, 0x62, 0x75, 0x72,
   0x67, 0x0d, 0x0a, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x40, 0x61, 0x6c, 0x6c, 0x79, 0x76,
   0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x0d, 0x0a, 0x0d, 0x0a, 0x47, 0x65, 0x73, 0x63, 0x68, 0xc3, 0xa4,
   0x66, 0x74, 0x73, 0x7a, 0x77, 0x65, 0x63, 0x6b, 0x3a, 0x0d, 0x0a, 0x2d, 0x20, 0x45, 0x69, 0x6e,
   0x62, 0x69, 0x6e, 0x64, 0x75, 0x6e, 0x67, 0x20, 0x69, 0x6e, 0x20, 0x64, 0x65, 0x6e, 0x20, 0x52,
   0x65, 0x67, 0x69, 0x73, 0x74, 0x72, 0x69, 0x65, 0x72, 0x75, 0x6e, 0x67, 0x73, 0x2d, 0x20, 0x75,
   0x6e, 0x64, 0x20, 0x41, 0x6e, 0x6d, 0x65, 0x6c, 0x64, 0x65, 0x70, 0x72, 0x6f, 0x7a, 0x65, 0x73,
   0x73, 0x20, 0x61, 0x6d, 0x20, 0x57, 0x65, 0x62, 0x70, 0x6f, 0x72, 0x74, 0x61, 0x6c, 0x20, 0x77,
   0x77, 0x77, 0x2e, 0x6d, 0x65, 0x69, 0x6e, 0x2d, 0x63, 0x6f, 0x63, 0x6b, 0x70, 0x69, 0x74, 0x2e,
   0x64, 0x65, 0x20, 0x2d, 0x0d, 0x0a, 0x0d, 0x0a, 0x44, 0x69, 0x65, 0x20, 0x66, 0xc3, 0xbc, 0x72,
   0x20, 0x53, 0x69, 0x65, 0x20, 0x7a, 0x75, 0x73, 0x74, 0xc3, 0xa4, 0x6e, 0x64, 0x69, 0x67, 0x65,
   0x20, 0x44, 0x61, 0x74, 0x65, 0x6e, 0x73, 0x63, 0x68, 0x75, 0x74, 0x7a, 0x62, 0x65, 0x68, 0xc3,
   0xb6, 0x72, 0x64, 0x65, 0x20, 0x69, 0x73, 0x74, 0x3a, 0x0d, 0x0a, 0x48, 0x61, 0x6d, 0x62, 0x75,
   0x72, 0x67, 0x69, 0x73, 0x63, 0x68, 0x65, 0x72, 0x20, 0x44, 0x61, 0x74, 0x65, 0x6e, 0x73, 0x63,
   0x68, 0x75, 0x74, 0x7a, 0x62, 0x65, 0x61, 0x75, 0x66, 0x74, 0x72, 0x61, 0x67, 0x74, 0x65, 0x72,
   0x0d, 0x0a, 0x4b, 0x6c, 0x6f, 0x73, 0x74, 0x65, 0x72, 0x77, 0x61, 0x6c, 0x6c, 0x20, 0x36, 0x0d,
   0x0a, 0x32, 0x30, 0x30, 0x39, 0x35, 0x20, 0x48, 0x61, 0x6d, 0x62, 0x75, 0x72, 0x67, 0x0d, 0x0a,
   0x30, 0x34, 0x30, 0x2f, 0x34, 0x32, 0x38, 0x20, 0x35, 0x34, 0x2d, 0x34, 0x30, 0x20, 0x34, 0x30,
   0x0d, 0x0a, 0x6d, 0x61, 0x69, 0x6c, 0x62, 0x6f, 0x78, 0x40, 0x64, 0x61, 0x74, 0x65, 0x6e, 0x73,
   0x63, 0x68, 0x75, 0x74, 0x7a, 0x2e, 0x68, 0x61, 0x6d, 0x62, 0x75, 0x72, 0x67, 0x2e, 0x64, 0x65,
   0x0d, 0x0a, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x64, 0x61, 0x74,
   0x65, 0x6e, 0x73, 0x63, 0x68, 0x75, 0x74, 0x7a, 0x2e, 0x68, 0x61, 0x6d, 0x62, 0x75, 0x72, 0x67,
   0x2e, 0x64, 0x65, 0x0d, 0x0a, 0x41, 0x6e, 0x73, 0x70, 0x72, 0x65, 0x63, 0x68, 0x70, 0x61, 0x72,
   0x74, 0x6e, 0x65, 0x72, 0x3a, 0x20, 0x50, 0x72, 0x6f, 0x66, 0x2e, 0x20, 0x44, 0x72, 0x2e, 0x20,
   0x4a, 0x6f, 0x68, 0x61, 0x6e, 0x6e, 0x65, 0x73, 0x20, 0x43, 0x61, 0x73, 0x70, 0x61, 0x72, 0xa7,
   0x68, 0x31, 0x66, 0x04, 0x20, 0x2e, 0x15, 0x78, 0x88, 0x58, 0xe5, 0x6a, 0x91, 0xa4, 0x59, 0xbb,
   0x70, 0x86, 0x94, 0x3a, 0x5a, 0x3a, 0xb8, 0x79, 0xf8, 0x8f, 0x72, 0xee, 0xe7, 0x2d, 0x5b, 0x82,
   0x02, 0xb0, 0x35, 0x94, 0x3d, 0x04, 0x20, 0x80, 0x5a, 0xd7, 0x54, 0x56, 0x8e, 0x47, 0x2c, 0x47,
   0x61, 0xd5, 0x2d, 0x41, 0x0f, 0xb9, 0x91, 0x28, 0xab, 0x4c, 0xe2, 0xd7, 0x50, 0xfd, 0xa3, 0xa8,
   0xda, 0x8f, 0xba, 0x67, 0xbb, 0x14, 0xeb, 0x04, 0x20, 0xe9, 0xb2, 0xb7, 0xe1, 0x43, 0x0e, 0xfd,
   0xe9, 0xe9, 0x9a, 0x25, 0x60, 0x3a, 0xd3, 0x2e, 0x26, 0x71, 0xef, 0xb6, 0xb0, 0x0d, 0x92, 0x14,
   0x39, 0x42, 0x8d, 0xf9, 0x82, 0xce, 0x16, 0x8d, 0x44,
};
static unsigned char mein_cockpit_dvcert[] = {
   0x7f, 0x21, 0x81, 0xe7, 0x7f, 0x4e, 0x81, 0xa0, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x0e, 0x44, 0x45,
   0x43, 0x56, 0x43, 0x41, 0x65, 0x49, 0x44, 0x30, 0x30, 0x31, 0x30, 0x32, 0x7f, 0x49, 0x4f, 0x06,
   0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86, 0x41, 0x04, 0x46, 0x5f,
   0x4b, 0x87, 0x11, 0xde, 0x56, 0xa2, 0x4a, 0x03, 0x1a, 0xec, 0xc3, 0x74, 0x8c, 0x0a, 0x13, 0xa6,
   0x3f, 0x56, 0x67, 0x5c, 0x8d, 0xfc, 0x76, 0x8e, 0x1f, 0xfe, 0x69, 0xf9, 0x92, 0x19, 0x2d, 0xab,
   0x1a, 0x41, 0xec, 0x37, 0xc3, 0xae, 0x23, 0xe1, 0x53, 0x90, 0x65, 0x44, 0xab, 0x9b, 0x84, 0xb4,
   0x50, 0xc0, 0xbf, 0x2e, 0xe4, 0x62, 0x57, 0x6a, 0x33, 0xf4, 0xa5, 0x0a, 0x17, 0x07, 0x5f, 0x20,
   0x10, 0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30,
   0x31, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, 0x02, 0x53,
   0x05, 0x40, 0x05, 0x01, 0xff, 0x87, 0x5f, 0x25, 0x06, 0x01, 0x01, 0x01, 0x01, 0x01, 0x07, 0x5f,
   0x24, 0x06, 0x01, 0x02, 0x00, 0x02, 0x01, 0x07, 0x5f, 0x37, 0x40, 0x1b, 0x93, 0x4a, 0x7e, 0xc4,
   0xd2, 0x01, 0xe8, 0xae, 0xb0, 0xe3, 0xbe, 0xda, 0x06, 0x60, 0x6b, 0x7b, 0x47, 0x10, 0xaf, 0xd3,
   0x9a, 0x13, 0x14, 0x7d, 0x63, 0x2c, 0xf6, 0x72, 0xd6, 0x4b, 0x2f, 0x94, 0xd9, 0x93, 0x55, 0xad,
   0x4a, 0x3b, 0x16, 0x07, 0x04, 0x86, 0x14, 0x9e, 0xd3, 0x53, 0xe0, 0xdc, 0x9d, 0x8c, 0xa7, 0x6b,
   0x28, 0xf7, 0x94, 0x33, 0x07, 0x3b, 0x43, 0x40, 0xb3, 0x8e, 0x97,
};
static unsigned char mein_cockpit_cert[] = {
   0x7f, 0x21, 0x82, 0x01, 0x48, 0x7f, 0x4e, 0x82, 0x01, 0x00, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x10,
   0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30, 0x31,
   0x7f, 0x49, 0x4f, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86,
   0x41, 0x04, 0x30, 0xc6, 0x55, 0x64, 0x02, 0x9f, 0xcb, 0xa9, 0x93, 0x4a, 0x46, 0x7c, 0x5d, 0x53,
   0xb4, 0x0f, 0x87, 0xc3, 0x0f, 0x1f, 0x04, 0xb5, 0x0f, 0x2b, 0xa7, 0x6a, 0xf0, 0xc7, 0x4b, 0x8f,
   0x2a, 0xc0, 0x57, 0xcb, 0x17, 0x6d, 0x2a, 0x1d, 0x15, 0x0b, 0x53, 0x1f, 0xe5, 0x9f, 0x4f, 0x31,
   0x5d, 0x22, 0xd2, 0xcf, 0x45, 0x63, 0x60, 0x2f, 0x9c, 0xe0, 0x85, 0xda, 0x62, 0x24, 0x0f, 0xd2,
   0x9d, 0x7a, 0x5f, 0x20, 0x0e, 0x44, 0x45, 0x30, 0x30, 0x30, 0x30, 0x30, 0x34, 0x39, 0x30, 0x30,
   0x34, 0x46, 0x4e, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02,
   0x02, 0x53, 0x05, 0x00, 0x00, 0x00, 0x00, 0x04, 0x5f, 0x25, 0x06, 0x01, 0x01, 0x01, 0x02, 0x00,
   0x09, 0x5f, 0x24, 0x06, 0x01, 0x01, 0x01, 0x02, 0x01, 0x00, 0x65, 0x5e, 0x73, 0x2d, 0x06, 0x09,
   0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x02, 0x80, 0x20, 0x2b, 0xa8, 0x57, 0xae, 0xc0,
   0x87, 0xcc, 0xcd, 0x2d, 0xd5, 0x26, 0xfd, 0xd8, 0x2e, 0xe3, 0x3e, 0xa2, 0x89, 0x40, 0x8b, 0x3c,
   0x21, 0x61, 0xf3, 0x66, 0x80, 0xa9, 0xc3, 0x84, 0xc1, 0x43, 0xfb, 0x73, 0x2d, 0x06, 0x09, 0x04,
   0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x80, 0x20, 0xbe, 0x46, 0x23, 0xde, 0x47, 0xdc,
   0x5f, 0x42, 0x24, 0x02, 0xf8, 0x51, 0xb2, 0x5c, 0x88, 0xd8, 0xf4, 0x66, 0xb1, 0xf4, 0x5c, 0xc9,
   0x7e, 0x11, 0xe8, 0x09, 0xaf, 0xf4, 0xa3, 0xe4, 0x5f, 0xe3, 0x5f, 0x37, 0x40, 0x49, 0xff, 0x9b,
   0xf0, 0xf3, 0x0a, 0x9e, 0x61, 0x16, 0x76, 0xa6, 0xc7, 0x95, 0xea, 0x55, 0x23, 0x4b, 0x2a, 0xe4,
   0x3f, 0x69, 0x0f, 0x8a, 0x77, 0xa7, 0x6a, 0x23, 0x8a, 0x9c, 0xa3, 0x98, 0x28, 0x31, 0x3c, 0x0a,
   0x30, 0x18, 0x39, 0xeb, 0x39, 0x1e, 0xb3, 0x1a, 0xd5, 0x78, 0x1a, 0xcd, 0xf1, 0xdf, 0x83, 0xc1,
   0x69, 0xb1, 0x3a, 0xf9, 0xa6, 0x1f, 0x58, 0x72, 0xf5, 0xff, 0x70, 0xa7, 0x8c
};

/* https://www.service-bw.de/idm-web-portal/page/protected/index/index.faces?action=init&eid=true */
static unsigned char service_bw_dvcert[] = {
    0x7f, 0x21, 0x81, 0xe7, 0x7f, 0x4e, 0x81, 0xa0, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x0e, 0x44, 0x45, /* .!...N.._)..B.DE */
    0x43, 0x56, 0x43, 0x41, 0x65, 0x49, 0x44, 0x30, 0x30, 0x31, 0x30, 0x32, 0x7f, 0x49, 0x4f, 0x06, /* CVCAeID00102.IO. */
    0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86, 0x41, 0x04, 0x46, 0x5f, /* ............A.F_ */
    0x4b, 0x87, 0x11, 0xde, 0x56, 0xa2, 0x4a, 0x03, 0x1a, 0xec, 0xc3, 0x74, 0x8c, 0x0a, 0x13, 0xa6, /* K...V.J....t.... */
    0x3f, 0x56, 0x67, 0x5c, 0x8d, 0xfc, 0x76, 0x8e, 0x1f, 0xfe, 0x69, 0xf9, 0x92, 0x19, 0x2d, 0xab, /* ?Vg\..v...i...-. */
    0x1a, 0x41, 0xec, 0x37, 0xc3, 0xae, 0x23, 0xe1, 0x53, 0x90, 0x65, 0x44, 0xab, 0x9b, 0x84, 0xb4, /* .A.7..#.S.eD.... */
    0x50, 0xc0, 0xbf, 0x2e, 0xe4, 0x62, 0x57, 0x6a, 0x33, 0xf4, 0xa5, 0x0a, 0x17, 0x07, 0x5f, 0x20, /* P....bWj3....._  */
    0x10, 0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30, /* .DEDVeIDDTR10140 */
    0x31, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, 0x02, 0x53, /* 1.L............S */
    0x05, 0x40, 0x05, 0x01, 0xff, 0x87, 0x5f, 0x25, 0x06, 0x01, 0x01, 0x01, 0x01, 0x01, 0x07, 0x5f, /* .@...._%......._ */
    0x24, 0x06, 0x01, 0x02, 0x00, 0x02, 0x01, 0x07, 0x5f, 0x37, 0x40, 0x1b, 0x93, 0x4a, 0x7e, 0xc4, /* $......._7@..J~. */
    0xd2, 0x01, 0xe8, 0xae, 0xb0, 0xe3, 0xbe, 0xda, 0x06, 0x60, 0x6b, 0x7b, 0x47, 0x10, 0xaf, 0xd3, /* .........`k{G... */
    0x9a, 0x13, 0x14, 0x7d, 0x63, 0x2c, 0xf6, 0x72, 0xd6, 0x4b, 0x2f, 0x94, 0xd9, 0x93, 0x55, 0xad, /* ...}c,.r.K/...U. */
    0x4a, 0x3b, 0x16, 0x07, 0x04, 0x86, 0x14, 0x9e, 0xd3, 0x53, 0xe0, 0xdc, 0x9d, 0x8c, 0xa7, 0x6b, /* J;.......S.....k */
    0x28, 0xf7, 0x94, 0x33, 0x07, 0x3b, 0x43, 0x40, 0xb3, 0x8e, 0x97                                /* (..3.;C@... */
};
static unsigned char service_bw_cert[] = {
    0x7f, 0x21, 0x82, 0x01, 0x48, 0x7f, 0x4e, 0x82, 0x01, 0x00, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x10, /* .!..H.N..._)..B. */
    0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30, 0x31, /* DEDVeIDDTR101401 */
    0x7f, 0x49, 0x4f, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86, /* .IO............. */
    0x41, 0x04, 0x4f, 0xba, 0x55, 0x36, 0xf0, 0x0f, 0xbd, 0xce, 0x28, 0xcb, 0xee, 0xd6, 0x0b, 0x88, /* A.O.U6....(..... */
    0xc9, 0x24, 0xbb, 0xcd, 0x58, 0x04, 0xb6, 0x08, 0x29, 0x73, 0x58, 0xea, 0x95, 0x73, 0x7c, 0x0b, /* .$..X...)sX..s|. */
    0xb8, 0xb3, 0x9c, 0x45, 0x81, 0x42, 0xaf, 0x86, 0xb5, 0xea, 0x02, 0xb8, 0x33, 0x20, 0xc5, 0x0f, /* ...E.B......3 .. */
    0xed, 0x94, 0x72, 0x79, 0xdc, 0x3d, 0x1e, 0xe1, 0x8e, 0x10, 0x89, 0x07, 0x2c, 0x5c, 0xcc, 0x5a, /* ..ry.=......,\.Z */
    0xbc, 0x16, 0x5f, 0x20, 0x0e, 0x44, 0x45, 0x30, 0x30, 0x30, 0x30, 0x30, 0x35, 0x35, 0x30, 0x30, /* .._ .DE000005500 */
    0x30, 0x56, 0x31, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, /* 0V1.L........... */
    0x02, 0x53, 0x05, 0x00, 0x01, 0x00, 0xdb, 0x04, 0x5f, 0x25, 0x06, 0x01, 0x02, 0x00, 0x01, 0x02, /* .S......_%...... */
    0x03, 0x5f, 0x24, 0x06, 0x01, 0x02, 0x00, 0x01, 0x02, 0x04, 0x65, 0x5e, 0x73, 0x2d, 0x06, 0x09, /* ._$.......e^s-.. */
    0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x02, 0x80, 0x20, 0x40, 0x56, 0xea, 0x70, 0x4b, /* .......... @V.pK */
    0xa2, 0x91, 0xe2, 0x6b, 0x21, 0xfc, 0x1e, 0x4e, 0x37, 0xd9, 0x37, 0x5c, 0xb9, 0x3b, 0xbc, 0x91, /* ...k!..N7.7\.;.. */
    0xea, 0x41, 0x91, 0x2f, 0x72, 0x6c, 0xcc, 0x3a, 0xff, 0x6f, 0x64, 0x73, 0x2d, 0x06, 0x09, 0x04, /* .A./rl.:.ods-... */
    0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x80, 0x20, 0x4c, 0xb0, 0x13, 0x22, 0x73, 0x7a, /* ......... L.."sz */
    0xc5, 0x6c, 0x6c, 0xac, 0x4d, 0xef, 0xce, 0x69, 0x07, 0x19, 0x0e, 0xae, 0x58, 0xb4, 0xeb, 0x1b, /* .ll.M..i....X... */
    0x7e, 0xd3, 0x47, 0x53, 0x81, 0x27, 0x75, 0xb5, 0x7a, 0xbb, 0x5f, 0x37, 0x40, 0x25, 0x61, 0x7f, /* ~.GS.'u.z._7@%a. */
    0xd6, 0xa2, 0x92, 0xbc, 0x1e, 0x04, 0x7b, 0x6c, 0x46, 0x42, 0x0c, 0x8b, 0xfd, 0x21, 0x28, 0xbc, /* ......{lFB...!(. */
    0x5c, 0xb0, 0x78, 0x00, 0x20, 0xe3, 0x0e, 0x24, 0x16, 0xc8, 0xf3, 0xd4, 0x81, 0x0b, 0x77, 0x29, /* \.x. ..$......w) */
    0x5e, 0x90, 0xfa, 0x59, 0x30, 0xa5, 0x31, 0x69, 0x8f, 0xef, 0x21, 0x9c, 0x01, 0xb8, 0xa5, 0xc6, /* ^..Y0.1i..!..... */
    0xe6, 0xe5, 0x40, 0x1d, 0x05, 0x33, 0x55, 0xd5, 0x5a, 0x74, 0xac, 0x47, 0xf5                    /* ..@..3U.Zt.G. */
};
static unsigned char service_bw_cert_2[] = {
    0x7f, 0x21, 0x82, 0x01, 0x48, 0x7f, 0x4e, 0x82, 0x01, 0x00, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x10, /* .!..H.N..._)..B. */
    0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30, 0x31, /* DEDVeIDDTR101401 */
    0x7f, 0x49, 0x4f, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86, /* .IO............. */
    0x41, 0x04, 0x21, 0xdf, 0x03, 0x60, 0x1f, 0x93, 0xa2, 0x8d, 0x57, 0xbd, 0xcb, 0x7f, 0xff, 0xd5, /* A.!..`....W..... */
    0x74, 0x80, 0x21, 0x28, 0x23, 0x29, 0x33, 0xb5, 0xdf, 0x4e, 0xfe, 0x55, 0x83, 0x57, 0x72, 0xae, /* t.!(#)3..N.U.Wr. */
    0xbb, 0xf9, 0x10, 0xe0, 0x73, 0x7c, 0xce, 0x19, 0x98, 0x5b, 0xd3, 0xf1, 0x93, 0x6c, 0x7a, 0x6c, /* ....s|...[...lzl */
    0xf9, 0x92, 0xe5, 0xe6, 0x47, 0x4d, 0xd3, 0xcf, 0x07, 0xee, 0x7e, 0x95, 0x8b, 0x2a, 0xd4, 0xc8, /* ....GM....~..*.. */
    0x1d, 0xf2, 0x5f, 0x20, 0x0e, 0x44, 0x45, 0x30, 0x30, 0x30, 0x30, 0x30, 0x35, 0x35, 0x30, 0x30, /* .._ .DE000005500 */
    0x30, 0x58, 0x49, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, /* 0XI.L........... */
    0x02, 0x53, 0x05, 0x00, 0x01, 0x00, 0xdb, 0x04, 0x5f, 0x25, 0x06, 0x01, 0x02, 0x00, 0x02, 0x00, /* .S......_%...... */
    0x03, 0x5f, 0x24, 0x06, 0x01, 0x02, 0x00, 0x02, 0x00, 0x04, 0x65, 0x5e, 0x73, 0x2d, 0x06, 0x09, /* ._$.......e^s-.. */
    0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x02, 0x80, 0x20, 0x40, 0x56, 0xea, 0x70, 0x4b, /* .......... @V.pK */
    0xa2, 0x91, 0xe2, 0x6b, 0x21, 0xfc, 0x1e, 0x4e, 0x37, 0xd9, 0x37, 0x5c, 0xb9, 0x3b, 0xbc, 0x91, /* ...k!..N7.7\.;.. */
    0xea, 0x41, 0x91, 0x2f, 0x72, 0x6c, 0xcc, 0x3a, 0xff, 0x6f, 0x64, 0x73, 0x2d, 0x06, 0x09, 0x04, /* .A./rl.:.ods-... */
    0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x80, 0x20, 0x4c, 0xb0, 0x13, 0x22, 0x73, 0x7a, /* ......... L.."sz */
    0xc5, 0x6c, 0x6c, 0xac, 0x4d, 0xef, 0xce, 0x69, 0x07, 0x19, 0x0e, 0xae, 0x58, 0xb4, 0xeb, 0x1b, /* .ll.M..i....X... */
    0x7e, 0xd3, 0x47, 0x53, 0x81, 0x27, 0x75, 0xb5, 0x7a, 0xbb, 0x5f, 0x37, 0x40, 0x2b, 0x54, 0xc6, /* ~.GS.'u.z._7@+T. */
    0x2f, 0x58, 0x25, 0xf9, 0x5a, 0xb3, 0x89, 0x05, 0x67, 0xfc, 0xa0, 0x5b, 0x09, 0x2e, 0x19, 0x31, /* /X%.Z...g..[...1 */
    0x8d, 0xb1, 0xcc, 0xe9, 0xbd, 0x2b, 0x2f, 0x4d, 0x27, 0x04, 0xaa, 0x69, 0x62, 0x4d, 0x15, 0x58, /* .....+/M'..ibM.X */
    0xd1, 0xb5, 0x75, 0xe7, 0x84, 0x06, 0xad, 0x96, 0x81, 0x5d, 0x74, 0xa6, 0x1a, 0xef, 0x5e, 0x37, /* ..u......]t...^7 */
    0xdf, 0x7a, 0xe0, 0x18, 0x47, 0x94, 0x78, 0x97, 0x23, 0xcc, 0x7d, 0xb2, 0x02                   /* .z..G.x.#.}.. */
};
static unsigned char service_bw_cert_3[] = {
    0x7f, 0x21, 0x82, 0x01, 0x48, 0x7f, 0x4e, 0x82, 0x01, 0x00, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x10, /* .!..H.N..._)..B. */
    0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30, 0x31, /* DEDVeIDDTR101401 */
    0x7f, 0x49, 0x4f, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86, /* .IO............. */
    0x41, 0x04, 0x3f, 0x5f, 0xd5, 0x01, 0x17, 0x2e, 0xb5, 0xf3, 0xbc, 0x78, 0xb3, 0x49, 0xb4, 0xb4, /* A.?_.......x.I.. */
    0xd0, 0xda, 0xe4, 0x2d, 0x3e, 0xc9, 0xa6, 0x77, 0x40, 0x3e, 0xb7, 0xcb, 0x88, 0xcd, 0x86, 0x4d, /* ...->..w@>.....M */
    0x5e, 0x2a, 0x35, 0x79, 0x6c, 0x98, 0x30, 0x9f, 0x7d, 0x29, 0x99, 0xc6, 0xae, 0x9a, 0xb7, 0x94, /* ^*5yl.0.})...... */
    0x72, 0x18, 0x56, 0x1c, 0x8b, 0x22, 0x91, 0x73, 0x98, 0x45, 0x94, 0x21, 0xfc, 0xd8, 0x34, 0x6c, /* r.V..".s.E.!..4l */
    0xe4, 0x04, 0x5f, 0x20, 0x0e, 0x44, 0x45, 0x30, 0x30, 0x30, 0x30, 0x30, 0x35, 0x35, 0x30, 0x30, /* .._ .DE000005500 */
    0x30, 0x59, 0x46, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, /* 0YF.L........... */
    0x02, 0x53, 0x05, 0x00, 0x01, 0x00, 0xdb, 0x04, 0x5f, 0x25, 0x06, 0x01, 0x02, 0x00, 0x02, 0x00, /* .S......_%...... */
    0x07, 0x5f, 0x24, 0x06, 0x01, 0x02, 0x00, 0x02, 0x00, 0x08, 0x65, 0x5e, 0x73, 0x2d, 0x06, 0x09, /* ._$.......e^s-.. */
    0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x02, 0x80, 0x20, 0x40, 0x56, 0xea, 0x70, 0x4b, /* .......... @V.pK */
    0xa2, 0x91, 0xe2, 0x6b, 0x21, 0xfc, 0x1e, 0x4e, 0x37, 0xd9, 0x37, 0x5c, 0xb9, 0x3b, 0xbc, 0x91, /* ...k!..N7.7\.;.. */
    0xea, 0x41, 0x91, 0x2f, 0x72, 0x6c, 0xcc, 0x3a, 0xff, 0x6f, 0x64, 0x73, 0x2d, 0x06, 0x09, 0x04, /* .A./rl.:.ods-... */
    0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x80, 0x20, 0x4c, 0xb0, 0x13, 0x22, 0x73, 0x7a, /* ......... L.."sz */
    0xc5, 0x6c, 0x6c, 0xac, 0x4d, 0xef, 0xce, 0x69, 0x07, 0x19, 0x0e, 0xae, 0x58, 0xb4, 0xeb, 0x1b, /* .ll.M..i....X... */
    0x7e, 0xd3, 0x47, 0x53, 0x81, 0x27, 0x75, 0xb5, 0x7a, 0xbb, 0x5f, 0x37, 0x40, 0x51, 0x50, 0x1b, /* ~.GS.'u.z._7@QP. */
    0x46, 0x1d, 0xd0, 0xe9, 0x80, 0x09, 0x25, 0xc8, 0xb7, 0x49, 0xd1, 0x35, 0x85, 0x47, 0x7d, 0x7b, /* F.....%..I.5.G}{ */
    0xc8, 0xd3, 0x40, 0x40, 0xb2, 0x34, 0x64, 0x52, 0xf4, 0x57, 0xd5, 0x24, 0x23, 0x5a, 0xaf, 0xfe, /* ..@@.4dR.W.$#Z.. */
    0x5e, 0x9d, 0x6e, 0xf4, 0x4b, 0x03, 0xd1, 0x8f, 0x73, 0x90, 0xaf, 0x14, 0xcf, 0x1b, 0x85, 0xb7, /* ^.n.K...s....... */
    0x73, 0xb1, 0x65, 0x50, 0x30, 0xf8, 0xf7, 0x5a, 0x05, 0x9a, 0x0b, 0x53, 0x75,                   /* s.eP0..Z...Su */
};
static unsigned char service_bw_cert_4[] = {
   0x7f, 0x21, 0x82, 0x01, 0x48, 0x7f, 0x4e, 0x82, 0x01, 0x00, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x10, /* .!..H.N..._)..B. */
   0x44, 0x45, 0x44, 0x56, 0x65, 0x49, 0x44, 0x44, 0x54, 0x52, 0x31, 0x30, 0x31, 0x34, 0x30, 0x31, /* DEDVeIDDTR101401 */
   0x7f, 0x49, 0x4f, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x86, /* .IO............. */
   0x41, 0x04, 0x31, 0xeb, 0xeb, 0xd5, 0x20, 0x2a, 0x28, 0x74, 0x57, 0xa9, 0x55, 0x5a, 0xd5, 0x47, /* A.1... *(tW.UZ.G */
   0x10, 0xfb, 0xd7, 0x97, 0x59, 0xf6, 0xa7, 0x77, 0x97, 0x7c, 0x68, 0xd9, 0x74, 0xa6, 0x45, 0x3d, /* ....Y..w.|h.t.E= */
   0xad, 0xcc, 0x9e, 0xb1, 0x20, 0x93, 0x82, 0xc0, 0xe9, 0x38, 0x1d, 0xce, 0xf1, 0xb2, 0x0a, 0x6b, /* .... ....8.....k */
   0xd6, 0xdd, 0x5e, 0x56, 0x06, 0x83, 0x94, 0xa2, 0xf3, 0x45, 0x3a, 0x20, 0x64, 0x40, 0x04, 0xbd, /* ..^V.....E: d@.. */
   0x2f, 0x91, 0x5f, 0x20, 0x0e, 0x44, 0x45, 0x30, 0x30, 0x30, 0x30, 0x30, 0x35, 0x35, 0x30, 0x30, /* /._ .DE000005500 */
   0x30, 0x59, 0x56, 0x7f, 0x4c, 0x12, 0x06, 0x09, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x02, /* 0YV.L........... */
   0x02, 0x53, 0x05, 0x00, 0x01, 0x00, 0xdb, 0x04, 0x5f, 0x25, 0x06, 0x01, 0x02, 0x00, 0x02, 0x00, /* .S......_%...... */
   0x09, 0x5f, 0x24, 0x06, 0x01, 0x02, 0x00, 0x02, 0x01, 0x00, 0x65, 0x5e, 0x73, 0x2d, 0x06, 0x09, /* ._$.......e^s-.. */
   0x04, 0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x02, 0x80, 0x20, 0x40, 0x56, 0xea, 0x70, 0x4b, /* .......... @V.pK */
   0xa2, 0x91, 0xe2, 0x6b, 0x21, 0xfc, 0x1e, 0x4e, 0x37, 0xd9, 0x37, 0x5c, 0xb9, 0x3b, 0xbc, 0x91, /* ...k!..N7.7\.;.. */
   0xea, 0x41, 0x91, 0x2f, 0x72, 0x6c, 0xcc, 0x3a, 0xff, 0x6f, 0x64, 0x73, 0x2d, 0x06, 0x09, 0x04, /* .A./rl.:.ods-... */
   0x00, 0x7f, 0x00, 0x07, 0x03, 0x01, 0x03, 0x01, 0x80, 0x20, 0x4c, 0xb0, 0x13, 0x22, 0x73, 0x7a, /* ......... L.."sz */
   0xc5, 0x6c, 0x6c, 0xac, 0x4d, 0xef, 0xce, 0x69, 0x07, 0x19, 0x0e, 0xae, 0x58, 0xb4, 0xeb, 0x1b, /* .ll.M..i....X... */
   0x7e, 0xd3, 0x47, 0x53, 0x81, 0x27, 0x75, 0xb5, 0x7a, 0xbb, 0x5f, 0x37, 0x40, 0x5d, 0xce, 0xda, /* ~.GS.'u.z._7@].. */
   0xe9, 0xbf, 0x00, 0xb2, 0x70, 0x6c, 0x3f, 0x81, 0xd1, 0x20, 0x28, 0x0f, 0x1f, 0xbf, 0xdf, 0xe8, /* ....pl?.. (..... */
   0x08, 0x6e, 0x7d, 0xd3, 0xe6, 0xb3, 0xac, 0x45, 0xd2, 0x5b, 0xd1, 0xad, 0xde, 0x54, 0x17, 0xb7, /* .n}....E.[...T.. */
   0xe9, 0x44, 0x06, 0xfa, 0x47, 0x21, 0x99, 0xf5, 0x41, 0x36, 0x8c, 0xeb, 0xcd, 0xce, 0xcf, 0x9d, /* .D..G!..A6...... */
   0xc2, 0x50, 0x58, 0x61, 0x8c, 0xdf, 0x72, 0xe0, 0xd3, 0x00, 0xf3, 0xba, 0x76,                   /* .PXa..r.....v */
};

/* https://github.com/bcgit/bc-java/blob/master/pkix/src/test/resources/org/bouncycastle/eac/test/REQ_18102010.csr */
static unsigned char req_18102010_csr[] = {
   0x7f, 0x21, 0x82, 0x01, 0x55, 0x7f, 0x4e, 0x82, 0x01, 0x15, 0x5f, 0x29, 0x01, 0x00, 0x7f, 0x49, /* .!..U.N..._)...I */
   0x81, 0xff, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x03, 0x81, 0x1d, /* ................ */
   0x00, 0xd7, 0xc1, 0x34, 0xaa, 0x26, 0x43, 0x66, 0x86, 0x2a, 0x18, 0x30, 0x25, 0x75, 0xd1, 0xd7, /* ...4.&Cf.*.0%u.. */
   0x87, 0xb0, 0x9f, 0x07, 0x57, 0x97, 0xda, 0x89, 0xf5, 0x7e, 0xc8, 0xc0, 0xff, 0x82, 0x1c, 0x68, /* ....W....~.....h */
   0xa5, 0xe6, 0x2c, 0xa9, 0xce, 0x6c, 0x1c, 0x29, 0x98, 0x03, 0xa6, 0xc1, 0x53, 0x0b, 0x51, 0x4e, /* ..,..l.)....S.QN */
   0x18, 0x2a, 0xd8, 0xb0, 0x04, 0x2a, 0x59, 0xca, 0xd2, 0x9f, 0x43, 0x83, 0x1c, 0x25, 0x80, 0xf6, /* .*...*Y...C..%.. */
   0x3c, 0xcf, 0xe4, 0x41, 0x38, 0x87, 0x07, 0x13, 0xb1, 0xa9, 0x23, 0x69, 0xe3, 0x3e, 0x21, 0x35, /* <..A8.....#i.>!5 */
   0xd2, 0x66, 0xdb, 0xb3, 0x72, 0x38, 0x6c, 0x40, 0x0b, 0x84, 0x39, 0x04, 0x0d, 0x90, 0x29, 0xad, /* .f..r8l@..9...). */
   0x2c, 0x7e, 0x5c, 0xf4, 0x34, 0x08, 0x23, 0xb2, 0xa8, 0x7d, 0xc6, 0x8c, 0x9e, 0x4c, 0xe3, 0x17, /* ,~\.4.#..}...L.. */
   0x4c, 0x1e, 0x6e, 0xfd, 0xee, 0x12, 0xc0, 0x7d, 0x58, 0xaa, 0x56, 0xf7, 0x72, 0xc0, 0x72, 0x6f, /* L.n....}X.V.r.ro */
   0x24, 0xc6, 0xb8, 0x9e, 0x4e, 0xcd, 0xac, 0x24, 0x35, 0x4b, 0x9e, 0x99, 0xca, 0xa3, 0xf6, 0xd3, /* $...N..$5K...... */
   0x76, 0x14, 0x02, 0xcd, 0x85, 0x1d, 0x00, 0xd7, 0xc1, 0x34, 0xaa, 0x26, 0x43, 0x66, 0x86, 0x2a, /* v........4.&Cf.* */
   0x18, 0x30, 0x25, 0x75, 0xd0, 0xfb, 0x98, 0xd1, 0x16, 0xbc, 0x4b, 0x6d, 0xde, 0xbc, 0xa3, 0xa5, /* .0%u......Km.... */
   0xa7, 0x93, 0x9f, 0x86, 0x39, 0x04, 0xc0, 0xa6, 0x24, 0x01, 0xc2, 0xb7, 0xba, 0xf2, 0xde, 0x3d, /* ....9...$......= */
   0x5e, 0x2a, 0x58, 0x9e, 0xab, 0x71, 0x2f, 0x9e, 0x4f, 0x65, 0xde, 0x09, 0xe4, 0x2e, 0x9a, 0x41, /* ^*X..q/.Oe.....A */
   0x47, 0xf2, 0x62, 0xdf, 0x00, 0xf6, 0xd3, 0x95, 0x7c, 0x76, 0x73, 0xc1, 0x53, 0x76, 0xf6, 0x48, /* G.b.....|vs.Sv.H */
   0xaf, 0xf0, 0x42, 0x22, 0x42, 0x85, 0x0c, 0x45, 0x2e, 0x93, 0x25, 0x10, 0x50, 0x2d, 0x87, 0x01, /* ..B"B..E..%.P-.. */
   0x01, 0x5f, 0x20, 0x0b, 0x42, 0x45, 0x54, 0x45, 0x53, 0x54, 0x30, 0x30, 0x30, 0x30, 0x31, 0x5f, /* ._ .BETEST00001_ */
   0x37, 0x38, 0x4a, 0x16, 0xd0, 0x8b, 0xb3, 0x11, 0x7e, 0x2e, 0xc0, 0xb2, 0x0b, 0xc2, 0x72, 0xeb, /* 78J.....~.....r. */
   0x2d, 0x66, 0x77, 0x57, 0xbc, 0x63, 0x57, 0xc7, 0x92, 0x33, 0x5e, 0x5b, 0xb1, 0xc4, 0x33, 0x35, /* -fwW.cW..3^[..35 */
   0xb6, 0xe2, 0xbd, 0xd4, 0x76, 0xfb, 0xce, 0x72, 0xd6, 0x20, 0x8c, 0xec, 0x88, 0x44, 0x4a, 0xcb, /* ....v..r. ...DJ. */
   0xee, 0x98, 0x1b, 0xbd, 0x86, 0x92, 0x8e, 0x28, 0x16, 0xca,                                     /* .......(.. */
};

/* https://www.ejbca.org/older_releases/ejbca_3_9/htdocs/cvccas.html#Creating%20CVCAs%20and%20DVs */
static unsigned char MY_cvreq[] = {
   0x7f, 0x21, 0x82, 0x02, 0x47, 0x7f, 0x4e, 0x82, 0x01, 0x3d, 0x5f, 0x29, 0x01, 0x00, 0x42, 0x0d, /* .!..G.N..=_)..B. */
   0x53, 0x45, 0x43, 0x56, 0x43, 0x41, 0x50, 0x4b, 0x30, 0x30, 0x30, 0x30, 0x31, 0x7f, 0x49, 0x82, /* SECVCAPK00001.I. */
   0x01, 0x15, 0x06, 0x0a, 0x04, 0x00, 0x7f, 0x00, 0x07, 0x02, 0x02, 0x02, 0x01, 0x02, 0x81, 0x82, /* ................ */
   0x01, 0x00, 0xc9, 0xb8, 0x12, 0xda, 0xd6, 0x21, 0x57, 0x0f, 0xb5, 0x96, 0x2b, 0xae, 0xb5, 0xe1, /* .......!W...+... */
   0x1b, 0xed, 0x9a, 0x26, 0x69, 0x80, 0x55, 0xe2, 0x4d, 0x69, 0x33, 0x9e, 0x9e, 0x4c, 0x7e, 0xeb, /* ...&i.U.Mi3..L~. */
   0x09, 0xb8, 0xda, 0x81, 0xbb, 0x3e, 0x3a, 0xcd, 0xe1, 0x40, 0x20, 0x97, 0x1e, 0xac, 0x4b, 0x4a, /* .....>:..@ ...KJ */
   0xc5, 0xc1, 0xed, 0x6c, 0xea, 0x69, 0x03, 0x95, 0x36, 0x17, 0x03, 0xad, 0xaa, 0x2b, 0x74, 0xca, /* ...l.i..6....+t. */
   0x19, 0x20, 0xd9, 0x72, 0xd3, 0x11, 0xff, 0x73, 0x87, 0x97, 0xb0, 0x72, 0x14, 0x86, 0x5a, 0xe8, /* . .r...s...r..Z. */
   0xc6, 0xf7, 0x26, 0x3b, 0xec, 0x13, 0xd7, 0x7c, 0xb8, 0x37, 0x6b, 0x78, 0xb0, 0x57, 0x20, 0x56, /* ..&;...|.7kx.W V */
   0xb0, 0x40, 0xf8, 0xcf, 0x82, 0xfb, 0xd6, 0x17, 0xef, 0x07, 0x24, 0xed, 0x90, 0x1a, 0x73, 0xf7, /* .@........$...s. */
   0x40, 0x45, 0x8f, 0x03, 0x14, 0xb2, 0x24, 0xb3, 0x92, 0xe4, 0x70, 0x33, 0xfd, 0x8f, 0x79, 0x38, /* @E....$...p3..y8 */
   0x49, 0x36, 0x6e, 0x72, 0x0d, 0x95, 0x63, 0x2c, 0xd1, 0x5f, 0x37, 0xb5, 0x5d, 0x89, 0xb8, 0xba, /* I6nr..c,._7.]... */
   0xec, 0xf4, 0x61, 0xef, 0xba, 0x10, 0x92, 0x87, 0xe4, 0xfc, 0x5b, 0xcc, 0x35, 0x56, 0xba, 0xd2, /* ..a.......[.5V.. */
   0xee, 0x91, 0xb1, 0x3d, 0x6e, 0xac, 0x97, 0x4c, 0xef, 0xee, 0x66, 0x0a, 0xf1, 0x8e, 0xb3, 0x42, /* ...=n..L..f....B */
   0xa8, 0x5b, 0x31, 0x4c, 0x09, 0xd9, 0xb7, 0xad, 0x9a, 0xc8, 0x22, 0xf5, 0xa6, 0x40, 0x14, 0xb1, /* .[1L......"..@.. */
   0x30, 0xf4, 0x4b, 0xe3, 0x60, 0x15, 0x0e, 0x06, 0x3f, 0x1b, 0xb7, 0x5f, 0xdb, 0xb6, 0x0c, 0x1c, /* 0.K.`...?.._.... */
   0x96, 0x7a, 0xa6, 0xc9, 0xc4, 0x60, 0x2e, 0xa7, 0xdf, 0x4c, 0xb0, 0xf7, 0x50, 0x45, 0x9f, 0x38, /* .z...`...L..PE.8 */
   0x6a, 0x98, 0x37, 0xa6, 0xe3, 0x1b, 0xa0, 0x46, 0x7a, 0x73, 0xab, 0x5c, 0xf4, 0x46, 0xa1, 0x4c, /* j.7....Fzs.\.F.L */
   0x61, 0xa6, 0xd6, 0x7b, 0xc3, 0x6c, 0x20, 0x19, 0x3f, 0x93, 0xa6, 0xc5, 0x5d, 0xc8, 0x5e, 0x0a, /* a..{.l .?...].^. */
   0x2a, 0x83, 0x82, 0x03, 0x01, 0x00, 0x01, 0x5f, 0x20, 0x0d, 0x43, 0x5a, 0x44, 0x56, 0x54, 0x45, /* *......_ .CZDVTE */
   0x53, 0x54, 0x53, 0x45, 0x30, 0x30, 0x30, 0x5f, 0x37, 0x82, 0x01, 0x00, 0x6a, 0x8b, 0x5a, 0x38, /* STSE000_7...j.Z8 */
   0x1d, 0xec, 0x41, 0xdc, 0xc7, 0xf1, 0xe5, 0xc9, 0xe3, 0x09, 0xa4, 0x70, 0xf2, 0xa3, 0xec, 0x0b, /* ..A........p.... */
   0x7c, 0xf1, 0xc3, 0xe6, 0xd1, 0x5c, 0x1d, 0xed, 0xd5, 0x8d, 0x69, 0xcc, 0x0f, 0xc4, 0x97, 0x2d, /* |....\....i....- */
   0x07, 0x2f, 0xcf, 0xf2, 0xfd, 0x26, 0x29, 0x5c, 0xa2, 0x2c, 0xe0, 0x3b, 0x59, 0xa4, 0x20, 0xc8, /* ./...&)\.,.;Y. . */
   0x47, 0x8c, 0xca, 0x0d, 0x22, 0xb7, 0x8e, 0x4b, 0xfd, 0xd7, 0x8a, 0x15, 0x32, 0x93, 0x37, 0x0e, /* G..."..K....2.7. */
   0xa9, 0x45, 0x74, 0xba, 0x98, 0xe1, 0xa0, 0x07, 0x57, 0x81, 0x4d, 0xe7, 0xee, 0x81, 0x2a, 0xf6, /* .Et.....W.M...*. */
   0xc6, 0x61, 0x91, 0x1f, 0x8a, 0x9b, 0x35, 0x6d, 0xaf, 0x2b, 0x3d, 0x91, 0x13, 0x2f, 0x37, 0x22, /* .a....5m.+=../7" */
   0xbc, 0xa9, 0x63, 0x22, 0xb5, 0xdb, 0x7f, 0x47, 0xd5, 0x48, 0xd8, 0x61, 0x9c, 0x20, 0x1f, 0x33, /* ..c"...G.H.a. .3 */
   0xd0, 0xf1, 0x2a, 0xb9, 0x71, 0x64, 0xfd, 0x74, 0x98, 0xb6, 0xf4, 0x71, 0x6b, 0x39, 0x25, 0x21, /* ..*.qd.t...qk9%! */
   0xa3, 0x03, 0x68, 0xb4, 0x0e, 0x51, 0xf5, 0xf3, 0x32, 0x8a, 0xb5, 0xe0, 0x0e, 0x77, 0xc8, 0x32, /* ..h..Q..2....w.2 */
   0x86, 0xd8, 0x12, 0x8e, 0x5c, 0xf6, 0x16, 0x78, 0xac, 0x1e, 0x72, 0x14, 0x94, 0x74, 0xb5, 0x76, /* ....\..x..r..t.v */
   0x52, 0xc8, 0x14, 0x00, 0x43, 0x7d, 0xbb, 0xcd, 0x4b, 0xdf, 0x96, 0x4f, 0x33, 0x4d, 0xf0, 0x99, /* R...C}..K..O3M.. */
   0x45, 0x8b, 0x7e, 0xd0, 0xa5, 0x20, 0xbe, 0x0b, 0x95, 0xed, 0xf1, 0x8e, 0x0d, 0x9c, 0x68, 0x97, /* E.~.. ........h. */
   0x74, 0xc0, 0xda, 0x47, 0x59, 0x75, 0xe5, 0x59, 0xfa, 0x98, 0xdb, 0x30, 0xf6, 0x56, 0x1a, 0x01, /* t..GYu.Y...0.V.. */
   0xc2, 0x23, 0xfc, 0x43, 0xa5, 0x65, 0xc1, 0xa8, 0x72, 0x75, 0x57, 0x02, 0x58, 0x72, 0xdc, 0xc9, /* .#.C.e..ruW.Xr.. */
   0x16, 0x93, 0x4b, 0x9e, 0xf3, 0x95, 0x5b, 0xbf, 0x08, 0x0f, 0x01, 0xf0, 0xcf, 0xd4, 0x01, 0x05, /* ..K...[......... */
   0xf0, 0x97, 0x2a, 0x43, 0xac, 0x4d, 0xed, 0x42, 0xcc, 0x87, 0xcc, 0x56,                         /* ..*C.M.B...V */
};


struct cv_cert {
    const unsigned char *cv_cert;
    const size_t cv_cert_len;
    const unsigned char *cv_cert_desc;
    const size_t cv_cert_desc_len;
    const unsigned char *cv_cert_request;
    const size_t cv_cert_request_len;
};

static const struct cv_cert cv_certs[] = {
    {cvc1, sizeof cvc1, cvc1_desc, sizeof cvc1_desc, NULL, 0},
    {cvc2, sizeof cvc2, NULL, 0, NULL, 0},
    {wrong_tc_ecdh_cvca, sizeof wrong_tc_ecdh_cvca, NULL, 0, NULL, 0},
    {kba_dvca_cert, sizeof kba_dvca_cert, NULL, 0, NULL, 0},
    {kba_cert, sizeof kba_cert, kba_cert_desc, sizeof kba_cert_desc, NULL, 0},
    {kba_dvca_cert_2, sizeof kba_dvca_cert_2, NULL, 0, NULL, 0},
    {kba_cert_2, sizeof kba_cert_2, kba_cert_desc_2, sizeof kba_cert_desc_2, NULL, 0},
    {kba_cert_3, sizeof kba_cert_3, kba_cert_desc_2, sizeof kba_cert_desc_2, NULL, 0},
    {kba_cert_4, sizeof kba_cert_4, kba_cert_desc_2, sizeof kba_cert_desc_2, NULL, 0},
    {schufa_dvcert, sizeof schufa_dvcert, NULL, 0, NULL, 0},
    {schufa_cert, sizeof schufa_cert, schufa_desc, sizeof schufa_desc, NULL, 0},
    {schufa_dvcert, sizeof schufa_dvcert, NULL, 0, NULL, 0},
    {schufa_cert, sizeof schufa_cert, schufa_desc, sizeof schufa_desc, NULL, 0},
    {mein_cockpit_dvcert, sizeof mein_cockpit_dvcert, NULL, 0, NULL, 0},
    {mein_cockpit_cert, sizeof mein_cockpit_cert, mein_cockpit_desc, sizeof mein_cockpit_desc, NULL, 0},
    {allianz_dvcert, sizeof allianz_dvcert, NULL, 0, NULL, 0},
    {allianz_cert, sizeof allianz_cert, allianz_desc, sizeof allianz_desc, NULL, 0},
    {service_bw_dvcert, sizeof service_bw_dvcert, NULL, 0, NULL, 0},
    {service_bw_cert, sizeof service_bw_cert, NULL, 0, NULL, 0},
    {service_bw_cert_2, sizeof service_bw_cert_2, NULL, 0, NULL, 0},
    {service_bw_cert_3, sizeof service_bw_cert_3, NULL, 0, NULL, 0},
    {service_bw_cert_4, sizeof service_bw_cert_4, NULL, 0, NULL, 0},
    {NULL, 0, NULL, 0, req_18102010_csr, sizeof req_18102010_csr},
    {NULL, 0, NULL, 0, MY_cvreq, sizeof MY_cvreq},
};


struct pace_params {
    int protocol;
    int std_dp;
};
struct ta_params {
    int protocol;
    int curve;
};
struct ca_params {
    int protocol;
    int curve;
};
struct ri_params {
    int protocol;
    int std_dp;
};
struct pace_secret {
    enum s_type type;
    char *secret;
};

static char pace_raw[0xff];
static const struct pace_secret dynamic_pace_secrets[] = {
    { PACE_PUK, "1234567890" },
    { PACE_CAN, "654321" },
    { PACE_PIN, "123456" },
    {
        PACE_MRZ,
        /* ePA MRZ: https:// www.bsi.bund.de/cae/servlet/contentblob/650964/normal/37609/ePA_beideSeiten.jpg */
        /* ePA MRZ: http://www.cio.bund.de/cln_093/sid_AF344A548BB315C1B49C3F627F830E35/SharedDocs/Kurzmeldungen/DE/2009/091217_finales_muster_neuer_personalausweis.html */
        "IDD<<T220001293<<<<<<<<<<<<<<<6408125<1010318D<<<<<<<<<<<<<<MUSTERMANN<<ERIKA<<<<<<<<<<<<<"
    },
    { PACE_RAW, pace_raw },
};
/******************************************************************************/

/***************************** Helper functions *******************************/

#if 0
static void hexdump(const char *title, const BUF_MEM *s)
{
    size_t n=0;

    fprintf(stdout,"%s",title);

    if (!s) {
        fprintf(stdout,"(null)\n");
    } else {
        for(; n < s->length; ++n)
        {
            if((n%16) == 0)
                fprintf(stdout,"\n    ");
            fprintf(stdout,"%02x:",(unsigned char) s->data[n]);
        }
        fprintf(stdout,"\n");
    }
}
static void hexdump(const char *title, const unsigned char *s, size_t l)
{
    size_t n=0;

    fprintf(stdout,"%s",title);

    if (!s) {
        fprintf(stdout,"(null)\n");
    } else {
        for(; n < l; ++n)
        {
            if((n%16) == 0)
                fprintf(stdout,"\n    ");
            fprintf(stdout,"%02x:",(unsigned char) s[n]);
        }
        fprintf(stdout,"\n");
    }
}
printf("%s:%d\n", __FILE__, __LINE__);
#endif

#if 0
#ifdef EMSCRIPTEN
static int predictable_index = 0;
/* we use tc_dh_shared_secret_k as it at least looks randomly */
const static char *predictable_buffer = tc_dh_shared_secret_k;
const static size_t predictable_max = sizeof tc_dh_shared_secret_k;
RAND_METHOD predictable;

#define MIN(a,b) (a<b?a:b)
int predictable_bytes(unsigned char *buf, int num)
{
   int count = 0;
   int i = predictable_index;
   int to_copy;

   while (count < num) {
      if (i >= predictable_max)
         i = 0;

      to_copy = MIN(predictable_max-i, num);

      memcpy(buf, predictable_buffer+i, to_copy);

      count += to_copy;
      i += to_copy;
   }

   if (i >= predictable_max)
      predictable_index = 0;
   else
      predictable_index = i;

   return count;
}

void set_rand(void)
{
   memset(&predictable, 0, sizeof predictable);
   predictable.bytes = predictable_bytes;
   predictable.pseudorand = predictable_bytes;
   RAND_set_rand_method(&predictable);
}
#else
void set_rand(void) { }
#endif
#endif


static EVP_PKEY *
generate_signature_key(int curve)
{
    RSA *rsa = NULL;
    EC_KEY *ec = NULL;
    DH *dh = NULL;
    BIGNUM *bn = NULL;

    EVP_PKEY *key = EVP_PKEY_new();
    if (!key)
        goto err;

    switch(curve) {
        case 0:
        case 1:
        case 2:
            if (!EVP_PKEY_set_std_dp(key, curve))
                goto err;
            dh = EVP_PKEY_get1_DH(key);
            if (!dh
                    || !DH_generate_key(dh)
                    || !EVP_PKEY_set1_DH(key, dh))
                goto err;
            break;

        case 8:
        case 9:
        case 10:
        case 11:
        case 12:
        case 13:
        case 14:
        case 15:
        case 16:
        case 17:
        case 18:
            if (!EVP_PKEY_set_std_dp(key, curve))
                goto err;
            ec = EVP_PKEY_get1_EC_KEY(key);
            if (!ec
                    || !EC_KEY_generate_key(ec)
                    || !EVP_PKEY_set1_EC_KEY(key, ec))
                goto err;
            break;

        default:
            /* RSA where curve specifies the keylength */
            if(((bn = BN_new()) == NULL)
                    || !BN_set_word(bn, RSA_F4)
                    || ((rsa = RSA_new()) == NULL)
                    || !RSA_generate_key_ex(rsa, curve, bn, NULL)
                    || !EVP_PKEY_set1_RSA(key, rsa))
                goto err;
            break;
    }

    if (bn)
        BN_clear_free(bn);
    if (rsa)
        RSA_free(rsa);
    if (dh)
        DH_free(dh);
    if (ec)
        EC_KEY_free(ec);

    return key;

err:
    if (bn)
        BN_clear_free(bn);
    if (rsa)
        RSA_free(rsa);
    if (dh)
        DH_free(dh);
    if (ec)
        EC_KEY_free(ec);
    if (key)
        EVP_PKEY_free(key);

    return NULL;
}

/* Initialize an EVP_PKEY container from buffers containing the private and public
 * key */
static int
EVP_PKEY_set_keys_buf(EVP_PKEY *evp_pkey,
           const BUF_MEM *privkey, const BUF_MEM *pubkey, BN_CTX *bn_ctx)
{
   if (!privkey || !pubkey)
      return 0;

   return EVP_PKEY_set_keys(evp_pkey,
         (const unsigned char *) privkey->data, privkey->length,
         (const unsigned char *) pubkey->data, pubkey->length, bn_ctx);
}

/* Compare the generator given in a buffer to the generator contained in an
 * EVP_PKEY container */
static int
check_generator(EVP_PKEY *evp_pkey, const BUF_MEM generator, BN_CTX *bn_ctx)
{
    EC_KEY *ec_key = NULL;
    DH *dh = NULL;
    EC_POINT *ec_point = NULL;
    BIGNUM *bn = NULL;
    const BIGNUM *g;
    int ok = 0;
    const EC_GROUP *group;

    switch (EVP_PKEY_base_id(evp_pkey)) {
        case EVP_PKEY_EC:
            ec_key = EVP_PKEY_get1_EC_KEY(evp_pkey);
            if (!ec_key)
               goto err;
            group = EC_KEY_get0_group(ec_key);
            if (!group)
               goto err;
            ec_point = EC_POINT_new(group);
            if (!ec_point
                    || !EC_POINT_oct2point(group, ec_point,
                        (unsigned char *) generator.data, generator.length, bn_ctx)
                    || EC_POINT_cmp(group, ec_point,
                        EC_GROUP_get0_generator(group), bn_ctx) != 0)
                goto err;
            break;
        case EVP_PKEY_DH:
        case EVP_PKEY_DHX:
            dh = EVP_PKEY_get1_DH(evp_pkey);
            bn = BN_bin2bn((unsigned char *) generator.data, generator.length, bn);
            DH_get0_pqg(dh, NULL, NULL, &g);
            if (!dh || !bn || BN_cmp(g, bn) != 0)
                goto err;
            break;
        default:
            goto err;
            break;
    }

    ok = 1;

err:
    if (ec_key)
        EC_KEY_free(ec_key);
    if (ec_point)
        EC_POINT_clear_free(ec_point);
    if (dh)
        DH_free(dh);
    if (bn)
        BN_clear_free(bn);

    return ok;
}

static int
buf_eq_buf(const BUF_MEM *buf1, const BUF_MEM *buf2)
{
    if (buf1 && buf1->length) {
        if (!buf2 || !buf2->length || buf1->length != buf2->length
                   || memcmp(buf1->data, buf2->data, buf1->length) != 0)
            return 0;
    } else {
        if (buf2 && buf2->length)
            return 0;
    }

    return 1;
}
#if 0
static int
buf_eq_str(const BUF_MEM *buf, const unsigned char *str, size_t len)
{
    if (len) {
        if (!buf || buf->length != len || memcmp(buf->data, str, len) != 0)
            return 0;
    } else {
        if (buf && buf->length)
            return 0;
    }

    return 1;
}
#endif

static void
print_desc(int protocol, int curve)
{
    const char *sn;
    sn = OBJ_nid2sn(protocol);
    if (curve > 31) {
        if (verbose)
            printf("  Testing RSA %d Parameters with %s\n", curve, sn);
        else
            printf("  Testing  RSA %4d  with %-*s ", curve, 32, sn);
    } else {
        if (verbose)
            printf("  Testing Std. Domain Parameters %d with %s\n", curve, sn);
        else
            printf("  Testing Std. DP %2d with %-*s ", curve, 32, sn);
    }
}
/******************************************************************************/

/****************************** Unit tests ************************************/

/* Do a test of the EAC protocols emulating both parties (PICC and PCD)
 * Keys are generated on the fly, only the secret used for PACE and the exact
 * protocol OIDs have to specified in advance. Right now only the PACE protocol
 * is tested. */
static int
dynamic_eac_test(const struct pace_secret pace_secret,
                 const struct pace_params pace_params,
                 const struct ta_params ta_params,
                 const struct ca_params ca_params,
                 const struct ri_params ri_params)
{
    int failed = 1;
    PACE_SEC *s = NULL;
    EAC_CTX *pcd_ctx = NULL, *picc_ctx = NULL;
    BUF_MEM *enc_nonce = NULL,
            *static_pub_picc = NULL, *static_pub_pcd = NULL,
            *ephemeral_pub_picc = NULL, *ephemeral_pub_pcd = NULL,
            *token_pcd = NULL, *token_picc = NULL,
            *id_picc = NULL, *ta_nonce = NULL,
            *ta_comp_ephemeral_key_for_ca = NULL, *ta_signature = NULL,
            *ca_picc_pubkey = NULL, *ca_pcd_pubkey = NULL, *ca_nonce = NULL,
            *ca_token = NULL, *ca_picc_comp_pubkey = NULL, *ri = NULL,
            *sector_pub = NULL;
    BIO *bio = NULL;

    print_desc(pace_params.protocol, pace_params.std_dp);


    /* Initialization */
    picc_ctx = EAC_CTX_new();
    pcd_ctx = EAC_CTX_new();
    s = PACE_SEC_new(pace_secret.secret, strlen(pace_secret.secret), pace_secret.type);
    CHECK(0, picc_ctx && pcd_ctx && s, "Creating Data");
    if (pace_params.protocol == NID_id_PACE_ECDH_IM_3DES_CBC_CBC
          || pace_params.protocol == NID_id_PACE_ECDH_IM_AES_CBC_CMAC_128
          || pace_params.protocol == NID_id_PACE_ECDH_IM_AES_CBC_CMAC_192
          || pace_params.protocol == NID_id_PACE_ECDH_IM_AES_CBC_CMAC_256
          || pace_params.protocol == NID_id_PACE_DH_IM_3DES_CBC_CBC
          || pace_params.protocol == NID_id_PACE_DH_IM_AES_CBC_CMAC_128
          || pace_params.protocol == NID_id_PACE_DH_IM_AES_CBC_CMAC_192
          || pace_params.protocol == NID_id_PACE_DH_IM_AES_CBC_CMAC_256) {
       picc_ctx->tr_version = EAC_TR_VERSION_2_01;
       pcd_ctx->tr_version = EAC_TR_VERSION_2_01;
    }
    CHECK(0, EAC_CTX_init_pace(picc_ctx, pace_params.protocol, pace_params.std_dp)
            && EAC_CTX_init_pace(pcd_ctx, pace_params.protocol, pace_params.std_dp),
            "Initializing PACE");


    /* PACE */
    enc_nonce = PACE_STEP1_enc_nonce(picc_ctx, s);
    CHECK(1, enc_nonce,
            "PACE step 1: MRTD generated and encrypted nonce.");

    CHECK(1, PACE_STEP2_dec_nonce(pcd_ctx, s, enc_nonce),
            "PACE step 2: Terminal decrypted the correct nonce");

    static_pub_picc = PACE_STEP3A_generate_mapping_data(picc_ctx);
    static_pub_pcd = PACE_STEP3A_generate_mapping_data(pcd_ctx);
    CHECK(1, static_pub_pcd && static_pub_picc,
            "PACE step 3a: Generated key pairs for mapping");

    if (picc_ctx->pace_ctx->protocol == NID_id_PACE_ECDH_IM_3DES_CBC_CBC
          || picc_ctx->pace_ctx->protocol == NID_id_PACE_ECDH_IM_AES_CBC_CMAC_128
          || picc_ctx->pace_ctx->protocol == NID_id_PACE_ECDH_IM_AES_CBC_CMAC_192
          || picc_ctx->pace_ctx->protocol == NID_id_PACE_ECDH_IM_AES_CBC_CMAC_256
          || picc_ctx->pace_ctx->protocol == NID_id_PACE_DH_IM_3DES_CBC_CBC
          || picc_ctx->pace_ctx->protocol == NID_id_PACE_DH_IM_AES_CBC_CMAC_128
          || picc_ctx->pace_ctx->protocol == NID_id_PACE_DH_IM_AES_CBC_CMAC_192
          || picc_ctx->pace_ctx->protocol == NID_id_PACE_DH_IM_AES_CBC_CMAC_256) {
            /* Integrated Mapping is special, only the terminal generates the key */
            CHECK(0, PACE_STEP3A_map_generator(pcd_ctx, static_pub_pcd),
                    "Computing mapped domain parameters");
    } else {
       CHECK(0, PACE_STEP3A_map_generator(pcd_ctx, static_pub_picc),
             "Computing mapped domain parameters");
    }
    CHECK(1, PACE_STEP3A_map_generator(picc_ctx, static_pub_pcd),
            "PACE step 3a: Mapped domain parameters on MRTD and Terminal");

    ephemeral_pub_picc = PACE_STEP3B_generate_ephemeral_key(picc_ctx);
    ephemeral_pub_pcd = PACE_STEP3B_generate_ephemeral_key(pcd_ctx);
    CHECK(1, PACE_STEP3B_compute_shared_secret(picc_ctx, ephemeral_pub_pcd)
            && PACE_STEP3B_compute_shared_secret(pcd_ctx, ephemeral_pub_picc)
            && buf_eq_buf(picc_ctx->pace_ctx->ka_ctx->shared_secret,
                    pcd_ctx->pace_ctx->ka_ctx->shared_secret),
            "PACE step 3b: computed shared secret");

    CHECK(1, PACE_STEP3C_derive_keys(picc_ctx)
            &&  PACE_STEP3C_derive_keys(pcd_ctx)
            &&  buf_eq_buf(picc_ctx->pace_ctx->ka_ctx->k_enc, pcd_ctx->pace_ctx->ka_ctx->k_enc)
            &&  buf_eq_buf(picc_ctx->pace_ctx->ka_ctx->k_mac, pcd_ctx->pace_ctx->ka_ctx->k_mac),
            "PACE step 3c: Derived authentication and encryption key");

    token_pcd = PACE_STEP3D_compute_authentication_token(pcd_ctx, ephemeral_pub_picc);
    token_picc = PACE_STEP3D_compute_authentication_token(picc_ctx, ephemeral_pub_pcd);
    CHECK(1, PACE_STEP3D_verify_authentication_token(pcd_ctx, token_picc) == 1
            && PACE_STEP3D_verify_authentication_token(picc_ctx, token_pcd) == 1,
            "PACE step 3d: Verified authentication tokens");

    /* Initialize secure channel */
    CHECK(0, EAC_CTX_set_encryption_ctx(pcd_ctx, EAC_ID_PACE)
            && EAC_CTX_set_encryption_ctx(picc_ctx, EAC_ID_PACE),
            "Set encryption/authentication context");
    id_picc = EAC_Comp(picc_ctx, EAC_ID_PACE, ephemeral_pub_picc);
    CHECK(0, id_picc,
            "Computed ID of MRTD and Terminal");

    OK;
    print_desc(ta_params.protocol, ta_params.curve);


    /* TODO write a clean wrapper to use for this kind of initialization */
    pcd_ctx->ta_ctx->protocol = ta_params.protocol;
    pcd_ctx->ta_ctx->priv_key = generate_signature_key(ta_params.curve);
    CHECK(0, pcd_ctx->ta_ctx->priv_key,
            "Failed to generate Terminal's TA private key");
    picc_ctx->ta_ctx->protocol = ta_params.protocol;
    picc_ctx->ta_ctx->pub_key = EVP_PKEY_dup(pcd_ctx->ta_ctx->priv_key);
    CHECK(0, picc_ctx->ta_ctx->pub_key,
            "Failed to import Terminal's TA public key to MRTD");

    /* Initialize the CA context for both parties */
    CHECK(0, EAC_CTX_init_ca(picc_ctx, ca_params.protocol, ca_params.curve)
            && EAC_CTX_init_ca(pcd_ctx, ca_params.protocol, ca_params.curve),
            "Initializing CA");
    /* Domain parameters for CA are now set for PCD and PICC, usually we would
     * like to simply generate a key on the PICC side. Since there is no
     * generic EVP_PKEY wrapper for this, we simply overwrite it with an
     * initialized one. */
    EVP_PKEY_free(picc_ctx->ca_ctx->ka_ctx->key);
    picc_ctx->ca_ctx->ka_ctx->key = generate_signature_key(ca_params.curve);
    CHECK(0, picc_ctx->ca_ctx->ka_ctx->key,
            "Failed to generate CA key");

    ta_comp_ephemeral_key_for_ca = TA_STEP3_generate_ephemeral_key(pcd_ctx);
    CHECK(1, ta_comp_ephemeral_key_for_ca,
            "TA step 3: Generated ephemeral key for subsequent CA");

    ta_nonce = TA_STEP4_get_nonce(picc_ctx);
    CHECK(1, ta_nonce && TA_STEP4_set_nonce(pcd_ctx, ta_nonce),
            "TA step 4: Generated nonce");

    ta_signature = TA_STEP5_sign(pcd_ctx, ta_comp_ephemeral_key_for_ca,
           id_picc, NULL);
    CHECK(1, ta_signature,
            "TA step 5: Created signature");

    CHECK(1, TA_STEP6_verify(picc_ctx, ta_comp_ephemeral_key_for_ca,
                id_picc, NULL, ta_signature) == 1,
            "TA step 6: Verified signature");

    OK;
    print_desc(ca_params.protocol, ca_params.curve);


    ca_picc_comp_pubkey = CA_STEP1_get_pubkey(picc_ctx);
    ca_picc_pubkey = get_pubkey(picc_ctx->ca_ctx->ka_ctx->key,
            picc_ctx->bn_ctx);
    CHECK(1, ca_picc_comp_pubkey && ca_picc_pubkey,
            "CA step 1: Exported MRTD's public key");

    ca_pcd_pubkey = CA_STEP2_get_eph_pubkey(pcd_ctx);
    CHECK(1, ca_pcd_pubkey,
            "CA step 2: Exported terminal's public key");

    CHECK(1, CA_STEP3_check_pcd_pubkey(picc_ctx,
                ta_comp_ephemeral_key_for_ca, ca_pcd_pubkey) == 1,
            "CA step 3: Checked terminal's public key");

    CHECK(1, CA_STEP4_compute_shared_secret(picc_ctx, ca_pcd_pubkey)
            && CA_STEP4_compute_shared_secret(pcd_ctx, ca_picc_pubkey),
            "CA step 4: Computed the shared secret");

    CHECK(1, CA_STEP5_derive_keys(picc_ctx, ca_pcd_pubkey, &ca_nonce, &ca_token),
            "CA step 5: Derived MRTD's keys");

    CHECK(1, CA_STEP6_derive_keys(pcd_ctx, ca_nonce, ca_token),
            "CA step 6: Derived terminal's keys");


    OK;
    print_desc(ri_params.protocol, ri_params.std_dp);

    CHECK(0, EAC_CTX_init_ri(picc_ctx, ri_params.protocol, ri_params.std_dp)
            && EAC_CTX_init_ri(pcd_ctx, ri_params.protocol, ri_params.std_dp),
            "Initializing RI");

    sector_pub = get_pubkey(pcd_ctx->ri_ctx->static_key, pcd_ctx->bn_ctx);
    CHECK(1, sector_pub, "RI step 1: Retrieving terminal's public key");

    ri = RI_STEP2_compute_identifier(picc_ctx, sector_pub);
    CHECK(1, ri, "RI step 2: Computing sector identifier");

    failed = 0;

err:
    if (debug) {
        bio = BIO_new_fp(stdout, BIO_NOCLOSE|BIO_FP_TEXT);
        if (bio) {
            BIO_printf(bio, "    PICC's EAC Context:\n");
            EAC_CTX_print_private(bio, picc_ctx, 6);
            BIO_printf(bio, "    PCD's EAC Context:\n");
            EAC_CTX_print_private(bio, pcd_ctx, 6);
            BIO_printf(bio, "    PACE step 1: Encrypted Nonce:\n");
            BUF_MEM_print(bio, enc_nonce, 6);
            BIO_printf(bio, "    PACE step 3a: PCD's mapping data:\n");
            BUF_MEM_print(bio, static_pub_pcd, 6);
            BIO_printf(bio, "    PACE step 3a: PICC's mapping data:\n");
            BUF_MEM_print(bio, static_pub_picc, 6);
            BIO_printf(bio, "    PACE step 3d: PCD's Authentication Token:\n");
            BUF_MEM_print(bio, token_pcd, 6);
            BIO_printf(bio, "    PACE step 3d: PICC's Authentication Token:\n");
            BUF_MEM_print(bio, token_picc, 6);
            BIO_printf(bio, "    PICC's compressed PACE public key (ID ICC):\n");
            BUF_MEM_print(bio, id_picc, 6);
            BIO_printf(bio, "    TA step 3: PCD's compressed CA public key:\n");
            BUF_MEM_print(bio, ta_comp_ephemeral_key_for_ca, 6);
            BIO_printf(bio, "    TA step 5: PCD's signature:\n");
            BUF_MEM_print(bio, ta_signature, 6);
            BIO_printf(bio, "    Sector-specific identifier:\n");
            BUF_MEM_print(bio, ri, 6);
            BIO_free_all(bio);
        }
    }
    if (token_picc)
        BUF_MEM_free(token_picc);
    if (token_pcd)
        BUF_MEM_free(token_pcd);
    if (ephemeral_pub_picc)
        BUF_MEM_free(ephemeral_pub_picc);
    if (ephemeral_pub_pcd)
        BUF_MEM_free(ephemeral_pub_pcd);
    if (static_pub_picc)
        BUF_MEM_free(static_pub_picc);
    if (static_pub_pcd)
        BUF_MEM_free(static_pub_pcd);
    if (enc_nonce)
        BUF_MEM_free(enc_nonce);
    if (id_picc)
        BUF_MEM_free(id_picc);
    if (ta_nonce)
        BUF_MEM_free(ta_nonce);
    if (ta_comp_ephemeral_key_for_ca)
        BUF_MEM_free(ta_comp_ephemeral_key_for_ca);
    if (ta_signature)
        BUF_MEM_free(ta_signature);
    if (ca_picc_comp_pubkey)
        BUF_MEM_free(ca_picc_comp_pubkey);
    if (ca_picc_pubkey)
        BUF_MEM_free(ca_picc_pubkey);
    if (ca_pcd_pubkey)
        BUF_MEM_free(ca_pcd_pubkey);
    if (ca_nonce)
        BUF_MEM_free(ca_nonce);
    if (ca_token)
        BUF_MEM_free(ca_token);
    if (ri)
        BUF_MEM_free(ri);
    if (sector_pub)
        BUF_MEM_free(sector_pub);
    EAC_CTX_clear_free(picc_ctx);
    EAC_CTX_clear_free(pcd_ctx);
    PACE_SEC_clear_free(s);

    END;
}

/* Perform dynamic tests of the EAC protocol using different combinations of the
 * parameters (pace_secret, protocols and standardized domain parameters)*/
static int
do_dynamic_eac_tests(void)
{
    size_t i, j, k, l, m;
    int failed = 0;
    const struct pace_params dynamic_pace_params[] = {
       { NID_id_PACE_DH_GM_3DES_CBC_CBC, 0 },
       { NID_id_PACE_DH_GM_AES_CBC_CMAC_128, 1},
       { NID_id_PACE_DH_GM_AES_CBC_CMAC_192, 2 },
       { NID_id_PACE_DH_GM_AES_CBC_CMAC_256, 0 },
       { NID_id_PACE_DH_IM_3DES_CBC_CBC, 1},
       { NID_id_PACE_DH_IM_AES_CBC_CMAC_128, 2 },
       { NID_id_PACE_DH_IM_AES_CBC_CMAC_192, 0 },
       { NID_id_PACE_DH_IM_AES_CBC_CMAC_256, 1},
       { NID_id_PACE_ECDH_GM_3DES_CBC_CBC, 11 },
       { NID_id_PACE_ECDH_GM_3DES_CBC_CBC, 16 },
       { NID_id_PACE_ECDH_GM_AES_CBC_CMAC_128, 9 },
       { NID_id_PACE_ECDH_GM_AES_CBC_CMAC_128, 12 },
       { NID_id_PACE_ECDH_GM_AES_CBC_CMAC_192, 18 },
       { NID_id_PACE_ECDH_GM_AES_CBC_CMAC_192, 10 },
       { NID_id_PACE_ECDH_GM_AES_CBC_CMAC_256, 15 },
       { NID_id_PACE_ECDH_IM_3DES_CBC_CBC, 13 },
       { NID_id_PACE_ECDH_IM_AES_CBC_CMAC_128, 14 },
       { NID_id_PACE_ECDH_IM_AES_CBC_CMAC_192, 8 },
       { NID_id_PACE_ECDH_IM_AES_CBC_CMAC_256, 17 },
    };
    const struct ta_params dynamic_ta_params[] = {
       { NID_id_TA_RSA_v1_5_SHA_1, 1024 },
       { NID_id_TA_RSA_v1_5_SHA_256, 1280 },
       { NID_id_TA_RSA_PSS_SHA_1, 1536 },
       { NID_id_TA_RSA_PSS_SHA_256, 2048 },
       { NID_id_TA_RSA_v1_5_SHA_512, 3072 },
       { NID_id_TA_RSA_PSS_SHA_512, 1024 },
       { NID_id_TA_ECDSA_SHA_1, 8 },
       { NID_id_TA_ECDSA_SHA_224, 9 },
       { NID_id_TA_ECDSA_SHA_256, 10 },
       { NID_id_TA_ECDSA_SHA_384, 11 },
       { NID_id_TA_ECDSA_SHA_512, 12 },
    };
    const struct ca_params dynamic_ca_params[] = {
       { NID_id_CA_DH_3DES_CBC_CBC, 0 },
       { NID_id_CA_DH_AES_CBC_CMAC_128, 1 },
       { NID_id_CA_DH_AES_CBC_CMAC_192, 2 },
       { NID_id_CA_DH_AES_CBC_CMAC_256, 0 },
       { NID_id_CA_ECDH_3DES_CBC_CBC, 8 },
       { NID_id_CA_ECDH_AES_CBC_CMAC_128, 11 },
       { NID_id_CA_ECDH_AES_CBC_CMAC_192, 13 },
       { NID_id_CA_ECDH_AES_CBC_CMAC_256, 14 },
    };
    const struct ri_params dynamic_ri_params[] = {
       { NID_id_RI_DH_SHA_1, 0 },
       { NID_id_RI_DH_SHA_224, 1 },
       { NID_id_RI_DH_SHA_256, 2 },
       { NID_id_RI_DH_SHA_384, 0 },
       { NID_id_RI_DH_SHA_512, 1 },
       { NID_id_RI_ECDH_SHA_1, 8 },
       { NID_id_RI_ECDH_SHA_224, 9 },
       { NID_id_RI_ECDH_SHA_256, 10 },
       { NID_id_RI_ECDH_SHA_384, 11 },
       { NID_id_RI_ECDH_SHA_512, 12 },
       { NID_id_RI_ECDH_SHA_1, 13 },
       { NID_id_RI_ECDH_SHA_224, 14 },
       { NID_id_RI_ECDH_SHA_256, 15 },
       { NID_id_RI_ECDH_SHA_384, 16 },
       { NID_id_RI_ECDH_SHA_512, 17 },
       { NID_id_RI_ECDH_SHA_1, 18 },
    };

    printf("Dynamic EAC tests:\n");

    for (i = 0, j = 0, k = 0, l = 0, m = 0;
            i < (sizeof dynamic_pace_params) / sizeof(*dynamic_pace_params);
            i++, j++, k++, l++, m++) {
        if (j >= (sizeof dynamic_pace_secrets) / sizeof(*dynamic_pace_secrets)) {
            j = 0;
        }
        if (k >= (sizeof dynamic_ta_params) / sizeof(*dynamic_ta_params)) {
            k = 0;
        }
        if (l >= (sizeof dynamic_ca_params) / sizeof(*dynamic_ca_params)) {
            l = 0;
        }
        if (m >= (sizeof dynamic_ri_params) / sizeof(*dynamic_ri_params)) {
            m = 0;
        }
        failed += dynamic_eac_test(dynamic_pace_secrets[j], dynamic_pace_params[i],
                dynamic_ta_params[k], dynamic_ca_params[l], dynamic_ri_params[m]);
    }

    return failed;
}

/* Encrypt and decrypt some data with the symmetric keys derived during PACE */
static int
test_enc(EAC_CTX *ctx, unsigned int send_sequence_counter,
        const BUF_MEM *data, const BUF_MEM *enc)
{
    int ok = 0;
    BUF_MEM *pad = NULL, *enc_buf = NULL, *dec_buf = NULL, *unpadded = NULL;

    CHECK(0, EAC_set_ssc(ctx, send_sequence_counter),
            "Initializing Send Sequence Counter");

    pad = EAC_add_iso_pad(ctx, data);
    enc_buf = EAC_encrypt(ctx, pad);
    CHECK(0, buf_eq_buf(enc_buf, enc),
            "Encrypting");

    dec_buf = EAC_decrypt(ctx, enc_buf);
    CHECK(0, buf_eq_buf(dec_buf, pad),
            "Decrypting");

    unpadded = EAC_remove_iso_pad(dec_buf);
    CHECK(0, buf_eq_buf(unpadded, data),
            "Removing padding");

    CHECK(1, 1, "Encrypting/Decrypting");

    ok = 1;

err:
    if (pad)
        BUF_MEM_free(pad);
    if (enc_buf)
        BUF_MEM_free(enc_buf);
    if (dec_buf)
        BUF_MEM_free(dec_buf);
    if (unpadded)
        BUF_MEM_free(unpadded);

    return ok;
}

/* Authenticate some data with the symmetric keys derived during PACE */
static int
test_auth(EAC_CTX *ctx, unsigned int send_sequence_counter,
        const BUF_MEM *data, const BUF_MEM *mac)
{
    int ok = 0;
    BUF_MEM *pad = NULL, *auth = NULL;

    CHECK(0, EAC_set_ssc(ctx, send_sequence_counter),
            "Initializing Send Sequence Counter");

    pad = EAC_add_iso_pad(ctx, data);
    auth = EAC_authenticate(ctx, pad);
    CHECK(1, buf_eq_buf(auth, mac), "Authenticating");

    CHECK(1, EAC_verify_authentication(ctx, pad, mac), "Verifying");

    ok = 1;

err:
    if (pad)
        BUF_MEM_free(pad);
    if (auth)
        BUF_MEM_free(auth);

    return ok;
}

static const struct eac_worked_example *current_tc;
CVC_CERT * dummy_lookup_cvca_cert(const unsigned char *chr, size_t car_len)
{
   const unsigned char *p = (const unsigned char *) current_tc->ta_cvca.data;
   return CVC_d2i_CVC_CERT(NULL, &p, current_tc->ta_cvca.length);
}

/* Perform one EAC protocol run using static test data */
static int
static_eac_test(struct eac_worked_example *tc)
{
    int failed = 1, i;
    PACE_SEC *pace_sec = NULL;
    BUF_MEM *enc_nonce = NULL, *token_picc = NULL, *token_pcd = NULL,
            *ta_comp_ephemeral_key_for_ca = NULL, *signature = NULL,
            *ephemeral_pub_picc = NULL, *id_picc = NULL,
            *ta_pcd_eph_pub_key_for_ca = NULL,
            *ta_pcd_comp_eph_pub_key_for_ca = NULL,
            *parsed_ca_picc_pub_key = NULL;
    EAC_CTX *picc_ctx = NULL, *pcd_ctx = NULL;
    BIO *bio = NULL;

    if (!tc)
       goto err;

    print_desc(tc->pace_info_oid, tc->pace_curve);

    /* PACE */

    /* Initialization */
    pace_sec = PACE_SEC_new(tc->password, strlen(tc->password),
            tc->password_type);
    picc_ctx = EAC_CTX_new();
    pcd_ctx = EAC_CTX_new();
    CA_disable_passive_authentication(pcd_ctx);
    CHECK(1, EAC_CTX_init_ef_cardsecurity((unsigned char *) tc->ef_cardsecurity.data,
                tc->ef_cardsecurity.length, picc_ctx)
            && EAC_CTX_init_ef_cardaccess((unsigned char *) tc->ef_cardaccess.data,
                tc->ef_cardaccess.length, pcd_ctx)
            && picc_ctx->pace_ctx->version == tc->pace_version
            && picc_ctx->pace_ctx->protocol == tc->pace_info_oid
            && pcd_ctx->pace_ctx->version == tc->pace_version
            && pcd_ctx->pace_ctx->protocol == tc->pace_info_oid,
            "Parsing EF.CardAccess or EF.CardSecurity");


    /* Generate, En-/Decrypt Nonce */
    picc_ctx->pace_ctx->nonce = BUF_MEM_create_init((unsigned char *) tc->pace_nonce.data, tc->pace_nonce.length);
    enc_nonce = BUF_MEM_create_init((unsigned char *) tc->pace_enc_nonce.data, tc->pace_enc_nonce.length);
    CHECK(1, PACE_STEP2_dec_nonce(pcd_ctx, pace_sec, enc_nonce)
            && buf_eq_buf(pcd_ctx->pace_ctx->nonce, &tc->pace_nonce),
            "PACE step 2: Terminal decrypted the correct nonce");


    /* Set static key pair and perform mapping to the ephemeral generator */
    /* Since the static keys are pre-defined we can omit the call to
     * PACE_STEP3A_generate_mapping_data */
    CHECK(1, EVP_PKEY_set_keys_buf(pcd_ctx->pace_ctx->static_key,
                   &tc->pace_static_pcd_priv_key, &tc->pace_static_pcd_pub_key,
                   pcd_ctx->bn_ctx)
            && PACE_STEP3A_map_generator(pcd_ctx, &tc->pace_static_picc_pub_key)
            && EVP_PKEY_set_keys_buf(picc_ctx->pace_ctx->static_key,
                &tc->pace_static_picc_priv_key, &tc->pace_static_picc_pub_key,
                picc_ctx->bn_ctx)
            && PACE_STEP3A_map_generator(picc_ctx, &tc->pace_static_pcd_pub_key)
            && check_generator(picc_ctx->pace_ctx->ka_ctx->key,
                tc->pace_eph_generator, picc_ctx->bn_ctx)
            && check_generator(pcd_ctx->pace_ctx->ka_ctx->key,
                tc->pace_eph_generator, pcd_ctx->bn_ctx),
            "PACE step 3a: Mapped to ephemeral domain parameters");


    /* Set the ephemeral key pair and perform the key agreement */
    /* Since the static keys are pre-defined we can omit the call to
     * PACE_STEP3B_generate_ephemeral_key */
    pcd_ctx->pace_ctx->my_eph_pubkey =
        BUF_MEM_create_init(tc->pace_eph_pcd_pub_key.data,
                tc->pace_eph_pcd_pub_key.length);
    picc_ctx->pace_ctx->my_eph_pubkey =
        BUF_MEM_create_init(tc->pace_eph_picc_pub_key.data,
                tc->pace_eph_picc_pub_key.length);
    CHECK(1, EVP_PKEY_set_keys_buf(pcd_ctx->pace_ctx->ka_ctx->key,
                   &tc->pace_eph_pcd_priv_key, &tc->pace_eph_pcd_pub_key,
                   pcd_ctx->bn_ctx)
            && EVP_PKEY_set_keys_buf(picc_ctx->pace_ctx->ka_ctx->key,
                &tc->pace_eph_picc_priv_key, &tc->pace_eph_picc_pub_key,
                picc_ctx->bn_ctx)
            && PACE_STEP3B_compute_shared_secret(pcd_ctx, &tc->pace_eph_picc_pub_key)
            && buf_eq_buf(pcd_ctx->pace_ctx->ka_ctx->shared_secret, &tc->pace_shared_secret_k)
            && PACE_STEP3B_compute_shared_secret(picc_ctx, &tc->pace_eph_pcd_pub_key)
            && buf_eq_buf(picc_ctx->pace_ctx->ka_ctx->shared_secret, &tc->pace_shared_secret_k),
            "PACE step 3b: computed shared secret");


    /* Derive session keys */
    CHECK(1, PACE_STEP3C_derive_keys(pcd_ctx)
            && buf_eq_buf(pcd_ctx->pace_ctx->ka_ctx->k_enc, &tc->pace_k_enc)
            && buf_eq_buf(pcd_ctx->pace_ctx->ka_ctx->k_mac, &tc->pace_k_mac)
            && PACE_STEP3C_derive_keys(picc_ctx)
            && buf_eq_buf(picc_ctx->pace_ctx->ka_ctx->k_enc, &tc->pace_k_enc)
            && buf_eq_buf(picc_ctx->pace_ctx->ka_ctx->k_mac, &tc->pace_k_mac),
            "PACE step 3c: Derived Keys");


    /* Derive and verify authentication tokens */
    token_pcd = PACE_STEP3D_compute_authentication_token(pcd_ctx,
            &tc->pace_eph_picc_pub_key);
    token_picc = PACE_STEP3D_compute_authentication_token(picc_ctx,
            &tc->pace_eph_pcd_pub_key);
    CHECK(1, buf_eq_buf(token_pcd, &tc->pace_authentication_token_pcd)
            && buf_eq_buf(token_picc, &tc->pace_authentication_token_picc)
            && PACE_STEP3D_verify_authentication_token(picc_ctx, token_pcd) == 1
            && PACE_STEP3D_verify_authentication_token(pcd_ctx, token_picc) == 1,
            "PACE step 3d: Computed and verified authentication tokens");


    /* Initialize secure channel */
    CHECK(0, EAC_CTX_set_encryption_ctx(pcd_ctx, EAC_ID_PACE)
            && EAC_CTX_set_encryption_ctx(picc_ctx, EAC_ID_PACE),
            "Initializing encryption");
    /* picc_ctx and pcd_ctx look the same with respect to SM, it doesn't matter
     * which one we are using for encryption/decryption or authentication */

    /* En-/Decrypt some test data */
    for (i = 0; (size_t) i < tc->pace_encrypt_decrypt_len; i++) {
        if (!test_enc(pcd_ctx, tc->pace_encrypt_decrypt[i].ssc,
                    &tc->pace_encrypt_decrypt[i].data,
                    &tc->pace_encrypt_decrypt[i].cipher))
            goto err;
    }

    /* Authenticate some test data */
    for (i = 0; (size_t) i < tc->pace_authenticate_len; i++) {
        if (!test_auth(picc_ctx, tc->pace_authenticate[i].ssc,
                    &tc->pace_authenticate[i].data,
                    &tc->pace_authenticate[i].mac))
            goto err;
    }

    /* Terminal Authentication*/
    TA_disable_checks(picc_ctx);
    TA_disable_checks(pcd_ctx);
    CA_disable_passive_authentication(picc_ctx);
    CA_disable_passive_authentication(pcd_ctx);

    /* We need to chose one of the supported CA keys specified in the EF.CardAccess
     * BEFORE Terminal Authentication. Therefore, we need to initialize the CA
     * context before TA. */
    parsed_ca_picc_pub_key = CA_get_pubkey(pcd_ctx, (unsigned char *) tc->ef_cardsecurity.data,
            tc->ef_cardsecurity.length);
    CHECK(1, buf_eq_buf(parsed_ca_picc_pub_key, &tc->ca_picc_pub_key)
            && CA_set_key(pcd_ctx,
                (unsigned char *) tc->ca_pcd_priv_key.data, tc->ca_pcd_priv_key.length,
                (unsigned char *) tc->ca_pcd_pub_key.data, tc->ca_pcd_pub_key.length)
            && CA_set_key(picc_ctx,
                (unsigned char *) tc->ca_picc_priv_key.data, tc->ca_picc_priv_key.length,
                (unsigned char *) tc->ca_picc_pub_key.data, tc->ca_picc_pub_key.length),
            "Initializing Chip Authentication");

    /* Initialize the TA contexts. The PICC looks up the trust anchor (CVCA) on
     * demand and the PCD gets initialized with the terminal certificate */
    current_tc = tc;
    CHECK(1, EAC_CTX_init_ta(pcd_ctx, (unsigned char *) tc->ta_pcd_key.data,
                tc->ta_pcd_key.length, (unsigned char *) tc->ta_cert.data,
                tc->ta_cert.length)
            && EAC_CTX_set_cvca_lookup(picc_ctx, dummy_lookup_cvca_cert),
            "Initializing Terminal Authentication");
    OK;
    print_desc(pcd_ctx->ta_ctx->protocol, tc->ta_curve);

    /* Import the certificate chain. Currently we only support chains of length
     * three (CVCA certificate, DV certificate and terminal certificate). The
     * trust anchor is already imported during the EAC_CTX_init_ta call above */
    CHECK(1, TA_STEP2_import_certificate(picc_ctx, (unsigned char *) tc->ta_dv_cert.data,
                tc->ta_dv_cert.length)
            && TA_STEP2_import_certificate(picc_ctx, (unsigned char *) tc->ta_cert.data,
                tc->ta_cert.length),
            "TA step 2: Imported and verified terminal's certificate");

    /* Import the specified nonce. This makes TA_STEP4_get_nonce superfluous. */
    CHECK(1, TA_STEP4_set_nonce(picc_ctx, &tc->ta_nonce)
            && TA_STEP4_set_nonce(pcd_ctx, &tc->ta_nonce),
            "TA STEP 4: Imported nonce");

    /* Create the signature. Since the challenge is pre-specified we do not
     * generate it on our own (and therefore don't call TA_STEP4_get_nonce */
    ephemeral_pub_picc = BUF_MEM_create_init(tc->pace_eph_picc_pub_key.data, tc->pace_eph_picc_pub_key.length);
    id_picc = EAC_Comp(picc_ctx, EAC_ID_PACE, ephemeral_pub_picc);
    ta_pcd_eph_pub_key_for_ca = BUF_MEM_create_init(tc->ca_pcd_pub_key.data, tc->ca_pcd_pub_key.length);
    ta_pcd_comp_eph_pub_key_for_ca = EAC_Comp(picc_ctx, EAC_ID_CA, ta_pcd_eph_pub_key_for_ca);
    signature = TA_STEP5_sign(pcd_ctx, ta_pcd_comp_eph_pub_key_for_ca, id_picc, &tc->ta_auxdata);
    CHECK(1, signature,
            "TA step 5: Signed nonce, public key, auxiliary data");

    /* Verify the response */
    if (picc_ctx->ta_ctx->protocol == NID_id_TA_RSA_v1_5_SHA_1
          || picc_ctx->ta_ctx->protocol == NID_id_TA_RSA_v1_5_SHA_256
          || picc_ctx->ta_ctx->protocol == NID_id_TA_RSA_v1_5_SHA_512) {
       /* In case of a probabilistic signature our signature will not be
        * equal to the signature specified in the worked example. Therefore,
        * we only compare the signatures in the deterministic case */
       CHECK(0, buf_eq_buf(signature, &tc->ta_pcd_signature),
             "Signature does match test data");
    }
    if (picc_ctx->ta_ctx->protocol == NID_id_TA_RSA_v1_5_SHA_1
          || picc_ctx->ta_ctx->protocol == NID_id_TA_RSA_v1_5_SHA_256
          || picc_ctx->ta_ctx->protocol == NID_id_TA_RSA_v1_5_SHA_512
          || picc_ctx->ta_ctx->protocol == NID_id_TA_RSA_PSS_SHA_1
          || picc_ctx->ta_ctx->protocol == NID_id_TA_RSA_PSS_SHA_256
          || picc_ctx->ta_ctx->protocol == NID_id_TA_RSA_PSS_SHA_512
          || picc_ctx->ta_ctx->protocol == NID_id_TA_ECDSA_SHA_1
          || picc_ctx->ta_ctx->protocol == NID_id_TA_RSA_v1_5_SHA_256
          || picc_ctx->ta_ctx->protocol == NID_id_TA_ECDSA_SHA_224
          || picc_ctx->ta_ctx->protocol == NID_id_TA_ECDSA_SHA_256
          || picc_ctx->ta_ctx->protocol == NID_id_TA_ECDSA_SHA_384
          || picc_ctx->ta_ctx->protocol == NID_id_TA_ECDSA_SHA_512) {
       /* Verify both our as well as the pre-defined signature (equal in the
        * deterministic case). */
       CHECK(1, TA_STEP6_verify(picc_ctx, ta_pcd_comp_eph_pub_key_for_ca, id_picc,
                &tc->ta_auxdata, signature) == 1
             && TA_STEP6_verify(picc_ctx, ta_pcd_comp_eph_pub_key_for_ca, id_picc,
                &tc->ta_auxdata, &tc->ta_pcd_signature) == 1,
             "TA step 6: Verified signature");
    } else {
       CHECK(1, 0, "Unknown TA protocol OID");
    }

    OK;
    print_desc(tc->ca_info_oid, tc->ca_curve);


    /* Chip Authentication */
    /* Generate the ephemeral CA key */
    ta_comp_ephemeral_key_for_ca = CA_STEP2_get_eph_pubkey(pcd_ctx);
    CHECK(1, buf_eq_buf(ta_comp_ephemeral_key_for_ca, &tc->ca_pcd_pub_key),
            "CA step 2: Encoded PCD's ephemeral key");

    /* We do not have the compressed public key and thus can't perform
     * CA_STEP3_check_pcd_pubkey */

    /* Perform the key agreement and check the results. */
    CHECK(1, CA_STEP4_compute_shared_secret(picc_ctx, ta_comp_ephemeral_key_for_ca)
            && buf_eq_buf(picc_ctx->ca_ctx->ka_ctx->shared_secret,
                &tc->ca_shared_secret_k)
            && CA_STEP4_compute_shared_secret(pcd_ctx, &tc->ca_picc_pub_key)
            && buf_eq_buf(pcd_ctx->ca_ctx->ka_ctx->shared_secret,
                &tc->ca_shared_secret_k),
            "CA step 4: Compute shared secret");

    /* Derive session keys */
    CHECK(1, CA_STEP6_derive_keys(pcd_ctx, &tc->ca_nonce, &tc->ca_picc_token)
            && buf_eq_buf(pcd_ctx->ca_ctx->ka_ctx->shared_secret,
                &tc->ca_shared_secret_k)
            && buf_eq_buf(pcd_ctx->ca_ctx->ka_ctx->k_enc,
                &tc->ca_k_enc)
            && buf_eq_buf(pcd_ctx->ca_ctx->ka_ctx->k_mac,
                &tc->ca_k_mac),
            "CA step 6: Derived keys for terminal");

    /* Initialize secure channel (here keys were derived for PCD only) */
    CHECK(1, EAC_CTX_set_encryption_ctx(pcd_ctx, EAC_ID_CA),
            "Initializing encryption");

    failed = 0;

err:
    if (debug) {
        bio = BIO_new_fp(stdout, BIO_NOCLOSE|BIO_FP_TEXT);
        if (bio && tc) {
            CVC_CERT *cert = NULL;
            if (CVC_d2i_CVC_CERT(&cert, (const unsigned char **) &tc->ta_cvca.data, tc->ta_cvca.length)) {
               BIO_printf(bio, "    CVCA:\n");
               CVC_print(bio, cert, 6);
            }
            if (CVC_d2i_CVC_CERT(&cert, (const unsigned char **) &tc->ta_dv_cert.data, tc->ta_dv_cert.length)) {
               BIO_printf(bio, "    DVCA:\n");
               CVC_print(bio, cert, 6);
            }
            if (cert)
               CVC_CERT_free(cert);
            BIO_printf(bio, "    Terminal Certificate:\n");
            CVC_print(bio, picc_ctx->ta_ctx->current_cert, 0);
            BIO_printf(bio, "    PICC's EAC Context:\n");
            EAC_CTX_print_private(bio, picc_ctx, 6);
            BIO_printf(bio, "    PCD's EAC Context:\n");
            EAC_CTX_print_private(bio, pcd_ctx, 6);
            BIO_printf(bio, "    PACE step 1: Encrypted Nonce:\n");
            BUF_MEM_print(bio, enc_nonce, 6);
            BIO_printf(bio, "    PACE step 3a: PCD's mapping data:\n");
            BUF_MEM_print(bio, &tc->pace_eph_pcd_pub_key, 6);
            BIO_printf(bio, "    PACE step 3a: PICC's mapping data:\n");
            BUF_MEM_print(bio, &tc->pace_eph_picc_pub_key, 6);
            BIO_printf(bio, "    PACE step 3d: PCD's Authentication Token:\n");
            BUF_MEM_print(bio, token_pcd, 6);
            BIO_printf(bio, "    PACE step 3d: PICC's Authentication Token:\n");
            BUF_MEM_print(bio, token_picc, 6);
            BIO_printf(bio, "    PICC's compressed PACE public key (ID ICC):\n");
            BUF_MEM_print(bio, id_picc, 6);
            BIO_printf(bio, "    TA step 3: PCD's compressed CA public key:\n");
            BUF_MEM_print(bio, ta_comp_ephemeral_key_for_ca, 6);
            BIO_printf(bio, "    TA step 5: PCD's signature:\n");
            BUF_MEM_print(bio, &tc->ta_pcd_signature, 6);
            BIO_free_all(bio);
        }
    }

    if (ephemeral_pub_picc)
        BUF_MEM_free(ephemeral_pub_picc);
    if (id_picc)
        BUF_MEM_free(id_picc);
    if (ta_pcd_eph_pub_key_for_ca)
        BUF_MEM_free(ta_pcd_eph_pub_key_for_ca);
    if (ta_pcd_comp_eph_pub_key_for_ca)
        BUF_MEM_free(ta_pcd_comp_eph_pub_key_for_ca);
    if (parsed_ca_picc_pub_key)
        BUF_MEM_free(parsed_ca_picc_pub_key);
    if (signature)
        BUF_MEM_free(signature);
    if (ta_comp_ephemeral_key_for_ca)
        BUF_MEM_free(ta_comp_ephemeral_key_for_ca);
    if (enc_nonce)
        BUF_MEM_free(enc_nonce);
    if (token_picc)
        BUF_MEM_free(token_picc);
    if (token_pcd)
        BUF_MEM_free(token_pcd);
    EAC_CTX_clear_free(picc_ctx);
    EAC_CTX_clear_free(pcd_ctx);
    PACE_SEC_clear_free(pace_sec);

    END;
}

/* Perform all the tests specified in the EAC worked example */
static int
test_worked_examples(void)
{
    size_t i;
    int failed = 0;
    struct eac_worked_example eac_examples[] = {
       {   /* EAC worked example - ECDH */
          /* ef_cardaccess */
          { sizeof tc_ecdh_ef_cardaccess, tc_ecdh_ef_cardaccess, sizeof tc_ecdh_ef_cardaccess, },
          /* ef_cardsecurity */
          { sizeof tc_ecdh_ef_cardsecurity, tc_ecdh_ef_cardsecurity, sizeof tc_ecdh_ef_cardsecurity, },
          /* pace_info_oid */
          NID_id_PACE_ECDH_GM_AES_CBC_CMAC_128,
          /* pace_version */
          2,
          /* pace_curve */
          0x0D,
          /* password */
          "123456",
          /* password_type */
          PACE_PIN,
          /* pace_nonce */
          { sizeof tc_ecdh_nonce, tc_ecdh_nonce, sizeof tc_ecdh_nonce, },
          /* pace_enc_nonce */
          { sizeof tc_ecdh_nonce_enc, tc_ecdh_nonce_enc, sizeof tc_ecdh_nonce_enc, },
          /* pace_static_pcd_priv_key */
          { sizeof tc_ecdh_map_pcd_priv_key, tc_ecdh_map_pcd_priv_key, sizeof tc_ecdh_map_pcd_priv_key, },
          /* pace_static_pcd_pub_key */
          { sizeof tc_ecdh_map_pcd_pub_key, tc_ecdh_map_pcd_pub_key, sizeof tc_ecdh_map_pcd_pub_key, },
          /* pace_static_picc_priv_key */
          { sizeof tc_ecdh_map_picc_priv_key, tc_ecdh_map_picc_priv_key, sizeof tc_ecdh_map_picc_priv_key, },
          /* pace_static_picc_pub_key */
          { sizeof tc_ecdh_map_picc_pub_key, tc_ecdh_map_picc_pub_key, sizeof tc_ecdh_map_picc_pub_key, },
          /* pace_shared_secret_h */
          { sizeof tc_ecdh_map_shared_secret_h, tc_ecdh_map_shared_secret_h, sizeof tc_ecdh_map_shared_secret_h, },
          /* pace_eph_generator */
          { sizeof tc_ecdh_map_generator, tc_ecdh_map_generator, sizeof tc_ecdh_map_generator, },
          /* pace_eph_pcd_priv_key */
          { sizeof tc_ecdh_pcd_priv_key, tc_ecdh_pcd_priv_key, sizeof tc_ecdh_pcd_priv_key, },
          /* pace_eph_pcd_pub_key */
          { sizeof tc_ecdh_pcd_pub_key, tc_ecdh_pcd_pub_key, sizeof tc_ecdh_pcd_pub_key, },
          /* pace_eph_picc_priv_key */
          { sizeof tc_ecdh_picc_priv_key, tc_ecdh_picc_priv_key, sizeof tc_ecdh_picc_priv_key, },
          /* pace_eph_picc_pub_key */
          { sizeof tc_ecdh_picc_pub_key, tc_ecdh_picc_pub_key, sizeof tc_ecdh_picc_pub_key, },
          /* pace_shared_secret_k */
          { sizeof tc_ecdh_shared_secret_k, tc_ecdh_shared_secret_k, sizeof tc_ecdh_shared_secret_k, },
          /* pace_k_mac */
          { sizeof tc_ecdh_k_mac, tc_ecdh_k_mac, sizeof tc_ecdh_k_mac, },
          /* pace_k_enc */
          { sizeof tc_ecdh_k_enc, tc_ecdh_k_enc, sizeof tc_ecdh_k_enc, },
          /* pace_authentication_token_picc */
          { sizeof tc_ecdh_authentication_token_picc, tc_ecdh_authentication_token_picc, sizeof tc_ecdh_authentication_token_picc, },
          /* pace_authentication_token_pcd */
          { sizeof tc_ecdh_authentication_token_pcd, tc_ecdh_authentication_token_pcd, sizeof tc_ecdh_authentication_token_pcd, },
          /* pace_encrypt_decrypt */
          tc_ecdh_enc_dec,
          /* pace_encrypt_decrypt_len */
          (sizeof tc_ecdh_enc_dec)/sizeof *tc_ecdh_enc_dec,
          /* pace_authenticate */
          tc_ecdh_pace_authenticate,
          /* pace_authenticate_len */
          (sizeof tc_ecdh_pace_authenticate)/sizeof *tc_ecdh_pace_authenticate,
          /* ca_curve */
          0x0D,
          /* ca_version */
          2,
          /* ca_info_oid */
          NID_id_CA_ECDH_AES_CBC_CMAC_128,
          /* ca_picc_priv_key */
          { sizeof tc_ecdh_ca_picc_priv_key, tc_ecdh_ca_picc_priv_key, sizeof tc_ecdh_ca_picc_priv_key, },
          /* ca_picc_pub_key */
          { sizeof tc_ecdh_ca_picc_pub_key, tc_ecdh_ca_picc_pub_key, sizeof tc_ecdh_ca_picc_pub_key, },
          /* ca_pcd_priv_key */
          { sizeof tc_ecdh_ca_pcd_priv_key, tc_ecdh_ca_pcd_priv_key, sizeof tc_ecdh_ca_pcd_priv_key, },
          /* ca_pcd_pub_key */
          { sizeof tc_ecdh_ca_pcd_pub_key, tc_ecdh_ca_pcd_pub_key, sizeof tc_ecdh_ca_pcd_pub_key, },
          /* ca_nonce */
          { sizeof tc_ecdh_ca_nonce, tc_ecdh_ca_nonce, sizeof tc_ecdh_ca_nonce, },
          /* ca_picc_token */
          { sizeof tc_ecdh_ca_picc_token, tc_ecdh_ca_picc_token, sizeof tc_ecdh_ca_picc_token, },
          /* ca_shared_secret_k */
          { sizeof tc_ecdh_ca_shared_secret_k, tc_ecdh_ca_shared_secret_k, sizeof tc_ecdh_ca_shared_secret_k, },
          /* ca_k_mac */
          { sizeof tc_ecdh_ca_k_mac, tc_ecdh_ca_k_mac, sizeof tc_ecdh_ca_k_mac, },
          /* ca_k_enc */
          { sizeof tc_ecdh_ca_k_enc, tc_ecdh_ca_k_enc, sizeof tc_ecdh_ca_k_enc, },
          /* ta_curve */
          13,
          /* ta_pcd_key */
          { sizeof tc_ecdh_ta_pcd_key, tc_ecdh_ta_pcd_key, sizeof tc_ecdh_ta_pcd_key, },
          /* ta_nonce */
          { sizeof tc_ecdh_ta_nonce, tc_ecdh_ta_nonce, sizeof tc_ecdh_ta_nonce, },
          /* ta_auxdata */
          { 0, NULL, 0, },
          /* ta_pcd_signature */
          { sizeof tc_ecdh_ta_pcd_signature, tc_ecdh_ta_pcd_signature, sizeof tc_ecdh_ta_pcd_signature, },
          /* ta_cert */
          { sizeof tc_ecdh_ta_cert, tc_ecdh_ta_cert, sizeof tc_ecdh_ta_cert, },
          /* ta_cvca */
          { sizeof tc_ecdh_cvca_cert, tc_ecdh_cvca_cert, sizeof tc_ecdh_cvca_cert, },
          /* ta_dv_cert */
          { sizeof tc_ecdh_dv_cert, tc_ecdh_dv_cert, sizeof tc_ecdh_dv_cert, },
       },
       {   /* EAC worked example - DH */
          /* ef_cardaccess */
          { sizeof tc_dh_ef_cardaccess, tc_dh_ef_cardaccess, sizeof tc_dh_ef_cardaccess, },
          /* ef_cardsecurity */
          { sizeof tc_dh_ef_cardsecurity, tc_dh_ef_cardsecurity, sizeof tc_dh_ef_cardsecurity, },
          /* pace_info_oid */
          NID_id_PACE_DH_GM_AES_CBC_CMAC_128,
          /* pace_version */
          2,
          /* pace_curve */
          0x00,
          /* password */
          "123456",
          /* password_type */
          PACE_PIN,
          /* pace_nonce */
          { sizeof tc_dh_nonce, tc_dh_nonce, sizeof tc_dh_nonce, },
          /* pace_enc_nonce */
          { sizeof tc_dh_nonce_enc, tc_dh_nonce_enc, sizeof tc_dh_nonce_enc, },
          /* pace_static_pcd_priv_key */
          { sizeof tc_dh_map_pcd_priv_key, tc_dh_map_pcd_priv_key, sizeof tc_dh_map_pcd_priv_key, },
          /* pace_static_pcd_pub_key */
          { sizeof tc_dh_map_pcd_pub_key, tc_dh_map_pcd_pub_key, sizeof tc_dh_map_pcd_pub_key, },
          /* pace_static_picc_priv_key */
          { sizeof tc_dh_map_picc_priv_key, tc_dh_map_picc_priv_key, sizeof tc_dh_map_picc_priv_key, },
          /* pace_static_picc_pub_key */
          { sizeof tc_dh_map_picc_pub_key, tc_dh_map_picc_pub_key, sizeof tc_dh_map_picc_pub_key, },
          /* pace_shared_secret_h */
          { sizeof tc_dh_map_shared_secret_h, tc_dh_map_shared_secret_h, sizeof tc_dh_map_shared_secret_h, },
          /* pace_eph_generator */
          { sizeof tc_dh_map_generator, tc_dh_map_generator, sizeof tc_dh_map_generator, },
          /* pace_eph_pcd_priv_key */
          { sizeof tc_dh_pcd_priv_key, tc_dh_pcd_priv_key, sizeof tc_dh_pcd_priv_key, },
          /* pace_eph_pcd_pub_key */
          { sizeof tc_dh_pcd_pub_key, tc_dh_pcd_pub_key, sizeof tc_dh_pcd_pub_key, },
          /* pace_eph_picc_priv_key */
          { sizeof tc_dh_picc_priv_key, tc_dh_picc_priv_key, sizeof tc_dh_picc_priv_key, },
          /* pace_eph_picc_pub_key */
          { sizeof tc_dh_picc_pub_key, tc_dh_picc_pub_key, sizeof tc_dh_picc_pub_key, },
          /* pace_shared_secret_k */
          { sizeof tc_dh_shared_secret_k, tc_dh_shared_secret_k, sizeof tc_dh_shared_secret_k, },
          /* pace_k_mac */
          { sizeof tc_dh_k_mac, tc_dh_k_mac, sizeof tc_dh_k_mac, },
          /* pace_k_enc */
          { sizeof tc_dh_k_enc, tc_dh_k_enc, sizeof tc_dh_k_enc, },
          /* pace_authentication_token_picc */
          { sizeof tc_dh_authentication_token_picc, tc_dh_authentication_token_picc, sizeof tc_dh_authentication_token_picc, },
          /* pace_authentication_token_pcd */
          { sizeof tc_dh_authentication_token_pcd, tc_dh_authentication_token_pcd, sizeof tc_dh_authentication_token_pcd, },
          /* pace_encrypt_decrypt */
          tc_dh_enc_dec,
          /* pace_encrypt_decrypt_len */
          (sizeof tc_dh_enc_dec)/sizeof *tc_dh_enc_dec,
          /* pace_authenticate */
          tc_dh_pace_authenticate,
          /* pace_authenticate_len */
          (sizeof tc_dh_pace_authenticate)/sizeof *tc_dh_pace_authenticate,
          /* ca_curve */
          0x00,
          /* ca_version */
          2,
          /* ca_info_oid */
          NID_id_CA_DH_AES_CBC_CMAC_128,
          /* ca_picc_priv_key */
          { sizeof tc_dh_ca_picc_priv_key, tc_dh_ca_picc_priv_key, sizeof tc_dh_ca_picc_priv_key, },
          /* ca_picc_pub_key */
          { sizeof tc_dh_ca_picc_pub_key, tc_dh_ca_picc_pub_key, sizeof tc_dh_ca_picc_pub_key, },
          /* ca_pcd_priv_key */
          { sizeof tc_dh_ca_pcd_priv_key, tc_dh_ca_pcd_priv_key, sizeof tc_dh_ca_pcd_priv_key, },
          /* ca_pcd_pub_key */
          { sizeof tc_dh_ca_pcd_pub_key, tc_dh_ca_pcd_pub_key, sizeof tc_dh_ca_pcd_pub_key, },
          /* ca_nonce */
          { sizeof tc_dh_ca_nonce, tc_dh_ca_nonce, sizeof tc_dh_ca_nonce, },
          /* ca_picc_token */
          { sizeof tc_dh_ca_picc_token, tc_dh_ca_picc_token, sizeof tc_dh_ca_picc_token, },
          /* ca_shared_secret_k */
          { sizeof tc_dh_ca_shared_secret_k, tc_dh_ca_shared_secret_k, sizeof tc_dh_ca_shared_secret_k, },
          /* ca_k_mac */
          { sizeof tc_dh_ca_k_mac, tc_dh_ca_k_mac, sizeof tc_dh_ca_k_mac, },
          /* ca_k_enc */
          { sizeof tc_dh_ca_k_enc, tc_dh_ca_k_enc, sizeof tc_dh_ca_k_enc, },
          /* ta_curve */
          0,
          /* ta_pcd_key */
          { sizeof tc_dh_ta_pcd_key, tc_dh_ta_pcd_key, sizeof tc_dh_ta_pcd_key, },
          /* ta_nonce */
          { sizeof tc_dh_ta_nonce, tc_dh_ta_nonce, sizeof tc_dh_ta_nonce, },
          /* ta_auxdata */
          { 0, NULL, 0, },
          /* ta_pcd_signature */
          { sizeof tc_dh_ta_pcd_signature, tc_dh_ta_pcd_signature, sizeof tc_dh_ta_pcd_signature, },
          /* ta_cert */
          { sizeof tc_dh_ta_cert, tc_dh_ta_cert, sizeof tc_dh_ta_cert, },
          /* ta_cvca */
          { sizeof tc_dh_cvca, tc_dh_cvca, sizeof tc_dh_cvca, },
          /* ta_dv_cert */
          { sizeof tc_dh_dv_cert, tc_dh_dv_cert, sizeof tc_dh_dv_cert, },
       },
    };


    printf("EAC worked examples:\n");

    for (i = 0; i<(sizeof eac_examples)/sizeof(*eac_examples); i++) {
        failed += static_eac_test(&eac_examples[i]);
    }

    return failed;
}

/* Try to parse an EF.CardSecurity structure and extract the content */
static X509_STORE *store = NULL;
X509_STORE *lookup_csca_cert(unsigned long issuer_name_hash)
{
    return store;
}
static int
test_ef_cardsecurity_parsing(const struct tc_ef_cardsecurity tc)
{
    BUF_MEM *pubkey = NULL;
    static X509 *csca_cert = NULL;
    const unsigned char *p;
    int failed = 1;
    BIO *bio = NULL;

    EAC_CTX *ctx = NULL;

    ctx = EAC_CTX_new();
    CHECK(1, ctx && EAC_CTX_init_ef_cardaccess((unsigned char *) tc.ef_cardaccess.data,
                tc.ef_cardaccess.length, ctx) && ctx->ca_ctx,
            "Parsed EF.CardAccess");


    p = (unsigned char *) tc.csca.data;
    d2i_X509(&csca_cert, &p, tc.csca.length);

    /* Initialize the trust store */
    if (!store)
       store = X509_STORE_new();
    CHECK(1, store, "Failed to create truststore");
    X509_STORE_add_cert(store, csca_cert);


    ctx->ca_ctx->flags = 0;
    ctx->ca_ctx->lookup_csca_cert = lookup_csca_cert;

    pubkey = CA_get_pubkey(ctx, (unsigned char *) tc.ef_cardsecurity.data, tc.ef_cardsecurity.length);
    CHECK(1, pubkey, "Parsed EF.CardSecurity");

    failed = 0;

err:
    if (debug) {
        bio = BIO_new_fp(stdout, BIO_NOCLOSE|BIO_FP_TEXT);
        if (bio) {
            BIO_printf(bio, "    EAC Context:\n");
            EAC_CTX_print_private(bio, ctx, 6);
            BIO_free_all(bio);
        }
    }
    if (pubkey)
        BUF_MEM_free(pubkey);
    if (ctx)
        EAC_CTX_clear_free(ctx);

    return failed;
}

/* Try to parse an EF.CardAccess structure and extract the content */
static int
test_ef_cardaccess_parsing(const BUF_MEM tc_ef_cardaccess )
{
    int failed = 1;
    EAC_CTX *ctx = NULL;
    BIO *bio = NULL;

    ctx = EAC_CTX_new();
    CHECK(1, ctx && EAC_CTX_init_ef_cardaccess((unsigned char *) tc_ef_cardaccess.data,
                tc_ef_cardaccess.length, ctx),
            "Parsed EF.CardAccess");

    failed = 0;

err:
    if (debug) {
        bio = BIO_new_fp(stdout, BIO_NOCLOSE|BIO_FP_TEXT);
        if (bio) {
            BIO_printf(bio, "    EAC Context:\n");
            EAC_CTX_print_private(bio, ctx, 6);
            BIO_free_all(bio);
        }
    }
    EAC_CTX_clear_free(ctx);

    return failed;
}

/* Try to parse a CV certificate and optionally the corresponding certificate
 * description*/
static int
test_cv_cert_parsing(const struct cv_cert tc)
{
    BIO *bio = NULL;
    CVC_CERT *cvc_cert = NULL;
    CVC_CERTIFICATE_DESCRIPTION *desc = NULL;
    CVC_CERT_REQUEST *request = NULL;
    const unsigned char *p;
    int failed = 1;

    if (tc.cv_cert && tc.cv_cert_len) {
        p = tc.cv_cert;
        cvc_cert = CVC_d2i_CVC_CERT(NULL, &p, tc.cv_cert_len);
        CHECK(1, cvc_cert,
                "Parsed CV Certificate");
    }

    if (tc.cv_cert_desc && tc.cv_cert_desc_len) {
        p = tc.cv_cert_desc;
        desc = d2i_CVC_CERTIFICATE_DESCRIPTION(NULL, &p, tc.cv_cert_desc_len);
        CHECK(1, desc,
                "Parsed Certificate Description");
    }

    if (tc.cv_cert && tc.cv_cert_len && tc.cv_cert_desc && tc.cv_cert_desc_len) {
        CHECK(1, 1 == CVC_check_description(cvc_cert, tc.cv_cert_desc, tc.cv_cert_desc_len),
                "Certificate Description matches Certificate");
    }

    if (tc.cv_cert_request && tc.cv_cert_request_len) {
        p = tc.cv_cert_request;
        request = d2i_CVC_CERT_REQUEST(NULL, &p, tc.cv_cert_request_len);
        CHECK(1, request,
                "Parsed Certificate Request");
        CHECK(1, 1 == CVC_verify_request_signature(request),
                "certificate request verified");
    }

    failed = 0;

err:
    if (debug) {
        bio = BIO_new_fp(stdout, BIO_NOCLOSE|BIO_FP_TEXT);
        if (bio) {
            BIO_printf(bio, "    Card Verifiable Certificate:\n");
            CVC_print(bio, cvc_cert, 6);
            certificate_description_print(bio, desc, 6);
            BIO_printf(bio, "    Certificate Request:\n");
            certificate_request_print(bio, request, 6);
        }
    }

    if (desc)
        CVC_CERTIFICATE_DESCRIPTION_free(desc);
    if (cvc_cert)
        CVC_CERT_free(cvc_cert);
    if (request)
        CVC_CERT_REQUEST_free(request);
    if (bio)
        BIO_free_all(bio);

    return failed;
}

/* Perform the parsing of EF.CardAccess and CV certificates for various test
 * vectors */
static int
test_parsing(void)
{
    size_t i;
    int failed = 0;

    printf("Parsing capabilities:\n");

    printf("  EF.CardSecurity            %s", verbose ? "\n" : " ");
    for (i = 0; i<(sizeof ef_cardsecurity_tests)/sizeof(*ef_cardsecurity_tests); i++) {
        failed += test_ef_cardsecurity_parsing(ef_cardsecurity_tests[i]);
    }
    TESTEND;

    printf("  EF.CardAccess              %s", verbose ? "\n" : " ");
    for (i = 0; i<(sizeof ef_cardaccess_files)/sizeof(*ef_cardaccess_files); i++) {
        failed += test_ef_cardaccess_parsing(ef_cardaccess_files[i]);
    }
    TESTEND;

    printf("  Card Verifiable Certificate%s", verbose ? "\n" : " ");
    for (i = 0; i<(sizeof cv_certs)/sizeof(*cv_certs); i++) {
        failed += test_cv_cert_parsing(cv_certs[i]);
    }

    END;
}



/******************************************************************************/

int
main(int argc, char *argv[])
{
    int i;
    size_t j;
    int failed = 0;
    unsigned char c;

    for (i=1; i<argc; i++)
    {
        if ((strcmp(argv[i], "--verbose") == 0)
                || (strcmp(argv[i], "-v") == 0)) {
            verbose++;
            continue;
        }
        if ((strcmp(argv[i], "--debug") == 0)
                || (strcmp(argv[i], "-d") == 0)) {
            verbose++;
            debug++;
            continue;
        }
        if ((strcmp(argv[i], "--help") == 0)
                || (strcmp(argv[i], "-h") == 0)) {
            fprintf(stderr,
                    "%s tests the OpenSSL implementation of EAC\n"
                    "\n"
                    "  Extended Access Control (EAC) includes Password Authentication\n"
                    "  Connection Establishment (PACE), Terminal Authentication (TA)\n"
                    "  and Chip Authentication (CA).\n"
                    "\n"
                    "Usage: %s [Options]\n"
                    "\n"
                    "Options:\n"
                    "  -h,--help         show this help message and exit\n"
                    "  -v,--verbose      more verbosity\n"
                    "  -d,--debug        even more verbosity\n"
                    "     --version      print version information and exit\n"
                    , argv[0], argv[0]
            );
            return 0;
        }
        if (strcmp(argv[i], "--version") == 0) {
            fprintf(stderr,
                    "%s 0.6\n"
                    "\n"
                    "Written by Frank Morgner and Dominik Oepen.\n"
                    , argv[0]
            );
            return 0;
        }

        fprintf(stderr, "unrecognized option \"%s\"\n", argv[i]);
        return 1;
    }

    /* all possible characters as character string */
    for (c = 0x20, j = 0; j < (sizeof pace_raw) - 1; c++, j++) {
        while (c <= 0x1f || (c >= 0x7f && c <= 0x9f)) {
            c++;
        }
        pace_raw[j] = c;
    }
    pace_raw[(sizeof pace_raw) - 1] = '\0';

    EAC_init();
    failed += test_parsing();
    failed += test_worked_examples();
    failed += do_dynamic_eac_tests();
    EAC_cleanup();

    if (failed)
        printf("%d error%s collected.\n", failed, failed>1 ? "s" : "");
    else
        printf("Everything works as expected.\n");

    return failed;
}
