/*
 * Decompiled with CFR 0.152.
 */
package org.graalvm.compiler.code;

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import jdk.vm.ci.code.CodeUtil;
import org.graalvm.compiler.code.CompilationResult;

public class HexCodeFile {
    public static final String NEW_LINE = CodeUtil.NEW_LINE;
    public static final String SECTION_DELIM = " <||@";
    public static final String COLUMN_END = " <|@";
    public static final Pattern SECTION = Pattern.compile("(\\S+)\\s+(.*)", 32);
    public static final Pattern COMMENT;
    public static final Pattern OPERAND_COMMENT;
    public static final Pattern JUMP_TABLE;
    public static final Pattern LOOKUP_TABLE;
    public static final Pattern HEX_CODE;
    public static final Pattern PLATFORM;
    public static final String EMBEDDED_HCF_OPEN = "<<<HexCodeFile";
    public static final String EMBEDDED_HCF_CLOSE = "HexCodeFile>>>";
    public final Map<Integer, List<String>> comments = new TreeMap<Integer, List<String>>();
    public final Map<Integer, List<String>> operandComments = new TreeMap<Integer, List<String>>();
    public final byte[] code;
    public final ArrayList<CompilationResult.JumpTable> jumpTables = new ArrayList();
    public final String isa;
    public final int wordWidth;
    public final long startAddress;

    public HexCodeFile(byte[] code, long startAddress, String isa, int wordWidth) {
        this.code = code;
        this.startAddress = startAddress;
        this.isa = isa;
        this.wordWidth = wordWidth;
    }

    public static HexCodeFile parse(String input, int sourceOffset, String source, String sourceName) {
        return new Parser((String)input, (int)sourceOffset, (String)source, (String)sourceName).hcf;
    }

    public String toString() {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        this.writeTo(baos);
        return baos.toString();
    }

    public String toEmbeddedString() {
        return EMBEDDED_HCF_OPEN + NEW_LINE + this.toString() + EMBEDDED_HCF_CLOSE;
    }

    public void writeTo(OutputStream out) {
        PrintStream ps = out instanceof PrintStream ? (PrintStream)out : new PrintStream(out);
        ps.printf("Platform %s %d %s%n", this.isa, this.wordWidth, SECTION_DELIM);
        ps.printf("HexCode %x %s %s%n", this.startAddress, HexCodeFile.hexCodeString(this.code), SECTION_DELIM);
        for (CompilationResult.JumpTable jumpTable : this.jumpTables) {
            ps.printf("JumpTable %d %d %d %d %s%n", jumpTable.position, jumpTable.entrySize, jumpTable.low, jumpTable.high, SECTION_DELIM);
        }
        for (Map.Entry entry : this.comments.entrySet()) {
            int pos = (Integer)entry.getKey();
            for (String comment : (List)entry.getValue()) {
                ps.printf("Comment %d %s %s%n", pos, comment, SECTION_DELIM);
            }
        }
        for (Map.Entry entry : this.operandComments.entrySet()) {
            for (String c : (List)entry.getValue()) {
                ps.printf("OperandComment %d %s %s%n", entry.getKey(), c, SECTION_DELIM);
            }
        }
        ps.flush();
    }

    public static String hexCodeString(byte[] code) {
        if (code == null) {
            return "";
        }
        StringBuilder sb = new StringBuilder(code.length * 2);
        for (byte b : code) {
            String hex = Integer.toHexString(b & 0xFF);
            if (hex.length() == 1) {
                sb.append('0');
            }
            sb.append(hex);
        }
        return sb.toString();
    }

    public void addComment(int pos, String comment) {
        List<String> list = this.comments.get(pos);
        if (list == null) {
            list = new ArrayList<String>();
            this.comments.put(pos, list);
        }
        list.add(HexCodeFile.encodeString(comment));
    }

    public void addOperandComment(int pos, String comment) {
        List<String> list = this.comments.get(pos);
        if (list == null) {
            list = new ArrayList<String>(1);
            this.comments.put(pos, list);
        }
        list.add(HexCodeFile.encodeString(comment));
    }

    public static void addAnnotations(HexCodeFile hcf, List<CompilationResult.CodeAnnotation> annotations) {
        if (annotations == null || annotations.isEmpty()) {
            return;
        }
        for (CompilationResult.CodeAnnotation a : annotations) {
            if (a instanceof CompilationResult.JumpTable) {
                CompilationResult.JumpTable table = (CompilationResult.JumpTable)a;
                hcf.jumpTables.add(table);
                continue;
            }
            if (!(a instanceof CompilationResult.CodeComment)) continue;
            CompilationResult.CodeComment comment = (CompilationResult.CodeComment)a;
            hcf.addComment(comment.position, comment.value);
        }
    }

    public static String encodeString(String input) {
        int index;
        String s = input;
        while ((index = s.indexOf(SECTION_DELIM)) != -1) {
            s = s.substring(0, index) + " < |@" + s.substring(index + SECTION_DELIM.length());
        }
        while ((index = s.indexOf(COLUMN_END)) != -1) {
            s = s.substring(0, index) + " < @" + s.substring(index + COLUMN_END.length());
        }
        return s;
    }

    static {
        OPERAND_COMMENT = COMMENT = Pattern.compile("(\\d+)\\s+(.*)", 32);
        JUMP_TABLE = Pattern.compile("(\\d+)\\s+(\\d+)\\s+(-{0,1}\\d+)\\s+(-{0,1}\\d+)\\s*");
        LOOKUP_TABLE = Pattern.compile("(\\d+)\\s+(\\d+)\\s+(\\d+)\\s+(\\d+)\\s*");
        HEX_CODE = Pattern.compile("(\\p{XDigit}+)(?:\\s+(\\p{XDigit}*))?");
        PLATFORM = Pattern.compile("(\\S+)\\s+(\\S+)", 32);
    }

    static class Parser {
        final String input;
        final String inputSource;
        String isa;
        int wordWidth;
        byte[] code;
        long startAddress;
        HexCodeFile hcf;

        Parser(String input, int sourceOffset, String source, String sourceName) {
            this.input = input;
            this.inputSource = sourceName;
            this.parseSections(sourceOffset, source);
        }

        void makeHCF() {
            if (this.hcf == null && this.isa != null && this.wordWidth != 0 && this.code != null) {
                this.hcf = new HexCodeFile(this.code, this.startAddress, this.isa, this.wordWidth);
            }
        }

        void checkHCF(String section, int offset) {
            this.check(this.hcf != null, offset, section + " section must be after Platform and HexCode section");
        }

        void check(boolean condition, int offset, String message) {
            if (!condition) {
                this.error(offset, message);
            }
        }

        Error error(int offset, String message) {
            throw new Error(this.errorMessage(offset, message));
        }

        void warning(int offset, String message) {
            PrintStream err = System.err;
            err.println("Warning: " + this.errorMessage(offset, message));
        }

        String errorMessage(int offset, String message) {
            assert (offset < this.input.length());
            InputPos inputPos = this.filePos(offset);
            int lineEnd = this.input.indexOf(NEW_LINE, offset);
            int lineStart = offset - inputPos.col;
            String line = lineEnd == -1 ? this.input.substring(lineStart) : this.input.substring(lineStart, lineEnd);
            return String.format("%s:%d: %s%n%s%n%" + (inputPos.col + 1) + "s", this.inputSource, inputPos.line, message, line, "^");
        }

        InputPos filePos(int index) {
            assert (this.input != null);
            int lineStart = this.input.lastIndexOf(NEW_LINE, index) + 1;
            String l = this.input.substring(lineStart, lineStart + 10);
            PrintStream out = System.out;
            out.println("YYY" + this.input.substring(index, index + 10) + "...");
            out.println("XXX" + l + "...");
            int pos = this.input.indexOf(NEW_LINE, 0);
            int line = 1;
            while (pos > 0 && pos < index) {
                ++line;
                pos = this.input.indexOf(NEW_LINE, pos + 1);
            }
            return new InputPos(line, index - lineStart);
        }

        void parseSections(int offset, String source) {
            assert (this.input.startsWith(source, offset));
            int index = 0;
            int endIndex = source.indexOf(HexCodeFile.SECTION_DELIM);
            while (endIndex != -1) {
                while (source.charAt(index) <= ' ') {
                    ++index;
                }
                String section = source.substring(index, endIndex).trim();
                this.parseSection(offset + index, section);
                index = endIndex + HexCodeFile.SECTION_DELIM.length();
                endIndex = source.indexOf(HexCodeFile.SECTION_DELIM, index);
            }
        }

        int parseInt(int offset, String value) {
            try {
                return Integer.parseInt(value);
            }
            catch (NumberFormatException e) {
                throw this.error(offset, "Not a valid integer: " + value);
            }
        }

        void parseSection(int offset, String section) {
            if (section.isEmpty()) {
                return;
            }
            assert (this.input.startsWith(section, offset));
            Matcher m = SECTION.matcher(section);
            this.check(m.matches(), offset, "Section does not match pattern " + SECTION);
            String header = m.group(1);
            String body = m.group(2);
            int headerOffset = offset + m.start(1);
            int bodyOffset = offset + m.start(2);
            if (header.equals("Platform")) {
                this.check(this.isa == null, bodyOffset, "Duplicate Platform section found");
                m = PLATFORM.matcher(body);
                this.check(m.matches(), bodyOffset, "Platform does not match pattern " + PLATFORM);
                this.isa = m.group(1);
                this.wordWidth = this.parseInt(bodyOffset + m.start(2), m.group(2));
                this.makeHCF();
            } else if (header.equals("HexCode")) {
                this.check(this.code == null, bodyOffset, "Duplicate Code section found");
                m = HEX_CODE.matcher(body);
                this.check(m.matches(), bodyOffset, "Code does not match pattern " + HEX_CODE);
                String hexAddress = m.group(1);
                this.startAddress = Long.valueOf(hexAddress, 16);
                String hexCode = m.group(2);
                if (hexCode == null) {
                    this.code = new byte[0];
                } else {
                    this.check(hexCode.length() % 2 == 0, bodyOffset, "Hex code length must be even");
                    this.code = new byte[hexCode.length() / 2];
                    for (int i = 0; i < this.code.length; ++i) {
                        String hexByte = hexCode.substring(i * 2, (i + 1) * 2);
                        this.code[i] = (byte)Integer.parseInt(hexByte, 16);
                    }
                }
                this.makeHCF();
            } else if (header.equals("Comment")) {
                this.checkHCF("Comment", headerOffset);
                m = COMMENT.matcher(body);
                this.check(m.matches(), bodyOffset, "Comment does not match pattern " + COMMENT);
                int pos = this.parseInt(bodyOffset + m.start(1), m.group(1));
                String comment = m.group(2);
                this.hcf.addComment(pos, comment);
            } else if (header.equals("OperandComment")) {
                this.checkHCF("OperandComment", headerOffset);
                m = OPERAND_COMMENT.matcher(body);
                this.check(m.matches(), bodyOffset, "OperandComment does not match pattern " + OPERAND_COMMENT);
                int pos = this.parseInt(bodyOffset + m.start(1), m.group(1));
                String comment = m.group(2);
                this.hcf.addOperandComment(pos, comment);
            } else if (header.equals("JumpTable")) {
                this.checkHCF("JumpTable", headerOffset);
                m = JUMP_TABLE.matcher(body);
                this.check(m.matches(), bodyOffset, "JumpTable does not match pattern " + JUMP_TABLE);
                int pos = this.parseInt(bodyOffset + m.start(1), m.group(1));
                int entrySize = this.parseInt(bodyOffset + m.start(2), m.group(2));
                int low = this.parseInt(bodyOffset + m.start(3), m.group(3));
                int high = this.parseInt(bodyOffset + m.start(4), m.group(4));
                this.hcf.jumpTables.add(new CompilationResult.JumpTable(pos, low, high, entrySize));
            } else {
                this.error(offset, "Unknown section header: " + header);
            }
        }

        static class InputPos {
            final int line;
            final int col;

            InputPos(int line, int col) {
                this.line = line;
                this.col = col;
            }
        }
    }
}

