// -*- C++ -*-

/* This software was produced by NIST, an agency of the U.S. government,
 * and by statute is not subject to copyright in the United States.
 * Recipients of this software assume all responsibilities associated
 * with its operation, modification and maintenance. However, to
 * facilitate maintenance we ask that before distributing modified
 * versions of this software, you first contact the authors at
 * oof_manager@nist.gov. 
 */

#include <oofconfig.h>

#include <algorithm>
#include <assert.h>
#include <iomanip>
#include <iostream>
#include <map>
#include <math.h>		// for sqrt()
#include <stdlib.h>		// for abs()
#include <vector>

#include "common/activearea.h"
#include "common/cmicrostructure.h"
#include "common/coord.h"
#include "common/geometry.h"
#include "common/lock.h"
#include "common/printvec.h"
#include "common/pixelattribute.h"
#include "common/pixelsetboundary.h"
#include "common/random.h"

using namespace std;		// TODO: Don't do this.

// TODO: Are groups_attributes_lock and category_lock still required?
// Shouldn't the locks in the Who class be sufficient?

//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//

long CMicrostructure::globalMicrostructureCount = 0; // used for code testing
static SLock globalMicrostructureCountLock;

long get_globalMicrostructureCount() {
  return CMicrostructure::globalMicrostructureCount;
}

CMicrostructure::CMicrostructure(const std::string &name,
				 const ICoord *isz, const Coord *sz) 
  : pxlsize_(*isz),
    size_(*sz),
    attributeMap(nAttributes()),
    attributeGlobalData(nAttributes()),
    categorized(false),
    ncategories(0),
    name_(name)
{
    globalMicrostructureCountLock.acquire();
    ++globalMicrostructureCount;
    globalMicrostructureCountLock.release();
#if DIM == 2
  delta_ = Coord((*sz)(0)/(*isz)(0), (*sz)(1)/(*isz)(1));
  categorymap = Array<int>((*isz)(0), (*isz)(1));
#elif DIM == 3
  delta_ = Coord((*sz)(0)/(*isz)(0), (*sz)(1)/(*isz)(1), (*sz)(2)/(*isz)(2));
  categorymap = Array<int>((*isz)(0), (*isz)(1), (*isz)(2));
#endif
  // Initialize pixel attribute maps.  Lock probably not actually
  // required -- who would contend for data during construction? --
  // but certainly harmless, and maybe safe.
  // std::cerr << "Acquire." << std::endl;
  // category_lock.acquire();
  for(std::vector<Array<PixelAttribute*> >::size_type i=0; i<attributeMap.size(); i++) {
    Array<PixelAttribute*> &map = attributeMap[i];
    const PxlAttributeRegistration *pareg =
      PxlAttributeRegistration::getRegistration(i);
    attributeGlobalData[i] = pareg->createAttributeGlobalData(this);
    map.resize(pxlsize_);
    for(Array<PixelAttribute*>::iterator j=map.begin(); j!=map.end(); ++j) {
      map[j] = pareg->createAttribute(j.coord());
    }
  }
}

CMicrostructure::~CMicrostructure() {
  destroy();
  globalMicrostructureCountLock.acquire();
  --globalMicrostructureCount;
  globalMicrostructureCountLock.release();
}

// This routine, and the constructor, could lock the
// groups_attributes_lock for writing, but it's probably not required.
void CMicrostructure::destroy() {
  for(PixelGroupDict::iterator i=pixelgroups.begin(); i!=pixelgroups.end(); ++i)
    delete (*i).second;
  pixelgroups.clear();
//   for(std::vector<PixelGroup*>::size_type i=0; i<pixelgroups.size(); i++) 
//     delete pixelgroups[i];
//   pixelgroups.resize(0);

  for(std::vector<PixelSetBoundary*>::iterator i=categoryBdys.begin();
      i!=categoryBdys.end(); ++i)
    delete *i;
  categoryBdys.clear();

  for(std::vector<Array<PixelAttribute*> >::size_type i=0;
      i<attributeMap.size(); i++) {
    Array<PixelAttribute*> &map = attributeMap[i];
    for(Array<PixelAttribute*>::iterator j=map.begin(); j!=map.end(); ++j)
      delete *j;
  }
  attributeMap.resize(0);
  for(std::vector<PixelAttributeGlobalData*>::size_type i=0;
      i<attributeGlobalData.size(); i++)
    delete attributeGlobalData[i];
  attributeGlobalData.resize(0);
}

Array<PixelAttribute*> &CMicrostructure::getAttributeMap(std::size_t which)
  const
{
  groups_attributes_lock.read_acquire();
#ifdef DEBUG
  assert(which >= 0);
  assert(which < attributeMap.size());
#endif
  Array<PixelAttribute*> &res = attributeMap[which];
  groups_attributes_lock.read_release();
  return res;
}

PixelAttributeGlobalData *
CMicrostructure::getAttributeGlobalData(std::size_t which)
  const 
{
  return attributeGlobalData[which];
}

TimeStamp &CMicrostructure::getTimeStamp() {
  return timestamp;
}

const TimeStamp &CMicrostructure::getTimeStamp() const {
  return timestamp;
}

std::size_t CMicrostructure::nGroups() const {
  groups_attributes_lock.read_acquire();
  std::size_t res = pixelgroups.size();
  groups_attributes_lock.read_release();
  return res;
}

// Convert a Coord in the physical space to pixel coordinates (without
// rounding to the nearest integer).
Coord CMicrostructure::physical2Pixel(const Coord &pt) const {
  return Coord(pt(0)/delta_(0), pt(1)/delta_(1));
}

// Return the physical space coordinates of the lower-left corner of a
// pixel.
Coord CMicrostructure::pixel2Physical(const ICoord &pxl) const {
  return Coord(pxl(0)*delta_(0), pxl(1)*delta_(1));
}

// Return the physical space coordinates of a given non-integer
// coordinate in pixel space.
Coord CMicrostructure::pixel2Physical(const Coord &pt) const {
#if DIM == 2
  return Coord(pt(0)*delta_(0), pt(1)*delta_(1));
#elif DIM == 3
  return Coord(pt(0)*delta_(0), pt(1)*delta_(1), pt(2)*delta_(2));
#endif
}

// Return the coordinates of the pixel that contains the given point.
ICoord CMicrostructure::pixelFromPoint(const Coord &pt) const {
  Coord p = physical2Pixel(pt);
  int xx = (int) floor(p(0));
  int yy = (int) floor(p(1));
  if(xx >= pxlsize_(0))
    --xx;
  if(xx < 0.0)			// round-off can make xx==-1.
    xx = 0.0;
  if(yy >= pxlsize_(1))
    --yy;
  if(yy < 0.0)
    yy = 0.0;
  return ICoord(xx, yy);
}

bool CMicrostructure::contains(const ICoord &ip) const {
  if ((ip(0)>=0 && ip(0)<pxlsize_(0)) && (ip(1)>=0 && ip(1)<pxlsize_(1)))
    return true;
  return false;
}

std::vector<ICoord> CMicrostructure::shuffledPix() const {
  std::vector<ICoord> pix;
  pix.reserve(pxlsize_(0)*pxlsize_(1));
  for(int i=0; i<pxlsize_(0); i++)
    for(int j=0; j<pxlsize_(1); j++) {
      ICoord p(i, j);
      if(activearea->isActive(p))
	pix.push_back(p);
    }
  shuffleVector(pix);
  return pix;
}

PixelGroup *CMicrostructure::findGroup(const std::string &name) const {
  // Get an existing group.  Don't create one if it doesn't already exist.
  groups_attributes_lock.read_acquire();
  PixelGroup *res = 0;

  PixelGroupDict::const_iterator i = pixelgroups.find(name);
  if(i != pixelgroups.end())
    res = (*i).second;

//   for(std::vector<PixelGroup*>::size_type i=0; i<pixelgroups.size(); i++) {
//     if(pixelgroups[i]->name() == name) {
//       res = pixelgroups[i];
//       break;
//     }
//   }
  groups_attributes_lock.read_release();
  return res;
}


PixelGroup *CMicrostructure::getGroup(const std::string &name,
				      bool *newness) {
  // Get an existing group, or create one if necessary.  Set the "newness"
  // pointer according to whether or not a new group is created.
  *newness = false;
  // findGroup independently handles the groups_attributes_lock.
  PixelGroup *grp = findGroup(name);
  if(grp) return grp;
  groups_attributes_lock.write_acquire();
  PixelGroup *newgrp = new PixelGroup(name, &pxlsize_, this);
//   pixelgroups.push_back(newgrp);
  pixelgroups.insert(PixelGroupDict::value_type(name, newgrp));
  ++timestamp;
  *newness = true;
  groups_attributes_lock.write_release();
  return newgrp;
}

void CMicrostructure::removeGroup(const std::string &name) {
  // Remove a group from the list of pixel groups, and mark it as
  // defunct.  Defunct groups will be removed from the groupmap later.
  category_lock.acquire();
  groups_attributes_lock.read_acquire();
  
  PixelGroupDict::iterator i = pixelgroups.find(name);
  if(i != pixelgroups.end()) {
    PixelGroup *grp = (*i).second;
    pixelgroups.erase(i);
    grp->set_defunct();
    defunctgroups.push_back(grp);
    categorized = false;
    ++timestamp;
  }
  groups_attributes_lock.read_release();
  category_lock.release();
}

void CMicrostructure::removeAllGroups() {
  category_lock.acquire();
  groups_attributes_lock.read_acquire();
  for(PixelGroupDict::iterator i=pixelgroups.begin(); i!=pixelgroups.end(); ++i)
    {
      PixelGroup *grp = (*i).second;
      grp->set_defunct();
      defunctgroups.push_back(grp);
    }
  pixelgroups.clear();
  categorized = false;
  ++timestamp;
  groups_attributes_lock.read_release();
  category_lock.release();
}

std::vector<std::string> *CMicrostructure::groupNames() const {
  std::vector<std::string> *roster = new std::vector<std::string>;
  groups_attributes_lock.read_acquire();
  for(PixelGroupDict::const_iterator i=pixelgroups.begin();
      i!=pixelgroups.end(); ++i)
    {
      roster->push_back((*i).first);
    }
//   for(std::vector<PixelGroup*>::size_type i=0; i<pixelgroups.size(); ++i)
//     roster->push_back(pixelgroups[i]->name());
  groups_attributes_lock.read_release();
  return roster;
}

void CMicrostructure::renameGroupC(const std::string &oldname,
				   const std::string &newname)
{
  PixelGroup *grp = findGroup(oldname);
  grp->rename(newname);
  pixelgroups.erase(oldname);
  pixelgroups.insert(PixelGroupDict::value_type(newname, grp));
}

// Comparison function for PixelAttribute vectors, used when mapping
// attributes to categories.

static bool ltAttributes(const std::vector<PixelAttribute*> &pavec0,
			 const std::vector<PixelAttribute*> &pavec1)
{
  std::size_t n0 = pavec0.size();
  std::size_t n1 = pavec1.size();
  std::size_t n = n0;
  if(n1 < n0) n = n1;
  for(std::size_t i=0; i<n; i++) {
    if(*pavec0[i] < *pavec1[i]) return true;
    if(*pavec1[i] < *pavec0[i]) return false;
  }
  if(n0 < n1) return true;
  return false;
}

// A CatMap is an STL map between vectors of PixelAttributes and
// categories.  This is not the same as CMicrostructure::categorymap,
// which is an array of categories (ie, a map from 2D space to
// categories).

typedef std::map<const std::vector<PixelAttribute*>, int,
		 bool (*)(const std::vector<PixelAttribute*>&,
			  const std::vector<PixelAttribute*>&) > CatMap;

// This function should only be run with the category_lock acquired.
// It is the caller's responsibility to do this -- all callers are
// within the CMicrostructure class, because this function (and the
// lock) is private.
void CMicrostructure::categorize() const {
  groups_attributes_lock.read_acquire();
  CatMap catmap(ltAttributes);	// maps lists of groups to categories
  representativePixels.resize(0);
  for(std::vector<PixelSetBoundary*>::iterator p=categoryBdys.begin();
      p!=categoryBdys.end(); ++p)
    delete *p;
  categoryBdys.clear();

  ncategories = 0;
  std::size_t nattrs = attributeMap.size();
  // loop over pixels in the microstructure
  for(Array<int>::iterator i=categorymap.begin(); i!=categorymap.end(); ++i) {

    // TODO: Instead of adding all pixels to the PixelSetBoundary and
    // searching for the edges later, this should check the categories
    // of the neighboring pixels and only add the boundary segments,
    // like the 3D code does.
    
    // construct a list of the attributes of this pixel
    std::vector<PixelAttribute*> attrs(nattrs);
    const ICoord &where = i.coord();
    for(int j=0; j<nattrs; j++)
      attrs[j] = attributeMap[j][where];
    // See if this list of attributes has been seen already
    CatMap::iterator cat = catmap.find(attrs);
    if(cat == catmap.end()) {	// category not found
      catmap[attrs] = ncategories;
      categorymap[i] = ncategories;
      // representativePixels.push_back(i.coord());
      representativePixels.push_back(where);

      PixelSetBoundary *psb = new PixelSetBoundary(this);
      psb->add_pixel(where);
      categoryBdys.push_back(psb);

      ncategories++;
    }
    else {
      categorymap[i] = (*cat).second; // assign previously found category
      categoryBdys[ (*cat).second ]->add_pixel(where);
    }
  }

  // It may not be obvious to the casual reader, but at this point the
  // defunct pixel groups have been removed from the GroupLists, and
  // so the groups can actually be deleted.  What has happened is that
  // the search for categories has called operator<() on pairs of
  // PixelAttributes.  GroupList is a PixelAttribute.  GroupList's
  // operator<() calls GroupList.members(), which sorts the list and
  // removes the defunct groups from it.  PixelGroups are the only
  // PixelAttribute that is specifically stored in the Microstructure,
  // so they get special treatment here.

  if(defunctgroups.size() > 0) {
    for(std::vector<PixelGroup*>::size_type i=0; i<defunctgroups.size(); ++i)
      delete defunctgroups[i];
    defunctgroups.resize(0);
  }

  // Find the boundaries.
  for(std::vector<PixelSetBoundary*>::iterator i = categoryBdys.begin();
      i!=categoryBdys.end(); ++i )
    {
      (*i)->find_boundary();
    }
 categorized = true;
 groups_attributes_lock.read_release();
} // end CMicrostructure::categorize()

unsigned int CMicrostructure::nCategories() const {
  category_lock.acquire();
  if(!categorized)
    categorize();
  unsigned int res = ncategories;
  category_lock.release();
  return res;
}

int CMicrostructure::category(const ICoord *where) const {
  category_lock.acquire();
  if(!categorized)
    categorize();
  int cat = categorymap[*where];
  category_lock.release();
  return cat;
}

int CMicrostructure::category(const ICoord &where) const {
  category_lock.acquire();
  if(!categorized)
    categorize();
  int res = categorymap[where];
  category_lock.release();
  return res;
}

int CMicrostructure::category(int x, int y) const {
  category_lock.acquire();
  if(!categorized)
    categorize();
  int res = categorymap[ICoord(x,y)];
  category_lock.release();
  return res;
}

// Special version for finding the category of the pixel under an
// arbitrary point.
int CMicrostructure::category(const Coord &where) const {
  category_lock.acquire();
  if(!categorized) 
    categorize();
  int res = categorymap[pixelFromPoint(where)];
  category_lock.release();
  return res;
}

const ICoord &CMicrostructure::getRepresentativePixel(std::size_t category)
  const
{
  category_lock.acquire();
  if(!categorized)
    categorize();
  ICoord &res = representativePixels[category];
  category_lock.release();
  return res;
}

const Array<int> *CMicrostructure::getCategoryMap() const {
  category_lock.acquire();
  if(!categorized)
    categorize();
  category_lock.release();
  return &categorymap;
}

static bool strictLessThan_Attributes(const std::vector<PixelAttribute*> &p0,
				      const std::vector<PixelAttribute*> &p1)
{
  std::size_t n0 = p0.size();
  std::size_t n1 = p1.size();
  std::size_t n = n0;
  if(n1 < n0) n = n1;
  for(std::size_t i=0; i<n; i++) {
    if(p0[i]->strictLessThan(*p1[i])) return true;
    if(p1[i]->strictLessThan(*p0[i])) return false;
  }
  if(n0 < n1) return true;
  return false;
}

// getCategoryMapRO() is a combination of getCategoryMap() and
// categorize().  It doesn't compute representative pixels or category
// boundaries, and it is strictly const -- it doesn't even change any
// mutable data in the CMicrostructure.  It uses
// strictLessThan_Attributes instead of operator< when comparing
// PixelAttributes.  It's used when writing a Microstructure to a file.

const Array<int> *CMicrostructure::getCategoryMapRO() const {
  groups_attributes_lock.read_acquire();
#if DIM == 2
  Array<int> *localmap = new Array<int>(pxlsize_(0), pxlsize_(1));
#elif DIM == 3
  Array<int> *localmap = new Array<int>(pxlsize_(0), pxlsize_(1), pxlsize_(2));
#endif
  CatMap catmap(strictLessThan_Attributes);
  int ncats = 0;
  std::size_t nattrs = attributeMap.size();
  // loop over pixels in the microstructure
  for(Array<int>::iterator i=localmap->begin(); i!=localmap->end(); ++i) {
    const ICoord &where = i.coord();
    // construct a list of the attributes of this pixel
    std::vector<PixelAttribute*> attrs(nattrs);
    for(std::size_t j=0; j<nattrs; j++)
      attrs[j] = attributeMap[j][where];
    // See if this list of attributes has been seen already
    CatMap::iterator cat = catmap.find(attrs);
    if(cat == catmap.end()) {	// category not found
      catmap[attrs] = ncats;
      *i = ncats;
      ncats++;
    }
    else {
      *i = (*cat).second; // assign previously found category
    }
  }
  groups_attributes_lock.read_release();
  return localmap;
}

void CMicrostructure::recategorize() {
  // std::cerr << "Acquire, recategorize." << std::endl;
  category_lock.acquire();
  categorized = false;
  ++timestamp;
  category_lock.release();
  // std::cerr << "Release." << std::endl;
}

//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//

static const ICoord east(1, 0);
static const ICoord west(-1, 0);
static const ICoord north(0, 1);
static const ICoord south(0, -1);
static const ICoord northeast(1, 1);
static const ICoord northwest(-1, 1);
static const ICoord southeast(1, -1);
static const ICoord southwest(-1, -1);


// Geometry routines for identifying pixels in the microstructure that
// are under segments and elements. 

// Return a list (vector) of pixels underlying a segment.  It's the
// responsibility of the caller to delete the vector.

std::vector<ICoord> CMicrostructure::segmentPixels(const Coord &c0,
						   const Coord &c1,
						   bool &vertical_horizontal,
						   bool &flipped)
  const
{
  // Coordinates of endpoints in pixel space (real).
  Coord p0(physical2Pixel(c0));
  Coord p1(physical2Pixel(c1));
  // Coordinates of pixels containing the endpoints (integer).  Note
  // that we *don't* use CMicrostructure::pixelFromPoint() here,
  // because we have to treat the pixel boundary cases differently.
  ICoord ip0((int) floor(p0(0)), (int) floor(p0(1)));
  ICoord ip1((int) floor(p1(0)), (int) floor(p1(1)));

  // If an endpoint lies exactly on a pixel boundary, then the correct
  // choice for the pixel "containing" the endpoint depends on which
  // direction the segment is going.  The rule is that some part of
  // the segment must lie within the selected pixel. 
  if(ip0(0) == p0(0) && ip1(0) < ip0(0))
    ip0(0) -= 1;
  if(ip1(0) == p1(0) && ip0(0) < ip1(0))
    ip1(0) -= 1;
  if(ip0(1) == p0(1) && ip1(1) < ip0(1))
    ip0(1) -= 1;
  if(ip1(1) == p1(1) && ip0(1) < ip1(1))
    ip1(1) -= 1;

  // Round off error may have put a point out of bounds.  Fix it.
  if(ip0(0) == pxlsize_(0)) ip0(0) -= 1;
  if(ip0(1) == pxlsize_(1)) ip0(1) -= 1;
  if(ip1(0) == pxlsize_(0)) ip1(0) -= 1;
  if(ip1(1) == pxlsize_(1)) ip1(1) -= 1;
  if(ip0(0) < 0) ip0(0) = 0;
  if(ip0(1) < 0) ip0(1) = 0;
  if(ip1(0) < 0) ip1(0) = 0;
  if(ip1(1) < 0) ip1(1) = 0;

  // For vertical and horizontal segments that exactly lie along
  // the pixel boundaries, we need to pick a right row or column of pixels.
  // For instance,
  //
  //  |    element B  |
  //  |xxxxxxxxxxxxxxx|
  //  b---------------a   segment a-b lies on the pixel boundary
  //  |ooooooooooooooo|
  //  |    element A  |
  //
  //  From element A's point of view, pixels along the segment a-b
  //  should be "ooooo", whereas from element B's p.o.v, corresponding
  //  pixels should be "xxxxx".
  //  Followings will deal with this adjustment.

  // The users of the pixel data may need to know if the order of the
  // pixels has been flipped.
  flipped = false;

  // This assumes we are traversing an element in counter clockwise
  // order.
  vertical_horizontal=false;
  // Horizontal
  if ((p0(1) == p1(1) && p0(1) == ip0(1)) && p0(0) > p1(0)) {
    if(ip0(1) >= 1) ip0(1) -= 1;
    if(ip1(1) >= 1) ip1(1) -= 1;
    vertical_horizontal=true;
  }
  // Vertical 
  if ((p0(0) == p1(0) && p0(0) == ip0(0)) && p0(1) < p1(1)) {
    if(ip0(0) >= 1) ip0(0) -= 1;
    if(ip1(0) >= 1) ip1(0) -= 1;
    vertical_horizontal=true;
  }

  // For vertical and horizontal segments on skeleton(microstructure)
  // edges, all that matters is that the chosen pixels aren't outside
  // the microstructure.  After the preceding check, the only way that
  // an end pixel can be outside is if the segment lies along the top
  // or right edges.
  int maxx = pxlsize_(0);
  int maxy = pxlsize_(1);
  if(ip0(0) == maxx && ip1(0) == maxx) {
    ip0(0) = maxx - 1;
    ip1(0) = maxx - 1;
  }
  if(ip0(1) == maxy && ip1(1) == maxy) {
    ip0(1) = maxy - 1;
    ip1(1) = maxy - 1;
  }

  ICoord id = ip1 - ip0;	// distance between pixel endpoints

  if(id(0) == 0 && id(1) == 0) { // segment is entirely within one pixel
    return std::vector<ICoord>(1, ip0);
  }
  
  if(id(0) == 0) {	
    // Segment is contained within a single column of pixels.
    int npix = abs(id(1)) + 1;
    std::vector<ICoord> pixels(npix);
    int x = ip0(0);
    int y0 = (ip0(1) < ip1(1)) ? ip0(1) : ip1(1);
    flipped = ip1(1) < ip0(1);
    for(int i=0; i<npix; i++)
      pixels[i] = ICoord(x, y0+i);
    return pixels;
  }
  
  if(id(1) == 0) {	
    // Segment is contained within a single row of pixels.
    int npix = abs(id(0)) + 1;
    std::vector<ICoord> pixels(npix);
    int y = ip0(1);
    int x0 = (ip0(0) < ip1(0)) ? ip0(0) : ip1(0);
    flipped = ip1(0) < ip0(0);
    for(int i=0; i<npix; i++)
      pixels[i] = ICoord(x0+i, y);
    return pixels;
  }

  // Segment is diagonal.

  std::vector<ICoord> pixels;

  // Is it bigger in the x or y direction?  That determines whether
  // we loop over rows or columns below.
  Coord pd = p1 - p0;
  // It's important to use pd and not id when deciding which range is
  // bigger!  If the integer ranges are equal but the real y range is
  // bigger, if we use id we can end up missing some pixels.
  if(fabs(pd(0)) >= fabs(pd(1))) {
    // x range is bigger than y range, so loop over x.
    // Make sure that p0 is to the left of p1.
    if(ip0(0) > ip1(0)) {
      ICoord itemp(ip1);
      ip1 = ip0;
      ip0 = itemp;
      Coord temp(p1);
      p1 = p0;
      p0 = temp;
      flipped = true;
    }
    
    pixels.reserve(2*abs(id(0))); // biggest possible size
    double x0 = p0(0);
    double y0 = p0(1);
    double slope = (p1(1) - y0)/(p1(0) - x0);
    pixels.push_back(ip0);
    // Iterate over columns of pixels.  Compute the y intercepts at
    // the boundaries between the columns (ie, integer values of x).
    // Whenever the integer part of the y intercept changes, we need
    // to include an extra pixel (at the new y value) in the previous
    // column.
    int lasty = ip0(1);	// previous integer part of y intercept
    for(int x=ip0(0)+1; x<=ip1(0); ++x) {
      int y = int(floor(y0 + slope*(x-x0)));
      if(y >=0 && y < maxy) {
	if(y != lasty)
	  pixels.push_back(ICoord(x-1, y));
	pixels.push_back(ICoord(x,y));
      }
      lasty = y;
    }
    // The last pixel may not have been included yet, if there's one
    // more y intercept change to come, so make sure it's included.
    if(pixels.back() != ip1)
      pixels.push_back(ip1);
  } // fabs(pd(0)) >= fabs(pd(1))
  else {
    // fabs(pd(1)) > fabs(pd(0))
    // y range is bigger than x range, so loop over y.
    if(ip0(1) > ip1(1)) {
      ICoord itemp(ip1);
      ip1 = ip0;
      ip0 = itemp;
      Coord temp(p1);
      p1 = p0;
      p0 = temp;
      flipped = true;
    }
    pixels.reserve(2*abs(id(1)));
    double x0 = p0(0);
    double y0 = p0(1);
    double slope = (p1(0) - x0)/(p1(1) - y0); // dx/dy
    pixels.push_back(ip0);
    int lastx = ip0(0);
    // iterate over rows of pixels
    for(int y=ip0(1)+1; y<=ip1(1); ++y) {
      int x = int(floor(x0 + slope*(y-y0)));
      if(x >= 0 && x < maxx) {
	if(x != lastx)
	  pixels.push_back(ICoord(x, y-1));
	pixels.push_back(ICoord(x,y));
      }
      lastx = x;
    }
    if(pixels.back() != ip1)
      pixels.push_back(ip1);
  }
// #ifdef DEBUG
//   // Check that the pixels abut each other properly
//   for(unsigned int i=1; i<pixels.size(); i++) {
//     ICoord diff = pixels[i] - pixels[i-1];
//     if(diff != east && diff != west && diff != north && diff != south) {
//       std::cerr << "CMicrostructure::segmentPixels: p0=" << p0
// 		<< " p1=" << p1 << std::endl;
//       std::cerr << "CMicrostructure::segmentPixels: ip0=" << ip0
// 		<< " ip1=" << ip1 << std::endl;
//       std::cerr << "CMicrostructure::segmentPixels: pixels=" << *pixels
// 		<< std::endl;
//       throw ErrProgrammingError("Error in CMicrostructure::segmentPixels",
// 				__FILE__, __LINE__);
//     }
//   }
// #endif	// DEBUG
  return pixels;
} // end CMicrostructure::segmentPixels

MarkInfo::MarkInfo(const ICoord &size)
  : markedpixels(size, false)
{
  // Initial setting of markedregion will be overwritten... why is it done?
  markedregion = markedpixels.subarray(ICoord(0,0), markedpixels.size());
}


// Set the bounds of the marking region and clear it.  The
// markedpixels array is an array of booleans, used as an intermediate
// result when marking complicated things like elements.

MarkInfo *CMicrostructure::beginMarking(const CRectangle &bbox) const {
  MarkInfo *mm = new MarkInfo(sizeInPixels());
  ICoord p0 = pixelFromPoint(bbox.lowerleft());
  ICoord p1 = pixelFromPoint(bbox.upperright()) + northeast;
  if(p1(0) >= pxlsize_(0)) p1(0) = pxlsize_(0);
  if(p1(1) >= pxlsize_(1)) p1(1) = pxlsize_(1);
  mm->markedregion = mm->markedpixels.subarray(p0, p1);
  mm->markedregion.clear(false);
  return mm;
}

// Return a vector containing all the marked pixels.  It's the
// responsibility of the caller to delete the vector.

std::vector<ICoord> *CMicrostructure::markedPixels(MarkInfo *mm) const {
  return mm->markedregion.pixels(true); // returns new'd vector
}

// Mark the pixels underlying a segment.

void CMicrostructure::markSegment(MarkInfo *mm, 
				  const Coord &c0, const Coord &c1) const
{
  bool dummy;
  const std::vector<ICoord> pixels(segmentPixels(c0, c1, dummy, dummy));
  for(const ICoord &pxl : pixels) {
    mm->markedregion.set(pxl);
  }
}


// Mark the pixels under a triangle by marking the pixels under its
// edges, then using a burn algorithm to mark the ones inside.

void CMicrostructure::markTriangle(MarkInfo *mm, const Coord &c0,
				   const Coord &c1, const Coord &c2)
  const
{
  markSegment(mm, c0, c1);
  markSegment(mm, c1, c2);
  markSegment(mm, c2, c0);
  // Find an unmarked point at which to start the burn algorithm.  For
  // small or narrow triangles, the center point may lie within a
  // pixel's distance of the edge and be already marked.
  ICoord start = pixelFromPoint(1./3.*(c0 + c1 + c2));
  if(mm->markedregion[start]) {
    // Center is already marked.  Unmarked pixels, if any, will lie
    // near the triangle's shortest edge.  Look for an unmarked pixel
    // by starting at the midpoint of the shortest edge and moving
    // towards the center of the triangle.  This code is really ugly
    // since it hardly seems worthwhile to put c0,c1,c2 in an array
    // and iterate over them.
    int shortest = 0;
    double shortlen = norm2(c1 - c2);
    double dd = norm2(c2 - c0);
    if(dd < shortlen) {
      shortlen = dd;
      shortest = 1;
    }
    dd = norm2(c0 - c1);
    if(dd < shortlen) {
      shortest = 2;
    }
    Coord mid;			// midpoint of shortest side
    Coord r;			// unit vector from center of short
				// side toward the opposite corner.
    switch (shortest) {
    case 0:
      mid = 0.5*(c1 + c2);
      r = c0 - mid;
      break;
    case 1:
      mid = 0.5*(c2 + c0);
      r = c1 - mid;
      break;
    case 2:
      mid = 0.5*(c0 + c1);
      r = c2 - mid;
      break;
    }
    double normr = sqrt(norm2(r));
    double maxdist = normr/3.0;	// no point in moving beyond center
    r = (1./normr)*r;		// unit vector
    double d = 1.0;
    bool ok = false;		// found a starting point yet?
    while(d < maxdist && !ok) {
      start = pixelFromPoint(mid + d*r);
      if(!mm->markedregion[start])
	ok = true;
      d += 1.0;
    }
    if(!ok) return;		// all interior pixels must already be marked
  }
  // Burn algorithm.  If a site is unmarked, the mark_site routine
  // marks it and puts it on the activesites list.  The main loop
  // (here) removes sites from the list and calls mark_site on their
  // neighbors.  Since the edges of the triangle are already marked,
  // they don't become active, and the burn won't spread beyond the
  // edges.
  std::vector<ICoord> activesites; // sites whose neighbors need to be checked
  mm->mark_site(activesites, start); // come on baby, light my fire
  while(activesites.size() > 0) {
    const ICoord here = activesites.back();
    activesites.pop_back();
    mm->mark_site(activesites, here + east);
    mm->mark_site(activesites, here + west);
    mm->mark_site(activesites, here + north);
    mm->mark_site(activesites, here + south);
    mm->mark_site(activesites, here + northeast);
    mm->mark_site(activesites, here + northwest);
    mm->mark_site(activesites, here + southeast);
    mm->mark_site(activesites, here + southwest);
  }
}

void MarkInfo::mark_site(std::vector<ICoord> &activesites,
			     const ICoord &here)
{
  if(markedregion.contains(here) && !markedregion[here]) {
    markedregion.set(here);
    activesites.push_back(here);
  }
}

void CMicrostructure::endMarking(MarkInfo *mm) const {
  delete mm;
}

//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//

// Utility functions used by CMicrostructure::getSegmentSections()

// #ifdef DEBUG
// double SegmentSection::span(int comp) const {
//   return fabs((p1[comp] - p0[comp]));
// }
// #endif // DEBUG

double SegmentSection::pixelLength() const {
  return sqrt(norm2(p1 - p0));
}

double SegmentSection::physicalLength() const {
  return sqrt(norm2(ms->pixel2Physical(p1 - p0)));
}

Coord SegmentSection::physicalPt0() const {
  return ms->pixel2Physical(p0);
}

Coord SegmentSection::physicalPt1() const {
  return ms->pixel2Physical(p1);
}
  

std::ostream &operator<<(std::ostream &os, const SegmentSection &s) {
  return os << "SegmentSection(" << s.physicalPt0() << ", " << s.physicalPt1()
	    << ", " << s.category << ", " << s.stepcategory << ")"
	    << " length=" << sqrt(norm2(s.physicalPt0() - s.physicalPt1()));
}

// Find the distance in the comp direction spanned by two adjacent sections.
static double spantwo(const SegmentSection &s0, const SegmentSection &s1,
		      int comp)
{
  if(s0.physicalPt0() == s1.physicalPt1()) {
    return fabs(s0.physicalPt1()[comp] - s1.physicalPt0()[comp]);
  }
  assert(s0.physicalPt1() == s1.physicalPt0());
  return fabs(s0.physicalPt0()[comp] - s1.physicalPt1()[comp]);
}

bool isint(double x) {
  return floor(x) == x;
}

int getInitialCat(const CMicrostructure *ms, const Coord &pt0, const Coord &pt1)
{
  // Get the category to use at the beginning of the segment that goes
  // from pt0 to pt1 (in pixel units).  It's not just category(pt0)
  // because that can give the wrong answer if the point lies on a
  // pixel boundary and the segment goes to the left or down.
  double x0 = pt0[0];
  double y0 = pt0[1];
  if(isint(x0) && (x0 > pt1[0] || (x0==pt1[0] && y0 < pt1[1])))
    // going left from a pixel boundary or vertically up along a boundary
    x0 -= 0.5;
  if(isint(y0) && (y0 >= pt1[1] || (y0 == pt1[1] && x0 > pt1[0])))
    // going down from a pixel boundary or horizontally left along a boundary
    y0 -= 0.5;
  // We can't just call category(x0, y0) because that assumes the
  // coordinates are physical, not pixel.  That means that we have to
  // check for x and y on the left and upper boundaries of the
  // microstructure too, the way that pixelFromPoint does.
  ICoord sz = ms->sizeInPixels();
  if(x0 >= sz[0]) x0 -= 1;
  if(y0 >= sz[1]) y0 -= 1;
  int cat = ms->category(ICoord(x0, y0));
  return cat;
}

// Given a vector of SegmentSections, compute the sum of their lengths
// and the dominant category.  indices indicates which sections to
// include.

void CMicrostructure::segmentCats(const std::vector<SegmentSection> &sections,
				  const std::vector<int> &indices,
				  double &totallength, int &category)
  const
{
  std::vector<double> catlength(ncategories, 0);
  for(int j=0; j<indices.size(); j++) {
    const SegmentSection &segj = sections[indices[j]];
    catlength[segj.category] += segj.pixelLength(); 
  }
  // Find which category dominates the short sections, and their total
  // length.
  double maxlen = -1;
  category = -1;
  totallength = 0.0;
  for(int c=0; c<ncategories; c++) {
    totallength += catlength[c];
    if(catlength[c] > maxlen) {
      category = c;
      maxlen = catlength[c];
    }
  }
}

//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//

// Split a directed line segment (c0, c1) into sections such that each
// section lies over pixels with a single category.  If a stairstep is
// detected, treat it as a single section. Sections less than
// minlength are merged with their neighbors if both neighbors are the
// same category, or at the end of the segment. Return of list of
// SegmentSection objects, which give the end points and category each
// section.

std::vector<SegmentSection*>* CMicrostructure::getSegmentSections(
				  const Coord *c0, const Coord *c1,
				  double minlength)
  const
{
  // Get the pixels under the segment.
  bool flipped, dummy;

  std::vector<SegmentSection> sections;
  std::vector<ICoord> pixels(segmentPixels(*c0, *c1, dummy, flipped));

  // Pixel-space coordinates of the ends of segment
  Coord pt0(physical2Pixel(*c0));
  Coord pt1(physical2Pixel(*c1));

  bool vertical = (pt0[0] == pt1[0]);
  double slope = 0.0;		// Only one of slope and invslope is used.
  double invslope = 0.0;
  if(!vertical) {
    slope = (pt1[1] - pt0[1])/(pt1[0] - pt0[0]);
    invslope = 1./slope;
  }

  // segmentPixels doesn't guarantee that the pixels are in order from
  // c0 to c1.  If flipped is true, the pixels are in reverse order,
  // so we iterate over them manually.
  int npxls = pixels.size();
  int i0 = flipped ? npxls-1 : 0; // initial pixel index
  int i1 = flipped ? -1 : npxls;  // final pixel index
  int idir = flipped ? -1 : 1;	  // pixel index increment

  // Find the category of the starting point.  This is not just
  // category(pt0), because if the segment starts on a boundary going
  // left or down, we want the pixel to the left or below the starting
  // point.
  int prevcat = getInitialCat(this, pt0, pt1);
  Coord curpt =  pt0;	      // starting point of the current section
  ICoord prevpxl = pixels[i0]; // previous pixel
  // The step category is the category to assign to the segment if it
  // is part of a stairstep boundary.  This is determined by how the
  // segment ends meet the category boundary, and is not necessarily
  // the same as the segment category.
  int stepcat = -1;

  // Loop over the pixels under the segment, looking for category
  // changes.  Start at the second pixel (i0+idir) because we're
  // looking for endpoints of sections.

  for(int i=i0+idir; i!=i1; i+=idir) {
    ICoord pxl = pixels[i];	// the pixel to examine
    int cat = category(pxl);
    Coord pt;			// next transition point
    if(cat != prevcat) {
      // Find the actual intersection point of the segment and the
      // previous pixel, in the direction of pt1.
      ICoord delta = pxl - prevpxl; // Will be either x, -x, y, or -y
      if(delta[0] == 1) {
	// We've moved one pixel in the +x direction from the last
	// pixel.  The transition point is on the left side of this
	// pixel, x = pxl[0]
	assert(delta[1] == 0 && !vertical);
	pt = Coord(pxl[0],  pt0[1] + slope*(pxl[0] - pt0[0]));
	if(slope > 0) {
	  //           1
	  //   +----+-/--+
	  //   |....|/   |   Shaded pixel is prevpxl.
	  //   |..../    |   Segment goes in the 01 direction
	  //   |.../|    |
	  //   +--/-+----+	  
	  //     0
	  stepcat = prevcat;
	}
	else {
	  //     0       
	  //   +--\-+----+
	  //   |...\|    |  
	  //   |....\    |
	  //   |....|\   |
	  //   +----+-\--+	  
	  //           1
	  stepcat = cat;
	}
      }
      else if(delta[0] == -1) {
	// Moving left.  The transition point is on the left side of
	// the previous pixel, x = prevpxl[0].
	assert(delta[1] == 0 && !vertical);
	pt = Coord(prevpxl[0], pt0[1] + slope*(prevpxl[0] - pt0[0]));
	if(slope > 0) {
	  //           0
	  //   +----+-/--+
	  //   |    |/...| 
	  //   |    /....| 
	  //   |   /|....|
	  //   +--/-+----+	  
	  //     1
	  stepcat = prevcat;
	}
	else {
	  //     1       
	  //   +--\-+----+
	  //   |   \|....|  
	  //   |    \....|
	  //   |    |\...|
	  //   +----+-\--+	  
	  //           0
	  stepcat = cat;
	}
      }
      else if(delta[1] == 1) {
	// Moving up one from the previous pixel.  Intersection is at
	// the bottom of this pixel, y=pxl[1]
	assert(delta[0] == 0);
	if(vertical) {
	  pt = Coord(pt0[0], pxl[1]);
	  stepcat = -1;
	}
	else {
	  pt = Coord(pt0[0] + invslope*(pxl[1] - pt0[1]), pxl[1]);
	  if(slope > 0) {
	    //    +----+ /1 	 
	    //    |    |/	 	 
	    //    |    /	 	 
	    //    |   /|	 	 
	    //    +--/-+	 	 
	    //    |./..|	 	 
	    //    |/...|	 	 
	    //    /....|	 	 
	    //  0/+----+
	    stepcat = cat;
	  }
	  else {
	    //  1\+----+    	 
	    //    \    | 	 	 
	    //    |\   |	 	 
	    //    | \  |	 	 
	    //    +--\-+	 	 
	    //    |...\|	 	 
	    //    |....\      Segment goes in
	    //    |....|\     the 01 direction
	    //    +----+ 0	
	    stepcat = prevcat;
	  }
	}
      }
      else if(delta[1] == -1) {
	// Moving down one from the previous pixel. Intersection is at
	// the top of this pixel, y = prevpxl[1].
	assert(delta[0] == 0);
	if(vertical)
	  pt = Coord(pt0[0], prevpxl[1]);
	else {
	  pt = Coord(pt0[0] + invslope*(prevpxl[1] - pt0[1]), prevpxl[1]);
	  if(slope > 0) {
	    //    +----+ /0 	 
	    //    |....|/	 	 
	    //    |..../	 	 
	    //    |.../|	 	 
	    //    +--/-+	 	 
	    //    | /  |	 	 
	    //    |/   |	 	 
	    //    /    |	 	 
	    //  1/+----+
	    stepcat = cat;
	  }
	  else {
	    //  0\+----+    	 
	    //    \....| 	 	 
	    //    |\...|	 	 
	    //    |.\..|	 	 
	    //    +--\-+	 	 
	    //    |   \|	 	 
	    //    |    \      Segment goes in
	    //    |    |\     the 01 direction
	    //    +----+ 1	
	    stepcat = prevcat;
	  }
	}
      }
      else {
	throw ErrProgrammingError("Error in getSegmentSections",
				  __FILE__, __LINE__);
      }
      sections.emplace_back(this, curpt, pt, prevcat, stepcat);
      curpt = pt;
      prevcat = cat;
    } // end if cat != prevcat
    prevpxl = pxl;
  } // end loop over segment pixels

  // Add the final point.  There can be no more transition points.  If
  // this section is part of a stairstep, its category must be the
  // same as the previous section's stairstep category.
  if(sections.size() > 0) {
    stepcat = sections.back().stepcategory;
    sections.emplace_back(this, curpt, pt1, prevcat, stepcat);
  }
  else {
    // There were no transition points found, so there can be no
    // stairsteps.  The value of stepcat is irrelevant.
    sections.emplace_back(this, pt0, pt1, prevcat, -1);
  }

  ICoord totalspan = pixels[npxls-1] - pixels[0];

  int nsections = sections.size();
  // adaSections are ones where steps have been replaced with
  // ramps. Since this is an intermediate result and won't outlive
  // this routine or be exported to Python, it can be vector of
  // SegmentSections, not a pointer to a vector of pointers to
  // SegmentSections.
  std::vector<SegmentSection> adaSections; 

  // There are no stairsteps if the entire segment is horizontal or
  // vertical or if there aren't enough sections.
  // TODO PYTHON3: Should we eliminate stairsteps of only two
  // sections?  It would probably be best to treat that as a special
  // case.

  if(nsections <= 2 || totalspan[0] == 0 || totalspan[1] == 0) {
    // If there are no steps, there still might be short segment to be
    // eliminated.  The code that does that, after the stairstep
    // elimination, expects to get a vector of SegmentSections, not
    // pointers, so we have to copy the list here.
    for(SegmentSection &seg : sections) 
      adaSections.push_back(seg);
  }
  else {
    // Eliminate stairstep sections.  A stairstep is a set of sections
    // that alternate categories (A,B,A,[B...]), and in which
    // transition points occupy rows or columns next to the second
    // previous point.

    // ....|....1    |    |    |
    // ....|....|\   |    |    |     Sections (1,2) and (3,4) have category A
    // ....|....| \  |    |    |     and sections (2,3) and (4,5) have
    // ----+----+--2-+----+----+-    category B.
    // ....|....|...\|    |    |  
    // ....|....|....3    |    |     The whole section from 1 to 5 should
    // ....|.B..|....|\   |  A |     be assigned category A, because A
    // ----+----+----+-4--+----+     is on the left.
    // ....|....|....|..\.|    | 
    // ....|....|....|...\|    |     If the segment went the other way, the
    // ....|....|....|....5....+     section from 5 to 1 would be category B.
  
    // 1 and 3 are exactly one pixel apart horizontally, and would be
    // even if the boundary were steeper.  2 and 4 are exactly one
    // pixel apart vertically, and would be even if the boundary were
    // flatter.  (The pixel spacing is exactly 1.0 because of the way
    // that the intersections were calculated.)

    // If the horizontal distance between steps (the tread) is larger
    // than the vertical distance (the riser) in pixel units, then the
    // riser will be 1.0, and the sum of the y components of two
    // adjacent sections will be 1.0.  If the riser is larger than the
    // tread, then the sum of the x components will be one.  Since the
    // angle is the same across the whole segment, the segment
    // geometry tells whether to use the x or y components of the
    // lengths of the sections.
    Coord diff = pt1 - pt0;
    int comp = fabs(diff[0]) > fabs(diff[1]) ? 1 : 0;


    int startstep = 0;	 // index of first potential stairstep section
    int sec = 0;	 // index of section we're examining

    // Loop over sections, looking for stairsteps.  Each iteration
    // skips sections that don't have the right size in the "comp"
    // direction, and then combines the following sections into a
    // single new section.  The comp-sizes of the sections must add
    // pairwise to 1, starting with an integer comp-component, and the
    // sections must alternate between two categories.
    do {
      // We're either at the beginning of the step search, or have just
      // finished a set of stairs.  If the next sections don't have the
      // right lengths, or if we're too close to the end of the section
      // list to form a stairstep, just copy the sections to the result.

      while(sec < nsections &&
	    !(sec < nsections-2 &&
	      isint(sections[sec].p0[comp]) &&
	      spantwo(sections[sec], sections[sec+1], comp) == 1.0))
	{
	  adaSections.push_back(SegmentSection(sections[sec]));
	  sec++;			// Go on to the next section.
	}

      if(sec < nsections-2) {
	// The comp-components of this section and the next sum to
	// one. Search for the end section.  Pairs of sections need to sum
	// to 1 (in the comp direction) and their categories must
	// alternate between two values.
	int cat0 = sections[sec].category;
	int cat1 = sections[sec+1].category;
	int nextpair = sec + 2;
	while(nextpair < nsections-2 &&
	      isint(sections[nextpair+1].p1[comp]) &&
	      sections[nextpair].category == cat0 &&
	      sections[nextpair+1].category == cat1 &&
	      spantwo(sections[nextpair], sections[nextpair+1],comp)==1.0)
	  {
	    nextpair += 2;
	  }
    
	if(nextpair == sec + 2) {
	  // We didn't find a step.  Add the two sections to the output
	  // and look again.
	  adaSections.push_back(sections[sec]);
	  adaSections.push_back(sections[sec+1]);
	}
	else {
	  // The sections from sec through nextpair-1 can be combined.  The
	  // category is the stepcategory of any of the sections.
	
	  adaSections.emplace_back(this,
				   sections[sec].p0,
				   sections[nextpair-1].p1,
				   sections[sec].stepcategory,
				   sections[sec].stepcategory);
	}
	sec = nextpair;     // potential start of next stairstep
      }

    } while(sec < nsections);
  } // end if there might have been stair steps

  // Take one more pass through the new sections to eliminate ones
  // that are shorter than minlength, and to see if the segments
  // immediately before or after a stairstep can be joined to it.
  // This can happen if a stair starts right after a long segment with
  // the same category as the first step on the stair.  Copy the
  // sections to the result vector.

  nsections = adaSections.size();
  std::vector<SegmentSection*> *result = new std::vector<SegmentSection*>;
  result->reserve(nsections);
  double totallength = 0;
  int maxcat = -1;     // dominant category of a set of short sections
  // Consecutive short sections are accumulated until a non-short
  // section is found. shortSections contains the indices (in
  // adaSections) of the current set of short sections.
  std::vector<int> shortSections;
  for(int i=0; i<adaSections.size(); i++) {
    // lastcat is the category of the pixels under the previously
    // found section.  Its default value, -1, is never a valid
    // category.
    int lastcat = result->empty() ? -1 : result->back()->category;
    SegmentSection &section = adaSections[i];
    if(section.pixelLength() < minlength) {
      // This section is too short.  Store it and deal with it later.
      shortSections.push_back(i);
    }
    else {
      // This section is not too short.  Is it the first one after a
      // stretch of one or more short sections?
      if(shortSections.empty()) { // Previous sections were not short.
	if(section.category == lastcat) {
	  // This section is the same category as the previous one.
	  // Extend previous section.
	  result->back()->p1 = section.p1;
	}
	else {
	  result->push_back(new SegmentSection(section));
	}
      }	// end if the previous segment was not short
      else {
	// This section is the first non-short one after a sequence of
	// short ones.  Combine all the short ones and decide what to
	// do with them.

	// segmentCats returns the total length of the short sections
	// and the dominant pixel category.
	segmentCats(adaSections, shortSections, totallength, maxcat);

	if((maxcat == lastcat && maxcat == section.category) ||
	   (totallength < minlength && lastcat == section.category)) {
	  // The combined short sections are the same category as the
	  // current and previous long sections, so merge all of them.
	  // This can't happen when result is empty, because lastcat
	  // would be -1 then.
	  result->back()->p1 = section.p1;
	}
	else if(maxcat == lastcat) {
	  // Absorb the short sections into the previous section and
	  // add the current section.  This also can't happen when
	  // result is empty.
	  result->back()->p1 = adaSections[shortSections.back()].p1;
	  result->push_back(new SegmentSection(section));
	}
	else if(maxcat == section.category) {
	  // Create a new section from the short ones and the current one.
	  result->push_back(new SegmentSection(section));
	  result->back()->p0 = adaSections[shortSections.front()].p0;
	}
	else if(totallength < minlength) {
	  // the short sections don't match the category of either of
	  // the long sections on either side, but are too short to
	  // include.  Split them between the two long sections.
	  Coord midpoint;
	  if(result->size() > 0) {
	    midpoint = 0.5*(result->back()->p1 + section.p0);
	    result->back()->p1 = midpoint;
	  }
	  else {
	    midpoint = adaSections[shortSections.front()].p0;
	  }
	  result->push_back(new SegmentSection(section));
	  result->back()->p0 = midpoint;
	}
	else {
	  // The assembled short sections are long enough to form
	  // their own section, but don't match the category of the
	  // long sections on either side.  Insert it and the new
	  // section.
	  // std::cerr << "CMicrostructure::getSegmentSections:"
	  // 	    << " combined short sections are long enough" << std::endl;
	  result->push_back(new SegmentSection(this,
				       adaSections[shortSections.front()].p0,
				       adaSections[shortSections.back()].p1,
				       maxcat, maxcat));
	  result->push_back(new SegmentSection(section));
	}
      }	// end if the previous section was short but this one isn't
      
      // After processing the long section and maybe incorporating
      // short ones, get ready for the next batch of short ones.
      shortSections.clear();
      totallength = 0.0;
      maxcat = -1;
    } // end this section was not short
  }   // end loop over sections

  // If any short sections are left, they are at the end of the whole
  // segment.
  if(!shortSections.empty()) {
    double totallength;
    int maxcat;
    segmentCats(adaSections, shortSections, totallength, maxcat);
    if(result->empty() ||	   // no previous sections
       (totallength > minlength && // or sum of lengths isn't still too short
	maxcat != result->back()->category)) // and category doesn't match
      {
	// Make a new section from the total set of short sections,
	// and append it.
	result->push_back(new SegmentSection(this,
			     adaSections[shortSections.front()].p0,
			     adaSections[shortSections.back()].p1,
					   maxcat,
					   maxcat));
    }
    else {
      // absorb short lengths into the last part of result.
      result->back()->p1 = adaSections[shortSections.back()].p1;
      }
    } // end if there are shortSections left over
    
  return result;
} // end CMicrostructure::getSegmentSections

//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//=\\=//

double CMicrostructure::edgeHomogeneity(const Coord &c0, const Coord &c1) const
{
  std::vector<SegmentSection*> *sections = getSegmentSections(&c0, &c1, 0);
  if(sections->size() == 1)
    return 1.0;
  std::vector<double> slengths(nCategories(), 0.0);
  for(SegmentSection *s : *sections) {
    slengths[s->category] += s->physicalLength();
    delete s;
  }
  delete sections;

  double lmax = 0.0;
  for(double length : slengths)
    if(lmax < length)
      lmax = length;

  return lmax/sqrt(norm2(c1 - c0));
}



