//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

// <iterator>

// move_iterator

// requires RandomAccessIterator<Iter>
//   unspecified operator[](difference_type n) const;
//
//  constexpr in C++17

#include "support/test_config.h"

#include <oneapi/dpl/iterator>
#include <oneapi/dpl/type_traits>

#include "support/test_iterators.h"
#include "support/utils.h"

template <class It>
bool
test(It i, typename dpl::iterator_traits<It>::difference_type n, typename dpl::iterator_traits<It>::value_type x)
{
    typedef typename dpl::iterator_traits<It>::value_type value_type;
    const dpl::move_iterator<It> r(i);
    value_type rr = r[n];
    return (rr == x);
}

struct do_nothing
{
    void
    operator()(void*) const
    {
    }
};

bool
kernel_test()
{
    sycl::queue deviceQueue = TestUtils::get_test_queue();
    bool ret = true;
    {
        sycl::range<1> numOfItems{1};
        sycl::buffer<bool, 1> buffer1(&ret, numOfItems);
        deviceQueue.submit([&](sycl::handler& cgh) {
            auto ret_access = buffer1.get_access<sycl::access::mode::write>(cgh);
            cgh.single_task<class KernelTest>([=]() {
                {
                    char s[] = "1234567890";
                    ret_access[0] &= test(random_access_iterator<char*>(s + 5), 4, '0');
                    ret_access[0] &= test(s + 5, 4, '0');
                }

                {
                    constexpr const char* p = "123456789";
                    typedef dpl::move_iterator<const char*> MI;
                    constexpr MI it1 = dpl::make_move_iterator(p);
                    static_assert(it1[0] == '1');
                    static_assert(it1[5] == '6');
                }
            });
        });
    }
    return ret;
}

int
main()
{
    auto ret = kernel_test();
    EXPECT_TRUE(ret, "Wrong result of move_iterator check with operator[] in kernel_test()");

    return TestUtils::done();
}
