/*******************************************************************************
* Copyright 2019-2020 Intel Corporation
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*******************************************************************************/

#ifndef CPU_JIT_AVX512_CORE_BF16_CONVOLUTION_HPP
#define CPU_JIT_AVX512_CORE_BF16_CONVOLUTION_HPP

#include "c_types_map.hpp"
#include "dnnl_thread.hpp"
#include "memory_tracking.hpp"
#include "utils.hpp"

#include "cpu_barrier.hpp"
#include "cpu_convolution_pd.hpp"
#include "cpu_reducer.hpp"

#include "jit_avx512_core_bf16_conv_kernel.hpp"
#include "jit_transpose_src_utils.hpp"

namespace dnnl {
namespace impl {
namespace cpu {

struct jit_avx512_core_bf16_convolution_fwd_t : public primitive_impl_t {
    struct pd_t : public cpu_convolution_fwd_pd_t {
        pd_t(engine_t *engine, const convolution_desc_t *adesc,
                const primitive_attr_t *attr,
                const typename pd_t::base_class *hint_fwd_pd)
            : cpu_convolution_fwd_pd_t(engine, adesc, attr, hint_fwd_pd)
            , jcp_() {}

        DECLARE_COMMON_PD_T(JIT_IMPL_NAME_HELPER("jit_bf16:", jcp_.isa, ""),
                jit_avx512_core_bf16_convolution_fwd_t);

        status_t init() {
            bool ok = true && mayiuse(avx512_core) && is_fwd()
                    && set_default_alg_kind(alg_kind::convolution_direct)
                    && (expect_data_types(data_type::bf16, data_type::bf16,
                                data_type::undef, data_type::bf16,
                                data_type::undef)
                            || expect_data_types(data_type::bf16,
                                    data_type::bf16, data_type::undef,
                                    data_type::f32, data_type::undef))
                    && IMPLICATION(with_bias(),
                            utils::one_of(weights_md(1)->data_type,
                                    data_type::f32, data_type::bf16))
                    && attr()->has_default_values(
                            primitive_attr_t::skip_mask_t::post_ops)
                    && !has_zero_dim_memory();
            if (!ok) return status::unimplemented;

            status_t status = jit_avx512_core_bf16_fwd_kernel::init_conf(jcp_,
                    *desc(), src_md_, weights_md_, dst_md_, bias_md_, *attr(),
                    dnnl_get_max_threads());
            if (status != status::success) return status::unimplemented;

            auto scratchpad = scratchpad_registry().registrar();
            jit_avx512_core_bf16_fwd_kernel::init_scratchpad(scratchpad, jcp_);

            return status::success;
        }

        jit_conv_conf_t jcp_;
    };

    jit_avx512_core_bf16_convolution_fwd_t(const pd_t *apd)
        : primitive_impl_t(apd) {
        kernel_ = new jit_avx512_core_bf16_fwd_kernel(
                pd()->jcp_, *pd()->attr());
    }
    ~jit_avx512_core_bf16_convolution_fwd_t() { delete kernel_; }

    typedef typename prec_traits<data_type::bf16>::type src_data_t;
    typedef typename prec_traits<data_type::bf16>::type wei_data_t;

    virtual status_t execute(const exec_ctx_t &ctx) const override {
        if (pd()->ndims() == 3)
            execute_forward_1d(ctx);
        else if (pd()->ndims() == 4)
            execute_forward_2d(ctx);
        else if (pd()->ndims() == 5)
            execute_forward_3d(ctx);
        else
            return status::unimplemented;

        if (pd()->wants_zero_pad_dst()) ctx.memory(DNNL_ARG_DST)->zero_pad();

        return status::success;
    }

private:
    void prepare_padded_bias(const char *&bias,
            const memory_tracking::grantor_t &scratchpad) const;
    void execute_forward_1d(const exec_ctx_t &ctx) const;
    void execute_forward_2d(const exec_ctx_t &ctx) const;
    void execute_forward_3d(const exec_ctx_t &ctx) const;
    const pd_t *pd() const { return (const pd_t *)primitive_impl_t::pd(); }

    jit_avx512_core_bf16_fwd_kernel *kernel_;
};

struct jit_avx512_core_bf16_convolution_bwd_data_t : public primitive_impl_t {
    struct pd_t : public cpu_convolution_bwd_data_pd_t {
        pd_t(engine_t *engine, const convolution_desc_t *adesc,
                const primitive_attr_t *attr,
                const convolution_fwd_pd_t *hint_fwd_pd)
            : cpu_convolution_bwd_data_pd_t(engine, adesc, attr, hint_fwd_pd)
            , jcp_() {}

        DECLARE_COMMON_PD_T(JIT_IMPL_NAME_HELPER("jit_bf16:", jcp_.isa, ""),
                jit_avx512_core_bf16_convolution_bwd_data_t);

        status_t init() {
            using namespace prop_kind;
            bool ok = true && mayiuse(avx512_core) && is_bwd_d()
                    && set_default_alg_kind(alg_kind::convolution_direct)
                    && (expect_data_types(data_type::f32, data_type::bf16,
                                data_type::undef, data_type::bf16,
                                data_type::undef)
                            || expect_data_types(data_type::bf16,
                                    data_type::bf16, data_type::undef,
                                    data_type::bf16, data_type::undef))
                    && attr()->has_default_values() && !has_zero_dim_memory();
            if (!ok) return status::unimplemented;

            status_t status = jit_avx512_core_bf16_bwd_data_kernel::init_conf(
                    jcp_, *desc(), diff_src_md_, weights_md_, diff_dst_md_,
                    dnnl_get_max_threads());
            return status;
        }

        jit_conv_conf_t jcp_;
    };

    jit_avx512_core_bf16_convolution_bwd_data_t(const pd_t *apd)
        : primitive_impl_t(apd) {
        kernel_ = new jit_avx512_core_bf16_bwd_data_kernel(pd()->jcp_);
    }
    ~jit_avx512_core_bf16_convolution_bwd_data_t() { delete kernel_; };

    typedef typename prec_traits<data_type::bf16>::type diff_dst_data_t;
    typedef typename prec_traits<data_type::bf16>::type wei_data_t;

    virtual status_t execute(const exec_ctx_t &ctx) const override {
        if (pd()->ndims() < 5)
            execute_backward_data(ctx);
        else if (pd()->ndims() == 5)
            execute_backward_data_3d(ctx);
        else
            assert(!"invalid dimension");

        return status::success;
    }

private:
    void execute_backward_data(const exec_ctx_t &ctx) const;
    void execute_backward_data_3d(const exec_ctx_t &ctx) const;
    const pd_t *pd() const { return (const pd_t *)primitive_impl_t::pd(); }
    jit_avx512_core_bf16_bwd_data_kernel *kernel_;
};

struct jit_avx512_core_bf16_convolution_bwd_weights_t
    : public primitive_impl_t {
    struct pd_t : public cpu_convolution_bwd_weights_pd_t {
        pd_t(engine_t *engine, const convolution_desc_t *adesc,
                const primitive_attr_t *attr,
                const convolution_fwd_pd_t *hint_fwd_pd)
            : cpu_convolution_bwd_weights_pd_t(engine, adesc, attr, hint_fwd_pd)
            , jcp_() {}

        DECLARE_COMMON_PD_T(JIT_IMPL_NAME_HELPER("jit_bf16:", jcp_.isa, ""),
                jit_avx512_core_bf16_convolution_bwd_weights_t);

        status_t init() {
            bool ok = true && mayiuse(avx512_core) && is_bwd_w()
                    && set_default_alg_kind(alg_kind::convolution_direct)
                    && (expect_data_types(data_type::bf16, data_type::bf16,
                                data_type::undef, data_type::bf16,
                                data_type::undef)
                            || expect_data_types(data_type::bf16,
                                    data_type::f32, data_type::undef,
                                    data_type::bf16, data_type::undef))
                    && IMPLICATION(with_bias(),
                            utils::one_of(diff_bias_md_.data_type,
                                    data_type::f32, data_type::bf16))
                    && attr()->has_default_values() && !has_zero_dim_memory();
            if (!ok) return status::unimplemented;

            status_t status = jit_avx512_core_bf16_conv_bwd_weights_kernel_f32::
                    init_conf(jcp_, *desc(), src_md_, diff_weights_md_,
                            diff_bias_md_, diff_dst_md_,
                            dnnl_get_max_threads());
            if (status != status::success) return status;

            init_balancers();

            auto scratchpad = scratchpad_registry().registrar();
            jit_avx512_core_bf16_conv_bwd_weights_kernel_f32::init_scratchpad(
                    scratchpad, jcp_);

            auto reducer_bia_scratchpad = memory_tracking::registrar_t(
                    scratchpad, memory_tracking::names::prefix_reducer_bia);
            reducer_bia_conf_.init_scratchpad(reducer_bia_scratchpad);
            return status;
        }

        jit_conv_conf_t jcp_;
        typename cpu_reducer_t<data_type::f32>::conf_t reducer_bia_conf_;

    private:
        void init_balancers() {
            const size_t max_buffer_size = jcp_.nthr * 3 * 5 * 5 * 16 * 16;
            if (with_bias()) {
                reducer_bia_conf_.init(reduce_balancer_t(jcp_.nthr,
                        jcp_.oc_block, jcp_.ngroups * jcp_.nb_oc, jcp_.mb,
                        max_buffer_size, true));
            }
        }
    };

    jit_avx512_core_bf16_convolution_bwd_weights_t(const pd_t *apd);
    ~jit_avx512_core_bf16_convolution_bwd_weights_t() {
        delete kernel_;
        delete trans_kernel_;
        delete trans_dst_kernel_;
        delete acc_ker_;
        delete reducer_bias_;
    }

    typedef typename prec_traits<data_type::bf16>::type src_data_t;
    typedef typename prec_traits<data_type::bf16>::type diff_dst_data_t;

    virtual status_t execute(const exec_ctx_t &ctx) const override {
        execute_backward_weights(ctx);
        return status::success;
    }

private:
    void execute_backward_weights(const exec_ctx_t &ctx) const;
    void prepare_scratchpad_data(const exec_ctx_t &ctx) const;
    struct thread_info_t;
    void compute_diff_weights_2d(const thread_info_t *) const;
    void compute_diff_weights_3d(const thread_info_t *) const;
    void compute_diff_weights(const thread_info_t *) const;
    void reduce_and_convert_diff_weights(const thread_info_t *) const;
    void compute_diff_bias(const thread_info_t *, const exec_ctx_t &ctx) const;

    const pd_t *pd() const { return (const pd_t *)primitive_impl_t::pd(); }

    size_t tr_src_buf_number(const thread_info_t *ti, int g, int ic) const;
    size_t tr_diff_dst_buf_number(const thread_info_t *ti, int g, int oc) const;
    void trans_src(
            src_data_t *tr_src1, const src_data_t *src1, int my_work) const;
    void trans_dst(diff_dst_data_t *tr_diff_dst1,
            const diff_dst_data_t *diff_dst1, int my_work) const;

    int nthr_, nthr_mb_, nthr_g_, nthr_oc_b_, nthr_ic_b_;

    jit_avx512_core_bf16_conv_bwd_weights_kernel_f32 *kernel_;

    cpu_accumulator_1d_t<data_type::f32> *acc_ker_;
    cpu_reducer_t<data_type::f32> *reducer_bias_;

    jit_trans_src_t *trans_kernel_;
    jit_trans_dst_t *trans_dst_kernel_;
};

} // namespace cpu
} // namespace impl
} // namespace dnnl

#endif

// vim: et ts=4 sw=4 cindent cino+=l0,\:4,N-s
