!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch, M. Verstraete
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

subroutine X(mesh_batch_dotp_matrix)(mesh, aa, bb, dot, symm, reduce)
  type(mesh_t),      intent(in)    :: mesh
  type(batch_t),     intent(in)    :: aa
  type(batch_t),     intent(in)    :: bb
  R_TYPE,            intent(inout) :: dot(:, :)
  logical, optional, intent(in)    :: symm         !< for the moment it is ignored
  logical, optional, intent(in)    :: reduce

  integer :: ist, jst, idim, sp, block_size, ep, ip, ldaa, ldbb, indb, jndb
  R_TYPE :: ss, tmp1, tmp2
  R_TYPE, allocatable :: dd(:, :)
  logical :: use_blas, conj
  type(accel_mem_t) :: dot_buffer
  type(profile_t), save :: prof_copy, prof_gemmcl, prof, profgemm
#ifdef HAVE_MPI
  logical :: reduce_
  type(profile_t), save :: profcomm
#endif
  
  PUSH_SUB(X(mesh_batch_dotp_matrix))
  call profiling_in(prof, "DOTP_BATCH")

#ifdef HAVE_MPI
  reduce_ = .true.
  if(present(reduce)) reduce_ = reduce
#endif
  conj = .false.

  ASSERT(aa%dim == bb%dim)
  ASSERT(batch_status(aa) == batch_status(bb))

  SAFE_ALLOCATE(dd(1:aa%nst, 1:bb%nst))
  ! This has to be set to zero by hand since NaN * 0 = NaN.
  dd(1:aa%nst, 1:bb%nst) = R_TOTYPE(CNST(0.0))

  use_blas = .false.
  
  select case(batch_status(aa))
  case(BATCH_NOT_PACKED)
    use_blas = associated(aa%X(psicont)) .and. associated(bb%X(psicont)) .and. (.not. mesh%use_curvilinear) .and. (aa%dim == 1)

    if(use_blas) then
      call profiling_in(profgemm, "DOTP_BATCH_GEMM")

      ldaa = size(aa%X(psicont), dim = 1)
      ldbb = size(bb%X(psicont), dim = 1)

      call lalg_gemmt(aa%nst, bb%nst, mesh%np, R_TOTYPE(mesh%volume_element), &
        aa%X(psicont), bb%X(psicont), R_TOTYPE(M_ZERO), dd)

    else

      block_size = hardware%X(block_size)

      do idim = 1, aa%dim
        do sp = 1, mesh%np, block_size
          ep = min(mesh%np, sp + block_size - 1)

          if(mesh%use_curvilinear) then

            do ist = 1, aa%nst
              indb = batch_ist_idim_to_linear(aa, (/ist, idim/))
              do jst = 1, bb%nst
                jndb = batch_ist_idim_to_linear(bb, (/jst, idim/))

                ss = M_ZERO
                do ip = sp, ep
                  ss = ss + mesh%vol_pp(ip)*R_CONJ(aa%states_linear(indb)%X(psi)(ip))*bb%states_linear(jndb)%X(psi)(ip)
                end do
                dd(ist, jst) = dd(ist, jst) + ss

              end do
            end do

          else

            do ist = 1, aa%nst
              indb = batch_ist_idim_to_linear(aa, (/ist, idim/))
              do jst = 1, bb%nst
                jndb = batch_ist_idim_to_linear(bb, (/jst, idim/))

                dd(ist, jst) = dd(ist, jst) + mesh%volume_element*&
                  blas_dot(ep - sp + 1, aa%states_linear(indb)%X(psi)(sp), 1, bb%states_linear(jndb)%X(psi)(sp), 1)
              end do
            end do

          end if
        end do
      end do

    end if
  case(BATCH_PACKED)
    ASSERT(.not. mesh%use_curvilinear)
    use_blas = aa%dim == 1

    if(use_blas) then
      conj = .true.
      call profiling_in(profgemm, "DOTP_BATCH_GEMM")

      ldaa = aa%pack%size(1)
      ldbb = bb%pack%size(1)
      call blas_gemm(transa = 'n', transb = 'c', m = aa%nst, n = bb%nst, k = mesh%np, &
        alpha = R_TOTYPE(mesh%volume_element), &
        a = aa%pack%X(psi)(1, 1), lda = ldaa, &
        b = bb%pack%X(psi)(1, 1), ldb = ldbb, &
        beta = R_TOTYPE(M_ZERO), c = dd(1, 1), ldc = aa%nst)
      
    else

      do ist = 1, aa%nst
        do jst = 1, bb%nst
          tmp1 = M_ZERO
          tmp2 = M_ZERO
          do ip = 1, mesh%np
            tmp1 = tmp1 + R_CONJ(aa%pack%X(psi)(2*ist - 1, ip))*bb%pack%X(psi)(2*jst - 1, ip)
            tmp2 = tmp2 + R_CONJ(aa%pack%X(psi)(2*ist    , ip))*bb%pack%X(psi)(2*jst    , ip)
          end do
          dd(ist, jst) = mesh%volume_element*(tmp1 + tmp2)
        end do
      end do

    end if

  case(BATCH_DEVICE_PACKED)
    ASSERT(.not. mesh%use_curvilinear)

    call accel_create_buffer(dot_buffer, ACCEL_MEM_WRITE_ONLY, R_TYPE_VAL, aa%nst*bb%nst)

    call profiling_in(prof_gemmcl, "DOTP_BATCH_CL_GEMM")
    
    call X(accel_gemm)(transA = CUBLAS_OP_N, transB = CUBLAS_OP_T, &
      M = int(aa%nst, 8), N = int(bb%nst, 8), K = int(mesh%np, 8), alpha = R_TOTYPE(M_ONE), &
      A = aa%pack%buffer, offA = 0_8, lda = int(aa%pack%size(1), 8), &
      B = bb%pack%buffer, offB = 0_8, ldb = int(bb%pack%size(1), 8), beta = R_TOTYPE(M_ZERO), &
      C = dot_buffer, offC = 0_8, ldc = int(aa%nst, 8))

    call profiling_count_operations(dble(mesh%np)*aa%nst*bb%nst*(R_ADD + 2*R_MUL))

    call accel_finish()
    call profiling_out(prof_gemmcl)

    call profiling_in(prof_copy, 'DOTP_BATCH_COPY')
    call accel_read_buffer(dot_buffer, aa%nst*bb%nst, dd)
    call profiling_count_transfers(aa%nst*bb%nst, dd(1, 1))
    call accel_finish()
    call profiling_out(prof_copy)

    call accel_release_buffer(dot_buffer)

    forall(ist = 1:aa%nst, jst = 1:bb%nst) dd(ist, jst) = mesh%volume_element*dd(ist, jst)

  case default
    ASSERT(.false.)
    
  end select

  if(batch_status(aa) /= BATCH_DEVICE_PACKED) then
    if(mesh%use_curvilinear) then
      call profiling_count_operations(dble(mesh%np)*aa%nst*bb%nst*(R_ADD + 2*R_MUL))
    else
      call profiling_count_operations(dble(mesh%np)*aa%nst*bb%nst*(R_ADD + R_MUL))
    end if
  end if

  if(use_blas) call profiling_out(profgemm)

#ifdef HAVE_MPI
  if(mesh%parallel_in_domains .and. reduce_) then
    call profiling_in(profcomm, "DOTP_BATCH_REDUCE")
    call comm_allreduce(mesh%mpi_grp%comm, dd)
    call profiling_out(profcomm)
  end if
#endif

  if(conj) then
    forall(ist = 1:aa%nst, jst = 1:bb%nst) dot(aa%states(ist)%ist, bb%states(jst)%ist) = R_CONJ(dd(ist, jst))
  else
    forall(ist = 1:aa%nst, jst = 1:bb%nst) dot(aa%states(ist)%ist, bb%states(jst)%ist) = dd(ist, jst)
  end if

  SAFE_DEALLOCATE_A(dd)

  call profiling_out(prof)
  POP_SUB(X(mesh_batch_dotp_matrix))
end subroutine X(mesh_batch_dotp_matrix)

!-----------------------------------------------------------------

subroutine X(mesh_batch_dotp_self)(mesh, aa, dot, reduce)
  type(mesh_t),      intent(in)    :: mesh
  type(batch_t),     intent(in)    :: aa
  R_TYPE,            intent(inout) :: dot(:, :)
  logical, optional, intent(in)    :: reduce

  integer :: ist, jst, idim, sp, block_size, ep, ip, lda, indb, jndb
  R_TYPE :: ss
  type(profile_t), save :: prof, profgemm, profcomm
  logical :: use_blas, reduce_
  R_TYPE, allocatable :: dd(:, :)

  PUSH_SUB(X(mesh_batch_dotp_self))

  ! some limitations of the current implementation
  ASSERT(ubound(dot, dim = 1) >= aa%nst .and. ubound(dot, dim = 2) >= aa%nst)

  if(batch_status(aa) /= BATCH_NOT_PACKED) then
    call X(mesh_batch_dotp_matrix)(mesh, aa, aa, dot, reduce)
    POP_SUB(X(mesh_batch_dotp_self))
    return
  end if

  reduce_ = .true.
  if(present(reduce)) reduce_ = reduce

  use_blas = associated(aa%X(psicont)) .and. (.not. mesh%use_curvilinear)

  SAFE_ALLOCATE(dd(1:aa%nst, 1:aa%nst))
  ! This has to be set to zero by hand since NaN * 0 = NaN.
  dd(1:aa%nst, 1:aa%nst) = R_TOTYPE(CNST(0.0))

  call profiling_in(prof, "BATCH_DOTP_SELF")

  if(use_blas) then
    call profiling_in(profgemm, "BATCH_HERK")

    lda = size(aa%X(psicont), dim = 1)*aa%dim

    call blas_herk('l', 'c', aa%nst, mesh%np, mesh%vol_pp(1), aa%X(psicont)(1, 1, 1), &
      lda, M_ZERO, dd(1, 1), ubound(dd, dim = 1))

    if(aa%dim == 2) then
      call blas_herk('l', 'c', aa%nst, mesh%np, mesh%vol_pp(1), aa%X(psicont)(1, 2, 1), &
        lda, M_ONE, dd(1, 1), ubound(dd, dim = 1))
    end if

  else

    block_size = hardware%X(block_size)

    do idim = 1, aa%dim
      do sp = 1, mesh%np, block_size
        ep = min(mesh%np, sp + block_size - 1)

        if(mesh%use_curvilinear) then

          do ist = 1, aa%nst
            indb = batch_ist_idim_to_linear(aa, (/ist, idim/))
            do jst = 1, ist
              jndb = batch_ist_idim_to_linear(aa, (/jst, idim/))
              ss = M_ZERO
              do ip = sp, ep
                ss = ss + mesh%vol_pp(ip)*R_CONJ(aa%states_linear(indb)%X(psi)(ip))*aa%states_linear(jndb)%X(psi)(ip)
              end do
              dd(ist, jst) = dd(ist, jst) + ss

            end do
          end do

        else

          do ist = 1, aa%nst
            indb = batch_ist_idim_to_linear(aa, (/ist, idim/))
            do jst = 1, ist
              jndb = batch_ist_idim_to_linear(aa, (/jst, idim/))
              dd(ist, jst) = dd(ist, jst) + mesh%volume_element*&
                blas_dot(ep - sp + 1, aa%states_linear(indb)%X(psi)(sp), 1, aa%states_linear(jndb)%X(psi)(sp), 1)
            end do
          end do

        end if
      end do
    end do
  end if

  if(mesh%use_curvilinear) then
    call profiling_count_operations(dble(mesh%np)*aa%nst**2*(R_ADD + 2*R_MUL))
  else
    call profiling_count_operations(dble(mesh%np)*aa%nst**2*(R_ADD + R_MUL))
  end if

  if(use_blas) call profiling_out(profgemm)

  if(mesh%parallel_in_domains .and. reduce_) then
    call profiling_in(profcomm, "BATCH_SELF_REDUCE")
    call comm_allreduce(mesh%mpi_grp%comm, dd)
    call profiling_out(profcomm)
  end if

  forall(ist = 1:aa%nst)
    forall(jst = 1:aa%nst) 
      dot(aa%states(ist)%ist, aa%states(jst)%ist) = dd(ist, jst)
      dot(aa%states(jst)%ist, aa%states(ist)%ist) = R_CONJ(dd(ist, jst))
    end forall
  end forall

  SAFE_DEALLOCATE_A(dd)

  call profiling_out(prof)
  POP_SUB(X(mesh_batch_dotp_self))
end subroutine X(mesh_batch_dotp_self)

! --------------------------------------------------------------------------

subroutine X(mesh_batch_dotp_vector)(mesh, aa, bb, dot, reduce, cproduct)
  type(mesh_t),      intent(in)    :: mesh
  type(batch_t),     intent(in)    :: aa
  type(batch_t),     intent(in)    :: bb
  R_TYPE,            intent(inout) :: dot(:)
  logical, optional, intent(in)    :: reduce
  logical, optional, intent(in)    :: cproduct

  integer :: ist, indb, idim, ip, status
  logical :: reduce_, cproduct_
  type(profile_t), save :: prof, profcomm
  R_TYPE, allocatable :: tmp(:), cltmp(:, :)
  type(accel_mem_t)  :: dot_buffer

  PUSH_SUB(X(mesh_batch_dotp_vector))
  call profiling_in(prof, "DOTPV_BATCH")

  reduce_ = .true.
  if(present(reduce)) reduce_ = reduce
  
  cproduct_ = optional_default(cproduct, .false.)
  
  ASSERT(aa%nst == bb%nst)
  ASSERT(aa%dim == bb%dim)

  status = batch_status(aa)
  ASSERT(batch_status(bb) == status)

  select case(status)
  case(BATCH_NOT_PACKED)
    do ist = 1, aa%nst
      dot(ist) = M_ZERO
      do idim = 1, aa%dim
        indb = batch_ist_idim_to_linear(aa, (/ist, idim/))
        dot(ist) = dot(ist) + X(mf_dotp)(mesh, aa%states_linear(indb)%X(psi), bb%states_linear(indb)%X(psi),& 
           reduce = .false., dotu = cproduct_)
      end do
    end do

  case(BATCH_PACKED)

    SAFE_ALLOCATE(tmp(1:aa%nst_linear))

    tmp = M_ZERO
    
    if(mesh%use_curvilinear) then
      if(.not. cproduct_) then
        !$omp parallel do private(ip, ist) reduction(+:tmp)
        do ip = 1, mesh%np
          do ist = 1, aa%nst_linear
            tmp(ist) = tmp(ist) + mesh%vol_pp(ip)*R_CONJ(aa%pack%X(psi)(ist, ip))*bb%pack%X(psi)(ist, ip)
          end do
        end do
      else
        !$omp parallel do private(ip, ist) reduction(+:tmp)
        do ip = 1, mesh%np
          do ist = 1, aa%nst_linear
            tmp(ist) = tmp(ist) + mesh%vol_pp(ip)*aa%pack%X(psi)(ist, ip)*bb%pack%X(psi)(ist, ip)
          end do
        end do
      end if
    else
      if(.not. cproduct_) then
        !$omp parallel do private(ip, ist) reduction(+:tmp)
        do ip = 1, mesh%np
          do ist = 1, aa%nst_linear
            tmp(ist) = tmp(ist) + R_CONJ(aa%pack%X(psi)(ist, ip))*bb%pack%X(psi)(ist, ip)
          end do
        end do
      else
        !$omp parallel do private(ip, ist) reduction(+:tmp)
        do ip = 1, mesh%np
          do ist = 1, aa%nst_linear
            tmp(ist) = tmp(ist) + aa%pack%X(psi)(ist, ip)*bb%pack%X(psi)(ist, ip)
          end do
        end do
      end if
    end if

    do ist = 1, aa%nst
      dot(ist) = M_ZERO
      do idim = 1, aa%dim
        indb = batch_ist_idim_to_linear(aa, (/ist, idim/))
        dot(ist) = dot(ist) + mesh%volume_element*tmp(indb)
      end do
    end do

    SAFE_DEALLOCATE_A(tmp)

  case(BATCH_DEVICE_PACKED)

    call accel_create_buffer(dot_buffer, ACCEL_MEM_WRITE_ONLY, R_TYPE_VAL, aa%pack%size(1))

    do ist = 1, aa%nst_linear
      call X(accel_dot)(n = int(mesh%np, 8), &
        x = aa%pack%buffer, offx = int(ist - 1, 8), incx = int(aa%pack%size(1), 8), &
        y = bb%pack%buffer, offy = int(ist - 1, 8), incy = int(bb%pack%size(1), 8), &
        res = dot_buffer, offres = int(ist - 1, 8))
    end do

    SAFE_ALLOCATE(cltmp(1:aa%pack%size(1), 1))

    call accel_read_buffer(dot_buffer, aa%pack%size(1), cltmp)

    call accel_release_buffer(dot_buffer)


    do ist = 1, aa%nst
      dot(ist) = M_ZERO
      do idim = 1, aa%dim
        indb = batch_ist_idim_to_linear(aa, (/ist, idim/))
        dot(ist) = dot(ist) + mesh%volume_element*cltmp(indb, 1)
      end do
    end do

  end select

  if(mesh%parallel_in_domains .and. reduce_) then
    call profiling_in(profcomm, "DOTPV_BATCH_REDUCE")
    call comm_allreduce(mesh%mpi_grp%comm, dot, dim = aa%nst)
    call profiling_out(profcomm)
  end if
  
  call profiling_count_operations(aa%nst*dble(mesh%np)*(R_ADD + R_MUL)*types_get_size(batch_type(aa))/types_get_size(TYPE_FLOAT))

  call profiling_out(prof)
  POP_SUB(X(mesh_batch_dotp_vector))
end subroutine X(mesh_batch_dotp_vector)

!--------------------------------------------------------------------------------------

!> This functions exchanges points of a mesh according to a certain
!! map. Two possible maps can be given. Only one map argument must be present.

subroutine X(mesh_batch_exchange_points)(mesh, aa, forward_map, backward_map)
  type(mesh_t),      intent(in)    :: mesh            !< The mesh descriptor.
  type(batch_t),     intent(inout) :: aa              !< A batch which contains the mesh functions whose points will be exchanged.
  integer, optional, intent(in)    :: forward_map(:)  !< A map which gives the destination of the value each point.
  logical, optional, intent(in)    :: backward_map    !< A map which gives the source of the value of each point.
  logical :: packed_on_entry

#ifdef HAVE_MPI
  integer :: ip, ipg, npart, ipart, ist, pos, nstl, np_points, np_inner, np_bndry
  integer, allocatable :: send_count(:), recv_count(:), send_disp(:), recv_disp(:), &
       points_inner(:), points_bndry(:), partno_inner(:), partno_bndry(:)
  integer, allocatable :: send_count_nstl(:), recv_count_nstl(:), send_disp_nstl(:), recv_disp_nstl(:)
  R_TYPE, allocatable  :: send_buffer(:, :), recv_buffer(:, :)
#endif

  PUSH_SUB(X(mesh_batch_exchange_points))

  ASSERT(present(backward_map) .neqv. present(forward_map))
  ASSERT(batch_type(aa) == R_TYPE_VAL)
  packed_on_entry = batch_status(aa) == BATCH_NOT_PACKED
  if (packed_on_entry) then
    call batch_unpack(aa, force=.true.)
  end if

  if(.not. mesh%parallel_in_domains) then
    message(1) = "Not implemented for the serial case. Really, only in parallel."
    call messages_fatal(1)
  else

#ifdef HAVE_MPI
    npart = mesh%mpi_grp%size
    nstl = aa%nst_linear

    SAFE_ALLOCATE(send_count(1:npart))
    SAFE_ALLOCATE(recv_count(1:npart))
    SAFE_ALLOCATE(send_count_nstl(1:npart))
    SAFE_ALLOCATE(recv_count_nstl(1:npart))
    SAFE_ALLOCATE(send_disp_nstl(1:npart))
    SAFE_ALLOCATE(recv_disp_nstl(1:npart))
    SAFE_ALLOCATE(send_buffer(1:nstl, mesh%np))
    SAFE_ALLOCATE(recv_buffer(1:nstl, mesh%np))

    if(present(forward_map)) then

      SAFE_ALLOCATE(send_disp(1:npart))
      SAFE_ALLOCATE(recv_disp(1:npart))
      SAFE_ALLOCATE(points_inner(1:mesh%np))
      SAFE_ALLOCATE(points_bndry(1:mesh%np))
      ASSERT(ubound(forward_map, dim = 1) == mesh%np_global)

      send_count = 0
      np_inner   = 0
      np_bndry   = 0
      np_points  = 0
      do ip = 1, mesh%np
        ! Get the temporally global point
        ipg = mesh%vp%local(mesh%vp%xlocal + ip - 1)
        ! Store the global point
        ! Global index can be either in the mesh or in the boundary.
        ! Different treatment is needed for each case.
        if (ipg > mesh%np_global) then
          np_bndry = np_bndry + 1
          points_bndry(np_bndry) = forward_map(ipg) - mesh%np_global
        else
          np_inner = np_inner + 1
          points_inner(np_inner) = forward_map(ipg)
        end if
        np_points = np_points + 1
      end do

      SAFE_ALLOCATE(partno_inner(1:np_points))
      SAFE_ALLOCATE(partno_bndry(1:np_points))
      call partition_get_partition_number(mesh%inner_partition, np_inner, &
           points_inner, partno_inner)
      call partition_get_partition_number(mesh%bndry_partition, np_bndry, &
           points_bndry, partno_bndry)
      SAFE_DEALLOCATE_A(points_inner)
      SAFE_DEALLOCATE_A(points_bndry)
      do ip = 1, np_inner
        ! the destination
        ipart = partno_inner(ip)
        INCR(send_count(ipart), 1)
      end do
      do ip = 1, np_bndry
        ! the destination
        ipart = partno_bndry(ip)
        INCR(send_count(ipart), 1)
      end do
      ASSERT(sum(send_count) == mesh%np)

      ! Receiving number of points is the inverse matrix of the sending points
      call mpi_debug_in(mesh%mpi_grp%comm, C_MPI_ALLTOALL)
      call MPI_Alltoall(send_count(1), 1, MPI_INTEGER, &
                        recv_count(1), 1, MPI_INTEGER, &
                        mesh%mpi_grp%comm, mpi_err)
      call mpi_debug_out(mesh%mpi_grp%comm, C_MPI_ALLTOALL)
      ASSERT(sum(recv_count) == mesh%np)

      send_disp(1) = 0
      recv_disp(1) = 0
      do ipart = 2, npart
        send_disp(ipart) = send_disp(ipart - 1) + send_count(ipart - 1)
        recv_disp(ipart) = recv_disp(ipart - 1) + recv_count(ipart - 1)
      end do

      ASSERT(send_disp(npart) + send_count(npart) == mesh%np)
      ASSERT(recv_disp(npart) + recv_count(npart) == mesh%np)

      ! Pack for sending
      send_count = 0
      ! First inner points
      do ip = 1, np_inner
        !the destination
        ipart = partno_inner(ip)
        INCR(send_count(ipart), 1)
        pos = send_disp(ipart) + send_count(ipart)
        forall(ist = 1:nstl) send_buffer(ist, pos) = aa%states_linear(ist)%X(psi)(ip)
      end do
      ! Then boundary points
      do ip = 1, np_bndry
        !the destination
        ipart = partno_bndry(ip)
        INCR(send_count(ipart), 1)
        pos = send_disp(ipart) + send_count(ipart)
        forall(ist = 1:nstl) send_buffer(ist, pos) = aa%states_linear(ist)%X(psi)(ip)
      end do

      SAFE_DEALLOCATE_A(partno_bndry)
      SAFE_DEALLOCATE_A(partno_inner)

      send_count_nstl = send_count * nstl
      send_disp_nstl = send_disp * nstl
      recv_count_nstl = recv_count * nstl
      recv_disp_nstl = recv_disp * nstl
      call mpi_debug_in(mesh%mpi_grp%comm, C_MPI_ALLTOALLV)
      call MPI_Alltoallv(send_buffer(1, 1), send_count_nstl, send_disp_nstl, R_MPITYPE, &
        recv_buffer(1, 1), recv_count_nstl, recv_disp_nstl, R_MPITYPE, mesh%mpi_grp%comm, mpi_err)
      call mpi_debug_out(mesh%mpi_grp%comm, C_MPI_ALLTOALLV)

      recv_count = 0
      do ipg = 1, mesh%np_global
        if(mesh%vp%part_vec(forward_map(ipg)) == mesh%vp%partno) then
          ip = vec_global2local(mesh%vp, forward_map(ipg), mesh%vp%partno)
          ASSERT(ip /= 0)
          ipart = mesh%vp%part_vec(ipg)
          INCR(recv_count(ipart), 1)
          pos = recv_disp(ipart) + recv_count(ipart)
          forall(ist = 1:nstl) aa%states_linear(ist)%X(psi)(ip) = recv_buffer(ist, pos)
        end if
      end do

      SAFE_DEALLOCATE_A(send_disp)
      SAFE_DEALLOCATE_A(recv_disp)
    
    else ! backward map

      recv_count = mesh%vp%recv_count
      ASSERT(sum(recv_count) == mesh%np)

      send_count = mesh%vp%send_count
      ASSERT(sum(send_count) == mesh%np)

      ASSERT(mesh%vp%send_disp(npart) + send_count(npart) == mesh%np)
      ASSERT(mesh%vp%recv_disp(npart) + recv_count(npart) == mesh%np)

      ! Pack for sending
      send_count = 0  
      do ip = 1, mesh%np
        ipart = mesh%vp%part_local(ip)
        INCR(send_count(ipart), 1)
        pos = mesh%vp%send_disp(ipart) + send_count(ipart)
        forall(ist = 1:nstl) send_buffer(ist, pos) = aa%states_linear(ist)%X(psi)(ip) 
      end do
      
      send_count_nstl = send_count * nstl
      send_disp_nstl = mesh%vp%send_disp * nstl
      recv_count_nstl = recv_count * nstl
      recv_disp_nstl = mesh%vp%recv_disp * nstl
      call mpi_debug_in(mesh%mpi_grp%comm, C_MPI_ALLTOALLV)
      call MPI_Alltoallv(send_buffer(1, 1), send_count_nstl, send_disp_nstl, R_MPITYPE, &
        recv_buffer(1, 1), recv_count_nstl, recv_disp_nstl, R_MPITYPE, mesh%mpi_grp%comm, mpi_err)
      call mpi_debug_out(mesh%mpi_grp%comm, C_MPI_ALLTOALLV)

      ! Unpack on receiving
      recv_count = 0
      do ip = 1, mesh%np
        ! get the destination
        ipart = mesh%vp%part_local_rev(ip)
        INCR(recv_count(ipart), 1)
        pos = mesh%vp%recv_disp(ipart) + recv_count(ipart)
        forall(ist = 1:nstl) aa%states_linear(ist)%X(psi)(ip) = recv_buffer(ist, pos)
      end do

    end if

    SAFE_DEALLOCATE_A(send_count)
    SAFE_DEALLOCATE_A(recv_count)
    SAFE_DEALLOCATE_A(send_buffer)
    SAFE_DEALLOCATE_A(recv_buffer)
    SAFE_DEALLOCATE_A(send_count_nstl)
    SAFE_DEALLOCATE_A(recv_count_nstl)
    SAFE_DEALLOCATE_A(send_disp_nstl)
    SAFE_DEALLOCATE_A(recv_disp_nstl)
#endif
  end if

  if (packed_on_entry) then
    call batch_pack(aa)
  end if
  POP_SUB(X(mesh_batch_exchange_points))
end subroutine X(mesh_batch_exchange_points)

! -----------------------------------------------------
!> This function should not be called directly, but through mesh_batch_nrm2.
subroutine X(priv_mesh_batch_nrm2)(mesh, aa, nrm2)
  type(mesh_t),            intent(in)    :: mesh
  type(batch_t),           intent(in)    :: aa
  FLOAT,                   intent(out)   :: nrm2(:)

  integer :: ist, idim, indb, ip
  R_TYPE :: a0
  FLOAT, allocatable :: scal(:), ssq(:)
  type(accel_mem_t)  :: nrm2_buffer
  type(profile_t), save :: prof

  PUSH_SUB(X(priv_mesh_batch_nrm2))
  call profiling_in(prof, 'MESH_BATCH_NRM2')

  select case(batch_status(aa))
  case(BATCH_NOT_PACKED)
    do ist = 1, aa%nst
      nrm2(ist) = M_ZERO
      do idim = 1, aa%dim
        indb = batch_ist_idim_to_linear(aa, (/ist, idim/))
        nrm2(ist) = hypot(nrm2(ist), X(mf_nrm2)(mesh, aa%states_linear(indb)%X(psi), reduce = .false.))
      end do
    end do

  case(BATCH_PACKED)
    
    SAFE_ALLOCATE(scal(1:aa%nst_linear))
    SAFE_ALLOCATE(ssq(1:aa%nst_linear))

    scal = M_ZERO
    ssq  = M_ONE
    
    if(.not. mesh%use_curvilinear) then

      do ip = 1, mesh%np
        do ist = 1, aa%nst_linear
          a0 = aa%pack%X(psi)(ist, ip)
          if(abs(a0) <= M_EPSILON) cycle
          if(scal(ist) < abs(a0)) then
            ssq(ist) = M_ONE + ssq(ist)*(scal(ist)/abs(a0))**2
            scal(ist) = abs(a0)
          else
            ssq(ist) = ssq(ist) + (abs(a0)/scal(ist))**2
          end if
        end do
      end do

    else

      do ip = 1, mesh%np
        do ist = 1, aa%nst_linear
          a0 = aa%pack%X(psi)(ist, ip)
          if(a0 == R_TOTYPE(M_ZERO)) cycle
          if(scal(ist) < abs(a0)) then
            ssq(ist) =  mesh%vol_pp(ip)*M_ONE + ssq(ist)*(scal(ist)/abs(a0))**2
            scal(ist) = abs(a0)
          else
            ssq(ist) = ssq(ist) + mesh%vol_pp(ip)*(abs(a0)/scal(ist))**2
          end if
        end do
      end do

    end if

    do ist = 1, aa%nst
      nrm2(ist) = M_ZERO
      do idim = 1, aa%dim
        indb = batch_ist_idim_to_linear(aa, (/ist, idim/))
         nrm2(ist) = hypot(nrm2(ist), scal(indb)*sqrt(mesh%volume_element*ssq(indb)))
      end do
    end do

  case(BATCH_DEVICE_PACKED)

    ASSERT(.not. mesh%use_curvilinear)

    SAFE_ALLOCATE(ssq(1:aa%pack%size(1)))

    call accel_create_buffer(nrm2_buffer, ACCEL_MEM_WRITE_ONLY, TYPE_FLOAT, aa%pack%size(1))

    do ist = 1, aa%nst_linear

      call X(accel_nrm2)(N = int(mesh%np, 8), X = aa%pack%buffer, offx = int(ist - 1, 8), incx = int(aa%pack%size(1), 8), &
        res = nrm2_buffer, offres = int(ist - 1, 8))
      
    end do

    call accel_read_buffer(nrm2_buffer, aa%pack%size(1), ssq)

    call accel_release_buffer(nrm2_buffer)

    do ist = 1, aa%nst
      nrm2(ist) = M_ZERO
      do idim = 1, aa%dim
        indb = batch_ist_idim_to_linear(aa, (/ist, idim/))
        nrm2(ist) = hypot(nrm2(ist), sqrt(mesh%volume_element)*ssq(indb))
      end do
    end do

    SAFE_DEALLOCATE_A(ssq)

  end select
  
  ! REDUCTION IS REQUIRED, THIS IS DONE BY THE CALLING FUNCTION

  call profiling_out(prof)
  POP_SUB(X(priv_mesh_batch_nrm2))
end subroutine X(priv_mesh_batch_nrm2)


!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
