!! Copyright (C) 2011 U. De Giovannini
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.


! ---------------------------------------------------------
!> Write the photoelectron wavefunctions in real space
! ---------------------------------------------------------
subroutine pes_mask_output_states(st, gr, geo, dir, outp, mask)
  type(states_t),   intent(in) :: st
  type(grid_t),     intent(in) :: gr
  type(geometry_t), intent(in) :: geo
  character(len=*), intent(in) :: dir
  type(output_t),   intent(in) :: outp
  type(pes_mask_t), intent(in) :: mask

  integer :: ik, ist, idim, is, ierr
  character(len=80) :: fname
  type(unit_t) :: fn_unit

  CMPLX, allocatable :: PsiAB(:,:,:,:)
  FLOAT,allocatable :: RhoAB(:,:) 
  type(cube_function_t) :: cf
  type(mesh_t):: mesh   
  
  type(batch_t)        :: psib
  type(density_calc_t) :: dens_calc

  PUSH_SUB(pes_mask_output_states)
  
  mesh= gr%mesh

  SAFE_ALLOCATE(PsiAB(1:mesh%np_part,1:st%d%dim,1:st%nst,1:st%d%nik))
  SAFE_ALLOCATE(RhoAB(1:mesh%np_part,1:st%d%nspin))

  call cube_function_null(cf)    
  call zcube_function_alloc_RS(mask%cube, cf, force_alloc = .true.)

  RhoAB= M_ZERO
  
  !Calculate the pes density \Psi_A + \Psi_B on the simulation Box  
  call density_calc_init(dens_calc, st, gr, RhoAB)

  do ik = st%d%kpt%start, st%d%kpt%end
    do ist = st%st_start, st%st_end
      do idim = 1, st%d%dim

        cf%zRs=M_z0

        call pes_mask_K_to_X(mask,mesh,mask%k(:,:,:, idim, ist, ik),cf%zRs)

        call pes_mask_cube_to_mesh(mask, cf, PsiAB(:, idim, ist, ik))        

        if (mask%mode /= PES_MASK_MODE_PASSIVE) then 
          PsiAB(:, idim, ist, ik) = PsiAB(:, idim, ist, ik) + st%zpsi(:, idim, ist, ik) 
        end if
        
      end do
    end do
     
    call batch_init(psib, st%d%dim, st%st_start, st%st_end, PsiAB(:, :, st%st_start:, ik))
    call density_calc_accumulate(dens_calc, ik, psib) 
    call batch_end(psib)

  end do
  
  call density_calc_end(dens_calc)

  ! THE OUTPUT 
  if(iand(outp%what, C_OUTPUT_PES_DENSITY) /= 0) then
    fn_unit = units_out%length**(-gr%mesh%sb%dim)
    do is = 1, st%d%nspin
      if(st%d%nspin == 1) then
        write(fname, '(a)') 'pes_den'
      else
        write(fname, '(a,i1)') 'pes_den-sp', is
      endif
      call dio_function_output(outp%how, dir, fname, gr%fine%mesh, &
        RhoAB(:, is), fn_unit, ierr, is_tmp = .false., geo = geo, grp = st%dom_st_kpt_mpi_grp)
    end do
  end if


  if(iand(outp%what, C_OUTPUT_PES_WFS) /= 0) then
    fn_unit = sqrt(units_out%length**(-gr%mesh%sb%dim))
    do ist = st%st_start, st%st_end
!        if(loct_isinstringlist(ist, outp%wfs_list)) then
        do ik = st%d%kpt%start, st%d%kpt%end
          do idim = 1, st%d%dim
            if(st%d%nik > 1) then
              if(st%d%dim > 1) then
                write(fname, '(a,i3.3,a,i4.4,a,i1)') 'pes_wf-k', ik, '-st', ist, '-sd', idim
              else
                write(fname, '(a,i3.3,a,i4.4)')      'pes_wf-k', ik, '-st', ist
              endif
            else
              if(st%d%dim > 1) then
                write(fname, '(a,i4.4,a,i1)')        'pes_wf-st', ist, '-sd', idim
              else
                write(fname, '(a,i4.4)')             'pes_wf-st', ist
              endif
            endif
              
            call zio_function_output(outp%how, dir, fname, gr%mesh, &
              PsiAB(1:, idim, ist, ik), fn_unit, ierr, is_tmp = .false., geo = geo)

          end do
        end do
 !       end if
    end do
  end if

  SAFE_DEALLOCATE_A(PsiAB)
  SAFE_DEALLOCATE_A(RhoAB)

  call zcube_function_free_RS(mask%cube, cf)


  POP_SUB(pes_mask_output_states)
end subroutine pes_mask_output_states

! ---------------------------------------------------------
!
!> Calculates the momentum-resolved photoelectron probability
!!\f[
!!            P(k) = \sum_i |\Psi_{B,i}(k)|^2 
!!\f]
! ---------------------------------------------------------
subroutine pes_mask_create_full_map(mask, st, pesK, wfAk)
  type(pes_mask_t), intent(in)  :: mask
  type(states_t),   intent(in)  :: st
  FLOAT, target,    intent(out) :: pesK(:,:,:)
  CMPLX, optional,  intent(in)  :: wfAk(:,:,:,:,:,:)

  integer :: ist, ik, kx, ky, kz, idim
  FLOAT   :: scale
  FLOAT, pointer :: pesKloc(:,:,:)

! #ifdef HAVE_MPI
!   type(profile_t), save :: reduce_prof
! #endif

  PUSH_SUB(pes_mask_create_full_map)

  pesK = M_ZERO
  if (mask%cube%parallel_in_domains) then
    SAFE_ALLOCATE(pesKloc(1:mask%ll(1),1:mask%ll(2),1:mask%ll(3)))
    pesKloc = M_ZERO
  else 
    pesKloc => pesK
  end if

  do ik = st%d%kpt%start, st%d%kpt%end
    do ist = st%st_start, st%st_end

      do kx = 1, mask%ll(1)
        do ky = 1, mask%ll(2)
          do kz = 1, mask%ll(3)

            if(present(wfAk))then
              pesKloc(kx, ky, kz) = pesKloc(kx, ky, kz) + st%occ(ist, ik) * &
                sum(abs(mask%k(kx, ky, kz, :, ist, ik) + wfAk(kx,ky,kz,:, ist, ik)  )**2)
            else
              pesKloc(kx, ky, kz) = pesKloc(kx, ky, kz) + st%occ(ist, ik) * sum(abs(mask%k(kx, ky, kz, :, ist, ik)  )**2)
            end if
            
          end do
        end do
      end do

    end do
  end do


  if(st%parallel_in_states .or. st%d%kpt%parallel) then
!     call profiling_in(reduce_prof, "pes_DENSITY_REDUCE")
    call comm_allreduce(st%st_kpt_mpi_grp%comm, pesKloc)
!     call profiling_out(reduce_prof)
  end if  

  
  if (mask%cube%parallel_in_domains) then

    call dcube_function_allgather(mask%cube, pesK, pesKloc, gatherfs = .true.)
    
!     if(mask%pw_map_how .eq. PW_MAP_PFFT) then
!       call dcube_function_allgather(mask%cube, pesK, pesKloc, & 
!                                     order = (/2,3,1/))
! 
!     else if(mask%pw_map_how .eq. PW_MAP_PNFFT) then
!       
!       
! !       pesK(mask%fs_istart(1):mask%fs_istart(1)+mask%ll(1)-1, &
! !            mask%fs_istart(2):mask%fs_istart(2)+mask%ll(2)-1, &
! !            mask%fs_istart(3):mask%fs_istart(3)+mask%ll(3)-1) = &
! !         pesKloc(1:mask%ll(1),1:mask%ll(2),1:mask%ll(3))
! !       
! !       write(file, '(i3.3)') mpi_world%rank
! !       file = "fullmap_"//trim(file)
! !       print *,"write file: ",file
! !       call pes_mask_dump_full_mapM(pesK, file, mask%Lk)
! 
! 
!       
!       call dcube_function_allgather(mask%cube, pesK, pesKloc, gatherfs = .true.)
! !                                     order = (/1,3,2/))
! !     else
! !       call dcube_function_allgather(mask%cube, pesK, pesKloc)
!       
!     end if
    
!     call dcube_function_allgather(mask%cube, pesK, pesKloc, gatherfs = .true.)

    SAFE_DEALLOCATE_P(pesKloc) 
  end if

  ! This is needed in order to normalize the Fourier integral 
  scale = M_ONE
  do idim=1, mask%mesh%sb%dim
    scale = scale *( mask%spacing(idim)/sqrt(M_TWO*M_PI))**2
  end do
  pesK = pesK *scale 

  POP_SUB(pes_mask_create_full_map)
end subroutine pes_mask_create_full_map

! --------------------------------------------------------
!
!>  Qshep interpolation helper function initialization.
!!  Generates the linearized version of pesK (cube_f) and the associated
!!  qshep interpolator opbject (interp).
!
! ---------------------------------------------------------
subroutine pes_mask_interpolator_init(pesK, Lk, dim, cube_f, interp)
  FLOAT,          intent(in)    :: pesK(:,:,:)
  FLOAT,          intent(in)    :: Lk(:)
  integer,        intent(in)    :: dim
  FLOAT, pointer, intent(inout) :: cube_f(:)
  type(qshep_t),  intent(out)   :: interp
  
  integer :: np, ii, ll(3), ix, iy, iz
  FLOAT   :: KK(3)
  FLOAT, allocatable ::  kx(:),ky(:),kz(:)

  PUSH_SUB(pes_mask_interpolator_init)


  call messages_write("Initializing Qshep interpolator... ")
  call messages_info()

  ll = 1
  do ii = 1, dim
    ll(ii) = size(pesK,ii) 
  end do
  
  
  np = ll(1)*ll(2)*ll(3)  

  !check dim
  if (dim  <  2 .or. dim > 3) then
    message(1) = "This interpolator works only for 2 <= dim <= 3." 
    call messages_fatal(1)
  end if
  
  SAFE_ALLOCATE(cube_f(1:np))

  SAFE_ALLOCATE(kx(1:np))
  SAFE_ALLOCATE(ky(1:np))
  SAFE_ALLOCATE(kz(1:np))

  cube_f = M_ZERO
  kx = M_ZERO
  ky = M_ZERO
  kz = M_ZERO


  ii=1
  do ix = 1, ll(1)
    KK(1) = Lk(ix)
    do iy = 1, ll(2)
      KK(2) = Lk(iy)
      do iz = 1, ll(3)
        KK(3) = Lk(iz)

        cube_f(ii) =  pesK(ix,iy,iz)

        kx(ii) = KK(1)
        ky(ii) = KK(2)
        kz(ii) = KK(3)

        ii = ii +1
      end do 
    end do
  end do 
  


  select case(dim)
    case (2)
      call qshep_init(interp, np, cube_f, kx, ky) 
    case (3)
      call qshep_init(interp, np, cube_f, kx, ky, kz) 
  end select
 
  SAFE_DEALLOCATE_A(kx)    
  SAFE_DEALLOCATE_A(ky)    
  SAFE_DEALLOCATE_A(kz)    
 
  call messages_write("done")
  call messages_new_line()
  call messages_info()
  

  POP_SUB(pes_mask_interpolator_init)
end subroutine pes_mask_interpolator_init

! ---------------------------------------------------------
!>  Destroy the interpolation objects
! ---------------------------------------------------------
subroutine pes_mask_interpolator_end(cube_f, interp)
  FLOAT, pointer, intent(inout) :: cube_f(:)
  type(qshep_t),  intent(inout) :: interp

  PUSH_SUB(pes_mask_interpolator_end)
  
  call qshep_end(interp)
  
  SAFE_DEALLOCATE_P(cube_f)
  
  POP_SUB(pes_mask_interpolator_end)
end subroutine pes_mask_interpolator_end



! ---------------------------------------------------------
subroutine pes_mask_output_full_mapM(pesK, file, Lk, how, sb)
  FLOAT,             intent(in) :: pesK(:,:,:)
  character(len=*),  intent(in) :: file
  FLOAT,             intent(in) :: Lk(:)
  integer,           intent(in) :: how
  type(simul_box_t), intent(in) :: sb 
  
  integer :: iunit
  integer :: ierr
  character(len=512) :: filename
  type(cube_t) :: cube
  type(cube_function_t) :: cf
  integer :: ll(3),ii
  FLOAT :: dk(3)  

  PUSH_SUB(pes_mask_output_full_mapM)

  ll = 1
  do ii = 1, 3
    ll(ii) = size(pesK,ii) 
  end do

  call cube_init(cube, ll, sb)
  call cube_function_null(cf)
  call dcube_function_alloc_RS(cube, cf, force_alloc = .true.)
  cf%dRS = pesK

  dk= abs(Lk(2)-Lk(1))
  do ii = 1, 3
    dk(ii) = units_from_atomic(sqrt(units_out%energy), dk(ii))
  end do
  
#if defined(HAVE_NETCDF)  
  
  if(iand(how, C_OUTPUT_HOW_NETCDF) /= 0) then
    filename = trim(file)//".ncdf"
    write(message(1), '(a)') 'Writing netcdf format file: '
    call messages_info(1)
  
    call dout_cf_netcdf(filename, ierr, cf, cube, sb%dim, dk(:) , & 
          .false., sqrt(units_out%energy)**sb%dim)

  end if

#endif
  
  if(iand(how, C_OUTPUT_HOW_VTK) /= 0)  then
    filename = trim(file)//".vtk"
    write(message(1), '(a)') 'Writing vtk format file: '
    call messages_info(1)
        
    call dout_cf_vtk(filename, ierr, cf, cube, dk(:),& 
      sqrt(units_out%energy)**sb%dim)
      
  else
    write(message(1), '(a)') 'Writing ASCII format file: '
    call messages_info(1)
    call out_ascii()
  end if
  
  call cube_end(cube)
  call dcube_function_free_RS(cube, cf)

  POP_SUB(pes_mask_output_full_mapM)
  
contains  

! ---------------------------------------------------------
! Just dump the matrix in ascii form x,y,z, val. As the output
! can be pretty big is probably better to use netcdf instead.
! ---------------------------------------------------------
  subroutine out_ascii()

    integer :: ii, ix, iy, iz
    FLOAT ::  KK(3)
    integer :: ll(3)

    PUSH_SUB(pes_mask_output_full_mapM.out_ascii)

    iunit = io_open(file, action='write')


    ll = 1
    do ii = 1, 3
      ll(ii) = size(pesK,ii) 
    end do

    do ix = 1, ll(1)
      KK(1) = Lk(ix)
      do iy = 1, ll(2)
        KK(2) = Lk(iy)
        do iz = 1, ll(3)
          KK(3) = Lk(iz)
 
            write(iunit, '(es19.12,2x,es19.12,2x,es19.12,2x,es19.12)') &
                    units_from_atomic(sqrt(units_out%energy), KK(1)),&
                    units_from_atomic(sqrt(units_out%energy), KK(2)),&
                    units_from_atomic(sqrt(units_out%energy), KK(3)),&
                    pesK(ix,iy,iz) 
 
          end do  
        end do      
      end do

      call io_close(iunit)
  
      POP_SUB(pes_mask_output_full_mapM.out_ascii)
  end subroutine out_ascii
  
end subroutine pes_mask_output_full_mapM



! ---------------------------------------------------------
subroutine pes_mask_output_full_mapM_cut(pesK, file, Lk, dim, pol, dir, integrate)
  FLOAT,            intent(in) :: pesK(:,:,:)
  FLOAT,            intent(in) :: Lk(:)
  character(len=*), intent(in) :: file
  integer,          intent(in) :: dim
  FLOAT,            intent(in) :: pol(3)
  integer,          intent(in) :: dir
  integer,          intent(in) :: integrate

  integer              :: ii, ix, iy, iunit
  FLOAT                :: KK(3),temp
  integer              :: ll(3)
  integer, allocatable :: idx(:,:)
  FLOAT, allocatable   :: Lk_(:,:)
  FLOAT                :: rotation(1:dim,1:dim)
! integration
  FLOAT                :: K, KKK(3), theta, phi, Dphi, Dk
  integer              :: iph, Nphi


  FLOAT, pointer :: cube_f(:)
  type(qshep_t) :: interp

  PUSH_SUB(pes_mask_output_full_mapM_cut)
  
  iunit = io_open(file, action='write')

  ll = 1
  do ii = 1, dim
    ll(ii) = size(pesK, ii) 
  end do

  ASSERT(size(pesK, 1) == size(Lk,1))

  SAFE_ALLOCATE(idx(1:maxval(ll(:)), 1:3))
  SAFE_ALLOCATE(Lk_(1:size(Lk,1), 1:3))

  Dk = abs(Lk(2)-Lk(1)) 
  
  do ii = 1, 3
    Lk_(:,ii) = Lk(:)
    call sort(Lk_(1:ll(ii), ii), idx(1:ll(ii), ii)) !We need to sort the k-vectors in order to dump in gnuplot format
  end do  
  
  
  if (sum((pol-(/0 ,0 ,1/))**2)  <= 1E-14 .and. integrate == INTEGRATE_NONE) then !no need to rotate and interpolate
    
    do ix = 1, ll(1)
      KK(1) = Lk_(ix, 1)
      do iy = 1, ll(2)
        KK(2) = Lk_(iy, 2)
      

        select case (dir)
          case (1)
            temp = pesK(idx(ll(dir)/2 + 1, 1), idx(ix, 2), idx(iy, 3))    
    
          case (2)
            temp = pesK(idx(ix, 1), idx(ll(dir)/2 + 1, 2), idx(iy, 3))    
      
          case (3)
            temp = pesK(idx(ix, 1), idx(iy, 2), idx(ll(dir)/2 + 1, 3))    
    
        end select
        
        write(iunit, '(es19.12,2x,es19.12,2x,es19.12)') &
                units_from_atomic(sqrt(units_out%energy), KK(1)),&
                units_from_atomic(sqrt(units_out%energy), KK(2)),&
                temp
 
       
      end do
      write(iunit, *)  
    
    end do
  
    
  else 
    ! We set the z-axis along the polarization 
    call generate_rotation_matrix(rotation, (/M_ZERO, M_ZERO, M_ONE/), pol )

    if(in_debug_mode) then
      print *,"Rotate z-axis over the zenith axis"
      print *,rotation(1,:)
      print *,rotation(2,:)
      print *,rotation(3,:)
    end if

    call pes_mask_interpolator_init(pesK, Lk, dim, cube_f, interp)

    do ix = 1, ll(1)
     do iy = 1, ll(2)

       !cut 
       select case (dir)
         case (1)
           KK(1) = M_ZERO         
           KK(2) = Lk_(ix, 1)
           KK(3) = Lk_(iy, 2)

         case (2)
           KK(1) = Lk_(ix, 1)
           KK(2) = M_ZERO
           KK(3) = Lk_(iy, 2)
 
         case (3)
           KK(1) = Lk_(ix, 1)
           KK(2) = Lk_(iy, 2)
           KK(3) = M_ZERO

       end select

       temp = qshep_interpolate(interp, cube_f, matmul(rotation,KK(1:3)) )

       select case (integrate)
         case (INTEGRATE_PHI)
           temp = M_ZERO
           K = sqrt(KK(1)**2 + KK(2)**2 + KK(3)**2)

           Nphi = 360
           Dphi = M_TWO * M_PI/Nphi

           do iph = 0, Nphi
             phi = iph * Dphi
             theta = atan2(sqrt(KK(1)**2+KK(2)**2),KK(3))

             KKK(1) = K *sin(theta)*cos(phi) 
             KKK(2) = K *sin(theta)*sin(phi)
             KKK(3) = K *cos(theta)
        
             temp = temp + &
                           abs(qshep_interpolate(interp, cube_f, matmul(rotation,KKK(1:3)) ))
           end do
           temp = temp * Dphi     
           
         case (INTEGRATE_KX)
           temp = M_ZERO
           do ii =1, ll(1)
              KKK(:) = KK(:) + (/Lk(ii), M_ZERO, M_ZERO/)
              temp = temp + &
                            abs(qshep_interpolate(interp, cube_f, matmul(rotation,KKK(1:3)) ))
           end do
           temp = temp * Dk  
         
         case (INTEGRATE_KY)
           temp = M_ZERO
           do ii =1, ll(2)
              KKK(:) = KK(:) + (/M_ZERO, Lk(ii), M_ZERO/)
              temp = temp + &
                            abs(qshep_interpolate(interp, cube_f, matmul(rotation,KKK(1:3)) ))
           end do
           temp = temp * Dk  
         
         case (INTEGRATE_KZ)
           temp = M_ZERO
           do ii =1, ll(3)
              KKK(:) = KK(:) + (/M_ZERO, M_ZERO, Lk(ii)/)
              temp = temp + &
                            abs(qshep_interpolate(interp, cube_f, matmul(rotation,KKK(1:3)) ))
           end do
           temp = temp * Dk  

       end select
   
       write(iunit, '(es19.12,2x,es19.12,2x,es19.12)') &
               units_from_atomic(sqrt(units_out%energy), Lk_(ix, 1)),&
               units_from_atomic(sqrt(units_out%energy), Lk_(iy, 2)),&
               temp
  
     end do
     write(iunit, *)  
    end do


  end if

  call io_close(iunit)
  
  SAFE_DEALLOCATE_A(idx) 
  SAFE_DEALLOCATE_A(Lk_) 
  
  POP_SUB(pes_mask_output_full_mapM_cut)
end subroutine pes_mask_output_full_mapM_cut

! ---------------------------------------------------------
subroutine pes_mask_output_ar_polar_M(pesK, file, Lk, dim, dir, Emax, Estep)
  FLOAT,            intent(in) :: pesK(:,:,:)
  character(len=*), intent(in) :: file
  FLOAT,            intent(in) :: Lk(:)
  integer,          intent(in) :: dim
  FLOAT,            intent(in) :: Emax
  FLOAT,            intent(in) :: Estep
  FLOAT,            intent(in) :: dir(:) 

  integer :: ii
  FLOAT ::  KK(3)

  integer :: nn, ie
  FLOAT  :: step
  FLOAT, allocatable ::  pesM(:,:)

  ! needed for interpolation in 2D and 3D 
  FLOAT, pointer :: cube_f(:)
  type(qshep_t) :: interp

  FLOAT :: Dtheta, Dphi, theta, phi, EE
  integer :: Nphi, ith, iph, Nth
  integer :: ll(1:3)
  FLOAT :: vref(1:dim), rotation(1:dim,1:dim)
  FLOAT :: eGrid(3), thGrid(3), phiBounds(2)
  type(profile_t), save :: prof

  call profiling_in(prof, "pesMask_ar_polar")

  PUSH_SUB(pes_mask_output_ar_polar_M)

  ! we do the calculation assuming the polarization along the x-axis 
  vref = M_ZERO
  vref(1) = M_ONE
  !the rotation matrix from the x-axis to the actual polarization direction
  call generate_rotation_matrix(rotation, vref, dir)
  
  
  ll = 1
  do ii = 1, dim
    ll(ii) = size(pesK,ii) 
  end do

  step= Estep
  nn  = int(Emax/step)

  Nth = 300 
  Dtheta = M_PI/Nth

  Nphi = 360
  Dphi = M_TWO * M_PI/Nphi

  SAFE_ALLOCATE(pesM(1:Nth,1:nn))
  pesM = M_ZERO


  !in 1D we do not interpolate 
  if (  (dim  ==  1) ) then 
    message(1)="Impossible to obtain angle-dependent quantities in 1D."
    call messages_fatal(1)

  else

    call pes_mask_interpolator_init(pesK, Lk, dim, cube_f, interp)

    select case(dim)
    case(2)

  


    case(3)

      do ith = 1, Nth
        theta = (ith - 1) * Dtheta 
        do ie = 1, nn
          EE = (ie -1) * step

          do iph = 0, Nphi
            phi = iph * Dphi

            KK(2) = sqrt(M_TWO*EE)*sin(theta)*cos(phi) 
            KK(3) = sqrt(M_TWO*EE)*sin(theta)*sin(phi)
            KK(1) = sqrt(M_TWO*EE)*cos(theta)
            
            !sometimes the interpolator gives negative values therefore the abs()  
            pesM(ith,ie) = pesM(ith,ie) + &
                          abs(qshep_interpolate(interp, cube_f, matmul(rotation,KK(1:3)) ))
          end do
          pesM(ith,ie) = pesM(ith,ie) * sqrt(M_TWO*EE)     

        end do
      end do

      pesM = pesM * Dphi

    end select

    call pes_mask_interpolator_end(cube_f, interp)

  end if


  eGrid(1)= M_ZERO
  eGrid(2)= Emax
  eGrid(3)= step

  thGrid(1)= M_ZERO
  thGrid(2)= M_PI
  thGrid(3)= Dtheta

  phiBounds(1) = M_ZERO
  phiBounds(2) = M_TWO * M_PI

  call  pes_mask_write_2D_map(file, pesM, 2, thGrid, eGrid, dir, phiBounds)

  SAFE_DEALLOCATE_A(pesM)

  POP_SUB(pes_mask_output_ar_polar_M)

  call profiling_out(prof)

end subroutine pes_mask_output_ar_polar_M


! ---------------------------------------------------------
subroutine pes_mask_output_ar_plane_M(pesK, file, Lk, dim, dir, Emax, Estep)
  FLOAT,            intent(in) :: pesK(:,:,:)
  character(len=*), intent(in) :: file
  FLOAT,            intent(in) :: Lk(:)
  integer,          intent(in) :: dim
  FLOAT,            intent(in) :: Emax
  FLOAT,            intent(in) :: Estep
  FLOAT,            intent(in) :: dir(:) 

  integer :: ii, ix, iy
  FLOAT ::  KK(3)

  integer :: nn, nx, ny
  FLOAT  :: step, eGrid(3)
  FLOAT, allocatable ::  pesM(:,:)

  ! needed for interpolation in 2D and 3D 
  FLOAT, pointer :: cube_f(:)
  type(qshep_t) :: interp

  FLOAT :: Dphi, theta, phi, Ex, Ey, EE, phiBounds(2)
  integer :: Nphi, iph
  integer :: ll(1:3)
  FLOAT :: vref(1:dim), rotation(1:dim,1:dim)
  type(profile_t), save :: prof

  call profiling_in(prof, "PESMask_ar_plane")

  PUSH_SUB(pes_mask_output_ar_plane_M)

  ! we do the calculation assuming the polarization along the x-axis 
  vref = M_ZERO
  vref(1) = M_ONE
  !the rotation matrix from the x-axis to the actual polarization direction
  call generate_rotation_matrix(rotation, vref, dir)
  
  
  ll = 1
  do ii = 1, dim
    ll(ii) = size(pesK,ii) 
  end do

  step= Estep
  nn  = int(Emax/step)

  nx = 2*nn
  ny = nn  


  Nphi = 360
  Dphi = M_TWO * M_PI/Nphi

  SAFE_ALLOCATE(pesM(1:2*nn,1:nn))
  pesM = M_ZERO


  !in 1D we do not interpolate 
  if (  (dim  ==  1) ) then 
    message(1)="Impossible to obtain angle-dependent quantities in 1D."
    call messages_fatal(1)

  else

    call pes_mask_interpolator_init(pesK, Lk, dim, cube_f, interp)

    select case(dim)
    case(2)

      do ix = 1, nx
        Ex = (ix -nx/2 -1)*step 
        do iy = 1, ny
          Ey = (iy-1)*step
          EE =sqrt(Ex**2 + Ey**2)
          theta = atan2(Ey,Ex)

          KK(1) = sqrt(2*EE)*cos(theta) 
          KK(2) = sqrt(2*EE)*sin(theta)

          pesM(ix,iy) = pesM(ix,iy) + qshep_interpolate(interp, cube_f, KK(1:2))

        end do
      end do


    case(3)

      do ix = 1, nx
        Ex = (ix -nx/2 -1)*step 
        do iy = 1, ny
          Ey = (iy-1)*step
          EE =sqrt(Ex**2 + Ey**2)

          do iph = 0, Nphi
            phi = iph * Dphi
            theta = atan2(Ey,Ex)

            KK(2) = sqrt(M_TWO*EE)*sin(theta)*cos(phi) 
            KK(3) = sqrt(M_TWO*EE)*sin(theta)*sin(phi)
            KK(1) = sqrt(M_TWO*EE)*cos(theta)
            
            !sometimes the interpolator gives negative values therefore the abs()  
            pesM(ix,iy) = pesM(ix,iy) + &
                          abs(qshep_interpolate(interp, cube_f, matmul(rotation,KK(1:3)) ))
          end do
          pesM(ix,iy) = pesM(ix,iy) * sqrt(M_TWO*EE)     

        end do
      end do

      pesM = pesM * Dphi

    end select

    call pes_mask_interpolator_end(cube_f, interp)

  end if


  eGrid(1)= M_ZERO
  eGrid(2)= Emax
  eGrid(3)= step
  
  phiBounds(1) = M_ZERO
  phiBounds(2) = M_TWO * M_PI
  
  call pes_mask_write_2D_map(file, pesM, 1, eGrid, eGrid, dir, phiBounds)

  SAFE_DEALLOCATE_A(pesM)

  POP_SUB(pes_mask_output_ar_plane_M)

  call profiling_out(prof)

end subroutine pes_mask_output_ar_plane_M

! ---------------------------------------------------------
subroutine pes_mask_output_ar_spherical_cut_M(pesK, file, Lk, dim, dir, Emin, Emax, Estep)
  FLOAT,            intent(in) :: pesK(:,:,:)
  character(len=*), intent(in) :: file
  FLOAT,            intent(in) :: Lk(:)
  integer,          intent(in) :: dim
  FLOAT,            intent(in) :: Emin
  FLOAT,            intent(in) :: Emax
  FLOAT,            intent(in) :: Estep
  FLOAT,            intent(in) :: dir(:) 

  integer :: ii
  FLOAT ::  KK(3)

  integer :: nn, ie
  FLOAT  :: step
  FLOAT, allocatable ::  pesM(:,:)

  ! needed for interpolation in 2D and 3D 
  FLOAT, pointer :: cube_f(:)
  type(qshep_t) :: interp

  FLOAT :: Dtheta, Dphi, theta, phi, EE
  integer :: Nphi, ith, iph, Nth
  integer :: ll(1:3)
  FLOAT :: vref(1:dim), rotation(1:dim,1:dim)
  FLOAT :: phGrid(3), thGrid(3), eBounds(2)
  type(profile_t), save :: prof

  call profiling_in(prof, "PESMask_ar_spherical_cut")

  PUSH_SUB(pes_mask_output_ar_spherical_cut_M)

  ! by default the polarization is along the z-axis 
  vref = M_ZERO
  vref(3) = M_ONE
  !the rotation matrix from the x-axis to the actual polarization direction
  call generate_rotation_matrix(rotation, vref, dir)
  
  
  ll = 1
  do ii = 1, dim
    ll(ii) = size(pesK,ii) 
  end do

  step= Estep
  nn  = int(abs(Emax-Emin)/step)

  Nth = 300 
  Dtheta = M_PI/Nth

  Nphi = 360
  Dphi = M_TWO * M_PI/Nphi

  SAFE_ALLOCATE(pesM(1:Nphi,1:Nth))
  pesM = M_ZERO


  !in 1D we do not interpolate 
  if (  (dim  ==  1) ) then 
    message(1)="Impossible to obtain angle-dependent quantities in 1D."
    call messages_fatal(1)

  else

    call pes_mask_interpolator_init(pesK, Lk, dim, cube_f, interp)

    select case(dim)
    case(2)
  


    case(3)

      do iph = 1, Nphi
        phi = (iph - 1) * Dphi

        do ith = 1, Nth
          theta = (ith - 1) * Dtheta 

          do ie = 0, nn
            EE = Emin + ie * step

            KK(1) = sqrt(M_TWO*EE)*sin(theta)*cos(phi) 
            KK(2) = sqrt(M_TWO*EE)*sin(theta)*sin(phi)
            KK(3) = sqrt(M_TWO*EE)*cos(theta)
            
            !sometimes the interpolator gives negative values therefore the abs()  
            pesM(iph,ith) = pesM(iph,ith) + &
                          abs(qshep_interpolate(interp, cube_f, matmul(rotation,KK(1:3)) ))
          end do
          pesM(iph,ith) = pesM(iph,ith) * sqrt(M_TWO*EE)     

        end do
      end do

      pesM = pesM * Dphi

    end select

    call pes_mask_interpolator_end(cube_f, interp)

  end if


  phGrid(1)= M_ZERO
  phGrid(2)= M_TWO * M_PI
  phGrid(3)= Dphi

  thGrid(1)= M_ZERO
  thGrid(2)= M_PI
  thGrid(3)= Dtheta

  eBounds(1) = Emin
  eBounds(2) = Emax

  call  pes_mask_write_2D_map(file, pesM, 4, phGrid, thGrid, dir, eBounds)

  SAFE_DEALLOCATE_A(pesM)

  POP_SUB(pes_mask_output_ar_spherical_cut_M)

  call profiling_out(prof)

end subroutine pes_mask_output_ar_spherical_cut_M


! ========================================================================
!>  Common interface to write 2D maps in gnuplot with header files for 
!!  different objects. The modes are:
!!
!!  - 1 Angle- and energy-resolved on cartesian coordinates
!!  - 2 Angle- and energy-resolved in polar coordinates
!!  - 3 Velocity map on a plane
!!  - 4 Spherical cut
!
! ========================================================================
subroutine pes_mask_write_2D_map(file, pesM, mode, xGrid, yGrid, vv, intSpan)
  character(len=*), intent(in) :: file
  FLOAT,            intent(in) :: pesM(:,:)
  integer,          intent(in) :: mode
  FLOAT,            intent(in) :: xGrid(:)   !< max min and step for the x axis
  FLOAT,            intent(in) :: yGrid(:)   !< max min and step for the y axis
  FLOAT,            intent(in) :: vv(:)      !< for mode=1,2 indicate the Zenith axis for mode 3 the cutting plane
  FLOAT, optional,  intent(in) :: intSpan(:) !< for integrated quantities indicate the integral region    

  integer :: nx,ny, iunit, ix,iy

  PUSH_SUB(pes_mask_write_2D_map)

  nx = size(pesM,1)
  ny = size(pesM,2)

  iunit = io_open(file, action='write')

  select case (mode)
    case(1)
      !!Angle Energy Cartesian
        write(iunit, '(a)') '##################################################'

        write(iunit, '(a)') '#'        
        write(iunit, '(a1,a20,a1,f10.2,a2,f10.2,a2,f10.2,a1)')&
                                              "#"," Zenith axis: ","(",&
                                              vv(1),", ",vv(2),", ",vv(3),")"
        write(iunit, '(a1,a20,a1,f10.2,a2,f10.2,a1)')&
                                              "#"," Integrated in phi: ","(",&
                                              intSpan(1),", ",intSpan(2),")"
        write(iunit, '(a)') '#'        


        write(iunit, '(a1,a19,2x,a19,2x,a19)') '#', str_center("E*cos(th)", 19),&
                                              str_center("E*sin(th)", 19),&
                                              str_center("P(E)", 19)
        write(iunit, '(a1,a19,2x,a19,2x,a19)') &
          '#', str_center('['//trim(units_abbrev(units_out%energy)) // ']', 19), &
               str_center('['//trim(units_abbrev(units_out%energy)) // ']', 19), & 
               str_center('[1/' //trim(units_abbrev(units_out%energy))//']', 19)
        write(iunit, '(a)') '##################################################'
    case (2)
    !!Angle Energy polar
      write(iunit, '(a)') '##################################################'

      write(iunit, '(a)') '#'        
      write(iunit, '(a1,a20,a1,f10.2,a2,f10.2,a2,f10.2,a1)')&
                                            "#"," Zenith axis: ","(",&
                                            vv(1),", ",vv(2),", ",vv(3),")"
      write(iunit, '(a1,a20,a1,f10.2,a2,f10.2,a1)')&
                                            "#"," Integrated in phi: ","(",&
                                            intSpan(1),", ",intSpan(2),")"
      write(iunit, '(a)') '#'        
      
      
      write(iunit, '(a1,a19,2x,a19,2x,a19)') '#', str_center("Theta", 19),&
                                            str_center("E", 19),&
                                            str_center("P(E)", 19)
      write(iunit, '(a1,a19,2x,a19,2x,a19)') &
        '#', str_center('[rad]', 19), &
             str_center('['//trim(units_abbrev(units_out%energy)) // ']', 19), & 
             str_center('[1/' //trim(units_abbrev(units_out%energy))//']', 19)
      write(iunit, '(a)') '##################################################'

    case (3)
    !!Velocity map
      write(iunit, '(a)') '##################################################'
      write(iunit, '(a1,a19,2x,a19,2x,a19)') '#', str_center("p1", 19),&
                                            str_center("p2", 19),&
                                            str_center("P(p)", 19)
      write(iunit, '(a1,a19,2x,a19,2x,a19)') &
        '#', str_center('[sqrt('//trim(units_abbrev(units_out%energy)) // ')]', 19), & 
             str_center('[sqrt('//trim(units_abbrev(units_out%energy)) // ')]', 19), & 
             str_center('[1/' //trim(units_abbrev(units_out%energy))//']', 19)
      write(iunit, '(a)') '##################################################'

      case (4)
      !!Spherical Cut
        write(iunit, '(a)') '##################################################'

        write(iunit, '(a)') '#'        
        write(iunit, '(a1,a20,a1,f10.2,a2,f10.2,a2,f10.2,a1)')&
                                              "#"," Zenith axis: ","(",&
                                              vv(1),", ",vv(2),", ",vv(3),")"
        write(iunit, '(a1,a20,a1,es19.12,a2,es19.12,a1,a19)')&
                                              "#"," Integrated in E: ","(",&
                                              intSpan(1),", ",intSpan(2),")",&
                      str_center('['//trim(units_abbrev(units_out%energy)) // ']', 19)
        write(iunit, '(a)') '#'        
      
      
        write(iunit, '(a1,a19,2x,a19,2x,a19)') '#', str_center("Phi", 19),&
                                              str_center("Theta", 19),&
                                              str_center("P(Phi,Theta)", 19)
        write(iunit, '(a1,a19,2x,a19,2x,a19)') &
          '#', str_center('[rad]', 19), &
               str_center('[rad]', 19), & 
               str_center('[1/' //trim(units_abbrev(units_out%energy))//']', 19)
        write(iunit, '(a)') '##################################################'

    
  end select

  do ix = 1, nx
    do iy = 1, ny

      select case (mode)
        case (1)
          write(iunit, '(es19.12,2x,es19.12,2x,es19.12)') &
                  units_from_atomic(units_out%energy, (ix -nx/2 - 1) * xGrid(3) + xGrid(1) ),&
                  units_from_atomic(units_out%energy, (iy - 1) * yGrid(3) + yGrid(1) ), pesM(ix,iy)
        case(2)
          write(iunit, '(es19.12,2x,es19.12,2x,es19.12)') &   
                  (ix - 1) * xGrid(3) + xGrid(1), &
                  units_from_atomic(units_out%energy, (iy - 1) * yGrid(3) + yGrid(1) ), pesM(ix,iy)
        case(3)
          write(iunit, '(es19.12,2x,es19.12,2x,es19.12)') &   
                  (ix - 1) * xGrid(3) + xGrid(1), &
                  (iy - 1) * yGrid(3) + yGrid(1), pesM(ix,iy)

        case(4)
          write(iunit, '(es19.12,2x,es19.12,2x,es19.12)') &   
                  (ix - 1) * xGrid(3) + xGrid(1), &
                  (iy - 1) * yGrid(3) + yGrid(1), pesM(ix,iy)
    
      end select

    end do
    write(iunit,*) 
  end do
  
  call io_close(iunit)


  POP_SUB(pes_mask_write_2D_map)
end subroutine pes_mask_write_2D_map




! ---------------------------------------------------------
subroutine pes_mask_output_power_totalM(pesK, file, Lk, dim, Emax, Estep, interpolate)
  FLOAT,            intent(in) :: pesK(:,:,:)
  character(len=*), intent(in) :: file
  FLOAT,            intent(in) :: Lk(:)
  integer,          intent(in) :: dim
  FLOAT,            intent(in) :: Emax
  FLOAT,            intent(in) :: Estep
  logical,          intent(in) :: interpolate

  integer :: ii, ix, iy, iz
  FLOAT ::  KK(3),vec

  integer :: nn
  FLOAT  :: step
  FLOAT, allocatable :: npoints(:), pes(:)

  ! needed for interpolation in 2D and 3D 
  FLOAT, pointer :: cube_f(:)
  type(qshep_t) :: interp

  FLOAT :: Dtheta, Dphi, theta, phi,EE
  integer :: Ntheta, Nphi, ith, iph
  integer :: ll(3)

  PUSH_SUB(pes_mask_output_power_totalM)

  ll = 1
  do ii = 1, dim
    ll(ii) = size(pesK,ii) 
  end do

  step= Estep
  nn  = int(Emax/step)


  Ntheta = 360
  Dtheta = M_TWO*M_PI/Ntheta

  Nphi = 180
  Dphi = M_PI/Nphi

  SAFE_ALLOCATE(pes(1:nn))
  pes = M_ZERO

  SAFE_ALLOCATE(npoints(1:nn))
  npoints = M_ZERO

  !in 1D we do not interpolate 
  if ( (.not. interpolate) .or.  (dim  ==  1) ) then 

    do ix = 1,ll(1)
      KK(1) = Lk(ix)
      do iy = 1, ll(2)
        KK(2) = Lk(iy)
        do iz = 1, ll(3)
          KK(3) = Lk(iz)

          if(KK(1) /= 0 .or. KK(2) /= 0 .or. KK(3) /= 0) then
            ! the power spectrum
            vec = sum(KK(1:dim)**2) / M_TWO
            ii = int(vec / step) + 1

            if(ii <= nn) then

              pes(ii) = pes(ii)+pesK(ix,iy,iz)
              npoints(ii) = npoints(ii) + M_ONE

            end if
          end if

        end do
      end do
    end do

    do ii = 2, nn
      ! npoints==0.0 when pes==0.0
      if(pes(ii)/=0.0)then
        EE = (ii-1)*step
        !Multiply for the correct Jacobian factor
        pes(ii) = pes(ii)*sqrt(M_TWO*EE)**(dim - 2) 
        pes(ii) = pes(ii) / npoints(ii)
      end if
    end do


    ! Interpolate the output
  else

    call pes_mask_interpolator_init(pesK, Lk, dim, cube_f, interp)

    select case(dim)
    case(2)

      do ii = 1, nn
        EE = (ii-1)*step
        do ith = 0, Ntheta
          theta = ith*Dtheta
          KK(1) = sqrt(2*EE)*cos(theta) 
          KK(2) = sqrt(2*EE)*sin(theta)
          pes(ii) = pes(ii) + qshep_interpolate(interp, cube_f, KK(1:2))
        end do
      end do

      pes = pes * Dtheta

    case(3)

      do ii = 1, nn
        EE = (ii-1)*step
        do ith = 0, Ntheta
          theta = ith * Dtheta
          do iph = 0, Nphi
            phi = iph * Dphi

            KK(2) = sqrt(M_TWO*EE)*sin(phi)*cos(theta) 
            KK(3) = sqrt(M_TWO*EE)*sin(phi)*sin(theta)
            KK(1) = sqrt(M_TWO*EE)*cos(phi)

            pes(ii) = pes(ii) + qshep_interpolate(interp, cube_f, KK(1:3))
          end do
        end do
        pes(ii) = pes(ii) * sqrt(M_TWO*EE)    
      end do

      pes = pes * Dtheta * Dphi

    end select

    call pes_mask_interpolator_end(cube_f, interp)

  end if


  if (interpolate) then 
    call pes_mask_write_power_total(file, step, pes)
  else 
    call pes_mask_write_power_total(file, step, pes, npoints)
  end if

  SAFE_DEALLOCATE_A(pes)
  SAFE_DEALLOCATE_A(npoints)

  POP_SUB(pes_mask_output_power_totalM)


end subroutine pes_mask_output_power_totalM


! ---------------------------------------------------------
subroutine pes_mask_output_power_total(mask, st, file, wfAk)
  type(pes_mask_t), intent(in) :: mask
  type(states_t),   intent(in) :: st
  character(len=*), intent(in) :: file
  CMPLX, optional,  intent(in) :: wfAk(:,:,:,:,:,:)

  integer :: ist, ik, ii, ix, iy, iz, idim
  FLOAT ::  KK(3),vec

  integer :: nn
  FLOAT  :: step,DE
  FLOAT, allocatable :: npoints(:), pes(:)

  ! needed for interpolation in 2D and 3D 
  FLOAT, allocatable :: cube_f(:), kx(:),ky(:),kz(:)
  FLOAT :: Dtheta, Dphi, theta, phi,EE
  type(qshep_t) :: interp
  integer :: np, Ntheta, Nphi, ith, iph


  type(profile_t), save :: prof
  call profiling_in(prof, "PESMASK_output")

  PUSH_SUB(pes_mask_output_power_total)


  step = mask%energyStep
  nn  = int(mask%energyMax/step)

  DE = (mask%Lk(2)-mask%Lk(1))**2/M_TWO
  
  np = mask%ll(1)*mask%ll(2)*mask%ll(3)  

  Ntheta = 36
  Dtheta = M_TWO*M_PI/Ntheta

  Nphi = 18
  Dphi = M_PI/Nphi

  SAFE_ALLOCATE(pes(1:nn))
  pes = M_ZERO

  SAFE_ALLOCATE(npoints(1:nn))
  npoints = M_ZERO

  !in 1D we do not interpolate 
  if ((.not. mask%interpolate_out) .or. (mask%mesh%sb%dim  ==  1) ) then 

    do ix = 1, mask%ll(1)
      KK(1) = mask%Lk(ix)
      do iy = 1, mask%ll(2)
        KK(2) = mask%Lk(iy)
        do iz = 1, mask%ll(3)
          KK(3) = mask%Lk(iz)
          
          if(KK(1) /= 0 .or. KK(2) /= 0 .or. KK(3) /= 0) then
            ! the power spectrum
            vec = sum(KK(1:mask%mesh%sb%dim)**2) / M_TWO
            ii = int(vec / step) + 1
            
            if(ii <= nn) then
              do ik = 1,st%d%nik
                do ist = 1, st%nst
                  if(present(wfAk))then
                    pes(ii) = pes(ii) + st%occ(ist, ik) * &
                      sum(abs(mask%k(ix, iy, iz, :, ist, ik) + wfAk(ix,iy,iz,:, ist, ik)  )**2)
                  else 
                    pes(ii) = pes(ii) + st%occ(ist, ik) * sum(abs(mask%k(ix, iy, iz, :, ist, ik)  )**2)
                  end if
                end do
              end do
              npoints(ii) = npoints(ii) + M_ONE
            end if
            !Multiply for the correct Jacobian factor
!            pes(ii) = pes(ii)*sqrt(2*vec)**(mask%mesh%sb%dim - 2) 
          end if
          
        end do
      end do
    end do
    

  ! Interpolate the output
  else

    select case(mask%mesh%sb%dim)
      case(2)
      
        SAFE_ALLOCATE(cube_f(1:np))    
        SAFE_ALLOCATE(kx(1:np))    
        SAFE_ALLOCATE(ky(1:np))    
        
        cube_f = M_ZERO
        kx = M_ZERO
        ky = M_ZERO
        npoints = M_ONE
        
        ii=1
        do ix = 1, mask%ll(1)
          KK(1) = mask%Lk(ix)
          do iy = 1, mask%ll(2)
            KK(2) = mask%Lk(iy)
            do iz = 1, mask%ll(3)
              KK(3) = mask%Lk(iz)
              
              do ik = 1,st%d%nik
                do ist = 1, st%nst
                  if (present(wfAk)) then
                    cube_f(ii) = cube_f(ii) + st%occ(ist, ik) * &
                      sum(abs(mask%k(ix, iy, iz, :, ist, ik) + wfAk(ix,iy,iz,:, ist, ik)  )**2)
                  else
                    cube_f(ii) = cube_f(ii) + st%occ(ist, ik) * sum(abs(mask%k(ix, iy, iz, :, ist, ik)  )**2)
                  end if
                end do
              end do
              
              kx(ii) = KK(1)
              ky(ii) = KK(2)
              
              ii = ii +1
            end do
          end do
        end do
        
        call qshep_init(interp, np, cube_f, kx, ky)
        
        do ii = 1, nn
          EE = ii*step
          do ith = 0, Ntheta
            theta = ith*Dtheta
            KK(1) = sqrt(2*EE)*cos(theta) 
            KK(2) = sqrt(2*EE)*sin(theta)
            pes(ii) = pes(ii) + qshep_interpolate(interp, cube_f, KK(1:2))
          end do
!        pes(ii) = pes(ii) * sqrt(2*EE)      
          pes(ii) = pes(ii)      
        end do
        
        pes = pes * Dtheta
        
        
        call qshep_end(interp)
        
        SAFE_DEALLOCATE_A(cube_f)    
        SAFE_DEALLOCATE_A(kx)    
        SAFE_DEALLOCATE_A(ky)    
        
      case(3)

        SAFE_ALLOCATE(cube_f(1:np))    
        SAFE_ALLOCATE(kx(1:np))    
        SAFE_ALLOCATE(ky(1:np))    
        SAFE_ALLOCATE(kz(1:np))    
        
        cube_f = M_ZERO
        kx = M_ZERO
        ky = M_ZERO
        kz = M_ZERO
        npoints = M_ONE
        
        ii=1
        do ix = 1, mask%ll(1)
          KK(1) = mask%Lk(ix)
          do iy = 1, mask%ll(2)
            KK(2) = mask%Lk(iy)
            do iz = 1, mask%ll(3)
              KK(3) = mask%Lk(iz)
              
              do ik = 1,st%d%nik
                do ist = 1, st%nst
                  if (present(wfAk)) then
                    cube_f(ii) = cube_f(ii) + st%occ(ist, ik) * &
                      sum(abs(mask%k(ix, iy, iz, :, ist, ik) + wfAk(ix,iy,iz,:, ist, ik)  )**2)
                  else
                    cube_f(ii) = cube_f(ii) + st%occ(ist, ik) * sum(abs(mask%k(ix, iy, iz, :, ist, ik)  )**2)
                  end if
                end do
              end do
              
              kx(ii) = KK(1)
              ky(ii) = KK(2)
              kz(ii) = KK(3)
              
              ii = ii +1
            end do
          end do
        end do
        
        call qshep_init(interp, np, cube_f, kx, ky, kz)

        do ii = 1, nn
          EE = ii*step
          do ith = 0, Ntheta
            theta = ith * Dtheta
            do iph = 0, Nphi
              phi = iph * Dphi
              KK(1) = sqrt(2*EE)*sin(phi)*cos(theta) 
              KK(2) = sqrt(2*EE)*sin(phi)*sin(theta)
              KK(3) = sqrt(2*EE)*cos(phi)
              pes(ii) = pes(ii) + qshep_interpolate(interp, cube_f, KK(1:3))
              pes(ii) = pes(ii) * sin(phi)
            end do
          end do
!        pes(ii) = pes(ii) * 2*EE    
          pes(ii) = pes(ii) * sqrt(2*EE)    
        end do

        pes = pes * Dtheta * Dphi
        
        call qshep_end(interp)
        
        SAFE_DEALLOCATE_A(cube_f)    
        SAFE_DEALLOCATE_A(kx)    
        SAFE_DEALLOCATE_A(ky)    
        SAFE_DEALLOCATE_A(kz)    
        
    end select
    
  end if
  
  
  do idim=1, mask%mesh%sb%dim
    pes = pes *( mask%spacing(idim)/sqrt(M_TWO*M_PI))**2
  end do
  
  if (mask%interpolate_out) then 
    call pes_mask_write_power_total(file, step, pes)
  else 
    call pes_mask_write_power_total(file, step, pes, npoints)
  end if
  
  
  SAFE_DEALLOCATE_A(pes)
  SAFE_DEALLOCATE_A(npoints)
  
  POP_SUB(pes_mask_output_power_total)
  
  call profiling_out(prof)
  
end subroutine pes_mask_output_power_total


! ---------------------------------------------------------
subroutine pes_mask_write_power_total(file, step, pes, npoints)
  character(len=*), intent(in) :: file
  FLOAT,            intent(in) :: step
  FLOAT,            intent(in) :: pes(:)
  FLOAT, optional,  intent(in) :: npoints(:)

  integer :: nn, iunit, ii

  PUSH_SUB(pes_mask_write_power_total)

  nn = size(pes,1)

  iunit = io_open(file, action='write')

  !!Header
  write(iunit, '(a)') '##################################################'
  write(iunit, '(a1,a18,a18)') '#', str_center("E", 18), str_center("P(E)", 18)
  write(iunit, '(a1,a18,a18)') &
    '#', str_center('['//trim(units_abbrev(units_out%energy)) // ']', 18), &
    str_center('[1/' //trim(units_abbrev(units_out%energy))//']', 18)
  write(iunit, '(a)') '##################################################'
 

  do ii = 1, nn
    if(present(npoints)) then
      if(npoints(ii) > 0) then
        write(iunit, '(es19.12,2x,es19.12,2x,es19.12)')  units_from_atomic(units_out%energy, (ii - 1) * step), pes(ii), npoints(ii)
      end if
    else
      write(iunit, '(es19.12,2x,es19.12)')  units_from_atomic(units_out%energy, (ii - 1) * step), pes(ii)
    end if
  end do
  
  call io_close(iunit)


  POP_SUB(pes_mask_write_power_total)
end subroutine pes_mask_write_power_total


  
! ---------------------------------------------------------
!
!> This routine is the main routine dedicated to the output 
!! of PES data
!
! ---------------------------------------------------------
subroutine pes_mask_output(mask, mesh, st, outp, file, gr, geo, iter)
  type(pes_mask_t),  intent(in)    :: mask
  type(mesh_t),      intent(in)    :: mesh
  type(states_t),    intent(in)    :: st
  character(len=*),  intent(in)    :: file
  type(output_t),    intent(in)    :: outp
  type(grid_t),      intent(inout) :: gr
  type(geometry_t),  intent(in)    :: geo
  integer,           intent(in)    :: iter

  CMPLX, allocatable :: wfAk(:,:,:,:,:,:) 
  FLOAT :: pesK(1:mask%fs_n_global(1),1:mask%fs_n_global(2),1:mask%fs_n_global(3)),pol(3)
  integer :: ist, ik, idim, ierr, st1, st2, k1, k2
  character(len=100) :: fn
  character(len=256) :: dir
  type(cube_function_t) :: cf1

  type(profile_t), save :: prof

  PUSH_SUB(pes_mask_output)
  call profiling_in(prof, "PESMASK_out")
  
  !Output info for easy post-process
  if(mpi_grp_is_root(mpi_world)) call pes_mask_write_info(mask, "td.general")
 

  !Photoelectron wavefunction and density in real space
  if(iand(outp%what, C_OUTPUT_PES_WFS) /= 0  .or.  iand(outp%what, C_OUTPUT_PES_DENSITY) /= 0 ) then
    write(dir, '(a,i7.7)') "td.", iter  ! name of directory
    call  pes_mask_output_states(st, gr, geo, dir, outp, mask)
  end if

  !Write the output in the td.00iter directories
  dir = file 
  if(iand(outp%what, C_OUTPUT_PES) /= 0 ) then
    write(dir, '(a,i7.7,a)') "td.", iter,"/PESM"  ! name of directory
  end if

 !The contribution of \Psi_A(x,t2) to the PES 
  if(mask%add_psia) then 
    st1 = st%st_start
    st2 = st%st_end
    k1 = st%d%kpt%start
    k2 = st%d%kpt%end
    SAFE_ALLOCATE(wfAk(1:mask%ll(1), 1:mask%ll(2), 1:mask%ll(3),1:st%d%dim,st1:st2,k1:k2))
    wfAk = M_z0
  
    call cube_function_null(cf1)    
    call zcube_function_alloc_RS(mask%cube, cf1, force_alloc = .true.) 
    call  cube_function_alloc_FS(mask%cube, cf1, force_alloc = .true.) 

    do ik = st%d%kpt%start, st%d%kpt%end
      do ist =  st%st_start, st%st_end
        do idim = 1, st%d%dim
          call pes_mask_mesh_to_cube(mask, st%zpsi(:, idim, ist, ik), cf1)
          cf1%zRs = (M_ONE - mask%cM%zRs**10) * cf1%zRs ! mask^10 is practically a box function
          call pes_mask_X_to_K(mask,mesh,cf1%zRs,cf1%Fs)
          wfAk(:,:,:,idim, ist, ik) = cf1%Fs
        end do
      end do
    end do

    call zcube_function_free_RS(mask%cube, cf1)
    call  cube_function_free_FS(mask%cube, cf1)
  end if 

  !Create the full momentum-resolved PES matrix
  pesK = M_ZERO
  if(mask%add_psia) then 
    call pes_mask_create_full_map(mask, st, pesK, wfAk)
  else 
    call pes_mask_create_full_map(mask, st, pesK)
  end if
  
  if(mpi_grp_is_root(mpi_world)) then ! only root node writes the output
    ! Output the full matrix in binary format for subsequent post-processing 
    write(fn, '(a,a)') trim(dir), '_map.obf'
    call io_binary_write(io_workpath(fn), mask%fs_n_global(1)*mask%fs_n_global(2)*mask%fs_n_global(3), pesK, ierr)

    ! Output the k resolved PES on plane kz=0
    write(fn, '(a,a)') trim(dir), '_map.z=0'
    pol = (/M_ZERO, M_ZERO, M_ONE/)
    call pes_mask_output_full_mapM_cut(pesK, fn, mask%Lk, mask%mesh%sb%dim, pol = pol, &
                                     dir = 3, integrate = INTEGRATE_NONE )

    ! Total power spectrum 
    write(fn, '(a,a)') trim(dir), '_power.sum'
    call pes_mask_output_power_totalM(pesK,fn, mask%Lk, mask%mesh%sb%dim, mask%energyMax, mask%energyStep, mask%interpolate_out)

  end if

  if(mask%add_psia) then 
   SAFE_DEALLOCATE_A(wfAk)
  end if
  call profiling_out(prof)
  
  POP_SUB(pes_mask_output)
end subroutine pes_mask_output

! ---------------------------------------------------------
!> Read pes info.
! ---------------------------------------------------------
subroutine pes_mask_read_info(dir, dim, Emax, Estep, ll, Lk,RR)
  character(len=*), intent(in)  :: dir
  integer,          intent(out) :: dim  
  FLOAT,            intent(out) :: Emax
  FLOAT,            intent(out) :: Estep
  integer,          intent(out) :: ll
  FLOAT, pointer,   intent(out) :: Lk(:)
  FLOAT, pointer,   intent(out) :: RR(:)


  character(len=256) :: filename,dummy

  integer :: iunit,ii


  PUSH_SUB(pes_mask_read_info)


  filename = trim(dir)//'pes'
  iunit = io_open(filename, action='read', status='old', is_tmp = .true.)

  SAFE_ALLOCATE(RR(1:2))

  rewind(iunit)
  read(iunit, *) dummy, dim
  read(iunit,'(a10,2x,es19.12)') dummy, RR(1)
  read(iunit,'(a10,2x,es19.12)') dummy, RR(2)
  read(iunit, *) dummy, Emax
  read(iunit, *) dummy, Estep
  read(iunit, *) 

  read(iunit, *) dummy, ll
   
  SAFE_ALLOCATE(lk(1:ll))

  do ii=1, ll
    read(iunit, '(es19.12)') Lk(ii)
  end do 


  call io_close(iunit)       

  POP_SUB(pes_mask_read_info)
end subroutine pes_mask_read_info


! ---------------------------------------------------------
!> Output pes info
! ---------------------------------------------------------
subroutine pes_mask_write_info(mask, dir)
  type(pes_mask_t), intent(in) :: mask
  character(len=*), intent(in) :: dir

  character(len=256) :: filename

  integer :: iunit,ii

  PUSH_SUB(pes_mask_write_info)

  filename = trim(dir)//'/pes'

  iunit = io_open(filename, action='write', is_tmp = .true.)

  write(iunit, '(a10,2x,i2)') 'dim', mask%mesh%sb%dim
  write(iunit, '(a10,2x,es19.12)') 'Mask R1', mask%mask_R(1)
  write(iunit, '(a10,2x,es19.12)') 'Mask R2', mask%mask_R(2)
  write(iunit, '(a10,2x,es19.12)') 'Emax', mask%energyMax
  write(iunit, '(a10,2x,es19.12)') 'Estep', mask%energyStep
  write(iunit, '(a)') '-------'

  write(iunit, '(a,2x,i18)') 'nK', mask%ll(1)
  do ii=1, mask%ll(1)
    write(iunit, '(es19.12)')  mask%Lk(ii)
  end do 


  call io_close(iunit)       

  POP_SUB(pes_mask_write_info)
end subroutine pes_mask_write_info


! ---------------------------------------------------------
!
! ---------------------------------------------------------
subroutine pes_mask_dump(restart, mask, st, ierr)
  type(restart_t),  intent(in)  :: restart
  type(pes_mask_t), intent(in)  :: mask
  type(states_t),   intent(in)  :: st
  integer,          intent(out) :: ierr

  character(len=80) :: filename, path, lines(2)
  integer :: itot, ik, ist, idim, ll(3), np, iunit, err, err2
  CMPLX, pointer :: gwf(:,:,:)
  type(profile_t), save :: prof
   
  PUSH_SUB(pes_mask_dump)

  ierr = 0

  if (restart_skip(restart)) then
    POP_SUB(pes_mask_dump)
    return
  end if

  if (in_debug_mode) then
    message(1) = "Debug: Writing PES mask restart."
    call messages_info(1)
  end if

  call profiling_in(prof, "PESMASK_dump")

  iunit = restart_open(restart, 'pes_mask')
  write(lines(1), '(a10,2x,es19.12)') 'Mask R1', mask%mask_r(1)
  write(lines(2), '(a10,2x,es19.12)') 'Mask R2', mask%mask_r(2)
  call restart_write(restart, iunit, lines, 2, err)
  if (err /= 0) ierr = ierr + 1
  call restart_close(restart, iunit)

  ll(1:3) = mask%fs_n_global(1:3)
  np = ll(1)*ll(2)*ll(3) 

  err2 = 0
  do ik = st%d%kpt%start, st%d%kpt%end
    do ist = st%st_start, st%st_end
      do idim = 1, st%d%dim

        itot = ist + (ik - 1)*st%nst +  (idim - 1)*st%nst*st%d%kpt%nglobal

        write(filename,'(i10.10)') itot

        !FIXME: the following should not use io_binary_write directly. Instead, the
        !task of writing the function to the file should be done by the restart module.

        path = trim(restart_dir(restart))//'/pes_'//trim(filename)//'.obf'

        if (mask%cube%parallel_in_domains) then
          SAFE_ALLOCATE(gwf(1:ll(1),1:ll(2),1:ll(3))) 
          !call zcube_function_allgather(mask%cube, gwf, wf, transpose = .true.)
          call zcube_function_allgather(mask%cube, gwf, mask%k(:,:,:, idim, ist, ik), gatherfs = .true.)
          
          !if(mask%pw_map_how .eq. PW_MAP_PFFT) then
          !  call zcube_function_allgather(mask%cube, gwf, wf, order = (/2,3,1/))
          ! 
          !else if(mask%pw_map_how .eq. PW_MAP_PNFFT) then
          !  call zcube_function_allgather(mask%cube, gwf, wf, order = (/2,3,1/))
          !                                      
          !end if
          
        else
          gwf => mask%k(:,:,:, idim, ist, ik)
        end if
        
        !we need to check both cube and mesh mpi_grp to make sure only the root node writes the
        !output and preserve state parallelization 
        if (mpi_grp_is_root(mask%cube%mpi_grp) .and. mpi_grp_is_root(mask%mesh%mpi_grp)) then 
          call io_binary_write(path, np, gwf(:,:,:), err)
          if (err /= 0) then
            err2 = err2 + 1
            message(1) = "Unable to write PES mask restart data to '"//trim(path)//"'."
            call messages_warning(1)
          end if
        end if

        if (mask%cube%parallel_in_domains) then
          SAFE_DEALLOCATE_P(gwf)
        end if

      end do
    end do
  end do
  if (err2 /= 0) ierr = ierr + 2

  if (in_debug_mode) then
    message(1) = "Debug: Writing PES mask restart done."
    call messages_info(1)
  end if

  call profiling_out(prof)

  POP_SUB(pes_mask_dump)
end subroutine pes_mask_dump

! ---------------------------------------------------------
subroutine pes_mask_load(restart, mask, st, ierr)
  type(restart_t),  intent(in)    :: restart
  type(pes_mask_t), intent(inout) :: mask
  type(states_t),   intent(inout) :: st
  integer,          intent(out)   :: ierr

  character(len=80) :: filename, path
  integer :: itot, ik, ist, idim , np, err, err2, iunit, ll(3)
  character(len=128) :: lines(2)
  character(len=7) :: dummy
  FLOAT, allocatable :: rr(:)

  PUSH_SUB(pes_mask_load)

  ierr = 0

  if (restart_skip(restart)) then
    ierr = -1
    POP_SUB(pes_mask_load)
    return
  end if

  if (in_debug_mode) then
    message(1) = "Debug: Reading PES mask restart."
    call messages_info(1)
  end if


  SAFE_ALLOCATE(rr(1:2))
  iunit = restart_open(restart, 'pes_mask')
  call restart_read(restart, iunit, lines, 2, err)
  if (err /= 0) then    
    ierr = ierr + 1
  else
    read(lines(1),'(a10,2x,es19.12)') dummy, rr(1)
    read(lines(2),'(a10,2x,es19.12)') dummy, rr(2)
  end if
  call restart_close(restart, iunit)


  ll(1:3) = mask%ll(1:3)
  np =ll(1)*ll(2)*ll(3) 

  err2 = 0
  do ik = st%d%kpt%start, st%d%kpt%end
    do ist = st%st_start, st%st_end
      do idim = 1, st%d%dim
        itot = ist + (ik-1) * st%nst+  (idim-1) * st%nst*st%d%kpt%nglobal

        write(filename,'(i10.10)') itot

        !FIXME: the following should not use io_binary_read directly. Instead, the
        !task of reading the function to the file should be done by the restart module.
       
        path = trim(restart_dir(restart))//'/pes_'//trim(filename)//'.obf'
        
        call io_binary_read(path,np, mask%k(:,:,:, idim, ist, ik), err)
        if (err /= 0) then
          err2 = err2 + 1
          message(1) = "Unable to read PES mask restart data from '"//trim(path)//"'."
          call messages_warning(1)
        end if

      end do
    end do
  end do
  if (err2 /= 0) ierr = ierr + 2

  if (ierr == 0) then
    if (rr(1) /= mask%mask_r(1) .or. rr(2) /= mask%mask_r(2)) then
      message(1) = "PhotoElectronSpectrum = pes_mask : The mask parameters have changed."
      message(2) = "I will restart mapping from the previous context."
      call messages_warning(2)
      call pes_mask_restart_map(mask, st, rr)
    end if
  end if

  SAFE_DEALLOCATE_A(rr)

  if (in_debug_mode) then
    message(1) = "Debug: Reading PES mask restart done."
    call messages_info(1)
  end if

  POP_SUB(pes_mask_load)
end subroutine pes_mask_load


! ---------------------------------------------------------
subroutine pes_mask_restart_map(mask, st, RR)
  type(pes_mask_t), intent(inout) :: mask
  type(states_t),   intent(inout) :: st
  FLOAT,            intent(in)    :: RR(2)

  integer :: itot, ik, ist, idim, np
  integer :: ll(3)
!   CMPLX, allocatable :: wf1(:,:,:),wf2(:,:,:)
  FLOAT, allocatable :: M_old(:,:,:)
  type(cube_function_t):: cf1,cf2

  PUSH_SUB(pes_mask_restart_map)


  ll(1:3) = mask%ll(1:3)
  np =ll(1)*ll(2)*ll(3)

  SAFE_ALLOCATE(M_old(1:mask%ll(1), 1:mask%ll(2), 1:mask%ll(3)))
  call cube_function_null(cf1)    
  call zcube_function_alloc_RS(mask%cube, cf1, force_alloc = .true.) 
  call  cube_function_alloc_FS(mask%cube, cf1, force_alloc = .true.) 
  call cube_function_null(cf2)    
  call zcube_function_alloc_RS(mask%cube, cf2, force_alloc = .true.)


  call pes_mask_generate_mask_function(mask,mask%mesh,mask%shape, RR, M_old)
  
  itot = 1
  do ik = st%d%kpt%start, st%d%kpt%end
    do ist = st%st_start, st%st_end
      do idim = 1, st%d%dim
        cf1%zRs = M_z0
        call pes_mask_K_to_X(mask, mask%mesh, mask%k(:,:,:, idim, ist, ik), cf1%zRs)
        call pes_mask_mesh_to_cube(mask, st%zpsi(:, idim, ist, ik), cf2)
        cf2%zRs = cf1%zRs + cf2%zRs ! the whole pes orbital in real space 
        cf1%zRs = cf2%zRs* mask%cM%zRs !modify the orbital in A
        call pes_mask_cube_to_mesh(mask, cf1, st%zpsi(:, idim, ist, ik))
        cf2%zRs = cf2%zRs * (mask%cM%zRs-M_old) ! modify the k-orbital in B 
        call pes_mask_X_to_K(mask, mask%mesh, cf2%zRs, cf1%Fs)
        mask%k(:,:,:, idim, ist, ik) = mask%k(:,:,:, idim, ist, ik) - cf1%Fs
      end do
    end do
  end do
  SAFE_DEALLOCATE_A(M_old)
  
  call zcube_function_free_RS(mask%cube, cf1)
  call  cube_function_free_FS(mask%cube, cf1)
  call zcube_function_free_RS(mask%cube, cf2)

  POP_SUB(pes_mask_restart_map)
end subroutine pes_mask_restart_map

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
