!! Copyright (C) 2004 E.S. Kadantsev, M. Marques
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!
!! $Id: linear_solver.F90 13524 2015-03-24 16:53:51Z dstrubbe $

#include "global.h"

module linear_solver_m
  use batch_m
  use batch_ops_m
  use datasets_m
  use derivatives_m
  use global_m
  use grid_m
  use hamiltonian_m
  use lalg_basic_m
  use linear_response_m
  use loct_m
  use parser_m
  use math_m
  use mesh_m
  use mesh_batch_m
  use mesh_function_m
  use messages_m
  use profiling_m
  use preconditioners_m
  use scf_tol_m
  use smear_m
  use solvers_m
  use states_m

  implicit none

  private

  integer, public, parameter :: &
       LS_CG              = 5,  &
       LS_BICGSTAB        = 4,  &
       LS_MULTIGRID       = 7,  &
       LS_QMR_SYMMETRIC   = 81, &
       LS_QMR_SYMMETRIZED = 82, &
       LS_QMR_DOTP        = 83, &
       LS_QMR_GENERAL     = 84, &
       LS_SOS             = 9

  public ::                              &
       linear_solver_t,                  &
       linear_solver_init,               &
       linear_solver_end,                &
       dlinear_solver_solve_HXeY,        & 
       zlinear_solver_solve_HXeY,        &
       dlinear_solver_solve_HXeY_batch,  & 
       zlinear_solver_solve_HXeY_batch,  &
       linear_solver_ops_per_iter,       &
       linear_solver_obsolete_variables

  type linear_solver_t
     integer                :: solver         
     type(preconditioner_t) :: pre
     integer                :: max_iter
  end type linear_solver_t

  type(profile_t), save :: prof, prof_batch

  type linear_solver_args_t
    type(linear_solver_t), pointer :: ls
    type(hamiltonian_t),   pointer :: hm
    type(grid_t),          pointer :: gr
    type(states_t),        pointer :: st
    integer                        :: ist
    integer                        :: ik
    FLOAT                          :: dshift
    CMPLX                          :: zshift
  end type linear_solver_args_t

  type(linear_solver_args_t) :: args

contains

  ! ---------------------------------------------------------
  subroutine linear_solver_init(this, gr, prefix, states_are_real, def_solver)
    type(linear_solver_t),  intent(out)   :: this
    type(grid_t),           intent(in)    :: gr
    character(len=*),       intent(in)    :: prefix
    logical,                intent(in)    :: states_are_real !< for choosing solver
    integer, optional,      intent(in)    :: def_solver

    integer :: fsolver
    integer :: defsolver_ 

    PUSH_SUB(linear_solver_init)

    !%Variable LinearSolver
    !%Type integer
    !%Default qmr_symmetric
    !%Section Linear Response::Solver
    !%Description
    !% Method for solving linear equations, which occur for Sternheimer linear
    !% response and OEP. The solvers vary in speed, reliability (ability to
    !% converge), and domain of applicability. QMR solvers are most reliable.
    !%Option bicgstab 4
    !% Biconjugate gradients stabilized. Slower than <tt>cg</tt>, but more reliable.
    !% General matrices.
    !%Option cg 5
    !% Conjugate gradients. Fast but unreliable. Hermitian matrices only
    !% (no eta in Sternheimer).
    !%Option multigrid 7
    !% Multigrid solver, currently only Gauss-Jacobi (experimental).
    !% Slow, but fairly reliable. General matrices.
    !%Option qmr_symmetric 81
    !% Quasi-minimal residual solver, for (complex) symmetric matrices. [Real symmetric
    !% is equivalent to Hermitian.] Slightly slower than <tt>bicgstab</tt> but more reliable.
    !% For Sternheimer, must be real wavefunctions, but can have eta.
    !%Option qmr_symmetrized 82
    !% Quasi-minimal residual solver, using the symmetrized form <math>A^\dagger A x = A^\dagger y</math> instead of
    !% <math>A x = y</math>. Reliable but very slow. General matrices.
    !%Option qmr_dotp 83
    !% Quasi-minimal residual solver, for Hermitian matrices, using the
    !% symmetric algorithm with conjugated dot product (experimental). Slightly slower than <tt>bicgstab</tt>
    !% but more reliable. Can always be used in Sternheimer.
    !%Option qmr_general 84
    !% Quasi-minimal residual solver, for general matrices, using the
    !% most general form of the algorithm. Slow and unreliable.
    !%Option sos 9
    !% Sum over states: the Sternheimer equation is solved by using
    !% the explicit solution in terms of the ground-state
    !% wavefunctions. You need unoccupied states to use this method.
    !% Unlike the other methods, may not give the correct answer.
    !%End

    if(present(def_solver)) then
      defsolver_ = def_solver
    else
      if(conf%devel_version) then
        defsolver_ = LS_QMR_DOTP
      else
        if(states_are_real) then
          defsolver_ = LS_QMR_SYMMETRIC
          ! in this case, it is equivalent to LS_QMR_DOTP
        else
          defsolver_ = LS_QMR_SYMMETRIZED
        endif
      endif
    endif

    if (parse_isdef(datasets_check(trim(prefix)//"LinearSolver")) /= 0 ) then 
      call parse_integer(datasets_check(trim(prefix)//"LinearSolver"), defsolver_, fsolver)
    else
      call parse_integer(datasets_check("LinearSolver"), defsolver_, fsolver)
    end if

    ! set up pointer for dot product and norm in QMR solvers
    call mesh_init_mesh_aux(gr%mesh)

    !the last 2 digits select the linear solver
    this%solver = mod(fsolver, 100)

    call preconditioner_init(this%pre, gr, prefix)

    !%Variable LinearSolverMaxIter
    !%Type integer
    !%Default 1000
    !%Section Linear Response::Solver
    !%Description
    !% Maximum number of iterations the linear solver does, even if
    !% convergence is not achieved.
    !%End
    if (parse_isdef(datasets_check(trim(prefix)//"LinearSolverMaxIter")) /= 0) then 
      call parse_integer(datasets_check(trim(prefix)//"LinearSolverMaxIter"), 1000, this%max_iter)
    else
      call parse_integer(datasets_check("LinearSolverMaxIter"), 1000, this%max_iter)
    end if

    write(message(1),'(a)') 'Linear Solver'
    call messages_print_stress(stdout, trim(message(1)))
    
    ! solver 
    select case(this%solver)
      case(LS_CG)
        message(1)='Linear Solver: Conjugate Gradients'

      case(LS_BICGSTAB)
        message(1)='Linear Solver: Biconjugate Gradients Stabilized'

      case(LS_MULTIGRID)
        message(1)='Multigrid (currently only Gauss-Jacobi - EXPERIMENTAL)'

      case(LS_QMR_SYMMETRIC)
        message(1)='Linear Solver: Quasi-Minimal Residual, for symmetric matrix'

      case(LS_QMR_SYMMETRIZED)
        message(1)='Linear Solver: Quasi-Minimal Residual, for symmetrized matrix'

      case(LS_QMR_DOTP)
        message(1)='Linear Solver: Quasi-Minimal Residual, symmetric with conjugated dot product'

      case(LS_QMR_GENERAL)
        message(1)='Linear Solver: Quasi-Minimal Residual, general algorithm'

      case(LS_SOS)
        message(1)='Linear Solver: Sum-over-States'
    end select

    call messages_info(1)
    
    call messages_print_stress(stdout)

    if(this%solver == LS_MULTIGRID) call messages_experimental("Multigrid linear solver")
    if(this%solver == LS_QMR_DOTP)  call messages_experimental("QMR solver (symmetric with conjugated dot product)")

    POP_SUB(linear_solver_init)

  end subroutine linear_solver_init

  ! ---------------------------------------------------------
  subroutine linear_solver_end(this)
    type(linear_solver_t), intent(inout) :: this
    this%solver = -1

    call preconditioner_end(this%pre)

  end subroutine linear_solver_end


  ! ---------------------------------------------------------
  integer function linear_solver_ops_per_iter(this) result(n)
    type(linear_solver_t), intent(inout) :: this
    
    select case(this%solver)
    case(LS_BICGSTAB)
      n = 2
    case default ! LS_CG, LS_MULTIGRID, LS_QMR, LS_SOS
      n = 1
    end select
  
  end function linear_solver_ops_per_iter

  ! ----------------------------------------------------------
  
  subroutine linear_solver_obsolete_variables(old_prefix, new_prefix)
    character(len=*),    intent(in)    :: old_prefix
    character(len=*),    intent(in)    :: new_prefix
    
    call messages_obsolete_variable(trim(old_prefix)//"LinearSolver", trim(new_prefix)//"LinearSolver")
    call messages_obsolete_variable(trim(old_prefix)//"LinearSolverMaxIter", trim(new_prefix)//"LinearSolverMaxIter")

    call preconditioner_obsolete_variables(old_prefix, new_prefix)

  end subroutine linear_solver_obsolete_variables

#include "undef.F90"

#include "real.F90"

#include "linear_solver_inc.F90"

#include "undef.F90"

#include "complex.F90"
#include "linear_solver_inc.F90"

end module linear_solver_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
