!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!
!! $Id: grid.F90 2307 2006-07-29 00:50:22Z appel $

#include "global.h"

module species_pot_m
  use atom_m
  use curvilinear_m
  use datasets_m
  use double_grid_m
  use geometry_m
  use global_m
  use grid_m
  use io_m
  use io_function_m
  use loct_math_m
  use math_m
  use mesh_function_m
  use mesh_m
  use messages_m
  use mpi_m
  use parser_m
  use poisson_m
  use profiling_m
  use ps_m
  use root_solver_m
  use simul_box_m
  use solids_m
  use species_m
  use splines_m
  use submesh_m
  use unit_m
  use unit_system_m
  use varinfo_m

  implicit none

  private
  public ::                      &
    species_get_density,         &
    species_get_nlcc,            &
    species_get_orbital,         &
    species_get_orbital_submesh, &
    species_get_local,           &
    species_atom_density

  type(mesh_t), pointer :: mesh_p
  FLOAT, allocatable :: rho_p(:)
  FLOAT, allocatable :: grho_p(:, :)
  FLOAT :: alpha_p
  FLOAT :: pos_p(MAX_DIM)

contains


  ! ---------------------------------------------------------
  subroutine species_atom_density(mesh, sb, atom, spin_channels, rho)
    type(mesh_t),         intent(in)    :: mesh
    type(simul_box_t),    intent(in)    :: sb
    type(atom_t), target, intent(in)    :: atom
    integer,              intent(in)    :: spin_channels
    FLOAT,                intent(inout) :: rho(:, :) !< (mesh%np, spin_channels)

    integer :: isp, ip, in_points, nn, icell
    FLOAT :: rr, x, pos(1:MAX_DIM)
    FLOAT :: psi1, psi2, xx(MAX_DIM), yy(MAX_DIM), rerho, imrho
    type(species_t), pointer :: spec
    type(ps_t), pointer :: ps

#if defined(HAVE_MPI)
    integer :: in_points_red
#endif
    type(periodic_copy_t) :: pp

    PUSH_SUB(species_atom_density)

    ASSERT(spin_channels == 1 .or. spin_channels == 2)

    spec => atom%spec
    rho = M_ZERO

    ! build density ...
    select case (species_type(spec))
    case (SPEC_FROM_FILE, SPEC_USDEF, SPEC_FULL_DELTA, SPEC_FULL_GAUSSIAN, SPEC_PS_CPI, SPEC_PS_FHI) ! ... from userdef
      do isp = 1, spin_channels
        rho(1:mesh%np, isp) = M_ONE
        x = (species_zval(spec)/real(spin_channels, REAL_PRECISION)) / dmf_integrate(mesh, rho(:, isp))
        rho(1:mesh%np, isp) = x * rho(1:mesh%np, isp)
      end do

    case (SPEC_CHARGE_DENSITY)
      ! We put, for the electron density, the same as the positive density that 
      ! creates the external potential.

      call periodic_copy_init(pp, sb, spread(M_ZERO, dim=1, ncopies = sb%dim), &
        range = M_TWO * maxval(sb%lsize(1:sb%dim)))

      rho = M_ZERO
      do icell = 1, periodic_copy_num(pp)
        yy(1:sb%dim) = periodic_copy_position(pp, sb, icell)
        do ip = 1, mesh%np
          call mesh_r(mesh, ip, rr, origin = atom%x, coords = xx)
          xx(1:sb%dim) = xx(1:sb%dim) + yy(1:sb%dim)
          rr = sqrt(dot_product(xx(1:sb%dim), xx(1:sb%dim)))
          call parse_expression(rerho, imrho, sb%dim, xx, rr, M_ZERO, trim(species_rho_string(spec)))
          rho(ip, 1) = rho(ip, 1) + rerho
        end do
      end do
      call periodic_copy_end(pp)
      if(spin_channels > 1) then
        rho(:, 1) = M_HALF*rho(:, 1)
        rho(:, 2) = rho(:, 1)
      end if
      ! rescale to match the valence charge
      do isp = 1, spin_channels
        x = species_zval(spec) / dmf_integrate(mesh, rho(:, isp))
        rho(1:mesh%np, isp) = x * rho(1:mesh%np, isp)
      end do

    case (SPEC_POINT, SPEC_JELLI) ! ... from jellium
      in_points = 0
      do ip = 1, mesh%np
        call mesh_r(mesh, ip, rr, origin = atom%x)
        if(rr <= species_jradius(spec)) then
          in_points = in_points + 1
        end if
      end do

#if defined(HAVE_MPI)
      if(mesh%parallel_in_domains) then
        call MPI_Allreduce(in_points, in_points_red, 1, MPI_INTEGER, MPI_SUM, mesh%vp%comm, mpi_err)
        in_points = in_points_red
      end if
#endif

      if(in_points > 0) then
        ! This probably should be done inside the mesh_function_m module.
 
        if (mesh%use_curvilinear) then
          do ip = 1, mesh%np
            call mesh_r(mesh, ip, rr, origin = atom%x)
            if(rr <= species_jradius(spec)) then
              rho(ip, 1:spin_channels) = species_zval(spec) /   &
                (mesh%vol_pp(ip)*real(in_points*spin_channels, REAL_PRECISION))
            end if
          end do
        else
          do ip = 1, mesh%np
            call mesh_r(mesh, ip, rr, origin = atom%x)
            if(rr <= species_jradius(spec)) then
              rho(ip, 1:spin_channels) = species_zval(spec) /   &
                (mesh%vol_pp(1)*real(in_points*spin_channels, REAL_PRECISION))
            end if
          end do
        end if
      end if

    case (SPEC_JELLI_SLAB) ! ... from jellium slab
      in_points = 0
      do ip = 1, mesh%np
        rr = abs( mesh%x( ip, 3 ) )
        if( rr <= species_jthick(spec)/M_TWO ) then
          in_points = in_points + 1
        end if
      end do

#if defined(HAVE_MPI)
      if(mesh%parallel_in_domains) then
        call MPI_Allreduce(in_points, in_points_red, 1, MPI_INTEGER, MPI_SUM, mesh%vp%comm, mpi_err)
        in_points = in_points_red
      end if
#endif

      if(in_points > 0) then
        ! This probably should be done inside the mesh_function_m module.

        if (mesh%use_curvilinear) then
          do ip = 1, mesh%np
            rr = abs( mesh%x( ip, 3 ) )
            if( rr <= species_jthick(spec)/M_TWO ) then
              rho(ip, 1:spin_channels) = species_zval(spec) /   &
                (mesh%vol_pp(ip)*real(in_points*spin_channels, REAL_PRECISION))
            end if
          end do
        else
          do ip = 1, mesh%np
            rr = abs( mesh%x( ip, 3 ) )
            if( rr <= species_jthick(spec)/M_TWO ) then
              rho(ip, 1:spin_channels) = species_zval(spec) /   &
                (mesh%vol_pp(1)*real(in_points*spin_channels, REAL_PRECISION))
            end if
          end do
        end if
      end if

    case (SPEC_PS_PSF, SPEC_PS_HGH, SPEC_PS_UPF, SPEC_PSPIO) ! ...from pseudopotential

      ! the outer loop sums densities over atoms in neighbour cells
      pos(1:MAX_DIM) = M_ZERO
      ps => species_ps(spec)

      call periodic_copy_init(pp, sb, atom%x, &
        range = spline_cutoff_radius(ps%Ur(1, 1), ps%projectors_sphere_threshold))

      do icell = 1, periodic_copy_num(pp)
        pos(1:sb%dim) = periodic_copy_position(pp, sb, icell)
        do ip = 1, mesh%np
          call mesh_r(mesh, ip, rr, origin = pos)
          rr = max(rr, r_small)
          do nn = 1, ps%conf%p
            select case(spin_channels)
            case(1)
              if(rr >= spline_range_max(ps%Ur(nn, 1))) cycle

              psi1 = spline_eval(ps%Ur(nn, 1), rr)
              rho(ip, 1) = rho(ip, 1) + ps%conf%occ(nn, 1)*psi1*psi1 /(M_FOUR*M_PI)
            case(2)
              if(rr >= spline_range_max(ps%Ur(nn, 1))) cycle
              if(rr >= spline_range_max(ps%Ur(nn, 2))) cycle

              psi1 = spline_eval(ps%Ur(nn, 1), rr)
              psi2 = spline_eval(ps%Ur(nn, 2), rr)
              rho(ip, 1) = rho(ip, 1) + ps%conf%occ(nn, 1)*psi1*psi1 /(M_FOUR*M_PI)
              rho(ip, 2) = rho(ip, 2) + ps%conf%occ(nn, 2)*psi2*psi2 /(M_FOUR*M_PI)
            end select
          end do
        end do
      end do
  
      call periodic_copy_end(pp)
      nullify(ps)

    end select

    POP_SUB(species_atom_density)
  end subroutine species_atom_density
  ! ---------------------------------------------------------

  subroutine species_get_density(spec, pos, mesh, rho)
    type(species_t),    target, intent(in)  :: spec
    FLOAT,                      intent(in)  :: pos(:)
    type(mesh_t),       target, intent(in)  :: mesh
    FLOAT,                      intent(out) :: rho(:)

    type(root_solver_t) :: rs
    logical :: conv
    integer :: dim
    FLOAT   :: x(1:MAX_DIM+1), chi0(MAX_DIM), startval(MAX_DIM + 1)
    FLOAT   :: delta, alpha, beta, xx(MAX_DIM), yy(MAX_DIM), rr, imrho, rerho
    FLOAT   :: dist2, dist2_min
    integer :: icell, ipos, ip
    type(periodic_copy_t) :: pp
    type(ps_t), pointer :: ps
    logical :: have_point
#ifdef HAVE_MPI
    real(8) :: local_min(2), global_min(2)
#endif
    type(submesh_t)       :: sphere
    type(profile_t), save :: prof
    FLOAT,    allocatable :: rho_sphere(:)
    FLOAT, parameter      :: threshold = CNST(1e-6)
    FLOAT                 :: norm_factor
    
    PUSH_SUB(species_get_density)

    call profiling_in(prof, "SPECIES_DENSITY")

    select case(species_type(spec))

    case(SPEC_PS_PSF, SPEC_PS_HGH, SPEC_PS_CPI, SPEC_PS_FHI, SPEC_PS_UPF, SPEC_PSPIO)
      ps => species_ps(spec)

      call submesh_init_sphere(sphere, mesh%sb, mesh, pos, spline_cutoff_radius(ps%nlr, threshold))
      SAFE_ALLOCATE(rho_sphere(1:sphere%np))
      
      forall(ip = 1:sphere%np) rho_sphere(ip) = sphere%x(ip, 0)
      if(sphere%np > 0) call spline_eval_vec(ps%nlr, sphere%np, rho_sphere)

      rho(1:mesh%np) = M_ZERO

      ! A small amount of charge is missing with the cutoff, we
      ! renormalize so that the long range potential is exact
      norm_factor = abs(species_zval(spec)/dsm_integrate(mesh, sphere, rho_sphere))
      
      do ip = 1, sphere%np
        rho(sphere%map(ip)) = rho(sphere%map(ip)) + norm_factor*rho_sphere(ip)
      end do

      SAFE_DEALLOCATE_A(rho_sphere)
      call submesh_end(sphere)
      nullify(ps)

    case(SPEC_FULL_DELTA)

      dist2_min = huge(delta)
      ipos = 0

      do ip = 1, mesh%np

        rho(ip) = M_ZERO

        dist2 = sum((mesh%x(ip, 1:mesh%sb%dim) - pos(1:mesh%sb%dim))**2)
        if (dist2 < dist2_min) then
          ipos = ip
          dist2_min = dist2
        end if

      end do

      write(message(1), '(3a,f5.2,3a)') &
        "Info: spec_full_delta species ", trim(species_label(spec)), &
        " atom displaced ", units_from_atomic(units_out%length, sqrt(dist2_min)), &
        " [ ", trim(units_abbrev(units_out%length)), " ]"
      call messages_info(1)

      have_point = .true.
#ifdef HAVE_MPI
      ! in parallel we have to find the minimum of the whole grid
      if(mesh%parallel_in_domains) then

        local_min = (/ dist2_min, dble(mesh%mpi_grp%rank)/)
        call MPI_Allreduce(local_min, global_min, 1, MPI_2DOUBLE_PRECISION, MPI_MINLOC, mesh%mpi_grp%comm, mpi_err)

        if(mesh%mpi_grp%rank /= nint(global_min(2))) have_point = .false.

      end if
#endif
      if(have_point) then
        if(mesh%use_curvilinear) then
          rho(ipos) = -species_z(spec)/mesh%vol_pp(ipos)
        else
          rho(ipos) = -species_z(spec)/mesh%vol_pp(1)
        end if
      end if

    case(SPEC_FULL_GAUSSIAN)

      ! periodic copies are not considered in this routine
      if(simul_box_is_periodic(mesh%sb)) then
        call messages_experimental("spec_full_gaussian for periodic systems")
      end if

      ! --------------------------------------------------------------
      ! Constructs density for an all-electron atom with the procedure
      ! sketched in Modine et al. [Phys. Rev. B 55, 10289 (1997)],
      ! section II.B
      ! --------------------------------------------------------------
      dim = mesh%sb%dim

      SAFE_ALLOCATE(rho_p(1:mesh%np))
      SAFE_ALLOCATE(grho_p(1:mesh%np, 1:dim+1))

      mesh_p => mesh
      pos_p = pos

      ! Initial guess.
      call curvilinear_x2chi(mesh%sb, mesh%cv, pos, chi0)
      delta   = mesh%spacing(1)
      alpha   = sqrt(M_TWO)*species_sigma(spec)*delta
      alpha_p = alpha  ! global copy of alpha
      beta    = M_ONE

      ! the first dim variables are the position of the delta function
      startval(1:dim) = CNST(1.0)

      ! the dim+1 variable is the normalization of the delta function
      startval(dim+1) = beta

      ! get a better estimate for beta
      call getrho(startval)
      beta = M_ONE / dmf_integrate(mesh, rho_p)
      startval(dim+1) = beta

      ! solve equation
      call root_solver_init(rs, dim+1, &
        solver_type=ROOT_NEWTON, maxiter=500, abs_tolerance=CNST(1.0e-10))
      call droot_solver_run(rs, func, x, conv, startval=startval)

      if(.not.conv) then
        write(message(1),'(a)') 'Internal error in get_species_density.'
        call messages_fatal(1)
      end if

      ! we want a charge of -Z
      rho = -species_z(spec)*rho_p

      nullify(mesh_p)
      SAFE_DEALLOCATE_A(grho_p)
      SAFE_DEALLOCATE_A(rho_p)


    case(SPEC_CHARGE_DENSITY)

      call periodic_copy_init(pp, mesh%sb, spread(M_ZERO, dim=1, ncopies = mesh%sb%dim), &
        range = M_TWO * maxval(mesh%sb%lsize(1:mesh%sb%dim)))

      rho = M_ZERO
      do icell = 1, periodic_copy_num(pp)
        yy(1:mesh%sb%dim) = periodic_copy_position(pp, mesh%sb, icell)
        do ip = 1, mesh%np
          call mesh_r(mesh, ip, rr, origin = pos, coords = xx)
          xx(1:mesh%sb%dim) = xx(1:mesh%sb%dim) + yy(1:mesh%sb%dim)
          rr = sqrt(dot_product(xx(1:mesh%sb%dim), xx(1:mesh%sb%dim)))
          call parse_expression(rerho, imrho, mesh%sb%dim, xx, rr, M_ZERO, trim(species_rho_string(spec)))
          rho(ip) = rho(ip) - rerho
        end do
      end do
      rr = species_zval(spec) / abs(dmf_integrate(mesh, rho(:)))
      rho(1:mesh%np) = rr * rho(1:mesh%np)

      call periodic_copy_end(pp)

    end select

    call profiling_out(prof)
    POP_SUB(species_get_density)
  end subroutine species_get_density


  ! ---------------------------------------------------------
  subroutine func(xin, ff, jacobian)
    FLOAT, intent(in)  :: xin(:)
    FLOAT, intent(out) :: ff(:), jacobian(:,:)

    FLOAT, allocatable :: xrho(:)
    integer :: idir, jdir, dim

    PUSH_SUB(func)

    dim = mesh_p%sb%dim

    call getrho(xin)
    SAFE_ALLOCATE(xrho(1:mesh_p%np))

    ! First, we calculate the function ff.
    do idir = 1, dim
      xrho(1:mesh_p%np) = rho_p(1:mesh_p%np) * mesh_p%x(1:mesh_p%np, idir)
      ff(idir) = dmf_integrate(mesh_p, xrho) - pos_p(idir)
    end do
    ff(dim+1) = dmf_integrate(mesh_p, rho_p) - M_ONE

    ! Now the jacobian.
    do idir = 1, dim
      do jdir = 1, dim+1
        xrho(1:mesh_p%np) = grho_p(1:mesh_p%np, jdir) * mesh_p%x(1:mesh_p%np, idir)
        jacobian(idir, jdir) = dmf_integrate(mesh_p, xrho)
      end do
    end do
    do jdir = 1, dim+1
      xrho(1:mesh_p%np) = grho_p(1:mesh_p%np, jdir)
      jacobian(dim+1, jdir) = dmf_integrate(mesh_p, xrho)
    end do

    SAFE_DEALLOCATE_A(xrho)
    POP_SUB(func)
  end subroutine func

  ! ---------------------------------------------------------
  subroutine species_get_nlcc(spec, pos, mesh, rho_core)
    type(species_t), target, intent(in)  :: spec
    FLOAT,                   intent(in)  :: pos(MAX_DIM)
    type(mesh_t),            intent(in)  :: mesh
    FLOAT,                   intent(out) :: rho_core(:)

    integer :: icell
    type(periodic_copy_t) :: pp
    type(ps_t), pointer :: ps

    PUSH_SUB(species_get_nlcc)

    ! only for 3D pseudopotentials, please
    if(species_is_ps(spec)) then
      ps => species_ps(spec)
      rho_core = M_ZERO
      call periodic_copy_init(pp, mesh%sb, pos, range = spline_cutoff_radius(ps%core, ps%projectors_sphere_threshold))
      do icell = 1, periodic_copy_num(pp)
        call dmf_put_radial_spline(mesh, ps%core, periodic_copy_position(pp, mesh%sb, icell), rho_core, add = .true.)
      end do
      call periodic_copy_end(pp)
    else
      rho_core = M_ZERO
    end if

    POP_SUB(species_get_nlcc)
  end subroutine species_get_nlcc

  ! ---------------------------------------------------------
  subroutine getrho(xin)
    FLOAT, intent(in) :: xin(:)

    integer :: ip, jp, idir, dim
    FLOAT   :: r, chi(MAX_DIM)

    PUSH_SUB(getrho)

    dim = mesh_p%sb%dim
    rho_p = M_ZERO
    do ip = 1, mesh_p%np

      jp = ip
      if(mesh_p%parallel_in_domains) &
        jp = mesh_p%vp%local(mesh_p%vp%xlocal(mesh_p%vp%partno)+ip-1)

      chi(1:dim) = mesh_p%idx%lxyz(jp, 1:dim) * mesh_p%spacing(1:dim) + mesh_p%sb%box_offset(1:dim) 

      r = sqrt( sum( (chi(1:dim) - xin(1:dim))**2 ) )

      if( (r/alpha_p)**2 < CNST(10.0)) then
        grho_p(ip, dim+1) = exp(-(r/alpha_p)**2)
        rho_p(ip)         = xin(dim+1) * grho_p(ip, dim+1)
      else
        grho_p(ip, dim+1) = M_ZERO
        rho_p(ip)         = M_ZERO
      end if

      do idir = 1, dim
        grho_p(ip, idir) = (M_TWO/alpha_p**2) * (chi(idir)-xin(idir)) * rho_p(ip)
      end do
    end do

    POP_SUB(getrho)
  end subroutine getrho 


  ! ---------------------------------------------------------
  subroutine species_get_local(spec, mesh, x_atom, vl, Imvl)
    type(species_t), target, intent(in)  :: spec
    type(mesh_t),            intent(in)  :: mesh
    FLOAT,                   intent(in)  :: x_atom(:)
    FLOAT,                   intent(out) :: vl(:)
    FLOAT,         optional, intent(out) :: Imvl(:) !< cmplxscl: imaginary part of the potential

    FLOAT :: a1, a2, Rb2 ! for jellium
    FLOAT :: xx(MAX_DIM), r, r2
    integer :: ip, err, idim
    type(ps_t), pointer :: ps
    CMPLX :: zpot

    type(profile_t), save :: prof

    PUSH_SUB(species_get_local)
    call profiling_in(prof, "SPECIES_GET_LOCAL")

      select case(species_type(spec))
      case(SPEC_USDEF)

        do ip = 1, mesh%np
          
          xx = M_ZERO
          xx(1:mesh%sb%dim) = mesh%x(ip,1:mesh%sb%dim) - x_atom(1:mesh%sb%dim)
          r = sqrt(sum(xx(1:mesh%sb%dim)**2))
          
          ! Note that as the spec%user_def is in input units, we have to convert
          ! the units back and forth
          forall(idim = 1:mesh%sb%dim) xx(idim) = units_from_atomic(units_inp%length, xx(idim))
          r = units_from_atomic(units_inp%length, r)
          zpot = species_userdef_pot(spec, mesh%sb%dim, xx, r)
          vl(ip)   = units_to_atomic(units_inp%energy, real(zpot))
          if(present(Imvl)) then!cmplxscl
            Imvl(ip) = units_to_atomic(units_inp%energy, aimag(zpot))            
          end if

        end do


      case(SPEC_FROM_FILE)

        call dio_function_input(trim(species_filename(spec)), mesh, vl, err)
        if(err .ne. 0) then
          write(message(1), '(a)')    'Error loading file '//trim(species_filename(spec))//'.'
          write(message(2), '(a,i4)') 'Error code returned = ', err
          call messages_fatal(2)
        end if

      case(SPEC_POINT, SPEC_JELLI)
        a1 = species_z(spec)/(M_TWO*species_jradius(spec)**3)
        a2 = species_z(spec)/species_jradius(spec)
        Rb2= species_jradius(spec)**2
        
        do ip = 1, mesh%np
          
          xx(1:mesh%sb%dim) = mesh%x(ip, 1:mesh%sb%dim) - x_atom(1:mesh%sb%dim)
          r = sqrt(sum(xx(1:mesh%sb%dim)**2))
          
          if(r <= species_jradius(spec)) then
            vl(ip) = (a1*(r*r - Rb2) - a2)
          else
            vl(ip) = -species_z(spec)/r
          end if
          
        end do
      
      case(SPEC_JELLI_SLAB)
        a1 = M_TWO *M_PI * species_z(spec)/ (M_FOUR *mesh%sb%lsize(1) *mesh%sb%lsize(2) )

        do ip = 1, mesh%np

          r = abs( mesh%x(ip, 3 ) )

          if(r <= species_jthick(spec)/M_TWO ) then
            vl(ip) = a1 *( r*r/species_jthick(spec) + species_jthick(spec)/M_FOUR )
          else
            vl(ip) = a1 *r
          end if

        end do

      case(SPEC_PS_PSF, SPEC_PS_HGH, SPEC_PS_CPI, SPEC_PS_FHI, SPEC_PS_UPF, SPEC_PSPIO)
       
        ps => species_ps(spec)

        do ip = 1, mesh%np
          r2 = sum((mesh%x(ip, 1:mesh%sb%dim) - x_atom(1:mesh%sb%dim))**2)
          if(r2 < spline_range_max(ps%vlr_sq)) then
            vl(ip) = spline_eval(ps%vlr_sq, r2)
          else
            vl(ip) = P_PROTON_CHARGE*species_zval(spec)/sqrt(r2)
          end if
        end do

        nullify(ps)
        
      case(SPEC_FULL_DELTA, SPEC_FULL_GAUSSIAN, SPEC_CHARGE_DENSITY)
        vl(1:mesh%np) = M_ZERO
        
      end select

      call profiling_out(prof)
    POP_SUB(species_get_local)
  end subroutine species_get_local

  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  !> Places, in the function phi (defined in each point of the mesh), the
  !! iorb-th atomic orbital. The orbitals are obtained from the species data
  !! type, and are numbered from one to species_niwfs(spec). It may happen
  !! that there are different orbitals for each spin-polarization direction,
  !! and therefore the orbital is also characterized by the label "is".
  !!
  !! In order to put the orbital in the mesh, it is necessary to know where
  !! the species is, and this is given by the vector "pos".
  !!
  !! \todo Most of this work should be done inside the species
  !! module, and we should get rid of species_iwf_i, species_ifw_l, etc.
  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  subroutine species_get_orbital(spec, mesh, iorb, ispin, pos, orb)
    type(species_t), target, intent(in)     :: spec
    type(mesh_t),            intent(in)     :: mesh
    integer,                 intent(in)     :: iorb
    integer,                 intent(in)     :: ispin   !< The spin index.
    FLOAT,                   intent(in)     :: pos(:)  !< The position of the atom.
    FLOAT,                   intent(out)    :: orb(:)  !< The function defined in the mesh where the orbitals is returned.

    integer :: i, l, m, ip, icell
    FLOAT :: r2, x(1:MAX_DIM), radius
    FLOAT, allocatable :: xf(:, :), ylm(:), lorb(:)
    type(ps_t), pointer :: ps
    type(periodic_copy_t) :: pc

    PUSH_SUB(species_get_orbital)

    call species_iwf_ilm(spec, iorb, ispin, i, l, m)

    radius = min(species_get_iwf_radius(spec, iorb, ispin), maxval(mesh%sb%lsize))

    call periodic_copy_init(pc, mesh%sb, pos, range = radius)

    orb = M_ZERO

    SAFE_ALLOCATE(lorb(1:mesh%np))

    do icell = 1, periodic_copy_num(pc)

      if(species_is_ps(spec)) then

        ps => species_ps(spec)
        SAFE_ALLOCATE(xf(1:mesh%np, 1:mesh%sb%dim))
        SAFE_ALLOCATE(ylm(1:mesh%np))

        do ip = 1, mesh%np
          x(1:mesh%sb%dim) = mesh%x(ip, 1:mesh%sb%dim) - periodic_copy_position(pc, mesh%sb, icell)
          r2 = sum(x(1:mesh%sb%dim)**2)
          xf(ip, 1:mesh%sb%dim) = x(1:mesh%sb%dim)
          
          if(r2 < spline_range_max(ps%ur_sq(i, ispin))) then
            lorb(ip) = spline_eval(ps%ur_sq(i, ispin), r2)
          else
            lorb(ip) = M_ZERO
          end if

        end do

        call loct_ylm(mesh%np, xf(1, 1), xf(1, 2), xf(1, 3), l, m, ylm(1))

        do ip = 1, mesh%np
          orb(ip) = orb(ip) + lorb(ip)*ylm(ip)
        end do

        SAFE_DEALLOCATE_A(xf)
        SAFE_DEALLOCATE_A(ylm)

        nullify(ps)

      else

        do ip = 1, mesh%np
          x(1:mesh%sb%dim) = mesh%x(ip, 1:mesh%sb%dim) - pos(1:mesh%sb%dim)
          r2 = sum(x(1:mesh%sb%dim)**2)
          select case(mesh%sb%dim)
          case(1)
            orb(ip) = orb(ip) + exp(-species_omega(spec)*r2/M_TWO) * hermite(i - 1, x(1)*sqrt(species_omega(spec)))
          case(2)
             orb(ip) = orb(ip) + exp(-species_omega(spec)*r2/M_TWO) * &
              hermite(i - 1, x(1)*sqrt(species_omega(spec))) * hermite(l - 1, x(2)*sqrt(species_omega(spec)))
          case(3)
             orb(ip) = orb(ip) + exp(-species_omega(spec)*r2/M_TWO) * hermite(i - 1, x(1)*sqrt(species_omega(spec)))* &
              hermite(l - 1, x(2)*sqrt(species_omega(spec)))*hermite(m - 1, x(3)*sqrt(species_omega(spec)))
          end select
        end do
      end if

    end do

    SAFE_DEALLOCATE_A(lorb)

    call periodic_copy_end(pc)

    POP_SUB(species_get_orbital)
  end subroutine species_get_orbital


  ! ---------------------------------------------------------
  subroutine species_get_orbital_submesh(spec, submesh, iorb, ispin, pos, phi, derivative)
    type(species_t), target, intent(in)  :: spec       !< The species.
    type(submesh_t),         intent(in)  :: submesh    !< The submesh descriptor where the orbital will be calculated.
    integer,                 intent(in)  :: iorb       !< The index of the orbital to return.
    integer,                 intent(in)  :: ispin      !< The spin index.
    FLOAT,                   intent(in)  :: pos(:)     !< The position of the atom.
    FLOAT,                   intent(out) :: phi(:)     !< The function defined in the mesh where the orbitals is returned.
    logical,       optional, intent(in)  :: derivative !< If present and .true. returns the derivative of the orbital.

    integer :: i, l, m, ip
    FLOAT :: r2, x(1:MAX_DIM), sqrtw, ww
    FLOAT, allocatable :: ylm(:)
    type(ps_t), pointer :: ps
    type(spline_t) :: dur
    logical :: derivative_
    
    if(submesh%np == 0) return

    PUSH_SUB(species_get_orbital_submesh)

    derivative_ = optional_default(derivative, .false.)

    ASSERT(ubound(phi, dim = 1) >= submesh%np)

    call species_iwf_ilm(spec, iorb, ispin, i, l, m)

    if(species_is_ps(spec)) then
      ps => species_ps(spec)
      
      forall(ip = 1:submesh%np) phi(ip) = submesh%x(ip, 0)

      if(.not. derivative_) then
        call spline_eval_vec(ps%ur(i, ispin), submesh%np, phi)
      else
        call spline_init(dur)
        call spline_der(ps%ur(i, ispin), dur)
        call spline_eval_vec(dur, submesh%np, phi)
        call spline_end(dur)
      end if

      SAFE_ALLOCATE(ylm(1:submesh%np))

      call loct_ylm(submesh%np, submesh%x(1, 1), submesh%x(1, 2), submesh%x(1, 3), l, m, ylm(1))

      do ip = 1, submesh%np
        phi(ip) = phi(ip)*ylm(ip)
      end do

      SAFE_DEALLOCATE_A(ylm)

      nullify(ps)
    else
      
      ASSERT(.not. derivative_)

      ww = species_omega(spec)
      sqrtw = sqrt(ww)

      select case(submesh%mesh%sb%dim)
      case(1)
        do ip = 1, submesh%np
          x(1:submesh%mesh%sb%dim) = submesh%mesh%x(submesh%map(ip), 1:submesh%mesh%sb%dim) - pos(1:submesh%mesh%sb%dim)
          r2 = sum(x(1:submesh%mesh%sb%dim)**2)
          phi(ip) = exp(-ww*r2/M_TWO)*hermite(i - 1, x(1)*sqrtw)
        end do
      case(2)
        do ip = 1, submesh%np
          x(1:submesh%mesh%sb%dim) = submesh%mesh%x(submesh%map(ip), 1:submesh%mesh%sb%dim) - pos(1:submesh%mesh%sb%dim)
          r2 = sum(x(1:submesh%mesh%sb%dim)**2)
          phi(ip) = exp(-ww*r2/M_TWO)*hermite(i - 1, x(1)*sqrtw)*hermite(l - 1, x(2)*sqrtw)
        end do
      case(3)
        do ip = 1, submesh%np
          x(1:submesh%mesh%sb%dim) = submesh%mesh%x(submesh%map(ip), 1:submesh%mesh%sb%dim) - pos(1:submesh%mesh%sb%dim)
          phi(ip) = exp(-ww*r2/M_TWO)*hermite(i - 1, x(1)*sqrtw)*hermite(l - 1, x(2)*sqrtw)*hermite(m - 1, x(3)*sqrtw)
        end do
      end select
      
    end if

    POP_SUB(species_get_orbital_submesh)
  end subroutine species_get_orbital_submesh

end module species_pot_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
