!! Copyright (C) 2022 F. Bonafé
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!
#include "global.h"

module mxll_field_to_medium_oct_m
  use clock_oct_m
  use debug_oct_m
  use global_oct_m
  use grid_oct_m
  use interaction_with_partner_oct_m
  use interaction_partner_oct_m
  use mesh_oct_m
  use messages_oct_m
  use namespace_oct_m
  use profiling_oct_m
  use quantity_oct_m
  use states_mxll_oct_m

  implicit none

  private
  public ::                    &
    mxll_field_to_medium_t
  
  type, extends(interaction_with_partner_t) :: mxll_field_to_medium_t
    private
    FLOAT, allocatable, public     :: partner_E_field(:,:) !< E field generated by partner
    FLOAT, allocatable, public     :: system_E_field(:,:) !< E field in system (Medium) grid
                                                          ! (B field is not needed for non-magnetic media)
    type(grid_t), pointer, public  :: system_gr !< pointer to grid of the Medium system
    integer, allocatable, public   :: partner_to_system_map(:)
    integer, public                :: partner_np

  contains
    procedure :: init => mxll_field_to_medium_init
    procedure :: calculate => mxll_field_to_medium_calculate
    procedure :: calculate_energy => mxll_field_to_medium_calculate_energy
    final :: mxll_field_to_medium_finalize
  end type mxll_field_to_medium_t


  interface mxll_field_to_medium_t
    module procedure mxll_field_to_medium_constructor
  end interface mxll_field_to_medium_t

contains

  ! ---------------------------------------------------------
  function mxll_field_to_medium_constructor(partner) result(this)
    class(interaction_partner_t), target, intent(inout) :: partner
    class(mxll_field_to_medium_t),               pointer       :: this

    PUSH_SUB(mxll_field_to_medium_constructor)

    SAFE_ALLOCATE(this)

    this%label = "mxll_field_to_medium"
    this%partner => partner

    this%n_system_quantities = 0
    SAFE_ALLOCATE(this%system_quantities(1:this%n_system_quantities))

    this%n_partner_quantities = 1
    SAFE_ALLOCATE(this%partner_quantities(1:this%n_partner_quantities))
    this%partner_quantities(1) = E_FIELD

    POP_SUB(mxll_field_to_medium_constructor)
  end function mxll_field_to_medium_constructor


  subroutine mxll_field_to_medium_init(this, gr)
    class(mxll_field_to_medium_t), intent(inout) :: this
    type(grid_t), target, intent(in)             :: gr

    PUSH_SUB(mxll_field_to_medium_init)

    this%system_gr => gr
    SAFE_ALLOCATE(this%system_E_field(1:gr%mesh%np, 1:3))
    this%system_E_field(:,:) = M_zero

    POP_SUB(mxll_field_to_medium_init)
  end subroutine mxll_field_to_medium_init

  ! ---------------------------------------------------------
  subroutine mxll_field_to_medium_finalize(this)
    type(mxll_field_to_medium_t), intent(inout) :: this

    PUSH_SUB(mxll_field_to_medium_finalize)
    
    call interaction_with_partner_end(this)
    SAFE_DEALLOCATE_A(this%partner_E_field)
    SAFE_DEALLOCATE_A(this%system_E_field)
    SAFE_DEALLOCATE_A(this%partner_to_system_map)

    POP_SUB(mxll_field_to_medium_finalize)
  end subroutine mxll_field_to_medium_finalize

  ! ---------------------------------------------------------
  subroutine mxll_field_to_medium_calculate(this)
    class(mxll_field_to_medium_t), intent(inout) :: this

    integer :: ip_med, ip
    type(profile_t), save :: prof

    PUSH_SUB(mxll_field_to_medium_calculate)

    call profiling_in(prof,"MXLL_FIELD_TO_MEDIUM_CALCULATE")

    do ip = 1, this%partner_np
      ip_med = this%partner_to_system_map(ip)
      if (ip_med > 0) then
        this%system_E_field(ip_med,1:3) = this%partner_E_field(ip,1:3)
      end if
    end do

    call profiling_out(prof)
    POP_SUB(mxll_field_to_medium_calculate)
  end subroutine mxll_field_to_medium_calculate


  ! ---------------------------------------------------------
  subroutine mxll_field_to_medium_calculate_energy(this)
    class(mxll_field_to_medium_t),    intent(inout) :: this

    PUSH_SUB(mxll_field_to_medium_calculate_energy)

    ! interaction energy is zero, since it is only re-gridding the quantities of one system
    ! on the mesh of the other
    this%energy = M_ZERO

    POP_SUB(mxll_field_to_medium_calculate_energy)
  end subroutine mxll_field_to_medium_calculate_energy

end module mxll_field_to_medium_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
