#!/usr/bin/env python3

"""
This file is used for generating i8x16 related test cases
which inherites from the 'SimdCmpCase' class and overloads
with the 'get_test_cases' method.
"""

from simd_compare import SimdCmpCase


# Generate i8x16 test case
class Simdi8x16CmpCase(SimdCmpCase):

    # set lane type
    LANE_TYPE = 'i8x16'

    BINARY_OPS = ['eq', 'ne', 'lt_s', 'lt_u', 'le_s', 'le_u', 'gt_s', 'gt_u', 'ge_s', 'ge_u']

    # Overload base class method and set test data for i32x4.
    def get_case_data(self):

        case_data = []

        # i8x16.eq  (i8x16) (i8x16)
        # hex vs hex
        case_data.append(['#', 'eq'])
        case_data.append(['#', 'i8x16.eq  (i8x16) (i8x16)'])
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['eq', ['0xFF', '0xFF'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', ['0x00', '0x00'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', ['0xF0', '0xF0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', ['0x0F', '0x0F'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', [['0xFF', '0x00'], ['0xFF', '0x00']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', [['0x00', '0xFF'], ['0x00', '0xFF']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['eq', ['0xFF', '255'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', ['0xFF', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', ['0x80', '128'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', ['0x80', '-128'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['eq', ['-1', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', ['0', '0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', ['255', '255'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', ['255', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', [['255', '0'], ['255', '0']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', [['0', '255'], ['0', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '-0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['eq', [['0x00', '0x00', '0x00', '0xc3', '0x00', '0x00', '0xfe', '0xc2', '0x00', '0x00', '0x80', '0xbf', '0x00', '0x00', '0x00', '0x00'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '-1', ['i8x16', 'f32x4', 'i8x16']])
        case_data.append(['eq', [['0x00', '0x00', '0x80', '0x3f', '0x00', '0x00', '0xfe', '0x42', '0x00', '0x00', '0x00', '0x43', '0x00', '0x00', '0x7f', '0x43'],
                          ['1.0', '127.0', '128.0', '255.0']], '-1', ['i8x16', 'f32x4', 'i8x16']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['eq', ['0x0F', '0xF0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', [['0x00', '0xFF'], ['0xFF', '0x00']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0xFF', '0xAB', '0xAA', '0x1B', '0x1A', '0x0B', '0x0A', '0x12', '0x11', '0x10', '0x09', '0x04', '0x03', '0x02', '0x01', '0x00']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['0', '0', '0', '0', '0', '0', '0', '-1', '-1', '0', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['eq', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '-0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['0', '0', '0', '0', '0', '0', '0', '-1', '-1', '0', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])

        # i8x16.eq  (i8x16) (i16x8)
        case_data.append(['#', 'i8x16.eq  (i8x16) (i16x8)'])
        case_data.append(['eq', ['0xFF', '0xFFFF'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['eq', ['255', '65535'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['eq', ['0', '0'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['eq', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E']], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['eq', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['eq', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'],
                          ['-128', '-128', '0', '0', '1', '1', '255', '255']], ['-1', '0', '-1', '0', '-1', '-1', '-1', '-1', '-1', '0', '-1', '0', '-1', '0', '-1', '0'], ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['eq', ['0x55', '0xAAAA'], '0', ['i8x16', 'i16x8', 'i8x16']])

        # i8x16.eq  (i8x16) (i32x4)
        case_data.append(['#', 'i8x16.eq  (i8x16) (i32x4)'])
        case_data.append(['eq', ['0xFF', '0xFFFFFFFF'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['eq', ['255', '4294967295'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['eq', ['0', '0'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['eq', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['eq', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['eq', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'],
                          ['-128', '0', '1', '255']], ['-1', '0', '0', '0', '-1', '-1', '-1', '-1', '-1', '0', '0', '0', '-1', '0', '0', '0'], ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['eq', ['0x55', '0xAAAAAAAA'], '0', ['i8x16', 'i32x4', 'i8x16']])

        # ne
        # i8x16.ne  (i8x16) (i8x16)
        case_data.append(['#', 'ne'])
        case_data.append(['#', 'i8x16.ne  (i8x16) (i8x16)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['ne', ['0xFF', '0xFF'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', ['0x00', '0x00'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', ['0xF0', '0xF0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', ['0x0F', '0x0F'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', [['0xFF', '0x00'], ['0xFF', '0x00']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', [['0x00', '0xFF'], ['0x00', '0xFF']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['ne', ['0xFF', '255'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', ['0xFF', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', ['0x80', '128'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', ['0x80', '-128'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['ne', ['-1', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', ['0', '0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', ['255', '255'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', ['255', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', [['255', '0'], ['255', '0']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', [['0', '255'], ['0', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '-0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['ne', [['0x00', '0x00', '0x00', '0xc3', '0x00', '0x00', '0xfe', '0xc2', '0x00', '0x00', '0x80', '0xbf', '0x00', '0x00', '0x00', '0x00'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '0', ['i8x16', 'f32x4', 'i8x16']])
        case_data.append(['ne', [['0x00', '0x00', '0x80', '0x3f', '0x00', '0x00', '0xfe', '0x42', '0x00', '0x00', '0x00', '0x43', '0x00', '0x00', '0x7f', '0x43'],
                          ['1.0', '127.0', '128.0', '255.0']], '0', ['i8x16', 'f32x4', 'i8x16']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['ne', ['0x0F', '0xF0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', [['0x00', '0xFF'], ['0xFF', '0x00']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0xFF', '0xAB', '0xAA', '0x1B', '0x1A', '0x0B', '0x0A', '0x12', '0x11', '0x10', '0x09', '0x04', '0x03', '0x02', '0x01', '0x00']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['-1', '-1', '-1', '-1', '-1', '-1', '-1', '0', '0', '-1', '-1', '-1', '-1', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ne', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '-0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['-1', '-1', '-1', '-1', '-1', '-1', '-1', '0', '0', '-1', '-1', '-1', '-1', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])

        # i8x16.ne  (i8x16) (i16x8)
        case_data.append(['#', 'i8x16.ne (i8x16) (i16x8)'])
        case_data.append(['ne', ['0xFF', '0xFFFF'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ne', ['255', '65535'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ne', ['0', '0'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ne', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E']], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ne', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ne', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'],
                          ['-128', '-128', '0', '0', '1', '1', '255', '255']], ['0', '-1', '0', '-1', '0', '0', '0', '0', '0', '-1', '0', '-1', '0', '-1', '0', '-1'], ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ne', ['0x55', '0xAAAA'], '-1', ['i8x16', 'i16x8', 'i8x16']])

        # i8x16.ne  (i8x16) (i32x4)
        case_data.append(['#', 'i8x16.ne  (i8x16) (i32x4)'])
        case_data.append(['ne', ['0xFF', '0xFFFFFFFF'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ne', ['255', '4294967295'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ne', ['0', '0'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ne', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ne', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ne', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'], ['-128', '0', '1', '255']],
                          ['0', '-1', '-1', '-1', '0', '0', '0', '0', '0', '-1', '-1', '-1', '0', '-1', '-1', '-1'], ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ne', ['0x55', '0xAAAAAAAA'], '-1', ['i8x16', 'i32x4', 'i8x16']])

        # lt_s
        case_data.append(['#', 'lt_s'])
        case_data.append(['#', 'i8x16.lt_s  (i8x16) (i8x16)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['lt_s', ['0xFF', '0xFF'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', ['0x00', '0x00'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', ['0xF0', '0xF0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', ['0x0F', '0x0F'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', [['0xFF', '0x00'], ['0xFF', '0x00']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', [['0x00', '0xFF'], ['0x00', '0xFF']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['lt_s', ['0xFF', '255'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', ['0xFF', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', ['0x80', '128'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', ['0x80', '-128'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['lt_s', ['-1', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', ['0', '0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', ['255', '255'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', ['255', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', [['255', '0'], ['255', '0']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', [['0', '255'], ['0', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '-0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['lt_s', [['0x00', '0x00', '0x00', '0xc3', '0x00', '0x00', '0xfe', '0xc2', '0x00', '0x00', '0x80', '0xbf', '0x00', '0x00', '0x00', '0x00'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '0', ['i8x16', 'f32x4', 'i8x16']])
        case_data.append(['lt_s', [['0x00', '0x00', '0x80', '0x3f', '0x00', '0x00', '0xfe', '0x42', '0x00', '0x00', '0x00', '0x43', '0x00', '0x00', '0x7f', '0x43'],
                          ['1.0', '127.0', '128.0', '255.0']], '0', ['i8x16', 'f32x4', 'i8x16']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['lt_s', ['0x0F', '0xF0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', [['0x00', '0xFF'], ['0xFF', '0x00']], ['0', '0', '0', '0', '0', '0', '0', '0', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0xFF', '0xAB', '0xAA', '0x1B', '0x1A', '0x0B', '0x0A', '0x12', '0x11', '0x10', '0x09', '0x04', '0x03', '0x02', '0x01', '0x00']],
                          ['0', '0', '0', '-1', '-1', '-1', '0', '-1', '0', '-1', '0', '0', '0', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['-1', '-1', '-1', '0', '-1', '-1', '-1', '0', '0', '0', '0', '0', '-1', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_s', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '-0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['-1', '-1', '-1', '0', '-1', '-1', '-1', '0', '0', '0', '0', '0', '-1', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['#', 'i8x16.lt_s  (i8x16) (i16x8)'])
        case_data.append(['lt_s', ['0xFF', '0xFFFF'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['lt_s', ['255', '65535'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['lt_s', ['0', '0'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['lt_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E']], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['lt_s', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['lt_s', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'], ['-128', '-128', '0', '0', '1', '1', '255', '255']],
                          ['0', '-1', '0', '-1', '0', '0', '0', '0', '0', '0', '0', '0', '0', '-1', '0', '-1'], ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['lt_s', ['0x55', '0xAAAA'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['#', 'i8x16.lt_s  (i8x16) (i32x4)'])
        case_data.append(['lt_s', ['0xFF', '0xFFFFFFFF'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['lt_s', ['255', '4294967295'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['lt_s', ['0', '0'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['lt_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['lt_s', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['lt_s', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'], ['-128', '0', '1', '255']],
                          ['0', '-1', '-1', '-1', '0', '0', '0', '0', '0', '0', '0', '0', '0', '-1', '-1', '-1'], ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['lt_s', ['0x55', '0xAAAAAAAA'], '0', ['i8x16', 'i32x4', 'i8x16']])

        # lt_u
        case_data.append(['#', 'lt_u'])
        case_data.append(['#', 'i8x16.lt_u  (i8x16) (i8x16)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['lt_u', ['0xFF', '0xFF'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', ['0x00', '0x00'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', ['0xF0', '0xF0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', ['0x0F', '0x0F'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', [['0xFF', '0x00'], ['0xFF', '0x00']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', [['0x00', '0xFF'], ['0x00', '0xFF']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['lt_u', ['0xFF', '255'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', ['0xFF', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', ['0x80', '128'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', ['0x80', '-128'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['lt_u', ['-1', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', ['0', '0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', ['255', '255'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', ['255', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', [['255', '0'], ['255', '0']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', [['0', '255'], ['0', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '-0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['lt_u', [['0x00', '0x00', '0x00', '0xc3', '0x00', '0x00', '0xfe', '0xc2', '0x00', '0x00', '0x80', '0xbf', '0x00', '0x00', '0x00', '0x00'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '0', ['i8x16', 'f32x4', 'i8x16']])
        case_data.append(['lt_u', [['0x00', '0x00', '0x80', '0x3f', '0x00', '0x00', '0xfe', '0x42', '0x00', '0x00', '0x00', '0x43', '0x00', '0x00', '0x7f', '0x43'],
                          ['1.0', '127.0', '128.0', '255.0']], '0', ['i8x16', 'f32x4', 'i8x16']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['lt_u', ['0x0F', '0xF0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', [['0x00', '0xFF'], ['0xFF', '0x00']], ['-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '0', '0', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0xFF', '0xAB', '0xAA', '0x1B', '0x1A', '0x0B', '0x0A', '0x12', '0x11', '0x10', '0x09', '0x04', '0x03', '0x02', '0x01', '0x00']],
                          ['-1', '-1', '-1', '-1', '-1', '-1', '0', '-1', '0', '-1', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['-1', '-1', '-1', '0', '0', '0', '0', '0', '0', '-1', '-1', '-1', '-1', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['lt_u', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '-0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['-1', '-1', '-1', '0', '0', '0', '0', '0', '0', '-1', '-1', '-1', '-1', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['#', 'i8x16.lt_u  (i8x16) (i16x8)'])
        case_data.append(['lt_u', ['0xFF', '0xFFFF'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['lt_u', ['255', '65535'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['lt_u', ['0', '0'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['lt_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E']], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['lt_u', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['lt_u', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'],
                          ['-128', '-128', '0', '0', '1', '1', '255', '255']], ['0', '-1', '0', '-1', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['lt_u', ['0x55', '0xAAAA'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['#', 'i8x16.lt_u  (i8x16) (i32x4)'])
        case_data.append(['lt_u', ['0xFF', '0xFFFFFFFF'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['lt_u', ['255', '4294967295'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['lt_u', ['0', '0'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['lt_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['lt_u', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['lt_u', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'],
                          ['-128', '0', '1', '255']], ['0', '-1', '-1', '-1', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['lt_u', ['0x55', '0xAAAAAAAA'], '-1', ['i8x16', 'i32x4', 'i8x16']])

        # le_s
        case_data.append(['#', 'le_s'])
        case_data.append(['#', 'i8x16.le_s  (i8x16) (i8x16)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['le_s', ['0xFF', '0xFF'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', ['0x00', '0x00'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', ['0xF0', '0xF0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', ['0x0F', '0x0F'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', [['0xFF', '0x00'], ['0xFF', '0x00']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', [['0x00', '0xFF'], ['0x00', '0xFF']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF']], '-1', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['le_s', ['0xFF', '255'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', ['0xFF', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', ['0x80', '128'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', ['0x80', '-128'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['le_s', ['-1', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', ['0', '0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', ['255', '255'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', ['255', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', [['255', '0'], ['255', '0']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', [['0', '255'], ['0', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '-0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['le_s', [['0x00', '0x00', '0x00', '0xc3', '0x00', '0x00', '0xfe', '0xc2', '0x00', '0x00', '0x80', '0xbf', '0x00', '0x00', '0x00', '0x00'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '-1', ['i8x16', 'f32x4', 'i8x16']])
        case_data.append(['le_s', [['0x00', '0x00', '0x80', '0x3f', '0x00', '0x00', '0xfe', '0x42', '0x00', '0x00', '0x00', '0x43', '0x00', '0x00', '0x7f', '0x43'],
                          ['1.0', '127.0', '128.0', '255.0']], '-1', ['i8x16', 'f32x4', 'i8x16']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['le_s', ['0x0F', '0xF0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', [['0x00', '0xFF'], ['0xFF', '0x00']], ['0', '0', '0', '0', '0', '0', '0', '0', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0xFF', '0xAB', '0xAA', '0x1B', '0x1A', '0x0B', '0x0A', '0x12', '0x11', '0x10', '0x09', '0x04', '0x03', '0x02', '0x01', '0x00']],
                          ['0', '0', '0', '-1', '-1', '-1', '0', '-1', '0', '-1', '0', '0', '0', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['-1', '-1', '-1', '0', '-1', '-1', '-1', '-1', '-1', '0', '0', '0', '-1', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_s', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '-0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['-1', '-1', '-1', '0', '-1', '-1', '-1', '-1', '-1', '0', '0', '0', '-1', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['#', 'i8x16.le_s  (i8x16) (i16x8)'])
        case_data.append(['le_s', ['0xFF', '0xFFFF'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['le_s', ['255', '65535'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['le_s', ['0', '0'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['le_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E']], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['le_s', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['le_s', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'],
                          ['-128', '-128', '0', '0', '1', '1', '255', '255']],
                          ['-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '0', '-1', '0', '-1', '-1', '-1', '-1'], ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['le_s', ['0x55', '0xAAAA'], ['0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['#', 'i8x16.le_s  (i8x16) (i32x4)'])
        case_data.append(['le_s', ['0xFF', '0xFFFFFFFF'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['le_s', ['255', '4294967295'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['le_s', ['0', '0'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['le_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['le_s', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['le_s', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'], ['-128', '0', '1', '255']],
                          ['-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '0', '0', '0', '-1', '-1', '-1', '-1'], ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['le_s', ['0x55', '0xAAAAAAAA'], '0', ['i8x16', 'i32x4', 'i8x16']])

        # le_u
        case_data.append(['#', 'le_u'])
        case_data.append(['#', 'i8x16.le_u  (i8x16) (i8x16)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['le_u', ['0xFF', '0xFF'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', ['0x00', '0x00'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', ['0xF0', '0xF0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', ['0x0F', '0x0F'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', [['0xFF', '0x00'], ['0xFF', '0x00']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', [['0x00', '0xFF'], ['0x00', '0xFF']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF']], '-1', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['le_u', ['0xFF', '255'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', ['0xFF', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', ['0x80', '128'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', ['0x80', '-128'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['le_u', ['-1', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', ['0', '0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', ['255', '255'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', ['255', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', [['255', '0'], ['255', '0']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', [['0', '255'], ['0', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '-0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['le_u', [['0x00', '0x00', '0x00', '0xc3', '0x00', '0x00', '0xfe', '0xc2', '0x00', '0x00', '0x80', '0xbf', '0x00', '0x00', '0x00', '0x00'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '-1', ['i8x16', 'f32x4', 'i8x16']])
        case_data.append(['le_u', [['0x00', '0x00', '0x80', '0x3f', '0x00', '0x00', '0xfe', '0x42', '0x00', '0x00', '0x00', '0x43', '0x00', '0x00', '0x7f', '0x43'],
                          ['1.0', '127.0', '128.0', '255.0']], '-1', ['i8x16', 'f32x4', 'i8x16']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['le_u', ['0x0F', '0xF0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', [['0x00', '0xFF'], ['0xFF', '0x00']], ['-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '0', '0', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0xFF', '0xAB', '0xAA', '0x1B', '0x1A', '0x0B', '0x0A', '0x12', '0x11', '0x10', '0x09', '0x04', '0x03', '0x02', '0x01', '0x00']],
                          ['-1', '-1', '-1', '-1', '-1', '-1', '0', '-1', '0', '-1', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['-1', '-1', '-1', '0', '0', '0', '0', '-1', '-1', '-1', '-1', '-1', '-1', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['le_u', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '-0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['-1', '-1', '-1', '0', '0', '0', '0', '-1', '-1', '-1', '-1', '-1', '-1', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['#', 'i8x16.le_u  (i8x16) (i16x8)'])
        case_data.append(['le_u', ['0xFF', '0xFFFF'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['le_u', ['255', '65535'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['le_u', ['0', '0'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['le_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E']], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['le_u', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['le_u', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'],
                          ['-128', '-128', '0', '0', '1', '1', '255', '255']], ['-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '0', '-1', '0', '-1', '0', '-1', '0'], ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['le_u', ['0x55', '0xAAAA'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['#', 'i8x16.le_u  (i8x16) (i32x4)'])
        case_data.append(['le_u', ['0xFF', '0xFFFFFFFF'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['le_u', ['255', '4294967295'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['le_u', ['0', '0'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['le_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['le_u', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['le_u', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'], ['-128', '0', '1', '255']],
                          ['-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '0', '0', '0', '-1', '0', '0', '0'], ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['le_u', ['0x55', '0xAAAAAAAA'], '-1', ['i8x16', 'i32x4', 'i8x16']])

        # gt_s
        case_data.append(['#', 'gt_s'])
        case_data.append(['#', 'i8x16.gt_s  (i8x16) (i8x16)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['gt_s', ['0xFF', '0xFF'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', ['0x00', '0x00'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', ['0xF0', '0xF0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', ['0x0F', '0x0F'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', [['0xFF', '0x00'], ['0xFF', '0x00']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', [['0x00', '0xFF'], ['0x00', '0xFF']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['gt_s', ['0xFF', '255'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', ['0xFF', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', ['0x80', '128'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', ['0x80', '-128'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['gt_s', ['-1', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', ['0', '0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', ['255', '255'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', ['255', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', [['255', '0'], ['255', '0']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', [['0', '255'], ['0', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '-0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['gt_s', [['0x00', '0x00', '0x00', '0xc3', '0x00', '0x00', '0xfe', '0xc2', '0x00', '0x00', '0x80', '0xbf', '0x00', '0x00', '0x00', '0x00'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '0', ['i8x16', 'f32x4', 'i8x16']])
        case_data.append(['gt_s', [['0x00', '0x00', '0x80', '0x3f', '0x00', '0x00', '0xfe', '0x42', '0x00', '0x00', '0x00', '0x43', '0x00', '0x00', '0x7f', '0x43'],
                          ['1.0', '127.0', '128.0', '255.0']], '0', ['i8x16', 'f32x4', 'i8x16']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['gt_s', ['0x0F', '0xF0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', [['0x00', '0xFF'], ['0xFF', '0x00']], ['-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '0', '0', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0xFF', '0xAB', '0xAA', '0x1B', '0x1A', '0x0B', '0x0A', '0x12', '0x11', '0x10', '0x09', '0x04', '0x03', '0x02', '0x01', '0x00']],
                          ['-1', '-1', '-1', '0', '0', '0', '-1', '0', '-1', '0', '-1', '-1', '-1', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['0', '0', '0', '-1', '0', '0', '0', '0', '0', '-1', '-1', '-1', '0', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_s', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '-0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['0', '0', '0', '-1', '0', '0', '0', '0', '0', '-1', '-1', '-1', '0', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['#', 'i8x16.gt_s  (i8x16) (i16x8)'])
        case_data.append(['gt_s', ['0xFF', '0xFFFF'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['gt_s', ['255', '65535'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['gt_s', ['0', '0'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['gt_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E']], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['gt_s', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['gt_s', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'], ['-128', '-128', '0', '0', '1', '1', '255', '255']],
                          ['0', '0', '0', '0', '0', '0', '0', '0', '0', '-1', '0', '-1', '0', '0', '0', '0'], ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['gt_s', ['0x55', '0xAAAA'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['#', 'i8x16.gt_s  (i8x16) (i32x4)'])
        case_data.append(['gt_s', ['0xFF', '0xFFFFFFFF'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['gt_s', ['255', '4294967295'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['gt_s', ['0', '0'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['gt_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['gt_s', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'], ['2206368128', '16776957', '2130837760', '4294901120']],
                          ['0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['gt_s', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'], ['-128', '0', '1', '255']],
                          ['0', '0', '0', '0', '0', '0', '0', '0', '0', '-1', '-1', '-1', '0', '0', '0', '0'], ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['gt_s', ['0x55', '0xAAAAAAAA'], '-1', ['i8x16', 'i32x4', 'i8x16']])

        # gt_u
        case_data.append(['#', 'gt_u'])
        case_data.append(['#', 'i8x16.gt_u  (i8x16) (i8x16)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['gt_u', ['0xFF', '0xFF'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', ['0x00', '0x00'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', ['0xF0', '0xF0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', ['0x0F', '0x0F'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', [['0xFF', '0x00'], ['0xFF', '0x00']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', [['0x00', '0xFF'], ['0x00', '0xFF']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['gt_u', ['0xFF', '255'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', ['0xFF', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', ['0x80', '128'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', ['0x80', '-128'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['gt_u', ['-1', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', ['0', '0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', ['255', '255'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', ['255', '-1'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', [['255', '0'], ['255', '0']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', [['0', '255'], ['0', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '-0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['gt_u', [['0x00', '0x00', '0x00', '0xc3', '0x00', '0x00', '0xfe', '0xc2', '0x00', '0x00', '0x80', '0xbf', '0x00', '0x00', '0x00', '0x00'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '0', ['i8x16', 'f32x4', 'i8x16']])
        case_data.append(['gt_u', [['0x00', '0x00', '0x80', '0x3f', '0x00', '0x00', '0xfe', '0x42', '0x00', '0x00', '0x00', '0x43', '0x00', '0x00', '0x7f', '0x43'],
                          ['1.0', '127.0', '128.0', '255.0']], '0', ['i8x16', 'f32x4', 'i8x16']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['gt_u', ['0x0F', '0xF0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', [['0x00', '0xFF'], ['0xFF', '0x00']], ['0', '0', '0', '0', '0', '0', '0', '0', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0xFF', '0xAB', '0xAA', '0x1B', '0x1A', '0x0B', '0x0A', '0x12', '0x11', '0x10', '0x09', '0x04', '0x03', '0x02', '0x01', '0x00']],
                          ['0', '0', '0', '0', '0', '0', '-1', '0', '-1', '0', '-1', '-1', '-1', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['0', '0', '0', '-1', '-1', '-1', '-1', '0', '0', '0', '0', '0', '0', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['gt_u', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '-0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['0', '0', '0', '-1', '-1', '-1', '-1', '0', '0', '0', '0', '0', '0', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['#', 'i8x16.gt_u  (i8x16) (i16x8)'])
        case_data.append(['gt_u', ['0xFF', '0xFFFF'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['gt_u', ['255', '65535'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['gt_u', ['0', '0'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['gt_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E']], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['gt_u', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['gt_u', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'], ['-128', '-128', '0', '0', '1', '1', '255', '255']],
                          ['0', '0', '0', '0', '0', '0', '0', '0', '0', '-1', '0', '-1', '0', '-1', '0', '-1'], ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['gt_u', ['0x55', '0xAAAA'], '0', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['#', 'i8x16.gt_u  (i8x16) (i32x4)'])
        case_data.append(['gt_u', ['0xFF', '0xFFFFFFFF'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['gt_u', ['255', '4294967295'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['gt_u', ['0', '0'], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['gt_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '0', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['gt_u', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'], ['2206368128', '16776957', '2130837760', '4294901120']],
                          ['0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['gt_u', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'], ['-128', '0', '1', '255']],
                          ['0', '0', '0', '0', '0', '0', '0', '0', '0', '-1', '-1', '-1', '0', '-1', '-1', '-1'], ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['gt_u', ['0x55', '0xAAAAAAAA'], ['0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i32x4', 'i8x16']])

        # ge_s
        case_data.append(['#', 'ge_s'])
        case_data.append(['#', 'i8x16.ge_s  (i8x16) (i8x16)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['ge_s', ['0xFF', '0xFF'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', ['0x00', '0x00'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', ['0xF0', '0xF0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', ['0x0F', '0x0F'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', [['0xFF', '0x00'], ['0xFF', '0x00']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', [['0x00', '0xFF'], ['0x00', '0xFF']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF']], '-1', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['ge_s', ['0xFF', '255'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', ['0xFF', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', ['0x80', '128'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', ['0x80', '-128'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['ge_s', ['-1', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', ['0', '0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', ['255', '255'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', ['255', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', [['255', '0'], ['255', '0']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', [['0', '255'], ['0', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '-0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['ge_s', [['0x00', '0x00', '0x00', '0xc3', '0x00', '0x00', '0xfe', '0xc2', '0x00', '0x00', '0x80', '0xbf', '0x00', '0x00', '0x00', '0x00'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '-1', ['i8x16', 'f32x4', 'i8x16']])
        case_data.append(['ge_s', [['0x00', '0x00', '0x80', '0x3f', '0x00', '0x00', '0xfe', '0x42', '0x00', '0x00', '0x00', '0x43', '0x00', '0x00', '0x7f', '0x43'],
                          ['1.0', '127.0', '128.0', '255.0']], '-1', ['i8x16', 'f32x4', 'i8x16']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['ge_s', ['0x0F', '0xF0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', [['0x00', '0xFF'], ['0xFF', '0x00']], ['-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '0', '0', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0xFF', '0xAB', '0xAA', '0x1B', '0x1A', '0x0B', '0x0A', '0x12', '0x11', '0x10', '0x09', '0x04', '0x03', '0x02', '0x01', '0x00']],
                          ['-1', '-1', '-1', '0', '0', '0', '-1', '0', '-1', '0', '-1', '-1', '-1', '0', '0', '0'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['0', '0', '0', '-1', '0', '0', '0', '-1', '-1', '-1', '-1', '-1', '0', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_s', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '-0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['0', '0', '0', '-1', '0', '0', '0', '-1', '-1', '-1', '-1', '-1', '0', '-1', '-1', '-1'],
                          ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['#', 'i8x16.ge_s  (i8x16) (i16x8)'])
        case_data.append(['ge_s', ['0xFF', '0xFFFF'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ge_s', ['255', '65535'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ge_s', ['0', '0'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ge_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E']], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ge_s', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ge_s', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'], ['-128', '-128', '0', '0', '1', '1', '255', '255']],
                          ['-1', '0', '-1', '0', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '0', '-1', '0'], ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ge_s', ['0x55', '0xAAAA'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['#', 'i8x16.ge_s  (i8x16) (i32x4)'])
        case_data.append(['ge_s', ['0xFF', '0xFFFFFFFF'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ge_s', ['255', '4294967295'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ge_s', ['0', '0'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ge_s', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ge_s', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ge_s', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'], ['-128', '0', '1', '255']],
                          ['-1', '0', '0', '0', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '0', '0', '0'], ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ge_s', ['0x55', '0xAAAAAAAA'], '-1', ['i8x16', 'i32x4', 'i8x16']])

        # ge_u
        # i8x16.ge_u  (i8x16) (i8x16)
        case_data.append(['#', 'ge_u'])
        case_data.append(['#', 'i8x16.ge_u  (i8x16) (i8x16)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['ge_u', ['0xFF', '0xFF'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', ['0x00', '0x00'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', ['0xF0', '0xF0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', ['0x0F', '0x0F'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', [['0xFF', '0x00'], ['0xFF', '0x00']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', [['0x00', '0xFF'], ['0x00', '0xFF']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF']], '-1', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['ge_u', ['0xFF', '255'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', ['0xFF', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', ['0x80', '128'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', ['0x80', '-128'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['ge_u', ['-1', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', ['0', '0'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', ['255', '255'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', ['255', '-1'], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', [['255', '0'], ['255', '0']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', [['0', '255'], ['0', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '-0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i8x16', 'i8x16', 'i8x16']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['ge_u', [['0x00', '0x00', '0x00', '0xc3', '0x00', '0x00', '0xfe', '0xc2', '0x00', '0x00', '0x80', '0xbf', '0x00', '0x00', '0x00', '0x00'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '-1', ['i8x16', 'f32x4', 'i8x16']])
        case_data.append(['ge_u', [['0x00', '0x00', '0x80', '0x3f', '0x00', '0x00', '0xfe', '0x42', '0x00', '0x00', '0x00', '0x43', '0x00', '0x00', '0x7f', '0x43'],
                          ['1.0', '127.0', '128.0', '255.0']], '-1', ['i8x16', 'f32x4', 'i8x16']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['ge_u', ['0x0F', '0xF0'], '0', ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', [['0x00', '0xFF'], ['0xFF', '0x00']], ['0', '0', '0', '0', '0', '0', '0', '0', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x09', '0x10', '0x11', '0x12', '0x0A', '0x0B', '0x1A', '0x1B', '0xAA', '0xAB', '0xFF'],
                          ['0xFF', '0xAB', '0xAA', '0x1B', '0x1A', '0x0B', '0x0A', '0x12', '0x11', '0x10', '0x09', '0x04', '0x03', '0x02', '0x01', '0x00']],
                          ['0', '0', '0', '0', '0', '0', '-1', '0', '-1', '0', '-1', '-1', '-1', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', [['0x80', '0x81', '0x82', '0x83', '0xFD', '0xFE', '0xFF', '0x00', '0x00', '0x01', '0x02', '0x7F', '0x80', '0xFD', '0xFE', '0xFF'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['0', '0', '0', '-1', '-1', '-1', '-1', '-1', '-1', '0', '0', '0', '0', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])
        case_data.append(['ge_u', [['128', '129', '130', '131', '253', '254', '255', '-0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['255', '254', '253', '128', '127', '2', '1', '0', '-0', '-1', '-2', '-3', '-125', '-126', '-127', '-128']],
                          ['0', '0', '0', '-1', '-1', '-1', '-1', '-1', '-1', '0', '0', '0', '0', '-1', '-1', '-1'], ['i8x16', 'i8x16', 'i8x16']])

        # i8x16.ge_u (i8x16) (i16x8)
        case_data.append(['#', 'i8x16.ge_u (i8x16) (i16x8)'])
        case_data.append(['ge_u', ['0xFF', '0xFFFF'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ge_u', ['255', '65535'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ge_u', ['0', '0'], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ge_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E']], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ge_u', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '-1', ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ge_u', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'],
                          ['-128', '-128', '0', '0', '1', '1', '255', '255']], ['-1', '0', '-1', '0', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1'], ['i8x16', 'i16x8', 'i8x16']])
        case_data.append(['ge_u', ['0x55', '0xAAAA'], ['0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i16x8', 'i8x16']])

        # i8x16.ge_u (i8x16) (i32x4)
        case_data.append(['#', 'i8x16.ge_u (i8x16) (i32x4)'])
        case_data.append(['ge_u', ['0xFF', '0xFFFFFFFF'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ge_u', ['255', '4294967295'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ge_u', ['0', '0'], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ge_u', [['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ge_u', [['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '-1', ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ge_u', [['-128', '-128', '-128', '-128', '0', '0', '0', '0', '1', '1', '1', '1', '255', '255', '255', '255'],
                          ['-128', '0', '1', '255']], ['-1', '0', '0', '0', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1', '-1'], ['i8x16', 'i32x4', 'i8x16']])
        case_data.append(['ge_u', ['0x55', '0xAAAAAAAA'], ['0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0'], ['i8x16', 'i32x4', 'i8x16']])

        return case_data


def gen_test_cases():
    i8x16 = Simdi8x16CmpCase()
    i8x16.gen_test_cases()


if __name__ == '__main__':
    i8x16 = Simdi8x16CmpCase()
    i8x16.gen_test_cases()
