module Spec.Frodo.Test

open FStar.Mul
open Lib.IntTypes
open Lib.RawIntTypes
open Lib.Sequence
open Lib.ByteSequence

open Spec.Frodo.KEM.KeyGen
open Spec.Frodo.KEM.Encaps
open Spec.Frodo.KEM.Decaps
open Spec.Frodo.Params

open FStar.All

// Reducing e.g. v (size 64) to 64 requires access to Lib.IntTypes implementation,
// hence why we befriend Lib.IntTypes although it might be possible to prove this
// opaquely.
friend Lib.IntTypes

#reset-options "--max_fuel 0 --max_ifuel 1"

let print_and_compare (#len: size_nat) (test_expected: lbytes len) (test_result: lbytes len)
  : ML bool =
  IO.print_string "\nResult:   ";
  List.iter (fun a -> IO.print_string (UInt8.to_string (u8_to_UInt8 a))) (to_list test_result);
  IO.print_string "\nExpected: ";
  List.iter (fun a -> IO.print_string (UInt8.to_string (u8_to_UInt8 a))) (to_list test_expected);
  for_all2 (fun a b -> uint_to_nat #U8 a = uint_to_nat #U8 b) test_expected test_result

let compare (#len: size_nat) (test_expected: lbytes len) (test_result: lbytes len) =
  for_all2 (fun a b -> uint_to_nat #U8 a = uint_to_nat #U8 b) test_expected test_result

let test_frodo
  (keypaircoins: list uint8 {List.Tot.length keypaircoins == 2 * crypto_bytes + bytes_seed_a})
  (enccoins: list uint8 {List.Tot.length enccoins == bytes_mu})
  (ss_expected: list uint8 {List.Tot.length ss_expected == crypto_bytes})
  (pk_expected: list uint8 {List.Tot.length pk_expected == crypto_publickeybytes})
  (ct_expected: list uint8 {List.Tot.length ct_expected == crypto_ciphertextbytes})
  (sk_expected: list uint8 {List.Tot.length sk_expected == crypto_secretkeybytes})
  : ML bool =
  let keypaircoins = createL keypaircoins in
  let enccoins = createL enccoins in
  let ss_expected = createL ss_expected in
  let pk_expected = createL pk_expected in
  let ct_expected = createL ct_expected in
  let sk_expected = createL sk_expected in
  let pk, sk = crypto_kem_keypair_ keypaircoins in
  let ct, ss1 = crypto_kem_enc_ enccoins pk in
  let ss2 = crypto_kem_dec ct sk in
  let r_pk = compare pk_expected pk in
  let r_sk = compare sk_expected sk in
  let r_ct = compare ct_expected ct in
  let r_ss = print_and_compare ss1 ss2 in
  let r_ss1 = print_and_compare ss_expected ss2 in
  r_pk && r_sk && r_ct && r_ss && r_ss1

//
// Test1. FrodoKEM-64. AES128
//
let test1_keypaircoins =
  List.Tot.map u8_from_UInt8
    [
      0x4buy; 0x62uy; 0x2duy; 0xe1uy; 0x35uy; 0x01uy; 0x19uy; 0xc4uy; 0x5auy; 0x9fuy; 0x2euy; 0x2euy;
      0xf3uy; 0xdcuy; 0x5duy; 0xf5uy; 0x0auy; 0x75uy; 0x9duy; 0x13uy; 0x8cuy; 0xdfuy; 0xbduy; 0x64uy;
      0xc8uy; 0x1cuy; 0xc7uy; 0xccuy; 0x2fuy; 0x51uy; 0x33uy; 0x45uy; 0xd5uy; 0xa4uy; 0x5auy; 0x4cuy;
      0xeduy; 0x06uy; 0x40uy; 0x3cuy; 0x55uy; 0x57uy; 0xe8uy; 0x71uy; 0x13uy; 0xcbuy; 0x30uy; 0xeauy
    ]

let test1_enccoins =
  List.Tot.map u8_from_UInt8
    [
      0x08uy; 0xe2uy; 0x55uy; 0x38uy; 0x48uy; 0x4cuy; 0xd7uy; 0xf1uy; 0x61uy; 0x32uy; 0x48uy; 0xfeuy;
      0x6cuy; 0x9fuy; 0x6buy; 0x4euy
    ]

let test1_pk_expected =
  List.Tot.map u8_from_UInt8
    [
      0x92uy; 0x44uy; 0x4duy; 0x91uy; 0xa2uy; 0xaduy; 0xaduy; 0x05uy; 0x2cuy; 0xa2uy; 0x3duy; 0xe5uy;
      0xfbuy; 0x9duy; 0xf9uy; 0xe1uy; 0xccuy; 0xc4uy; 0x1buy; 0x9euy; 0x7buy; 0x1buy; 0x52uy; 0x1auy;
      0xbeuy; 0x8euy; 0xc1uy; 0xd0uy; 0x62uy; 0x60uy; 0x53uy; 0x71uy; 0xb6uy; 0xc4uy; 0x42uy; 0x5euy;
      0xa7uy; 0x3fuy; 0x37uy; 0x02uy; 0xdeuy; 0xf5uy; 0x26uy; 0x8buy; 0xccuy; 0x1auy; 0xc1uy; 0x64uy;
      0x7cuy; 0x9euy; 0x14uy; 0x0buy; 0x74uy; 0x6auy; 0xc9uy; 0x2fuy; 0xd2uy; 0xb3uy; 0x9duy; 0xfeuy;
      0xfauy; 0xa4uy; 0x08uy; 0x0duy; 0x57uy; 0xa3uy; 0xf8uy; 0xc3uy; 0xefuy; 0x9buy; 0x6cuy; 0x29uy;
      0x0fuy; 0x3fuy; 0xa0uy; 0x60uy; 0xe6uy; 0x4auy; 0x6buy; 0xa9uy; 0x63uy; 0xc7uy; 0x12uy; 0xc4uy;
      0xeeuy; 0x01uy; 0x89uy; 0xc8uy; 0x9duy; 0x41uy; 0xf7uy; 0x2duy; 0xeduy; 0xf6uy; 0xdcuy; 0x0buy;
      0x97uy; 0x94uy; 0x7fuy; 0xf2uy; 0x20uy; 0x86uy; 0xbduy; 0x39uy; 0x7cuy; 0x4auy; 0x59uy; 0x1duy;
      0xf0uy; 0xcduy; 0x3auy; 0xa1uy; 0x9buy; 0xb7uy; 0x27uy; 0x50uy; 0x82uy; 0xffuy; 0x2duy; 0xcduy;
      0x6cuy; 0x6buy; 0x6fuy; 0x28uy; 0x38uy; 0x34uy; 0x18uy; 0x09uy; 0x69uy; 0x1duy; 0x2fuy; 0x52uy;
      0xd0uy; 0x1duy; 0xdfuy; 0x70uy; 0xd6uy; 0x36uy; 0x48uy; 0xe4uy; 0x17uy; 0x26uy; 0x82uy; 0x01uy;
      0x70uy; 0x96uy; 0x60uy; 0x25uy; 0x21uy; 0xf2uy; 0xc3uy; 0x59uy; 0x7duy; 0xbcuy; 0x6cuy; 0x31uy;
      0xc7uy; 0xb5uy; 0x17uy; 0xdduy; 0x9cuy; 0xf2uy; 0xdcuy; 0x90uy; 0xc2uy; 0x02uy; 0xe0uy; 0x37uy;
      0xc8uy; 0xacuy; 0x55uy; 0xb5uy; 0xd6uy; 0xe9uy; 0x1fuy; 0x82uy; 0xc0uy; 0x70uy; 0x14uy; 0x0auy;
      0x6fuy; 0x03uy; 0x47uy; 0xa6uy; 0x5duy; 0x37uy; 0x55uy; 0xa2uy; 0xc0uy; 0xd3uy; 0xe7uy; 0xf2uy;
      0x7duy; 0x8euy; 0x77uy; 0x89uy; 0x94uy; 0x74uy; 0x3cuy; 0xe5uy; 0xbeuy; 0x34uy; 0x41uy; 0x30uy;
      0x59uy; 0x69uy; 0x48uy; 0xafuy; 0x28uy; 0xb7uy; 0xacuy; 0x96uy; 0x32uy; 0x0fuy; 0xc0uy; 0x4auy;
      0xcduy; 0x6duy; 0x1fuy; 0x6euy; 0x7euy; 0x71uy; 0xf8uy; 0xd8uy; 0xd0uy; 0x27uy; 0x3cuy; 0x50uy;
      0xeauy; 0x8euy; 0x6auy; 0x3buy; 0x4duy; 0x20uy; 0x49uy; 0xd5uy; 0x2euy; 0x6buy; 0xa8uy; 0xb7uy;
      0x7auy; 0x9buy; 0x9euy; 0xf8uy; 0xd7uy; 0x52uy; 0x67uy; 0x8euy; 0xbfuy; 0x95uy; 0x76uy; 0x5euy;
      0x28uy; 0x6euy; 0xa7uy; 0x14uy; 0x45uy; 0x38uy; 0xd6uy; 0x5duy; 0x28uy; 0x69uy; 0x49uy; 0xb5uy;
      0x38uy; 0xf2uy; 0x52uy; 0x17uy; 0xd4uy; 0xb2uy; 0xd5uy; 0x33uy; 0x60uy; 0xf7uy; 0x1duy; 0x3euy;
      0x64uy; 0xdbuy; 0x34uy; 0xaeuy; 0xb6uy; 0xb6uy; 0x23uy; 0x62uy; 0x3euy; 0xa7uy; 0x01uy; 0x08uy;
      0xfduy; 0x37uy; 0xe5uy; 0x91uy; 0x1cuy; 0xa7uy; 0x1buy; 0x30uy; 0x21uy; 0x4duy; 0x9buy; 0xecuy;
      0x7cuy; 0x51uy; 0x0buy; 0x3fuy; 0x16uy; 0x7buy; 0xe5uy; 0x45uy; 0x76uy; 0x62uy; 0xa7uy; 0x61uy;
      0x41uy; 0xbduy; 0xc5uy; 0x05uy; 0xeeuy; 0x3buy; 0x60uy; 0xe7uy; 0x6auy; 0xfbuy; 0xd0uy; 0x00uy;
      0x98uy; 0xcauy; 0xb4uy; 0xb9uy; 0x2euy; 0x4fuy; 0xe0uy; 0xe4uy; 0x80uy; 0xfeuy; 0x71uy; 0x27uy;
      0x48uy; 0x01uy; 0xd4uy; 0x05uy; 0x97uy; 0x9buy; 0xfcuy; 0xe9uy; 0x5buy; 0xe0uy; 0xc4uy; 0x24uy;
      0xe8uy; 0x9buy; 0x76uy; 0xecuy; 0x1cuy; 0xb9uy; 0xafuy; 0xe3uy; 0x41uy; 0x7euy; 0xc5uy; 0x3fuy;
      0x40uy; 0x02uy; 0x85uy; 0xf4uy; 0xccuy; 0x04uy; 0xd8uy; 0x1duy; 0x1fuy; 0x4cuy; 0xdbuy; 0xc9uy;
      0x9duy; 0x07uy; 0x58uy; 0x3cuy; 0x76uy; 0x6fuy; 0xa4uy; 0xeduy; 0xf1uy; 0x78uy; 0xa2uy; 0x6auy;
      0x82uy; 0x84uy; 0xdeuy; 0x6buy; 0x0auy; 0x93uy; 0x8cuy; 0xdauy; 0xbauy; 0x64uy; 0x20uy; 0x00uy;
      0xb2uy; 0xe8uy; 0x24uy; 0xc1uy; 0xa4uy; 0x93uy; 0x6cuy; 0x04uy; 0xe8uy; 0x23uy; 0xb5uy; 0x4fuy;
      0xc2uy; 0x84uy; 0x10uy; 0xd5uy; 0x0duy; 0x8buy; 0x49uy; 0xb4uy; 0x2fuy; 0xabuy; 0x5fuy; 0xbduy;
      0x40uy; 0xa5uy; 0x71uy; 0xf7uy; 0x1buy; 0x9euy; 0x8fuy; 0x46uy; 0x67uy; 0x5duy; 0x63uy; 0x51uy;
      0x27uy; 0x49uy; 0x65uy; 0x95uy; 0x58uy; 0x22uy; 0x32uy; 0xafuy; 0x32uy; 0x32uy; 0x54uy; 0xf0uy;
      0xeauy; 0x3cuy; 0x12uy; 0x71uy; 0x74uy; 0x9buy; 0xe9uy; 0xb8uy; 0xbbuy; 0x81uy; 0xb5uy; 0x7duy;
      0xd3uy; 0x77uy; 0x0duy; 0x67uy; 0xb6uy; 0x4duy; 0x92uy; 0x16uy; 0xf3uy; 0x20uy; 0xb8uy; 0x02uy;
      0x39uy; 0xc2uy; 0x21uy; 0x2auy; 0xceuy; 0xecuy; 0xb3uy; 0xa8uy; 0x14uy; 0x1duy; 0x79uy; 0x5fuy;
      0xe4uy; 0x37uy; 0x69uy; 0xf5uy; 0x31uy; 0x39uy; 0x11uy; 0x3buy; 0x3cuy; 0xf0uy; 0xabuy; 0xf0uy;
      0xc4uy; 0x97uy; 0xbauy; 0x98uy; 0x5cuy; 0x48uy; 0x96uy; 0x51uy; 0xeauy; 0x0fuy; 0x82uy; 0xc3uy;
      0x09uy; 0x85uy; 0x49uy; 0x89uy; 0x4duy; 0x33uy; 0xabuy; 0x12uy; 0xabuy; 0x7buy; 0x7buy; 0x15uy;
      0x1cuy; 0xd5uy; 0x1auy; 0x84uy; 0x99uy; 0x16uy; 0x68uy; 0x1buy; 0xffuy; 0x20uy; 0x99uy; 0x4euy;
      0x35uy; 0x9auy; 0xd0uy; 0xbauy; 0x34uy; 0x47uy; 0xf2uy; 0x1duy; 0x73uy; 0x43uy; 0xa1uy; 0x4buy;
      0x0auy; 0x51uy; 0xe2uy; 0x8fuy; 0x85uy; 0x0euy; 0x9buy; 0xa1uy; 0x02uy; 0x5cuy; 0xcfuy; 0x11uy;
      0x37uy; 0x7buy; 0x08uy; 0x54uy; 0xbfuy; 0xe1uy; 0xf4uy; 0xbcuy; 0x57uy; 0xc6uy; 0x71uy; 0x01uy;
      0xe0uy; 0x49uy; 0x4duy; 0x67uy; 0x6cuy; 0x31uy; 0xb5uy; 0x36uy; 0x89uy; 0xdbuy; 0x89uy; 0xbbuy;
      0x20uy; 0xe6uy; 0x9euy; 0xb8uy; 0x72uy; 0x54uy; 0xc9uy; 0x96uy; 0x42uy; 0x9fuy; 0xe5uy; 0x18uy;
      0xfduy; 0xe9uy; 0xd5uy; 0x5buy; 0x30uy; 0x54uy; 0x56uy; 0xfauy; 0x4duy; 0x9euy; 0x63uy; 0x68uy;
      0x32uy; 0x1duy; 0xe8uy; 0xe8uy; 0x3duy; 0x11uy; 0xb9uy; 0x54uy; 0x7fuy; 0xa3uy; 0x24uy; 0xa8uy;
      0xe0uy; 0x6auy; 0x89uy; 0xbauy; 0x21uy; 0xc6uy; 0xd3uy; 0xdbuy; 0x15uy; 0x0euy; 0x3buy; 0xccuy;
      0x1auy; 0x96uy; 0xebuy; 0xa2uy; 0xf4uy; 0x08uy; 0x75uy; 0x5buy; 0xfauy; 0xf4uy; 0xbbuy; 0x95uy;
      0xc2uy; 0x84uy; 0x81uy; 0xaauy; 0x83uy; 0xfeuy; 0xc1uy; 0x02uy; 0xa2uy; 0xb0uy; 0xfduy; 0x90uy;
      0xaauy; 0x2euy; 0xe4uy; 0x5cuy; 0x69uy; 0x25uy; 0xb8uy; 0x93uy; 0xf3uy; 0xcfuy; 0xdfuy; 0x9buy;
      0xf9uy; 0xeduy; 0xf7uy; 0xeduy; 0x97uy; 0x0euy; 0x54uy; 0x9fuy; 0xceuy; 0x05uy; 0x77uy; 0x0fuy;
      0x85uy; 0xfeuy; 0x3cuy; 0xa4uy; 0x6euy; 0x6duy; 0x9euy; 0xcbuy; 0x4auy; 0xcduy; 0xf6uy; 0x36uy;
      0x27uy; 0x7buy; 0x33uy; 0x1buy; 0x9cuy; 0x71uy; 0xd3uy; 0x8euy; 0xe8uy; 0x37uy; 0x27uy; 0x2cuy;
      0x74uy; 0x0buy; 0x22uy; 0x86uy; 0xcbuy; 0x03uy; 0x90uy; 0x00uy; 0x54uy; 0x86uy; 0x68uy; 0x01uy;
      0x77uy; 0x35uy; 0xaauy; 0xceuy; 0x1auy; 0x8euy; 0x4auy; 0x75uy; 0xf1uy; 0x24uy; 0x86uy; 0xaauy;
      0xc1uy; 0x80uy; 0xabuy; 0xacuy; 0xc5uy; 0x2buy; 0xb4uy; 0x3duy; 0x8duy; 0x02uy; 0x08uy; 0xa2uy;
      0xebuy; 0x36uy; 0xb4uy; 0xd8uy; 0x1buy; 0xa6uy; 0xd2uy; 0x8cuy; 0x3duy; 0x9fuy; 0x6cuy; 0xa4uy;
      0x69uy; 0x78uy; 0x95uy; 0x10uy; 0x96uy; 0xc3uy; 0xb3uy; 0xfcuy; 0x97uy; 0xdcuy; 0x0fuy; 0x9cuy;
      0xc4uy; 0x09uy; 0xe9uy; 0xcauy; 0xd2uy; 0x02uy; 0x60uy; 0x1auy; 0x5euy; 0xc8uy; 0x5auy; 0x5cuy;
      0x68uy; 0xd2uy; 0xa8uy; 0xeeuy; 0xffuy; 0x42uy; 0x99uy; 0xc1uy; 0xcbuy; 0xd2uy; 0x37uy; 0x9buy;
      0x2duy; 0xacuy; 0xd6uy; 0x0auy; 0x75uy; 0xb5uy; 0x50uy; 0x1buy; 0x2cuy; 0xbcuy; 0xf1uy; 0x09uy;
      0xf3uy; 0x10uy; 0x8duy; 0xeeuy; 0x67uy; 0x63uy; 0x64uy; 0x23uy; 0x44uy; 0x12uy; 0x20uy; 0x92uy;
      0xcduy; 0x29uy; 0xf5uy; 0x60uy; 0xa3uy; 0x27uy; 0x61uy; 0xb6uy; 0xb8uy; 0x30uy; 0xeauy; 0x9cuy;
      0xa0uy; 0xf0uy; 0x05uy; 0x5auy; 0x98uy; 0x5cuy; 0x18uy; 0x34uy; 0x1buy; 0xc7uy; 0x13uy; 0x8cuy;
      0x92uy; 0x70uy; 0x60uy; 0x28uy; 0x61uy; 0xd5uy; 0x76uy; 0x43uy; 0x0euy; 0xf7uy; 0xbcuy; 0x0fuy;
      0x3buy; 0xa6uy; 0x11uy; 0x8fuy; 0x69uy; 0x21uy; 0xe5uy; 0x55uy; 0x3cuy; 0x1cuy; 0x0fuy; 0xebuy;
      0xa2uy; 0x65uy; 0x83uy; 0x14uy; 0xe3uy; 0x3cuy; 0x23uy; 0xe4uy; 0x28uy; 0x9euy; 0x36uy; 0x66uy;
      0x93uy; 0x32uy; 0x38uy; 0x15uy; 0x0auy; 0xbbuy; 0x82uy; 0xf5uy; 0x7buy; 0x02uy; 0xc4uy; 0x80uy;
      0xfcuy; 0x7buy; 0xd7uy; 0xcauy; 0xdeuy; 0x44uy; 0xa3uy; 0x6euy; 0xcauy; 0x97uy; 0x12uy; 0x4auy;
      0x2duy; 0x9buy; 0xc4uy; 0x14uy; 0x61uy; 0xcbuy; 0x73uy; 0x56uy; 0x90uy; 0xdduy; 0x17uy; 0xf5uy;
      0xcfuy; 0xe9uy; 0xa1uy; 0x62uy; 0xabuy; 0x65uy; 0x4buy; 0xd8uy; 0x52uy; 0x1duy; 0x90uy; 0x07uy;
      0xfbuy; 0x5fuy; 0x03uy; 0x15uy; 0x9duy; 0xbbuy; 0x66uy; 0x44uy; 0xd6uy; 0xb4uy; 0x23uy; 0xd2uy;
      0x52uy; 0x14uy; 0x35uy; 0x4buy; 0x12uy; 0xf0uy; 0x73uy; 0xb6uy; 0x69uy; 0x38uy; 0x88uy; 0x66uy;
      0x68uy; 0x7euy; 0xdcuy; 0xf3uy; 0x9euy; 0x0buy; 0xe2uy; 0xdeuy; 0x1fuy; 0x31uy; 0x19uy; 0xe4uy;
      0x7auy; 0x54uy; 0xafuy; 0xaduy; 0x98uy; 0xd9uy; 0x34uy; 0xb1uy; 0xd2uy; 0x4buy; 0x85uy; 0x74uy;
      0x03uy; 0xebuy; 0x4euy; 0x47uy; 0x0euy; 0x92uy; 0xb0uy; 0xc1uy; 0xaeuy; 0x15uy; 0x78uy; 0xa2uy;
      0x09uy; 0x95uy; 0x5fuy; 0x29uy
  ]

let test1_sk_expected =
  List.Tot.map u8_from_UInt8
    [
      0x4buy; 0x62uy; 0x2duy; 0xe1uy; 0x35uy; 0x01uy; 0x19uy; 0xc4uy; 0x5auy; 0x9fuy; 0x2euy; 0x2euy;
      0xf3uy; 0xdcuy; 0x5duy; 0xf5uy; 0x92uy; 0x44uy; 0x4duy; 0x91uy; 0xa2uy; 0xaduy; 0xaduy; 0x05uy;
      0x2cuy; 0xa2uy; 0x3duy; 0xe5uy; 0xfbuy; 0x9duy; 0xf9uy; 0xe1uy; 0xccuy; 0xc4uy; 0x1buy; 0x9euy;
      0x7buy; 0x1buy; 0x52uy; 0x1auy; 0xbeuy; 0x8euy; 0xc1uy; 0xd0uy; 0x62uy; 0x60uy; 0x53uy; 0x71uy;
      0xb6uy; 0xc4uy; 0x42uy; 0x5euy; 0xa7uy; 0x3fuy; 0x37uy; 0x02uy; 0xdeuy; 0xf5uy; 0x26uy; 0x8buy;
      0xccuy; 0x1auy; 0xc1uy; 0x64uy; 0x7cuy; 0x9euy; 0x14uy; 0x0buy; 0x74uy; 0x6auy; 0xc9uy; 0x2fuy;
      0xd2uy; 0xb3uy; 0x9duy; 0xfeuy; 0xfauy; 0xa4uy; 0x08uy; 0x0duy; 0x57uy; 0xa3uy; 0xf8uy; 0xc3uy;
      0xefuy; 0x9buy; 0x6cuy; 0x29uy; 0x0fuy; 0x3fuy; 0xa0uy; 0x60uy; 0xe6uy; 0x4auy; 0x6buy; 0xa9uy;
      0x63uy; 0xc7uy; 0x12uy; 0xc4uy; 0xeeuy; 0x01uy; 0x89uy; 0xc8uy; 0x9duy; 0x41uy; 0xf7uy; 0x2duy;
      0xeduy; 0xf6uy; 0xdcuy; 0x0buy; 0x97uy; 0x94uy; 0x7fuy; 0xf2uy; 0x20uy; 0x86uy; 0xbduy; 0x39uy;
      0x7cuy; 0x4auy; 0x59uy; 0x1duy; 0xf0uy; 0xcduy; 0x3auy; 0xa1uy; 0x9buy; 0xb7uy; 0x27uy; 0x50uy;
      0x82uy; 0xffuy; 0x2duy; 0xcduy; 0x6cuy; 0x6buy; 0x6fuy; 0x28uy; 0x38uy; 0x34uy; 0x18uy; 0x09uy;
      0x69uy; 0x1duy; 0x2fuy; 0x52uy; 0xd0uy; 0x1duy; 0xdfuy; 0x70uy; 0xd6uy; 0x36uy; 0x48uy; 0xe4uy;
      0x17uy; 0x26uy; 0x82uy; 0x01uy; 0x70uy; 0x96uy; 0x60uy; 0x25uy; 0x21uy; 0xf2uy; 0xc3uy; 0x59uy;
      0x7duy; 0xbcuy; 0x6cuy; 0x31uy; 0xc7uy; 0xb5uy; 0x17uy; 0xdduy; 0x9cuy; 0xf2uy; 0xdcuy; 0x90uy;
      0xc2uy; 0x02uy; 0xe0uy; 0x37uy; 0xc8uy; 0xacuy; 0x55uy; 0xb5uy; 0xd6uy; 0xe9uy; 0x1fuy; 0x82uy;
      0xc0uy; 0x70uy; 0x14uy; 0x0auy; 0x6fuy; 0x03uy; 0x47uy; 0xa6uy; 0x5duy; 0x37uy; 0x55uy; 0xa2uy;
      0xc0uy; 0xd3uy; 0xe7uy; 0xf2uy; 0x7duy; 0x8euy; 0x77uy; 0x89uy; 0x94uy; 0x74uy; 0x3cuy; 0xe5uy;
      0xbeuy; 0x34uy; 0x41uy; 0x30uy; 0x59uy; 0x69uy; 0x48uy; 0xafuy; 0x28uy; 0xb7uy; 0xacuy; 0x96uy;
      0x32uy; 0x0fuy; 0xc0uy; 0x4auy; 0xcduy; 0x6duy; 0x1fuy; 0x6euy; 0x7euy; 0x71uy; 0xf8uy; 0xd8uy;
      0xd0uy; 0x27uy; 0x3cuy; 0x50uy; 0xeauy; 0x8euy; 0x6auy; 0x3buy; 0x4duy; 0x20uy; 0x49uy; 0xd5uy;
      0x2euy; 0x6buy; 0xa8uy; 0xb7uy; 0x7auy; 0x9buy; 0x9euy; 0xf8uy; 0xd7uy; 0x52uy; 0x67uy; 0x8euy;
      0xbfuy; 0x95uy; 0x76uy; 0x5euy; 0x28uy; 0x6euy; 0xa7uy; 0x14uy; 0x45uy; 0x38uy; 0xd6uy; 0x5duy;
      0x28uy; 0x69uy; 0x49uy; 0xb5uy; 0x38uy; 0xf2uy; 0x52uy; 0x17uy; 0xd4uy; 0xb2uy; 0xd5uy; 0x33uy;
      0x60uy; 0xf7uy; 0x1duy; 0x3euy; 0x64uy; 0xdbuy; 0x34uy; 0xaeuy; 0xb6uy; 0xb6uy; 0x23uy; 0x62uy;
      0x3euy; 0xa7uy; 0x01uy; 0x08uy; 0xfduy; 0x37uy; 0xe5uy; 0x91uy; 0x1cuy; 0xa7uy; 0x1buy; 0x30uy;
      0x21uy; 0x4duy; 0x9buy; 0xecuy; 0x7cuy; 0x51uy; 0x0buy; 0x3fuy; 0x16uy; 0x7buy; 0xe5uy; 0x45uy;
      0x76uy; 0x62uy; 0xa7uy; 0x61uy; 0x41uy; 0xbduy; 0xc5uy; 0x05uy; 0xeeuy; 0x3buy; 0x60uy; 0xe7uy;
      0x6auy; 0xfbuy; 0xd0uy; 0x00uy; 0x98uy; 0xcauy; 0xb4uy; 0xb9uy; 0x2euy; 0x4fuy; 0xe0uy; 0xe4uy;
      0x80uy; 0xfeuy; 0x71uy; 0x27uy; 0x48uy; 0x01uy; 0xd4uy; 0x05uy; 0x97uy; 0x9buy; 0xfcuy; 0xe9uy;
      0x5buy; 0xe0uy; 0xc4uy; 0x24uy; 0xe8uy; 0x9buy; 0x76uy; 0xecuy; 0x1cuy; 0xb9uy; 0xafuy; 0xe3uy;
      0x41uy; 0x7euy; 0xc5uy; 0x3fuy; 0x40uy; 0x02uy; 0x85uy; 0xf4uy; 0xccuy; 0x04uy; 0xd8uy; 0x1duy;
      0x1fuy; 0x4cuy; 0xdbuy; 0xc9uy; 0x9duy; 0x07uy; 0x58uy; 0x3cuy; 0x76uy; 0x6fuy; 0xa4uy; 0xeduy;
      0xf1uy; 0x78uy; 0xa2uy; 0x6auy; 0x82uy; 0x84uy; 0xdeuy; 0x6buy; 0x0auy; 0x93uy; 0x8cuy; 0xdauy;
      0xbauy; 0x64uy; 0x20uy; 0x00uy; 0xb2uy; 0xe8uy; 0x24uy; 0xc1uy; 0xa4uy; 0x93uy; 0x6cuy; 0x04uy;
      0xe8uy; 0x23uy; 0xb5uy; 0x4fuy; 0xc2uy; 0x84uy; 0x10uy; 0xd5uy; 0x0duy; 0x8buy; 0x49uy; 0xb4uy;
      0x2fuy; 0xabuy; 0x5fuy; 0xbduy; 0x40uy; 0xa5uy; 0x71uy; 0xf7uy; 0x1buy; 0x9euy; 0x8fuy; 0x46uy;
      0x67uy; 0x5duy; 0x63uy; 0x51uy; 0x27uy; 0x49uy; 0x65uy; 0x95uy; 0x58uy; 0x22uy; 0x32uy; 0xafuy;
      0x32uy; 0x32uy; 0x54uy; 0xf0uy; 0xeauy; 0x3cuy; 0x12uy; 0x71uy; 0x74uy; 0x9buy; 0xe9uy; 0xb8uy;
      0xbbuy; 0x81uy; 0xb5uy; 0x7duy; 0xd3uy; 0x77uy; 0x0duy; 0x67uy; 0xb6uy; 0x4duy; 0x92uy; 0x16uy;
      0xf3uy; 0x20uy; 0xb8uy; 0x02uy; 0x39uy; 0xc2uy; 0x21uy; 0x2auy; 0xceuy; 0xecuy; 0xb3uy; 0xa8uy;
      0x14uy; 0x1duy; 0x79uy; 0x5fuy; 0xe4uy; 0x37uy; 0x69uy; 0xf5uy; 0x31uy; 0x39uy; 0x11uy; 0x3buy;
      0x3cuy; 0xf0uy; 0xabuy; 0xf0uy; 0xc4uy; 0x97uy; 0xbauy; 0x98uy; 0x5cuy; 0x48uy; 0x96uy; 0x51uy;
      0xeauy; 0x0fuy; 0x82uy; 0xc3uy; 0x09uy; 0x85uy; 0x49uy; 0x89uy; 0x4duy; 0x33uy; 0xabuy; 0x12uy;
      0xabuy; 0x7buy; 0x7buy; 0x15uy; 0x1cuy; 0xd5uy; 0x1auy; 0x84uy; 0x99uy; 0x16uy; 0x68uy; 0x1buy;
      0xffuy; 0x20uy; 0x99uy; 0x4euy; 0x35uy; 0x9auy; 0xd0uy; 0xbauy; 0x34uy; 0x47uy; 0xf2uy; 0x1duy;
      0x73uy; 0x43uy; 0xa1uy; 0x4buy; 0x0auy; 0x51uy; 0xe2uy; 0x8fuy; 0x85uy; 0x0euy; 0x9buy; 0xa1uy;
      0x02uy; 0x5cuy; 0xcfuy; 0x11uy; 0x37uy; 0x7buy; 0x08uy; 0x54uy; 0xbfuy; 0xe1uy; 0xf4uy; 0xbcuy;
      0x57uy; 0xc6uy; 0x71uy; 0x01uy; 0xe0uy; 0x49uy; 0x4duy; 0x67uy; 0x6cuy; 0x31uy; 0xb5uy; 0x36uy;
      0x89uy; 0xdbuy; 0x89uy; 0xbbuy; 0x20uy; 0xe6uy; 0x9euy; 0xb8uy; 0x72uy; 0x54uy; 0xc9uy; 0x96uy;
      0x42uy; 0x9fuy; 0xe5uy; 0x18uy; 0xfduy; 0xe9uy; 0xd5uy; 0x5buy; 0x30uy; 0x54uy; 0x56uy; 0xfauy;
      0x4duy; 0x9euy; 0x63uy; 0x68uy; 0x32uy; 0x1duy; 0xe8uy; 0xe8uy; 0x3duy; 0x11uy; 0xb9uy; 0x54uy;
      0x7fuy; 0xa3uy; 0x24uy; 0xa8uy; 0xe0uy; 0x6auy; 0x89uy; 0xbauy; 0x21uy; 0xc6uy; 0xd3uy; 0xdbuy;
      0x15uy; 0x0euy; 0x3buy; 0xccuy; 0x1auy; 0x96uy; 0xebuy; 0xa2uy; 0xf4uy; 0x08uy; 0x75uy; 0x5buy;
      0xfauy; 0xf4uy; 0xbbuy; 0x95uy; 0xc2uy; 0x84uy; 0x81uy; 0xaauy; 0x83uy; 0xfeuy; 0xc1uy; 0x02uy;
      0xa2uy; 0xb0uy; 0xfduy; 0x90uy; 0xaauy; 0x2euy; 0xe4uy; 0x5cuy; 0x69uy; 0x25uy; 0xb8uy; 0x93uy;
      0xf3uy; 0xcfuy; 0xdfuy; 0x9buy; 0xf9uy; 0xeduy; 0xf7uy; 0xeduy; 0x97uy; 0x0euy; 0x54uy; 0x9fuy;
      0xceuy; 0x05uy; 0x77uy; 0x0fuy; 0x85uy; 0xfeuy; 0x3cuy; 0xa4uy; 0x6euy; 0x6duy; 0x9euy; 0xcbuy;
      0x4auy; 0xcduy; 0xf6uy; 0x36uy; 0x27uy; 0x7buy; 0x33uy; 0x1buy; 0x9cuy; 0x71uy; 0xd3uy; 0x8euy;
      0xe8uy; 0x37uy; 0x27uy; 0x2cuy; 0x74uy; 0x0buy; 0x22uy; 0x86uy; 0xcbuy; 0x03uy; 0x90uy; 0x00uy;
      0x54uy; 0x86uy; 0x68uy; 0x01uy; 0x77uy; 0x35uy; 0xaauy; 0xceuy; 0x1auy; 0x8euy; 0x4auy; 0x75uy;
      0xf1uy; 0x24uy; 0x86uy; 0xaauy; 0xc1uy; 0x80uy; 0xabuy; 0xacuy; 0xc5uy; 0x2buy; 0xb4uy; 0x3duy;
      0x8duy; 0x02uy; 0x08uy; 0xa2uy; 0xebuy; 0x36uy; 0xb4uy; 0xd8uy; 0x1buy; 0xa6uy; 0xd2uy; 0x8cuy;
      0x3duy; 0x9fuy; 0x6cuy; 0xa4uy; 0x69uy; 0x78uy; 0x95uy; 0x10uy; 0x96uy; 0xc3uy; 0xb3uy; 0xfcuy;
      0x97uy; 0xdcuy; 0x0fuy; 0x9cuy; 0xc4uy; 0x09uy; 0xe9uy; 0xcauy; 0xd2uy; 0x02uy; 0x60uy; 0x1auy;
      0x5euy; 0xc8uy; 0x5auy; 0x5cuy; 0x68uy; 0xd2uy; 0xa8uy; 0xeeuy; 0xffuy; 0x42uy; 0x99uy; 0xc1uy;
      0xcbuy; 0xd2uy; 0x37uy; 0x9buy; 0x2duy; 0xacuy; 0xd6uy; 0x0auy; 0x75uy; 0xb5uy; 0x50uy; 0x1buy;
      0x2cuy; 0xbcuy; 0xf1uy; 0x09uy; 0xf3uy; 0x10uy; 0x8duy; 0xeeuy; 0x67uy; 0x63uy; 0x64uy; 0x23uy;
      0x44uy; 0x12uy; 0x20uy; 0x92uy; 0xcduy; 0x29uy; 0xf5uy; 0x60uy; 0xa3uy; 0x27uy; 0x61uy; 0xb6uy;
      0xb8uy; 0x30uy; 0xeauy; 0x9cuy; 0xa0uy; 0xf0uy; 0x05uy; 0x5auy; 0x98uy; 0x5cuy; 0x18uy; 0x34uy;
      0x1buy; 0xc7uy; 0x13uy; 0x8cuy; 0x92uy; 0x70uy; 0x60uy; 0x28uy; 0x61uy; 0xd5uy; 0x76uy; 0x43uy;
      0x0euy; 0xf7uy; 0xbcuy; 0x0fuy; 0x3buy; 0xa6uy; 0x11uy; 0x8fuy; 0x69uy; 0x21uy; 0xe5uy; 0x55uy;
      0x3cuy; 0x1cuy; 0x0fuy; 0xebuy; 0xa2uy; 0x65uy; 0x83uy; 0x14uy; 0xe3uy; 0x3cuy; 0x23uy; 0xe4uy;
      0x28uy; 0x9euy; 0x36uy; 0x66uy; 0x93uy; 0x32uy; 0x38uy; 0x15uy; 0x0auy; 0xbbuy; 0x82uy; 0xf5uy;
      0x7buy; 0x02uy; 0xc4uy; 0x80uy; 0xfcuy; 0x7buy; 0xd7uy; 0xcauy; 0xdeuy; 0x44uy; 0xa3uy; 0x6euy;
      0xcauy; 0x97uy; 0x12uy; 0x4auy; 0x2duy; 0x9buy; 0xc4uy; 0x14uy; 0x61uy; 0xcbuy; 0x73uy; 0x56uy;
      0x90uy; 0xdduy; 0x17uy; 0xf5uy; 0xcfuy; 0xe9uy; 0xa1uy; 0x62uy; 0xabuy; 0x65uy; 0x4buy; 0xd8uy;
      0x52uy; 0x1duy; 0x90uy; 0x07uy; 0xfbuy; 0x5fuy; 0x03uy; 0x15uy; 0x9duy; 0xbbuy; 0x66uy; 0x44uy;
      0xd6uy; 0xb4uy; 0x23uy; 0xd2uy; 0x52uy; 0x14uy; 0x35uy; 0x4buy; 0x12uy; 0xf0uy; 0x73uy; 0xb6uy;
      0x69uy; 0x38uy; 0x88uy; 0x66uy; 0x68uy; 0x7euy; 0xdcuy; 0xf3uy; 0x9euy; 0x0buy; 0xe2uy; 0xdeuy;
      0x1fuy; 0x31uy; 0x19uy; 0xe4uy; 0x7auy; 0x54uy; 0xafuy; 0xaduy; 0x98uy; 0xd9uy; 0x34uy; 0xb1uy;
      0xd2uy; 0x4buy; 0x85uy; 0x74uy; 0x03uy; 0xebuy; 0x4euy; 0x47uy; 0x0euy; 0x92uy; 0xb0uy; 0xc1uy;
      0xaeuy; 0x15uy; 0x78uy; 0xa2uy; 0x09uy; 0x95uy; 0x5fuy; 0x29uy; 0xfbuy; 0xffuy; 0x03uy; 0x00uy;
      0x00uy; 0x00uy; 0x03uy; 0x00uy; 0xfeuy; 0xffuy; 0xfduy; 0xffuy; 0x01uy; 0x00uy; 0xffuy; 0xffuy;
      0x02uy; 0x00uy; 0x02uy; 0x00uy; 0xfeuy; 0xffuy; 0x01uy; 0x00uy; 0x03uy; 0x00uy; 0xffuy; 0xffuy;
      0xfbuy; 0xffuy; 0xfeuy; 0xffuy; 0x06uy; 0x00uy; 0x06uy; 0x00uy; 0x01uy; 0x00uy; 0xfcuy; 0xffuy;
      0xfduy; 0xffuy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0x01uy; 0x00uy; 0xffuy; 0xffuy; 0x00uy; 0x00uy;
      0xfcuy; 0xffuy; 0x02uy; 0x00uy; 0x04uy; 0x00uy; 0x01uy; 0x00uy; 0x02uy; 0x00uy; 0xffuy; 0xffuy;
      0x02uy; 0x00uy; 0x03uy; 0x00uy; 0x01uy; 0x00uy; 0xfeuy; 0xffuy; 0x04uy; 0x00uy; 0xfeuy; 0xffuy;
      0x06uy; 0x00uy; 0xffuy; 0xffuy; 0x03uy; 0x00uy; 0x00uy; 0x00uy; 0xfduy; 0xffuy; 0x00uy; 0x00uy;
      0x00uy; 0x00uy; 0x00uy; 0x00uy; 0xffuy; 0xffuy; 0x01uy; 0x00uy; 0xfduy; 0xffuy; 0xfeuy; 0xffuy;
      0x00uy; 0x00uy; 0xfeuy; 0xffuy; 0x04uy; 0x00uy; 0xfduy; 0xffuy; 0xfduy; 0xffuy; 0x00uy; 0x00uy;
      0xffuy; 0xffuy; 0xfbuy; 0xffuy; 0x00uy; 0x00uy; 0x01uy; 0x00uy; 0xfeuy; 0xffuy; 0xfbuy; 0xffuy;
      0x02uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0xfeuy; 0xffuy; 0xfeuy; 0xffuy; 0xffuy; 0xffuy;
      0x06uy; 0x00uy; 0xffuy; 0xffuy; 0x00uy; 0x00uy; 0xfeuy; 0xffuy; 0x01uy; 0x00uy; 0x02uy; 0x00uy;
      0x01uy; 0x00uy; 0x01uy; 0x00uy; 0x01uy; 0x00uy; 0xffuy; 0xffuy; 0xffuy; 0xffuy; 0x01uy; 0x00uy;
      0xfbuy; 0xffuy; 0x05uy; 0x00uy; 0x02uy; 0x00uy; 0xfduy; 0xffuy; 0x02uy; 0x00uy; 0x05uy; 0x00uy;
      0xfduy; 0xffuy; 0x04uy; 0x00uy; 0x02uy; 0x00uy; 0x04uy; 0x00uy; 0xfbuy; 0xffuy; 0xfeuy; 0xffuy;
      0x01uy; 0x00uy; 0x02uy; 0x00uy; 0xfcuy; 0xffuy; 0x03uy; 0x00uy; 0x00uy; 0x00uy; 0x01uy; 0x00uy;
      0x02uy; 0x00uy; 0xffuy; 0xffuy; 0xffuy; 0xffuy; 0xffuy; 0xffuy; 0xfcuy; 0xffuy; 0x01uy; 0x00uy;
      0xffuy; 0xffuy; 0x01uy; 0x00uy; 0xfduy; 0xffuy; 0x03uy; 0x00uy; 0x00uy; 0x00uy; 0x04uy; 0x00uy;
      0xffuy; 0xffuy; 0xfcuy; 0xffuy; 0x01uy; 0x00uy; 0x03uy; 0x00uy; 0xffuy; 0xffuy; 0x00uy; 0x00uy;
      0x01uy; 0x00uy; 0x02uy; 0x00uy; 0xfduy; 0xffuy; 0xfeuy; 0xffuy; 0x01uy; 0x00uy; 0xffuy; 0xffuy;
      0x03uy; 0x00uy; 0xffuy; 0xffuy; 0xffuy; 0xffuy; 0xfeuy; 0xffuy; 0xfduy; 0xffuy; 0x01uy; 0x00uy;
      0xfeuy; 0xffuy; 0x03uy; 0x00uy; 0x02uy; 0x00uy; 0xffuy; 0xffuy; 0x01uy; 0x00uy; 0xfeuy; 0xffuy;
      0x00uy; 0x00uy; 0xffuy; 0xffuy; 0xfduy; 0xffuy; 0xffuy; 0xffuy; 0x01uy; 0x00uy; 0xffuy; 0xffuy;
      0x02uy; 0x00uy; 0x02uy; 0x00uy; 0xfeuy; 0xffuy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0x01uy; 0x00uy;
      0xfauy; 0xffuy; 0x02uy; 0x00uy; 0x04uy; 0x00uy; 0xfduy; 0xffuy; 0xffuy; 0xffuy; 0x02uy; 0x00uy;
      0x03uy; 0x00uy; 0x08uy; 0x00uy; 0xfbuy; 0xffuy; 0x03uy; 0x00uy; 0x00uy; 0x00uy; 0xfduy; 0xffuy;
      0x08uy; 0x00uy; 0x04uy; 0x00uy; 0xfduy; 0xffuy; 0x02uy; 0x00uy; 0x02uy; 0x00uy; 0x01uy; 0x00uy;
      0x04uy; 0x00uy; 0x00uy; 0x00uy; 0xfduy; 0xffuy; 0xffuy; 0xffuy; 0xffuy; 0xffuy; 0x00uy; 0x00uy;
      0x01uy; 0x00uy; 0x02uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0x04uy; 0x00uy;
      0xffuy; 0xffuy; 0x01uy; 0x00uy; 0xffuy; 0xffuy; 0x01uy; 0x00uy; 0xfduy; 0xffuy; 0x01uy; 0x00uy;
      0xffuy; 0xffuy; 0x01uy; 0x00uy; 0x01uy; 0x00uy; 0x06uy; 0x00uy; 0xfeuy; 0xffuy; 0x01uy; 0x00uy;
      0xfduy; 0xffuy; 0xfeuy; 0xffuy; 0x00uy; 0x00uy; 0x02uy; 0x00uy; 0xfcuy; 0xffuy; 0xffuy; 0xffuy;
      0x00uy; 0x00uy; 0xffuy; 0xffuy; 0x02uy; 0x00uy; 0x00uy; 0x00uy; 0xfduy; 0xffuy; 0xfeuy; 0xffuy;
      0x05uy; 0x00uy; 0x05uy; 0x00uy; 0xfeuy; 0xffuy; 0x03uy; 0x00uy; 0x02uy; 0x00uy; 0x04uy; 0x00uy;
      0x00uy; 0x00uy; 0x01uy; 0x00uy; 0x05uy; 0x00uy; 0x02uy; 0x00uy; 0xfcuy; 0xffuy; 0xfeuy; 0xffuy;
      0x01uy; 0x00uy; 0xffuy; 0xffuy; 0xffuy; 0xffuy; 0x02uy; 0x00uy; 0xffuy; 0xffuy; 0xfcuy; 0xffuy;
      0xfeuy; 0xffuy; 0x02uy; 0x00uy; 0xfcuy; 0xffuy; 0xfeuy; 0xffuy; 0xfduy; 0xffuy; 0xfcuy; 0xffuy;
      0x02uy; 0x00uy; 0xffuy; 0xffuy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0xfcuy; 0xffuy; 0x04uy; 0x00uy;
      0x01uy; 0x00uy; 0x04uy; 0x00uy; 0xfduy; 0xffuy; 0xffuy; 0xffuy; 0xfbuy; 0xffuy; 0xfduy; 0xffuy;
      0xffuy; 0xffuy; 0x04uy; 0x00uy; 0x03uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0xfcuy; 0xffuy;
      0x01uy; 0x00uy; 0xfduy; 0xffuy; 0xffuy; 0xffuy; 0xfduy; 0xffuy; 0x03uy; 0x00uy; 0xffuy; 0xffuy;
      0xfeuy; 0xffuy; 0x05uy; 0x00uy; 0x01uy; 0x00uy; 0x03uy; 0x00uy; 0xfauy; 0xffuy; 0x02uy; 0x00uy;
      0x00uy; 0x00uy; 0xfeuy; 0xffuy; 0xfeuy; 0xffuy; 0x02uy; 0x00uy; 0xfeuy; 0xffuy; 0xfduy; 0xffuy;
      0xffuy; 0xffuy; 0x01uy; 0x00uy; 0x04uy; 0x00uy; 0x01uy; 0x00uy; 0x01uy; 0x00uy; 0x06uy; 0x00uy;
      0x04uy; 0x00uy; 0xffuy; 0xffuy; 0x02uy; 0x00uy; 0xfcuy; 0xffuy; 0x01uy; 0x00uy; 0x02uy; 0x00uy;
      0xffuy; 0xffuy; 0x00uy; 0x00uy; 0x01uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0xfeuy; 0xffuy;
      0xfeuy; 0xffuy; 0x02uy; 0x00uy; 0xffuy; 0xffuy; 0xffuy; 0xffuy; 0x05uy; 0x00uy; 0x01uy; 0x00uy;
      0xfeuy; 0xffuy; 0x01uy; 0x00uy; 0x02uy; 0x00uy; 0x01uy; 0x00uy; 0xfeuy; 0xffuy; 0xfduy; 0xffuy;
      0x01uy; 0x00uy; 0xfeuy; 0xffuy; 0x03uy; 0x00uy; 0x03uy; 0x00uy; 0x02uy; 0x00uy; 0x04uy; 0x00uy;
      0x06uy; 0x00uy; 0x01uy; 0x00uy; 0x00uy; 0x00uy; 0xfeuy; 0xffuy; 0xfeuy; 0xffuy; 0x02uy; 0x00uy;
      0xfeuy; 0xffuy; 0x02uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0x01uy; 0x00uy; 0x01uy; 0x00uy;
      0x00uy; 0x00uy; 0x03uy; 0x00uy; 0xfcuy; 0xffuy; 0xfeuy; 0xffuy; 0xffuy; 0xffuy; 0xfbuy; 0xffuy;
      0xfduy; 0xffuy; 0x04uy; 0x00uy; 0x01uy; 0x00uy; 0x02uy; 0x00uy; 0x01uy; 0x00uy; 0xffuy; 0xffuy;
      0xfeuy; 0xffuy; 0xffuy; 0xffuy; 0xfcuy; 0xffuy; 0xfcuy; 0xffuy; 0x06uy; 0x00uy; 0x00uy; 0x00uy;
      0xfduy; 0xffuy; 0xfduy; 0xffuy; 0xffuy; 0xffuy; 0xfeuy; 0xffuy; 0xffuy; 0xffuy; 0xffuy; 0xffuy;
      0x03uy; 0x00uy; 0xfbuy; 0xffuy; 0x02uy; 0x00uy; 0x02uy; 0x00uy; 0xfduy; 0xffuy; 0xfduy; 0xffuy;
      0x02uy; 0x00uy; 0x02uy; 0x00uy; 0x02uy; 0x00uy; 0x03uy; 0x00uy; 0xfcuy; 0xffuy; 0xfbuy; 0xffuy;
      0x01uy; 0x00uy; 0x04uy; 0x00uy; 0x03uy; 0x00uy; 0x00uy; 0x00uy; 0xfduy; 0xffuy; 0xfduy; 0xffuy;
      0xfduy; 0xffuy; 0xffuy; 0xffuy; 0x00uy; 0x00uy; 0x01uy; 0x00uy; 0xf8uy; 0xffuy; 0x01uy; 0x00uy;
      0x00uy; 0x00uy; 0xfeuy; 0xffuy; 0xfcuy; 0xffuy; 0x01uy; 0x00uy; 0x02uy; 0x00uy; 0xfeuy; 0xffuy;
      0xffuy; 0xffuy; 0xffuy; 0xffuy; 0x04uy; 0x00uy; 0x02uy; 0x00uy; 0xfduy; 0xffuy; 0x01uy; 0x00uy;
      0xfduy; 0xffuy; 0x04uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0x01uy; 0x00uy; 0x00uy; 0x00uy;
      0x00uy; 0x00uy; 0x02uy; 0x00uy; 0x01uy; 0x00uy; 0x03uy; 0x00uy; 0x03uy; 0x00uy; 0xfauy; 0xffuy;
      0x03uy; 0x00uy; 0x00uy; 0x00uy; 0x02uy; 0x00uy; 0xffuy; 0xffuy; 0x01uy; 0x00uy; 0xfeuy; 0xffuy;
      0x01uy; 0x00uy; 0x04uy; 0x00uy; 0x03uy; 0x00uy; 0x01uy; 0x00uy; 0xfeuy; 0xffuy; 0xfduy; 0xffuy;
      0xffuy; 0xffuy; 0xffuy; 0xffuy; 0x01uy; 0x00uy; 0x05uy; 0x00uy; 0x02uy; 0x00uy; 0x02uy; 0x00uy;
      0xfduy; 0xffuy; 0x02uy; 0x00uy; 0xfeuy; 0xffuy; 0x03uy; 0x00uy; 0x01uy; 0x00uy; 0x01uy; 0x00uy;
      0x00uy; 0x00uy; 0x03uy; 0x00uy; 0x03uy; 0x00uy; 0xfeuy; 0xffuy; 0x00uy; 0x00uy; 0x02uy; 0x00uy;
      0xfcuy; 0xffuy; 0x02uy; 0x00uy; 0x02uy; 0x00uy; 0xfeuy; 0xffuy; 0x04uy; 0x00uy; 0x01uy; 0x00uy;
      0x06uy; 0x00uy; 0xfduy; 0xffuy; 0xfduy; 0xffuy; 0x03uy; 0x00uy; 0x00uy; 0x00uy; 0xfduy; 0xffuy;
      0xfeuy; 0xffuy; 0xffuy; 0xffuy; 0xffuy; 0xffuy; 0xfeuy; 0xffuy; 0x07uy; 0x00uy; 0x01uy; 0x00uy;
      0xffuy; 0xffuy; 0xffuy; 0xffuy; 0x01uy; 0x00uy; 0xfcuy; 0xffuy; 0xfduy; 0xffuy; 0x01uy; 0x00uy;
      0xffuy; 0xffuy; 0x03uy; 0x00uy; 0xffuy; 0xffuy; 0x00uy; 0x00uy; 0x03uy; 0x00uy; 0xffuy; 0xffuy;
      0x02uy; 0x00uy; 0xffuy; 0xffuy; 0x00uy; 0x00uy; 0xfeuy; 0xffuy; 0xffuy; 0xffuy; 0xfeuy; 0xffuy;
      0x02uy; 0x00uy; 0xfbuy; 0xffuy; 0xfcuy; 0xffuy; 0x01uy; 0x00uy; 0xffuy; 0xffuy; 0x01uy; 0x00uy;
      0x02uy; 0x00uy; 0xfduy; 0xffuy; 0xfeuy; 0xffuy; 0xfbuy; 0xffuy; 0x00uy; 0x00uy; 0xffuy; 0xffuy;
      0x03uy; 0x00uy; 0x01uy; 0x00uy; 0x03uy; 0x00uy; 0x01uy; 0x00uy; 0x01uy; 0x00uy; 0x01uy; 0x00uy;
      0xffuy; 0xffuy; 0xfauy; 0xffuy; 0x03uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0x00uy; 0xfbuy; 0xffuy;
      0xffuy; 0xffuy; 0x02uy; 0x00uy; 0xffuy; 0xffuy; 0x00uy; 0x00uy; 0xfeuy; 0xffuy; 0x05uy; 0x00uy;
      0x05uy; 0x00uy; 0x01uy; 0x00uy; 0xfeuy; 0xffuy; 0x01uy; 0x00uy; 0x01uy; 0x00uy; 0x03uy; 0x00uy;
      0xfeuy; 0xffuy; 0xffuy; 0xffuy; 0xfeuy; 0xffuy; 0x05uy; 0x00uy; 0x02uy; 0x00uy; 0xfcuy; 0xffuy;
      0xfeuy; 0xffuy; 0x05uy; 0x00uy; 0x01uy; 0x00uy; 0x00uy; 0x00uy; 0xfcuy; 0xffuy; 0x02uy; 0x00uy;
      0xfeuy; 0xffuy; 0x01uy; 0x00uy; 0xfcuy; 0xffuy; 0x02uy; 0x00uy; 0x01uy; 0x00uy; 0x03uy; 0x00uy;
      0x00uy; 0x00uy; 0xffuy; 0xffuy; 0xfbuy; 0xffuy; 0xfduy; 0xffuy; 0x00uy; 0x00uy; 0x02uy; 0x00uy;
      0x02uy; 0x00uy; 0xfbuy; 0xffuy; 0xfduy; 0xffuy; 0xfduy; 0xffuy; 0xffuy; 0xffuy; 0x01uy; 0x00uy
  ]

let test1_ct_expected =
  List.Tot.map u8_from_UInt8
    [
      0xf7uy; 0xa6uy; 0x93uy; 0xf8uy; 0x0duy; 0x9duy; 0xa5uy; 0x9euy; 0x3cuy; 0x22uy; 0x45uy; 0x2cuy;
      0x67uy; 0x29uy; 0x02uy; 0x61uy; 0x09uy; 0x41uy; 0x2cuy; 0xb0uy; 0x92uy; 0xfauy; 0x7duy; 0x55uy;
      0x04uy; 0x8auy; 0x27uy; 0xa1uy; 0x19uy; 0x89uy; 0xb2uy; 0x11uy; 0x8euy; 0x05uy; 0x5euy; 0xb2uy;
      0x4cuy; 0x50uy; 0x16uy; 0xa8uy; 0x2fuy; 0xbcuy; 0x5auy; 0x12uy; 0x2cuy; 0x72uy; 0x4cuy; 0xd8uy;
      0xa0uy; 0xd1uy; 0xc5uy; 0x35uy; 0xd7uy; 0x0duy; 0xe2uy; 0xb3uy; 0x54uy; 0x02uy; 0x7buy; 0x45uy;
      0x0auy; 0xd0uy; 0x98uy; 0x49uy; 0xf6uy; 0x29uy; 0xdduy; 0x52uy; 0x9duy; 0xe5uy; 0x9euy; 0x0fuy;
      0x2duy; 0x2cuy; 0x1cuy; 0x5cuy; 0x9duy; 0xd4uy; 0x8duy; 0x32uy; 0x0cuy; 0x72uy; 0x2fuy; 0x93uy;
      0xceuy; 0xb1uy; 0x52uy; 0xaeuy; 0x0cuy; 0x16uy; 0xfduy; 0x02uy; 0x46uy; 0xd4uy; 0x60uy; 0x2duy;
      0x3fuy; 0x40uy; 0x0fuy; 0xb2uy; 0x12uy; 0xf0uy; 0x14uy; 0x68uy; 0x0cuy; 0xb2uy; 0x44uy; 0xaduy;
      0x1buy; 0x4auy; 0x62uy; 0xf8uy; 0x4auy; 0x73uy; 0xe4uy; 0xebuy; 0x92uy; 0xa2uy; 0x57uy; 0x16uy;
      0x49uy; 0xf1uy; 0xd3uy; 0x4cuy; 0xa1uy; 0xffuy; 0xcfuy; 0x54uy; 0x6euy; 0xfduy; 0xa4uy; 0xb2uy;
      0x22uy; 0xf9uy; 0x17uy; 0xb1uy; 0xe1uy; 0x61uy; 0xacuy; 0x12uy; 0x6buy; 0xaeuy; 0x1cuy; 0x10uy;
      0x1euy; 0x43uy; 0x78uy; 0x3duy; 0x25uy; 0x1euy; 0x26uy; 0xbeuy; 0x09uy; 0x78uy; 0x04uy; 0xb3uy;
      0xfbuy; 0x31uy; 0x2auy; 0x5cuy; 0x00uy; 0x44uy; 0xabuy; 0x16uy; 0x78uy; 0x52uy; 0x13uy; 0xc3uy;
      0x92uy; 0x7fuy; 0x2auy; 0x44uy; 0x7duy; 0x84uy; 0x73uy; 0x86uy; 0xacuy; 0xfbuy; 0x7fuy; 0x79uy;
      0x44uy; 0x15uy; 0x91uy; 0xa6uy; 0xd4uy; 0x4euy; 0xe6uy; 0xccuy; 0xa0uy; 0x11uy; 0xe1uy; 0x67uy;
      0x65uy; 0x84uy; 0x2euy; 0x75uy; 0x1auy; 0xc1uy; 0x70uy; 0xf2uy; 0xdeuy; 0x45uy; 0x31uy; 0x78uy;
      0x6auy; 0x4cuy; 0xe8uy; 0x8duy; 0xf4uy; 0xcbuy; 0xefuy; 0x6euy; 0x6euy; 0xe3uy; 0xabuy; 0xe8uy;
      0x0cuy; 0x53uy; 0xceuy; 0xaeuy; 0x91uy; 0xe7uy; 0x44uy; 0x83uy; 0xd6uy; 0xb1uy; 0xf5uy; 0xbbuy;
      0x66uy; 0x59uy; 0x1fuy; 0xc4uy; 0x9euy; 0xd1uy; 0x73uy; 0x67uy; 0x7duy; 0xb1uy; 0x38uy; 0x64uy;
      0x3auy; 0xd1uy; 0x4fuy; 0x3fuy; 0x95uy; 0x32uy; 0xe0uy; 0x67uy; 0x17uy; 0xaduy; 0xb0uy; 0xf7uy;
      0x6cuy; 0x62uy; 0x3buy; 0x2auy; 0xcbuy; 0x21uy; 0xfcuy; 0x4euy; 0xefuy; 0xd4uy; 0x4cuy; 0x4fuy;
      0xa9uy; 0xc2uy; 0xe8uy; 0x74uy; 0x12uy; 0x53uy; 0x67uy; 0x9fuy; 0xbfuy; 0x4duy; 0x36uy; 0x8buy;
      0xf2uy; 0x53uy; 0x7cuy; 0x35uy; 0x53uy; 0x17uy; 0xaauy; 0x7auy; 0x11uy; 0x08uy; 0xb7uy; 0x3duy;
      0x4auy; 0x3cuy; 0xe3uy; 0x62uy; 0xcfuy; 0xb6uy; 0x1fuy; 0x14uy; 0x7duy; 0xdfuy; 0x4fuy; 0x72uy;
      0xdcuy; 0x9cuy; 0x02uy; 0x5auy; 0xfeuy; 0x07uy; 0xf5uy; 0x9cuy; 0xf1uy; 0x0buy; 0x51uy; 0x2fuy;
      0x28uy; 0xd3uy; 0xecuy; 0x78uy; 0x37uy; 0xdbuy; 0x1cuy; 0xefuy; 0x9auy; 0xfeuy; 0x19uy; 0xe9uy;
      0x7euy; 0xceuy; 0xc3uy; 0x5auy; 0xd8uy; 0x3auy; 0x7duy; 0x7euy; 0x9auy; 0x4euy; 0x2duy; 0xe1uy;
      0x01uy; 0x87uy; 0x5auy; 0x05uy; 0x3auy; 0xb6uy; 0xf4uy; 0x12uy; 0x93uy; 0x77uy; 0xd5uy; 0x82uy;
      0xeeuy; 0x61uy; 0x40uy; 0x2fuy; 0xd8uy; 0x7duy; 0x33uy; 0xeeuy; 0xcauy; 0x5cuy; 0xf0uy; 0xb8uy;
      0x7duy; 0x41uy; 0x35uy; 0xbbuy; 0x86uy; 0x62uy; 0xf3uy; 0xdduy; 0xa0uy; 0xd0uy; 0x36uy; 0xa0uy;
      0x50uy; 0x0euy; 0x50uy; 0xbauy; 0x26uy; 0x25uy; 0xf4uy; 0x15uy; 0xf6uy; 0x46uy; 0x8euy; 0x98uy;
      0x65uy; 0x84uy; 0xeauy; 0xbbuy; 0xdeuy; 0xd3uy; 0xf4uy; 0x29uy; 0x02uy; 0x5fuy; 0xdbuy; 0x1cuy;
      0xc4uy; 0xbduy; 0xdeuy; 0x47uy; 0x00uy; 0x0auy; 0x51uy; 0x34uy; 0x43uy; 0x86uy; 0x7duy; 0xc3uy;
      0xebuy; 0x80uy; 0x10uy; 0x45uy; 0xccuy; 0x4euy; 0xd1uy; 0x6euy; 0xbeuy; 0x84uy; 0xbauy; 0x45uy;
      0x4fuy; 0x1euy; 0x72uy; 0x99uy; 0xf2uy; 0x6cuy; 0xb9uy; 0x4duy; 0xeduy; 0x28uy; 0xdeuy; 0x61uy;
      0x1buy; 0x65uy; 0xe2uy; 0x7cuy; 0x64uy; 0x93uy; 0xf0uy; 0x71uy; 0x50uy; 0xe9uy; 0x1fuy; 0xdfuy;
      0x14uy; 0x72uy; 0xf6uy; 0xf5uy; 0xabuy; 0x03uy; 0x14uy; 0x91uy; 0xbauy; 0x44uy; 0x25uy; 0x39uy;
      0x46uy; 0x9duy; 0x48uy; 0x2auy; 0x4buy; 0x7cuy; 0xbeuy; 0x10uy; 0x5buy; 0x00uy; 0xbcuy; 0xf3uy;
      0x57uy; 0xe1uy; 0x2euy; 0x1buy; 0x4auy; 0x5duy; 0x59uy; 0xabuy; 0x0fuy; 0x70uy; 0x2cuy; 0x93uy;
      0x5auy; 0x06uy; 0x0duy; 0xbeuy; 0xb3uy; 0x41uy; 0x7duy; 0x21uy; 0xf2uy; 0x04uy; 0x51uy; 0x0duy;
      0x91uy; 0x08uy; 0x3euy; 0xffuy; 0x78uy; 0x6buy; 0x86uy; 0x5auy; 0xa5uy; 0xc4uy; 0x8duy; 0x4cuy;
      0x96uy; 0x40uy; 0xeauy; 0x7buy; 0x14uy; 0x26uy; 0x77uy; 0x44uy; 0x41uy; 0xaauy; 0xf1uy; 0x6cuy;
      0xf2uy; 0xc7uy; 0xfbuy; 0x00uy; 0x39uy; 0xb8uy; 0x7buy; 0x99uy; 0x64uy; 0x68uy; 0x35uy; 0xe0uy;
      0xf5uy; 0x15uy; 0xccuy; 0x12uy; 0x2buy; 0xf1uy; 0x3auy; 0x35uy; 0x69uy; 0xe2uy; 0xc2uy; 0x84uy;
      0x56uy; 0x95uy; 0x88uy; 0x16uy; 0xeduy; 0x87uy; 0x69uy; 0x48uy; 0x75uy; 0x9euy; 0x46uy; 0xdcuy;
      0x76uy; 0xc5uy; 0xc2uy; 0x48uy; 0xb6uy; 0xd7uy; 0x91uy; 0xbcuy; 0x53uy; 0x21uy; 0x65uy; 0x7fuy;
      0xa8uy; 0x62uy; 0xf4uy; 0x30uy; 0x00uy; 0x09uy; 0x0duy; 0xd6uy; 0xdeuy; 0xccuy; 0x54uy; 0x56uy;
      0xeauy; 0x3buy; 0xd9uy; 0x05uy; 0x63uy; 0xe9uy; 0xe2uy; 0x80uy; 0x00uy; 0x97uy; 0x0auy; 0x1auy;
      0x97uy; 0xd8uy; 0xa1uy; 0x54uy; 0xd3uy; 0x07uy; 0xf0uy; 0xeeuy; 0xe6uy; 0xd2uy; 0x82uy; 0x51uy;
      0xccuy; 0x0fuy; 0x8euy; 0x4fuy; 0x7euy; 0xe8uy; 0x5buy; 0xf7uy; 0xcduy; 0xb1uy; 0xa7uy; 0x05uy;
      0x2duy; 0xa2uy; 0xe1uy; 0x4duy; 0xfeuy; 0x30uy; 0xe8uy; 0x78uy; 0x26uy; 0x39uy; 0x52uy; 0x9cuy;
      0xa0uy; 0x82uy; 0x16uy; 0xfauy; 0xefuy; 0x7fuy; 0x8auy; 0x5auy; 0xcbuy; 0x2buy; 0x5euy; 0x57uy;
      0x31uy; 0x4fuy; 0x75uy; 0xa5uy; 0xe9uy; 0x9buy; 0x3buy; 0x27uy; 0x13uy; 0x6fuy; 0x47uy; 0x20uy;
      0xdduy; 0x33uy; 0x13uy; 0x93uy; 0x40uy; 0xe4uy; 0x56uy; 0x60uy; 0x1cuy; 0xfcuy; 0x02uy; 0x51uy;
      0x16uy; 0x1fuy; 0xa5uy; 0xd3uy; 0xc5uy; 0x27uy; 0xe5uy; 0x08uy; 0x9duy; 0x51uy; 0x16uy; 0xc7uy;
      0x70uy; 0x3duy; 0xd8uy; 0x81uy; 0xaauy; 0xb2uy; 0x67uy; 0x9cuy; 0x44uy; 0xc3uy; 0x0cuy; 0x01uy;
      0x33uy; 0xb1uy; 0xb3uy; 0xe8uy; 0x57uy; 0xf4uy; 0x17uy; 0xe5uy; 0xdbuy; 0x5auy; 0xaauy; 0xc1uy;
      0x25uy; 0x83uy; 0x44uy; 0x2fuy; 0x1euy; 0xdduy; 0x86uy; 0x19uy; 0x97uy; 0x50uy; 0x32uy; 0x0auy;
      0x18uy; 0x87uy; 0x2fuy; 0xe2uy; 0x60uy; 0xcauy; 0xdbuy; 0x3euy; 0x08uy; 0xb7uy; 0x9duy; 0x97uy;
      0xdbuy; 0x14uy; 0xc1uy; 0x3euy; 0x35uy; 0x74uy; 0x2euy; 0x53uy; 0x6auy; 0x60uy; 0x8euy; 0x58uy;
      0xf7uy; 0x97uy; 0x33uy; 0x20uy; 0xbfuy; 0x76uy; 0xebuy; 0x2fuy; 0x8duy; 0x2buy; 0x08uy; 0x46uy;
      0xf9uy; 0xfduy; 0xaeuy; 0x76uy; 0x0buy; 0x90uy; 0x37uy; 0x96uy; 0x87uy; 0x10uy; 0xc9uy; 0x4auy;
      0x54uy; 0x23uy; 0x66uy; 0x7euy; 0x31uy; 0xc7uy; 0x5cuy; 0xf0uy; 0xfcuy; 0x8buy; 0x2fuy; 0xdbuy;
      0xc7uy; 0x12uy; 0x02uy; 0x13uy; 0x31uy; 0x27uy; 0xa3uy; 0x95uy; 0x5duy; 0x16uy; 0xb1uy; 0x3fuy;
      0x56uy; 0xcduy; 0x65uy; 0xfcuy; 0x41uy; 0x87uy; 0x3buy; 0xaduy; 0x26uy; 0x5euy; 0xd5uy; 0x87uy;
      0x4fuy; 0xceuy; 0x3duy; 0xccuy; 0x7auy; 0x29uy; 0x68uy; 0x3fuy; 0xb2uy; 0x5auy; 0x6fuy; 0xa4uy;
      0xb8uy; 0xf5uy; 0xeeuy; 0x7fuy; 0x73uy; 0x9buy; 0x88uy; 0x15uy; 0x8cuy; 0x1euy; 0x9duy; 0xf4uy;
      0x1cuy; 0xa0uy; 0x41uy; 0x67uy; 0x50uy; 0xa5uy; 0x10uy; 0xf8uy; 0x16uy; 0x2auy; 0x0fuy; 0xdeuy;
      0x82uy; 0x4fuy; 0xeeuy; 0xd7uy; 0xd0uy; 0x8duy; 0x2cuy; 0x8fuy; 0x87uy; 0x3auy; 0x4buy; 0x68uy;
      0x2auy; 0xf2uy; 0x54uy; 0x66uy; 0x5duy; 0x2fuy; 0xeeuy; 0x1duy; 0x80uy; 0x57uy; 0xaauy; 0xc1uy;
      0x20uy; 0x43uy; 0x9buy; 0x9duy; 0x83uy; 0x6duy; 0xacuy; 0xd6uy; 0xbbuy; 0xdeuy; 0x58uy; 0x73uy;
      0x19uy; 0x9euy; 0x6euy; 0xaeuy; 0xdbuy; 0xccuy; 0x9euy; 0x43uy; 0x9fuy; 0x7auy; 0x62uy; 0x59uy;
      0xeeuy; 0x42uy; 0x29uy; 0x90uy; 0xf8uy; 0x93uy; 0x51uy; 0x1duy; 0x5duy; 0x4buy; 0xa1uy; 0x11uy;
      0xc7uy; 0x3duy; 0x53uy; 0xb1uy; 0xaeuy; 0x4euy; 0xeduy; 0x57uy; 0x1cuy; 0xbauy; 0x11uy; 0x2fuy;
      0xf4uy; 0xaduy; 0x7auy; 0x63uy; 0xf2uy; 0xb3uy; 0x75uy; 0x9euy; 0x5fuy; 0xeauy; 0xb9uy; 0xaauy;
      0x27uy; 0x34uy; 0x81uy; 0x9buy; 0x84uy; 0xe3uy; 0x3cuy; 0x0auy; 0x2cuy; 0xd8uy; 0x6auy; 0x99uy;
      0x00uy; 0x2buy; 0x56uy; 0x15uy; 0xcbuy; 0x42uy; 0x35uy; 0xccuy; 0x1duy; 0x1euy; 0xbauy; 0x1auy;
      0xb7uy; 0xbduy; 0x17uy; 0x2euy; 0xfduy; 0x2euy; 0x87uy; 0xd4uy; 0xfcuy; 0xcduy; 0x63uy; 0x09uy;
      0x1duy; 0x7cuy; 0xaauy; 0x65uy; 0x34uy; 0xcduy; 0x7buy; 0x27uy; 0x87uy; 0x24uy; 0xa7uy; 0x6euy;
      0x84uy; 0x3cuy; 0xfbuy; 0x5auy; 0x9duy; 0x99uy; 0xf8uy; 0x47uy; 0x74uy; 0x73uy; 0x98uy; 0x3euy;
      0x50uy; 0x04uy; 0x07uy; 0xa4uy; 0x10uy; 0xcbuy; 0x72uy; 0x20uy; 0xf9uy; 0xf2uy; 0xecuy; 0xefuy;
      0x02uy; 0x49uy; 0xa5uy; 0x5euy; 0x75uy; 0x54uy; 0x43uy; 0x15uy; 0x55uy; 0xb2uy; 0xf4uy; 0xe2uy;
      0x7duy; 0x74uy; 0xdfuy; 0xaduy; 0xd5uy; 0xbcuy; 0x2fuy; 0x30uy; 0xb4uy; 0x00uy; 0x79uy; 0x07uy;
      0x5fuy; 0xb2uy; 0x9duy; 0x0buy; 0xd8uy; 0xf2uy; 0x7cuy; 0x78uy; 0xa1uy; 0x40uy; 0x4auy; 0x0fuy;
      0x94uy; 0x12uy; 0x95uy; 0x96uy; 0x0auy; 0xb3uy; 0x70uy; 0x0buy; 0xaduy; 0x74uy; 0xf2uy; 0x74uy;
      0xe0uy; 0x63uy; 0xeeuy; 0xf8uy; 0x1auy; 0xbcuy; 0x28uy; 0x09uy; 0xdcuy; 0xafuy; 0x67uy; 0xbduy;
      0x31uy; 0x02uy; 0xb3uy; 0x73uy; 0xdauy; 0xd0uy; 0x61uy; 0x1auy; 0x5duy; 0x36uy; 0x26uy; 0xf0uy;
      0x94uy; 0x98uy; 0xa1uy; 0xafuy; 0xa6uy; 0xebuy; 0xeeuy; 0xb7uy; 0xc0uy; 0xdeuy; 0x25uy; 0xeeuy;
      0xb9uy; 0x83uy; 0x54uy; 0xa2uy; 0x59uy; 0xaauy; 0xbeuy; 0xcduy; 0x0euy; 0x32uy; 0x7fuy; 0x3auy;
      0x0buy; 0xd9uy; 0xd5uy; 0x33uy; 0x08uy; 0x23uy; 0x49uy; 0xacuy; 0xbauy; 0x96uy; 0x76uy; 0x82uy;
      0x44uy; 0xceuy; 0xc7uy; 0x03uy
  ]

let test1_ss_expected =
  List.Tot.map u8_from_UInt8
    [
      0xc3uy; 0xe1uy; 0x21uy; 0x95uy; 0x7fuy; 0x7buy; 0xeeuy; 0xc6uy; 0xa0uy; 0xb1uy; 0x5duy; 0x3duy;
      0xf0uy; 0x96uy; 0xe7uy; 0x0buy
    ]

let test () : ML unit =
  assert_norm (List.Tot.length test1_enccoins == 16);
  assert_norm (List.Tot.length test1_keypaircoins == 2 * crypto_bytes + bytes_seed_a);
  assert_norm (List.Tot.length test1_enccoins == bytes_mu);
  assert_norm (List.Tot.length test1_ss_expected == crypto_bytes);
  assert_norm (List.Tot.length test1_pk_expected == crypto_publickeybytes);
  assert_norm (List.Tot.length test1_ct_expected == crypto_ciphertextbytes);
  assert_norm (List.Tot.length test1_sk_expected == crypto_secretkeybytes);
  let result =
    test_frodo test1_keypaircoins
      test1_enccoins
      test1_ss_expected
      test1_pk_expected
      test1_ct_expected
      test1_sk_expected
  in
  if result
  then IO.print_string "\n\nFrodoKEM : Success!\n"
  else IO.print_string "\n\nFrodoKEM: Failure :(\n";
  result
