(* A loose implementation of version 3 of the UUID spec:

   Version 3 UUIDs use a scheme deriving a UUID via MD5 from a URL, a fully
   qualified domain name, an object identifier, a distinguished name (DN as used
   in Lightweight Directory Access Protocol), or on names in unspecified
   namespaces. Version 3 UUIDs have the form xxxxxxxx-xxxx-3xxx-xxxx-xxxxxxxxxxxx
   with hexadecimal digits x.
*)

open Core_kernel.Std
module Unix = Core_unix

(* The base module is responsible for generating unique string identifiers.  It should be
   clear to a reader that the id generated has an extremely high probability of uniqueness
   across all possible machines, processes, and threads of execution. *)
module Base = struct
  type t = {
    hostname : string;
    pid      : Pid.t;
    time     : Time.t;
    counter  : int;
  }

  let next_counter =
    let counter = ref 0 in
    (fun () ->
      (* In OCaml this doesn't allocate, and threads can't context switch except on
         allocation *)
      incr counter;
      !counter)
  ;;

  let to_string t =
    String.concat ~sep:"-" [
      t.hostname;
      Int.to_string (Pid.to_int t.pid);
      Float.to_string (Time.to_float t.time);
      Int.to_string t.counter
    ]
  ;;

  let create () =
    {
      hostname = Unix.gethostname ();
      pid      = Unix.getpid ();
      time     = Time.now ();
      counter  = next_counter ();
    }
  ;;
end

module T = struct
  type t = string with bin_io

  let create () =
    let digest = Digest.to_hex (Digest.string (Base.to_string (Base.create ()))) in
    let s = String.create 36 in
    s.[8]  <- '-';
    s.[13] <- '-';
    s.[18] <- '-';
    s.[23] <- '-';
    String.blit ~src:digest ~dst:s ~src_pos:0 ~dst_pos:0 ~len:8;
    String.blit ~src:digest ~dst:s ~src_pos:8 ~dst_pos:9 ~len:4;
    String.blit ~src:digest ~dst:s ~src_pos:12 ~dst_pos:14 ~len:4;
    String.blit ~src:digest ~dst:s ~src_pos:16 ~dst_pos:19 ~len:4;
    String.blit ~src:digest ~dst:s ~src_pos:20 ~dst_pos:24 ~len:12;
    s.[14] <- '3';
    s
  ;;

  let to_string = ident

  let of_string s =
    match String.split ~on:'-' s with
    | [a; b; c; d; e] ->
      begin try
        assert (String.length a = 8);
        assert (String.length b = 4);
        assert (String.length c = 4);
        assert (String.length d = 4);
        assert (String.length e = 12);
        (* we don't check for a 3 in the version position (14) because we want to be
          generous about accepting UUIDs generated by other versions of the protocol, and
          we want to be resilient to future changes in this algorithm. *)
        s
      with
      | _ -> failwithf "%s: not a valid UUID" s ()
      end
    | _ -> failwithf "%s: not a valid UUID" s ()
  ;;

end

include T

include Identifiable.Make (struct
  let module_name = "Core.Std.Uuid"
  include T
  include Sexpable.Of_stringable (T)
  let compare t1 t2 = String.compare t1 t2
  let hash t = String.hash t
end)

module Test = struct
  let test_size = 100_000

  let no_collisions l =
    let rec loop set l =
      match l with
      | []        -> true
      | t :: rest ->
        if Set.mem set t
        then false
        else loop (Set.add set t) rest
    in
    loop Set.empty l
  ;;

  let generate (n:int) =
    let rec loop acc n =
      if Int.(=) n 0 then acc
      else loop (create () :: acc) (n - 1)
    in
    loop [] n
  ;;

  let thread_test () =
    let res1 = ref [] in
    let res2 = ref [] in
    let thread1 = Thread.create (fun () -> res1 := generate test_size) () in
    let thread2 = Thread.create (fun () -> res2 := generate test_size) () in
    Thread.join thread1;
    Thread.join thread2;
    no_collisions (List.rev_append !res1 !res2)
  ;;

  TEST = no_collisions (generate test_size)
  TEST = thread_test ()
end
