/* 
 *  $Id: send.c,v 1.34 2005/08/05 20:10:43 rader Exp $
 */

#include <stdio.h>
#include <signal.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <errno.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#ifdef __FreeBSD__
#include <stdlib.h> 
#else
#include <malloc.h> 
#endif
#include "pingd.h"

extern struct ping_list_entry *ping_list[];
extern int packet_size;
extern char *ping_packet;
extern int pingd_socket;
extern int ping_list_size;
extern int polling_interval;
extern int interpacket_delay;
extern int debuggin;

static int next_node_on_list = 0;
static int nodes_pinged = 0;
static time_t ping_list_start_time;

/*------------------------------------------------------------------*/

void send_pings()
{
  int i, rc;
  time_t now;
  struct timeval tval;

  signal(SIGALRM, SIG_IGN);
  if ( next_node_on_list == 0 ) {
    now = time(&now);
    ping_list_start_time = now;
    debug("send_pings() ping list started at "); 
    debug_ctime(now); debug(" (%d)\n",now);
    verbose("ping list starting\n");
    nodes_pinged = 0;
    /* mark nodes that respond within polling interval as down... */
    for ( i = 0; i < ping_list_size; i++ ) {
      if ( ping_list[i]->n_state == PINGING &&
           ping_list[i]->n_next_ping <= now ) {
        verbose(" %s (%s) dropped a ping packet\n", 
          ping_list[i]->n_addr, ping_list[i]->n_name);
        mark_node_as_down(i);
      }
    }

  } else {
    debug("send_pings() continuing ping list\n");
  }
  now = time(&now);
  debug("  looking for a node to ping at "); 
  debug_ctime(now); debug(" (%d)\n",now);
  for ( i = next_node_on_list; i < ping_list_size; i++ ) {

    debug("  ping list node %d: addr=%s name=%s\n",
      i,ping_list[i]->n_addr, ping_list[i]->n_name);

    if ( ping_list[i]->n_next_ping <= now ) {

      memcpy(ping_packet, &ping_list[i]->n_header, sizeof(struct icmp));
#ifdef __DUMP_PACKETS__
      dump_packet();
#endif

      debug("  pinging %s\n",ping_list[i]->n_addr);
      gettimeofday(&tval,0);
      ping_list[i]->n_ping_start_time = tval;
      rc = sendto(pingd_socket, ping_packet, packet_size, 0, 
        &ping_list[i]->n_sock, sizeof(struct sockaddr));

      if ( rc != packet_size ) {
        debug("  ping failed: ");
        if ( debuggin ) { perror(""); }
        verbose(" %s (%s) dropped a ping packet\n", 
          ping_list[i]->n_addr, ping_list[i]->n_name);
        mark_node_as_down(i);
      } else {
        ping_list[i]->n_state = PINGING;
      }
      ping_list[i]->n_next_ping += polling_interval;
      next_node_on_list = i + 1;
      set_alarm_itimer(0, interpacket_delay * 1000);
      return;

    } else {

      debug("  skipping %s: next_ping=%d > now=%d\n",
        ping_list[i]->n_addr, ping_list[i]->n_next_ping, now );

    }

  }
  now = time(&now);
  debug("  all nodes have been pung\n"); 
  write_stats_file();
  next_node_on_list = 0;
  verbose("ping list is done\n");
  ping_list_start_time += polling_interval;
  if ( ping_list_start_time <= now ) {
    /* ping list exe time overran next ping list     */
    /* time so back schedule next ping list for one  */
    /* second from now...                            */
    debug("  next ping list time is less than now!\n");
    debug("  setting next ping list time to now plus one second\n");
    ping_list_start_time = now + 1;
  }
  debug("  next ping list at "); 
  debug_ctime(ping_list_start_time); debug(" (%d)\n",ping_list_start_time);
  set_alarm_itimer(ping_list_start_time, 0);
  return;
}

/*------------------------------------------------------------------*/

mark_node_as_down(i)
int i;
{
  ping_list[i]->n_state = NOT_PINGING;
  ping_list[i]->n_latency = NOT_RESPONDING;
  ping_list[i]->n_latency_data[ping_list[i]->n_next_data_point] =
    NOT_RESPONDING;
  ping_list[i]->n_next_data_point++;
  if ( ping_list[i]->n_next_data_point > (NUM_DATA_POINTS - 1) ) {
    ping_list[i]->n_next_data_point = 0;
  }
  debug("  node %d is down: addr=%s name=%s latency=%d ms ave=%d ms\n",
    i, ping_list[i]->n_addr, ping_list[i]->n_name,
  ping_list[i]->n_latency, ping_list[i]->n_latency_ave);
#ifdef __DEBUG_DATA_POINTS__
      /* assumes NUM_DATA_POINTS == 20... can't use loop here because */
      /* because it will get interrupted causing disjointed output... */
      if ( debuggin ) { 
        int idx; 
        idx = i;
        fprintf(stderr,"node %d data...\n", idx);
	fprintf(stderr,"  %d %d %d %d %d %d %d %d %d %d\n",
          ping_list[idx]->n_latency_data[0], ping_list[idx]->n_latency_data[1],
          ping_list[idx]->n_latency_data[2], ping_list[idx]->n_latency_data[3],
          ping_list[idx]->n_latency_data[4], ping_list[idx]->n_latency_data[5],
          ping_list[idx]->n_latency_data[6], ping_list[idx]->n_latency_data[7],
          ping_list[idx]->n_latency_data[8], ping_list[idx]->n_latency_data[9]);
	fprintf(stderr,"  %d %d %d %d %d %d %d %d %d %d (next=%d)\n",
          ping_list[idx]->n_latency_data[10], ping_list[idx]->n_latency_data[11],
          ping_list[idx]->n_latency_data[12], ping_list[idx]->n_latency_data[13],
          ping_list[idx]->n_latency_data[14], ping_list[idx]->n_latency_data[15],
          ping_list[idx]->n_latency_data[16], ping_list[idx]->n_latency_data[17],
          ping_list[idx]->n_latency_data[18], ping_list[idx]->n_latency_data[19],
          ping_list[idx]->n_next_data_point);
        fflush(stderr);
      }
#endif
}


/*------------------------------------------------------------------*/

set_alarm_itimer(when_sec,delta_ms)
time_t when_sec;
time_t delta_ms;
{
  time_t now;
  struct itimerval timer_time, old_timer_time;

  timer_time.it_interval.tv_sec = 0;
  timer_time.it_interval.tv_usec = 0;
  if ( when_sec != 0 ) { 
    now = time(&now);
    timer_time.it_value.tv_sec = when_sec - now;
    timer_time.it_value.tv_usec = 0;
    signal(SIGALRM, send_pings);
    setitimer(ITIMER_REAL, &timer_time, &old_timer_time);
    return;
  }
  if ( delta_ms != 0 ) { 
    timer_time.it_value.tv_sec = 0;
    timer_time.it_value.tv_usec = delta_ms;
    signal(SIGALRM, send_pings);
    setitimer(ITIMER_REAL, &timer_time, &old_timer_time);
    return;
  }
}

/*------------------------------------------------------------------*/

#ifdef __DUMP_PACKETS__
dump_packet()
{
  register int i;
  fprintf(stderr,"\n");
  for (i = 0; i < packet_size; i++) {
    fprintf(stderr,"0x%-2x ", ping_packet[i]);
    if (((i + 1) % 8) == 0) fprintf(stderr,"\n");
  }
  fprintf(stderr,"\n");
}
#endif

