const React = require('react');

const ms = require('ms');

const importJSX = require('import-jsx');

const {
  Box
} = require('ink');

const Parser = require('tap-parser');

const chalk = require('chalk'); // Pull in all the tags here so we can re-export them


const AssertCounts = importJSX('./assert-counts.js');
const AssertName = importJSX('./assert-name.js');
const Counts = importJSX('./counts.js');
const Footer = importJSX('./footer.js');
const Log = importJSX('./log.js');
const PassFail = importJSX('./pass-fail.js');
const Result = importJSX('./result.js');
const Runs = importJSX('./runs.js');
const StatusMark = importJSX('./status-mark.js');
const SuiteCounts = importJSX('./suite-counts.js');
const Summary = importJSX('./summary.js');
const TestPoint = importJSX('./test-point.js');
const Test = importJSX('./test.js');

class Base extends React.Component {
  get Summary() {
    return Summary;
  }

  get Runs() {
    return Runs;
  }

  get Log() {
    return Log;
  }

  get Footer() {
    return Footer;
  }

  constructor({
    tap
  }) {
    super();
    this.state = {
      // the stuff in the static section.  most importantly, errors in
      // real time, but also console output, test summaries, etc.
      log: [],
      // all tests, done and queued
      tests: [],
      // currently running
      runs: [],
      // tap.results at the end
      results: null,
      // counts of all relevant test points
      // debounced on this.assertCounts
      assertCounts: {
        total: 0,
        pass: 0,
        fail: 0,
        skip: 0,
        todo: 0
      },
      // a count of all the test suites run
      suiteCounts: {
        total: 0,
        pass: 0,
        fail: 0,
        skip: 0,
        todo: 0
      },
      // total elapsed time
      time: 0,
      bailout: null
    };
    this.start = Date.now();
    this.timer = setInterval(() => this.setState(prevState => ({ ...prevState,
      time: this.time,
      assertCounts: this.assertCounts
    })), 200); // keep counters on the object itself, to debounce

    this.assertCounts = {
      total: 0,
      pass: 0,
      fail: 0,
      skip: 0,
      todo: 0
    }, this.counterBouncer = null;
    tap.on('subtestAdd', t => this.addTest(t));
    tap.on('subtestStart', t => this.startTest(t));
    tap.on('subtestEnd', t => this.endTest(t));
    tap.on('end', () => this.endAll(tap));
    tap.once('bailout', message => this.bailout(message)); // Handle data that might come out of the tap object itself.

    tap.on('extra', this.onRaw(tap, 1));
    tap.parser.on('result', res => {
      if (res.fullname === 'TAP') {
        this.logRes(tap, res);
        this.inc(res.todo ? 'todo' : res.skip ? 'skip' : res.ok ? 'pass' : 'fail');
      }
    }); // consume the text stream, but ignore it.
    // we get all we need from the child test objects.

    this.tapResume(tap);
  }

  tapResume(tap) {
    tap.resume();
  }

  get time() {
    return Date.now() - this.start;
  }

  componentWillUnmount() {
    clearTimeout(this.counterBouncer);
    clearInterval(this.timer);
  }

  bailout(bailout, test = null) {
    this.bailedOut = bailout;
    return this.setState(prevState => prevState.bailout ? prevState : { ...prevState,
      runs: test ? prevState.runs.filter(t => t.childId !== test.childId) : [],
      // if we bail out, then we should only show the bailout,
      // or the counts get confusing, because we never receive a testEnd
      // for the other ones.
      tests: prevState.tests.filter(t => test ? t.childId === test.childId : t.results && t.results.bailout === bailout),
      bailout,
      assertCounts: this.assertCounts,
      time: this.time
    });
  }

  inc(type) {
    this.assertCounts.total++;
    this.assertCounts[type]++;
    if (this.counterBouncer) return;
    this.counterBouncer = setTimeout(() => {
      this.counterBouncer = null;
      this.setState(prevState => ({ ...prevState,
        assertCounts: this.assertCounts
      }));
    }, 50);
  }

  addTest(test) {
    this.setState(prevState => ({ ...prevState,
      tests: prevState.tests.concat(test),
      suiteCounts: { ...prevState.suiteCounts,
        total: prevState.suiteCounts.total + 1
      },
      assertCounts: this.assertCounts,
      time: this.time
    }));
    test.on('preprocess', options => options.stdio = 'pipe').on('process', proc => {
      proc.stderr.setEncoding('utf8');
      proc.stderr.on('data', this.onRaw(test, 2));
    }).parser.on('extra', this.onRaw(test, 1)).on('pass', res => this.inc('pass')).on('todo', res => (this.inc('todo'), this.logRes(test, res))).on('skip', res => (this.inc('skip'), this.logRes(test, res))).on('fail', res => (this.inc('fail'), this.logRes(test, res)));
  }

  onRaw(test, fd) {
    const p = ` ${fd}>`;
    return raw => {
      const pref = chalk.bold.dim(test.name + p + ' ');
      raw = raw.replace(/\n$/, '').replace(/^/gm, pref);
      this.setState(prevState => ({ ...prevState,
        log: prevState.log.concat({
          raw
        }),
        assertCounts: this.assertCounts
      }));
    };
  }

  logRes(test, res) {
    res.testName = test.name;
    this.setState(prevState => ({ ...prevState,
      log: prevState.log.concat({
        res
      }),
      assertCounts: this.assertCounts,
      time: this.time
    }));
  }

  startTest(test) {
    test.startTime = Date.now();
    test.once('bailout', message => this.bailout(message, test));
    this.setState(prevState => prevState.bailout ? prevState : { ...prevState,
      runs: prevState.runs.concat(test),
      assertCounts: this.assertCounts
    });
  }

  endTest(test) {
    test.endTime = Date.now(); // put it in the appropriate bucket.
    // live update assertion handed by tap.parser event

    const ok = test.results && test.results.ok;
    const skip = test.options.skip && ok !== false;
    const todo = test.options.todo && ok !== false;
    const bucket = skip ? 'skip' : todo ? 'todo' : !ok ? 'fail' : 'pass';
    this.setState(prevState => prevState.bailout ? prevState : { ...prevState,
      log: prevState.log.concat({
        test
      }),
      runs: prevState.runs.filter(t => t.childId !== test.childId),
      suiteCounts: { ...prevState.suiteCounts,
        [bucket]: prevState.suiteCounts[bucket] + 1
      },
      time: this.time,
      assertCounts: this.assertCounts
    });
  }

  endAll(tap) {
    clearInterval(this.timer);
    clearInterval(this.counterBouncer);
    this.setState(prevState => ({ ...prevState,
      results: tap.results,
      assertCounts: this.assertCounts,
      time: tap.time || this.time
    }));
  }

  render() {
    return /*#__PURE__*/React.createElement(Box, {
      flexDirection: "column"
    }, /*#__PURE__*/React.createElement(this.Log, {
      log: this.state.log
    }), /*#__PURE__*/React.createElement(this.Runs, {
      runs: this.state.runs
    }), this.state.results ? /*#__PURE__*/React.createElement(this.Summary, {
      tests: this.state.tests,
      results: this.state.results
    }) : '', /*#__PURE__*/React.createElement(this.Footer, {
      suiteCounts: this.state.suiteCounts,
      assertCounts: this.state.assertCounts,
      time: this.state.time
    }));
  }

}

Base.AssertCounts = AssertCounts;
Base.AssertName = AssertName;
Base.Counts = Counts;
Base.Footer = Footer;
Base.Log = Log;
Base.PassFail = PassFail;
Base.Result = Result;
Base.Runs = Runs;
Base.StatusMark = StatusMark;
Base.SuiteCounts = SuiteCounts;
Base.Summary = Summary;
Base.TestPoint = TestPoint;
Base.Test = Test;
module.exports = Base;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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